/*----------------------------------------------------------------

 rank - numerical computation of the rank of a matrix

 Call: numeric::rank(A, <eps>)
 Parameter:

  A   --  m x n matrix (DOM_ARRAY, DOM_HFARRAY, or category Cat::Matrix)
  eps --  a positive numerical value; default value 10^(-DIGITS)
                                                                   

  Details:

    The singular values of A are computed. If smax is the
    largest singular value, the number of singular values
    is returned that are larger than eps*singmax.
    This number is the numerical rank of the matrix.


  Examples: 

    //-----------------------------
    A:= matrix([[1,  1,   I] , 
                [1,exp(1),2] , 
                [1,  2,   3] ]):
    numeric::rank(A)

                            2
    //-----------------------------

    A:= matrix([[1, 1, 1, 0, 0, 0, 0],
                [2, 2, 2, 0, 0, 0, 0],
                [3, 3, 3, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 1, 1],
                [0, 0, 0, 0, 0, 1, 1]]):
    linalg::rank(A) = numeric::rank(A)

                          2 = 2
    //-----------------------------
    

    A:= matrix([[1,  1,   I], 
                [1,exp(1),I], 
                [1,  2,   I]]):
    numeric::rank(A)
                            2

    //-----------------------------
    numeric::rank(matrix([[1, 1, 10^30],
                          [1, 1, 10^30]]))

                            1

    //-----------------------------
    // The rank of the following matrix is 2, but the
    // the small singular value is neglected (regarded 
    // as zero relative to the largest value, because
    // eps = 10^(-DIGITS) by default). Hence, rank = 1
    // is returned:

    A:= matrix([[1,        1, 10^30],
                [10^(-30), 1, 10^30]]):

    numeric::singularvalues(A)

               [1.414213562e30, 1.780411173e14]


    numeric::rank(A)
                            1

    // Choose eps = 10^(-20) to detect the full rank:

    numeric::rank(A, 10^(-20))

                            2

-----------------------------------------------------------------*/

numeric::rank:=proc(A)
local eps, furtherargs, i, tmp, singvals, maxsingval;
begin

  if args(0) < 1 then error("expecting at least one argument") end_if;

  //-----------------
  // Check the matrix 
  //-----------------
  if domtype(args(1)) = DOM_ARRAY or
     domtype(args(1)) = DOM_HFARRAY then
     if op(A, [0, 1]) <> 2 then
        error("first argument: expecting a 2-dimensional array");
     end_if;
  elif A::dom::hasProp(Cat::Matrix) <> TRUE then
     error("expecting an array or a matrix of category 'Cat::Matrix'"):
  end_if;

  eps:= 10.0^(-DIGITS):
  furtherargs:= null():
  for i from 2 to args(0) do
      if domtype(float(args(i))) <> DOM_FLOAT then
         furtherargs:= furtherargs, args(i);
         next;
      end_if;
      eps:= float(args(i)):
      if domtype(eps) <> DOM_FLOAT then
         error("2nd argument: expecting a real numerical value"):
      end_if;
      if eps <= 0 then
         error("expecting a positive numerical value for the tolerance"):
      end_if;
  end_for;
  if (tmp:= {furtherargs} minus {Hard, HardwareFloats, Soft, SoftwareFloats}) <> {} then
     error("unexpected option '".expr2text(op(tmp, 1))."'");
  end_if;
  singvals:= numeric::singularvalues(A, furtherargs):
  maxsingval:= max(singvals):
  nops(select(singvals, x -> bool(x > eps*maxsingval)));
end_proc:
