/*--
numeric::rationalize - convert a floating point numbers to an exact 
                       or approximate rational number

Calling sequence:
rationalize(object);
rationalize(object, Exact);
rationalize(object), Minimize [, digits])
rationalize(object), Restore [, digits])

Parameters:
expr   -- MuPAD object of any basic type: expression, list, set, DOM_POLY, ...
digits -- optional integer, must satisfy 0 < digits <= DIGITS

Summary:
The function rationalize converts all floating-point numbers contained
in the expression expr to exact rational numbers. Lists, sets, tables
and arrays are also converted.

Domain-elements are not converted !!! (we do not know what the
map-method of the domain does, there is danger of an endless loop).

An "exact" conversion of a floating-point number f will be performed 
by default: wite f = sign(f) * m * 10^e with a mantissa 1<=m<10.
Then  sign(f)* round(m*10^digits)/10^digits * 10^e is returned.

The flag Minimize indicates, that a rational approximation of f with
smaller complexity (i.e., smaller  numerator/denominator)
should be computed.

The flag Restore indicates, that a rational approximation of f is
is searched for, such that 
      numeric::rationalize(float(rat),Restore) = rat
holds for rational numbers rat. This is purely heuristic.
It works reasonably well, if numerator and denominator of rat
are not too big! Restauration works better for higher DIGITS !

The accuracy of the conversion will depend on the value of the global
variable DIGITS, or the value of digits if specified as an integer.

For the default setting the rational r approximating the float f satisfies
   abs(r-f)< 10^(-digits)*f   (with the default digits = DIGITS)
With Minimize the rational r approximating the float f satisfies
   abs(r-f)< 10^(-digits)*f   (with the default digits = DIGITS)
With Restore the rational r approximating the float f satisfies
   abs(r-f)< 10^(-digits/2)*f   (with the default digits = DIGITS)

Examples:
>> DIGITS:=5:
>> numeric::rationalize(0.33333);                -> 33333/10000
// Note: (0.3333-1/3)> 1/3*10^(-DIGITS)!
>> numeric::rationalize(0.33333, Minimize);      -> 33333/10000
// Note: (0.33331-1/3)< 1/3*10^(-DIGITS)!
>> numeric::rationalize(0.333331, Minimize);     -> 1/3
>> numeric::rationalize(14.285);                 -> 2857/200
>> numeric::rationalize(14.2857, Minimize);      -> 100/7
>> numeric::rationalize(10.0e-11);               -> 1/10000000000
>> numeric::rationalize(float(PI), Minimize, 1); -> 3
>> numeric::rationalize(float(PI), Minimize, 5); -> 355/113
>> numeric::rationalize(0.2*a+b^(0.2*I))         -> a/5 + b^(1/5*I)
>> numeric::rationalize(float(1234/56789), Minimize); -> 49/2255
>> numeric::rationalize(float(123/4567));         -> 269323/10000000
>> numeric::rationalize(float(123/4567), Minimize); -> 23/854
>> numeric::rationalize(float(123/4567),Restore); -> 123/4567

--*/

numeric::rationalize:= proc(f)
   local flag, digits, opt, rec_rat, float2rat, ln10, furtherargs;
begin
   if args(0)<1 then error("expecting at least one argument"); end_if;
   furtherargs:=args(2..args(0));                      
   ln10:= slot(ln,"float")(10.0):// only needed to estimate exponents of floats,
                                 // moderate accuracy for ln10 suffices.

/* ------ internal methods ----- */


rec_rat:= proc(f)
name numeric::rationalize@rec_rat;
begin
   /* complete list of kernel data types, MuPAD 1.5, 25.9.99 */ 
   case domtype(f)
     // first: most frequently encountered types
     of DOM_EXPR   do return(map(f,rec_rat));
     of DOM_FLOAT  do return(float2rat(f));
     of DOM_COMPLEX do /* map doesn't work on complex numbers */
                    return(rec_rat(op(f,1))+rec_rat(op(f,2))*I)
     of DOM_INT    do 
     of DOM_RAT    do 
     of DOM_INTERVAL do // float intervals cannot be turned rational
     of DOM_IDENT  do return(f);
     of DOM_POLY   do return(mapcoeffs(f,rec_rat));
     // containers: probably encounterd only once 
     //             per call to numeric::rationalize:
     of DOM_LIST   do
     of DOM_SET    do
     of DOM_ARRAY  do
     of densematrix do
     of DOM_TABLE  do return(map(f,rec_rat));
     of DOM_HFARRAY do return(f);
     of matrix do return(matrix::mapNonZeroes(f, rec_rat));

     // exotic types, likely never to be encountered:
     of DOM_BOOL   do
     of DOM_STRING do
     of DOM_FAIL   do
     of DOM_FRAME  do
     of DOM_NIL    do
     of DOM_NULL   do  // can't happen because nargs(0)<>0
     of DOM_VAR    do  // should not happen, locals are NIL or initialized
     of DOM_PROC   do
     of DOM_EXEC   do
     of DOM_FUNC_ENV do
     of DOM_PROC_ENV do
     of DOM_DOMAIN do return(f);
     otherwise 
        // numeric::rationalize should not handle domain elements:
        // There is danger of unterminated recursion! E.g.:
        // f:= Dom::AlgebraicExtension(Dom::Float, x^2+1=0,x) (1.0);
        // map(f,rec_rat) yields rec_rat(f), so there is an infinite
        // recursion!
        // All objects of kernel domains are dealt with above, so
        // return(f);
        // is appropriate here.

        // Instead (as a safety measure for future implementation of
        // further kernel domains) we trap objects from library domains:

        // potential test for library domains:
        // Experimental Type::Internal is defined below
        //if not testtype(f, Type::Internal) then return(f) end_if;
        // much faster is:

        if domtype(extop(f,0)) = DOM_DOMAIN then
          // allow overloading
          if f::dom::numeric_rationalize<>FAIL then
            return(f::dom::numeric_rationalize(f, furtherargs))
          else  
            return(f)
          end_if;
        end_if;
       
        // Now, further (future) kernel types may be processed via
        if nops(f)>1 or op(f)<>f // e.g., f:= sin(0.1)
           then return(map(f, rec_rat))
        end_if;

        // finally:
        return(f);
   end_case;
end_proc:
/* ------------------------------ */
float2rat := proc(f) // approximate f (of type DOM_FLOAT) by rational
   local p0,p1,p2,q0,q1,q2,mant,expo,r,s,p,q,n,hi,tmp,unlocked;
   name numeric::rationalize@float2rat;
   save DIGITS;
begin
   if iszero(f) then return(0) end_if;
   if f = RD_INF then 
      return(infinity)
   elif f = RD_NINF then
      return(-infinity)
   elif f = RD_NAN then
      return(undefined)
   end_if;

   if f < 0 then f:= -f; s:= -1 else s:= 1 end_if;
   /* ------------------------------------------------------------*/
   if flag = Exact then
     // write f= mant*10^expo  with float mant:  1 <= mant < 10
     expo:= trunc(slot(ln,"float")(f)/ln10)-1;
     if f>=1 then expo:= expo + 1 end_if;
     // Now rationalize mantissa mant:= f*10^(-expo).
     // Do not(!!!) use round(f* 10^(digits-expo)), this may
     // yield results different from round((f*10^(-expo))*10^digits)
     // due to roundoff !!
     mant:= round((f*10^(-expo))*10^digits); 
     // now we have rationals and may combine 10^(-digits)*10^(expo)
     return(s*mant*10^(expo-digits)) 
   end_if;
   /* ------------------------------------------------------------*/
   DIGITS:= DIGITS+1; // Increase DIGITS for Minimize and RESTORE
   /* ------------------------------------------------------------*/
   if flag = Minimize then
     // write f= mant*10^expo with integer mant: 1<= mant <=10^(1+digits)
     expo:= trunc(slot(ln,"float")(f)/ln10) - digits - 1:
     if f>=1 then expo:= expo + 1 end_if;
     mant:= round( f*10^(-expo) ): 
     if expo >= 0 then return(s*mant*10^expo) end_if;
     // Use rational approximation p/q computed above 
     // and simplify via continued fraction expansion:
     expo:= 10^(-expo);
     p:= mant;  q:= expo; // rational to be simplified is p/q
     p0:= 0;  p1:= 1;
     q0:= 1;  q1:= 0;
     while TRUE do
       n:= mant div expo; r:= mant mod expo;
       p2:= n*p1+p0;  q2:= n*q1+q0;
       /* termination criterion: the approximation is exact within digits */
       /* decimal places, i.e, abs(p2/q2-p/q)| < 10^(-digits)*abs(p/q)   */
       if r=0 or 10^digits*specfunc::abs(p*q2-q*p2) < q*p2 
          then return(s*p2/q2)
       end_if;
       p0:= p1; p1:= p2; q0:= q1; q1:= q2; mant:= expo; expo:= r;
     end_while;
   end_if;
   /* ------------------------------------------------------------*/
   if flag=Restore then
    hi:= float(10^((digits+1)/2));
    n:= trunc(f);
    p0:= 1: q0:= 0:
    p1:= n: q1:= 1: 
    if n=0
       then unlocked:= FALSE //lock termination criterion, until there is a
       else unlocked:= TRUE  //first notrivial coefficient of the continued
    end_if;                  //fraction expansion
    while q1<hi do
       /* termination criterion: f = rational = p1/q1 */
       if iszero(f-n) then return(s*p1/q1); end_if;
       /* next step of continued fraction expansion */
       f:= 1/(f-n);  
       /* with n=trunc(old_f) we must have new_f=1/(old_f-n)>1 */
       /* stop, if loss of precision due to roundoff in f-n, i.e., if new_f <1 */
       if f<1 then return(s*p1/q1); end_if;
       n:= trunc(f); /* note that n >=1 is guaranteed! */
       tmp:= n*p1+p0; p0:=p1: p1:=tmp;
       tmp:= n*q1+q0; q0:=q1: q1:=tmp;
       /* termination criterion: stop when coefficient n of
          the continued fraction expansion is large (n>hi).
          Ignore the large coefficient n, i.e., use p0, q0: */
       if n>hi and unlocked then return(s*p0/q0) end_if;
       unlocked:= TRUE:
    end_while; 
    return(s*p1/q1);
   end_if:
   /* ------------------------------------------------------------*/
end_proc:

      /* ---------------------------- */
      /* main of numeric::rationalize */
      /* ---------------------------- */
      digits:= DIGITS;  // default
      flag := Exact;    // default
      for opt in [args(2..args(0))] do
          if opt = hold(Exact)    then flag:= Exact: next end_if;
          if opt = hold(Minimize) then flag:= Minimize: next end_if;
          if opt = hold(Restore)  then flag:= Restore: next end_if;
          if domtype(opt) = DOM_INT then
             if opt<0 then error("digits must be a positive integer") end_if;
             if opt>DIGITS then 
                   error("illegal requested precision > DIGITS") 
             end_if;
             digits:= opt;
             next;
          end_if;
          error("unknown option");
      end_for;
      rec_rat(f)
end_proc:

// end of file 

/*
unprotect(Type);
Type::Internal := new(Type, "Internal",
      proc(t,x) begin
        if args(0) <> 2 then return(FALSE) end_if;
        if contains({DOM_ARRAY, DOM_BOOL, DOM_COMPLEX, DOM_DOMAIN, DOM_EXEC, 
                     DOM_EXPR, DOM_FAIL, DOM_FLOAT, DOM_FUNC_ENV, DOM_IDENT,
                     DOM_INT, DOM_LIST, DOM_NIL, DOM_NULL, DOM_POLY,
                     DOM_PROC, DOM_PROC_ENV, DOM_RAT, DOM_SET, DOM_HFARRAY,
                     DOM_STRING, DOM_TABLE, DOM_VAR}, x::dom) 
           then TRUE
           else FALSE
        end_if;
      end_proc, FALSE, FALSE, "Rational"):
*/
