
//------   file  singval.mu  17.9.97   ---------------------
/*  Help file:

 singularvalues - numerical singular values of a matrix

 Call:

 numeric::singularvalues(A <, hard_soft>) 
                                                                   
 Parameter:

  A        --  a square matrix (DOM_ARRAY, DOM_HFARRAY or category Cat::Matrix)
  hard_soft -- either 'Hard', 'HardwareFloats', 'Soft' or 'SoftwareFloats'

                                                                   
 Synopsis:

 The singular values of an (m x n) matrix A are the p=min(m,n)
 real non-negative square roots of the eigenvalues of A'A (if p=n)
 or of AA' (if p=m), where A' is the Hermitean transpose of A.

 numeric::singularvalues(A); returns the sorted list [d[1],..,d[p]]
 of singular values d[1] >= .. >= d[p] >= 0 .
                                                                 
 Remark: Non-float numerical entries are allowed in the matrix. 
         They will be evaluated to floats in the course of the 
         computation.  Non-numeric symbolic entries will lead to 
         an error.      

 Remark: The singular values may also be computed via
               numeric::eigenvalues(AA');
         or    numeric::eigenvalues(A'A);
         However, in comparison with the singular value 
         routine, for ill-conditioned matrices about twice 
         as many DIGITS are required in the eigenvalue 
         routine to produce results of similar quality.
         
 Examples:                                                         
                                                                  
 >> m := 3: n := 2: A := array(1..m,1..n,[[1,2*I ],[2,3],[3,4]]):

 >> numeric::singularvalues(A);                                   

                        [6.411107219, 1.377571855]

 See also:  numeric::eigenvalues, numeric::eigenvectors,
            numeric::singularvectors, numeric::spectralradius
                                                                  */
//-----------------------------------------------------------------
numeric::singularvalues := proc(A)
local m, n, useHardwareFloats, HardwareFloatsRequested, _norm,
      mgtn, B, oldA, real, i, j, k, l, d, e, s, ss, c, h , f, g,
      eps, docancel, x, y, z;
save DIGITS;
begin
    if args(0) < 1 then
       error("expecting at least one argument");
    end_if;

    //------------------
    // Check the matrix
    //------------------
    if contains({DOM_ARRAY, DOM_HFARRAY}, domtype(args(1))) then
       if op(A, [0, 1]) <> 2 then
          error("first argument: expecting a 2-dimensional array");
       end_if;
       if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
         if domtype(A) = DOM_ARRAY then
            A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                   1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
         else
           A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                       1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]),
                       [op(A)]);
         end_if:
       end_if;
       [m, n]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
    elif A::dom::hasProp(Cat::Matrix)=TRUE then
       [m, n]:= A::dom::matdim(A):
    else
       error("expecting an array, an hfarray, or a matrix of category 'Cat::Matrix'"):
    end_if;

    //---------------------------------------------
    // Check the options
    //---------------------------------------------
    // set defaults
    if DIGITS <= 15 then
         useHardwareFloats:= TRUE;
    else useHardwareFloats:= FALSE;
    end_if;
    HardwareFloatsRequested:= FALSE;
    if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
    end_if:
  
    for i from 2 to args(0) do
        case args(i)
        of Hard do
        of HardwareFloats do
                       useHardwareFloats:= TRUE:
                       HardwareFloatsRequested:= TRUE;
                       break;
        of Soft do
        of SoftwareFloats do
                       useHardwareFloats:= FALSE:
                       HardwareFloatsRequested:= FALSE;
                       break;
        otherwise
           error("unknown option ".expr2text(args(i)));
        end_case;
    end_for;

    //--------------------------------
    // ----- use HardwareFloats ------
    //--------------------------------
    case useHardwareFloats // use case because we want to use break below
    of TRUE do
       if DIGITS > 15 then
          if HardwareFloatsRequested then
            warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                    " cannot be achieved with the requested 'HardwareFloats'."):
          end_if;
          useHardwareFloats:= FALSE;
          break;
       end_if;
       if not numeric::startHardwareFloats("singularvalues") then
          userinfo(1, "cannot start the hardware floats interface"):
          if HardwareFloatsRequested then
             warning("cannot start the 'HardwareFloats' interface, ".
                     "using 'SoftwareFloats' instead");
          end_if;
          useHardwareFloats:= FALSE;
          break;
       end_if;

       userinfo(1, "trzing hardware floats"):

       /*-----------------------------------------------------------
       //  convert to hfarray
       -----------------------------------------------------------*/
       // The hfa module expects an hfarray as input.
       // If A contains symbols or numerical stuff that
       // cannot be represented by hardware floats, the
       // conversion to an hfarray will fail. Use traperror
       // to detect these cases and fall back to software floats
       if domtype(A) = DOM_HFARRAY then
          B:= A
       else
          if traperror((
               B:= hfarray(1..m, 1..n, [op(A)]);
             )) <> 0 then
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             break;
          end_if;
       end_if:

       // normalize to avoid underflow/overflows in the hfa module
       _norm:= norm(B):
       if iszero(_norm) then
          return([float(0) $ n]);
       end_if:
       B:= B/_norm:

       /*-----------------------------------------------------------
       //  call the hfa module
       -----------------------------------------------------------*/
       if traperror((d:= hfa::singularvalues(B))) = 0 then
          if has(d, RD_NAN) or
             has(d, RD_INF) or
             has(d, RD_NINF) then
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
          else
             // undo normalization and sort:
             return(numeric::sort(map(d, _mult, _norm)));
          end_if:
       else
          userinfo(1, "'HardwareFloats' failed"):
       end_if:
    end_case; 
    //---------------------------------
    // end of useHardwareFloats = TRUE
    //---------------------------------


    //---------------------------------
    //---- use SoftwareFloats ---------
    //---------------------------------
    if HardwareFloatsRequested then
       warning("'HardwareFloats' failed, using 'SoftwareFloats' instead"):
    end_if;

    userinfo(1, "using software floats"):
    // convert to domtype DOM_ARRAY for speed
    if domtype(A) = DOM_HFARRAY then
       assert(nops(A) = m*n):
       A:= array(1..m, 1..n, [op(A)]);
    elif A::dom::hasProp(Cat::Matrix)=TRUE then
       A:= expr(A);
    end_if;
    if domtype(A) <> DOM_ARRAY then
       error("the matrix must be an array, an hfarray, ".
             "or a matrix of category 'Cat::Matrix'");
    end_if:

    DIGITS := DIGITS + max(10, trunc(min(m,n)/10));
    A:=map(A,float):
    if {op(map(A,domtype))} minus {DOM_FLOAT, DOM_COMPLEX} <> {} 
       then error("non-numeric entries not handled");
    end_if:
    if {op(map(A,(iszero@Im)))} = {TRUE}     // is matrix real ? 
       then real:=TRUE else real:=FALSE
    end_if:
    A:=subs(A,float(0)=0):

    if m >= n then
         mgtn:=TRUE;
    else mgtn:= FALSE; 
         [m, n]:= [n,m];
         oldA:=A: 
         A:=array(1..m,1..n);
         if real then ((A[i,j]:=oldA[j,i] )          $ i=1..m)$ j=1..n;
                 else ((A[i,j]:=conjugate(oldA[j,i]))$ i=1..m)$ j=1..n;
         end_if;
    end_if;
    d:=[0$n  ];  // initialize list to store diagonal       
    e:=[0$n-1];  // initialize list to store super diagonal 
    for j from 1 to n do 
        // Householder step from the left to obtain upper   
        // triangular form. Householder data are stored in  
        // lower triangular part of A.                      
        s:= _plus(specfunc::abs(A[i,j])^2 $ i=j+1..m);
        if iszero(s) then d[j]:=specfunc::abs(A[j,j]);
        else ss:=s+specfunc::abs(A[j,j])^2;
             s:=specfunc::sqrt(ss):
             h := 1 / (ss + specfunc::abs(A[j,j]) * s);
             if not iszero(A[j,j]) then s:=sign(A[j,j])*s;end_if:
             A[j,j]:=s+A[j,j]; d[j]:=specfunc::abs(s);
             for i from j+1 to n do
                 if real then   s:=_plus(A[k,j] *A[k,i]$ k=j..m) ;
                 else s:=_plus(conjugate(A[k,j])*A[k,i]$ k=j..m) ;
                 end_if;
                 f:=s*h;
                 (A[k,i]:=A[k,i]-f*A[k,j]) $ k=j..m ;
             end_for;
        end_if;
        // Householder step from the right to obtain upper  
        // bidiagonal form. Householder data are stored in  
        // strictly upper triangular part of A.             
        s:= _plus(specfunc::abs(A[j,i])^2 $ i=j+2..n);
        if iszero(s) then if j<n then e[j]:=specfunc::abs(A[j,j+1]); end_if;
        else ss:=s+specfunc::abs(A[j,j+1])^2;
             s:=specfunc::sqrt(ss):
             h := 1 / (ss + specfunc::abs(A[j,j+1]) * s);
             if not iszero(A[j,j+1]) then s:=sign(A[j,j+1])*s; end_if:
             A[j,j+1]:=s+A[j,j+1]; e[j]:=specfunc::abs(s); 
             for i from j+1 to m do
                 if real then   s:=_plus(A[j,k] *A[i,k] $ k=j+1..n) ;
                 else s:=_plus(conjugate(A[j,k])*A[i,k] $ k=j+1..n) ;
                 end_if;
                 f:=s*h;
                 (A[i,k]:=A[i,k]-f*A[j,k]) $ k=j+1..n ;
             end_for;
        end_if;
    end_for;

    // now A contains in its lower/strictly upper part the 
    // Householder data of the left/right transformation.  
    // The resulting real bi-diagonal matrix               
    //    [ d[1] e[1]         ]                            
    //    [      d[2] e[2]    ]  is stored in d and e.     
    //    [           ..   .. ]                            

// ------ now QR iteration to diagonalize bi-diagonal matrix ---- 

  e:=[0,op(e)];
  eps:=max(specfunc::abs(d[i])+specfunc::abs(e[i]) $ i=1..n)/10^DIGITS;
  for k from n downto 1 do
     repeat
       docancel:=TRUE;
       for l from k downto 1 do
          if specfunc::abs(e[l])<=eps then docancel:=FALSE; break end_if;
          if specfunc::abs(d[l-1]) <= eps then break end_if;
       end_for;
 
       if docancel then // cancellation of e[l] for l>1 
          c:=0;s:=1;
          for i from l to k do
              f:=s*e[i];e[i]:=c*e[i]:
              if specfunc::abs(f)<=eps then break; end_if;
              g:=d[i]; h:=specfunc::sqrt(f^2+g^2); d[i]:=h; c:=g/h; s:=-f/h;
          end_for;
       end_if;          // end of cancellation 

       z:=d[k]; if l=k then break; end_if;

       // compute shift 
       x:=d[l]; y:=d[k-1]; g:=e[k-1]; h:=e[k];
       f:= ( (y-z)*(y+z)+(g-h)*(g+h) ) / (2*h*y);
       g:=specfunc::sqrt(1.0+f^2); 
       if f<0 then f:= ( (x-z)*(x+z)+h*( y/(f-g)-h )  )/x ;
              else f:= ( (x-z)*(x+z)+h*( y/(f+g)-h )  )/x ;
       end_if;
 
       // do QR step 
       c:=1;s:=1;
       for i from l+1 to k do
          g:=e[i]; y:=d[i]: h:=s*g; g:=c*g;
          z:=specfunc::sqrt(f^2+h^2);  e[i-1]:=z;
          if iszero(z) then
             [c, s]:= [float(1), float(0)];
          else
             c:=f/z;
             s:=h/z;
          end_if;
          f:=x*c+g*s; g:=-x*s +g*c; h:=y*s; y:=y*c;
          z:=specfunc::sqrt(f^2+h^2); 
          d[i-1]:=z; 
          if iszero(z) then
             [c, s]:= [float(1), float(0)];
          else
             c:=f/z;
             s:=h/z;
          end_if;
          f:=c*g+s*y; x:=-s*g+c*y;
       end_for;
       e[l]:=0; e[k]:=f; d[k]:=x;
     until FALSE end_repeat;
     if z<0 then d[k]:=-z; end_if;
  end_for;

  // ------ sort singular values from large to small ------ 
  return(numeric::sort(d));
end_proc:
//--------  end of procedure singularvalues ---------------
