//---------------------------------------------------------
/*  Help file:

 singularvectors - numerical singular value decomposition of a matrix

 Call:

 numeric::singularvectors(A <,hard_soft> 
                          <, ReturnType = t>
                          <, NoLeftVectors> 
                          <, NoRightVectors> 
                          <, NoResidues>
                          <, NoWarning>) 

 numeric::svd(A <,hard_soft> <, NoLeftVectors> 
                <, NoRightVectors> <, NoResidues>) 
                                                                   
 Parameter:

  A         -- a matrix (DOM_ARRAY, DOM_HFARRAY, or category Cat::Matrix)
  hard_soft -- Soft, SoftwareFloats, Hard, HardwareFloats  
         t  -- the domain type of the left/right eigenvectors
               U, V  returned by the call.
               Either DOM_ARRAY, DOM_HFARRAY, matrix or densematrix

Synopsis:

  The singular value decomposition of an (m x n) matrix A is
  a factorisation A = U D V', where
  --  U is a unitary (m x m) matrix,
  --  D is a sparse (m x n) matrix with real entries D[i,i]=d[i],
      i=1..p, p=min(m,n),
  --  V is a unitary (n x n) matrix.
  V' denotes the Hermitean transpose of V.
  The i.th column of U is an eigenvector of AA' associated with 
  the eigenvalue d[i]^2, where d[i]=0 for i>p.
  The i.th column of V is an eigenvector of A'A associated with 
  the eigenvalue d[i]^2, where d[i]=0 for i>p.
 
  numeric::singularvectors(A, Options); returns the list 
  [ U,d,V,resU,resV ], where                           
  -- U is the unitary (m x m) matrix of left singular vectors. 
  -- d=[ d[1],...,d[p] ] is a sorted list of real singular 
     values d[1] >= d[2] >= .. >= d[p] >= 0, p=min(m,n).
  -- V is a unitary (n x n) matrix of right singular vectors. 
     It is normalized such that, in each colum, the first
     element of absolute magnitude >= 10^(-DIGITS) is real
     and positive.
  -- resU = [ resU[1],..,resU[m] ] is a list of residues
     associated with the left singular vectors:
             resU[i] = < A'u.i , A'u.i > - d[i]^2,
     where u.i= i.th column of U and <.,.> is the usual complex
     Euclidean scalar product.
  -- resV = [ resV[1],..,resV[n] ] is a list of residues
     associated with the right singular vectors:
             resV[i] = < A v.i , A v.i > - d[i]^2,
     where v.i= i.th column of V.

  The list d represents the (m x n) matrix
              [ d[1]  0  ..  ]
              [   0  ..  ..  ]  
          D = [  ..  .. d[p] ]  
              [   0  ..   0  ]
              [  ..  ..  ..  ]
              [   0  ..   0  ]
  if p=min(m,n)=n, or, if p=min(m,n)=m,
              [ d[1]  0  ..  0 ..  0 ]
          D = [   0  ..  .. .. .. .. ] 
              [  ..  .. d[p] 0 ..  0 ]

  U,V are of domtype DOM_ARRAY.

  If only right/left singular vectors are required, then the
  options NoLeftVectors/NoRightVectors suppress computation of 
  U/V and the corresponding residues resU/resV. The returned
  values for these data are NIL.

  Computation of the residues resU/resV may be suppressed by
  the option NoResidues (or the equivalent NoErrors). In this 
  case, the returned values for resU/resV are NIL.
                                                                  
  Remark: Non-float numerical entries are allowed in the matrix.
          They will be evaluated to floats in the course of the 
          computation. Non-numeric symbolic entries will lead to 
          an error.      

  Remark: If no singular vectors are required, then
          the singular values obtained from
          numeric::singularvectors(A,NoLeftVectors,NoRightVectors);
          are identical to those computed by
          numeric::singularvalues(A);
 
  Remark: The left/right singular data may also be 
          computed via
           ([d2,U,errorsU]):= numeric::eigenvectors(AA');
          or
           ([d2,V,errorsV]):= numeric::eigenvectors(A'A);
          where d2= [ d[1]^2,d[2]^2,.. ].  However, in comparison 
          with the singular routines, for ill-conditioned matrices 
          about twice as many DIGITS are required in the eigen 
          routines to produce results of similar quality.
          
  Examples: 

  >>  m:=3: n:=2: A:=array(1..m,1..n,[[1,2*I ],[2,3],[3,4]]):
  >>  sv:=numeric::singularvectors:
  >> ([U,d,V,resU,resV]):=sv(A,NoResidues):
  >> ([U,d,V,resU,resV]):=sv(A,NoLeftVectors):
  >> ([U,d,V,resU,resV]):=sv(A,NoLeftVectors,NoResidues):
  >> ([U,d,V,resU,resV]):=sv(A,NoRightVectors):
  >> ([U,d,V,resU,resV]):=sv(A,NoRightVectors,NoResidues):
  >> ([U,d,V,resU,resV]):=sv(A,NoLeftVectors,NoRightVectors):
  >>
  >> ([U,d,V,resU,resV]):=sv(A):                                   
  >> U;
               array(1..3, 1..3,
                 (1, 1) =  0.115306134   + 0.2577906207 I,
                 (1, 2) = -0.5998682768  + 0.73999056 I,
                 (1, 3) = -0.07612303987 + 0.08382001921 I,
                 (2, 1) =  0.5600113945  - 0.04296510345 I,
                 (2, 2) =  0.0102273326  - 0.1333040615 I,
                 (2, 3) = -0.8074122747  - 0.1214581624 I,
                 (3, 1) =  0.7755694367  - 0.05728680461 I,
                 (3, 2) = -0.1863196488  - 0.1999560922 I,
                 (3, 3) =  0.5636491964  + 0.05303210188 I
               )
  >> d;
                        [6.411107219, 1.377571855]
  >> V;
   +-                                                               -+
   |          0.5556040651,         - 0.8263616549 - 0.09181796165 I |
   | 0.8263616549 - 0.09181796165 I,          0.5556040651           |
   +-                                                               -+
  >> resU;
            [6.938893904e-18, 8.741117365e-19, 8.008055265e-38]
  >> resV;
                    [2.083474391e-17, 4.415536526e-19]

See also:  numeric::eigenvalues, numeric::eigenvectors, 
           numeric::singularvalues, numeric::spectralRadius
                                                                  */
//-----------------------------------------------------------------

numeric::singularvectors := proc(A)
local useHardwareFloats, HardwareFloatsRequested, returnType,
      colrangechanged, rowrangechanged,
      originalcolrange, originalrowrange,
      nowarning, _norm,
      macheps0, result, m, n, mgtn, oldA, B, U, V,
      real, i, j, k, l,  d, e, s, ss, c, h , hh , f, g,
      eps, docancel, x, y, z, OK,
      withU, withV, withResidues, position, Y,
      resU,resV, r, reorder, reorderU, reorderV;
save DIGITS;
begin
    if args(0) < 1 then error("expecting at least one argument") end_if;
    if args(0) > 7 then error("expecting no more than 7 arguments") end_if;

    //----------------------------------------------
    // Check the matrix and set default return types
    //----------------------------------------------
    colrangechanged:= FALSE;
    rowrangechanged:= FALSE;

    if contains({DOM_ARRAY, DOM_HFARRAY}, domtype(args(1))) then
       if op(A, [0, 1]) <> 2 then
          error("first argument: expecting a 2-dimensional array or hfarray");
       end_if;
       if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
         colrangechanged:= TRUE;
         rowrangechanged:= TRUE;
         originalrowrange:= op(A, [0, 2]):
         originalcolrange:= op(A, [0, 3]):
         if domtype(A) = DOM_ARRAY then
           A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                  1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
         else
           A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                       1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]),
                       [op(A)]);
         end_if;
       end_if;
       [m, n]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
       returnType:= domtype(args(1));
    elif A::dom::hasProp(Cat::Matrix)=TRUE then
       [m, n]:= A::dom::matdim(A):
       returnType:= Dom::Matrix();
       if A::dom::constructor = Dom::DenseMatrix then
            returnType:= Dom::DenseMatrix();
       elif A::dom <> Dom::Matrix() then
            A:= array(1..m, 1..n, [op(A)]):
       end_if;
    else
       error("1st argument: expecting an array, an hfarray, ".
            "or a matrix of category 'Cat::Matrix'"):
    end_if;
    if not rowrangechanged then
       originalrowrange:= 1..m;
    end_if:
    if not colrangechanged then
       originalcolrange:= 1..n;
    end_if;

    //---------------------------------------------
    // Check the options
    //---------------------------------------------
    // set defaults
    if DIGITS <= 15 then
         useHardwareFloats:= TRUE;       // default
    else useHardwareFloats:= FALSE;      // default
    end_if;
    HardwareFloatsRequested:= FALSE;     
    if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
    end_if:
    nowarning:= FALSE;                   // default
    withU:=TRUE;                         // default
    withV:=TRUE;                         // default
    withResidues:=TRUE;                  // default

    for i from 2 to args(0) do
        case args(i)
        of Hard do
        of HardwareFloats do
                       useHardwareFloats:= TRUE:
                       HardwareFloatsRequested:= TRUE;
                       break;
        of Soft do
        of SoftwareFloats do
                       useHardwareFloats:= FALSE:
                       HardwareFloatsRequested:= FALSE;
                       break;
        of NoLeftVectors do
                       withU:= FALSE:
                       break;
        of NoRightVectors do
                       withV:= FALSE:
                       break;
        of NoErrors do
        of NoResidues do
                       withResidues:= FALSE;
                       break;
        of NoWarning do
                       nowarning:= TRUE;
                       break;
        otherwise
           if type(args(i)) = "_equal" and
              lhs(args(i)) = ReturnType then
                returnType:= rhs(args(i));
                if not contains({DOM_ARRAY, 
                                 DOM_HFARRAY, 
                                 Dom::DenseMatrix(), 
                                 Dom::Matrix()}, returnType) 
                   then
                   error("illegal return type ".expr2text(args(i)).
                         " specified. Choose between DOM_ARRAY, ".
                         "Dom_HFARRAY, Dom::Matrix() or Dom::DenseMatrix()"):
                end_if;
           else error("unknown option ".expr2text(args(i)));
           end_if;
        end_case;
    end_for;

    //--------------------------------
    // ----- use HardwareFloats ------
    //--------------------------------
    macheps0:= 10.0^(-DIGITS);
    case useHardwareFloats // use case because we want to use break below
    of TRUE do
       if DIGITS > 15 then
          if HardwareFloatsRequested and not nowarning then
             warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                     " cannot be achieved with the requested 'HardwareFloats'.".
                     " Using 'SoftwareFloats' instead.");
          end_if;
          useHardwareFloats:= FALSE;
          break;
       end_if;

       if not numeric::startHardwareFloats("singularvectors") then
          userinfo(1, "cannot start the hardware floats interface"):
          if HardwareFloatsRequested and not nowarning then
             userinfo(1, "cannot start the 'HardwareFloats' interface"):
             warning("cannot start the 'HardwareFloats' interface, ".
                     "using 'SoftwareFloats' instead");
          end_if;
          useHardwareFloats := FALSE;
          break;
       end_if;

       userinfo(1, "trying hardware floats"):

       if domtype(A) <> DOM_HFARRAY then
           assert(nops(A) = m*n);
           OK:= traperror((
                           B:= hfarray(1..m, 1..n, [op(A)]);
                          )) = 0;
       else
          B:= A;
          OK:= TRUE:
       end_if:
       if (not OK) or
          has(B, RD_NAN) or
          has(B, RD_INF) or
          has(B, RD_NINF) then
            userinfo(1, "'HardwareFloats' failed"):
            useHardwareFloats:= FALSE;
            break;
       end_if:

       // normalize to avoid underflow/overflows in the hfa module
       _norm:= norm(B):

       // special case: nothing to be done for the 0 matrix
       if iszero(_norm) then
          U:= hfarray(1..n, 1..n) + 1;
          d:= [float(0) $ n]; 
          V:= hfarray(1..m, 1..m) + 1; 
          resU:= [float(0) $ n]; 
          resV:= [float(0) $ m];
          if m < n then
             [U,V]:= [V,U];
             [resU,resV]:= [resV,resU];
          end_if:
          break: // proceed to postprocessing
       end_if:
       B:= B/_norm:

       if traperror((
             result:= hfa::singularvectors(B, withU, withV, withResidues)
           )) = 0 then
         if has(result, RD_NAN) or
            has(result, RD_INF) or
            has(result, RD_NINF) then
               userinfo(1, "'HardwareFloats' failed"):
               useHardwareFloats:= FALSE;
               break;
         end_if:

         //------------------------------
         // The matrix of left eigenvectors
         //------------------------------
         U:= result[1]: // DOM_HFARRAY
         //------------------------------
         // d is a List of length min(m, n)
         //------------------------------
         d:= map(result[2], _mult, _norm); // DOM_LIST
         //------------------------------
         // The matrix of right eigenvectors
         //------------------------------
         V:= result[3]: // DOM_HFARRAY
         //------------------------------
         // The left residues:
         //------------------------------
         resU:= result[4];
         if resU <> NIL then
            resU:= map(resU, _mult, _norm^2); // DOM_LIST
         end_if:
         //------------------------------
         // The right residues:
         //------------------------------
         resV:= result[5];
         if resV <> NIL then
            resV:= map(resV, _mult, _norm^2); // DOM_LIST
         end_if:

         //-----------------------------------
         // sort singular values and singular
         // vectors according to numeric::sort
         //-----------------------------------
         d:= numeric::sort(d, TRUE);
         reorder:= map(d, op, 2);
         d:= map(d, op, 1);

         if withU then
            if resU <> NIL then
               for i from 1 to min(m, n) do
                   resU[i]:= resU[reorder[i]];
               end_for:
            end_if:
            reorderU:= reorder.[i $ i =  min(m, n) +1 .. m];
            // hfa::reorderColumns does a **referenced** change!!
            hfa::reorderColumns(U, reorderU);
         end_if;
         if withV then
            if resV <> NIL then
               for i from 1 to min(m, n) do
                   resV[i]:= resV[reorder[i]];
               end_for:
            end_if:
            reorderV:= reorder.[i $ i =  min(m, n) +1 .. n];
            // hfa::reorderColumns does a **referenced** change!!
            hfa::reorderColumns(V, reorderV);
         end_if:

         break;

       else // proceed to the branch useHardwareFloats = FALSE
         userinfo(1, "'HardwareFloats' failed"):
         useHardwareFloats:= FALSE;
         break;
       end_if;

    end_case; // 
    //--------------------------------
    // end of useHardwareFloats = TRUE
    //--------------------------------


    // use case instead of if useHardwareFloats = FALSE,
    // because we want to use break
    case useHardwareFloats
    of FALSE do
       //---------------------------------
       //---- use SoftwareFloats ---------
       //---------------------------------
       if HardwareFloatsRequested and not nowarning then
          warning("'HardwareFloats' failed, using 'SoftwareFloats' instead"):
       end_if;

      userinfo(1, "using software floats"):

      // convert to domtype DOM_ARRAY for speed 
      if A::dom::hasProp(Cat::Matrix)=TRUE then
         A:= expr(A);
      end_if; 

      DIGITS := DIGITS + max(10, trunc(min(m,n)/10));

      if domtype(A) <> DOM_HFARRAY then
         A:= map(A, float):
      end_if:

      if map({op(A)}, domtype) minus {DOM_FLOAT, DOM_COMPLEX} <> {} 
         then error("non-numeric entries not handled");
      end_if:

      // Indexed reading and writing for arrays is
      // as fast as lists. Hfarrays, however, are
      // significantly slower. So: use list or array
      if domtype(A) = DOM_HFARRAY then
         assert(nops(A) = m*n);
         A:= array(1..m, 1..n, [op(A)]);
      end_if;

      if domtype(A) <> DOM_ARRAY then
         error("the matrix must be an array, an hfarray, ".
               "or of category 'Cat::Matrix'");
      end_if:

      // special case: nothing to be done for the 0 matrix
      if map({op(A)},iszero) = {TRUE} then
          U:= array(1..m, 1..m, [[0.0 $ i-1, 1.0, (0.0 $ m - i)] $ i = 1..m]): 
          d:= [float(0) $ n]; 
          V:= array(1..n, 1..n, [[0.0 $ i-1, 1.0, (0.0 $ n - i)] $ i = 1..n]): 
          resU:= [float(0) $ m]: 
          resV:= [float(0) $ n]:
          if m < n then
             [U,V]:= [V,U];
             [resU,resV]:= [resV,resU];
          end_if:
          break: // proceed to postprocessing
      end_if;

      real:= bool(map({op(A)},iszero@Im) = {TRUE}):

      oldA:=A:

    if m >= n then 
         mgtn:= TRUE;
    else mgtn:= FALSE;
         ([m,n]):=([n,m]);
         ([withU,withV]):=([withV,withU]);
         A:= array(1..m,1..n);
         if real then ((A[i,j]:=oldA[j,i] )          $ i=1..m)$ j=1..n;
                 else ((A[i,j]:=conjugate(oldA[j,i]))$ i=1..m)$ j=1..n;
         end_if;
    end_if;

    d:=[0 $ n  ];  // initialize list to store diagonal       
    e:=[0 $ n-1];  // initialize list to store super diagonal 
    h:=[0 $ n];    // initialize list to store left factors   
    hh:=[0 $ n];   // initialize list to store right factors  
    // Householder transformation to reduce matrix to real  
    // bi-diagonal form                                     
    for j from 1 to n do 
        // Householder step from the left to obtain upper   
        // triangular form. Householder data are stored in  
        // lower triangular part of A.                      
        s:= _plus(specfunc::abs(A[i,j])^2 $ i=j+1..m);
        if iszero(s) then d[j]:=A[j,j];
        else ss:=s+specfunc::abs(A[j,j])^2;
             s:=specfunc::sqrt(ss):
             h[j] := 1 / (ss + specfunc::abs(A[j,j]) * s);
             if not iszero(A[j,j]) then s:=sign(A[j,j])*s;end_if:
             A[j,j]:=s+A[j,j]; d[j]:=-s;
             for i from j+1 to n do
                 if real then   s:=_plus(A[k,j] *A[k,i]$ k=j..m) ;
                 else s:=_plus(conjugate(A[k,j])*A[k,i]$ k=j..m) ;
                 end_if;
                 f:=s*h[j];
                 (A[k,i]:=A[k,i]-f*A[k,j]) $ k=j..m ;
             end_for;
        end_if;
        // Householder step from the right to obtain upper  
        // bidiagonal form. Householder data are stored in  
        // strictly upper triangular part of A.             
        s:= _plus(specfunc::abs(A[j,i])^2 $ i=j+2..n);
        if iszero(s) then if j<n then e[j]:=A[j,j+1]; end_if;
        else ss:=s+specfunc::abs(A[j,j+1])^2;
             s:=specfunc::sqrt(ss):
             hh[j] := 1 / (ss + specfunc::abs(A[j,j+1]) * s);
             if not iszero(A[j,j+1]) then s:=sign(A[j,j+1])*s; end_if:
             A[j,j+1]:=s+A[j,j+1]; e[j]:=-s; 
             for i from j+1 to m do
                 if real then   s:=_plus(A[j,k] *A[i,k] $ k=j+1..n) ;
                 else s:=_plus(conjugate(A[j,k])*A[i,k] $ k=j+1..n) ;
                 end_if;
                 f:=s*hh[j];
                 (A[i,k]:=A[i,k]-f*A[j,k]) $ k=j+1..n ;
             end_for;
        end_if;
    end_for;

    // now A contains in its lower/strictly upper part the 
    // Householder data of the left/right transformation.  
    // The resulting bi-diagonal matrix                    
    //    [ d[1] e[1]         ]                            
    //    [      d[2] e[2]    ]  is stored in d and e.     
    //    [           ..   .. ]                            

if withV then
    V:=array(1..n,1..n,[[0$n]$n]): (V[j,j]:=1)$ j=1..n;
    for j from n-1 downto 1 do     
        if not iszero(hh[j]) then
           for i from j+2 to n do
               s:=_plus(A[j,k]*V[k,i] $ k=j+2..n) ;
               f:=s*hh[j];
               if real then (V[k,i]:=V[k,i]-f*A[j,k])$ k=j+1..n;
               else (V[k,i]:=V[k,i]-f*conjugate(A[j,k]))$ k=j+1..n;
               end_if;
           end_for;
           if real then (V[k,j+1]:= A[j,k]/e[j]) $ k=j+1..n ; 
           else (V[k,j+1]:=conjugate(A[j,k]/e[j])) $ k=j+1..n ; 
           end_if;
           V[j+1,j+1]:=V[j+1,j+1]+1;
        end_if;
    end_for;
end_if;
 
if withU then
    U:=array(1..m,1..m,[[0$m]$m]); (U[j,j]:=1)$ j=1..m;
    for j from n downto 1 do    
      if not iszero(h[j]) then 
         for i from j+1 to m do
            if real then s:=_plus(A[k,j]*U[k,i] $ k=j..m) ;
            else s:=_plus(conjugate(A[k,j])*U[k,i] $ k=j..m) ;
            end_if;
            f:=s*h[j];
            (U[k,i]:=U[k,i]-f*A[k,j]) $ k=j..m ;
         end_for;
        (U[k,j]:=A[k,j]/d[j]) $ k=j..m;
        U[j,j]:=U[j,j]+1;
      end_if;
    end_for;
end_if;

   /* now [d[1],..,d[n]] and [e[1],..,e[n-1]] are the diagonal and
     superdiagonal of a complex bidiagonal matrix. Apply further
     transformation with phase matrices 
        [beta1     ] [d1 e1      ] [conjug(alpha1)         ]
        [   beta2  ] [   d2 e2   ] [       conjug(alpha2)  ]
        [       ...] [      .. ..] [                   ... ]
     (with alpha1=1) to obtain real bidiagonal form.
     Update matrices U and V.
   */

   if not real then
     if iszero(d[1]) then f:=1
                     else f:=sign(d[1]):    // f= beta.1 
                          d[1]:=specfunc::abs(d[1]); 
                          if withU then
                            (U[i,1]:=U[i,1]*f) $ i=1..m;
                          end_if;
     end_if;
     for j from 2 to n do
         if iszero(e[j-1]) then f:=1;       
                           else f:=f/sign(e[j-1]); // f = alpha.j 
                                e[j-1]:=specfunc::abs(e[j-1]);
                                if withV then
                                  (V[i,j]:=V[i,j]*f) $ i=1..n;
                                end_if;
         end_if;
         if iszero(d[j]) then f:=1;  
                         else f:=f*sign(d[j]);     // f = beta.j 
                              d[j]:=specfunc::abs(d[j]):
                              if withU then
                                 (U[i,j]:=U[i,j]*f) $ i=1..m;
                              end_if;
         end_if;
     end_for;
  end_if;

// ------ now QR iteration to diagonalize bi-diagonal matrix ---- 

  e:=[0,op(e)];
  eps:=max(specfunc::abs(d[i])+specfunc::abs(e[i]) $ i=1..n)/10^DIGITS;
  for k from n downto 1 do
     repeat
       docancel:=TRUE;
       for l from k downto 1 do
          if specfunc::abs(e[l])<=eps then docancel:=FALSE; break end_if;
          if specfunc::abs(d[l-1]) <= eps then break end_if;
       end_for;
 
       if docancel then // cancellation of e[l] for l>1 
          c:=0;s:=1;
          for i from l to k do
              f:=s*e[i];e[i]:=c*e[i]:
              if specfunc::abs(f)<=eps then break; end_if;
              g:=d[i]; h:=specfunc::sqrt(f^2+g^2); d[i]:=h; c:=g/h; s:=-f/h;
 
              if withU then
                for j from 1 to m do    // update matrix U 
                   y:=U[j,l-1]; z:=U[j,i]:
                   U[j,l-1]:= y*c+z*s; U[j,i]:=-y*s+z*c;
                end_for;
              end_if;
 
          end_for;
       end_if;          // end of cancellation 

       z:=d[k]; if l=k then break; end_if;

       // compute shift 
       x:=d[l]; y:=d[k-1]; g:=e[k-1]; h:=e[k];
       f:= ( (y-z)*(y+z)+(g-h)*(g+h) ) / (2*h*y);
       g:=specfunc::sqrt(float(1)+f^2); 
       if f<0 then f:= ( (x-z)*(x+z)+h*( y/(f-g)-h )  )/x ;
              else f:= ( (x-z)*(x+z)+h*( y/(f+g)-h )  )/x ;
       end_if;
 
       // do QR step 
       c:=1;s:=1;
       for i from l+1 to k do
          g:=e[i]; y:=d[i]: h:=s*g; g:=c*g;
          z:=specfunc::sqrt(f^2+h^2); 
          e[i-1]:=z; 
          if iszero(z) then
            [c, s]:= [float(1), float(0)];
          else
             c:=f/z;
             s:=h/z;
          end_if;
          f:=x*c+g*s; g:=-x*s +g*c; h:=y*s; y:=y*c;
 
          if withV then
            for j from 1 to n do
               x:=V[j,i-1];z:=V[j,i];V[j,i-1]:=x*c+z*s;V[j,i]:=-x*s+z*c;
            end_for;
          end_if;

          z:=specfunc::sqrt(f^2+h^2); 
          d[i-1]:=z; 
          if iszero(z) then
             [c, s]:= [float(1), float(0)];
          else
             c:= f/z; 
             s:= h/z;
          end_if;
          f:=c*g+s*y; x:=-s*g+c*y;

          if withU then
            for j from 1 to m do
               y:=U[j,i-1];
               z:=U[j,i];
               U[j,i-1]:=y*c+z*s;
               U[j,i]:=-y*s+z*c;
            end_for;
          end_if;

       end_for;
       e[l]:=0; e[k]:=f; d[k]:=x;
     until FALSE 
     end_repeat;
     if z<0 then 
         d[k]:=-z;
         if withV then 
            (V[j,k]:=-V[j,k]) $ j=1..n; 
         end_if;
     end_if;
  end_for;

// ------ sort singular values from large to small ------ 

  position:=[i$ i=1..n];
  for j from 2 to n do              // search for sorted position  
      z:=d[j];                      // of orig. eigenvalue d[j]    
      for i from j downto 2 do      // which becomes d[position[j]]
         if d[i-1] >= z then break end_if:
         d[i]:=d[i-1]: 
         position[i]:=position[i-1]:
      end_for;
      d[i]:=z;
      position[i]:=j:
  end_for:   

if withU then
  Y:= U: 
  for j from 1 to n do
    if j<>position[j] then 
       (U[i,j]:=Y[i,position[j]])$ i=1..m 
    end_if
  end_for;
end_if;

if withV then
  Y:=V: 
  for j from 1 to n do
    if j<>position[j] then 
       (V[i,j]:=Y[i,position[j]])$ i=1..n 
    end_if
  end_for;
end_if;

/*
  make first element > macheps0 in each column of V 
  positive real by U-> UP, V->VP with diagonal phase 
  matrix P. Note A = U D V^T = U*P * D * (V*P)^T.
*/

if not mgtn then 
  ([m,n]):=([n,m]);
  if withU and withV then ([U,V]):=([V,U]); end_if;
  if withU and not withV then V := U; end_if:
  if withV and not withU then U := V; end_if:
  ([withU,withV]):=([withV,withU]);
end_if;

if withV then
   for j from 1 to n do
       f:= float(1):
       for k from 1 to n do
           f:= specfunc::abs(V[k, j]);
           if f > macheps0 then
              f:= f/V[k, j];
              break;
           end_if;
       end_for;
       if float(f) <> float(1) then
         (V[i,j]:=V[i,j]*f) $ i=1..k-1;
          V[k,j]:=specfunc::abs(V[k,j]);
         (V[i,j]:=V[i,j]*f) $ i=k+1..n;
         if withU and j<=m then
             (U[i,j]:=U[i,j]*f) $ i=1..m;
         end_if;
       end_if;
   end_for;
end_if;

/* compute residues
      norm( A^T u.j )^2 - d[j]^2 of left sing. vectors u.j:=column(U,j)
  and norm( A v.j )^2 - d[j]^2  of right sing. vectors v.j:=column(V,j)
*/

  if withResidues then
   if withU then
     resU:=[0 $ j=1..m];
     for j from 1 to m do
        for i from 1 to n do
           if real then r[i]:=_plus(oldA[k,i]*U[k,j] $ k=1..m);
           else r[i]:=_plus(conjugate(oldA[k,i])*U[k,j] $ k=1..m);
           end_if;
        end_for;
        if j<=min(m,n) then z:=d[j]^2 else z:=0 end_if;
        if real then   resU[j]:=specfunc::abs(_plus(r[i] *r[i]$ i=1..n)-z);
        else resU[j]:=specfunc::abs(_plus(conjugate(r[i])*r[i]$ i=1..n)-z);
        end_if;
     end_for;
   end_if;
   if withV then
     resV:=[0 $ j=1..n];
     for j from 1 to n do
        for i from 1 to m do
           r[i]:=_plus( oldA[i,k]*V[k,j] $ k=1..n);
        end_for;
        if j<=min(m,n) then z:=d[j]^2 else z:=0; end_if;
        if real then   resV[j]:=specfunc::abs(_plus(r[i] *r[i]$ i=1..m)-z);
        else resV[j]:=specfunc::abs(_plus(conjugate(r[i])*r[i]$ i=1..m)-z);
        end_if;
     end_for;
   end_if;
  end_if;

/*
  d:= numeric::sort(d, TRUE);
  reorder:= map(d, op, 2);
  d:= map(d, op, 1);
*/
end_case: // of useHardwareFloats = FALSE


   //----------------------
   // prepare return values
   //----------------------
   d:= map(d, float):
   if withU then
        U:= map(U, float);
   else U:= NIL;
        resU:= NIL;
   end_if;
   if withV then 
        V:= map(V, float);
   else V:= NIL;
        resV:= NIL;
   end_if;
   if not withResidues then
      resU:= NIL;
      resV:= NIL;
   end_if;

   if withU or withV then
      if domtype(U) = returnType and
         domtype(V) = returnType and
         (not rowrangechanged) and
         (not colrangechanged) then
         // nothing to be done
      else
         case returnType
         of DOM_ARRAY do
            if withU then
               U:= array(originalrowrange, originalrowrange, [op(U)]);
            end_if;
            if withV then
               V:= array(originalcolrange, originalcolrange, [op(V)]);
            end_if;
            break;
         of DOM_HFARRAY do
            if withU then
               U:= hfarray(originalrowrange, originalrowrange, [op(U)]);
            end_if;
            if withV then
               V:= hfarray(originalcolrange, originalcolrange, [op(V)]);
            end_if;
            break;
         of Dom::DenseMatrix() do
            if withU then
               U:= map(U, float);
               U:= (Dom::DenseMatrix())::create(U):
            end_if;
            if withV then
               V:= map(V, float);
               V:= (Dom::DenseMatrix())::create(V):
            end_if;
            break;
         of Dom::Matrix() do
            if withU then
               U:= (Dom::Matrix())::create(U):
            end_if;
            if withV then
               V:= (Dom::Matrix())::create(V):
            end_if;
            break;
         otherwise
            error("illegal returnType"):
         end_case;
      end_if;
  end_if:

  return([U, d, V, resU, resV]) 

end_proc:

//---------------------------------------------
// Allow alternative calls (svd is the standard
// name for a singular value decomposition)
//---------------------------------------------
numeric::svd:= numeric::singularvectors:
numeric::singularVectors:= numeric::singularvectors:
//--------  end of procedure singularvectors ---------------
