/*
  numeric::sort -- sort a list of numerical values

Call:  numeric::sort(list_of_values)

Input is a list of real or complex values,
output is a sorted list of float values.

Sorting criterion:
  1) From large real parts to small real parts
  2) If tie, then from large abs(imaginary part) 
     to small abs(imaginary part)
  3) If tie (i.e., a complex conjugate pair)
     then positive imaginary part first

Remark: if any second argument is given,
   the return type is changed to
    [[vi1, i1], [vi2, i2], ...],
   where [i1, i2, ...] is the permutation of
   the [1, 2, 3, ...] indicating the sorting (i.e., 
   the i1-th element of the input list is now the 1st element,
   the i2-th element of the input list is now the 2nd element,
   etc.

This routine is used to sort the output of
  numeric::eigenvalues,
  numeric::eigenvectors, // version with 2 arguments
  numeric::polyroots,
  numeric::polyroots2,
  numeric::singularvalues
  numeric::singularvectors // version with 2 arguments


Examples:

  >> numeric::sort([PI, 0, PI-I, PI+I, 2, sqrt(2)]);

     [0.0, 1.414213562, 2.0, 3.141592653, 3.141592653 + 1.0 I,
         3.141592653 - 1.0 I]

  >> numeric::sort([-2.0, 2+10*I, 2-12*I, 0.3*I, 0.3]);

            [-2.0, 0.3 I, 0.3, 2.0 + 10.0 I, 2.0 - 12.0 I]
*/

numeric::sort:= proc(d) 
local extended, i;
begin
    extended:= bool(args(0) = 2);
    if args(0) < 1 then error("expecting at least one argument") end_if;
    if args(0) > 2 then error("expecting no more than 2 arguments") end_if;
    if domtype(d)<> DOM_LIST then error("expecting a list") end_if;
    // convert to float
    d:= map(d, float);
    if extended then
       d:= [[d[i], i] $ i = 1 .. nops(d)];
    end_if;
    // sort real floats by kernel sort, watch out for future float(0)=0
    if has(d, RD_NAN) then
       if extended then
         return(sort(d, (x, y) -> (x[1] > y[1])));
       else
         return(sort(d));
       end_if:
    end_if:
    if {op(map(d,domtype))} minus {DOM_FLOAT, DOM_INT} = {} then 
       return(sort(d, (x, y) -> (x > y)));
    end_if;
    // check for symbolic objects
    if extended then
       if {op(map(d,domtype@op, 1))} minus 
          {DOM_INT, DOM_RAT, DOM_FLOAT, DOM_COMPLEX, DOM_INTERVAL} <> {}
          then error("cannot sort symbolic objects") 
       end_if;
    else
       if {op(map(d,domtype))} minus 
          {DOM_INT, DOM_RAT, DOM_FLOAT, DOM_COMPLEX, DOM_INTERVAL} <> {}
          then error("cannot sort symbolic objects") 
       end_if;
    end_if;
    // final sorting
    if extended then
    sort(d, proc(a,b) local absa, absb; begin
            a:= a[1];
            b:= b[1];
            if domtype(a)=DOM_INTERVAL then
               a:= DOM_INTERVAL::center(a);
            end_if;
            if domtype(b)=DOM_INTERVAL then
               b:= DOM_INTERVAL::center(b);
            end_if;
            if domtype(a)=DOM_COMPLEX or
               domtype(b)=DOM_COMPLEX 
            then if op(a,1) > op(b,1) then return(TRUE) end_if;
                 if op(a,1) = op(b,1) then
                    absa:= specfunc::abs(a);
                    absb:= specfunc::abs(b);
                    if absa > absb then return(TRUE) end_if:
                    if absa = absb and Im(a) > Im(b) 
                       then return(TRUE)
                    end_if:
                 end_if;
            else if a > b then return(TRUE) end_if;
            end_if;
            return(FALSE)
         end_proc);
    else
    sort(d, proc(a,b) local absa, absb; begin
            if domtype(a)=DOM_INTERVAL then
               a:= DOM_INTERVAL::center(a);
            end_if;
            if domtype(b)=DOM_INTERVAL then
               b:= DOM_INTERVAL::center(b);
            end_if;
            if domtype(a)=DOM_COMPLEX or
               domtype(b)=DOM_COMPLEX 
            then if op(a,1) > op(b,1) then return(TRUE) end_if;
                 if op(a,1) = op(b,1) then
                    absa:= specfunc::abs(a);
                    absb:= specfunc::abs(b);
                    if absa > absb then return(TRUE) end_if:
                    if absa = absb and Im(a) > Im(b) 
                       then return(TRUE)
                    end_if:
                 end_if;
            else if a > b then return(TRUE) end_if;
            end_if;
            return(FALSE)
         end_proc);
    end_if:
end_proc:

