//----------------------------------------------------------------------
// spectralradius(A, x0) -- compute the spectral radius of a matrix
//      by the vector iteration x.i = (A^i x0)/norm(A^i x0) converging
//      to the eigenspace associated with the spectral radius
//                                                                      
// Calls:  numeric::spectralradius(A, <x0>, <maxit>, <mode>,
//                                 <NoWarning>, <ReturnType = t>)
//
// Parameter:
//        A  -- a square matrix (DOM_ARRAY, DOM_HFARRAY or Cat::Matrix)                      
//       x0  -- a start vector. If no start vector is provided by the
//              user, some random start vector is used internally.
//    maxit  -- maximal number of iterations to be used.
//              The default is 1000
//     mode  -- one of the flags HardwareFloats, Hard, SoftwareFloats, Soft
//        t  -- the domain type of the eigenvector x returned by the call.
//              Either DOM_LIST, DOM_ARRAY, DOM_HFARRAY, matrix or densematrix
//                                                                      
// Return Value: a list [rho , x, residue]                            
//                                                                      
//         Here rho is an approximation of the absolutely largest       
//         eigenvalue (i.e., abs(rho) is the spectral radius).          
//         For hermitean A the Rayleigh quotient rho = <x,Ax> is        
//         used with a normalized vector x:=A^i*x0/ norm(A^i x0),       
//         i some suitable power. For non hermitean A rho is            
//         computed as absolute largest value of the form               
//         rho = (Ax)[j]/x[j].                                          
//                                                                      
//         x = A^i*x0 / norm(A^i*x0) is the corresponding normalized    
//         eigenvector (array(1..n, [..])                               
//                                                                      
//         The  residue := norm(Ax - rho x)/norm(x) provides an         
//         error estimate. For hermitean matrices this is a             
//         rigorous upper bound for the error abs(rho-rho_exact).       
//                                                                      
// Warning: the iteration is terminated, if the approximations of the   
//          absolutely largest eigenvalue becomes stationary within     
//          the relative precision given by DIGITS.                     
//          The iteration may not converge, if the absolute value of    
//          the eigenvalue defining the spectral radius is not well     
//          separated from the absolute value of the other eigenvalues. 
//          In this case the iteration is terminated after              
//          maxit iterations with a warning.                               
//                                                                      
// The routine spectralradius is an ad-hoc (and not very sophisticated)  
// implementation of the power method to compute the largest eigenvalue 
// and a corresponding eigenvector.                                     
//                                                                      
// Example:                                                             
//                                                                      
//   x0:=array(1..2, [1$n]):                                            
//   data:= spectralradius(linalg::hilbert(n),x0,1000);                 
//   --            +-                          -+                 --    
//   | 1.267591879,| 0.8816735651, 0.4718598569 |, 6.931409507e-12 |    
//   --            +-                          -+                 --    
//                                                                      
//   eigenvalue:=op(data,1);                                            
//                              1.267591879                             
//   eigenvector:=op(data,2);                                           
//                            +-                          -+            
//                            | 0.8816735651, 0.4718598569 |            
//                            +-                          -+            
//   error:=op(data,3);                                                 
//                              6.931409507e-12                         
//                                                                      
//----------------------------------------------------------------------

numeric::spectralradius := proc(A)
local n, oldDIGITS, hermitean, iter, nnorm, maxit,
      useHardwareFloats, HardwareFloatsRequested,
      nowarning, returnType, returnTypes, OK,
      tmp, x0, x, i, Ax, j, rho, residue, 
      rangechanged, originalrange;
save DIGITS;
begin 
   maxit:= 1000;                  // default

   if args(0) < 1 then error("expecting at least one argument") end_if;

   //---------------------
   // check the matrix
   //---------------------
   // set the return type here, because matrices 
   // will change to arrays via A:= expr(A)
   returnType:= A::dom:
   if A::dom::hasProp(Cat::Matrix) = TRUE then 
      if A::dom::constructor = Dom::DenseMatrix then
         returnType:= Dom::DenseMatrix(): 
      else
         returnType:= Dom::Matrix(): 
      end_if:
      A:= expr(A) 
   end_if;
   if domtype(A) <> DOM_ARRAY and
      domtype(A) <> DOM_HFARRAY then
      error("the matrix must be an array, an hfarray, or of category 'Cat::Matrix'");
   end_if:
   if op(A,[0,1]) <> 2 then 
      error("the first argument is not a matrix"):
   end_if:
   if op(A, [0, 2, 1]) <> 1 or 
      op(A, [0, 3, 1]) <> 1 then
       if domtype(A) = DOM_ARRAY then
         A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
         A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                     1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]), [op(A)]);
       end_if:
   end_if;
   n:= op(A,[0,2,2]):
   if n <> op(A,[0,3,2]) then error("the first argument is not a square matrix"): end_if:

   //-------------------------------------
   // create a start vector, if necessary
   //-------------------------------------
   x0:= array(1..n, [frandom() $ k = 1..n]);
   // the default returnType = A::dom was set above

   if DIGITS <= 15 then
        useHardwareFloats:= TRUE;    // default
   else useHardwareFloats:= FALSE;   // default
   end_if;
   HardwareFloatsRequested:= FALSE;
   if domtype(A) = DOM_HFARRAY then
      useHardwareFloats:= TRUE;
      HardwareFloatsRequested:= TRUE; // implicit request
   end_if:
   nowarning:= FALSE;

   // The legal returnTypes:
   returnTypes:= {DOM_ARRAY, DOM_HFARRAY, DOM_LIST,
                  Dom::DenseMatrix(), Dom::Matrix()};

   //-----------------
   // scan for options
   //-----------------
   for i from 2 to args(0) do
      if contains(returnTypes, domtype(args(i))) then
         // This is the start vector. Overwrite the
         // random start vector set above
         x0:= args(i);
         returnType:= domtype(x0):   // default
         next;
      end_if; 
      case args(i)
      of Hard do
      of HardwareFloats do
         HardwareFloatsRequested:= TRUE;
         useHardwareFloats:= TRUE:
         break;
      of Soft do
      of SoftwareFloats do
         useHardwareFloats:= FALSE:
         HardwareFloatsRequested:= FALSE;
         break;
      of NoWarning do
         nowarning:= TRUE;
         break;
      otherwise
         if domtype(args(i)) = DOM_INT then
            maxit:= args(i);
            if maxit <= 0 then
              error("the maximal number of iterations must be specified ".
                "by a positive integer. Got: ".expr2text(maxit));
            end_if;
         elif type(args(i)) = "_equal" and
            lhs(args(i)) = ReturnType then
              returnType:= rhs(args(i));
              if not contains(returnTypes, returnType) then
                 error("illegal return type ".expr2text(args(i)).
                       " specified. Choose between DOM_ARRAY, DOM_HFARRAY, DOM_LIST, ".
                       "Dom::Matrix() or Dom::DenseMatrix()"):
              end_if;
         else error("unknown option ".expr2text(args(i)));
         end_if;
      end_case;
   end_for:

   // introduce some guard digits:
   oldDIGITS := DIGITS: 
   if not useHardwareFloats then
      DIGITS := DIGITS + trunc(n/10):
   end_if:
   
   if domtype(A) = DOM_HFARRAY then
     hermitean:= bool(A = hfa::conjugate(hfa::transpose(A)));
   else
     A:=map(A,float):
     if {op(map(A,domtype))} minus {DOM_FLOAT, DOM_COMPLEX} <> {}  
        then error("matrices with symbolic entries are not handled"); 
     end_if: 
     A:=subs(A,float(0)=0):
     hermitean:=TRUE:
     for i from 1 to n do
       for j from 1 to i do
         if A[i,j] <> conjugate(A[j,i]) then 
           hermitean:= FALSE;
           break;
        end_if:
       end_for:
       if not hermitean then 
          break; 
       end_if:
     end_for:
   end_if:

   if domtype(x0) <> DOM_HFARRAY then
      x0:= map(x0, float);
   end_if;

   rangechanged:= FALSE;

   case domtype(x0)
   of DOM_ARRAY do
   of DOM_HFARRAY do
      if op(x0, [0,1]) > 2 then 
         error("the start vector must be a 1- or 2-dimensional array or hfarray")
      end_if;
      if nops(x0) <> n then
         error("dimensions of matrix and start vector do not match");
      end_if;
      if op(x0, [0, 1]) = 2 then
         if domtype(x0) = DOM_ARRAY then
            x0:= array(1..n, [op(x0)]);
         elif domtype(x0) = DOM_HFARRAY then
            x0:= hfarray(1..n, [op(x0)]);
         else
            error("unexpected case");
         end_if;
      end_if:
      // is the index range = 1 .. n ?
      // If not, then reformat:
      if {op(x0, [0, i, 1]) $ i = 2 .. op(x0, [0, 1]) + 1} <> {1} then
         rangechanged:= TRUE;
         originalrange:= op(x0, [0, 2]);
         if domtype(x0) = DOM_ARRAY then
           x0:= array(1..n, [op(x0)]);
         elif domtype(x0) = DOM_HFARRAY then
           x0:= hfarray(1..n, [op(x0)]);
         end_if:
      end_if;
      break;
   of DOM_LIST do
      if nops(x0) <> n then
         error("dimensions of matrix and start vector do not match");
      end_if;
      if {op(map(x0,domtype))} minus {DOM_FLOAT, DOM_COMPLEX} <> {}  
        then error("start vectors with symbolic entries are not handled"):
      end_if: 
      x0:= array(1..n, x0);
      break;
   otherwise 
      if x0::dom::hasProp(Cat::Matrix) = TRUE then
         x0:= expr(x0):
         if nops(x0) <> n then
            error("dimensions of matrix and start vector do not match");
         end_if;
         if {op(map(x0,domtype))} minus {DOM_FLOAT, DOM_COMPLEX} <> {}  
            then error("start vectors with symbolic entries are not handled"):
         end_if: 
         x0:= array(1..n, [op(x0)]);
      else
        error("wrong type of start vector") 
      end_if:
   end_case;

   if {op(x0)}={float(0)} then
      error("the start vector must not vanish");
   end_if;

   //----------------------------------------------------
   // now, the work starts:
   //----------------------------------------------------

   //--------------------------------
   // ----- use HardwareFloats ------
   //--------------------------------
   case useHardwareFloats
   of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested and
           not nowarning then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'."):
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     // there is no hfa::spectralradius. Check that the hfa module
     // is operational by querying the existence of hfa::transpose
     if not numeric::startHardwareFloats("transpose") then
        userinfo(1, "cannot start the 'HardwareFloats' interface"):
        if HardwareFloatsRequested and
           not nowarning then
           warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
        end_if;
        useHardwareFloats := FALSE;
        break;
     end_if;

     userinfo(1, "trying hardware floats"):

     if domtype(A) <> DOM_HFARRAY then
        OK:= traperror((
               A:= hfarray(1..n, 1..n, [op(A)]);
             )) = 0;
     else
        OK:= TRUE:
     end_if:
     if (not OK) or
        has(A, RD_NAN) or
        has(A, RD_INF) or
        has(A, RD_NINF) then
          userinfo(1, "'HardwareFloats' failed"):
          useHardwareFloats:= FALSE;
          break;
     end_if:

     if domtype(x0) <> DOM_HFARRAY or
        op(x0, [0, 1]) < 2 then
        OK:= traperror((
               x0:= hfarray(1..n, 1..1, [op(x0)]);
             )) = 0:
     end_if;
     if (not OK) or
        has(x0, RD_NAN) or
        has(x0, RD_INF) or
        has(x0, RD_NINF) then
          userinfo(1, "'HardwareFloats' failed"):
          useHardwareFloats:= FALSE;
          break;
     end_if;

     x:= x0;
     Ax:= x0;

     for iter from 1 to maxit do 
       tmp:= hfa::abs(Ax):
       tmp:= hfa::transpose(tmp) * tmp;
       nnorm:= (tmp[1, 1])^(1/2):
       if iszero(nnorm) then
          // put the test 'A = zero-matrix ?' here. This is an exceptional
          // case and should only be tested when necessary:
          tmp:= hfa::abs(A):
          if iszero(max({op(tmp)})) then
            rho:= float(0);
            residue:= float(0);
            break;
          else
            error("bad start vector. The ".output::ordinal(iter-1).
                  " iterate is the zero vector"):
          end_if;
       end_if;
       x:= Ax/nnorm:
       Ax:= A*x;
       //---------------------------------------------------------
       // Ax is the next iterate for the eigenvector approximation.
       // Extract an approximation of the eigenvalue from the
       // approximate eigenvector
       //---------------------------------------------------------
       if hermitean then
          // use Rayleigh quotient rho=<x,Ax> to extract eigenvalue 
          rho:= hfa::transpose(hfa::abs(x))*hfa::abs(Ax);
          rho:= rho[1,1];
       else 
          rho:= 0; // use rho=max((Ax)[i]/x[i]) to extract eigenvalue 
          for i from 1 to n do 
            if (not iszero(x[i, 1])) and 
               specfunc::abs(rho*x[i, 1]) < specfunc::abs(Ax[i, 1]) then
             rho:= Ax[i, 1]/x[i, 1];
            end_if;
          end_for;
       end_if;
       tmp:= Ax - rho*x;
       tmp:= hfa::abs(tmp);
       tmp:= hfa::transpose(tmp) * tmp;
       residue:= (tmp[1,1])^(1/2); 
       if residue < 10^(-oldDIGITS)*specfunc::abs(rho) 
          then break:
       end_if:
       if iter = maxit and not nowarning then 
         warning("no convergence of vector iteration");
       end_if;
     end_for:
   end_case;

   if useHardwareFloats = FALSE then
       //---------------------------------
       //---- use SoftwareFloats ---------
       //---------------------------------
       if HardwareFloatsRequested and 
          not nowarning then
          warning("'HardwareFloats' failed, using 'SoftwareFloats' instead"):
       end_if;

       userinfo(1, "using software floats"):

       x:= x0:
       Ax:= x0;
       for iter from 1 to maxit do
         nnorm:=specfunc::sqrt(_plus(specfunc::abs(Ax[i])^2 $ i=1..n)):
         if iszero(nnorm) then
            // put the test 'A = zero-matrix ?' here. This is an exceptional
            // case and should only be tested when necessary:
            tmp:= map(A, specfunc::abs);
            if iszero(max({op(tmp)})) then
               // A = zero matrix
               rho:= float(0);
               residue:= float(0);
               break;
            else
               error("bad start vector. The ".output::ordinal(iter-1).
                     " iterate is the zero vector"):
            end_if;
         end_if:
         (x[i]:= Ax[i]/nnorm) $ i=1..n:
         (Ax[i]:= _plus(A[i,j]*x[j] $ j=1..n ) ) $ i=1..n:

         //---------------------------------------------------------
         // Ax is the next iterate for the eigenvector approximation.
         // Extract an approximation of the eigenvalue from the
         // approximate eigenvector
         //---------------------------------------------------------
         if hermitean then
            // use Rayleigh quotient rho=<x,Ax> to extract eigenvalue 
            rho:=_plus(conjugate(x[i])*Ax[i] $ i=1..n ):
         else 
            rho:=0; // use rho=max((Ax)[i]/x[i]) to extract eigenvalue 
            for i from 1 to n do 
              if (not iszero(x[i])) and
                 specfunc::abs(rho*x[i])<specfunc::abs(Ax[i]) then
                   rho:= Ax[i]/x[i];
              end_if;
            end_for;
         end_if;
         residue:= (_plus(specfunc::abs(Ax[i]-rho*x[i])^2 $ i=1..n))^(1/2):
         if residue < 10^(-oldDIGITS)*specfunc::abs(rho) then 
            break:
         end_if:
         if iter=maxit and not nowarning then 
            warning("no convergence of vector iteration");
         end_if;
       end_for:
   end_if;

   // In both branches HardwareFloats and SoftwareFloats
   // the results rho, x, residue were computed. We still
   // need to convert the eigenvector x to the requested
   // return type:

   // Reformat x to a 1-dimensional hfarray (in the Hardware
   // branch it was an hfarray(1..n, 1..1):
   if op(x, [0, 1]) = 2 then
     x:= array(1..n, [op(x)]);
   end_if;

   //----------------------
   // prepare return values
   //----------------------
   if domtype(x) <> returnType then
     case returnType
     of DOM_ARRAY do
        x:= array(1..n, [op(x)]);
        break;
     of DOM_HFARRAY do
        if traperror((x:= hfarray(1..n, [op(x)]))) <> 0 or
           has(x, RD_NAN) or
           has(x, RD_INF) or
           has(x, RD_NINF) then
           if not nowarning then
              warning("cannot return the result as an hfarray. ".
                      "Returning an array with software floats ".
                      "instead.");
           end_if:
           x:= array(1..n, [op(x)]);
        end_if:

        break;
     of DOM_LIST do
        x:= [op(x)];
        break;
     of matrix do
        x:= matrix(n, 1, [op(x)]);
        break;
     of densematrix do
        x:= densematrix(n, 1, [op(x)]);
        break;
     otherwise
       error("unexpected return type. Got: ".expr2text(returnType)):
     end_case;
   end_if:

   
/*
   if contains({DOM_ARRAY, DOM_HFARRAY}, domtype(x)) and 
      rangechanged then
      x:= subsop(x, 0 = (1, originalrange));
   end_if;
*/
   if rangechanged then
      if domtype(x) = DOM_ARRAY then
         x:= array(originalrange, [op(x)]);
      elif domtype(x) = DOM_HFARRAY then
         x:= hfarray(originalrange, [op(x)]);
      end_if;
   end_if:

   return([rho, x, residue]):
end_proc:
//--------  end of procedure spectralradius -------------------:w
// introduce an alias for numeric::spectralradius:

numeric::spectralRadius:= numeric::spectralradius:
