         
/*===========================================================================

numlib::contfrac -- continued fractions arithmetic

contfrac(e <,p>)
        computes a continued fraction for e (with precision p)

contfrac::rationalInterval(cf) 
        gives an interval [a, b] with rational bounds containing cf

contfrac::convert(a,b)	
        gives a continued fraction for the interval [a,b]

a + b   adds two continued fractions
a * b   multiplies two continued fractions
1/a     inverts a continued fraction
a^n     computes powers of a continued fraction

Convention: "..." represents a real between 1 and infinity, thus
        for example the continued fraction of 0 is 1/..., that
        of an integer n is n+1/...

Examples:

>> use( numlib,contfrac ):
>> a := contfrac(PI,5);

                                  1               
                        ---------------------- + 3
                                1                 
                        ------------------ + 7    
                              1                   
                        ------------- + 15        
                            1                     
                        --------- + 1             
                           ...                       

>> contfrac::interval(a);

                       [688/219, 355/113]

>> contfrac::convert(op(%)); 

                            1
                     ---------------- + 3
                          1
                     ------------ + 7
                        1
                     ------- + 15
                      1
                     --- + 2
                     ...

(we lost some precision, which is normal, because ... does not
 have a precise value)

Now we get a CF for the gold-ratio (1+sqrt(5))2 from that of sqrt(5),
via the fractional linear form L(x) = (x+1)/2:

>> b := contfrac(sqrt(5),4);

                                   1             
                          ------------------- + 2
                                 1               
                          --------------- + 4    
                               1                 
                          ----------- + 4        
                             1                   
                          ------- + 4            
                           ...                   



>> a+b; CF for PI+sqrt(5)

                                   1               
                        ----------------------- + 5
                                 1                 
                        ------------------- + 2    
                               1                   
                        --------------- + 1        
                             1                     
                        ----------- + 1            
                           1                       
                        ------- + 1                
                         1                         
                        --- + 5                    
                        ...                        

>> a*b; CF for PI*sqrt(5)

                                  1          
                             ------------ + 7
                                1            
                             ------- + 40    
                              1              
                             --- + 3         
                             ...             

>> 1/a; CF for 1/PI

                                    1             
                        --------------------------
                                  1               
                        ---------------------- + 3
                                1                 
                        ------------------ + 7    
                              1                   
                        ------------- + 15        
                            1                     
                        --------- + 1             
                         1                        
                        --- + 292                 
                        ...                       

>> a^3; CF for PI^3

                                  1           
                            ------------- + 31
                               1              
                            ------- + 159     
                             1                
                            --- + 3           
                            ...               

==========================================================================*/

alias(CONTFRAC = numlib::contfrac):

CONTFRAC:=newDomain("numlib::contfrac"):
CONTFRAC::create_dom:=hold(CONTFRAC):
CONTFRAC::info:= "domain 'numlib::contfrac': arithmetic with numerical continued fractions":
CONTFRAC::interface:= {
  hold(expr), hold(print), hold(rationalInterval), 
  hold(convert), hold(rational), hold(convert),
  hold(_plus), hold(_mult), hold( _power), hold(_invert), 
  hold(nthcoeff) }:

//------------------------------------------
// contfrac(x, <n>)
//   x - a numerical real value
//   n - a positive integer: the number of digits
//       of the floating point representation of 
//       irrationals to be approximated by the contfrac
//
// For irrational values x, contfrac returns the
// contfrac cf of minimal length satisfying
//     | x - cf | <= |x|*10^(-DIGITS)
// For rational coefficients x, contfrac returns
// the contfrac cf representing x precisely,
// except if n is specified.
//
// Internal representation of a numlib::contfrac:
//
//  op(numlib::contfrac(x)) = [a0, a1, ..], p)
//
// Here a0, a1, .. are the coefficients of a0 + 1/(a1 + 1/...),
// p is the rational value of the contfrac expansion.
//------------------------------------------
CONTFRAC::new:=
proc(e)
local floatEval, z, dz, macheps, l, n, pn, qn, pnold, qnold;
save DIGITS;
begin
  if args(0) < 1 then
     error("wrong number of arguments");
  end_if;
  if args(0) >= 2 then
     if domtype(args(2)) = DOM_INT and args(2) > 0 then
          DIGITS:= args(2)
     else error("the 2nd argument (the precision) must be an integer > 0");
     end_if;
  end_if;
  // need a copy z of the original input to pass to contfrac::function
  z:= e; 
  if domtype(e) = DOM_POLY then
     e:= op(e, 1);
  end_if;
  if args(0) >= 2 or not contains({DOM_RAT,DOM_INT}, type(e)) then
       floatEval:= TRUE;
       macheps:= 10.0^(-DIGITS);
       DIGITS:= DIGITS + 10;
       e:= float(e);
       if type(e) <> DOM_FLOAT then
         if type(e) = DOM_COMPLEX then
            error("contfrac of complex numbers not implemented");
         end_if;
//       return(contfrac::function(z, args(2..args(0))))
         error("cannot handle non-constant arguments; use 'contfrac' instead")
       end_if
  else floatEval:= FALSE
  end_if;
  if floatEval then
    z:= float(z):
    // approximate e = float(a) only to the leading
    // n decimals of the float approximation
    dz:= specfunc::abs(e)*macheps;
    n:= floor(e);
    // the algorithm uses a 2 step recursion
    //   ( p.(n-1)/q.(n-1), p.n/q.n ) -> p.n/q.n
    pnold:=1; 
    qnold:=0;
    pn:=n; 
    qn:=1;
    // initialize container l for the contfrac coefficients
    l:=[n];
    // stopping criterion:  | pn/qn - z | <=  |z|* 10^(-DIGITS).
    while (not iszero(e - n )) and 
    specfunc::abs(z - pn/qn) > dz do
       // next step of the contfrac expansion:
       e:= 1/(e-n); // z = a0 + 1/(a1 + 1/( ... + 1/(a.k + 1/e)..)
       if e < 1 then // wrong sign due to loss of precision, stop here
         break
       end_if;
       n:= floor(e); // next coeff of the contfrac
       l:= append(l, n); // append this coeff to the list
       // update the rational approximation
       [pnold, pn]:= [pn, n*pn + pnold];
       [qnold, qn]:= [qn, n*qn + qnold];
    end_while;
    z:= pn/qn;
  else // exact contfrac expansion for integers and rationals
    n:= floor(e);
    l:= [n];
    while not iszero(e - n) do
       // next step of the contfrac expansion:
       e:= 1/(e-n); // z = a0 + 1/(a1 + 1/( ... + 1/(a.k + 1/e)..)
       n:= floor(e); // next coeff of the contfrac
       l:= append(l, n); // append this coeff to the list
    end_while;
  end_if;

  return(new(CONTFRAC,l, z))
end_proc:

//-------------------------------------------------
// expr: returns the rational number
// (jngerhar, 6.09.01)
CONTFRAC::expr := proc(cf) begin extop(cf, 2) end_proc:

//-------------------------------------------------
// print: [a0, a1, ...] -->  a0 + 1/(a1 + 1/...),
CONTFRAC::print:=
proc(cf) 
  local l, res, i;
begin
  l:= extop(cf, 1);
  // vordefiniertes Standard-Domain fuer spezielle Ausgaben
  res:=stdlib::Exposed("...");
  if l = [0] then
    return(hold(_plus)(0, res))
  end_if;
  for i from nops(l) downto 2 do
    res := hold(_power)(hold(_plus)(l[i], res), -1);
  end_for;
  // avoid  0 + 1/(a1 + 1/...)
  if not iszero(l[1]) then
    res := hold(_plus)(l[1], res);
  end_if;
  return(res);
end_proc:

CONTFRAC::Content :=
proc(Out, data)
  local l, res, i;
begin
  l:= extop(data, 1);
  // vordefiniertes Standard-Domain fuer spezielle Ausgaben
  res:=Out(Symbol::dots);
  if l = [0] then
    return(Out::Capply(Out::Cplus, 0, res))
  end_if;
  for i from nops(l) downto 2 do
    res := Out::Capply(Out::Cdivide, Out(1),
           Out::Capply(Out::Cplus,(Out(l[i]), res)));
  end_for;
  // avoid  0 + 1/(a1 + 1/...)
  if not iszero(l[1]) then
    res := Out::Capply(Out::Cplus,(Out(l[1]), res));
  end_if;
  return(res);
end_proc:

CONTFRAC::TeX :=
  (cf, prio) -> generate::tex(dom::print(cf), prio):

// expr2text *must* return something parseable that yields an identical object
CONTFRAC::expr2text :=
(cf) -> "new(numlib::contfrac, " . expr2text(extop(cf)) . ")":

CONTFRAC::testtype:=
proc(x, T)
begin
if T = Type::Arithmetical then
  TRUE
else
  FAIL
end_if
end_proc:



/*-------------------------------------------------
// W. Oevel, 21.7.01: This is the new version of interval.
// (it used to be named 'approx')
//-------------------------------------------------
// contfrac::rationalInterval(cf : dom, <n : positive integer>)
// computes two rationals a, b such that
//   -   a <= cf <= b
//   -   a = cf or b = cf
// n - use only the coefficients 1 through n of cf.
//     If cf has less than n coefficients, then
//     n is replaced internally by the actual number
//     of coefficients in cf.
// Returns: the list [a, b]
//-------------------------------------------------*/
CONTFRAC::rationalInterval:=
proc(cf, n) // retuns an enclosing interval of the nth 
            // rational approximation (a list [a, b] of
            // two rationals a, b
  local l,r,s,i;
begin
  if args(0) < 1 then
     error("expecting at least one argument");
  end_if;
  if args(0) > 2 then
     error("expecting no more than two arguments");
  end_if;
  if domtype(cf) <> CONTFRAC then
     error("expecting an object of type 'numlib::contfrac'");
  end_if;
  if args(0) > 1 then
     n:= args(2);
     if domtype(n) <> DOM_INT or n <= 0 then
        error("the 2nd argument must be a positive integer");
     end_if;
  end_if;

  l:= extop(cf, 1);
  if args(0) = 1 or n > nops(l) then
       n:= nops(l);
  end_if;

  r:= extop(cf, 2);
  s:=op(l, n) + 1;
  for i from n-1 downto 1 do
     s:=op(l,i) + 1/s;
  end_for;
  return([min(r,s),max(r,s)])
end_proc:

/*-------------------------------------------------
// W. Oevel, 21.7.01: This is the new version of convert.
// (it used to be named 'unapprox')
// It computes a contfrac expansion of the left border of
// the input interval [a, b] and stops, when the contfrac
// is inside [a, b]. Thus, one gets the contfrac expansion
// of a with minimal number of coefficients such that the
// contfrac is inside [a, b]

  contfrac::convert(a, b)

  a, b -- numerical real values representing an
          interval [a, b]

-------------------------------------------------*/
CONTFRAC::convert:=
proc(a, b) // find a minimal contfrac expansion
           // of a that lies in [a,b] 
local e, n, pnold, qnold, pn, qn, l;
save DIGITS;
begin
  if args(0) <> 2 then
     error("expecting two arguments");
  end_if;
  if a = b then
     // let contfrac::new handel conversions to float etc.
     return(CONTFRAC(a))
  end_if;
  DIGITS:= DIGITS + 10;
  if {domtype(a)} minus {DOM_INT, DOM_RAT, DOM_FLOAT} <> {} then
     a:= float(a);
  end_if;
  if {domtype(a)} minus {DOM_INT, DOM_RAT, DOM_FLOAT} <> {} then
     error("cannot convert left boundary to a number");
  end_if;
  if {domtype(b)} minus {DOM_INT, DOM_RAT, DOM_FLOAT} <> {} then
     b:= float(b);
  end_if;
  if {domtype(b)} minus {DOM_INT, DOM_RAT, DOM_FLOAT} <> {} then
     error("cannot convert right boundary to a number");
  end_if;
  
  //----------------------------------
  // make sure that a <= b
  //----------------------------------
  if a > b then
     [a, b]:= [b, a]
  end_if;
  //----------------------------------
  // the work starts
  //----------------------------------
  e:= a;
  n:= floor(e);
  // the algorithm uses a 2 step recursion
  //   ( p.(n-1)/q.(n-1), p.n/q.n ) -> p.(n+1)/q.(n+1)
  pnold:=1;
  qnold:=0;
  pn:=n;
  qn:=1;
  // initialize container l for the contfrac coefficients
  l:=[n];
  // stopping criterion:  a <= pn/qn <= b
  while pn/qn < a or pn/qn > b do
    if iszero(e-n) then // extra check for rationals
       break;
    end_if;
    // next step of the contfrac expansion:
    e:= 1/(e-n); // z = a0 + 1/(a1 + 1/( ... + 1/(a.k + 1/e)..)
    if e < 1 then // wrong sign due to loss of precision, stop here
      break
    end_if;
    n:= floor(e); // next coeff of the contfrac
    l:= append(l, n); // append this coeff to the list
    // update the rational approximation
    [pnold, pn]:= [pn, n*pn + pnold];
    [qnold, qn]:= [qn, n*qn + qnold];
  end_while;
  return(new(CONTFRAC, l, pn/qn))
end_proc:

//-------------------------------------------------
// W. Oevel, 21.7.01: This is the new version of _plus.
// It interprets each contfrac as a rational and does
// exact arithmetic on the rationals. Finally, the
// result is turned into a contfrac
CONTFRAC::_plus:=proc() 
local s, i, x;
begin 
   // pass the arguments to the "super" structure
   // (stdlib) contfrac if symbolic contfracs
   // are involved. This method is just for
   // adding numlib::contfracs.
   if contains(map({args()}, domtype), contfrac) then
     return(contfrac::_plus(args()))
   end_if;
   s:= 0:
   for i from 1 to args(0) do
       x:= args(i):
       if domtype(x) = DOM_POLY then
          x:= op(x, 1); // Let CONTFRAC handle non-constant x 
       elif domtype(x) = CONTFRAC then
          x:= extop(x, 2);
       end_if;
       s:= s + x;
   end_for;
   if numeric::indets(s) minus Type::ConstantIdents = {} then
     return(CONTFRAC(s));
   else
     return(contfrac(s));
   end_if;
end_proc:

//-------------------------------------------------
// W. Oevel, 21.7.01: This is the new version of _mult.
// It interprets each contfrac as a rational and does
// exact arithmetic on the rationals. Finally, the
// result is turned into a contfrac
CONTFRAC::_mult:=proc() 
local s, i, x;
begin 
   // pass the arguments to the "super" structure
   // (stdlib) contfrac if symbolic contfracs
   // are involved. This method is just for
   // multiplying numlib::contfracs.
   if contains(map({args()}, domtype), contfrac) then
     return(contfrac::_mult(args()));
   end_if;
   s:= 1;
   for i from 1 to args(0) do
       x:= args(i):
       if domtype(x) = DOM_POLY then
          x:= op(x, 1); // Let CONTFRAC handle non-constant x 
       elif domtype(x) = CONTFRAC then
          x:= extop(x, 2);
       end_if;
       s:= s * x;
   end_for;
   if numeric::indets(s) minus Type::ConstantIdents = {} then
     return(CONTFRAC(s));
   else
     return(contfrac(s));
   end_if;
end_proc:

//-------------------------------------------------
//  x = a0 + 1/(a1 + 1/(a2 + ...))
// Case 1:   a0 =  0 --> 1/x = a1 + 1/(a2 + ...)
// Case 2:   a0 <> 0 --> 1/x = 1/(a0 + 1/(a1 + ...))
CONTFRAC::_invert:=proc(x) 
local l, p;
begin
   l:= extop(x, 1);
   if op(l,1) = 0 then 
      delete l[1];
   else l:=[0, op(l)] end_if;
   p:= extop(x, 2);
   if iszero(p) then
      error("division by zero");
   end_if;
   subsop(x, 1 = l, 2 = 1/p);
end_proc:

//-------------------------------------------------
CONTFRAC::_power:=proc(x, n) 
begin
   if n = 0 then return(CONTFRAC(1)) end_if;
   if n = 1 then return(x) end_if;
   if n = - 1 then return(CONTFRAC::_invert(x)) end_if;
   return(CONTFRAC(extop(x, 2)^n));
end_proc: 

//-------------------------------------------------
// W. Oevel, 21.7.01: new method coeff implemented
// jngerhar, 28.11.01: coeff --> nthcoeff, now must take two arguments
//
//  nthcoeff(cf, n) -- returns the n-th coefficient of cf,
//                     i.e., op(cf, [1, n]),
//                     and FAIL if n > nops(op(cf, 1))

CONTFRAC::nthcoeff:= proc(cf, n) 
local L;
begin
  if args(0) <> 2 then
     error("expecting two arguments");
  elif domtype(cf) <> CONTFRAC then
     error("expecting an object of type 'numlib::contfrac'");
  elif domtype(n) <> DOM_INT or n < 0 then
     error("the 2nd argument must be a nonnegative integer");
  end_if;

  L := extop(cf, 1);
  if n > nops(L) then
     return(FAIL);
  end_if;
  return(L[n]);
end_proc:

//-------------------------------------------------
// W. Oevel, 21.7.01: new method 'rational' implemented
//
//  rational(cf, n)
//     cf - a contfrac
//     n  - a positive integer
//
//  returns the rational number represented by the
//  coeffs 1 through n of cf.
CONTFRAC::rational:= proc(cf, n) 
local L, p, i;
begin
  if args(0) < 1 then
     error("expecting at least one argument");
  end_if;
  if args(0) > 2 then
     error("expecting no more than two arguments");
  end_if;
  if domtype(cf) <> CONTFRAC then
     error("expecting an object of type 'numlib::contfrac'");
  end_if;
  if args(0) > 1 then
     n:= args(2);
     if domtype(n) <> DOM_INT or n <= 0 then
        error("the 2nd argument must be a nonnegative integer");
     end_if;
  end_if;

  L:= extop(cf, 1);
  if args(0) = 1 or n > nops(L) then
       n:= nops(L);
  end_if;
  if n = nops(L) then
     return(extop(cf, 2))
  end_if;
  p:= L[n];
  for i from n - 1 downto 1 do
    p:= L[i] + 1/p;
  end_for;
  return(p);
end_proc:

//--------------------------------------------
unalias(CONTFRAC):
