/*

 numlib::contfracPeriodic(p, q, N)

 compute the contfrac of (p + q*sqrt(N)) where p, q are rationals

 returns two lists l1, l2 where l1 is the non-periodic part and l2 is
 the periodic part of the coeffs of the continued fraction, i.e.,
 the first operand of numlib::contfrac(p+q*sqrt(N)) equals the beginning
 of the list l1.l2.l2.l2.l2.... 

 if p + q*sqrt(N) is rational, only one list l1 is returned
*/

numlib::contfracPeriodic:=
proc(p: Type::Rational, q: Type::Rational, N: Type::PosInt)
  local
  alph: DOM_LIST,
  alphas: DOM_LIST,
  result: DOM_LIST,
  fl: DOM_INT,
  denominator: DOM_INT,
  i: DOM_INT;
begin
  if numlib::issqr(N) then
    p:= p + q*isqrt(N);
    q:= 0;
  end_if;
    
  if q=0 then
    return(op(numlib::contfrac(p), 1));
  end_if;

  if p=0 and q=1 then
    return(numlib::sqrt2cfrac(N));
  end_if;
  
  alph:= [p, q]; 
  result:= [];
  alphas:= [];

  // we proceed as follows: with alpha initially being p+q*sqrt(N),
  // represented as a list alph=[p, q], we do
  // result[j]:= floor(alph); alpha:= 1/alpha until some alpha occurs
  // for a second time
  
  repeat
    alphas:= alphas.[alph];
    fl:= floor(alph[1] + alph[2]*N^(1/2));
    alph[1]:= alph[1] - fl;
    result:= result.[fl];
    // invert alph:
    // 1/(p + q*sqrt(N)) = (p-q*sqrt(N))/(p^2 - q^2*N)
    denominator:= alph[1]^2 - alph[2]^2*N;
    alph[1]:= alph[1]/denominator;
    alph[2]:= -alph[2]/denominator;
  until (i:= contains(alphas, alph)) > 0 end_repeat;

  [op(result, 1..i-1)], [op(result, i..nops(result))];
end_proc: