/* ecm(n) tries to factor n using the ECM algorithm

This version uses the Montgomery form (8) from [2] which avoid's gcd's:

        b*y^2*z = x^3 + a*x^2*z + x*z^2

References:
[1] "Speeding the Pollard and Elliptic Curve Methods of Factorization", by Peter
   Montgomery, Math. of Comp. 48 (177), pages 243-264, January 1987.
[2] "Factorization of the tenth and eleventh Fermat numbers", by Richard Brent,
ftp://nimbus.anu.edu.au/pub/Brent/rpb...

Input: n - a positive integer
          B1 - (optional) the prime bound for step 1, default is 1000
                  or a list of primes, e.g. [2,3,47,345551]
          s - (optional) the seed sigma from (11) in [2]
Output: a factor of n, not necessarily prime 
            (could be 1 or n if n is prime, or if the algorithm fails)

Examples:
>> use(numlib, ecm):
>> ecm(137703491,100,6); Step 1
              17389
>> ecm(137703491,100,8); Step 1
               7919
>> ecm(137703491,100,13); Step 2
               7919
>> ecm(2^(2^6)+1,100,14);
              274177

From [2], page 15:
>> b:=[2,3,47,61,89,233,829,3607,18451,54869,75223,149827,345551]:
>> ecm((55^126+1)/2/17/37/89/109,b,805816989);
25233450176615986500234063824208915571213

Factorization of F_10: [2] page 17:
>> b:=[2,3,5,149,163,197,7187,18311,123677,226133,314263,4677853]:
>> ecm(2^1024+1,b,14152267);
4659775785220018543264560743076778192897

With Step 2:
>> b:=[2,3,5,149,163,197,7187,18311,123677,226133,314263]:
>> ecm((2^1024+1)/295760497253281793,b,14152267,4677853);

                 4659775785220018543264560743076778192897
D=315:1167s/540135mul D=1155:deg(q)=239,1690s/477469mul
D=2310:1768s/486441mul
use(numlib,ecm):
ecm(137703491,100,6):
setuserinfo(numlib,2):
sysassign(numlib::ecm_bestD,2310):

kepler: old=1732s/1045806mul (D=315)
*/

numlib::ecm:=
proc(n:Type::PosInt, B1, s, B2)
  local B1orig,p,u,v,x,z,a,b,i,q,l,j,x1,z1,x2,z2,w,m,nextp,basis,st,mul;
  save SEED; // needed due to use of 'random'
begin
  st:=time();
  mul:=0; // number of multiplications mod n 
  if n = 1 or n = 5 then
    return(n);
  end_if;
  if (p:=igcd(n,6))<>1 then 
    return(p);
  end_if;
  // now gcd(n,6)=1 
  if args(0)>1 then
    if type(B1)=DOM_LIST then // basis 
      B1orig:=B1;
      B1:=max(op(B1));
      basis:=append(B1orig,B1+1);
      nextp:=proc() begin op(basis,args(1)) end_proc;
    elif type(B1)=DOM_INT then
      nextp:=ithprime;
    else
      error("Second argument has wrong type");
    end_if;
  else
    B1:=1000;
    nextp:=ithprime;
  end_if;
  
  // generates a random starting point using (11) from [2] 
  if args(0)<3 then
    s:=random(6..n-1)();
    userinfo(10,"Starting point generated from sigma=".s);
  end_if;

  if args(0)<4 then
     B2:=100*B1;
  end_if;
  
  u:=(s^2-5) mod n; 
  v:=(4*s) mod n;
  x:=u^3 mod n; 
  z:=v^3 mod n; // starting point 
  userinfo(3,"starting point: (".x." : : ".z.")");
  a:=[((v-u)^3*(3*u+v)) mod n, (4*x*v) mod n];
  if (p:=igcd(n, a[2]))>1 then
    userinfo(1, "Lucky case: common divisor immediately found");
    return(p);
  end_if;
  a:=(a[1]/a[2]-2) mod n;
  b:=(a+2)/4 mod n;
  // Step 1 
  i:=1; 
  p:=nextp(i);
  repeat
    q:=p; 
    while (u:=q*p)<=B1 do
      q:=u;
    end_while; 
    q:=q-1;
      // multiply P=(x : : z) by q 
      /* at each point, mP = (x1 : : z1), (m+1)P = (x2 : : z2), starting with m=1,
         see p. 261 of [1] */
      // start from (P,2P) i.e. m=1 
    m:=1;
    u:=igcdex(z,n); 
    if u[1]<>1 then 
      return(u[1]);
    end_if;
    x:=(x*u[2]) mod n; 
    z:=1;
    x1:=x; 
    z1:=z;
    u:=(x+z)^2 mod n; 
    v:=(x-z)^2 mod n; 
    x2:=(u*v) mod n;
    w:=u-v; 
    u:=(v+b*w) mod n; 
    z2:=(w*u) mod n;
    mul:=mul+6;
    userinfo(3,"start: ".m."P=(".x1." : : ".z1."), ".(m+1)."P=(".x2." : : ".z2.")");
    l:=[]; 
    while q<>1 do 
      l:=append(l,q mod 2); 
      q:=q div 2;
    end_while;
    // l[j+1] is the bit corresponding to 2^j in q 
    for j from nops(l) downto 1 do
      if l[j]=0 then // (mP,(m+1)P) -> (2mP,(2m+1)P) i.e. m <- 2*m 
        // first compute (2m+1)P from mP, (m+1)P and P with 6 multiplications 
        u:=((x2-z2)*(x1+z1)) mod n; 
        v:=((x2+z2)*(x1-z1)) mod n;
        if m=1 then 
        x1:=x2; 
        z1:=z2; 
        mul:=mul-5 
      end_if; // 2mP = (m+1)P 
        x2:=(u+v)^2 mod n; // x2:=(z*w) mod n; (z=1) 
        w:=(u-v)^2 mod n; 
        z2:=(x*w) mod n;
        // then double mP -> 2mP with 5 multiplications 
        if m<>1 then
          u:=(x1+z1)^2 mod n; 
          v:=(x1-z1)^2 mod n;
          x1:=(u*v) mod n;
          w:=u-v; // 4 x1 z1 
          u:=(v+b*w) mod n; 
          z1:=(w*u) mod n;
        end_if;
        m:=2*m;
      else // (mP,(m+1)P) -> ((2m+1)P,(2m+2)P), i.e. m <- 2*m+1 
        // first compute (2m+1)P from mP, (m+1)P and P with 6 multiplications 
        u:=((x2-z2)*(x1+z1)) mod n; 
        v:=((x2+z2)*(x1-z1)) mod n;
        x1:=(u+v)^2 mod n; // x1:=(z*w) mod n; (z=1) 
        w:=(u-v)^2 mod n; 
        z1:=(x*w) mod n;
        // then double (m+1)P -> (2m+2)P with 5 multiplications 
        u:=(x2+z2)^2 mod n; 
        v:=(x2-z2)^2 mod n;
        x2:=(u*v) mod n;
        w:=u-v; // 4 x2 z2 
        u:=(v+b*w) mod n; 
        z2:=(w*u) mod n;
        m:=2*m+1;
      end_if;
      mul:=mul+10;
      userinfo(3,"after one step: ".m."P=(".x1." : : ".z1."), ".(m+1)."P=(".x2." : : ".z2.")");
    end_for;
    // now qP = (x1 : : z1) and (q+1)P = (x2 : : z2) 
    x:=x2; 
    z:=z2;
    userinfo(2,"finished p=".p." at time ".time());
    if (w:=igcd(n,z))<>1 then
      return(w)
    end_if;
    p:=nextp((i:=i+1));
  until p>B1 end_repeat;
  userinfo(1,"Step 1 took ".((time()-st) div 1000)."s for ".mul." multiplications");
  userinfo(1,"K1=".expr2text(mul*ln(2.0)/B1));
  numlib::ecm_step2(n,B1,B2,x,z,b)
end_proc:

/* returns the best value of D for step B2. It should satisfy:
(a) D about sqrt(B2)
(b) D/2 having as many prime divisors as possible (i.e. D product of prime
    factorials 2, 6, 30, 210, ...)
*/
numlib::ecm_bestD:=
proc(B2)
  local d;
begin
  d:=round(sqrt(B2/2.0));
  if d<15 then
    3*round(d/3)
  elif d<105 then
    15*round(d/15)
  elif d<1155 then
    105*round(d/105)
  elif d<15015 then
    1155*round(d/1155)
  else
    15015*round(d/15015)
  end_if
end_proc:

/* Step 2: improved standard continuation, cf [2] p. 7-8.
   Use polynomial arithmetic.
*/
numlib::ecm_step2:=
proc(n,B1,B2,x,z,b)
  local st,mul,Q,DD,nQ,i,x1,z1,u,v,w,x2,z2,m,q,y,d;
begin
   st:=time();
   y:=genident();
   Q:=[x,z]; mul:=0;
   DD:=numlib::ecm_bestD(B2);
   userinfo(1,"start step 2 with B2=".B2.", D=".DD); 
   userinfo(3,"Q=(".x." : : ".z.")");
   // with Q the point computed by Step 1, we precompute 2*d*Q for 0 < d <= D 
   // i*Q is stored in nQ[i] 
   nQ[1]:=Q; 
   q:=poly(1,[y],IntMod(n));
   for i from 2 to 2*DD do // compute i*Q 
      if i mod 2 = 0 then // i*Q = 2*(i/2)*Q 
         x1:=nQ[i/2]; 
         z1:=x1[2]; 
         x1:=x1[1];
         u:=(x1+z1)^2 mod n; 
         v:=(x1-z1)^2 mod n; 
         x2:=(u*v) mod n;
         w:=u-v; u:=(v+b*w) mod n; 
         z2:=(w*u) mod n;
         // normalizes z2 to 1 to save one multiplication below 
         u:=igcdex(z2,n);
         if u[1]<>1 then 
           return(u[1]);
         end_if;
         x2:=(x2*u[2]) mod n; 
         z2:=1;
         nQ[i]:=[x2,z2];
         if i<DD and igcd(i+1,DD)=1 then 
           q:=q*poly(y-x2,[y],IntMod(n));
         end_if;
      else // i*Q = (i+1)/2*Q+(i-1)/2*Q 
         x1:=nQ[(i-1)/2]; z1:=x1[2]; x1:=x1[1];
         x2:=nQ[(i+1)/2]; z2:=x2[2]; x2:=x2[1];
         u:=((x2-z2)*(x1+z1)) mod n; v:=((x2+z2)*(x1-z1)) mod n;
         w:=(u+v)^2 mod n; x1:=(z*w) mod n;
         w:=(u-v)^2 mod n; z1:=(x*w) mod n;
         nQ[i]:=[x1,z1]; 
      end_if; mul:=mul+6
   end_for;
   d:=degree(q); 
   userinfo(1,"degree(q)=".d);
   // then we compute mQ for m=1, 2*D+1, 4*D+1, ... 
   // loop invariant: 2DQ=(x2:z2),  (m-2D)Q=(x1:z1) (m-4*D)Q=(x:z) 
   x2:=nQ[2*DD]; z2:=x2[2]; x2:=x2[1];
   x1:=Q[1]; z1:=Q[2]; x:=nQ[2*DD-1][1]; z:=nQ[2*DD-1][2];
   userinfo(1,"initialization of Step 2 took ".((time()-st) div 1000)."s");
   m:=1; while m-DD<=B2 do
      userinfo(2,"m=".m." at time ".time().", mul=".mul);
      if m<>1 then 
         // compute m*Q = (m-2D)Q + 2DQ 
         u:=((x2-z2)*(x1+z1)) mod n; v:=((x2+z2)*(x1-z1)) mod n;
         w:=(u+v)^2 mod n; i:=x; x:=x1; x1:=(z*w) mod n;
         w:=(u-v)^2 mod n; z:=z1; z1:=(i*w) mod n; mul:=mul+6;
      end_if;
      // now nQ[m]:=[x1,z1]; 
      if m+2*DD>B1 then
         u:=igcdex(z1,n); // u[1] = u[2]*z1+u[3]*n 
         if u[1]<>1 then 
           return(u[1]);
         end_if;
         // normalizes z1 to 1 to save one multiplication below 
         x1:=(x1*u[2]) mod n; 
         z1:=1; 
         w:=evalp(q,y=x1); 
         mul:=mul+d;
         if (w:=igcd(w,n))<>1 then 
           return(w);
         end_if
      end_if;
      m:=m+2*DD;
   end_while;
   userinfo(1,"Step 2 took ".((time()-st) div 1000)."s for ".mul." multiplications");
   userinfo(1,"K2=".expr2text(mul/B2*ln(1.0*B2)));
   1
end_proc:

