
/*  */ /*  */ //  

/*++    
numlib::mpqs - the multi-polynomial quadratic sieve for factoring integers

numlib::mpqs(n,options) 

n -  integer to be factored

The following options can be given in any order:

Factorbase=l          -  l: list of primes
      use the members of l as a factorbase
MaxInFactorbase=maxfb     -  maxfb: positive integer
      use all suitable primes below maxfb as a factorbase
      (cannot be given together with Factorbase=l)
      The default is maxfb=300.
NumberOfPolynomials=k - k: positive integer
      Sieve the values of k polynomials (default:k=1)
SieveArrayLimit=M     -  M: positive integer
      For every polynomial f to be considered, find
      smooth values among the numbers f(-M)...f(M).
      (Default: M=1000).
Tolerance=T           -  T: positive real number 
      Allow numbers to pass the first step of the
      sieve if the non-smooth part of them does probably
      not exceed maxfb^T, where maxfb is the maximum 
      element of the factorbase. 
      (Default: T=2).
LargeFactorBound=B    -   B: positive integer
      After factoring the values that have passed the
      first step of the sieve, rule out those which 
      have an irreducible factor exceeding B.
                    (Default: B=maxfb^T).
InteractiveInput      
      Enter the above options interactively.

Gauss
            Use Gaussian elimination to solve the linear system

CollectInformation
      Collect information about the course of the 
      algorithm, and return them as a list.

++*/ 

alias(FBSIZE=1000):  // default size of maximum prime in factorbase 
alias(PRECMULT=100): // use logarithms with precision two decimal digits 
alias(ARRSIZE=10000): // default M for sieve array 
alias(NPOLYS=1):  // default number of polynomials used 

numlib::mpqs:=
proc(oldn)
local i,r,j,optionalargs,fb,l,fbprod,npolys,M,bound, n, found,
  largefactorbound, T, 
  factorizations,H,HH,ll,st, hilf, resu,k, xorlist, vecs, pivot,
  partialreports, fullreports,
  multiplier,collectinformation, result,
  sievepoly, sievestep, factorbase, cutoff, 
  gaussian;
save SEED;
begin

// procedure definitions : 

  factorbase:=
  proc(n:DOM_INT, size:DOM_INT):DOM_LIST
// given n, select those primes <> 2 for which n is a quadratic residue
// 2 is handled separately by the mpqs-algorithm
  begin
    select(select([$3..size],isprime),
           proc()
           begin
             case numlib::jacobi(n,args(1))
               of 1 do
                 TRUE; break
               of -1 do
                 FALSE; break
               of 0 do
                 if args(1)>8 then
                   // this cannot be the multiplier
                   found:=args(1)
                 end_if;
                 FALSE;
                 break
             otherwise
               error("Unexpected result from numlib::jacobi")
           end_case
      end_proc )
  end_proc:

  sievestep:=
  proc()
    local dlist, d, st;
  begin
// create list of d's
    d:=4*(floor(sqrt(sqrt(n)/(M*sqrt(2)))) div 4) + 3;
    userinfo(4,"Initial d chosen:".expr2text(d));
    st:=time();
    dlist:=[];
    while nops(dlist) < npolys do
      if isprime(d) and numlib::jacobi(d,n)=1 then
        dlist:=append(dlist,d);
      end_if;
      d:=d+4;
    end_while;
    userinfo(1,"Time for creating list of d's :".expr2text(time()-st));
   _concat(op(_concat(map(dlist,sievepoly))))        
  end_proc:

  sievepoly:=
  proc(d)
    local p,f,r,st,larray,result, a,b,c, h1, h2,
    /* private */ logp,q, rr, mask, bigmask, pprod, offset;

// sieves with f=ax^2+bx+c
/* returns a list of sieve reports, each of the form
  [f(r),u,smallf, largef,H] where H^2 \equiv f(r) mod n 
   and f(r)=u*smallf*largef */
// smallf is the product of primes in the factorbase 


  begin
    userinfo(5, "Generating coefficients for d= ".expr2text(d));
    a:=d^2 mod n;
    userinfo(10, "a =".expr2text(a));
    h1:=powermod(n,(d+1) div 4,d);
    userinfo(10, "h1 =".expr2text(h1));
    h2:= ((n-h1^2)/d)/(2*h1) mod d;
    userinfo(10, "h2 =".expr2text(h2));
    b:=irem(h1+h2*d, a);
    if b mod 2=0 then b:=a-b end_if;
    userinfo(10, "b = ".expr2text(b));
    c:=(b^2-n)/(4*a);
    userinfo(10, "c = ".expr2text(c));

    st:=time();
    r:=genident();
    f:=poly(a*r^2+b*r+c,[r]);


// as c=f(0) indicates whether f(0) is even or odd, initialize array
    if c mod 2=1 then
      larray:=[(0,round(PRECMULT*float(ln(2)))) $ M].[0];
    elif (a+b+c) mod 2 = 1 then // f(1) is odd
      larray:=[(round(PRECMULT*float(ln(2))),0) $M].
               [round(PRECMULT*float(ln(2)))]
    else
      larray:=[ round(PRECMULT*float(ln(2))) $2*M+1 ];
    end_if;


   // now, larray[i] contains the factorization of i-(M+1) 
   // and the factorization of i can be found in larray[i+M+1] 
   // since c is always even, zero is always a root of f for p=2 

   bigmask:=[0];
   pprod:=1;

   for p in fb do 

      if pprod*p > M then // flush and build a new mask 
         offset:= M mod pprod;
         larray:=zip(larray, [op(bigmask, pprod-offset+1..pprod)]
           ._concat(bigmask $((2*M+1) div pprod)+1),_plus);		
         pprod:=1;
         bigmask:=[0]
      end_if;

      logp:=round(PRECMULT*float(ln(p)));
      // solve f(rr)=0 mod p 
      if a mod p = 0 then
         // now b mod p \neq 0, because p does not divide n 
         rr:=c/b mod p 
      else
         q:=op(numlib::msqrts(n,p),1);
         rr:=[(q-b)/(2*a) mod p,(-q-b)/(2*a) mod p];
      end_if;
      // create mask 
      // mask corresponds to [ 0 mod p ... p-1 mod p ] 
      mask := [0 $ p];

      if domtype(rr)=DOM_LIST then
         // polynomial has two roots modulo p 
         mask[rr[1]+1]:=logp;
         mask[rr[2]+1]:=logp	    	      
      else
         // only one root 
         mask[rr+1]:=logp	   
      end_if;

      // combine with old bigmask to a new bigmask 
      // loop invariant: bigmask always has pprod elements 

      bigmask:=zip(_concat(bigmask $p), _concat(mask $pprod), _plus);
      pprod:=p*pprod

   end_for;

   // add mask to array 
   offset:= M mod pprod;
   larray:=zip(larray, [op(bigmask, pprod-offset+1..pprod)]
      ._concat(bigmask $((2*M+1) div pprod)+1),_plus);       
   userinfo(1,"Time for sieving ".expr2text(f).":".expr2text(time()-st));
   st:=time();
   larray:=select([$1..2*M+1], proc(u) begin larray[u]>=bound end_proc);

    userinfo(1,"Time for sieving ".expr2text(f).":".expr2text(time()-st));

    // find out which numbers really have small factors

    st:=time();
    result:=map(larray,
                proc(x)
                  local largef,ggcd,signum,smoothpart,H;

                begin
                  H:=(2*a*(x-(M+1))+b)/(2*d) mod n;
                  x:=evalp(f,r=x-(M+1));
                  signum:=sign(x);
                  largef:=abs(x);
                  smoothpart:=1;
                  while (ggcd:=igcd(largef,fbprod))<> 1 do
                    largef:=largef/ggcd;
                    smoothpart:=smoothpart*ggcd
                  end_while;
                [x,signum,smoothpart,largef,H]
                end_proc
                );
/* now result is a list of entries of the form [x,signum,smoothpart,largef,H], 
where signum is the sign, smoothpart is the product of factors over 
the factorbase, largef
  is the non-smooth part of x, and H is a square root of x modulo n */
// then x=signum*smoothpart*value holds 
    userinfo(1,"Time for trial divisions:".expr2text(time()-st));
    userinfo(3,"The polynomial ".expr2text(f) ." generated ".
             expr2text(nops(result)). " values conjectured to be smooth");
    result:=select(result,proc()
                          begin
                            op(args(1),4) < largefactorbound
                          end_proc );
    userinfo(3,"The polynomial ".expr2text(f) ." generated ".
             expr2text(nops(result))." sieve reports");
    userinfo(20,"Sieve reports:");
    for r in result do
      userinfo(20, expr2text(op(r,1))."=".expr2text(op(r,2))."*".
               expr2text(op(r,3))."*".expr2text(op(r,4))." ; H = ".
               expr2text(op(r,5)));
    end_for;
    result
  end_proc:


//   ----------  m a i n    p r o g r a m --------------


  n:=abs(oldn);
  if type(n)<>DOM_INT then
    error("First argument must be integer (number to be factored)");
  end_if;
  if isprime(n) or n=1 or n=0 then
    return(n)
  end_if;
// exclude small factors
  if (fb:=igcd(n, 30)) > 1 then
    return(fb)
  end_if;

// initialize options

  fb:= T:= M := largefactorbound := npolys := FAIL;
  
  collectinformation:= FALSE;

  gaussian:= FALSE;
 

// unless otherwise set 

// select multiplier 
multiplier:=1/n mod 8;
n:=n*multiplier;
userinfo(2,"Select multiplier ".expr2text(multiplier).", factoring ".
         expr2text(n));
result:=["Multiplier :".expr2text(multiplier)];

// get options

if args(0)>1 then
  optionalargs:=[args()];
  delete optionalargs[1];
  map(optionalargs,
      proc(x)
      begin
        if type(x) = "_equal" then
          case op(x,1)
            of hold(Factorbase) do
              fb:=op(x,2);
              if domtype(fb)<>DOM_LIST then
                context(hold(error)("Factorbase = [primes,...] expected"));
              end_if;
              break
            of hold(MaxInFactorbase) do
              if type(fb)=DOM_LIST then
                context(hold(error)("Options 'Factorbase' and 'MaxInFactorbase' cannot be combined"));
              end_if;
              fb:=factorbase(n,op(x,2));
              userinfo(4,"Factorbase chosen:".expr2text(fb));
              break
            of hold(LargeFactorBound) do
              largefactorbound:= op(x,2); 
              break
            of hold(NumberOfPolynomials) do
              npolys:=op(x,2); 
              break
            of hold(SieveArrayLimit) do
              M:=op(x,2); 
              break
            of hold(Tolerance) do
              T:=op(x,2); 
              break
            otherwise
              context(hold(error)("Unknown option '".expr2text(op(x,1))."'"));
          end_case;
        else
      case x
        of hold(InteractiveInput) do
          input("Enter limit for size of primes in factorbase",fb);
          fb:=factorbase(n,fb);
          userinfo(4,"Factorbase chosen:".expr2text(fb));
          print(Unquoted,
                "Sieving will be done on some interval [-M,M] .");
          input("Enter value for M", M);
          input("Enter number of polynomials to be used",
                   npolys);
          input("Enter sieve tolerance", T);
          input("Enter bound for maximum size of really encountered large factor", largefactorbound);
          break;
        of hold(CollectInformation) do
          collectinformation:=TRUE;
          break;
        of hold(Gauss) do
          gaussian:=TRUE;
          break
        otherwise 
          context(hold(error)("Unknown option '".expr2text(x)."'"));
      end_case;
    end_if;
 end_proc
      );  // end of map 
end_if;


// use defaults as far as options were not given 
// generate factorbase 
if type(fb)<>DOM_LIST then
  found:= FAIL;
  fb:=factorbase(n,FBSIZE);
  if found <> FAIL then
    userinfo(2, "Factor found while factorbase was chosen");
    return(found)
  end_if;
  userinfo(4, "Factorbase chosen: ". expr2text(fb))
end_if;
fbprod:=_mult(2,multiplier,op(fb)); // in any case

// select number of polynomials : default: 1 

if type(npolys)<>DOM_INT then
  npolys:=NPOLYS;
  userinfo(4, "Number of polynomials: ".expr2text(npolys))
end_if;

if type(T)=DOM_FAIL then
  T:=2;
  userinfo(4, "Tolerance: ".expr2text(T));
end_if;

if type(largefactorbound)=DOM_FAIL then
  largefactorbound:=fb[nops(fb)]^T;
  userinfo(4, "Bound for large factors: ".expr2text(largefactorbound))
end_if;

if type(M)<>DOM_INT then
  M:= ARRSIZE;
  userinfo(4, "Size of sieve array: ".expr2text(M))
end_if;


/* global bound: suggested by Silverman, but could also depend on polynomial */

bound:=round((float(ln(M*sqrt(n/2)))-float(T*ln(fb[nops(fb)])))*PRECMULT);
userinfo(4, "Bound for passing first step of the sieve: ".expr2text(bound));
userinfo(4, "Bound corresponds to a smooth part of ".
    expr2text(float(exp(bound/PRECMULT))));


if collectinformation then
   result:=result.["Size of factorbase:".expr2text(nops(fb))];
end_if;


st:=time();
l:=sievestep();
userinfo(1,"Total time for sieve-step: ".expr2text(time()-st));


userinfo(1,"Total number of sieve reports: ".expr2text( nops(l)));
if collectinformation then 
   result:=append(result," Sieve Reports: ".expr2text(nops(l)));
end_if;

/* now, eliminate sieve reports which are not usable because of their
  large factors */ 
// do *not* try to factor the large factors into primes 


fullreports:=split(l, proc(x) begin op(x,4)=1 end_proc);
partialreports:=op(fullreports,2);
fullreports:=op(fullreports,1);
userinfo(2, "Number of full reports: ".expr2text(nops(fullreports)));
userinfo(2, "Number of partial reports: ".expr2text(nops(partialreports)));

st:=time();


partialreports:=sort(partialreports, proc(x,y) begin x[4]<=y[4] end_proc);
userinfo(3, "Time for sorting reports: ".expr2text(time()-st));

// cycle-detection step 

ll:=null();  // meaning of ll: previous reports with the same large factor 
r:=null();  // old large factor (could be any object) 
i:=1;
while i<=nops(partialreports) do
        if partialreports[i][4]=r then
                // another report with the same large factor 
                ll:=ll+1;
                i:=i+1;
        else
                // new large factor 
                r:=partialreports[i][4];
                if ll=1 then 
                        // previous r occured only once 
                        delete partialreports[i-1]
                elif ll=2 then // collect 
                        fullreports:=append(fullreports,
                        zip(partialreports[i-2],partialreports[i-1],_mult));
                        delete partialreports[i-1];
                        delete partialreports[i-2];
                        i:=i-1;                           
                else // ll>2 
                        i:=i+1
                end_if;
                ll:=1;
        end_if;
end_while;

//flush 

if ll=1 then
                delete partialreports[i-1]
elif ll=2 then
                fullreports:=append(fullreports,
                zip(partialreports[i-2],partialreports[i-1],_mult));
                delete partialreports[i-1];
                delete partialreports[i-2];          
// else do nothing 
end_if;

l:=fullreports.partialreports;

userinfo(1,"Time for handling large factors:".expr2text(time()-st));
userinfo(20, "Partial sieve reports left after elimination: "
        .expr2text(partialreports));	

/* there are two ways how one could proceed : 
   a) Cycle detection
   b) simply adding all large factors to the factorbase
   Here, the second alternative is chosen
*/


if multiplier <> 1 then
        fb:=[2,multiplier].fb.sort([op({op(map(partialreports,op,4))})])
else
        fb:=[2].fb.sort([op({op(map(partialreports,op,4))})])
end_if;

userinfo(4, "Size of extended factorbase: ".expr2text(nops(fb)));
userinfo(4, "Number of useful sieve reports: ".expr2text(nops(l)));

if collectinformation then
   result:=append(result,"Size of extended factorbase:".expr2text(nops(fb)));
   result:=append(result,"Number of useful sieve reports:".expr2text(nops(l)));
end_if;


if nops(l)=0 then
  userinfo(4, "Cannot factor because of no sieve reports");
  if collectinformation then
        return(append(result,"No factor found"));
  else
        return(FAIL)
  end_if;
end_if;  


userinfo(2,"Handling negative sieve reports");

H:=map(l,op,5);
r:=select([$1..nops(l)],proc() begin op(l,[args(1),2])=-1 end_proc );
l:=map(l,op,1);
// get positive numbers 
if r<>[] then
   i:=r[1];
   HH:=H[i];
   ll:=l[i];
   map(r,proc() begin
        H[args(1)]:= H[args(1)]*HH mod n; 
        l[args(1)]:= l[args(1)]*ll;
      end_proc
   );
   delete l[i];
   delete H[i];
end_if;

userinfo(5, expr2text(nops(l))." sieve reports left");
userinfo(2,"Creating exponent vectors");
factorizations:=map(l, x -> coerce(ifactor(x), DOM_LIST));

userinfo(4, "Number of different factors in Factorizations of smooth parts:");
userinfo(4, map(factorizations, proc(x) begin nops(x) div 2 end_proc));

// create lists of odd exponents 

if gaussian then

ll:=map(factorizations, proc(factli)
        local i, result;

        begin
        result:=[];
        for i from 2 to nops(factli) step 2 do
                if factli[i+1] mod 2=1 then
                        result:=append(result, factli[i])
                end_if
        end_for; 
        result
        end_proc);
                
else

//  transpose

ll:=[ {} $ nops(fb)];

cutoff:=min(nops(factorizations), nops(fb)+10); // should suffice 

userinfo(3, "using only ".expr2text(cutoff). " reports");

// create a matrix with nops(fb) rows and cutoff columns 

userinfo(3,"Creating ".expr2text(nops(fb))." times ".expr2text(cutoff).
" matrix");

for i from 1 to cutoff do
  for j from 2 to nops(factorizations[i])-1 step 2 do
    if factorizations[i][j+1] mod 2=1 then
      contains(fb, factorizations[i][j]);
      ll[%]:=ll[%] union {i}
    end_if
  end_for
end_for;



userinfo(2,"Solving linear system");
userinfo(10, "System is ".expr2text(ll));

 

l:=Dom::SparseMatrixF2(nops(fb), cutoff, ll);


for i from 1 to 5 do // should suffice 

  userinfo(3, expr2text(i). "th attempt");
  st:=time();
  ll:=linalg::wiedemann(l, Dom::SparseMatrixF2(nops(fb), 1, [{}$nops(fb)]));
  userinfo(1, "Time for Wiedemann algorithm:".expr2text(time()-st));
  if ll=FAIL then
    userinfo(3, "Wiedemann algorithm failed");
    next;
  end_if;
  ll:=op(ll,1);
  if iszero(ll) then
    userinfo(3, "Wiedemann algorithm produced trivial solution");
    next;
  end_if;

  ll:=select([$1..cutoff], proc(j) begin not iszero(ll[j]) end_proc);
  userinfo(5, "Sieve reports to combine: ".expr2text(ll));
  
  HH:=1;
  r:=table();     
  for j from 1 to nops(ll) do
    HH:=HH*H[ll[j]] mod n;
    hilf:=factorizations[ll[j]];
    for k from 2 to nops(hilf) step 2 do
      if type(r[hilf[k]])="_index" then
        r[hilf[k]]:=hilf[k+1]
      else
        r[hilf[k]]:=r[hilf[k]]+hilf[k+1]
      end_if
    end_for;        
  end_for;
  map(r,_divide,2);
  map([op(%)], proc()
               begin
                 powermod(op(args(1),1),op(args(1),2), n)
               end_proc);
  1; for j in %2 do %*j mod n end_for;    
  // now l^2 \equiv H[i]^2 \bmod n 
  r:=igcd(HH-%,oldn);
  if r=1 or r=abs(oldn) then
    userinfo(2,"Solution number ".expr2text(i)." unusable")
  elif collectinformation then           
    return(append(result," Solution: ".expr2text(r)));
  else
    return(r)
  end_if;
  
end_for;

userinfo(3, "Time for solving linear system: ".expr2text(time()-st));

end_if;

if gaussian then


  
  xorlist:=
  proc(l1,l2)
  begin
    select(l1, proc(x) begin contains(l2, x)=0 end_proc).
    select(l2, proc(x) begin contains(l1, x)=0 end_proc)
  end_proc:
  
  userinfo(20, "System is ".expr2text(ll));
  resu:=[];
  delete i;
  vecs:=[[i]$i=1..nops(ll)];
  for i from 1 to nops(ll) do
    userinfo(5, "Working on ".expr2text(i)." th row");
    if ll[i]=[] then
      resu:=append(resu, vecs[i]);
      if nops(resu) > 10 then
        userinfo(5, "Enough solutions found");
        break
      end_if;  
    else
      pivot:=ll[i][nops(ll[i])];
      userinfo(5, "Pivot is ".expr2text(pivot));
      for j from i+1 to nops(ll) do
        if has(ll[j], pivot) then
          vecs[j]:=xorlist(vecs[i], vecs[j]);
          ll[j]:=xorlist(ll[j],ll[i]);
        end_if
      end_for;
    end_if;
  end_for;
  
  ll:=resu;



  userinfo(1,"Time for solving linear system:".expr2text(time()-st));
  userinfo(2,"Number of solutions: ".expr2text(nops(ll)));
  if collectinformation then
    result:=append(result,"Number of Solutions:".expr2text(nops(ll)));
  end_if;

  userinfo(20, "Solutions are: ".expr2text(ll));

  if ll=[] then
    if collectinformation then
      return(append(result,"No factor found"));
    else
      return(FAIL)
    end_if;
  end_if;

  for i from 1 to nops(ll) do
    HH:=1;
    r:=table();
    for j from 1 to nops(ll[i]) do
      HH:=HH*H[ll[i][j]] mod n;
      hilf:=factorizations[ll[i][j]];
      for k from 2 to nops(hilf) step 2 do
        if type(r[hilf[k]])="_index" then
          r[hilf[k]]:=hilf[k+1]
        else
          r[hilf[k]]:=r[hilf[k]]+hilf[k+1]
        end_if
      end_for;
    end_for;
    map(r,_divide,2);
    map([op(%)], proc()
                 begin
                   powermod(op(args(1),1),op(args(1),2), n)
                 end_proc);
    1; for j in %2 do %*j mod n end_for;
    // now l^2 \equiv H[i]^2 \bmod n
    r:=igcd(HH-%,oldn);
    if r=1 or r=abs(oldn) then
      userinfo(2,"Solution number ".expr2text(i)." unusable")
    elif collectinformation then
      return(append(result," Solution: ".expr2text(r)));
    else
      return(r)
    end_if;

  end_for;

end_if; // gaussian


if collectinformation then
   return(append(result,"No factor found"));
else
   return(FAIL)
end_if;

end_proc:


unalias(FBSIZE):
unalias(PRECMULT):
unalias(ARRSIZE):
unalias(NPOLYS):

