/* ---------------------------------------------------------------------------------------
   'ode::integratingFactor' -- computing integrating factors of ODEs
   ---------------------------------------------------------------------------------------
 
   BACKGROUND: This method does the necessary argument checking. Depending on the order 
               of the input ODE it calls the internal methods/heuristics to detect 
               integrating factors.  
 
   ---------------------------------------------------------------------------------------
 
   ARGUMENTS: Either an ordinary differential equation (ODE) of type 'ode' or 
              an equation/expression encoding an ODE 
 
   ---------------------------------------------------------------------------------------
 
   EXAMPLES: 

    Documented interface calls: 

     >> ode::integratingFactor(ode(1/x*diff(y(x),x)=1/x*y(x),y(x)))
          x exp(-x)
     
     >> ode::integratingFactor(2*y(x)*(1-y(x))*diff(y(x),x,x)-(1-2*y(x))*diff(y(x),x)^2+
                               y(x)*(1-y(x))*diff(y(x),x)*f(x), y(x))
                1
          -------------
          diff(y(x), x) 
     
     >> ode::integratingFactor(exp(y(x))+1/x*diff(y(x),x)+diff(y(x),x$2),y(x))
           2
          x  diff(y(x), x) + 2 x

     
    Undocumented internal call for passing 'solveOptions' and 'odeOptions': 

     >> ode::integratingFactor(exp(y(x))+1/x*diff(y(x),x)+diff(y(x),x$2),y,x,{},{})
           2
          x  diff(y(x), x) + 2 x

  --------------------------------------------------------------------------------------- */

ode::integratingFactor:= proc()
  local eq,y,x,mu,components,n,odeOptions,solveOptions;
begin 
  // Argument checking: note that 'ode::integratingFactor' is a documented function 
  //                    of the interface
  solveOptions:= {};
  odeOptions:= {};
  case args(0) 
  of 0 do 
    error("expecting at least one argument (an ordinary differential equation)");
  of 1 do // valid interface call
    if type(args(1)) <> ode then 
      error("unexpected argument: expecting an ordinary differential equation");
    else 
      components:= ode::getComponents(args());
      eq:= components[1];
       y:= components[2];
       x:= components[3];
    end_if; 
    break;
  of 2 do // valid interface call 
    if type(args(2)) = "function" and nops(args(2)) = 1 then 
      y:= op(args(2),0);
      x:= op(args(2),1);
    else 
      error("expecting a univariate function (dependent variable) as second argument");
    end_if;    
    if type(args(1)) = "_equal" then 
      eq:= lhs(args(1)) - rhs(args(1));
    elif domtype(args(1)) = DOM_EXPR then 
      eq:= args(1);
    else   
      error("expecting an ordinary differential equation as first argument")  
    end_if;      
    break;
  of 5 do // undocumented call for providing 'solveOptions' and 'odeOptions' 
    if domtype(args(1)) = DOM_EXPR and type(args(2)) = DOM_IDENT and 
       type(args(3)) = DOM_IDENT and type(args(4)) = DOM_SET and type(args(5)) = DOM_SET then 
      eq:= args(1);
       y:= args(2);
       x:= args(3);
      solveOptions:= args(4);
      odeOptions:= args(5);
      break;
    else 
      return(FAIL);
    end_if;
  otherwise 
    error("unexpected number of arguments");
  end_case;  
  n:= ode::order(eq,{y},{},{});
  mu:= FAIL;
  // It tuns out that properties are not helpful in the context 
  // of these algorithms. Using the standard property 'x in R_' 
  // slows down the simplification steps applied by the following
  // algorithms significantly. 
  unassume(x);
  if n = 1 then
    // Note that the old integrating factor methods always need to be applied
    // for compatibility reasons. But the new methods will only be used, if 
    // the option 'Type = IntegratingFactor' is used. 
    mu:= ode::integratingFactorsOrder1_1(eq,y,x,solveOptions,odeOptions);
    if contains(odeOptions, Type = IntegratingFactor) then 
      if mu = FAIL then 
        mu:= ode::integratingFactorsOrder1_2(eq,y,x,solveOptions,odeOptions);
      end_if;  
      //
      // NOTE: We do not use 'ode::integratingFactorsOrder1_3' since it does 
      // not produce helpful integrating factors, i.e. integrating factors are 
      // found, but they cannot be used to integrate any of the equations that
      // are of interest so far. 
      //
      //  if mu = FAIL then 
      //    mu:= ode::integratingFactorsOrder1_3(eq,y,x,solveOptions,odeOptions);
      //  end_if;
      //
      if mu = FAIL then 
        // muss vermutlich ausgeknippst werden
        mu:= ode::integratingFactorsOrder1_4(eq,y,x,solveOptions,odeOptions);
      end_if;  
      if mu = FAIL then 
        mu:= ode::integratingFactorsOrder1_5(eq,y,x,solveOptions,odeOptions);
      end_if;  
      if mu = FAIL then 
        // Do not use this currently - even if the option 
        //     'Type = IntegratingFactor'
        // is given. 
        // 
        // mu:= ode::integratingFactorsOrder1_6(eq,y,x,solveOptions,odeOptions);
      end_if;  
      if mu = FAIL then 
        mu:= ode::integratingFactorsOrder1_7(eq,y,x,solveOptions,odeOptions);
      end_if;  
    end_if;    
  elif n = 2 then 
    // Note that the old integrating factor methods always need to be applied
    // for compatibility reasons. But the new methods will only be used, if 
    // the option 'Type = IntegratingFactor' is used. 
    mu:= ode::integratingFactorsOrder2_1(eq,y,x,solveOptions,odeOptions);
    if mu = FAIL and contains(odeOptions, Type = IntegratingFactor) then 
      // further methods in the future
    end_if;  
  end_if;
 
  return(mu);
end_proc:


