/* 
   ode::integratingFactorsOrder2_1(eq,y,x)

   REFERENCE: [1] E.S. Cheb-Terrab and A.D. Roche: "Integrating factors for 
                  second order ODEs", J. Symb. Comp. 27(5), 505-519, 1999.

   DESCRIPTION: 

   ode::int_factors2(eq,y,x) tries to solve the second order ODE eq with
   respect to y(x) using the method of integrating factors (see Murphy p165).

   Output: either FAIL or a set of solutions.

   DETAILS: 

   The method used to find an integration factor is based on the paper [1]. 
   Once one has found an integration factor mu, then mu*eq is exact.
   The notations are, as far as possible, those used in the article and
   so we refer to the paper for the explanations of the algorithm.
 
   EXAMPLES: 

   1) Equation of Liouville's type:
        
        >> ode::solve(2*y(x)*(1-y(x))*diff(y(x),x,x)-(1-2*y(x))*diff(y(x),x)^2+
                      y(x)*(1-y(x))*diff(y(x),x)*f(x), y(x));

   2) Integrating factor :

        >> ode::solve((x+y(x)^2)*diff(y(x),x,x)-2*(x-y(x)^2)*diff(y(x),x)^3+
                      diff(y(x),x)*(1+4*y(x)*diff(y(x),x)), y(x));
*/

ode::integratingFactorsOrder2_1 :=
proc(eq,y,x,solveOptions,odeOptions)
  local q,eq0,parameq,Phi,Phi0,found,a,b,c,X,Y,phi,tau,A,B,nu,paramnu,nu1,nu2,
        denomnu,sing,eq1,eq2,firint1,firint2,dery,sys,res,gama,intay,tmp,F,mu,
        Fmu,muc,H,Hy,Hp,mutilde,foncF,elim,err,intOptions,optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                                   IgnoreAnalyticConstraints;
                                 else
                                   null();
                                 end_if;  
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  eq0:= eq;
  parameq:= indets(eq) minus {x,y,PI,CATALAN,EULER};
  userinfo(2,"trying to find an integrating factor");
  eq:= subs(eq,diff(y(x),x$2)=`#yp`,diff(y(x),x)=`#p`,y(x)=y);
  if testtype((eq:=poly(eq,[`#yp`])),Type::PolyExpr(`#yp`)) and degree(eq)=1 then
    Phi:= ode::normal(-coeff(eq,`#yp`,0)/coeff(eq,`#yp`,1),Expand=FALSE);
    Phi0:= Phi;
    found:= FALSE;
    if testtype(Phi,Type::PolyExpr(`#p`)) and degree((Phi:=poly(Phi,[`#p`])))<=2 then
      a:= ode::normal(coeff(Phi,2)); 
      b:= ode::normal(coeff(Phi,1));
      c:= ode::normal(coeff(Phi,0));
      if iszero(c) and not has(a,x) and not has(b,y) then
        // The ODE is a Liouville's equation (c.f. Murphy p. 385, Example 47) and so we can
        // solve it directly.
        found:= TRUE;
        userinfo(2,"Liouville's equation: integrating factor is :".
        " 1/diff(y(x),x)");
        if contains(odeOptions,"ReturnIntegratingFactorOnly") then 
          return(1/diff(y(x),x));  
        end_if;  
        X:= int(-b,x,intOptions);
        Y:= int(-a,y,intOptions);
        res:= solvelib::discreteSolve(int(exp(Y),y,intOptions) + 
                    genident("C")*int(exp(-X),x,intOptions) + 
                    genident("C"),y,op(solveOptions)):
        if not has(res, FAIL) then
          return(res union {genident("C")})
        else
          userinfo(2,"Liouville's equation failed : problem with solve")
        end_if;
      elif iszero(ode::normal(2*diff(a,x)-diff(b,y))) then
        // In this case we have two integrating factors of the form mu(x,y)
        phi:= diff(c,y)-a*c;
        if iszero(ode::normal(diff(a,x$2)-diff(a,x)*b-diff(phi,y))) then
          found:= TRUE;
          userinfo(2,"two integrating factors of the form mu(x,y)");
          tau:= diff(int(a,y),x,intOptions);
          A:= 2*tau-b;
          B:= phi+tau*(b-tau)-diff(b-tau,x);
          nu:= genident();
          eq:= diff(nu(x),x$2)-A*diff(nu(x),x)-B*nu(x);
          userinfo(2,"solving linear 2nd ODE for the integrating factors:",eq=0);
          sysassign(ode::depth,ode::depth+1);
          nu:= ode::solve_eq(eq,nu,x,{},solveOptions,odeOptions);
          sysassign(ode::depth,ode::depth-1); 
          if nu<>{FAIL} and type(nu)=DOM_SET and nops((nu:= select(nu,has,x)))>0 then
            nu:= map({nu[1]},_mult,exp(-int(a,y,intOptions)));
            paramnu:=indets(nu) minus {x,y,PI,CATALAN,EULER} minus parameq;
            nu1:=simplify(op(evalAt(nu, [paramnu[1]=0, paramnu[2]=1])),optIgnoreAnalyticConstraints);
            nu2:=simplify(op(evalAt(nu, [paramnu[1]=1, paramnu[2]=0])),optIgnoreAnalyticConstraints);
            userinfo(2,"the two integrating factors found are:", nu1, nu2);
            if contains(odeOptions,"ReturnIntegratingFactorOnly") then 
              return(nu1,nu2);
            end_if;  

            //            when multiplying by the integrating factor nu1 one may lose some
            //            solution of the initial DE eq0: so one has to check this !
            denomnu:=denom(ode::normal(nu1));
            sing:=solvelib::discreteSolve(denomnu, y, op(solveOptions));
            sing:=select(sing,
            proc(s)
            begin
              if traperror((res:=subs(eq0, y(x)=s, EvalChanges)))=0 then
                return(iszero(expand(res,optIgnoreAnalyticConstraints)))
              end_if:
              FALSE
            end_proc):
            
            //            We have two different ways now to proceed the resolution :
            //            - compute the two first integrals with respect to nu1 and nu2 and
            //              eliminate the derivative y' in these 2 equations to get the
            //              solution of the original equation
            //            - solve one the above first integral
            
            eq1:=subs(nu1, y=y(x))*(diff(y(x),x$2)-subs(op(Phi)[1],y=y(x),
            `#p`=diff(y(x),x)));
            eq2:=subs(nu2, y=y(x))*(diff(y(x),x$2)-subs(op(Phi)[1],y=y(x),
            `#p`=diff(y(x),x)));
            //           eq1 and eq2 are both exact differential equations
            
            if (firint1:=ode::firint_second_order(eq1,y,x,
              solveOptions,odeOptions)) = FAIL or
              (firint2:=ode::firint_second_order(eq2,y,x,
              solveOptions,odeOptions)) = FAIL then
              return(FAIL)
            end_if;
            firint1:=subs(firint1, diff(y(x),x)=`#p`, y(x)=y, EvalChanges);
            firint2:=subs(firint2, diff(y(x),x)=`#p`, y(x)=y, EvalChanges);
            dery:=solvelib::discreteSolve(firint1,`#p`,op(solveOptions));
            if type(dery)=DOM_SET then
              sys:={evalAt(firint2, `#p`=q) $ q in dery};
              err:=traperror((res:=_union(
              solvelib::discreteSolve(numer(simplify(q,optIgnoreAnalyticConstraints)),y,op(solveOptions)) 
              $ q in sys)));
              if err=0 and not has(res, FAIL) then
                return(map(res,simplify,optIgnoreAnalyticConstraints) union sing);
              end_if:
            else
              userinfo(2,"solving new equation :", firint1=0);
              return(sing union ode::solve_eq(firint1,y,x,{},solveOptions,odeOptions))
            end_if:
          else
            userinfo(2,"integrating factors failed");
            return(FAIL);
          end_if;
        end_if;
      else
        // In this case we have one integrating factor of the form mu(x,y).        
        phi:= diff(c,y)-a*c-diff(b,x);
        gama:= diff(a,x$2)+diff(a,x)*b+diff(phi,y);
        if iszero(ode::normal(diff(gama,y)-diff(a,x))) and iszero(ode::normal(diff(gama,x)+phi+b*gama-gama^2)) then
          found:= TRUE;
          intay:= int(a,y,intOptions);
          mu:= exp(int(-gama+diff(intay,x),x,intOptions)-intay);
          if mu<>1 then
            userinfo(2,"one integrating factor of the form mu(x,y)");
            userinfo(2,"integrating factor found is:", mu);
            if contains(odeOptions,"ReturnIntegratingFactorOnly") then 
              return(mu);
            end_if;  
            eq:= subs(mu,y=y(x))*(diff(y(x),x$2)-subs(op(Phi)[1],y=y(x),`#p`=diff(y(x),x)));
            userinfo(2,"solving new equation :",eq=0);
            // When multiplying by the integrating factor mu one may lose some
            // solution of the initial ODE eq0: so one has to check this!
            denomnu:= ode::normal(mu,List,Expand=FALSE);
            sing:= solvelib::discreteSolve(denomnu, y, op(solveOptions));
            sing:= select(sing,
                          proc(s)
                          begin
                            if traperror((res:=subs(eq0,y(x)=s,EvalChanges)))=0 then
                              return(iszero(expand(res,optIgnoreAnalyticConstraints)));
                            end_if;
                            return(FALSE);
                          end_proc):
            if has((res:=ode::exact_second_order(eq,y,x,solveOptions,odeOptions)),FAIL) then
              userinfo(2,"integrating factor method failed but found a first".
                         " order integral");
            else
              return(res union sing);
            end_if;
          end_if;
        end_if;
      end_if;
    end_if;
    
    //================================== SUBROUTINES ==================================
    
    // The following procedure mutilde computes \tilde{mu}(x) from Lemma 2 of [1]. 
    mutilde:= proc(phi,F,x,y,p)
      local phiy,phi1,phi2,phi3,phi4,res;
    begin
      phiy:= diff(phi,y);
      phi2:= ode::normal(diff(phiy*F,p));
      phi1:= ode::normal(phiy*F-p*phi2,Expand=FALSE);
      phi3:= ode::normal(-diff(phi*F,p),Expand=FALSE);
      phi4:= ode::normal(diff(F,p),Expand=FALSE);
      if not iszero(phi2) then
        if not has((res:= ode::normal(1/phi2*(diff(phi1,y)-diff(phi2,x)))),{p,y}) then
          return(simplify(exp(int(res,x,intOptions)),optIgnoreAnalyticConstraints));
        end_if;
      else
        if not has((res:= ode::normal(1/phi4*(diff(phi3,p)-diff(phi4,x)))),{p,y}) then
          return(simplify(exp(int(res,x,intOptions)),optIgnoreAnalyticConstraints));
        end_if;
      end_if;
      
      return(FAIL);
    end_proc:
        
    // The following procedure foncF computes {\cal F}(x,y') from Lemma 3 in [1].
    // Notations are, as far as possible, the same as the article.
    foncF:= proc(phi,x,y,p)
      local gama,fgama,gamay,fgamap,i,w,wx,wy,pp,C1,lambda,Psi,lambdax,lambdap,
            Psix,Psip,ps,eq;
    begin
      gama:= ode::normal(diff(phi,y),Expand=FALSE);
      fgama:= factor(gama);
      fgama:= coerce(fgama, DOM_LIST);
      pp:= genident("p"); 
      ps:= genident();
      if not iszero(gama) then        
        gamay:= ode::normal(diff(gama,y));
        if not iszero(ode::normal(diff(gamay/gama,p))) then // Case A
          fgamap:= select([i $ i=1..nops(fgama) div 2], 
                          () -> has(fgama[2*args(1)],p) and not has(fgama[2*args(1)],y));
          return(expand(_mult(fgama[1]*fgama[2*i]^(-fgama[2*i+1]) $ i in fgamap),optIgnoreAnalyticConstraints))
        else
          fgamap:= select([i $ i=1..nops(fgama) div 2], 
                          ()-> has(fgama[2*args(1)],y) and not has(fgama[2*args(1)],p));
          w:= expand(_mult(fgama[1]*fgama[2*i]^(fgama[2*i+1]) $ i in fgamap),optIgnoreAnalyticConstraints);
          if not iszero(gamay) then            
            if not iszero(ode::normal(diff(diff(ln(w),y),x))) and not iszero(ode::normal(diff(diff(ln(w),y),y))) then // Case C
              wx:= diff(w,x); 
              wy:= diff(w,y);
              pp:= ode::normal((diff(wx,y)*w-wx*wy)/(diff(wy,y)*w-wy^2),Expand=FALSE);
              return(ode::normal(w*(pp+p)/gama));
            elif not has((C1:=diff(ln(w),y)), {x,y,p}) then // Case E
              lambda:= ode::normal(C1*exp(C1*y)/gama,Expand=FALSE);
              Psi:= ode::normal(phi*lambda-C1*exp(C1*y),Expand=FALSE);
              lambdax:= diff(lambda,x); 
              lambdap:= diff(lambda,p);
              Psix:= diff(Psi,x); 
              Psip:= diff(Psi,p);
              eq:= ode::normal((ps+pp^2*C1)*lambdap+pp*(p*lambdap*C1+lambda*C1+
                          diff(lambdax,p)+diff(Psip,p))+2*Psip+lambdax+
                          p*diff(lambdax,p)+p*diff(Psip,p),Expand=FALSE);
              if iszero(ode::normal(lambdap)) then
                pp:= op(solvelib::discreteSolve(factor(eq),pp,op(solveOptions)));
              else
                eq:= ode::normal(eq/lambdap);
                if has(eq,p) then
                  pp:= op(solvelib::discreteSolve(factor(ode::normal(diff(eq,p))),pp,op(solveOptions)));
                else  // this is case F : not yet implemented...
                  pp:= FAIL;
                end_if;
              end_if;
              if pp<>FAIL and not has(pp,{y,p}) then
                return(ode::normal(w*(pp+p)/gama));
              else
                return(FAIL);
              end_if
            end_if;
          else // Case D
            lambda:= ode::normal(1/gama,Expand=FALSE);
            Psi:= ode::normal(1/gama*phi-y,Expand=FALSE);
            lambdax:= diff(lambda,x); 
            lambdap:= diff(lambda,p);
            Psix:= diff(Psi,x); 
            Psip:= diff(Psi,p);
            if iszero(ode::normal(lambdap)) then
              tmp:= diff(lambdax,p)+diff(Psip,p);
              if not iszero(tmp) then 
                pp:= ode::normal(-p-(lambdax+2*Psip)/tmp);
              else 
                return(FAIL);
              end_if;  
            elif not iszero(lambdap) then 
              ps:= ode::normal(-((diff(lambdax,p)+diff(Psip,p))*(p+pp(x))+lambdax+2*Psip)/lambdap,Expand=FALSE);
              eq:= ode::normal(lambda*diff(ps,x)+(diff(lambdax,x)+diff(Psip,x))*(p+pp(x))+
                          (lambdax+Psip)*ps+Psix-pp(x),Expand=FALSE);
              eq:= subs(eq,diff(pp(x),x)=ps);
              pp:= op(solvelib::discreteSolve(eq,pp(x),op(solveOptions)));
            end_if;
            if not has(pp, {y,p}) then
              return(ode::normal(w*(pp+p)/gama));
            end_if;
          end_if;
        end_if;  
        
      end_if;
      return(FAIL);
    end_proc:
    
    //================================== END OF SUBROUTINES ==================================
    
    // If we have not found an integrating factor of the form mu(x,y) so for. Now we look 
    // for one of the form mu(x,y').
    
    if not found and ((F:= foncF(Phi0,x,y,`#p`)))<>FAIL then
      if (mu:= mutilde(Phi0,F,x,y,`#p`))<>FAIL then
        // We have now to check for exactness condition by applying Euler's operator
        // to the total derivative of H = mu*(y''-phi) :
        H:= ode::normal(F*mu*(`#y2`-Phi0),Expand=FALSE);
        Hy:= subs(diff(H,y),y=y(x),`#p`=diff(y(x),x),`#y2`=diff(y(x),x$2));
        Hp:= subs(rewrite(diff(H,`#p`),D),y=y(x),`#p`=diff(y(x),x),`#y2`=diff(y(x),x$2));
        Fmu:= ode::normal(F*mu,Expand=FALSE);
        muc:= subs(Fmu,y=y(x),`#p`=diff(y(x),x));
        if muc<>1 and iszero(ode::normal(Hy-diff(Hp,x)+diff(muc,x$2))) then
          userinfo(2,"integrating factor found is:",muc);
          if contains(odeOptions,"ReturnIntegratingFactorOnly") then 
            return(muc);
          end_if;  
          eq:= muc*(diff(y(x),x$2)-subs(Phi0,y=y(x),`#p`=diff(y(x),x)));
          userinfo(2,"solving new equation :", eq=0);          
          denomnu:= ode::normal(Fmu,List,Expand=FALSE)[2];
          if has(denomnu,`#p`) then
            sing:= solve(denomnu,`#p`,op(solveOptions));
            if type(sing) <> DOM_SET then
              return(FAIL);
            end_if;
            sing:= map(sing,int,x);
            sing:= map(sing,_plus,genident("C"));
          elif has(denomnu, y) then
            sing:= solvelib::discreteSolve(denomnu,y,op(solveOptions));
          else
            sing:= {};
          end_if:
          sing:= select(sing,
                        proc(s)
                        begin
                          if traperror((res:= subs(eq0,y(x)=s,EvalChanges)))=0 then
                            return(iszero(expand(res,optIgnoreAnalyticConstraints)))
                          end_if:
                          return(FALSE);
                        end_proc):
          if has((res:= ode::exact_second_order(eq,y,x,solveOptions,odeOptions)),FAIL) then
            userinfo(2,"integrating factor method failed but found a first".
                       " order integral")
          else
            // NOTE: we have to check that the solution do not vanish for the integrating factor mu. 
            // Example: For exp(y(x))+1/x*diff(y(x),x)+diff(y(x),x$2) = 0 one obtains two elements 
            //          in res, one of which is not a solution! 
            elim:= proc(l)
                     begin
                       if type(l)=DOM_SET then
                         return(select(l,()-> not(iszero(subs(muc, y(x)=args(1),EvalChanges)))));
                       else
                         return(l);
                       end_if:
                     end_proc:
            if type(res)=DOM_SET then
              return(sing union elim(res))
            elif type(res) = piecewise and map({op(res)},elem -> type(op(elem,2)))={DOM_SET} then 
              return(sing union piecewise::extmap(res,elim));
            else
              return(sing union map(res, elim));
            end_if:
          end_if;
        end_if;
      end_if;
    end_if;
  end_if;
  
  return(FAIL);
end_proc:

