/* 
  ==================================================================================
  BACKGROUND: The methods in this file were created to support the implementations 
              based on the algorithms in 

              [CT1] E.S.Cheb-Terrab, A.D.Roche: Symmetries and First Order 
                    ODE Patterns, Computer Physics Communications, 113 
                    (1998) 239, 1998 
 
              [CT2] E.S.Cheb-Terrab, A.D.Roche: Integrating factors for 
                    second order ODEs, Journal of Symbolic Computation, 
                    Volume 27, pp. 501-519, 1999 

              [Geh] K.F.Gehrs: Algorithmic methods for ordinary differential 
                    equations, PhD thesis, University of Paderborn, 2007

  ==================================================================================
*/

/* 
  FUNCTION:   ode::splitFactor

  PARAMETERS: Phi   -- an expression  
              Vars1 -- a list of variables (DOM_IDENTs)
              Vars2 -- a list of variables (DOM_IDENTs)
              solveOptions -- usual set of options
              odeOptions   -- usual set of options

  DETAILS: This function checks whether 'Phi' can be written 
           as a product of two factors f1*f2, where f1 
           is an expression in the variables from Var1 and 
           f2 is an expression in the variables from Var2
           and the following condition holds: 

              not has(f1,Vars2) and not has(f2,Vars1)
  
           If such a decomposition exists, it returns a list 
           containing the two factors f1 and f2. 

           In case that 'odeOptions' contains the string 
           "AnyFactor", the method tries to compute any factor 
           depending on variables from Var1, but not on variables 
           from Var2. NOTE that this does not imply that the 
           other factor does not contain any of the variables 
           from Var1. In case "AnyFactor" is given and two 
           factors f1 and f2 are returned, the following holds: 
           f1 depends on variables from Var1 and not on variables 
           from Var2, but f2 can contain variables from Var1. 

           If such a decomposition does not exist, it returns
           'FAIL'. 

  EXAMPLES: 

    (1) >> Phi:= a*x^2*1/y*1/c*sqrt(c*y^2)+1/y*sin(x)*sqrt(c*y^2)+1/y*exp(x^2+x+1)*sqrt(c*y^2):
        >> splitFactor(Phi,[x],[y])
            [a*x^2 + c*sin(x) + c*exp(x^2 + x + 1), (c*y^2)^(1/2)/(c*y)]
    
    (2) >> Phi:= x+y+a*x^2*1/y*1/c*sqrt(c*y^2)+1/y*sin(x)*sqrt(c*y^2)+1/y*exp(x^2+x+1)*sqrt(c*y^2):
        >> splitFactor(Phi,[y],[x])
            FAIL   
*/

ode::splitFactor:=
proc(Phi,Vars1,Vars2,solveOptions={},odeOptions={}) 
  local Subst, f1, f2, h1, h2, n, p, q, ratPhi, s, i, j, m,  
        Vars1Extended, Vars2Extended, optIgnoreAnalyticConstraints;
begin 
  if not has(Phi,Vars1) then 
    return([1,Phi]);
  end_if;
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  Vars1Extended:= {op(Vars1)};
  Vars2Extended:= {op(Vars2)};
  [ratPhi,Subst]:= [rationalize(ode::normal(Phi,Expand=FALSE))];
  for s in Subst do 
    if has(rhs(s),Vars1) and has(rhs(s),Vars2) then 
      return(FAIL);
    elif has(rhs(s),Vars1) then 
      Vars1Extended:= Vars1Extended union {lhs(s)}
    elif has(rhs(s),Vars2) then 
      Vars2Extended:= Vars2Extended union {lhs(s)}
    end_if;
  end_for;
  n:= ode::normal(ratPhi,Rationalize=None,Expand=FALSE,List);
  if nops(Vars1Extended) > nops(Vars2Extended) then 
    p:= poly(n[1],[op(Vars1Extended)]);
    q:= poly(n[2],[op(Vars1Extended)]);
    h1:= gcd(coeff(p));
    h2:= gcd(coeff(q));
    f1:= subs(ode::normal(n[1]/n[2]/(h1/h2),Rationalize=None,Expand=FALSE),Subst);
    f2:= subs(h1/h2,Subst); 
  else 
    p:= poly(n[1],[op(Vars2Extended)]);
    q:= poly(n[2],[op(Vars2Extended)]);
    h1:= gcd(coeff(p));
    h2:= gcd(coeff(q));
    f2:= subs(ode::normal(n[1]/n[2]/(h1/h2),Rationalize=None,Expand=FALSE),Subst);
    f1:= subs(h1/h2,Subst); 
  end_if; 
  if not has(f1,Vars2) and not has(f2,Vars1) then 
    return([f1,f2]);
  elif contains(odeOptions,"AnyFactor") then 
    // last resort: call general factoring algorithm to find any factor 
    // depending on variables from Var1, but not on variables from Var2. 
    m:= coerce(factor(f1), DOM_LIST);
    m:= _mult(m[2*i]^(m[2*i+1]) $ i in select({j $ j=1..nops(m) div 2}, 
                                              () -> has(m[2*args(1)],Vars1)));
    if has(m,Vars1) then 
      return([m,ode::normal(f2*f1/m,Rationalize=None,Expand=FALSE)]);
    else
      return(FAIL);
    end_if;
  else 
    return(FAIL);
  end_if;

end_proc:

/* 
  FUNCTION:   ode::separateByProduct

  PARAMETERS: Phi -- an expression  
              y   -- dependent ODE variable (DOM_IDENT)
              x   -- independent ODE variable (DOM_IDENT)
              solveOptions -- usual set of options
              odeOptions   -- usual set of options

  DETAILS: This function checks whether 'Phi' can be written 
           as a product of two factors f1*f2, where  
  
                  diff(f1,y) = 0 = diff(f2,x). 

           If such a decomposition exists, it return a list 
           containing the two factors f1 and f2. 

           If such a decomposition does not exist, it returns
           'FAIL'. 

  EXAMPLES: 

    (1) >> Phi:= a*x^2*1/y*1/c*sqrt(c*y^2)+1/y*sin(x)*sqrt(c*y^2)+1/y*exp(x^2+x+1)*sqrt(c*y^2):
        >> separateByProduct(Phi,y,x)
            [a*x^2 + c*sin(x) + c*exp(x^2 + x + 1), 1/c/y*(c*y^2)^(1/2)]
    
    (2) >> Phi:= x+y+a*x^2*1/y*1/c*sqrt(c*y^2)+1/y*sin(x)*sqrt(c*y^2)+1/y*exp(x^2+x+1)*sqrt(c*y^2):
        >> separateByProduct(Phi,y,x)
            FAIL   
*/

ode::separateByProduct:=
proc(Phi,y,x,solveOptions={},odeOptions={}) 
begin
  return(ode::splitFactor(Phi,[x],[y],solveOptions,odeOptions));
end_proc:


/*
    Implementing Euler's operator (testing for exactness of an ODE)
  
    PARAMETER: 
                eq -- an expression in 'y(x)' and derivatives of 'y(x)' with 
                      respect to 'x' encoding an ordinary differential 
                      equation 
                 y -- dependent variable
                 x -- indepenent variable 
      solveOptions -- options for 'solve'
        odeOptions -- options for 'ode' 

    RETURN VALUE:  The method returns the differential expression in 'y(x)' 
                   and dervitatives of 'y(x)' with respect to 'x' obtained 
                   when applying Euler's operator to 'eq'.

                   NOTE: This especially means that the return value '0' 
                         means that 'eq' is exact, i.e. can be reduced in 
                         its order directly via integration. 

    REFERENCE:     See Section 2.5, pp. 78 of 

                   [1] K.F. Gehrs: Algorithmic methods for ordinary 
                                   differential equations, PhD thesis, 
                                   University of Paderborn, 2007                   
*/

ode::eulerOperator:= proc(eq,y,x,solveOptions={},odeOptions={})
  local Y, eqx, n, list_fi, list_fix, list_Dfi, i, j, k, res, intOptions,
        optIgnoreAnalyticConstraints, tmp; 
begin
  if not has(eq,y) then 
    if has(odeOptions,"ExactnessTest") then 
      return(TRUE);
    else   
      return(0);
    end_if;  
  end_if;
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;  
  intOptions:= null();             
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  n:= ode::order(eq,{y},solveOptions,odeOptions);
  Y:= genident("y"); 
  eq:= subs(eq, {diff(y(x),x$i) = Y[i] $ i=0..n});
  eqx:= subs(eq, {Y[i] = Y[i](x) $ i=0..n});
  list_fi:= [diff(eq,Y[i]) $ i=0..n];
  list_fix:= subs(list_fi, {Y[i] = Y[i](x) $ i=0..n});
  list_Dfi:= [];
  // For reference on the formula used below see [1], Section 2.5, pp. 78, 
  // formula (2.5.1).
  for i from 1 to n do
    list_Dfi:= append(list_Dfi, subs(diff(list_fix[i+1],x$i),
                             {diff(Y[j](x),x$k) = Y[j+k](x) $ k=1..i $ j=0..n}));
  end_for:
  res:= list_fix[1]+_plus((-1)^i*list_Dfi[i] $ i=1..n);
  // I experimented a lot also with using 'testeq', 'simplify', 'Simplify' 
  // etc. The combination of first 'expand' and then 'normal' turned out to 
  // be most suitable. 
  if not has(odeOptions,"ExactnessTest") then 
    res:= expand(res,optIgnoreAnalyticConstraints);
    res:= ode::normal(subs(res, {Y[i](x) = diff(y(x),x$i) $ i=0..n+1}));
    return(res);
  else 
    if ode::odeIszero(res) then 
      if traperror((tmp:= ode::normal(subs(expand(res,optIgnoreAnalyticConstraints),
                            {Y[i](x) = diff(y(x),x$i) $ i=0..n+1}))),MaxSteps=5) = 0 and 
         iszero(tmp) then 
        return(TRUE);
      end_if; 
      // NOTE: it may be necessary to decrease/increase the value of 'MaxSteps' in the 
      //       following call of 'simplify'. 
      if traperror((tmp:= simplify(ode::normal(subs(res,{Y[i](x) = diff(y(x),x$i) $ i=0..n+1})),
                                  optIgnoreAnalyticConstraints)),MaxSteps=120) = 0 and 
         iszero(tmp) then 
        return(TRUE);
      end_if;
      return(FALSE);
    else
      return(FALSE);
    end_if;  
  end_if;    
      
end_proc:


/*
    Check for quasi-linear ODE
  
    PARAMETER: 
                eq -- an expression in 'y(x)' and derivatives of 'y(x)' with 
                      respect to 'x' encoding an ordinary differential 
                      equation 
                 y -- dependent variable
                 x -- indepenent variable 
      solveOptions -- options for 'solve'
        odeOptions -- options for 'ode' 

    RETURN VALUE:  TRUE/FALSE
*/

ode::isQuasiLinear:= proc(eq,y,x,solveOptions={},odeOptions={})
   local yn,n,tmp;
begin
  yn:= genident();
  n:= ode::order(eq,{y},solveOptions,odeOptions);
  tmp:= subs(eq,diff(y(x),x$n)=yn,EvalChanges);
  if iszero(diff(tmp,yn)-1) then 
    return(TRUE);
  else   
    return(FALSE);  
  end_if;
end_proc:

/*
    Compute quasi-linear form of an ODE. 
  
    PARAMETER: 
                eq -- an expression in 'y(x)' and derivatives of 'y(x)' with 
                      respect to 'x' encoding an ordinary differential 
                      equation 
                 y -- dependent variable
                 x -- indepenent variable 
      solveOptions -- options for 'solve'
        odeOptions -- options for 'ode' 

    RETURN VALUE:  Quasi-linear form of the ODE (solved for highest derivative 
                   with coefficient of highest derivative =1) or FAIL. 

                   In detail:  

                        [  y^(n) - Phi(x,y,y',...,y^(n-1)),
                           Phi(x,y,y',...,y^(n-1)),
                           lcf, 
                           y,x,solveOptions,odeOptions    ]

                   where 'lcf' is the coefficient of 'y^(n)' in 'eq'.
                   In case of an ODE 'eq', where the highest deriviative 
                   appears in non-linear form, 'lcf' is FAIL. The reason is 
                   that the value of 'lcf' is used by the integrating factor
                   methods to compensate the transformation from 'eq' to its
                   quasi-linear form. The case 'lcf = FAIL' indicates that 
                   there is no "inverse transformation" available that helps 
                   to compute an integrating factor of 'eq' out of an 
                   integrating factor of its quasi-linear form.  
*/

ode::quasiLinearForm:= proc(eq,y,x,solveOptions={},odeOptions={})
  local yn,n,tmp,tmp2,lcf,Phi;
begin
  // Note: This procedure uses 'prog::remember' (see below). 
  //       This is absolutely necessary, since it is possibly 
  //       called several times with the same arguments. 
  yn:= genident();
  n:= ode::order(eq,{y},solveOptions,odeOptions);
  if ode::isQuasiLinear(eq,y,x,solveOptions,odeOptions) then  
    Phi:= diff(y(x),x$n)-eq;
    if not has(Phi,diff(y(x),x$n)) then 
      return([eq,Phi,1,y,x,solveOptions,odeOptions])
    else 
      Phi:= simplify(Phi);
      if not has(Phi,diff(y(x),x$n)) then 
        return([eq,Phi,1,y,x,solveOptions,odeOptions])
      end_if;
    end_if;  
  elif ode::depth>0 then 
    // This is the sitatuation where the ODE solver has called himself 
    // recursively. This means that it is working on a potentially 
    // very complicated automatically generated ODE. We currently 
    // only proceed if the ODE is at least already in quasilinear
    // form. 
    return(FAIL);
  end_if;
  tmp:= subs(eq,diff(y(x),x$n)=yn,EvalChanges);
  if not has(((lcf:= diff(tmp,yn))),yn) and not iszero(lcf) then 
    tmp2:= (1/lcf*eq);
    Phi:= diff(y(x),x$n)-tmp2;
    if not has(Phi,diff(y(x),x$n)) then 
      return([diff(y(x),x$n)-Phi,Phi,lcf,y,x,solveOptions,odeOptions]);
    else
      Phi:= simplify(Phi);
      if not has(Phi,diff(y(x),x$n)) then 
        return([diff(y(x),x$n)-Phi,Phi,lcf,y,x,solveOptions,odeOptions])
      end_if;
    end_if;  
  end_if;
  // Too costly at this time!
  // 
  //L:= solve(tmp,yn,op(solveOptions union {IgnoreSpecialCases}));
  //if type(L) = DOM_SET then 
  //  tmp:= L[1];
  //  return([diff(y(x),x$n)-tmp,tmp,FAIL,y,x,solveOptions,odeOptions]);
  //end_if;
  // 
  return(FAIL);
end_proc:

ode::quasiLinearForm:= prog::remember(ode::quasiLinearForm, property::depends):

/*
    Simplify expressions of the form 'exp(ln(...))'. 
  
    PARAMETER: 
                eq -- an expression 

    RETURN VALUE:  Simplified version of 'eq' obtained by applying 'simplify::exp'
                   in case that 'eq' contains an expression of the form 'exp(ln(..))'
                   and no functions that are "hard to evaluate" in the sense of 
                   'ReplaceHardToEval'. Otherwise the expression 'eq' is returned 
                   without any modifications applied to it.                
*/

/*
ode::simplifyExpOfLn:= proc(eq) 
begin   
  if testtype(eq,Type::Arithmetical) and 
     rationalize(eq,DescendInto=TRUE,ReplaceHardToEval)[2] = {} then  
    eq:= misc::maprec(eq,
                      {"exp"} = proc(elem)
                                begin
                                  if has(elem,ln) then 
                                    return(exp::simplify(elem));
                                  else
                                    return(elem);
                                  end_if;  
                                end_proc):
  end_if;                              

  return(eq);
end_proc:
*/

/*
ode::simplifyExpOfLn:= proc(eq) 
  local foundExpOfLn;
begin   
  if testtype(eq,Type::Arithmetical) then 
    if rationalize(eq,DescendInto=TRUE,ReplaceHardToEval)[2] = {} then  
      foundExpOfLn:= FALSE;
      misc::maprec(eq,
                   {"exp"} = proc(elem)
                             begin
                               if has(elem,ln) then 
                                 foundExpOfLn:= misc::breakmap();
                               end_if;
                               return(elem);
                             end_proc):
      if foundExpOfLn then 
        eq:= simplify::exp(eq);
      end_if;
    end_if;  
  end_if;

  return(eq);
end_proc:
*/


/*
    Check if an expression contains unresolved integrals, where the integrand contains 
    the dependent ODE variable or any of its derivatives.  
  
    PARAMETER: 
                eq -- an expression 
                 y -- dependent variable
                 x -- indepenent variable 
                 n -- order of the highest derivative of 'y' with respect to 'x' in 'eq'
      solveOptions -- options for 'solve'
        odeOptions -- options for 'ode' 
        

    RETURN VALUE:  Simplified version of 'eq' obtained by applying 'simplify::exp'
                   in case that 'eq' contains an expression of the form 'exp(ln(..))'
                   and no functions that are "hard to evaluate" in the sense of 
                   'ReplaceHardToEval'. Otherwise the expression 'eq' is returned 
                   without any modifications applied to it.                
*/

ode::checkIntegration:= proc(eq,y,x,solveOptions={},odeOptions={}) 
  local foundUnresolvedIntegral;
begin
  if not hastype(eq,"int") then 
    return(eq);
  end_if;  
  foundUnresolvedIntegral:= FALSE;
  misc::maprec(eq,{"int"} = proc(elem) 
                           begin
                             if has(op(elem,1),y) then 
                               foundUnresolvedIntegral:= misc::breakmap();
                             end_if;
                             elem;
                           end_proc);                         
  if not foundUnresolvedIntegral then 
    return(eq);
  end_if;
  //  eq:= misc::maprec(eq,{"int"} = proc(elem) 
  //                           local i;
  //                         begin
  //                           if has(op(elem,1),y) then 
  //                             int(simplify(op(elem,1)),op(elem,2..nops(elem)));
  //                           end_if;
  //                           elem;
  //                         end_proc);                         
  // foundUnresolvedIntegral:= FALSE;
  //misc::maprec(eq,{"int"} = proc(elem) 
  //                         begin
  //                           //if has(op(elem,1),[diff(y(x),x$i) $ i=0..n]) then 
  //                           if has(op(elem,1),y) then 
  //                             foundUnresolvedIntegral:= misc::breakmap();
  //                           end_if;
  //                           elem;
  //                         end_proc);                         
  //if not foundUnresolvedIntegral then 
  //  return(eq);
  //end_if;
  // Too costly at this time. 
  //
  //eq:= Simplify(eq,IgnoreAnalyticConstraints);
  //foundUnresolvedIntegral:= FALSE;
  //misc::maprec(eq,{"int"} = proc(elem) 
  //                           local i;
  //                         begin
  //                           //if has(op(elem,1),[diff(y(x),x$i) $ i=0..n]) then 
  //                           if has(op(elem,1),y) then 
  //                             foundUnresolvedIntegral:= misc::breakmap();
  //                           end_if;
  //                           elem;
  //                         end_proc);                         
  //if not foundUnresolvedIntegral then 
  //  return(eq);
  //end_if;
  
  return(FAIL);
end_proc:





