
/*++ ---------------- secondOrder.mu ---------------------
Description:
This file contains functions for computing liouvillian solutions
of second order (homogeneous) linear ordinary differential equations 
over the field of rational functions.

Functions: 

 - used Parameter:
    ++ Ly, y(x), r:DOM_EXPR
    ++ W, Wn, f, g:DOM_EXPR
    ++ deq        :DOM_EXPR
    ++ y,x,c,cset :DOM_IDENT
    ++ n, m1, m2  :DOM_INT
    ++ a          :DOM_LIST
    ++ Ly    = ordinary linear homogeneous differential equation 
    ++         over the rational functions
    ++ y(x)  = the operator (function) of Ly  
    ++ y     = name of the operator of Ly
    ++ x     = dependent variable of the operator y
    ++ n     = order of Ly
    ++ W     = wronskian of Ly
    ++ a     = list of the two smallest coefficients of the unique  
    ++         representation of Ly (i.e. leading coefficient of ly is 1)
    ++ Wn    = W^(1/n) or 1, the factor of the transformation
    ++ r,f,g = rational functions in x 
    ++         (r,f: rational invariants of lowest degree)
    ++ m1,m2 = degree of an invariant
    ++ c     = constant to be determined
    ++ cset  = set or list of constants or a single constant to be determined
    ++ deq   = determining equation of c or cset in x.

 - ode::secondOrder(Ly, y, x, < Transform | Irreducible >)
    ++ returns a fundamental system of liouvillian solutions of a 
    ++ second order ordinary homogeneous linear differential equation Ly
    ++ over the rational functions.
    ++ When option 'Transform' is given, the unimodular transform is
    ++ executed unconditionally.
    ++ When option 'Irreducible' is given, Ly is assumed to be irreducible.  
    ++ NOTE: currently this function only works properly for equations
    ++       for which the function 'ode::expsols' works properly.
    ++ Otherwise: no warranty!
 - ode::Hessian(f, m1, a, W, x)
    ++ returns the Hessian of the rational invariant f of degree m1.
 - ode::Jacobian(f, m1, g, m2, W, x)
    ++ returns the Jacobian of the two rational invariants f and g of degree
    ++ m1 and m2 respectively.
 - ode::solveDetEquation(deq, x, cset)
    ++ returns a solution of deq as a list of equations. This solutions
    ++ is determined in a way, that it solves deq for all regular points x.
 - ode::dihedralCase(r, a, W, x, Wn)
    ++ solves a second order linear ordinary differential equation whose
    ++ Galois group is isomorphic to a binary dihedral group, except for the
    ++ Quaternion group.
 - ode::QuaternionCase([r,f], a, W, x, Wn)
    ++ solves a second order linear ordinary differential equation whose
    ++ Galois group is isomorphic to the Quaternion group.
 - ode::tetrahedralCase(r, a, W, x, Wn)
    ++ solves a second order linear ordinary differential equation whose
    ++ Galois group is isomorphic to the tetrahedral group. 
 - ode::octahedralCase(r, a, W, x, Wn)
    ++ solves a second order linear ordinary differential equation whose
    ++ Galois group is isomorphic to the octahedral group. 
 - ode::icosahedralCase(r, a, W, x, Wn)
    ++ solves a second order linear ordinary differential equation whose
    ++ Galois group is isomorphic to the icosahedral group. 

See:
 - Fakler, W. (1997). On Second Order Homogenous Linear Differential Equations
   with Liouvillian Solutions. Theoretical Computer Science 187, 27-48.
 - Fakler, W. (1997). Algorithms for Solving Linear Ordinary Differential
   Equations. mathPAD 7 No.1, 50-59.
 - Ulmer, F., Weil, J.A. (1996). Note on Kovacic's Algorithm. 
   J. Symb. Comp. 22, 179-200.

Examples:
>> Wy := diff(y(x), x, x)+ diff(y(x), x)/(x*2-1)*(-2) + y(x)*(x*(-86) + x^2*25 
         + x^3*14 + x^4*263 + x^5*(-324) + x^6*108 + 27)/(x^2*144 + x^4*(-576) 
         + x^5*288 + x^6*576 + x^7*(-576) + x^8*144);
>> Hy := diff(y(x), x,x)+27*x/(8*(x^3-2)^2)*y(x);
>> Ky := diff(y(x), x, x) + y(x)*(x*(-27) + x^2*32 + 27)/(x^2*144 + x^3*(-288) 
         + x^4*144);
>> setuserinfo(ode, 10):
>> ode::secondOrder(Wy, y, x);

++*/

/* Dependencies:

  ifReducible.mu, RatSol.mu, tools.mu, mkUnimodular.mu   */

ode::secondOrder:=
proc(eq, y, x) 
  //options Transform, Irreducible 
  local o, m, a, W, trafo, sols, rvSet, solveOptions,odeOptions;
begin
  o := {args(4..args(0))};
  /*
   Note by Kai: The last two arguments MUST BE 'solveOptions' and 
                'odeOptions'.
  */
  if args(0) = 5 then 
    solveOptions:= args(args(0)-1);
    odeOptions:= args(args(0));
  else 
    solveOptions:= {};
    odeOptions:= {};
  end_if;    
  
  if not contains(o, Irreducible) then
    userinfo(5, "test, if the equation is reducible...");
    sols:= ode::ifReducible(eq, y, x, 2, solveOptions,odeOptions);
    if sols<>{} then
      return(map(sols,combine,exp)) end_if;
    if ode::possiblyLostSolutionsFlag and ode::printWarningsFlag then
      ode::odeWarning("Only Q-solvable exponential solutions will be found!")
    end_if;
  end_if;

  // Now we assume eq is irreducible! 
  if nops(freeIndets(eq))-1 > 6 then 
    return({})
  end_if;  
  userinfo(1, "Now we assume the equation is irreducible. ".
              "No warranty otherwise!");
  if contains(o, Transform) then
    W:= ode::mkUnimodular(eq, y, x, 2, Transform, solveOptions,odeOptions)
  else
    W:= ode::mkUnimodular(eq, y, x, 2, " ", solveOptions,odeOptions)
  end_if;
  eq:= numer(expr(W[1]));
  userinfo(10, "wronskian of equation is\n", W[2]);
  a:= W[3];
  trafo:= W[4];
  if trafo=1 then
    userinfo(5, "no unimodular transformation is needed");
    W:= W[2]
  else
    userinfo(5, "transformed equation is\n", eq);
    W:= 1
  end_if;

  // Now for rational invariants of degree 'm' try to apply the classification
  // methods from the reference papers listed in the header to this file.   
  for m in [4, 6, 8, 12] do
    userinfo(5, "calculate the rational invariants of ".
             output::ordinal(m) . " degree :");
    rvSet:= ode::ratInvars(eq,y,x,2,m,solveOptions,odeOptions);
    userinfo(5, "fundamental rational invariants of ".
             output::ordinal(m) . " degree are\n",
             rvSet);
    rvSet:= [op(rvSet)];
    case m
      of  4 do case nops(rvSet)
                      of 1 do 
                        userinfo(5, "dihedralCase");
                        return(ode::dihedralCase(rvSet[1], a, W, x,trafo,
                                                 solveOptions, odeOptions));
                        break;
                      of 2 do 
                        userinfo(5, "QuaternionCase");
                        return(ode::QuaternionCase(rvSet,a,W,x,trafo,
                                                   solveOptions, odeOptions));
                        break;
                      end_case; 
                    break;
      of  6 do if nops(rvSet)=1 then 
                 userinfo(5, "tetrahedralCase");
                 return(ode::tetrahedralCase(rvSet[1],a,W,x,trafo,
                                             solveOptions, odeOptions)) 
             end_if;
        break;
      of  8 do if nops(rvSet)=1 then 
                 userinfo(5, "octahedralCase");
                 return(ode::octahedralCase(rvSet[1],a,W,x,trafo,
                                            solveOptions, odeOptions)) 
               end_if;
        break;
      of 12 do if nops(rvSet)=1 then 
                 userinfo(5, "icosahedralCase");
                 return(ode::icosahedralCase(rvSet[1],a,W,x,trafo,
                                             solveOptions, odeOptions)) 
             end_if;
        break;
    end_case;
  end_for;
  return({})
end_proc:

ode::Hessian:=
proc(f, m, a, W, x, solveOptions, odeOptions)
  local dlogf;
begin
  assert(not iszero(f));
  assert(not iszero(W));
  dlogf:= ode::normal(diff(f,x)/f);
  ode::normal((m-1)/W^2*(dlogf^2+m*diff(dlogf, x)+m*a[2]*dlogf+m^2*a[1])*f^2)
end_proc:

ode::Jacobian:=
proc(f, m, g, n, W, x, solveOptions, odeOptions)
begin
  assert(not iszero(W));
  ode::normal((m*f*diff(g, x)-n*diff(f, x)*g)/W)
end_proc:

ode::solveDetEquation:=
proc(detEq, x, cset, solveOptions, odeOptions)
  local eqns, sol, s, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  // We used to have 
  //
  //   eqns:= coeff(poly(numer(simplify(detEq, optIgnoreAnalyticConstraints)), [x]));
  //
  // for the following line, but the simplification is not necessary. 
  eqns:= coeff(poly(numer(detEq), [x]));
  if domtype(eqns)<>DOM_LIST then
    eqns:=[eqns]
  end_if;
  case nops(cset)
    of 1 do //since we know there is a solution, we can do the following: 
      sol:= solvelib::discreteSolve(eqns,op(cset),MaxDegree=4,op(solveOptions));
      if not has(sol,FAIL) and sol <> {} then 
        return([op(cset) = op(sol, nops([op(sol)])) ]);
      else 
        return({});
      end_if;
    of 2 do // since we know there is a solution and as long as we are
            // computing only over the rationals (not in algebraic extensions)
            // solve always will find a solution. Thus we simply have to pick 
            // up a nice solution (in which no imaginary unit I is involved)
      cset:= [op(cset)];
      sol:= solvelib::discreteSolve(eqns,cset,MaxDegree=4,op(solveOptions))
            minus {[cset[1]=0,cset[2]=0]};
      s:= split(sol,has,I);
      if nops(s[2])<>0 then // there is a solution without imaginary I
        return(op(s[2],1))
      else
        return(op(s[1],1))
      end_if;
    otherwise
      error("illegal arguments")
  end_case
end_proc:

ode::dihedralCase:=
proc(r, a, W, x, trafo, solveOptions, odeOptions)
  local r1, r2, c, ci, detEq, integral, m, i, j, intOptions, optIgnoreAnalyticConstraints;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  r1:= diff(r, x);
  r2:= diff(r1, x);
  ci:= genident("C"); 
  detEq:= (4*r2*r-3*r1^2) + (W^2*ci^2+r1*a[2])*(4*r) + a[1]*(16*r^2);
  userinfo(10, "determining equation for",ci,"is\n", detEq);
  c:= ode::solveDetEquation(detEq, x, ci, solveOptions, odeOptions);
  if c = {} then 
    return({}) 
  end_if;
  userinfo(5, "constant", op(c));
  c := op(c, [1,2]);
  assert(not iszero(c));
  assert(not iszero(r));
  userinfo(5, "fundamental invariant of degree 4 is", "\n I_4 =", 1/c^2*r);
  //userinfo(5, "...equation is solved, try to compute the integral");
  r1:=factor(W^2/r);
  // Anpassung an Umstellung von Factored::_index (Walter 29.12.04)
  r1:= coerce(r1, DOM_LIST);
  r1:=_mult(sqrt(r1[1]), r1[2*i]^(r1[2*i+1]/2) $ i=1..nops(r1) div 2);
  m:=factor(trafo^4*r);

  // Anpassung an Umstellung von Factored::_index (Walter 29.12.04)
  m:= coerce(m, DOM_LIST);

  m:=_mult(m[2*i]^(m[2*i+1]/4) $ i in select({j $j=1..nops(m) div 2}, ()->has(m[2*args(1)],x)));
  
  integral:= int(r1, x, intOptions, IgnoreSpecialCases);
  if type(c)=DOM_COMPLEX and Re(c)=0 and not has(integral, I) then
    // We used to have 
    //     
    //     return(simplify(map({m*cos(-Im(c)/2*integral), 
    //                            m*sin(-Im(c)/2*integral)}, 
    //                           expand, optIgnoreAnalyticConstraints), 
    //                       optIgnoreAnalyticConstraints));
    //
    // but simpifying and expanding the expression does not seem necessary. 
    return({m*cos(-Im(c)/2*integral),m*sin(-Im(c)/2*integral)});
  elif not has(integral, int) then
    // We used to have 
    //     
    //    return(simplify(map({m*exp(-c/2*integral), 
    //                         m*exp(c/2*integral)}, 
    //                        expand, optIgnoreAnalyticConstraints), 
    //                    optIgnoreAnalyticConstraints)
    //
    // but simpifying and expanding the expression does not seem necessary. 
    return({m*exp(-c/2*integral),m*exp(c/2*integral)});
  else
    return({m*exp(-c/2*integral), m*exp(c/2*integral)});
  end_if:
end_proc:

ode::QuaternionCase:=
proc(rs, a, W, x, trafo, solveOptions, odeOptions)
  local r, r1, r2, c, ci, cj, detEq, integral, m, i, j, intOptions, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  ci:= genident("c");
  cj:= genident("c");
  r:= ci*rs[1]+cj*rs[2];
  r1:= diff(r, x);
  r2:= diff(r1, x);
  detEq:= (4*r2*r-3*r1^2) + (W^2+r1*a[2])*(4*r) + a[1]*(16*r^2);
  userinfo(10, "determining equation for {", ci,cj, "} is\n", detEq);
  c:= ode::solveDetEquation(detEq, x, [ci, cj], solveOptions, odeOptions);
  if c = {} then 
    return({}) 
  end_if;
  userinfo(5, "constants are", op(c));
  // We used to have 
  //
  //    r:= simplify(subs(r, op(c)), optIgnoreAnalyticConstraints);
  //
  // but simplification does not seem to be necessary.
  // OK if traperror((r:= Simplify(subs(r, op(c)), optIgnoreAnalyticConstraints))) <> 0 then 
  if traperror((r:= evalAt(r, op(c)))) <> 0 then 
    return({});
  end_if;  
  userinfo(5, "fundamental invariant of degree 4 is", "\n I_4 =", r);
  //  userinfo(5, "...eq is solved, try to compute the integral");
  //  putting assume(x>0) simplifies the result and should not have a bad
  //  consequence on the result
  assert(not iszero(r));
  r1:= factor(W^2/r);
  r1:= coerce(r1, DOM_LIST);
  r1:= _mult(sqrt(r1[1]), r1[2*i]^(r1[2*i+1]/2) $ i=1..nops(r1) div 2);

  m:=factor(trafo^4*r);
  m:= coerce(m, DOM_LIST);
  m:=_mult(m[2*i]^(m[2*i+1]/4) $ i in select({j $j=1..nops(m) div 2},
                                             t ->has(m[2*t],x)));
  
  integral:= int(r1, x, intOptions, IgnoreSpecialCases);
  
  return({m*exp(-1/2*integral), m*exp(1/2*integral)});
end_proc:

ode::tetrahedralCase:=
proc(r, a, W, x, trafo, solveOptions, odeOptions)
  local h, j, c, ci, detEq, i1, i2, i3, y, mp,Y;
  //save Y;
begin
  h:= ode::Hessian(r, 6, a, W, x, solveOptions, odeOptions);
  userinfo(5, "Hessian", h);
  j:= ode::Jacobian(r, 6, h, 8, W, x, solveOptions, odeOptions);
  userinfo(5, "Jacobian", j);
  ci:= genident("C");
  y:= genident("y");
  detEq :=(25*j^2+64*h^3)*ci^2 + 108000000*r^4;
  userinfo(10, "determining equation for",ci,"is\n", detEq);
  c:= ode::solveDetEquation(detEq, x, ci, solveOptions, odeOptions);
  if c = {} then 
    return({}) 
  end_if;
  userinfo(5, "constant", op(c));
  c:= op(c, [1,2]);
  i1:= c*r *1/4;
  i2:= -(c^2)/25*h *(-5)/80;
  i3:= -1/25*(c^3)/8*j *(-1)/16;
  userinfo(5, "fundamental invariants are",
           "\n I_1 =", i1, "\n I_2 =", i2, "\n I_3 =", i3);
  Y:=genident("_Y");
  mp := Y^24 + 10*i2*Y^16 + 5*i3*Y^12 - 15*i2^2*Y^8 - i2*i3*Y^4 + i1^4;
  
  return({RootOf(ode::printMinPol(mp,Y,4,6,solveOptions,odeOptions), Y) * trafo});
end_proc:

ode::octahedralCase:=
proc(r, a, W, x, trafo, solveOptions, odeOptions)
  local h, j, c, ci, detEq, i1, i2, y, mp,Y;
  //save Y;
begin
  h:= ode::Hessian(r, 8, a, W, x, solveOptions, odeOptions);
  userinfo(5, "Hessian", h);
  j:= ode::Jacobian(r, 8, h, 12, W, x, solveOptions, odeOptions);
  userinfo(5, "Jacobian", j);
  ci:= genident("C");
  y:= genident("y");
  detEq :=(49*j^2+144*h^3)*ci - 118013952*r^3*h;
  userinfo(10, "determining equation for", ci,"is\n", detEq);
  c:= ode::solveDetEquation(detEq, x, ci, solveOptions, odeOptions);
  if c = {} then 
    return({}) 
  end_if;
  userinfo(5, "constant", op(c));
  c := op(c, [1,2]);
  i1 := c*r *(-1)/16;
  i2 := (c^2)/9408*h *1/16;
  userinfo(5, "fundamental invariants are", "\n I_1 =", i1, "\n I_2 =", i2);
    //delete Y;
  Y:=genident("_Y");
  mp:= Y^48 + 20*i1*Y^40 + 70*i1^2*Y^32 + (2702*i2^2+100*i1^3)*Y^24 +
       (-1060*i1*i2^2+65*i1^4)*Y^16 + (78*i1^2*i2^2+16*i1^5)*Y^8 + i2^4;
        
  return({RootOf(ode::printMinPol(mp,Y,8,6,solveOptions,odeOptions), Y) * trafo});
end_proc:

ode::icosahedralCase:=
proc(r, a, W, x, trafo, solveOptions, odeOptions)
  local h, j, c, ci, detEq, i1, i2, i3, y, mp,Y;
  //save Y;
begin
  h:= ode::Hessian(r, 12, a, W, x, solveOptions, odeOptions);
  userinfo(5, "Hessian", h);
  j:= ode::Jacobian(r, 12, h, 20, W, x, solveOptions, odeOptions);
  userinfo(5, "Jacobian", j);
  ci:= genident("C");
  y:= genident("y");
  detEq:=(121*j^2+400*h^3)*ci + 1224502963200*r^5;
  userinfo(10, "determining equation for", ci, "is\n", detEq);
  c:= ode::solveDetEquation(detEq, x, ci, solveOptions, odeOptions);
  if c = {} then 
    return({}) 
  end_if;
  userinfo(5, "constant", op(c));
  c:= op(c, [1,2]);
  i1:= c*r *1/125;
  i2:= -(c^2)/121*h *(-1)/34375;
  i3:= -1/121*(c^3)/20*j *(-11)/3125;
  userinfo(5, "fundamental invariants are",
           "\n I_1 =", i1, "\n I_2 =", i2, "\n I_3 =", i3);
  //delete Y;
  Y:=genident("_Y");
  mp:= Y^120 + 20570*i2*Y^100 + 91*i3*Y^90 - 86135665*i2^2*Y^80 - 78254*i2*i3*Y^70 +
       (14993701690*i2^3+11137761250*i1^5)*Y^60 + 897941*i2^2*i3*Y^50 + 
       (-11602919295*i2^4+273542733750*i1^5*i2)*Y^40 + 
       (-151734*i2^3-6953000*i1^5)*i3*Y^30 + 
       (-503123324*i2^5-7854563750*i1^5*i2^2)*Y^20 + 
       (1331*i2^4+500*i1^5*i2)*i3*Y^10 + 3125*i1^10;
       
  return({RootOf(ode::printMinPol(mp,Y,10,12,solveOptions,odeOptions), Y) * trafo});
end_proc:

