
/*++ ---------------- thirdOrder.mu ---------------------
Description:
This file contains functions for computing liouvillian solutions
of third order (homogeneous) linear ordinary differential equations 
over the field of rational functions.

Functions: 

 - used Parameter:
    ++ Ly, y(x)   :DOM_EXPR
    ++ W, Wn,     :DOM_EXPR
    ++ deq        :DOM_EXPR
    ++ y,x,c,cset :DOM_IDENT
    ++ n, m1, m2  :DOM_INT
    ++ a          :DOM_LIST
    ++ Ly    = ordinary linear homogeneous differential equation 
    ++         over the rational functions
    ++ y(x)  = the operator (function) of Ly  
    ++ y     = name of the operator of Ly
    ++ x     = dependent variable of the operator y
    ++ n     = order of Ly
    ++ W     = wronskian of Ly
    ++ a     = list of the two smallest coefficients of the unique  
    ++         representation of Ly (i.e. leading coefficient of ly is 1)
    ++ Wn    = W^(1/n) or 1, the factor of the transformation
    ++ m1,m2 = degree of an invariant
    ++ c     = constant to be determined
    ++ cset  = set or list of constants or a single constant to be determined
    ++ deq   = determining equation of c or cset in x.

 - ode::thirdOrder(Ly, y, x, < Transform | Irreducible >)
    ++ returns a fundamental system of liouvillian solutions of a 
    ++ third order ordinary homogeneous linear differential equation Ly
    ++ over the rational functions.
    ++ When option 'Transform' is given, the unimodular transform is
    ++ executed unconditionally.
    ++ When option 'Irreducible' is given, Ly is assumed to be irreducible. 
    ++ NOTE: Only the reducible case is yet implemented! 
    ++ NOTE: currently this function only works properly for equations
    ++       for which the function ode::expsols works properly.
    ++ Otherwise: no warranty!


See:
 - Fakler, W. (1997). On Second Order Homogenous Linear Differential Equations
   with Liouvillian Solutions. Theoretical Computer Science 187, 27-48.
 - Fakler, W. (1997). Algorithms for Solving Linear Ordinary Differential
   Equations. mathPAD 7 No.1, 50-59.
 

Examples:
>> 

++*/

/* Dependencies:

  ifReducible.mu, RatSol.mu, tools.mu, mkUnimodular.mu, Lodo.mu */

ode::thirdOrder:=
  proc(eq, y, x) 
    //options Transform, Irreducible 
    //save Y;
    local o,sols,solveOptions,odeOptions;
  begin
    sysassign(ode::isReducibleFlag, FALSE);
    o := {args(4..args(0))};

    /*
     Note by Kai: The last two arguments MUST BE 'solveOptions' and 
                  'odeOptions'.
    */
    if args(0) = 5 then 
      solveOptions:= args(args(0)-1);
      odeOptions:= args(args(0));
    else 
      solveOptions:= {};
      odeOptions:= {};
    end_if;    
    
    if not contains(o, Irreducible) then
      userinfo(5, "test, if the equation is reducible...");
      sols:= ode::ifReducible(eq, y, x, 3, solveOptions,odeOptions);
      if sols <> {} then
        //sols := map(map(map(sols,_mult,trafo),expand),simplify);
        return( map(sols,combine,exp) ) end_if; 
    end_if;
    
    if not(ode::isReducibleFlag) and ode::printWarningsFlag then
      ode::odeWarning("For third order equations currently only the ".
          "reducible case is implemented.")
    end_if;
    return({});
  end_proc:

