/*++ ---------------- RatSol.mu ---------------------
Description:
This file contains functions for computing polynomial and rational solutions
of linear ordinary differential equations over the field of rational functions.

Functions: 

 - used Parameter:
    ++ Ly, y(x), R:DOM_EXPR
    ++ y, x       :DOM_IDENT
    ++ n, m, mu   :DOM_INT
    ++ A, a, ci   :DOM_LIST
    ++ P, Q       :DOM_POLY
    ++ M          :Dom::/*Dense*/Matrix
    ++ Ly   = ordinary linear differential equation over the rational functions
    ++ y(x) = the operator (function) of Ly  
    ++ y    = name of the operator of Ly
    ++ x    = dependant variable of the operator y
    ++ n    = order of Ly
    ++ m    = exponent of the symmetric power of Ly
    ++ mu   = sum of exponents of Ly
    ++ ci   = vectorized form Ly: [c_0,...,c_n] (denominators needs to be 
    ++        all of degree 0 in x)
    ++ A    = vectorized form Ly (i.e. all coefficients of Ly as polynomials)
    ++        [c_0,...,c_n] 
    ++ R    = Ansatz for rational solutions with a bounded but indeterminate
    ++        numerator and determined denominator
    ++ a    = list of the indeterminates of the numerator of R
    ++ P, Q = polynomials
    ++ M    = square matrix representing a linear differential system 

 - ode::rationalSolutions(Ly, y(x), < Generic >)
   ode::rationalSolutions(M,x)
    ++ returns a fundamental system of rational solutions of an ordinary
    ++ linear differential equation Ly over the rational functions or of a 
    ++ linear differential system of first order.
    ++ When the option Generic is given, a generic form of them is returned.
    
 - ode::ratSys(M,x)
    ++ computes a basis of rational solutions of a linear differential system
    ++ of first order using the method of cyclic vectors.
    
 - ode::ratsols(Ly, y, x, n), 
   ode::ratsols(ci, x)
    ++ returns a fundamental system of rational solutions of an ordinary
    ++ linear differential equation Ly over the rational functions.
    ++ NOTE: Ly should be normalized (i.e. denom(Ly)=1).
    
 - ode::rationalInvariants(Ly, y(x), m)
    ++ returns a fundamental system of rational solutions of the m-th
    ++ symmetric power of an ordinary linear differential equation Ly
    ++ over the rational functions.
    
 - ode::ratInvars(Ly, y, x, n, m)
    ++ returns a fundamental system of rational solutions of the m-th
    ++ symmetric power of an ordinary linear differential equation Ly
    ++ over the rational functions.
    ++ NOTE: Ly should be normalized (i.e. denom(Ly)=1).
    
 - ode::denomAnsatz(A, x, n, < "mu" >)
    ++ returns the Ansatz of the numerator for rational solutions of A.
    ++ When the option "mu" is given, the sum of the exponents 
    ++ are returned additionally.
    ++ NOTE: A should be normalized (i.e. all denominators must be one).

- ode::nu(Q, P, < Rem >)
    ++ computes the order of Q at P, which is the largest integer m
    ++ such that P^m divides Q.
    ++ When the option Rem is given, Q/P^m is returned.

REFERENCE:
     - M. Bronstein (1992). Integration and Differential Equations in
       Computer Algebra.  Programmirovanie 18, Nr. 5, pages 9-10,
     - Abramov, S.A., Kvansenko, K.Yu. (1991). Fast Algorithms to Search for the
       Rational Solutions of Linear Differential Equations with Polynomial 
       Coefficients. In: Proceedings of ISSAC '91, Bonn, Germany, 267-270.
     - Barkatou, M. (1997). An Efficient Algorithm for Computing Rational 
       Solutions of Systems of Linear Differential Equations. 
       Submitted to ISSAC '97 

Examples:
>> Ly := y(x)*6 + x*diff(y(x), x)*(-2) + diff(y(x), x, x)*(-x^2 + 1):
>> ode::rationalSolutions(Ly, y(x));
 
                                  2
                                {x  - 1/3}
 
>> ode::rationalInvariants(Ly, y(x), 2);

                            {         2       }
                            {  4   2 x        }
                            { x  - ---- + 1/9 }
                            {       3         }
++*/

ode::rationalSolutions:= proc(eq,z,o=" ",solveOptions={},odeOptions={}) //o: Option Generic 
  local sol;
begin
  if testtype(eq, Dom::Matrix) then
    return({}) // return(ode::ratSys(eq,z,o,solveOptions,odeOptions))
  end_if;
  if args(0) < 2 or args(0) > 5 then 
    error("expecting two or three arguments")
  elif args(0) = 3 and o <> Generic then 
    error("expecting option 'Generic' as third argument");
  elif args(0) = 4 or args(0) = 5 then
    if map({args(args(0)-1),args(args(0))},type) <> {DOM_SET} then 
      error("invalid arguments")    
    end_if;
  end_if;
  if type(z) <> "function" then
    error("expecting a function as the dependent variable")
  elif nops(z) <> 1 then 
    error("dependent variable must be a univariate function") 
  end_if; 
  if domtype(eq) <> DOM_EXPR then 
    error("expecting an ordinary differential equation as first argument")  
  end_if;
  if not has(eq,z) then 
    error("not an ordinary differential equation in ".expr2text(z))
  end_if;
  eq := ode::isLODE(rewrite(eq, diff), z, HlodeOverRF, solveOptions, odeOptions);
  case nops(eq)
  of 1 do 
    error("not an ordinary homogeneous linear differential ".
          "equation over the rational functions");
  of 4 do 
    if eq[4] < 1 then
      return(error("only defined for positive order equations"))
    end_if;
  end_case;
  // eq[1] := numer(normal(expr(eq[1]),Rationalize=None));
  eq[1]:= ode::removeDenominator(eq, solveOptions, odeOptions);
  // Here, 'eq' is a list of 4 elements!
  sol := ode::ratsols(eq, solveOptions,odeOptions);
  if o=Generic then
    return(_plus(op(map(sol, e->_mult(e,genident("C"))))))
  else
    return(sol);
  end_if
end_proc:


ode::ratSys:= proc(A,x,solveOptions={},odeOptions={})  
  local v,sol,s,M,n,yis,z,sys,solset,r,i,d;
begin
  // using cyclic vector method 
  userinfo(5,"reduce system to an equivalent scalar differential equation");
  v:= ode::cyclicVector(A,x,[],solveOptions,odeOptions);
  if v=[] then 
    return({}); 
  end_if;
  userinfo(10,"found cyclic vector");
  userinfo(5,"make scalar equation integral");
  d:= lcm(op(map(v[1],denom)));
  sol:= ode::ratsols([op(map(v[1],ode::normal@_mult,-d)),d],x,
                     "dummy","dummy",solveOptions,odeOptions);  
  userinfo(12,"rational solutions of scalar equation are ",sol);
  if sol={} then 
    return({}) 
  end_if;
  // construct linear system to determine rational solutions of A
  userinfo(5,"construct linear system to determine rational ".
             "matrix solutions");
  M:= Dom::Matrix(Dom::ExpressionField(normal));
  n:= linalg::nrows(v[2]);
  yis:= [genident("y") $ i=1..n];
  z:= genident("z");
  sys:= {op(zip(map([op(v[2]*M(yis))],expr),
                [diff(z(x),x$i) $ i=0..n-1],
                _equal))};
  solset:= linsolve(sys,yis);
  if solset = FAIL and ode::printWarningsFlag then
    ode::odeWarning("something is wrong");
  end_if;
  // determine rational solutions
  userinfo(5,"determine rational matrix solutions");
  r:= {};
  for s in sol do
    r:= r union {M([op(subs(solset,z(x)=s,EvalChanges),[i,2]) $ i=1..n])};
  end_for;
  return(r);
end_proc:
  
  
/* 
   Here the coefficients are assumed to be polynomial!!! 
   No normal is done, use 'ode::rationalSolutions for 
   rational coefficients. 
*/
ode::ratsols:= proc(eq,y,x,n,solveOptions,odeOptions) 
  local a, i, den, z, Ly, sol;
begin
  /*
     The number of parameters for calling this function orginally depends 
     on the fact whether 'eq' is a list or not. If 'eq' is a list, then 
     'x' and 'n' are not initialized by the function's call. Because of 
     the introduction of the new options 'solveOptions' and 'odeOptions', 
     the function has to be called with 6 arguments. If 'eq' is a list 
     then it will be called in the form 
  
        'ode::ratsols(eq,var,"dummy","dummy",solveOptions,odeOptions)'
  
     Note that the function 'ode::ratsols' is not used outside this
     file.                                                               */
  if type(eq)=DOM_LIST then 
    a:= eq; 
    x:= y; 
    n:= nops(a)-1;
    y:= genident("_y")
  else
    a:= ode::vectorize(eq,y,x,n,solveOptions,odeOptions) 
  end_if;
  userinfo(13, "determine ansatz of denominator");
  // We restrict our efforts to cases where the length of the list 'a' of 
  // polynomial coefficients of the corresponding linear differential operator
  // does not exceed the bound 2500. This heuristic may be changed/improved in 
  // the future. 
  if length(a) > 2500 then 
    return({});
  end_if;  
  if traperror((den:= ode::denomAnsatz(map(a,poly,[x]),x,n," ",
                                       solveOptions,odeOptions))) <> 0 or iszero(den) then
    return({})
  end_if;
  //userinfo(12,"computed denominator ``ansatz'' at time ".time());
  userinfo(10,"degree of denominator is ".degree(den));
  userinfo(5, "ansatz of denominator", den);
  assert(not iszero(den));
  z:= y(x)/den;
  Ly:= a[1]*z; 
  for i from 1 to n do
    z:= diff(z,x);
    Ly:= Ly+a[i+1]*z
  end_for;
  // Ly := numer(normal(expr(Ly),Rationalize=None));
  Ly:= ode::removeDenominator(Ly, y, x, n, solveOptions, odeOptions);
  userinfo(15, "transformed equation is", Ly);
  sol:= ode::polysols(Ly,y,x,n,0,solveOptions,odeOptions);
  //userinfo(12,"computed polynomial solutions at time ".time());
  return(map(map(sol,_mult,1/den),ode::normal,Rationalize=None));
end_proc:


ode::rationalInvariants:= proc(eq,z,m,solveOptions,odeOptions)
begin
  eq := ode::isLODE(rewrite(eq, diff), z, HlodeOverRF, solveOptions, odeOptions);
  case nops(eq) 
  of 1 do 
    error("not an ordinary homogeneous linear differential ".
          "equation over the rational functions");
  of 4 do 
    if eq[4] < 1 then
      error("only possible for positive order equations")
    end_if; 
  otherwise
    if domtype(m) <> DOM_INT or m < 1 then 
      return(error("only defined for positive integers"))
    end_if
  end_case;
  // eq[1] := numer(normal(expr(eq[1]),Rationalize=None));
  eq[1]:= ode::removeDenominator(eq, solveOptions, odeOptions);
  return(ode::ratInvars(eq, m, solveOptions,odeOptions));
end_proc:


ode::ratInvars:= proc(eq,y,x,n,m,solveOptions,odeOptions)
  local h, ord, a;
begin
  if m=1 then 
    return(eq) 
  end_if;
  userinfo(5, "compute the ".output::ordinal(m)." symmetric power");
  if n=2 then 
    eq:= ode::fracFreeSymPowerOrd2(eq, y, x, m, solveOptions, odeOptions);
    userinfo(5, "determine its rational solutions");
    return(ode::ratsols(ode::vectorize(eq,y,x,m+1,solveOptions,odeOptions),x,
                                       "dummy","dummy",solveOptions,odeOptions))
  end_if;
  eq:= ode::symPower(eq, y, x, n, m, solveOptions, odeOptions);
  userinfo(5, "...normalize it...");
  ord:= ode::getOrder(eq, y(x),solveOptions,odeOptions);
  a:= ode::vectorize(eq,y,x,ord,solveOptions,odeOptions);
  h:= lcm(op(map(map(a,denom),poly,[x])));
  if degree(h)<>0 then 
    h:= expr(h); 
    a:= map(map(a,_mult,h),ode::normal,Rationalize=None) 
  end_if; 
  userinfo(5, "determine its rational solutions");
  return(ode::ratsols(a,x,"dummy","dummy",solveOptions,odeOptions));
end_proc: 


ode::denomAnsatz:= proc(A,x,n,o=" ",solveOptions,odeOptions) 
                        // Option: "mu", the sum of Exponents 
local k,l,m,Q,mu,i,J,j,v,vmin,P,d,dp,minj,z;
begin 
  l:= factor(A[n+1]);
  l:= coerce(l, DOM_LIST);
  m:= nops(l) div 2; 
  mu:= 0;
  userinfo(14,"...factored leading coefficient, found ".expr2text(m).
              " factors");
  for i from 1 to m do
    Q[i]:= l[2*i];
    J[i]:= {};
    vmin:= infinity;
    for k from 0 to n do
      v:= ode::nu(A[k+1],Q[i],o=" ",solveOptions,odeOptions) - k;
      if v < vmin then
        J[i]:= {k}; 
        vmin:= v;
      elif v = vmin then
        J[i]:= J[i] union {k};
      end_if;
    end_for;
    dp:= diff(Q[i],x);
    P[i]:= 0;
    minj:=min(op(J[i]));
    // delete z;
    z:= genident();
    userinfo(14,"...determine exponents of ".output::ordinal(i)." factor");
    for j in J[i] do
      // improvement with respect to [Bronstein92]: 
      // factor out the terms z*(z-1)*...*(z-(minj-1)) 
      // if minj>0 
      P[i]:= P[i] + multcoeffs
      (dp^j*ode::nu(A[j+1],Q[i],Rem,solveOptions,odeOptions),_mult(z-k$ k=minj..j-1) )
    end_for;
    P[i]:= divide(P[i],Q[i],Rem);
    // P[i]:= polylib::resultant(P[i],Q[i],x);
    // As Q[i] is irrecucible, no resultant is needed, just get the
    // roots of P[i](0). 
    P[i]:= lcoeff(P[i]); // P[i](0) may be 0
    P[i]:= polylib::primpart(poly(P[i], [z]));
    // select negative roots 
    /* 
      We need to make sure that no errors occur due to the fact that the 
      polynomial 'P[i]' might not be a polynomial over 'Q[x]'.
      In the latter situation 'solvelib::iroots' produces an error message
      and we set 'd[i]' to be the empty set here. 
     */
    if not testtype(P[i], Type::PolyOf(Type::Rational,1)) or 
       traperror((d[i]:= select(solvelib::iroots(P[i]),_less,0))) <> 0 then 
      d[i]:= {};
    end_if;
    if d[i]={} then
      d[i]:= 0;
    else
      d[i]:= -min(op(d[i]));
    end_if;
    mu:= mu + d[i]*degree(Q[i]);
  end_for;
  Q:= expr(_mult(Q[i]^d[i]$ i=1..m));
  if o = "mu" then
    return(Q,mu);
  else
    return(Q);
  end_if
end_proc:


/* returns the largest integer m such that P^m divides Q
   or Q/P^m is o=Rem */
ode::nu:= proc(Q,P,o=" ", solveOptions,odeOptions) 
  local m,R;
begin
  if iszero(Q) then
    m:= infinity
  else
    m:= 0;
    repeat
      R:= divide(Q,P,Exact);
      if R <> FAIL then 
        m:= m+1;
        Q:= R; 
      else 
        break 
      end_if
    until 
      FALSE 
    end_repeat;
  end_if;
  if o = Rem then 
    return(Q); 
  else 
    return(m); 
  end_if;
end_proc:
    
