/*++ ---------------- ifReducible.mu ---------------------
Description:
This file contains functions for solving *reducible* ordinary linear 
(homogeneous) differential equations over the rational functions.
 

Functions: 

 - used Parameter:
    ++ Ly, y(x), v:DOM_EXPR
    ++ y, x       :DOM_IDENT
    ++ n          :DOM_INT
    ++ fl         :DOM_LIST
    ++ Ly    = ordinary linear homogeneous differential equation 
    ++ y(x)  = the operator (function) of Ly  
    ++ y     = name of the operator of Ly
    ++ x     = dependent variable of the operator y
    ++ n     = order of Ly
    ++ v     = special nonzero solution of Ly
    ++ fl    = list of Ly, the (irreducible) factors of an ordinary linear 
    ++         homogeneous differential equation, where the last factor of fl 
    ++         corresponds to right factor of the equation

 - ode::solveIfReducible(Ly, y(x))
    ++ returns the liouvillian solutions of Ly if Ly is reducible, 
    ++ otherwise an empty set is returned.
    ++ NOTE: currently this function only works for second and third order 
    ++       equations for which the function 'ode::expsols' works. 
 - ode::ifReducible(Ly, y, x, n)
    ++ returns the liouvillian solutions of Ly if Ly is reducible, 
    ++ otherwise an empty set is returned. 
    ++ NOTE: currently this function only works for second and third order 
    ++       equations for which the function 'ode::expsols' works.
 - ode::firstOrder(Ly,y,x)
    ++ returns the liouvillian solution of the first order equation Ly. The
    ++ integral inside will not be evaluated.
 - ode::reducibleCase(Ly, y, x, n)
    ++ returns (maybe not all) liouvillian solutions of a reducible 
    ++ n-th order equation Ly. This function cannot be a decision procedure
    ++ since there is neither (currently) a complete algorithm for 
    ++ factorization of the corresponding differential operator nor an 
    ++ algorithm for solving liouvillian solutions of order > 3. 
    ++ NOTE: currently this function only works for second and third order 
    ++       equations for which the function ode::expsols works.
 - ode::combineSolutionsOfFactors(fl,y,x,n)
    ++ returns liouvillian solutions, which can be determined by solving 
    ++ each factor and combining the solutions via variation of parameters.
    ++ Thus it is not a decision procedure for *all* liouvillian solutions,
    ++ see also 'ode::reducibleCase'.

See:
 - Walter, W. (1990). Gewhnliche Differentialgleichungen. 4th Edition.
   Berlin, Heidelberg, New York: Springer. 
   (for Variation of Parameters)

++*/

/* Dependencies:

    ExpSol.mu, dAlembert.mu, tools.mu, Lodo.mu
*/

/* Currently not used. 
ode::solveIfReducible:=
  proc(eq, z, solveOptions, odeOptions)
  begin
    eq := ode::isLODE(rewrite(eq, diff), z, HlodeOverRF, solveOptions, odeOptions);
    if nops(eq)<> 4 then 
      return(error("not an ordinary homogeneous linear differential ".
                           "equation over the rational functions")) end_if;    
    // eq[1] := numer(normal(expr(eq[1])));
    ode::ifReducible(eq, solveOptions, odeOptions)
  end_proc:
*/

ode::ifReducible:=
proc(eq, y, x, n, solveOptions, odeOptions)
  local sols, dim, v, w, EF, lodo, eq_adj, eq2, fac, b, e,de, Df,
        intOptions, complexity;
begin
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  if n=1 then
    return(ode::firstOrder(eq,y,x,solveOptions,odeOptions))
  end_if;
  if n>3 then
    error("currently only implemented up to third order equations")
  end_if;
  sysassign(ode::isReducibleFlag, FALSE);
  sols:= ode::expsols(ode::removeDenominator(eq, y, x, n, solveOptions, odeOptions), 
                      y, x, n, solveOptions, odeOptions);
  dim:= nops(sols);
  if dim=n then
    return( sols )
  end_if;
  Df:= genident();
  EF:= Dom::ExpressionField(normal);
  lodo:= Dom::LinearOrdinaryDifferentialOperator(Df,x,EF);
  if iszero(dim) then
    userinfo(2, "found no exponential solution");
    if n = 2 then
      return({})
    elif n = 3 then
      // try to find exponential solutions of the adjoint
      // equation
      userinfo(2,"try to find exponential solutions of the adjoint");
      eq_adj:= lodo::adjoint(lodo::mkLODO(eq,y(x)));
      sols:= ode::expsols(ode::removeDenominator(expr(eq_adj(y(x),Unsimplified)),
                                                 y,x,n,solveOptions, odeOptions),
                          y,x,n,solveOptions, odeOptions);
      dim:= nops(sols);
      if iszero(dim) then
        return({});
      else         // dim needs to be equal 1
        sysassign(ode::isReducibleFlag, TRUE);
        assert(nops(sols)=1);
        assert(not iszero(op(sols)));
        fac:= lodo::mkLODO([ode::normal(-diff(op(sols),x)/op(sols),Rationalize=None), 1]);
        eq2:= lodo::adjoint(lodo::rightQuotient(eq_adj, fac));
        eq2:= eq2(y(x),Unsimplified);
        sols:= ode::secondOrder(eq2,y,x,Irreducible,solveOptions,odeOptions);
        dim:= nops(sols);
        if iszero(dim) then
          sols:= ode::lookUp2ndOrderLinear(eq2,y,x,solveOptions,odeOptions);
          dim:=nops(sols);
        end_if;      
        if iszero(dim) then
          sols:=ode::specfunc(eq2,y,x,solveOptions,odeOptions);
          dim:=nops(sols);
        end_if:
        
        if iszero(dim) then
          return({})
        elif dim=1 then  // cannot treat this case currently, since
          // this means that we have only implicit
          // solutions given by a minimal polynomial
          //  --> solutions=RootOf(....)
          if ode::printWarningsFlag then
            ode::odeWarning("Only found implicit solutions. \n".
                    "         Cannot search for further solutions");
          end_if;
          return(sols)
        else     // search for a third solution by variation of
          // parameters
          userinfo(2, "two solutions found, searching for a ".
                              "third solution");
          b:= ode::firstOrder(lodo::adjoint(fac)(y(x),Unsimplified),
                               y,x,solveOptions, odeOptions);
          assert(not iszero(coeff(eq2,[diff(y(x),x,x)],1)));                     
          b:= op(b)/coeff(eq2,[diff(y(x),x,x)],1);
          v:= [op(sols)];
          w:= b/(ode::wronskian(v,x,EF,solveOptions,odeOptions));
          return(sols union
                 piecewise::extmap(-v[1]*int(v[2]*w,x,intOptions)
                                   +v[2]*int(v[1]*w,x,intOptions), DOM_SET))
        end_if;
      end_if;
    end_if
  else // found exponential solutions, try to find further solutions, 
       // i.e. dim > 0
    sysassign(ode::isReducibleFlag, TRUE);
    v:=op(sols);
    if n=2 then
      userinfo(2, "found an exponential solution and".
               " reduce the equation now...");
      de:=ode::secondOrderReduction(ode::normalize(eq,y,x,2,solveOptions,odeOptions),y,x,v,
                                    solveOptions,odeOptions);
      w:= op(ode::firstOrder(de, y, x, solveOptions, odeOptions));
      return(piecewise::extmap(v*int(w,x,intOptions), J -> sols union {J}))
    elif n=3 then
      // search for further solutions by variation of parameters
      userinfo(2, "found an exponential solution, try to find further ".
               "solutions");
      // should distinguish dim=1 or 2 but I don't feel to do it!
      w:= ode::normalize(
                  ode::wronskian([v,y(x)],x,EF,solveOptions,odeOptions),
                  y,x,dim,solveOptions,odeOptions);
      fac:= lodo::mkLODO(w,y(x));
      //  w   := ode::wronskian([v,y(x)],x,EF,solveOptions,odeOptions);
      //  fac := lodo::monic(lodo::mkLODO(w,y(x)));
      eq2 := lodo::rightQuotient(lodo::mkLODO(eq,y(x)), fac);
      complexity:= length(eq2(y(x),Unsimplified));
      if complexity < 200 then  
        b:= ode::liouvillianSolutions(eq2(y(x),Unsimplified),y(x),
                                      solveOptions,odeOptions);
      else                                 
        b:= {};
      end_if;  
      if b={} and complexity < 200 and nops(sols) = 1 then
        b:= ode::lookUp2ndOrderLinear(eq2(y(x),Unsimplified),y,x,solveOptions,odeOptions);        
      end_if:
      if b={} and complexity < 200  and nops(sols) = 1 then 
        b:= ode::specfunc(eq2(y(x),Unsimplified),y,x,solveOptions,odeOptions);
      end_if:
      if iszero(nops(b)) then return(sols) end_if;
      if dim=1 then
        assert(not iszero(v));
        for e in b do
          sols := sols union {v*hold(int)(e/v,x)};
        end_for;
        return(sols)
      else
        assert(nops(b)=1);
        v:= [v];
        w:= ode::wronskian(v,x,EF,solveOptions,odeOptions);
        assert(not iszero(w));
        w:= op(b,1)/w;
        return(sols union
               piecewise::extmap(-v[1]*int(v[2]*w,x,intOptions)
                                 +v[2]*int(v[1]*w,x,intOptions), DOM_SET))
      end_if
    end_if
  end_if
end_proc:

ode::firstOrder:=
  proc(eq, y, x, solveOptions,odeOptions) 
    local peq,intOptions,optIgnoreAnalyticConstraints;
  begin
    optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                  IgnoreAnalyticConstraints;
                else
                  null();
                end_if;
    intOptions:= null();            
    if has(solveOptions, IgnoreSpecialCases) then 
      intOptions:= intOptions,IgnoreSpecialCases;
    end_if;
    if has(solveOptions, IgnoreAnalyticConstraints) then   
      intOptions:= intOptions,IgnoreAnalyticConstraints;
    end_if;   
    peq:= ode::ode2poly(eq,y,x,1,solveOptions,odeOptions);
    if iszero(expr(coeff(peq,diff(y(x),x),1))) then 
      return({});
    else
      peq:= expr(-coeff(peq,y(x),1))/expr(coeff(peq,diff(y(x),x),1));
      peq:= int(peq,x,intOptions);
      peq:= piecewise::extmap(peq,
                              proc(J)
                                local expJ;
                              begin
                                expJ:= exp(J);
                                if type(expJ) = "exp" then
                                  {exp::simplify(expJ/*,optIgnoreAnalyticConstraints*/)}
                                else
                                  {expJ}
                                end_if
                              end_proc
                              );
      return(peq);                            
    end_if;  
  end_proc:


// This method is called by 'ode::liouvillianSolutions' in case that 
// the ODE to be solved is of order 4 or higher. We have no look-up
// or other specialized algorithms for 4th and higher order homogeneous
// linear ODEs. 
ode::reducibleCase:=
  proc(eq, y, x, n,solveOptions,odeOptions) 
    local EF, lodo, fl, Df;
    // save Df;
  begin
    if n=1 then 
      return(ode::firstOrder(eq,y,x,solveOptions, odeOptions)) 
    end_if;
    // delete Df;
    Df:= genident();
    EF:= Dom::ExpressionField(normal);  
    lodo:= Dom::LinearOrdinaryDifferentialOperator(Df,x,EF); 
    userinfo(1, "factorize equation");
    fl:= lodo::factors(lodo::mkLODO(eq, y(x)));
    if iszero(nops(fl)) then 
      return({}) 
    end_if;
    // 'func_call' just serves to evaluate the linear differential
    // operator 'fl' as 'y(x)'. 
    fl:= map(fl, lodo::func_call, y(x), Unsimplified);
    return(ode::combineSolutionsOfFactors(fl,y,x,n,solveOptions,odeOptions));
  end_proc:

  
ode::combineSolutionsOfFactors:=
  proc(fl,y,x,n,solveOptions,odeOptions) 
    local EF,i,sols,lsols,giveUp,ord,wr,w,v,u,nu,k,h,s,nls,intOptions;
  begin
    intOptions:= null();            
    if has(solveOptions, IgnoreSpecialCases) then 
      intOptions:= intOptions,IgnoreSpecialCases;
    end_if;
    if has(solveOptions, IgnoreAnalyticConstraints) then   
      intOptions:= intOptions,IgnoreAnalyticConstraints;
    end_if;   
    sols:= {};
    EF:= Dom::ExpressionField(normal);  
    giveUp:= FALSE;
    i:= nops(fl);
    if iszero(i) then 
      return({}) 
    end_if;
    
    userinfo(1, "solve factors and combine their solutions");
    
    while not(iszero(i) or giveUp) do 
      // the following lines are written instead of:                  
      // lsols:=ode::liouvillianSolutions(fl[i],y(x),"Irreducible");  
      lsols:= {};
      ord:= ode::getOrder(fl[i],y(x),solveOptions,odeOptions);
      userinfo(5, "solve factor of degree", ord);
      case ord
        of 0 do break;
        of 1 do 
          lsols:= {ode::firstord(fl[i],y,x,solveOptions, odeOptions)};
                            /*ode::firstOrder(fl[i],y,x);*/ 
                   // Note that 'ode::firstord' ist the generic method 
                   // to integrate a first order ODE, whereas 'ode::firstOrder'
                   // as defined above is restricted to the linear case. So it's
                   // better to call 'ode::firstord' instead of 'ode::firstOrder'
                   // here. 
          break;
        of 2 do if n>3 then 
                  lsols:= ode::secondOrder(fl[i],y,x,solveOptions,odeOptions)
                else 
                  lsols:= ode::secondOrder(fl[i],y,x,Irreducible,solveOptions, 
                                           odeOptions);
                end_if;
                // this if-statement is necessary since currently the 
                // lodo factorization is not complete.                
                break;
       otherwise 
         if ode::printWarningsFlag then
           ode::odeWarning("Cannot treat factors of degree >= 3".
                                   " currently. \n".
                         "         Hence, we may lose some ".
                       "Liouvillian solutions.");
          end_if;
      end_case;
      nls:=nops(lsols);
      if iszero(nls) then 
        break; 
      end_if;
      /* the following can only occure, when factor fl[i] is reducible 
        or we have found implicit solutions */
      if ord<>nls then 
        giveUp:=TRUE;
        if ord=2 and nls=1 and ode::printWarningsFlag then 
           ode::odeWarning("Found implicit solutions. \n".
                 "         Hence, cannot search for further solutions.");
        end_if;
      end_if;
      if iszero(nops(sols)) then sols:=lsols
       else
         // variation of parameters 
         userinfo(5, "combine solutions by variation of parameters");
         u:= [op(sols)];
         wr:= ode::normal(ode::wronskian(u,x,EF,solveOptions,odeOptions));
         // Since we can assume 'u' to be nontrivial, we can also 
         // assume that the Wronskian is nonzero. 
         assert(not iszero(wr)); 
         w:=[]; 
         nu:=nops(u);
         if nu=1 then 
           w:= [1] 
         else
           for k from 1 to nu do
             h:= u[k]; delete u[k];
             w:= append(w,ode::normal(ode::wronskian(u,x,EF,solveOptions,odeOptions)));
             u:= listlib::insertAt(u,h,k);
           end_for;
         end_if;
         for v in lsols do 
           s:= _plus(u[k]*(-1)^(nu+k)*int(ode::normal(v*w[k]/wr),x,intOptions) $ k=1..nu);
           sols:= piecewise::extmap(s, J -> sols union {J});   
         end_for;
      end_if;
      i:= i-1;
    end_while;
    
    return(sols);
  end_proc:

   


