/*++ ---------------- sympower.mu ---------------------
Description:
symmetricPower(Ly,y(x),m) computes the m-th symmetricPower of the 
linear homogeneous differential equation Ly. This is the lowest order
linear homogeneous differential equation whose solution space consists 
exactly of all possible m-th powerproducts of solutions of Ly.

Functions:

 - used Parameter:
    ++ Ly, y(x):DOM_EXPR
    ++ y, x    :DOM_IDENT
    ++ n, m    :DOM_INT
    ++ Ly   = ordinary linear differential equation
    ++ y(x) = the operator (function) of Ly  
    ++ y    = name of the operator of Ly
    ++ x    = dependant variable of the operator y
    ++ n    = order of Ly
    ++ m    = exponent of the symmetric power of Ly

 - ode::symmetricPower(Ly, y(x), m)
    ++ returns the m-th symmetricPower of an ordinary homogeneous
    ++ linear differential equation Ly.
    ++ The given arguments are tested.
 - ode::symPower(Ly, y, x, n, m)
    ++ returns the m-th symmetricPower of an ordinary
    ++ linear differential equation Ly.
    ++ NOTE: the reductum of Ly needs to be non-constant! Otherwise the result
    ++       maybe wrong!
 - ode::symPowerOrd2(Ly, y, x, m)
    ++ returns the m-th symmetricPower of a second order ordinary
    ++ linear differential equation Ly.
    ++ NOTE: the reductum of Ly needs to be non-constant! Otherwise the result
    ++       maybe wrong!
 - ode::fracFreeSymPowerOrd2(Ly, y, x, m)
    ++ returns the m-th symmetricPower of a second order ordinary
    ++ linear differential equation Ly, computed by a fraction free version.

See:
   - Singer, M. F. (1985). Solving homogeneous linear differential 
     equations in terms of second order linear differential equations. 
     Amer. J. Math. 107
   - Bronstein, M., Mulders, T., Weil, J.A. (1997). On Symmetric Powers of
     Differential Operators. In: Proceedings of ISSAC'97, Maui, Hawaii,
     156-163.

Example:

>> eq := diff(y(x),x,x)+1/x*diff(y(x),x)-1/(4*x^2)*y(x):
>> ode::symmetricPower(eq, y(x), 2);
 
                 3 diff(y(x), x, x)
                 ------------------ + diff(y(x), x, x, x)
                         x
Hurwitz-equation:
>> eq:=x^2*(x-1)^2*diff(y(x),x,x,x)+(7*x-4)*x*(x-1)*diff(y(x),x,x)+
 (72/7*(x^2-x)-20/9*(x-1)+3/4*x)*diff(y(x),x)+(72*11/7^3*(x-1)+5/8+2/63)*y(x):
>> ode::symmetricPower(eq, y(x), 2);
++*/

ode::symmetricPower:= proc(eq,z,m,solveOptions={},odeOptions={})
begin
  eq:= ode::isLODE(eq,z,Hlode,solveOptions,odeOptions);
  case nops(eq)
  of 1 do 
    error("not an ordinary homogeneous linear differential equation");
  of 4 do
    if eq[4] < 1 then
      error("only defined for positive order equations");
    end_if;
  otherwise
    if domtype(m) <> DOM_INT or m < 1 then 
      error("only defined for positive integers");
    end_if;
  end_case;
  if eq[4]=2 then
    delete eq[4];
    return(ode::symPowerOrd2(eq,m,solveOptions,odeOptions));
  else
    return(ode::symPower(eq,m,solveOptions,odeOptions));
  end_if;
end_proc:

ode::symPower:= proc(eq,z,x,n,m,solveOptions, odeOptions)
  local Y,dxn,c,i,j,ansatz,dimBasis,newDimBasis,diffvars,solset,term,tmp,
        optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions,IgnoreAnalyticConstraints) then 
              IgnoreAnalyticConstraints;
            else
              null();
            end_if;
  z:= z(x);
  Y:= z^m;
  dxn:= diff(z, x$n);
  diffvars:= [diff(z, x$j) $ j=0..n-1];
  assert(not iszero(coeff(eq,[dxn],1)));
  term:= expand(-coeff(eq,[dxn],0)/coeff(eq,[dxn],1),optIgnoreAnalyticConstraints);
  c[0]:= genident("c");
  ansatz:= c[0]*Y;
  for i from 1 to n-1 do
    Y:= diff(Y, x);
    c[i]:= genident("c");
    ansatz:= ansatz + c[i]*Y;
  end_for;    
  i:= n-1;
  newDimBasis:= nterms(ansatz, diffvars);
  repeat
    i:= i+1;
    c[i]:= genident("c");
    dimBasis:= newDimBasis;
    userinfo(5, "order is now: ", i);
    Y:= expand(subs(diff(Y, x),dxn=term,EvalChanges),optIgnoreAnalyticConstraints);
    ansatz:= expand(ansatz + c[i]*Y, optIgnoreAnalyticConstraints);
    newDimBasis:= nterms(ansatz, diffvars);
  until dimBasis = newDimBasis
  end_repeat; 
  repeat
    userinfo(5, "...solve ansatz");
    tmp:= [coeff(poly(subs(ansatz, c[i]=1), diffvars))];
    solset:= linsolve({op(tmp,j)=0 $j=1..dimBasis}, {c[j] $j=0..i-1});
    if solset <> FAIL and solset <> null() then
      return(subs( _plus(c[j]*diff(z, x$j) $j=0..i),{c[i]=1} union {op(solset)}, EvalChanges)); 
    end_if;
    i:= i+1;
    userinfo(5, "order is now: ", i);
    c[i]:= genident("c");
    Y:= expand(subs(diff(Y, x),dxn=term),optIgnoreAnalyticConstraints);
    ansatz:= expand(ansatz + c[i]*Y,optIgnoreAnalyticConstraints);
  until FALSE 
  end_repeat;
end_proc: 

ode::symPowerOrd2:= proc(eq,z,x,m,solveOptions,odeOptions) 
  local c, Y, i, j, dxn, sys, unk, ind, e, eqn, sols, 
        ansatz, dimBasis, newDimBasis, diffvars, solset, term, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
              IgnoreAnalyticConstraints;
            else
              null();
            end_if;
  z:= z(x);
  Y:= z^m;
  dxn:= diff(z, x$2);
  diffvars:= [diff(z, x$j) $ j=0..2]; 
  assert(not iszero(coeff(eq, [dxn],1)));
  term:= expand(-coeff(eq,[dxn],0)/coeff(eq,[dxn],1),optIgnoreAnalyticConstraints);
  c[0]:= genident("c");
  ansatz:= c[0]*Y;
  Y:= diff(Y, x);
  c[1]:= genident("c");
  ansatz:= ansatz + c[1]*Y;  
  i:= 1; 
  newDimBasis:= nterms(ansatz, diffvars);
  repeat
    i:= i+1;
    dimBasis:= newDimBasis;
    userinfo(5, "order is now: ", i);
    Y:= expand(subs(diff(Y, x), dxn=term,EvalChanges),optIgnoreAnalyticConstraints);
    c[i]:= genident("c");
    ansatz:= ansatz + c[i]*Y;
    newDimBasis:= nterms(ansatz, diffvars);
  until dimBasis = newDimBasis 
  end_repeat; 
  repeat
    userinfo(5, "...solve ansatz");
    sys:= {coeff(poly(ansatz, diffvars))};
    unk:= {c[j] $ j=0..i-1};
    // the following lines avoid an expensive call to linsolve 
    sols:= null();
    repeat
      e:= FALSE; 
      for eqn in sys do
        ind:= indets(eqn) intersect unk;
        if nops(ind)=1 then
          ind:= op(ind);
          if degree(eqn,[ind])=1 then
            sys:= sys minus {eqn}; 
            unk:= unk minus {ind};
            // normal is the expensive call 
            //eqn := ind=normal(-coeff(eqn,[ind],0)/coeff(eqn,[ind],1));
            assert(not iszero(coeff(eqn,[ind],1)));
            eqn:= ind = expand(-coeff(eqn,[ind],0)/coeff(eqn,[ind],1),optIgnoreAnalyticConstraints);
            sols:= sols,eqn; 
            sys:= subs(sys,eqn,EvalChanges); 
            e:= TRUE; 
            break;
          end_if
        end_if
      end_for;
    until e=FALSE 
    end_repeat;
    sys:= sys minus {0};
    if sys <> {} then
      solset:= linsolve(sys,unk);
    else 
      solset:= {} 
    end_if;
    if solset <> FAIL and solset <> null() then
      solset:= {sols} union solset;
      if has(solset, c[i]) then 
        solset:= subs(solset, c[i]=1) union {c[i]=1};
      else 
        solset:= solset union {c[i]=0};
      end_if;
      return(subs(_plus(c[j]*diff(z,x$j) $ j=0..i), solset,EvalChanges));
    end_if;
    i:= i+1;
    userinfo(5, "order is now: ", i);
    Y:= expand(subs(diff(Y, x),dxn=term,EvalChanges),optIgnoreAnalyticConstraints);
    c[i]:= genident("c");
    ansatz:=  ansatz + c[i]*Y;
  until FALSE 
  end_repeat;
end_proc:



ode::fracFreeSymPowerOrd2:= proc(eq,y,x,m,solveOptions,odeOptions)
  local veq,h,p,a,b,i,L,vl,p1,pe,po,optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
              IgnoreAnalyticConstraints;
            else
              null();
            end_if;
  vl:= [diff(y(x),x $ (m+1-i)) $ i=0..m,y(x)];
  L:= genident("_L");
  veq:= ode::vectorize(eq,y,x,2,solveOptions,odeOptions);
  h:= lcm(op(map(map(veq,denom),poly,[x])));
  if degree(h) <> 0 then 
    h:= expr(h); 
    veq:= map(map(veq,_mult,h),ode::normal,Rationalize=None,Expand=FALSE) 
  end_if; 
  p:= veq[3];
  a:= veq[2];
  b:= veq[1];
  p1:= collect(diff(p,x),x);
  pe:= gcd(gcd(p,p1),a-p1);
  assert(not iszero(pe));
  //po:= gcd(pe,ode::normal(b*p/pe,Rationalize=None,Expand=FALSE));
  po:= gcd(pe,divide(b*p,pe,[x],Quo));
  assert(not iszero(po));
  L[0]:= y(x);
  L[1]:= ode::normal(diff(L[0],x),Rationalize=None,Expand=FALSE);
  L[2]:= p*diff(y(x),x,x)+a*diff(y(x),x)+m*b*y(x);
  L[3]:= collect(p/pe*diff(L[2],x)+(2*(a-p1)/pe+p1/pe)*L[2]+
                 2*(m-1)*b*p/pe*L[1],vl,ode::normal);
  for i from 2 to trunc((m+1)/2) do
    L[2*i]:= op(mapcoeffs(
                 poly(p/po*diff(L[2*i-1],x)+L[2*i-1]*((2*i-1)*(a-p1)/po+
                      p1/po+(i-1)*p*diff(pe,x)/po/pe+(i-2)*p*diff(po,x)/po^2)+
                      (2*i-1)*(m-(2*i-2))*b*p/po/pe*L[2*i-2],
                      vl), ode::normal,Rationalize=None,Expand=FALSE),1);
    L[2*i+1]:= op(mapcoeffs(
                 poly(p/pe*diff(L[2*i],x)+L[2*i]*(2*i*(a-p1)/pe+p1/pe+
                        (i-1)*p*diff(pe,x)/pe^2+(i-1)*p*diff(po,x)/pe/po)+
                        2*i*(m-(2*i-1))*b*p/po/pe*L[2*i-1],
                        vl), ode::normal,Rationalize=None,Expand=FALSE),1);
  end_for;
                      
  return(collect(L[m+1],vl));
end_proc:

