/*
  REFERENCE. 
   [1] Kai Gehrs: Symbolic algorithms for the computation of    
       hypergeometric solutions of linear ODEs, internal paper, 
       2009 
  
  NOTE that especially all references within the code below refer 
  to this paper. The names for parameters, variables and the 
  numbering of equations, identities and facts are chosen due to 
  this work. The comments in this code are possibly not detailed 
  enough to understand the algorithmic ideas. 

  
  PARAMETERS.
   eq -- an expression encoding a 2nd order homogeneous linear ODE
    x -- the independent ODE variable
    y -- the dependent ODE variable 

   The (standard) options are as usual. 


  OUTPUT. 
   The methods returns a set of two independent solutions of 
   the 2nd order homogeneous linear ODE encoded by 'eq' or 'FAIL' 
   if no such solution could be found. 
      
*/

ode::lookUp2F1:= proc(eq,y,x,solveOptions={},odeOptions={})
  local intOptions, optIgnoreAnalyticConstraints, y0, y1, y2, A, B, 
        lcoeff, I1, J1, xpow, d, n, kTilde, k, J0, I0, denomFactors,
        numerI0, denomI0, p, q, Exponents2F1, L1, L2, a, aa, b, bb, 
        c, cc, dd, sol1, sol2, sys, fundSys, Sol_eq2F1, i, t, A_bar, 
        A_tilde, MoebiusTrans, eq2F1, eqI0, tmp, ind, m, unknownFunctions,
        Subs, sysSpecial;
  save MAXEFFORT;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  aa:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  bb:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa]));
  cc:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb]));
  dd:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc]));
   a:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd]));
   b:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,a]));
   c:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,a,b]));
   t:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,a,b,c]));
   eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) or (not has(eq,y0) and not has(eq,y1)) then 
    return(FAIL)
  end_if;
  unassume(x);
  lcoeff:= diff(eq,y2);
  A:= ode::normal(diff(eq,y1)); 
  B:= ode::normal(diff(eq,y0)); 
  // The algorithm cannot treat ODEs where 'A' and 'B' involve arbitrary functions
  // like 'f(x)' for a non-specified 'f'. We need to check this first: 
  unknownFunctions:= indets([A,B,lcoeff], All) minus indets([A,B,lcoeff]);
  unknownFunctions:= select(unknownFunctions, v -> domtype(eval(v)) = DOM_IDENT);
  if unknownFunctions <> {} then 
    return(FAIL);
  end_if;  
  // We consider ODEs of the form (2.1) from [1], i.e. if 'lcoeff <> 1' we need 
  // to divide by it.  
  if lcoeff <> 1 then 
    A:= ode::normal(A/lcoeff);
    B:= ode::normal(B/lcoeff);
  end_if;    
  //------------------------------------------------------------------------------  
  // [1], Step (1)
  //------------------------------------------------------------------------------  
  I1:= diff(A,x)/2 + A^2/4 - B; 
  //------------------------------------------------------------------------------  
  // [1], Step (2)
  //------------------------------------------------------------------------------  
  J1:= ode::normal(x^2*I1+1/4);
  //------------------------------------------------------------------------------  
  // [1], Step (3)
  //------------------------------------------------------------------------------  
  xpow:= {}:
  misc::maprec(combine(J1,optIgnoreAnalyticConstraints),
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc):  
  // Special treatment for the case 'xpow = {}'.
  if nops(xpow) = 0 then 
    // Give up only if 'J1' does not contain 'x' at all. 
    if has(J1,x) then 
      xpow:= {1};
    else 
      return(FAIL);
    end_if;  
  end_if;
  d:= lcm(op(map(xpow,denom)));
  xpow:= map(xpow,_mult,d);
  n:= gcd(op(xpow));
  map(xpow,_mult,1/n);
  kTilde:= d/n; 
  k:= 1/kTilde;
  if k =0 then 
    return(FAIL)
  end_if;
  //------------------------------------------------------------------------------  
  // [1], Step (4)
  //------------------------------------------------------------------------------  
  J0:= evalAt(J1/k^2,x=x^(1/k));
  //------------------------------------------------------------------------------  
  // [1], Step (5)
  //------------------------------------------------------------------------------  
  I0:= ode::normal((J0-1/4)/x^2); // ode::normalization is necessary due to the fact that 
                             // 'numer' and 'denom' are called on this expression 
                             // and may not ode::normalize sufficiently. 
  /* 
    Now check whether 'I0' is of the form as indicated by [1], Lemma 2.4. 
    If this is not the case, stop the computation and return 'FAIL'. 
  */
  numerI0:= numer(I0,Expand=TRUE);
  denomI0:= denom(I0,Expand=TRUE);
  if not iszero(expand(diff(numerI0,x$3),optIgnoreAnalyticConstraints)) or 
     not iszero(expand(diff(denomI0,x$7),optIgnoreAnalyticConstraints)) then  
    numerI0:= misc::maprec(combine(numerI0,IgnoreAnalyticConstraints),
                 {"_power"} = (elem -> _power(op(elem,1),ode::normal(op(elem,2)))));  
    denomI0:= misc::maprec(combine(denomI0,IgnoreAnalyticConstraints),
                 {"_power"} = (elem -> _power(op(elem,1),ode::normal(op(elem,2)))));  
    if not iszero(expand(diff(numerI0,x$3),optIgnoreAnalyticConstraints)) or 
       not iszero(expand(diff(denomI0,x$7),optIgnoreAnalyticConstraints)) then  
      return(FAIL);
    end_if;  
  end_if;  
  //------------------------------------------------------------------------------  
  // [1], Step (6)
  //------------------------------------------------------------------------------  
  denomI0:= factor(denomI0);
  //------------------------------------------------------------------------------  
  // [1], Step (7)
  //------------------------------------------------------------------------------  
  p:= degree(numerI0,x);
  //denomFactors:= select([op(expr(denomI0))],hastype,"_power");
  //q:= sort(map(denomFactors, elem -> if has(op(elem,1),x) then op(elem,2) end_if));
  //q1:= sort([coerce(denomI0,DOM_LIST)[i] $ i = 3..nops(denomI0) step 2]);
  denomFactors:= coerce(denomI0,DOM_LIST);
  q:= [];
  for i from 2 to nops(denomFactors) do 
    if has(denomFactors[i],x) then 
      q:= q.[denomFactors[i+1]];
    end_if;
  end_for;
  /* 
     Now '[p,q]' is a nested list of the form [1], (2.13). Now do the 
     classification due [1], Lemma 2.4. 
  */
  Exponents2F1:= [
      [2,[2,2,2]],[1,[1,2,2]],[0,[0,2,2]],[0,[1,1,2]], // [1], row [2*,[2*,2*,2*]]
      [1,[2,2,2]],[0,[1,2,2]],                         // [1], row [1*,[2*,2*,2*]]
      [0,[2,2,2]],                                     // [1], row [0 ,[2*,2*,2*]]
      [2,[2,2]],[1,[1,2]],[0,[1,1]],[0,[0,2]],         // [1], row [2*,[2*,2*]]
      [1,[2,2]],[0,[1,2]],                             // [1], row [1*,[2*,2*]]
      [0,[2,2]]                                        ]; 
  if contains(Exponents2F1,[p,q]) = 0 then 
    // Then 'ode::specfunc' is a good candidate for an algorithm to be 
    // tried out to find solutions. 
    return(FAIL);    
  end_if;
  //------------------------------------------------------------------------------  
  // [1], Step (8)
  //------------------------------------------------------------------------------  
  numerI0:= poly(numerI0,[x]);
  denomI0:= poly(denomI0,[x]);
  /*
     Now compute the general form of the invariant of the 2F1-equation after application of 
     Moebius transfrom with general parameters. Then use this invariant and 'I0' to set up
     a system of equations for determining 'aa','bb','cc','dd','a','b' and 'c'. The 
     necessary commands for setting up the system(s) of equations needed for Step (8)(i) and 
     Step (8)(ii) from [1] are here:

       >> Moebius:= (aa*x+bb)/(cc*x+dd):
       >> A:= ((a+b+1)*x-c)/(x^2-x):
       >> B:= b*a/(x^2-x):
       >> Inv_eq2F1Normal:= collect(ode::normal(diff(A,x)/2+A^2/4-B),x):
       >> t:= factor(ode::normal(diff(Moebius,x)^2*evalAt(Inv_eq2F1Normal,x=Moebius))):
       >> denomI0Unknowns:= poly(expand(expr(denom(t))),[x]);
       >> numerI0Unknowns:= poly(expand(expr(numer(t))),[x]);

     Note that in the following:
       - The coefficients of 'denomI0Unknowns' contribute the right-hand-sides 
         of the system defined right here before [1], Step (8)(i), i.e. 
         'sys = {(coeff(denomI0,i) = coeff(denomI0Unknowns,i)) $ i = 0..6}' 
       - The coefficients of 'numerI0Unknowns' contribute the right-hand-sides
         of the system defined in [1], Step (8)(ii), i.e.                                           
         'sys = {(coeff(numerI0,i) = coeff(numerI0Unknowns,i)) $ i = 0..2}' 
  */
  sys:= {coeff(denomI0,0) = 4*bb^4*dd^2 - 8*bb^3*dd^3 + 4*bb^2*dd^4,
         coeff(denomI0,1) = 16*aa*bb^3*dd^2 - 24*aa*bb^2*dd^3 + 16*bb^2*cc*dd^3 - 
                            24*bb^3*cc*dd^2 + 8*aa*bb*dd^4 + 8*bb^4*cc*dd, 
         coeff(denomI0,2) = 24*aa^2*bb^2*dd^2 - 24*aa^2*bb*dd^3 + 4*aa^2*dd^4 + 
                            32*aa*bb^3*cc*dd - 72*aa*bb^2*cc*dd^2 + 32*aa*bb*cc*dd^3 + 
                            4*bb^4*cc^2 - 24*bb^3*cc^2*dd + 24*bb^2*cc^2*dd^2, 
         coeff(denomI0,3) = 16*aa^3*bb*dd^2 - 8*aa^3*dd^3 + 48*aa^2*bb^2*cc*dd - 
                            72*aa^2*bb*cc*dd^2 + 16*aa^2*cc*dd^3 + 16*aa*bb^3*cc^2 - 
                            72*aa*bb^2*cc^2*dd + 48*aa*bb*cc^2*dd^2 - 8*bb^3*cc^3 + 
                            16*bb^2*cc^3*dd, 
         coeff(denomI0,4) = 4*aa^4*dd^2 + 32*aa^3*bb*cc*dd - 24*aa^3*cc*dd^2 + 
                            24*aa^2*bb^2*cc^2 - 72*aa^2*bb*cc^2*dd + 24*aa^2*cc^2*dd^2 -
                            24*aa*bb^2*cc^3 + 32*aa*bb*cc^3*dd + 4*bb^2*cc^4, 
         coeff(denomI0,5) = 16*aa^3*bb*cc^2 - 24*aa^2*bb*cc^3 + 16*aa^2*cc^3*dd - 
                            24*aa^3*cc^2*dd + 8*aa*bb*cc^4 + 8*aa^4*cc*dd, 
         coeff(denomI0,6) = 4*aa^4*cc^2 - 8*aa^3*cc^3 + 4*aa^2*cc^4          
         };
  //-----------------  
  // [1], Step (8)(i)
  //-----------------  
  L1:= solve(sys,{aa,bb,cc,dd},op(solveOptions),IgnoreAnalyticConstraints);
  // If no solution could be found, then return 'FAIL'. 
  if type(L1) <> DOM_SET or (nops(L1) = 0 or (nops(L1) > 0 and type(L1[1]) <> DOM_LIST)) then 
    return(FAIL);
  end_if;
  /* 
     We apply a heuristic startegy to choose a solution, which is as simple as possible. 
     Currently we use 'length' as a measure for complexity. 
     The function 'length' may be replaced by something more elaborate 
     if this shows up to be important in the future. 
  */
  L1:= [op(L1)];
  tmp:= map(map([op(L1)], elem -> map(elem,rhs)),length);
  m:= min(map(map([op(L1)], elem -> map(elem,rhs)),length));
  if type(m) = DOM_INT and m > 0 then 
    ind:= contains(tmp,m);
    sol1:= L1[ind];
  else 
    ind:= 1;
    sol1:= L1[1];
  end_if;
  //------------------
  // [1], Step (8)(ii)
  //------------------  
  sys:= {coeff(numerI0,0) = a^2*aa^2*bb^2*dd^2 - 
                            2*a^2*aa*bb^3*cc*dd + a^2*bb^4*cc^2 - 2*a*aa^2*b*bb^2*dd^2 + 
                            4*a*aa^2*b*bb*dd^3 - 2*a*aa^2*bb*c*dd^3 + 4*a*aa*b*bb^3*cc*dd - 
                            8*a*aa*b*bb^2*cc*dd^2 + 4*a*aa*bb^2*c*cc*dd^2 - 2*a*b*bb^4*cc^2 + 
                            4*a*b*bb^3*cc^2*dd - 2*a*bb^3*c*cc^2*dd + aa^2*b^2*bb^2*dd^2 - 
                            2*aa^2*b*bb*c*dd^3 - aa^2*bb^2*dd^2 + 2*aa^2*bb*c*dd^3 + 
                            aa^2*c^2*dd^4 - 2*aa^2*c*dd^4 - 2*aa*b^2*bb^3*cc*dd + 
                            4*aa*b*bb^2*c*cc*dd^2 + 2*aa*bb^3*cc*dd - 4*aa*bb^2*c*cc*dd^2 - 
                            2*aa*bb*c^2*cc*dd^3 + 4*aa*bb*c*cc*dd^3 + b^2*bb^4*cc^2 - 
                            2*b*bb^3*c*cc^2*dd - bb^4*cc^2 + 2*bb^3*c*cc^2*dd + 
                            bb^2*c^2*cc^2*dd^2 - 2*bb^2*c*cc^2*dd^2, 
         coeff(numerI0,1) = 2*a^2*aa^3*bb*dd^2 - 4*a^2*aa^2*bb^2*cc*dd + 2*a^2*aa*bb^3*cc^2 - 
                            4*a*aa^3*b*bb*dd^2 + 4*a*aa^3*b*dd^3 - 2*a*aa^3*c*dd^3 + 
                            8*a*aa^2*b*bb^2*cc*dd - 4*a*aa^2*b*bb*cc*dd^2 + 
                            2*a*aa^2*bb*c*cc*dd^2 - 4*a*aa*b*bb^3*cc^2 - 4*a*aa*b*bb^2*cc^2*dd + 
                            2*a*aa*bb^2*c*cc^2*dd + 4*a*b*bb^3*cc^3 - 2*a*bb^3*c*cc^3 + 
                            2*aa^3*b^2*bb*dd^2 - 2*aa^3*b*c*dd^3 - 2*aa^3*bb*dd^2 + 
                            2*aa^3*c*dd^3 - 4*aa^2*b^2*bb^2*cc*dd + 2*aa^2*b*bb*c*cc*dd^2 + 
                            4*aa^2*bb^2*cc*dd - 2*aa^2*bb*c*cc*dd^2 + 2*aa^2*c^2*cc*dd^3 - 
                            4*aa^2*c*cc*dd^3 + 2*aa*b^2*bb^3*cc^2 + 2*aa*b*bb^2*c*cc^2*dd - 
                            2*aa*bb^3*cc^2 - 2*aa*bb^2*c*cc^2*dd - 4*aa*bb*c^2*cc^2*dd^2 + 
                            8*aa*bb*c*cc^2*dd^2 - 2*b*bb^3*c*cc^3 + 2*bb^3*c*cc^3 + 
                            2*bb^2*c^2*cc^3*dd - 4*bb^2*c*cc^3*dd,
         coeff(numerI0,2) = a^2*aa^4*dd^2 - 2*a^2*aa^3*bb*cc*dd + a^2*aa^2*bb^2*cc^2 - 
                            2*a*aa^4*b*dd^2 + 4*a*aa^3*b*bb*cc*dd + 4*a*aa^3*b*cc*dd^2 - 
                            2*a*aa^3*c*cc*dd^2 - 2*a*aa^2*b*bb^2*cc^2 - 
                            8*a*aa^2*b*bb*cc^2*dd + 4*a*aa^2*bb*c*cc^2*dd + 
                            4*a*aa*b*bb^2*cc^3 - 2*a*aa*bb^2*c*cc^3 + aa^4*b^2*dd^2 - 
                            aa^4*dd^2 - 2*aa^3*b^2*bb*cc*dd - 2*aa^3*b*c*cc*dd^2 + 
                            2*aa^3*bb*cc*dd + 2*aa^3*c*cc*dd^2 + aa^2*b^2*bb^2*cc^2 + 
                            4*aa^2*b*bb*c*cc^2*dd - aa^2*bb^2*cc^2 - 4*aa^2*bb*c*cc^2*dd + 
                            aa^2*c^2*cc^2*dd^2 - 2*aa^2*c*cc^2*dd^2 - 2*aa*b*bb^2*c*cc^3 + 
                            2*aa*bb^2*c*cc^3 - 2*aa*bb*c^2*cc^3*dd + 4*aa*bb*c*cc^3*dd + 
                            bb^2*c^2*cc^4 - 2*bb^2*c*cc^4};    
  sol2:= FAIL;                           
  if traperror((sysSpecial:= evalAt(sys,sol1))) = 0 then 
    //-------------------  
    // [1], Step (8)(iii)
    //-------------------  
    L2:= solve(sysSpecial,{a,b,c},IgnoreAnalyticConstraints);
    if type(L2) <> DOM_SET or nops(L2) = 0 or (nops(L2) > 0 and type(L2[1]) <> DOM_LIST) then 
      sol2:= FAIL;
    else 
      sol2:= L2[1];
    end_if;  
  end_if;
  if sol2 = FAIL then 
    delete L1[ind]; // remove 'sol1' 
    L1:= sort(L1, (x,y) -> length(x) < length(y));
    for sol1 in L1 do 
      if sol2 = FAIL and type(sol1) = DOM_LIST and traperror((sysSpecial:= evalAt(sys,sol1))) = 0 then 
        //-------------------  
        // Still: [1], Step (8)(iii) (trying the other soltuions)
        //-------------------  
        L2:= solve(sysSpecial,{a,b,c},IgnoreAnalyticConstraints);
        if type(L2) <> DOM_SET or nops(L2) = 0 or (nops(L2) > 0 and type(L2[1]) <> DOM_LIST) then 
          next;
        else 
          sol2:= L2[1];
          break;
        end_if;  
      end_if;
    end_for;  
  end_if;  
  
  if sol2 = FAIL then 
    return(FAIL); 
  end_if;
  
  /* 
     Intermediate verification opportunity: To check the results up to here 
     use the following lines. If the last command gives 'TRUE', then everything 
     is o.k. 

       F:= simplify((aa*x^k+bb)/(cc*x^k+dd) | sol1);
       A:= ((a+b+1)*x-c)/(x^2-x);
       B:= b*a/(x^2-x);
       Inv_eq2F1Normal:= collect(ode::normal(diff(A,x)/2+A^2/4-B),x);
       Kai0:= simplify(Inv_eq2F1Normal | sol2);
       Kai1:= simplify(diff(F,x)^2*evalAt(Kai0,x=F) + 3*diff(F,x,x)^2/(4*diff(F,x)^2) - 
                        diff(F,x,x,x)/(2*diff(F,x)));
       print("Intermediate verification oppurtunity");
       print(testeq(simplify(simplify(I1)-Kai1,IgnoreAnalyticConstraints),0));
  */
  Sol_eq2F1:= [{hypergeom([a, b],[c],x), 
                x^(1-c)*hypergeom([b+1-c,a+1-c],[2-c],x)}, 
               {hypergeom([a,b],[a+b+1-c],1-x), 
                (1-x)^(c-a-b)*hypergeom([c-a,c-b],[c+1-a-b],1-x)}, // fr c=1
               {x^(-a)*hypergeom([a,a+1-c],[a+1-b],1/x),
                x^(-b)*hypergeom([b+1-c,b],[b+1-a],1/x)}
              ];
  fundSys:= FAIL;
  for i from 1 to 3 do 
    if traperror((fundSys:= evalAt(Sol_eq2F1[i],sol2))) = 0 and nops(fundSys) = 2 then 
      break;
    elif traperror((fundSys:= evalAt(Sol_eq2F1[i],sol2))) = 0 and i = 3 and nops(fundSys) = 1 then 
      break;
    else 
      fundSys:= FAIL;
    end_if;  
  end_for;  
  if fundSys = FAIL then 
    return(FAIL);
  end_if;  
  //------------------------------------------------------------------------------  
  // [1], Step (9)
  //------------------------------------------------------------------------------  
  // Compute 'A_tilde' as in [1], (2.15)
  eqI0:= y2-subs(I0,x=t)*y0;
  tmp:= evalAt(eqI0, y2 = (y2-diff(x^k,x,x)*(1/diff(x^k,x)*y1))*
                              1/diff(x^k,x)^2, 
                      t = x^k);
  A_tilde:= ode::normal(diff(tmp*1/diff(tmp,y2),y1));
  // Compute 'A_bar' as in [1], (2.16)
  eq2F1:= evalAt((x^2-x)*diff(y(x),x,x)+((a+b+1)*x-c)*diff(y(x),x)+b*a*y(x),sol2);
  MoebiusTrans:= simplify(evalAt((aa*t+bb)/(cc*t+dd),sol1));
  // Establishing transformation rules changing dependent variable. 
  Subs:= [0 $ 2]; 
  Subs[1]:= simplify(1/diff(MoebiusTrans,t)*diff(y(t),t)); // apply [1], (2.8) from Lemma 2.3
  for i from 2 to 2 do 
    Subs[i]:= simplify(1/diff(MoebiusTrans,t)*diff(Subs[i-1],t)); // apply [1], (2.9) from Lemma 2.3 
  end_for;
  Subs:= subs(Subs,[diff(y(t),t,t) = y2, diff(y(t),t) = y1, y(t) = y0]);
  tmp:= evalAt(eq2F1,[diff(y(x),x) = Subs[1], 
                      diff(y(x),x,x) = Subs[2], 
                      x = MoebiusTrans]); // finally transforming the idependent variable 
  A_bar:= ode::normal(diff(tmp*1/diff(tmp,y2),y1));
  //-----------------  
  // [1], Step (9)(i)
  //-----------------  
  if traperror((fundSys:= evalAt(fundSys,x=MoebiusTrans))) <> 0 then 
    return(FAIL)
  end_if; 
  //------------------  
  // [1], Step (9)(ii)
  //------------------  
  fundSys:= map(fundSys,_mult,exp(1/2*int(A_bar,t,intOptions)));
  //-------------------  
  // [1], Step (9)(iii)
  //-------------------  
  fundSys:= evalAt(fundSys,t=x^k); 
  //------------------  
  // [1], Step (9)(iv)
  //------------------  
  fundSys:= map(fundSys,_mult,exp(1/2*int(A_tilde-A,x,intOptions)));
  // Simplify the solutions to be returned as far as possible.   
  fundSys:= simplify(fundSys,IgnoreAnalyticConstraints);
  /* 
     Remove unnecessary constant factors from the fundamental set of 
     solutions, i.e.: if 'c*f(x)' has been found as a solution, we
     should better return 'f(x)' and remove 'c'. 
  */
  fundSys:= map(fundSys,elem -> if type(elem) = "_mult" then 
                                  _mult(op(select({op(elem)},has,x)))
                                else 
                                  elem
                                end_if);                              
  return(fundSys);                   
    
end_proc:



