/*
  =========================================
  LOOK-UP METHODS FOR 2nd ORDER LINEAR ODEs
  =========================================

  REFERENCE. Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 
  -----------
  PARAMETERS. 
  -----------

            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' may contain 'hypergeom' for 
                  returning the solution in terms of 'hypergeom'
                  in the most general case

 */
  

/* 
   This is the main function coordinating all look-ups (except for those
   in the frame of 'ode::specfunc' for 2nd order linear ODEs. 
*/
 
ode::lookUp2ndOrderLinear:=
proc(eq, y, x, solveOptions={}, odeOptions={}) 
  local sol, y0, y1, y2, dterm0, dterm1, dterm2, eqsubs, oldLookUpUsed, 
          intOptions, optIgnoreAnalyticConstraints, csts, inits;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(odeOptions,#inits) then 
    inits:= select(odeOptions,has,#inits)[1];
  else 
    inits:= {};
  end_if;  
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  oldLookUpUsed:= FALSE; 
  sol:= FAIL; 
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_1_eq_4_and_6_p_214(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_1_eq_7_p_215(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_1_eq_9_p_215(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_1_eq_10_p_215(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_2_eq_13_14_p_216(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_2_eq_15_16_17_p_216(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_2_eq_32_p_217(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_2_eq_33_34_35_p_217(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_2_eq_38_39_40_42_43_p_217_218(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_2_eq_45_46_p_218(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_2_eq_50_51_52_53_54_p_218(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_2_eq_55_56_57_58_59_60_p_218_219(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_4_eq_118_119_132_133_134_143_144_145_147_p_226_228_229_230(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_5_eq_149_p_230(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_5_eq_154_p_230(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_5_eq_169_170_p_233_234(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_5_eq_hypergeom(eq,y,x,solveOptions,odeOptions);
    if has(sol,hypergeom) then 
      sol:= FAIL;
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_6_eq_195_196_197_199_200_201_205_206_207_208_209_210_p_238(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_7_eq_215_217_218_219_228_229_230_231_233_234_p_240(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if; 
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_8_eq_242_243_245_247_248_249_250_251_252_254_to_265_p_244(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eqsubs:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if sol = FAIL or sol = {} then 
    /* Check whether the ODE is of the form 

              (a2*x+b2)*y''(x) + (a1*x+b1)*y'(x) + (a0*x+b0)*y(x)    (*)

       If so also try the old look-ups in 'ode::specfunc'. These provide 
       simpler solutions than the ones presented in the book by 
       Polyanin & Zaitsev for equations of type (*).
    */
    dterm0:= combine(expand(diff(eqsubs,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm1:= combine(expand(diff(eqsubs,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm2:= combine(expand(diff(eqsubs,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    if iszero(dterm2) then
      return(FAIL);
    end_if;
    if not has(ode::normal(diff(dterm0,x)),x) and  
       not has(ode::normal(diff(dterm1,x)),x) and 
       not has(ode::normal(diff(dterm2,x)),x) then 
      MAXEFFORT:= MAXEFFORT/20; 
      sol:= ode::specfunc(eq, y, x, solveOptions, odeOptions);
      MAXEFFORT:= 20*MAXEFFORT;
      oldLookUpUsed:= TRUE;
      if nops(sol) = 1 or sol={} then 
        sol:= FAIL; 
      end_if;
    end_if;
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_3_eq_61_62_63_64_69_70_71_72_73_74_105_106_107_108_p_219_to_225(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
   
  if sol = FAIL then 
    /* 
       Try this look-up as one of the last chances; it has to integrate 
       'kummerU' to generate the general solution from the particular 
    */ 
    sol:= ode::PZ_Sec_212_2_eq_18_19_20_21_22_24_25_26_27_28_p_216_217(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_3_eq_65_66_67_68_p_220(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_2_eq_29_30_31_p_217(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;

  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_5_eq_hypergeom(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL then 
    sol:= ode::PZ_Sec_212_1_to_8_eq_47_183_202_p_218(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL then 
    sol:= ode::PZ_Sec_215_1_eq_1_to_40_p_257_to_259(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL then 
    sol:= ode::PZ_Sec_215_1_eq_1_to_132_p_260_to_271(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;

  if sol = FAIL then 
    if traperror((sol:= ode::polynomialSolutions(eq,y(x)," ",solveOptions,odeOptions))) <> 0 or sol = {} then 
      sol:= FAIL;
    end_if;
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL then 
    if traperror((sol:= ode::rationalSolutions(eq,y(x)," ",solveOptions,odeOptions))) <> 0 or sol = {} then 
      sol:= FAIL;
    end_if;
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL then 
    if traperror((sol:= ode::exponentialSolutions(eq,y(x)," ",solveOptions,odeOptions))) <> 0 or sol = {} then 
      sol:= FAIL;
    end_if;
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL then 
    if traperror((sol:= ode::factor(eq,y,x,2,solveOptions union {},odeOptions))) <> 0 or sol = {} then 
      sol:= FAIL;
    else 
      csts:= freeIndets(sol) minus (indets(eq) union {y,x});
      if nops(csts) = 2 and nops(sol) = 1 then 
        //if traperror((sol:= evalAt(sol, [csts[1]=0,csts[2]=1]) union evalAt(sol, [csts[1]=1,csts[2]=0]))) <> 0 then
        if traperror((sol:= {diff(sol[1],csts[1]),diff(sol[1],csts[2])})) <> 0 and 
           nops(sol) = 2 and 
           not has(sol,csts[1]) and 
           not has(sol,csts[2]) then
          sol:= FAIL;
        end_if;
      else 
        sol:= FAIL;
      end_if;
    end_if;
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;

  if sol = FAIL then 
    /* 
       Try this look-up very late in the look-up hierarchy, since it mainly 
       returns complicated solutions in terms of 'hypergeom'. Note that if 
       this is used earlier it will definitely provide more complicated 
       solutions for equations which have elementary solutions (e.g. rational 
       functions solutions for eqs of type 192,193 etc.). 
    */ 
    sol:= ode::PZ_Sec_212_6_eq_186_190_194_p_237(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;

  /* 
     Not a good idea to be used in general due to complexity problems. 
     Even reducing 'MAXEFFORT' does not reduce, but in some cases even 
     increase the runtime behavior.
    
  if sol = FAIL and oldLookUpUsed = FALSE then 
    oldLookUpUsed:= TRUE;
    xpow:= {};
    misc::maprec(eqsubs,
                 {"_power"} = proc(elem)
                                begin
                                  if op(elem,1) = x then 
                                    xpow:= xpow union {op(elem,2)}
                                  end_if;
                                  elem;
                                end_proc);  
    if not hastype(xpow,DOM_RAT) then                             
      MAXEFFORT:= MAXEFFORT/20; 
      sol:= ode::specfunc(eq, y, x, solveOptions, odeOptions union {IgnoreAnalyticConstraints}); 
      MAXEFFORT:= MAXEFFORT/20; 
      if nops(sol) = 1 or sol={} then 
        sol:= FAIL; 
      end_if;
    end_if;
  end_if;  
  */
  
  if sol = FAIL then 
    // look-up methods for classes of ODEs involving 
    // arbitrary functions
    sol:= ode::PZ_Sec_219_1_eq_1_to_52_p_285_to_289(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL then 
    // look-up methods for classes of ODEs involving 
    // arbitrary functions and their derivatives
    sol:= ode::PZ_Sec_219_2_eq_53_to_91_p_289_to_292(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;

  if sol = FAIL then 
    sol:= ode::PZ_Sec_213_1_eq_1_to_79_p_246_to_252(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL then 
    sol:= ode::PZ_2nd_order_linear_look_up_heuristics_1(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;

  if sol = FAIL then 
    sol:= ode::lookUp2F1(eq, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  
  if sol = FAIL or sol = {} then 
    return({});
  else 
    if nops(sol) = 2 then
      return(sol);
    elif nops(sol) = 1 then 
      y0:= solvelib::getIdent(Any, indets([eq,y,x]));
      y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
      y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
      eqsubs:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
      dterm1:= combine(expand(diff(eqsubs,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
      dterm2:= combine(expand(diff(eqsubs,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
      return(sol union {sol[1]*int(exp(-int(dterm1/dterm2,x,intOptions))/sol[1]^2,x,intOptions)});
    end_if;   
  end_if;
end_proc:


/*
----------
REFERENCE. Implementation of (Section 2.1.2-1. eq. 4, p. 214) & 
----------                   (Section 2.1.2-1. eq. 6, p. 214) 
             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' may contain 'hypergeom' for 
                returning the solution in terms of 'hypergeom'
                in the most general case

*/

ode::PZ_Sec_212_1_eq_4_and_6_p_214:= proc(eq,y,x,solveOptions={},odeOptions={})
  local lcf, dterm, y0, y1, y2, a, b, xcoeff, res, optIgnoreAnalyticConstraints, f1, f2;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]); 
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  if (lcf:= diff(eq,y2)) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if;
  if not iszero(expand(diff(eq, y1),optIgnoreAnalyticConstraints)) then 
    return(FAIL);
  end_if;
  dterm:= combine(-diff(eq,y0),optIgnoreAnalyticConstraints);
  if has(ode::normal(diff(dterm,x,x)),x) then
    return(FAIL);
  end_if;
  a:= ode::normal(diff(dterm,x,x)/2);
  if iszero(a) then 
    return(FAIL);
  end_if; 
  b:= evalAt(dterm, x = 0);
  if not iszero((xcoeff:= evalAt(diff(dterm,x), x = 0))) then 
    /* This is the general case (Section 2.1.2-1. eq. 6, p. 214) */
    if contains(odeOptions, hypergeom) then  
      if traperror((res:= ({hypergeom([(4*a^(3/2) + 4*b*a - xcoeff^2)/(16*a^(3/2))],[1/2],(2*a*x+xcoeff)^2/(4*a^(3/2)))*
          exp(-x*(a*x+xcoeff)/(2*a^(1/2))),
          hypergeom([(12*a^(3/2) + 4*b*a - xcoeff^2)/(16*a^(3/2))],[3/2],(2*a*x+xcoeff)^2/(4*a^(3/2)))*
          (2*a*x+xcoeff)*exp(-x*(a*x+xcoeff)/(2*a^(1/2)))}))) = 0 then 
        return(res);
      else
        return(FAIL);
      end_if;
    else 
      dterm:= ode::normal(subs(dterm, x = x - xcoeff/(2*a)));
      a:= ode::normal(diff(dterm,x,x)/2);
      b:= evalAt(dterm, x = 0);
      if traperror((f1:= whittakerM(-b/(4*sqrt(a)), 1/4, 
                                    sqrt(a)*(x+xcoeff/(2*a))^2)/sqrt((x+xcoeff/(2*a))))) <> 0 then 
        f1:= undefined;
      end_if;
      if traperror((f2:= whittakerW(-b/(4*sqrt(a)), 1/4, 
                                    sqrt(a)*(x+xcoeff/(2*a))^2)/sqrt((x+xcoeff/(2*a))))) <> 0 then 
        f2:= undefined;
      end_if;  
      res:= {f1,f2} minus {undefined};
      if res <> {} then 
        return(res)
      end_if;  
    end_if;
  else 
    /* This is the special Weber case (Section 2.1.2-1. eq. 4, p. 214) */
    if traperror((f1:= whittakerM(-b/(4*sqrt(a)), 1/4, sqrt(a)*x^2)/sqrt(x))) <> 0 then 
      f1:= undefined;
    end_if;
    if traperror((f2:= whittakerW(-b/(4*sqrt(a)), 1/4, sqrt(a)*x^2)/sqrt(x))) <> 0 then 
      f2:= undefined;
    end_if;  
    res:= {f1,f2} minus {undefined};
    if res <> {} then 
      return(res)
    end_if;  
  end_if;
  
  return(FAIL);
end_proc:


/*
----------
REFERENCE. Implementation of (Section 2.1.2-1. eq. 7, p. 215)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_1_eq_7_p_215:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm, a, n, lcf, xpow, x1, optIgnoreAnalyticConstraints, res;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if; 
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if; 
  if (lcf:= diff(eq,y2)) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if;
  if not iszero(expand(diff(eq, y1),optIgnoreAnalyticConstraints)) then 
    return(FAIL);
  end_if;

  dterm:= combine(diff(eq,y0),optIgnoreAnalyticConstraints);
  
  xpow:= {};
  misc::maprec(dterm,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  

  xpow:= [op(xpow)];

  if nops(xpow) <> 1 or xpow = {-2} then 
    return(FAIL);
  end_if;

  x1:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  n:= xpow[1];

  dterm:= subsex(collect(subsex(dterm, x^n = x1), x1), x1 = x^n);

  if type(dterm) <> "_mult" then 
    return(FAIL);
  end_if;

  if has(ode::normal(dterm/x^(xpow[1])), x) then 
    return(FAIL)
  end_if;

  a:= -dterm | x = 1;
   
  if type(float(a)) = DOM_FLOAT and float(a) > 0 and 
     traperror((res:= {sqrt(x)*besselI(1/(n+2), 2*(a)^(1/2)*x^(n/2+1)/(n+2)), 
            sqrt(x)*besselK(1/(n+2), 2*(a)^(1/2)*x^(n/2+1)/(n+2))})) = 0 then 
    return(res);
  elif traperror((res:= {sqrt(x)*besselJ(1/(n+2), 2*I*(a)^(1/2)*x^(n/2+1)/(n+2)), 
                         sqrt(x)*besselY(1/(n+2), 2*I*(a)^(1/2)*x^(n/2+1)/(n+2))})) = 0 then 
    return(res);
  end_if;
  
  return(FAIL);
end_proc:



/*
----------
REFERENCE. Implementation of (Section 2.1.2-1. eq. 9, p. 215)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_1_eq_9_p_215:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm, e1, e2, a, b, n, op1, op2, xpow, x1, x2, lcf, 
        optIgnoreAnalyticConstraints, L;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  if (lcf:= diff(eq,y2)) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if;
  if not iszero(expand(diff(eq, y1),optIgnoreAnalyticConstraints)) then 
    return(FAIL);
  end_if;
 
  dterm:= combine(diff(eq,y0),optIgnoreAnalyticConstraints);

  xpow:= {};
  misc::maprec(dterm,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  if nops(xpow) <> 2 then 
    return(FAIL);
  end_if;
 
  xpow:= [op(xpow)];

  x1:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  x2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1]));

  dterm:= subsex(collect(subsex(dterm, 
                                [x^(xpow[1]) = x1, x^(xpow[2]) = x2]),
                         [x1,x2]),
                 [x1 = x^(xpow[1]), x2 = x^(xpow[2])]);
  e1:= FAIL;
  e2:= FAIL;
  if type(dterm) = "_plus" and nops(dterm) = 2 then 
    op1:= op(dterm,1);
    op2:= op(dterm,2);
    if has(op1, x) then 
      e1:= ode::normal(diff(op1,x)*x / op1);
      if has(e1,x) then 
        return(FAIL)
      end_if;
    end_if;
    if has(op2, x) then 
      e2:= ode::normal(diff(op2,x)*x / op2);
      if has(e2,x) then 
        return(FAIL)
      end_if;
    end_if; 
  else 
    return(FAIL);
  end_if;

  if e1=FAIL or e2=FAIL then 
    return(FAIL);
  end_if;  
  
  if not iszero(ode::normal(e1 - 2*(e2+1))) and  
     not iszero(ode::normal(e2 - 2*(e1+1))) then 
    return(FAIL);
  end_if;

  if iszero(ode::normal(e1 - 2*(e2+1))) then 
    // i.e. e1 = 2*n - 2
    n:= ode::normal((e1+2)/2);
    a:= -op1 | x = 1;
    b:= -op2 | x = 1; 
    if n = 0 then 
      return(FAIL) 
    end_if;
    if traperror(
       (L:= {whittakerM(-b/(a^(1/2)*(2*n)), 1/(2*n), (2*a^(1/2)*x^n)/n)/x^(n/2-1/2), 
             whittakerW(-b/(a^(1/2)*(2*n)), 1/(2*n), (2*a^(1/2)*x^n)/n)/x^(n/2-1/2)})) = 0 and 
       not contains(L, undefined) then 
      return(L)
    end_if;
  elif iszero(ode::normal(e2 - 2*(e1+1))) then 
    // i.e. e2 = 2*n - 2
    n:= ode::normal((e2+2)/2);
    a:= -op2 | x = 1;
    b:= -op1 | x = 1;
    if n = 0 then 
      return(FAIL) 
    end_if;
    if traperror(
       (L:= {whittakerM(-b/(a^(1/2)*(2*n)), 1/(2*n), (2*a^(1/2)*x^n)/n)/x^(n/2-1/2), 
             whittakerW(-b/(a^(1/2)*(2*n)), 1/(2*n), (2*a^(1/2)*x^n)/n)/x^(n/2-1/2)})) = 0 and 
       not contains(L, undefined) then 
      return(L)
    end_if;
  end_if;
  return(FAIL);
 
end_proc:


/*
----------
REFERENCE. Implementation of (Section 2.1.2-1. eq. 10, p. 215)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_1_eq_10_p_215:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm, e1, e2, a, b, n, op1, op2, xpow, x1, x2, lcf, 
        optIgnoreAnalyticConstraints, L;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  if (lcf:= diff(eq,y2)) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if;
  if not iszero(expand(diff(eq, y1),optIgnoreAnalyticConstraints)) then 
    return(FAIL);
  end_if;
  
  dterm:= combine(diff(eq,y0),optIgnoreAnalyticConstraints);

  xpow:= {};
  misc::maprec(dterm,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  if xpow = {4} then 
    xpow:= {4,1}
  end_if;
  if nops(xpow) <> 2 then 
    return(FAIL);
  end_if;

  xpow:= [op(xpow)];

  x1:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  x2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1]));

  dterm:= subsex(collect(subsex(dterm, 
                                [x^(xpow[1]) = x1, x^(xpow[2]) = x2]),
                         [x1,x2]),
                 [x1 = x^(xpow[1]), x2 = x^(xpow[2])]);

  if type(dterm) = "_plus" and nops(dterm) = 2 then 
    op1:= op(dterm,1);
    op2:= op(dterm,2);
    if has(op1, x) then 
      e1:= ode::normal(diff(op1,x)*x / op1);
      if has(e1,x) then 
        return(FAIL)
      end_if;
    else 
      return(FAIL);
    end_if;
    if has(op2, x) then 
      e2:= ode::normal(diff(op2,x)*x / op2);
      if has(e2,x) then 
        return(FAIL)
      end_if;
    else 
      return(FAIL);
    end_if; 
  else 
    return(FAIL);
  end_if;

  if not iszero(ode::normal(e1 - 2*(e2+1))) and  
     not iszero(ode::normal(e2 - 2*(e1+1))) then 
    return(FAIL);
  end_if;

  if iszero(ode::normal(e1 - 2*(e2+1))) then 
    // i.e. e1 = 2*n
    n:= ode::normal(e1/2);
    a:= -op1 | x = 1;
    b:= -op2 | x = 1; 
    if a = 1 and b = 2 and n = -4 then 
      // In this case both solutions coincide and only one 
      // solution is found. Maybe this is also useful, but 
      // currently I do not activate this and return 'FAIL'. 
      // But we should use the single solution to produce a 
      // fundamental set of solutions via variation of constants.
      return(FAIL) 
    end_if;

    if traperror(
       (L:= {x^(-n/2)*whittakerM(-b/(a^(1/2)*(2*n+2)), 1/(2*n+2), (2*a^(1/2)*x^(n+1))/(n+1)),
             x^(-n/2)*whittakerW(-b/(a^(1/2)*(2*n+2)), 1/(2*n+2), (2*a^(1/2)*x^(n+1))/(n+1))})) = 0 and 
       not contains(L, undefined) then 
      return(L)
    end_if;
  elif iszero(ode::normal(e2 - 2*(e1+1))) then 
    // i.e. e2 = 2*n
    n:= ode::normal(e2/2);
    a:= -op2 | x = 1;
    b:= -op1 | x = 1;
    if a = 1 and b = 2 and n = -4 then 
      // In this case both solutions coincide and only one 
      // solution is found. Maybe this is also useful, but 
      // currently I do not activate this and return 'FAIL'. 
      // But we should use the single solution to produce a 
      // fundamental set of solutions via variation of constants.
      return(FAIL) 
    end_if;

    if traperror(
       (L:= {x^(-n/2)*whittakerM(-b/(a^(1/2)*(2*n+2)), 1/(2*n+2), (2*a^(1/2)*x^(n+1))/(n+1)),
             x^(-n/2)*whittakerW(-b/(a^(1/2)*(2*n+2)), 1/(2*n+2), (2*a^(1/2)*x^(n+1))/(n+1))})) = 0 and 
       not contains(L, undefined) then 
      return(L)
    end_if;
  end_if;
  
  return(FAIL);
end_proc:
  

/*
----------
REFERENCE. Implementation of (Section 2.1.2-2. eq. 13, p. 216)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_2_eq_13_14_p_216:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, a, b, c, d, lcf, res, optIgnoreAnalyticConstraints;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  if (lcf:= diff(eq,y2)) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    elif not iszero(lcf) then 
      eq:= eq/lcf;
    end_if;
  end_if;
  dterm0:= combine(diff(eq,y0),optIgnoreAnalyticConstraints);
  dterm1:= combine(diff(eq,y1),optIgnoreAnalyticConstraints);
  if has(dterm1,x) or has(ode::normal(diff(dterm0,x,x)),x) then 
    return(FAIL)
  end_if;

  if iszero((b:= diff(dterm0, x,x)/2)) then 
    return(FAIL);
  end_if;

  a:= dterm1;
  
  if traperror((d:= dterm0 | x = 0)) <> 0 or 
     traperror((c:= diff(dterm0, x) | x = 0)) <>0 then 
    return(FAIL);
  end_if;  
   
  if traperror((res:= ({((c + 2*b*x)*hypergeom([((b*(d*4*I - a^2*I))/16 + (3*b^(3/2))/4 - (c^2*I)/16)/b^(3/2)], [3/2], 
                                                ((c + 2*b*x)^2*I)/(4*b^(3/2))))/exp((x*(b*x*I + a*b^(1/2) + c*I))/(2*b^(1/2))), 
                        hypergeom([((b*(d*4*I - a^2*I))/16 + b^(3/2)/4 - (c^2*I)/16)/b^(3/2)], [1/2], 
                                   ((c + 2*b*x)^2*I)/(4*b^(3/2)))/exp((x*(b*x*I + a*b^(1/2) + c*I))/(2*b^(1/2)))})
      )) = 0 then 
    return(res);
  else 
    return(FAIL);
  end_if;
end_proc:


/*
----------
REFERENCE. Implementation of (Section 2.1.2-2. eq. 15 & 16 & 17, p. 216)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_2_eq_15_16_17_p_216:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, e1, e2, e3, a, b, n, t, 
        xpow, x1, x2, x3, lcf, l, c1, c2, c3, found, intOptions, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  if ((lcf:= diff(eq,y2))) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if;
  dterm0:= combine(diff(eq,y0),optIgnoreAnalyticConstraints);
  dterm1:= combine(diff(eq,y1),optIgnoreAnalyticConstraints);
  if has(dterm1,x) or iszero(dterm1) then 
    return(FAIL)
  else
    a:= dterm1;
  end_if;

  xpow:= {};
  misc::maprec(dterm0,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  

  if xpow = {2,4} then 
    xpow:= {1,2,4};
  end_if;

  if nops(xpow) <> 3 then 
    return(FAIL);
  end_if;

  if traperror((t:= subsex(dterm0,[x^(xpow[1])=0,x^(xpow[2])=0,x^(xpow[3])=0]))) = 0 and 
     not iszero(t) then 
    return(FAIL);
  end_if;  

  xpow:= [op(xpow)];
  found:= FALSE;
  
  for l in [[1,2,3],[2,1,3],[2,3,1],
            [1,3,2],[3,1,2],[3,2,1]] do
    if ode::odeIszero(xpow[l[1]]-2*xpow[l[2]]) and 
       iszero(expand(xpow[l[1]]-2*xpow[l[2]],optIgnoreAnalyticConstraints)) then 
      e1:= xpow[l[1]];
      e2:= xpow[l[2]];
      if ((e3:= xpow[l[3]])) = e2-1 then 
        found := TRUE;
        break;
      end_if;
    end_if;
  end_for;

  if found = FALSE then 
    return(FAIL);
  end_if;
 
  n:= e2;

  x1:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  x2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1]));
  x3:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1,x2]));

  dterm0:= poly(subsex(dterm0, 
                      [x^e1 = x1, x^e2 = x2, x^e3 = x3]),
               [x1,x2,x3]);
        
  c1:= expr(coeff(dterm0,x1));
  c2:= expr(coeff(dterm0,x2));
  c3:= expr(coeff(dterm0,x3));

  if not iszero(a) then 
    b:= c2/a:

    if ode::odeIszero(c3 - n*b) and ode::odeIszero(c1 + b^2) and 
       iszero(expand(c3 - n*b,optIgnoreAnalyticConstraints)) and 
       iszero(expand(c1 + b^2,optIgnoreAnalyticConstraints)) and 
       not iszero(n+1) then 
      return({exp(-b/(n+1)*x^(n+1)), 
              exp(-b/(n+1)*x^(n+1))*int(exp(-a*x)/exp(-b/(n+1)*x^(n+1))^2,x,intOptions)});
    end_if;

    b:= -c2/a:

    if ode::odeIszero(c3 - n*b) and 
       ode::odeIszero(c1 + b^2) and 
       iszero(expand(c3 - n*b,optIgnoreAnalyticConstraints)) and 
       iszero(expand(c1 + b^2,optIgnoreAnalyticConstraints)) and 
       not iszero(n+1) then 
      return({exp(-b/(n+1)*x^(n+1)-a*x),
              exp(-b/(n+1)*x^(n+1)-a*x)*int(exp(-a*x)/exp(-b/(n+1)*x^(n+1)-a*x)^2,x,intOptions)});
    end_if;
  end_if;
    
  return(FAIL)    

end_proc:

  
/*
----------
REFERENCE. Implementation of (Section 2.1.2-2. eq. 18 & 19 & 20 & 21 & 
                              22 & 24 & 25 & 26 & 27 & 28, p. 216, 217)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_2_eq_18_19_20_21_22_24_25_26_27_28_p_216_217:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, a, b, c, d, lcf, intOptions, 
        optIgnoreAnalyticConstraints, f;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  if ((lcf:= diff(eq,y2))) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if;
  dterm0:= combine(diff(eq,y0),optIgnoreAnalyticConstraints);
  dterm1:= combine(diff(eq,y1),optIgnoreAnalyticConstraints);
  dterm2:= combine(diff(eq,y2),optIgnoreAnalyticConstraints);
  if iszero(dterm0) or iszero(dterm1) then 
    return(FAIL)
  end_if;
  if has(((a:= ode::normal(diff(dterm1,x)))),x) or has(((c:= ode::normal(diff(dterm0,x)))),x) then 
    return(FAIL);
  end_if;
  if traperror(dterm1 | x=0) = 0 and traperror(dterm0 | x=0) = 0 then 
    b:= dterm1 | x=0;
    d:= dterm0 | x=0;
    MAXEFFORT:= MAXEFFORT/100;
    if ode::odeIszero(c,optIgnoreAnalyticConstraints) and 
       ode::odeIszero(b,optIgnoreAnalyticConstraints) and 
       iszero(expand(c,optIgnoreAnalyticConstraints)) and 
       iszero(expand(b,optIgnoreAnalyticConstraints)) then 
      if traperror((f:= (x*kummerU(1-d/(2*a),3/2,(a*x^2)/2))/exp((a*x^2)/2))) = 0 then 
        return({f,f*int(exp(-int(dterm1/dterm2,x,intOptions))/f^2,x,intOptions)});
      else
        return(FAIL);
      end_if;
    else
      if traperror((f:= kummerU(((d*a^2)/2-(b*a*c)/2+c^2/2)/a^3,1/2,-(x*a^2+b*a-2*c)^2/(2*a^3))/exp((c*x)/a))) = 0 then 
        return({f,f*int(exp(-int(dterm1/dterm2,x,intOptions))/f^2,x,intOptions)});
      else
        return(FAIL);
      end_if;
    end_if;
  end_if;  
end_proc:


/*
----------
REFERENCE. Implementation of (Section 2.1.2-2. eq. 29 & 30 & 31 p. 217)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_2_eq_29_30_31_p_217:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, a, b, c, d, e, lcf, res, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  if ((lcf:= diff(eq,y2))) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if; 
  dterm0:= combine(diff(eq,y0),optIgnoreAnalyticConstraints);
  dterm1:= combine(diff(eq,y1),optIgnoreAnalyticConstraints);
  if iszero(dterm0) or iszero(dterm1) then 
    return(FAIL)
  end_if;
  if has(((a:= ode::normal(diff(dterm1,x)))),x) or has(((c:= ode::normal(diff(dterm0,x,x)/2))),x) then 
    return(FAIL);
  end_if;
  b:= dterm1 | x=0;
  d:= diff(dterm0,x) | x=0;
  e:= dterm0 | x=0;
  if not iszero(expand(2*a-4*c,optIgnoreAnalyticConstraints)) and 
     ode::odeIszero(d-e*b) and 
     ode::odeIszero(c-a*e+e^2) and 
     iszero(expand(d-e*b,optIgnoreAnalyticConstraints)) and 
     iszero(expand(c-a*e+e^2,optIgnoreAnalyticConstraints)) then 
    return({exp(-1/2*x^2*e), 
            exp(-e*(-2*e+a)/(2*a-4*e)*x^2)*erf((-2*x*e+a*x+b)/(2*a-4*e)^(1/2))});    
  elif not iszero(expand(c-a^2/4,optIgnoreAnalyticConstraints)) then 
    if traperror((res:= ({-(((2^(1/2)*(2*d - a*b))/(a^2 - 4*c)^(3/4) - 2^(1/2)*x*(a^2 - 4*c)^(1/4))*
             hypergeom([1/2 - (4*c*(a^2 - 4*c)^(1/2) + 4*a*c - 8*c*e - 
                        a^2*(a^2 - 4*c)^(1/2) + 2*b^2*c + 2*a^2*e - a^3 + 
                        2*d^2 - 2*a*b*d)/(4*(a^2 - 4*c)^(3/2))], [3/2], 
                        ((2^(1/2)*(2*d - a*b))/(2*(a^2 - 4*c)^(3/4)) - 
                        (2^(1/2)*x*(a^2 - 4*c)^(1/4))/2)^2))/exp(((a^2*x^2)/2 - 
                         2*c*x^2 - 2*d*x + b*x*(a^2 - 4*c)^(1/2) + a*b*x + 
                        (a*x^2*(a^2 - 4*c)^(1/2))/2)/(2*(a^2 - 4*c)^(1/2))), 
            hypergeom([-(4*c*(a^2 - 4*c)^(1/2) + 4*a*c - 8*c*e - a^2*(a^2 - 4*c)^(1/2) + 
                       2*b^2*c + 2*a^2*e - a^3 + 2*d^2 - 2*a*b*d)/(4*(a^2 - 4*c)^(3/2))],
                      [1/2], ((2^(1/2)*(2*d - a*b))/(2*(a^2 - 4*c)^(3/4)) - 
                      (2^(1/2)*x*(a^2 - 4*c)^(1/4))/2)^2)/exp(((a^2*x^2)/2 - 2*c*x^2 - 
                       2*d*x + b*x*(a^2 - 4*c)^(1/2) + a*b*x + (a*x^2*(a^2 - 
                       4*c)^(1/2))/2)/(2*(a^2 - 4*c)^(1/2)))}))) = 0 then
      return(res);
    else
      return(FAIL);
    end_if;
  elif not iszero(2*a*b - 4*d) then 
    return({airyAi((2*2^(1/3)*(a/2-e-(x*(4*d-2*a*b))/4+b^2/4))/(2*a*b - 4*d)^(2/3),0)/exp((a*x^2)/4 + (b*x)/2), 
            airyBi((2*2^(1/3)*(a/2 - e - (x*(4*d - 2*a*b))/4 + b^2/4))/(2*a*b - 4*d)^(2/3), 0)/exp((a*x^2)/4 + (b*x)/2)})
  elif iszero(2*a*b - 4*d) then
    return({1/exp((x*(b + a/2*x - 2*(b^2/4 + a/2 - e)^(1/2)))/2), 
            1/exp((x*(b + a/2*x + 2*(b^2/4 + a/2 - e)^(1/2)))/2)})
  else 
    return(FAIL);    
  end_if;
end_proc:



/*
----------
REFERENCE. Implementation of (Section 2.1.2-2. eq. 32, p. 217)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_2_eq_32_p_217:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, e1, e2, e3, e4, a, b, c, n, optIgnoreAnalyticConstraints, 
        xpow, x1, x2, x3, x4, lcf, l, c1, c2, c3, c4, found, intOptions;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  if ((lcf:= diff(eq,y2))) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if;
  dterm0:= combine(diff(eq,y0),optIgnoreAnalyticConstraints);
  dterm1:= combine(diff(eq,y1),optIgnoreAnalyticConstraints);
  if iszero(dterm1) or has((a:= ode::normal(diff(dterm1,x))),x) then 
    return(FAIL)
  end_if;
  if traperror((b:= (dterm1 | x=0))) <> 0 then 
    return(FAIL);
  end_if;
  
  xpow:= {};
  misc::maprec(dterm0,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  

  if nops(xpow) <> 4 then 
    return(FAIL);
  end_if;

  if traperror(eval(subsex(dterm0, 
                       [x^(xpow[1]) = 0, x^(xpow[2]) = 0, 
                        x^(xpow[3]) = 0, x^(xpow[4]) = 0]))) <> 0 or  
    not iszero(eval(subsex(dterm0, 
                       [x^(xpow[1]) = 0, x^(xpow[2]) = 0, 
                        x^(xpow[3]) = 0, x^(xpow[4]) = 0]))) then 
    return(FAIL);
  end_if;  

  xpow:= [op(xpow)];
  found:= FALSE;
  for l in combinat::permute([1,2,3,4]) do
    if iszero(xpow[l[1]]-2*xpow[l[2]]) then 
      e1:= xpow[l[1]];
      e3:= xpow[l[2]]; // = n
      if (xpow[l[3]] = e3+1 and xpow[l[4]] = e3-1) or
         (xpow[l[4]] = e3+1 and xpow[l[3]] = e3-1) then 
        e2:= e3 + 1;
        e4:= e3 - 1; 
        found := TRUE;
        break;
      end_if;
    end_if;
  end_for;

  if found = FALSE then 
    return(FAIL);
  end_if;
 
  n:= e3;

  x1:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  x2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1]));
  x3:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1,x2]));
  x4:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1,x2,x3]));

  if n <> -3 then 
    dterm0:= poly(subsex(dterm0, 
                        [x^e1 = x1, x^e2 = x2, x^e3 = x3, x^e4 = x4]),
                 [x1,x2,x3,x4]);
  else
    dterm0:= poly(subsex(dterm0, 
                        [x^e1 = x1, x^e4 = x4, x^e3 = x3, x^e2 = x2]),
                 [x1,x2,x3,x4]);    
  end_if;
        
  c1:= expr(coeff(dterm0,x1));
  c2:= expr(coeff(dterm0,x2));
  c3:= expr(coeff(dterm0,x3));
  c4:= expr(coeff(dterm0,x4));
  c:= c2/a;

  if ode::odeIszero(c1+c^2) and ode::odeIszero(c3-b*c) and ode::odeIszero(c4-c*n) and 
     iszero(expand(c1+c^2,optIgnoreAnalyticConstraints)) and 
     iszero(expand(c3-b*c,optIgnoreAnalyticConstraints)) and 
     iszero(expand(c4-c*n,optIgnoreAnalyticConstraints)) then 
    return({exp(-c/(n+1)*x^(n+1)), 
       exp(-c/(n+1)*x^(n+1))*int(exp(-1/2*a*x^2-b*x)/exp(-c/(n+1)*x^(n+1))^2,x,intOptions)});
  else 
    return(FAIL);
  end_if;

end_proc:


/*
----------
REFERENCE. Implementation of (Section 2.1.2-2. eq. 33 & 34 & 35, p. 217)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_2_eq_33_34_35_p_217:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, a, b, c, lcf, intOptions, optIgnoreAnalyticConstraints;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  if ((lcf:= diff(eq,y2))) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if;
  dterm0:= combine(diff(eq,y0),optIgnoreAnalyticConstraints);
  dterm1:= combine(diff(eq,y1),optIgnoreAnalyticConstraints);
  if iszero(dterm1) or iszero(dterm0) then 
    return(FAIL)
  end_if;
  if has((a:= ode::normal(1/2*diff(dterm1,x,x))),x) or 
     (ode::odeIszero(a) and 
      iszero(expand(a,optIgnoreAnalyticConstraints))) then
    return(FAIL)
  end_if;
  if traperror(iszero(expand(diff(dterm1,x) | x=0,optIgnoreAnalyticConstraints))) <> 0 or 
     not iszero(expand(diff(dterm1,x) | x=0,optIgnoreAnalyticConstraints)) then 
    return(FAIL);
  end_if;
  if ode::odeIszero(diff(dterm0,x)+a) and 
     iszero(expand(diff(dterm0,x)+a,optIgnoreAnalyticConstraints)) then 
    if traperror((b:= ode::normal((dterm0 | x=0)/(-a)))) <> 0 or 
       not iszero(expand((dterm1 | x=0) + a*b^2,optIgnoreAnalyticConstraints)) then
      return(FAIL);
    end_if;
    return({x - b, -int(exp(a*b^2*x - (a*x^3)/3)/(b - x)^2, x, intOptions)*(b - x)})
  elif not has(ode::normal(diff(dterm0,x,x)),x) then 
    if traperror(iszero(expand(diff(dterm0,x) | x=0,optIgnoreAnalyticConstraints))) = 0 and 
       iszero(expand(diff(dterm0,x) | x=0,optIgnoreAnalyticConstraints)) then
      b:= dterm1 | x=0;
      c:= diff(dterm0,x,x)/(2*a);
      if ode::odeIszero((dterm0 | x=1) - a*c - b*c + c^2) and 
         iszero(expand((dterm0 | x=1) - a*c - b*c + c^2,optIgnoreAnalyticConstraints)) then 
        return({int(exp(2*c*x)/exp((a*x^3)/3 + b*x), x, intOptions)/exp(c*x), 1/exp(c*x)})
      end_if;
    elif traperror(iszero(expand((diff(dterm0,x) | x=0)+a,optIgnoreAnalyticConstraints))) = 0 and 
         iszero(expand((diff(dterm0,x) | x=0)+a,optIgnoreAnalyticConstraints)) then 
      b:= (dterm1 | x=0)/2;
      if ode::odeIszero(diff(dterm0,x,x)/2-a*b) and 
         ode::odeIszero((dterm0 | x=1) -a*b + a - b^2) and 
         iszero(expand(diff(dterm0,x,x)/2-a*b,optIgnoreAnalyticConstraints)) and 
         iszero(expand((dterm0 | x=1) -a*b + a - b^2,optIgnoreAnalyticConstraints)) then
        return({x/exp(b*x), 
                (x*int(exp(2*b*x)/(x^2*exp((a*x^3)/3 + 2*b*x)), x, intOptions))/exp(b*x)})
      end_if;
    end_if;
  end_if;
  return(FAIL);
end_proc:


/*
----------
REFERENCE. Implementation of (Section 2.1.2-2. eq. 38 & 39 & 40 & 42 & 43, p. 217, 218)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_2_eq_38_39_40_42_43_p_217_218:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm1_x, dterm1_xx, dterm1_xxx, 
        dterm0_x, dterm0_xx, dterm0_xxx, a, b, c, lcf, intOptions, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if; 
  if ((lcf:= diff(eq,y2))) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if;
  dterm0:= combine(diff(eq,y0),optIgnoreAnalyticConstraints);
  dterm1:= combine(diff(eq,y1),optIgnoreAnalyticConstraints);
  if iszero(dterm1) or iszero(dterm0) then 
    return(FAIL)
  end_if;
  dterm0_x:= diff(dterm0,x);
  dterm0_xx:= diff(dterm0_x,x);
  dterm0_xxx:= diff(dterm0_xx,x);
  dterm1_x:= diff(dterm1,x);
  dterm1_xx:= diff(dterm1_x,x);
  dterm1_xxx:= diff(dterm1_xx,x);

  if not has(ode::normal(dterm1_x),x) or not has(ode::normal(dterm0_x),x) then
    return(FAIL);
  end_if;
  if has(ode::normal(dterm1_xxx),x) or has(ode::normal(dterm0_xxx),x) then
    return(FAIL);
  end_if;

  if has(ode::normal(dterm1_xx),x) and has(ode::normal(dterm0_xx),x) then
    a:= (dterm1 | x=0)/2;
    if not iszero(expand(dterm1_x | x = 0,optIgnoreAnalyticConstraints)) then
      return(FAIL);
    end_if;
    b:= (dterm1_xx | x=0)/2;
    if ode::odeIszero(dterm1_xxx - 6*a*b) and 
       iszero(expand(dterm1_xxx - 6*a*b,optIgnoreAnalyticConstraints)) then
      if ode::odeIszero((dterm0 | x=0) - a^2) and 
         ode::odeIszero(dterm0_xxx - 6*a^2*b) and 
         iszero(expand((dterm0 | x=0) - a^2,optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0_xxx - 6*a^2*b,optIgnoreAnalyticConstraints)) then
        return({(int(exp(2*a*x)/(exp((a*b*x^4)/4+(b*x^3)/3+2*a*x)*(a*x+1)^2),x,intOptions)*(a*x+1))/exp(a*x), 
                (a*x+1)/exp(a*x)});
      else
        return(FAIL);
      end_if;
    else
      return(FAIL);
    end_if;
  end_if;
  
  if has(ode::normal(dterm1_xx),x) and not has(ode::normal(dterm0_xx),x) and has(ode::normal(dterm0_x),x) then 
    a:= dterm1_xxx/6;
    b:= dterm1_x | x=0;
    if not iszero(expand((dterm1 | x=1) - (a+b),optIgnoreAnalyticConstraints)) or 
       not iszero(expand((dterm0 | x=0) - 2*b,optIgnoreAnalyticConstraints)) or 
       not iszero(expand(dterm0_xx - 8*a,optIgnoreAnalyticConstraints)) then 
      return(FAIL)
    end_if;
    return({x*exp(-1/4*a*x^4-1/2*b*x^2), 
            x*exp(-1/4*a*x^4-1/2*b*x^2)*int(exp(-(1/4*a*x^4+1/2*b*x^2))/(x*exp(-1/4*a*x^4-1/2*b*x^2))^2,x,intOptions)})
  end_if;

  if not has(ode::normal(dterm1_xx),x) and has(ode::normal(dterm0_xx),x) then 
    a:= dterm1_xx/2;
    b:= (dterm1_x | x=0);
    c:= (dterm1 | x=0);
    if not iszero(expand(dterm0_xxx/6 - a*b,optIgnoreAnalyticConstraints)) then 
      return(FAIL);
    end_if; 
    if ode::odeIszero(diff((dterm0 - a*b*x^3),x) - (b*c+2*a)) and 
       iszero(expand(diff((dterm0 - a*b*x^3),x) - (b*c+2*a),optIgnoreAnalyticConstraints)) then 
      return({exp(-1/3*a*x^3-c*x), 
        exp(-1/3*a*x^3-c*x)*int(exp(-(1/3*a*x^3+1/2*b*x^2+c*x))/(exp(-1/3*a*x^3-c*x))^2,x,intOptions)});
    end_if;
    if ode::odeIszero(((dterm0 - a*b*x^3)| x=1) - (a*c+b)) and 
       iszero(expand(((dterm0 - a*b*x^3)| x=1) - (a*c+b),optIgnoreAnalyticConstraints)) then
      return({exp(-1/2*b*x^2-c*x), 
        exp(-1/2*b*x^2-c*x)*int(exp(-(1/3*a*x^3+1/2*b*x^2+c*x))/(exp(-1/2*b*x^2-c*x))^2,x,intOptions)});
    end_if;
  end_if;

  if not has(ode::normal(dterm1_xx),x) and not has(ode::normal(dterm0_xx),x) then 
    a:= (dterm1 | x=0)/2;
    b:= (dterm1_x | x=0);
    if not iszero(expand(dterm1_xx - 2*a*b,optIgnoreAnalyticConstraints)) or 
       not iszero(expand((dterm0 | x=0) - a^2,optIgnoreAnalyticConstraints)) or 
       not iszero(expand(dterm0_xx - 2*a^2*b,optIgnoreAnalyticConstraints)) then 
      return(FAIL);
    end_if;
    return({(a*x+1)*exp(-a*x), 
       (a*x+1)*exp(-a*x)*int(exp(-(1/3*a*b*x^3+1/2*b*x^2+2*a*x))/((a*x+1)*exp(-a*x))^2,x,intOptions)})
  end_if;

  return(FAIL)
end_proc:


/*
----------
REFERENCE. Implementation of (Section 2.1.2-2. eq. 45, p. 218)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_2_eq_45_46_p_218:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, a, b, n, xpow, x1, x2, lcf, res, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  if ((lcf:= diff(eq,y2))) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if;
  dterm0:= combine(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/);
  dterm1:= combine(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/);
  if iszero(dterm1) or iszero(dterm0) then 
    return(FAIL)
  end_if;

  xpow:= {};
  misc::maprec(dterm1,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  if nops(xpow) <> 1 then 
    return(FAIL);
  end_if;

  n:= xpow[1];
  
  xpow:= {};
  misc::maprec(dterm0,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  if n <> -1 and nops(xpow) = 1 then 
    if not iszero(expand(xpow[1]-n+1,optIgnoreAnalyticConstraints)) then
      return(FAIL);
    end_if;
    a:= dterm1 | x=1;
    b:= dterm0 | x=1;
    if iszero(expand(b - a*(n+1),optIgnoreAnalyticConstraints)) or 
       iszero(expand(b + a,optIgnoreAnalyticConstraints)) then 
      /* 
         In this situations, the pattern for (eq. 2.1.2-2. 57) and 
         (eq. 2.1.2-2. 55) provide the simpler results. 
      */
      return(FAIL);
    end_if; 
    if has(a,x) or has(b,x) then 
      return(FAIL);
    end_if;
    if n <> -1 then 
      //if traperror((res:=({x*hypergeom([b/(a + a*n) + 1/(n + 1)], [1/(n + 1) + 1], -(a*x^(n + 1))/(n + 1)), 
      //     hypergeom([b/(a + a*n)], [1 - 1/(n + 1)], -(a*x^(n + 1))/(n + 1))}))) = 0 then
      if traperror((res:=({(x*hypergeom([(a - b + a*n)/(a*(n + 1))], [(n + 2)/(n + 1)], (a*x^(n + 1))/(n + 1)))/exp((a*x^(n + 1))/(n + 1)), 
           (x*hypergeom([-(b - a*n)/(a*(n + 1))], [n/(n + 1)], (a*x^(n + 1))/(n + 1)))/(exp((a*x^(n + 1))/(n + 1))*((a*x^(n + 1))/(n + 1))^(1/(n + 1)))}))) = 0 then
        return(res)
      else 
        return(FAIL);
      end_if;
    else 
      return(FAIL);
    end_if;  
  elif n <> -1 and nops(xpow) = 2 then 
    xpow:= [op(xpow)];
    if (iszero(expand(xpow[1] - 2*n,optIgnoreAnalyticConstraints)) and 
        iszero(expand(xpow[2] - n+1,optIgnoreAnalyticConstraints))) or 
       (iszero(expand(xpow[1] - n+1,optIgnoreAnalyticConstraints)) and 
        iszero(expand(xpow[2] - 2*n,optIgnoreAnalyticConstraints))) then 
      a:= (dterm1 | x=1)/2;
      x1:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
      x2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1]));
      if ode::odeIszero(((dterm0:= subsex(dterm0,[x^(2*n)=x1, x^(n-1)=x2]))) | (x1=0, x2=0)) and 
         iszero(expand(dterm0 | (x1=0, x2=0),optIgnoreAnalyticConstraints)) then
        if ode::odeIszero(diff(dterm0,x1) - a^2) and 
           ode::odeIszero(diff(dterm0,x2) - a*n) and 
           iszero(expand(diff(dterm0,x1) - a^2,optIgnoreAnalyticConstraints)) and 
           iszero(expand(diff(dterm0,x2) - a*n,optIgnoreAnalyticConstraints)) then 
          return({-1/exp((a*x^(n + 1))/(n + 1)), x/exp((a*x^(n + 1))/(n + 1))});
        end_if;
      else 
        return(FAIL)
      end_if;
    elif ode::odeIszero(xpow[1] - n+1) and 
         ode::odeIszero(xpow[2] - 2*n) and 
         iszero(expand(xpow[1] - n+1,optIgnoreAnalyticConstraints)) and 
         iszero(expand(xpow[2] - 2*n,optIgnoreAnalyticConstraints)) then 
      a:= (dterm1 | x=1)/2;
      x1:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
      x2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1]));
      if ode::odeIszero(((dterm0:= subsex(dterm0,[x^(2*n)=x1, x^(n-1)=x2]))) | (x1=0, x2=0)) and 
         iszero(expand(dterm0 | (x1=0, x2=0),optIgnoreAnalyticConstraints)) then
        if ode::odeIszero(diff(dterm0,x1) - a^2) and 
           ode::odeIszero(diff(dterm0,x2) - a*n) and 
           iszero(expand(diff(dterm0,x1) - a^2,optIgnoreAnalyticConstraints)) and 
           iszero(expand(diff(dterm0,x2) - a*n,optIgnoreAnalyticConstraints)) then 
          return({-1/exp((a*x^(n + 1))/(n + 1)), x/exp((a*x^(n + 1))/(n + 1))});
        end_if;
      else 
        return(FAIL)
      end_if;
    end_if;
    return(FAIL); 
  end_if;
  return(FAIL);
end_proc:


/*
----------
REFERENCE. Implementation of (Section 2.1.2-2. eq. 50 & 51 & 52 & 53 & 54, p. 218)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_2_eq_50_51_52_53_54_p_218:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, a, b, c, n, xpow1, xpow0, x1, x2, x3, lcf, 
        intOptions, optIgnoreAnalyticConstraints, dterm1_orig;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  if ((lcf:= diff(eq,y2))) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if; 
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1_orig:= dterm1;
  if iszero(dterm1) or iszero(dterm0) then 
    return(FAIL)
  end_if;
  x1:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  x2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1]));
  x3:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1,x2]));
  xpow0:= {};
  xpow1:= {};
  misc::maprec(dterm1,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow1:= xpow1 union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  misc::maprec(dterm0,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow0:= xpow0 union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  

  if nops(xpow1) = 1 and nops(xpow0) = 1 then 
    n:= xpow1[1];
    if ode::odeIszero(xpow1[1] - xpow0[1]) and 
       ode::odeIszero(xpow1[1] - xpow0[1]) and 
       iszero(expand(xpow1[1] - xpow0[1],optIgnoreAnalyticConstraints)) then 
      dterm1:= subsex(dterm1,[x^n = x1]);
      if has(dterm1, x) then 
        return(FAIL);
      end_if;
      a:= diff(dterm1, x1);
      b:= dterm1 | x1 = 0;
      dterm0:= subsex(dterm0,[x^n = x1]);
      c:= diff(dterm0, x1)/a;
      if not has(a,x) and not has(b,x) and not has(c,x) and 
         not has(a,x1) and not has(b,x1) and not has(c,x1) and
         ode::odeIszero((dterm0 | x1 = 0) - c*b + c^2) and 
         iszero(expand((dterm0 | x1 = 0) - c*b + c^2,optIgnoreAnalyticConstraints)) then 
        y0:= 1/exp(c*x);
        y1:= y0*int(exp(-int(dterm1_orig,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
  end_if;

  if nops(xpow1) = 1 and nops(xpow0) = 2 then
    n:= xpow1[1];
    if xpow0 <> {n,n-1} then 
      return(FAIL);
    end_if;    
    dterm1:= subsex(dterm1,[x^n = x1]);
    a:= diff(dterm1, x1);
    b:= (dterm1 | x1 = 0)/2;
    if type(float(n)) = DOM_FLOAT and n < 0 then
      dterm0:= subsex(dterm0,[x^(n-1) = x2, x^n = x1]);
    else 
      dterm0:= subsex(dterm0,[x^n = x1, x^(n-1) = x2]);
    end_if;

    if not iszero(expand(diff(dterm0,x1) - a*b,optIgnoreAnalyticConstraints)) or 
       not iszero(expand(diff(dterm0,x2) + a,optIgnoreAnalyticConstraints)) or 
       not iszero(expand((dterm0 | (x1=0, x2=0)) - b^2,optIgnoreAnalyticConstraints)) then 
      return(FAIL);
    end_if;
    if not has(a,x) and not has(b,x) and
       not has(a,x1) and not has(b,x1) then 
      y0:= x/exp(b*x);
      y1:= y0*int(exp(-int(dterm1_orig,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;

  if nops(xpow1) = 2 then 
    xpow1:= [op(xpow1)];
    if ode::odeIszero(xpow1[1] - xpow1[2]+1) and 
       iszero(expand(xpow1[1] - xpow1[2]+1,optIgnoreAnalyticConstraints)) then 
      n:= xpow1[2];
    elif ode::odeIszero(xpow1[2] - xpow1[1]+1) and 
         iszero(expand(xpow1[2] - xpow1[1]+1,optIgnoreAnalyticConstraints)) then
      n:= xpow1[1];
    else 
      return(FAIL);
    end_if;
    if type(float(n)) = DOM_FLOAT and n < 0 then
      dterm1:= subsex(dterm1,[x^(n-1) = x2, x^n = x1]);
    elif n = 1/2 then
      dterm1:= subsex(subsex(dterm1,[x^(n-1) = x2]), [1/x2 = x1]);      
    else
      dterm1:= subsex(dterm1,[x^n = x1, x^(n-1) = x2]);
    end_if;
    if xpow0 = {n} then 
      if not has(dterm1, x) then
        a:= (dterm1 | (x1=0, x2=0))/2;
        b:= diff(dterm1,x2);
        if not iszero(expand(diff(dterm1,x1) - a*b,optIgnoreAnalyticConstraints)) then 
          return(FAIL);
        end_if;
        dterm0:= subsex(dterm0,[x^n = x1]);
        if not has(a,x) and not has(b,x) and
           not has(a,x1) and not has(b,x1) and 
           not has(a,x2) and not has(b,x2) and
           ode::odeIszero((dterm0 | x1=0) - a^2) and 
           ode::odeIszero(diff(dterm0,x1) - a^2*b) and 
           iszero(expand((dterm0 | x1=0) - a^2,optIgnoreAnalyticConstraints)) and 
           iszero(expand(diff(dterm0,x1) - a^2*b,optIgnoreAnalyticConstraints)) then 
          y0:= (a*x + 1)/exp(a*x);
          y1:= y0*int(exp(-int(dterm1_orig,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        else
          return(FAIL);
        end_if;
      end_if;
    elif xpow0 = {op(xpow1)} then 
      if type(float(n)) = DOM_FLOAT and n < 0 then
        dterm0:= subsex(dterm0,[x^(n-1) = x2, x^n = x1]);
      elif n = 1/2 then
        dterm1:= subsex(subsex(dterm0,[x^(n-1) = x2]), [1/x2 = x1]);      
      else 
        dterm0:= subsex(dterm0,[x^n = x1, x^(n-1) = x2]);
      end_if;
      b:= diff(dterm1,x2)/2;
      a:= diff(dterm1,x1)/b;
      if not iszero(expand(diff(dterm1,x) + a^2,optIgnoreAnalyticConstraints)) then  
        return(FAIL);
      end_if;
      if not iszero(expand(diff(dterm0,x1) - a^2*b,optIgnoreAnalyticConstraints)) or 
         not iszero(expand(diff(dterm0,x2) - a*b,optIgnoreAnalyticConstraints)) or 
         not iszero(expand(diff(dterm0,x) + a^3,optIgnoreAnalyticConstraints)) then 
        return(FAIL);
      end_if;
      if not has(a,x) and not has(b,x) and 
         not has(a,x1) and not has(b,x1) and 
         not has(a,x2) and not has(b,x2) then  
        y0:= (a*x + 2)/exp(a*x);
        y1:= y0*int(exp(-int(dterm1_orig,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    else
      return(FAIL);
    end_if;
  end_if;

  if nops(xpow1) = 3 then 
    xpow1:= [op(xpow1)];
    if (iszero(expand(xpow1[1] - xpow1[3]-2,optIgnoreAnalyticConstraints)) and 
        iszero(expand(xpow1[2] - xpow1[3]-1,optIgnoreAnalyticConstraints))) or 
       (iszero(expand(xpow1[2] - xpow1[3]-2,optIgnoreAnalyticConstraints)) and 
        iszero(expand(xpow1[1] - xpow1[3]-1,optIgnoreAnalyticConstraints))) then 
      n:= xpow1[3];
    elif (iszero(expand(xpow1[2] - xpow1[1]-2,optIgnoreAnalyticConstraints)) and 
          iszero(expand(xpow1[3] - xpow1[1]-1,optIgnoreAnalyticConstraints))) or
         (iszero(expand(xpow1[3] - xpow1[1]-2,optIgnoreAnalyticConstraints)) and 
          iszero(expand(xpow1[2] - xpow1[1]-1,optIgnoreAnalyticConstraints))) then 
      n:= xpow1[1];
    elif (iszero(expand(xpow1[1] - xpow1[2]-2,optIgnoreAnalyticConstraints)) and 
          iszero(expand(xpow1[3] - xpow1[2]-1,optIgnoreAnalyticConstraints))) or
         (iszero(expand(xpow1[3] - xpow1[2]-2,optIgnoreAnalyticConstraints)) and 
          iszero(expand(xpow1[1] - xpow1[2]-1,optIgnoreAnalyticConstraints))) then
      n:= xpow1[2];
    else 
      return(FAIL);
    end_if;
    if type(float(n)) = DOM_FLOAT and n < 0 then
      dterm1:= subsex(dterm1,[x^n = x3, x^(n+1) = x2, x^(n+2) = x1]);
    else 
      dterm1:= subsex(dterm1,[x^(n+2) = x1, x^(n+1) = x2, x^n = x3]);
    end_if;
    if type(float(n)) = DOM_FLOAT and n < 0 then
      dterm0:= subsex(dterm0,[x^n = x3, x^(n+1) = x2]);
    else 
      dterm0:= subsex(dterm0,[x^(n+1) = x2, x^n = x3]);
    end_if;
    if has(dterm0, x) then 
      return(FAIL);
    end_if;
    a:= -diff(dterm0,x2);
    b:= -diff(dterm0,x3);
    if not iszero(expand(b,optIgnoreAnalyticConstraints)) then 
      c:= diff(dterm1,x3)/b;
    else
      c:= (diff(dterm1,x2)-b)/a;
    end_if;
    if not iszero(expand(diff(dterm1,x1) - a,optIgnoreAnalyticConstraints)) or 
       not iszero(expand(diff(dterm1,x2) - a*c - b,optIgnoreAnalyticConstraints)) or 
       not iszero(expand(diff(dterm1,x3) - b*c,optIgnoreAnalyticConstraints)) then 
      return(FAIL);
    end_if;
    if not has(a,x) and not has(b,x) and 
         not has(a,x1) and not has(b,x1) and 
         not has(a,x2) and not has(b,x2) and
         not has(a,x3) and not has(b,x3) then  
      y0:= c+x;
      y1:= y0*int(exp(-int(dterm1_orig,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;

  return(FAIL);
end_proc:


/*
----------
REFERENCE. Implementation of (Section 2.1.2-2. eq. 55 & 56 & 57 & 58 & 59 & 60, p. 218, 219)

             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
          eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
           y -- dependent variable
           x -- indepenent variable 
solveOptions -- options for 'solve'
  pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_2_eq_55_56_57_58_59_60_p_218_219:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, a, b, c, m, n, xpow1, xpow0, x1, x2, x3, lcf, 
        aa, bb, mm, nn, intOptions, optIgnoreAnalyticConstraints, dterm1_orig;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if; 
  if ((lcf:= diff(eq,y2))) <> 1 then
    if has(lcf, x) then 
      eq:= ode::normal(eq/lcf);
    else
      eq:= eq/lcf;
    end_if;
  end_if; 
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1_orig:= dterm1;
  if iszero(dterm1) or iszero(dterm0) then 
    return(FAIL)
  end_if;
  x1:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  x2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1]));
  x3:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1,x2]));
  xpow0:= {};
  xpow1:= {};
  misc::maprec(dterm0,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow0:= xpow0 union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  misc::maprec(dterm1,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow1:= xpow1 union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  if nops(xpow1) = 2 and nops(xpow0) < 3  and xpow0 <> xpow1 then
    n:= xpow1[1];
    m:= xpow1[2];
    if type(float(m-n)) = DOM_FLOAT and float(m-n) > 0 then
      dterm1:= subs(dterm1,[x^m = x1, x^n = x2]);
      dterm0:= subs(dterm0,[x^(m-1) = x1, x^(n-1) = x2]);
    else 
      dterm1:= subs(dterm1,[x^n = x2, x^m = x1]);
      dterm0:= subs(dterm0,[x^(n-1) = x2, x^(m-1) = x1]);
    end_if;
    a:= diff(dterm1,x2);
    b:= diff(dterm1,x1);
    if ode::odeIszero(diff(a,x)) and 
       ode::odeIszero(diff(b,x)) and 
       iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
       iszero(expand(diff(b,x),optIgnoreAnalyticConstraints)) and 
       not has(a,x1) and not has(b,x1) and not has(a,x2) and not has(b,x2) then
      if ode::odeIszero(diff(dterm0,x2)+a) and  
         ode::odeIszero(diff(dterm0,x1)+b) and 
         ode::odeIszero((dterm0 | (x1=1,x2=1)) + (a+b)) and 
         ode::odeIszero((dterm1 | (x1=1,x2=1)) - (a+b)) and 
         iszero(expand(diff(dterm0,x2)+a,optIgnoreAnalyticConstraints)) and  
         iszero(expand(diff(dterm0,x1)+b,optIgnoreAnalyticConstraints)) and 
         iszero(expand((dterm0 | (x1=1,x2=1)) + (a+b),optIgnoreAnalyticConstraints)) and 
         iszero(expand((dterm1 | (x1=1,x2=1)) - (a+b),optIgnoreAnalyticConstraints)) then 
        if m <> -1 and n <> -1 then 
          return({x,x*int(1/(x^2*exp((b*x^(m+1))/(m+1)+(a*x^(n+1))/(n+1))),x,intOptions)});
        elif m = -1 and n <> -1 then 
          return({x,x*int(1/(x^2*exp(b*ln(x)+(a*x^(n+1))/(n+1))),x,intOptions)});
        elif m <> -1 and n = -1 then
          return({x,x*int(1/(x^2*exp((b*x^(m+1))/(m+1)+a*ln(x))),x,intOptions)});
        elif m = -1 and n = -1 then 
          return({x,x*int(1/(x^2*exp((b*ln(x) + a*ln(x))),x,intOptions))});
        end_if;
      elif ode::odeIszero(diff(dterm0,x2)-a*n) and  
         ode::odeIszero(diff(dterm0,x1)-b*m) and 
         ode::odeIszero((dterm0 | (x1=1,x2=1)) - (a*n+b*m)) and 
         ode::odeIszero((dterm1 | (x1=1,x2=1)) - (a+b)) and 
         iszero(expand(diff(dterm0,x2)-a*n,optIgnoreAnalyticConstraints)) and  
         iszero(expand(diff(dterm0,x1)-b*m,optIgnoreAnalyticConstraints)) and 
         iszero(expand((dterm0 | (x1=1,x2=1)) - (a*n+b*m),optIgnoreAnalyticConstraints)) and 
         iszero(expand((dterm1 | (x1=1,x2=1)) - (a+b),optIgnoreAnalyticConstraints)) then 
        if m <> -1 and n <> -1 then 
          return({1/exp((b*x^(m+1))/(m+1)+(a*x^(n+1))/(n+1)), 
                  int(exp((2*b*x^(m+1))/(m+1)+(2*a*x^(n+1))/(n+1))/exp((b*x^(m+1))/(m+1)+ 
                      (a*x^(n+1))/(n+1)),x,intOptions)/exp((b*x^(m+1))/(m+1)+(a*x^(n+1))/(n+1))});
        else 
          return(FAIL);
        end_if;
      elif ode::odeIszero(diff(dterm0,x2)-a*(n+1)) and  
         ode::odeIszero(diff(dterm0,x1)-b*(m+1)) and 
         ode::odeIszero((dterm0 | (x1=1,x2=1)) - (a*(n+1)+b*(m+1))) and 
         ode::odeIszero((dterm1 | (x1=1,x2=1)) - (a+b)) and 
         iszero(expand(diff(dterm0,x2)-a*(n+1),optIgnoreAnalyticConstraints)) and  
         iszero(expand(diff(dterm0,x1)-b*(m+1),optIgnoreAnalyticConstraints)) and 
         iszero(expand((dterm0 | (x1=1,x2=1)) - (a*(n+1)+b*(m+1)),optIgnoreAnalyticConstraints)) and 
         iszero(expand((dterm1 | (x1=1,x2=1)) - (a+b),optIgnoreAnalyticConstraints)) then 
        if m <> -1 and n <> -1 then 
          return({(x*int(exp((2*b*x^(m+1))/(m+1)+(2*a*x^(n+1))/(n+1))/(x^2*exp((b*x^(m+1))/(m+1)+
                         (a*x^(n+1))/(n+1))),x,intOptions))/exp((b*x^(m+1))/(m+1)+(a*x^(n+1))/(n+1)),
                   x/exp((b*x^(m+1))/(m+1)+(a*x^(n+1))/(n+1))});
        else 
          return(FAIL);
        end_if;
      end_if;
    end_if;    
  elif nops(xpow1) = 1 and nops(xpow0) = 1 and not iszero(xpow0[1] - xpow1[1]) then 
    if nops(xpow0) > 1 then 
      return(FAIL);
    end_if; 
    n:= xpow1[1];
    dterm1:= subs(dterm1,[x^n = x1]);
    dterm0:= subs(dterm0,[x^(n-1) = x1]);
    a:= diff(dterm1,x1);
    if ode::odeIszero(diff(a,x)) and 
       iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
       not has(a,x1) then
      if ode::odeIszero(diff(dterm0,x1) + a) and
         ode::odeIszero((dterm0 | x1=1) + a) and 
         ode::odeIszero((dterm1 | x1=1) - a) and 
         iszero(expand(diff(dterm0,x1)+a,optIgnoreAnalyticConstraints)) and
         iszero(expand((dterm0 | x1=1) + a,optIgnoreAnalyticConstraints)) and 
         iszero(expand((dterm1 | x1=1) - a,optIgnoreAnalyticConstraints)) then 
        if n <> -1 then 
          return({x,x*int(1/(x^2*exp((a*x^(n+1))/(n+1))),x,intOptions)});
        else 
          return({x,x*int(1/(x^2*exp((a*ln(x)))),x,intOptions)});
        end_if;
      elif ode::odeIszero(diff(dterm0,x1)-a*n) and
         ode::odeIszero((dterm0 | x1=1) - a*n) and 
         ode::odeIszero((dterm1 | x1=1) - a) and 
         iszero(expand(diff(dterm0,x1)-a*n,optIgnoreAnalyticConstraints)) and
         iszero(expand((dterm0 | x1=1) - a*n,optIgnoreAnalyticConstraints)) and 
         iszero(expand((dterm1 | x1=1) - a,optIgnoreAnalyticConstraints)) then 
        if n <> -1 then 
          return({1/exp((a*x^(n+1))/(n+1)), 
                  int(exp((a*x^(n+1))/(n+1)),x,IgnoreSpecialCases,intOptions)/exp((a*x^(n+1))/(n+1))});
        else 
          return(FAIL);
        end_if;
      elif ode::odeIszero(diff(dterm0,x1)-a*(n+1)) and
         ode::odeIszero((dterm0 | x1=1) - a*(n+1)) and 
         ode::odeIszero((dterm1 | x1=1) - a) and 
         iszero(expand(diff(dterm0,x1)-a*(n+1),optIgnoreAnalyticConstraints)) and
         iszero(expand((dterm0 | x1=1) - a*(n+1),optIgnoreAnalyticConstraints)) and 
         iszero(expand((dterm1 | x1=1) - a,optIgnoreAnalyticConstraints)) then 
        if n <> -1 then 
          return({(x*int(exp((a*x^(n+1))/(n+1))/x^2,x,intOptions))/exp((a*x^(n+1))/(n+1)), 
                   x/exp((a*x^(n+1))/(n+1))});
        else 
          return(FAIL);
        end_if;
      end_if;
    end_if;    
  elif map(xpow0,expand,optIgnoreAnalyticConstraints) = map(xpow1,expand,optIgnoreAnalyticConstraints) then 
    if nops(xpow1) = 2 then 
      n:= xpow1[1];
      m:= xpow1[2];
      if type(float(m-n)) = DOM_FLOAT and float(m-n) > 0 then
        dterm1:= subs(dterm1,[x^m = x1, x^n = x2]);
        dterm0:= subs(dterm0,[x^m = x1, x^n = x2]);
      else 
        dterm1:= subs(dterm1,[x^n = x2, x^m = x1]);
        dterm0:= subs(dterm0,[x^n = x2, x^m = x1]);
      end_if;
      a:= diff(dterm1,x2);
      b:= diff(dterm1,x1);
      if not has(a,x1) and not has(b,x1) and not has(a,x2) and not has(b,x2) and 
         ode::odeIszero(diff(a,x)) and 
         ode::odeIszero(diff(b,x)) and 
         iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(diff(b,x),optIgnoreAnalyticConstraints)) then
        if not iszero(expand(a,optIgnoreAnalyticConstraints)) then 
          c:= diff(dterm0,x2)/a;
        else 
          c:= diff(dterm0,x1)/b;
        end_if:
        if ode::odeIszero(diff(dterm0,x2) - a*c) and 
           ode::odeIszero(diff(dterm0,x1) - b*c) and  
           iszero(expand(diff(dterm0,x2) - a*c,optIgnoreAnalyticConstraints)) and 
           iszero(expand(diff(dterm0,x1) - b*c,optIgnoreAnalyticConstraints)) and 
           traperror((dterm0 | (x1=0,x2=0))) = 0 and
           ode::odeIszero((dterm0 | (x1=0,x2=0)) + c^2) and 
           iszero(expand(dterm0 | (x1=0,x2=0),optIgnoreAnalyticConstraints) + c^2) then
          if m <> -1 and n <> -1 then 
            return({int(exp(2*c*x)/exp((b*x^(m+1))/(m+1)+(a*x^(n+1))/(n+1)),x,intOptions)/exp(c*x), 
                    1/exp(c*x)});
          elif m = -1 and n <> -1 then
            return({int(exp(2*c*x)/exp((b*ln(x)+(a*x^(n+1))/(n+1))),x,intOptions)/exp(c*x), 
                    1/exp(c*x)});        
          elif m <> -1 and n = -1 then
            return({int(exp(2*c*x)/exp((b*x^(m+1))/(m+1)+ a*ln(x)),x,intOptions)/exp(c*x), 
                    1/exp(c*x)});
          elif m = -1 and n = -1 then
            return({int(exp(2*c*x)/exp((b*ln(x)+a*ln(x))),x,intOptions)/exp(c*x), 
                    1/exp(c*x)});        
          end_if;
        end_if;
      end_if;      
    elif nops(xpow1) = 1 then 
      n:= xpow1[1];
      dterm1:= subs(dterm1,[x^n = x1]);
      dterm0:= subs(dterm0,[x^n = x1]);
      a:= diff(dterm1,x1);
      if traperror((b:= dterm1 | x1=0)) <> 0 then 
        return(FAIL);
      end_if;
      if not has(a,x) and not has(a,x1) and not has(b,x1) and 
         not has(b,x) and not iszero(expand(a,optIgnoreAnalyticConstraints)) then 
        c:= ode::normal(diff(dterm0,x1)/a);
      else 
        return(FAIL);
      end_if;
      if ode::odeIszero(diff(dterm0,x1) - a*c) and 
         ode::odeIszero((dterm0 | (x1=0)) + c^2 - c*b) and 
         iszero(expand(diff(dterm0,x1) - a*c,optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0 | (x1=0),optIgnoreAnalyticConstraints) + c^2 - c*b) then
        if n <> -1 then 
          return({int(exp(2*c*x)/exp((a*x^(n+1))/(n+1)+b*x),x,intOptions)/exp(c*x), 
                  1/exp(c*x)});
        elif n = -1 then
          return({int(exp(2*c*x)/exp(a*ln(x)+b*x),x,intOptions)/exp(c*x), 
                  1/exp(c*x)});
        end_if;
      end_if;
    end_if;
  elif nops(xpow1) = 2 and nops(xpow0) = 3 then 
    n:= xpow1[1];
    m:= xpow1[2];
    if map(xpow0,expand,optIgnoreAnalyticConstraints) = {m+n,m-1,n-1} then 
      if type(float(m-n)) = DOM_FLOAT and float(m-n) > 0 then
        dterm1:= subs(dterm1,[x^m = x1, x^n = x2]);
        dterm0:= subs(dterm0,[x^(m+n) = x1, x^(m-1) = x2, x^(n-1) = x3]);
      else 
        dterm1:= subs(dterm1,[x^n = x2, x^m = x1]);
        dterm0:= subs(dterm0,[x^(m+n) = x1, x^(n-1) = x3, x^(m-1) = x2]);
      end_if;
      a:= diff(dterm1,x2);
      b:= diff(dterm1,x1);
      aa:= a;
      bb:= b;
      nn:= n;
      mm:= m;
      
      if not has(a,x1) and not has(b,x1) and not has(a,x2) and not has(b,x2) and 
         ode::odeIszero(diff(a,x)) and 
         ode::odeIszero(diff(b,x)) and 
         iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(diff(b,x),optIgnoreAnalyticConstraints)) then
        if ode::odeIszero(diff(dterm0,x2) - a*(m+1)) and 
           iszero(expand(diff(dterm0,x2) - a*(m+1),optIgnoreAnalyticConstraints)) then 
          aa:= b;
          bb:= a;
          nn:= n;
          mm:= m;
        elif ode::odeIszero(diff(dterm0,x3) - a*(n+1)) and 
             iszero(expand(diff(dterm0,x3) - a*(n+1),optIgnoreAnalyticConstraints)) then
          aa:= b;
          bb:= a;
          nn:= m;
          mm:= n;
        elif ode::odeIszero(diff(dterm0,x2) - b*(m+1)) and 
             iszero(expand(diff(dterm0,x2) - b*(m+1),optIgnoreAnalyticConstraints)) then
          aa:= a;
          bb:= b;
          nn:= n; 
          mm:= m;
        elif ode::odeIszero(diff(dterm0,x3) - b*(n+1)) and 
             iszero(expand(diff(dterm0,x3) - b*(n+1),optIgnoreAnalyticConstraints)) then
          aa:= a;
          bb:= b;
          nn:= m;
          mm:= n;
        end_if;
        if ode::odeIszero(((dterm0) | (x1=1, x2=1, x3=1)) - (aa*bb + bb*(mm+1) - aa)) and 
           iszero(expand(((dterm0) | (x1=1, x2=1, x3=1)) - (aa*bb + bb*(mm+1) - aa),optIgnoreAnalyticConstraints)) then 
            return({(x*int(exp((2*bb*x^(mm+1))/(mm+1))/(x^2*exp((bb*x^(mm+1))/(mm+1) + 
                           (aa*x^(nn+1))/(nn+1))),x,intOptions))/exp((bb*x^(mm+1))/(mm+1)), 
                    x/exp((bb*x^(mm+1))/(mm+1))});
        end_if;
      end_if;                
    elif map(xpow0,expand,optIgnoreAnalyticConstraints) = {m+n, m, n-1} or map(xpow0,expand) = {m+n, n, m-1} then 
      if map(xpow0,expand,optIgnoreAnalyticConstraints) = {m+n, n, m-1} then 
        mm:= n;
        nn:= m;
        n:= nn;
        m:= mm;
      end_if;
      if type(float(m-n)) = DOM_FLOAT and float(m-n) > 0 then
        dterm1:= subs(dterm1,[x^m = x1, x^n = x2]);
        dterm0:= subs(dterm0,[x^(m+n) = x1, x^m = x2, x^(n-1) = x3]);
      else 
        dterm1:= subs(dterm1,[x^n = x2, x^m = x1]);
        dterm0:= subs(dterm0,[x^(m+n) = x1, x^(n-1) = x3, x^m = x2]);
      end_if;
      a:= diff(dterm1,x2);
      b:= diff(dterm1,x1);
      c:= dterm1 | (x1=0, x2=0);
      aa:= a;
      bb:= b;
      nn:= n;
      mm:= m;
      if not has(a,x1) and not has(b,x1) and not has(a,x2) and not has(b,x2) and
         ode::odeIszero(diff(a,x)) and 
         ode::odeIszero(diff(b,x)) and 
         iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(diff(b,x),optIgnoreAnalyticConstraints)) then
        if ode::odeIszero(diff(dterm0,x2) - a*m) and 
           ode::odeIszero(diff(dterm0,x2) - a*m) and 
           iszero(expand(diff(dterm0,x2) - a*m,optIgnoreAnalyticConstraints)) and 
           iszero(expand(diff(dterm0,x2) - a*m,optIgnoreAnalyticConstraints)) then 
          aa:= a;
          bb:= b;
          nn:= m;
          mm:= n;
        elif ode::odeIszero(diff(dterm0,x3) - a*n) and 
             iszero(expand(diff(dterm0,x3) - a*n,optIgnoreAnalyticConstraints)) then 
          aa:= a;
          bb:= b;
          nn:= n;
          mm:= m;
        elif ode::odeIszero(diff(dterm0,x2) - b*m) and 
             iszero(expand(diff(dterm0,x2) - b*m,optIgnoreAnalyticConstraints)) then 
          aa:= b;
          bb:= a;
          nn:= m;
          mm:= n;
        elif ode::odeIszero(diff(dterm0,x3) - b*n) and 
             iszero(expand(diff(dterm0,x3) - b*n,optIgnoreAnalyticConstraints)) then 
          aa:= b;
          bb:= a;
          nn:= n;
          mm:= m;
        end_if;
        if ode::odeIszero(((dterm0) | (x1=1, x2=1, x3=1)) - (aa*bb + bb*c + aa*nn)) and 
           iszero(expand(((dterm0) | (x1=1, x2=1, x3=1)) - (aa*bb + bb*c + aa*nn),optIgnoreAnalyticConstraints)) then
          return({int(exp(2*c*x+(2*aa*x^(nn+1))/(nn+1))/exp(c*x+(bb*x^(mm+1))/(mm+1)+
                       (aa*x^(nn+1))/(nn+1)),x,intOptions)/exp(c*x+(aa*x^(nn+1))/(nn+1)), 
                  1/exp(c*x+(aa*x^(nn+1))/(nn+1))})
        end_if;
      end_if;          
    end_if;
  end_if;
  return(FAIL);
end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.2-3. eq. 61,62,63,64,69,
                                70,71,72,73,74,105,106,107,108, pp. 219-225)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_3_eq_61_62_63_64_69_70_71_72_73_74_105_106_107_108_p_219_to_225:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, a0, a1, a2, b0, b1, b2, D, z, F,
        k, Bk, lambda, mu, nu, a, b, c, beta, w, res, intOptions, optIgnoreAnalyticConstraints;
  save MAXEFFORT;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();   
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if; 
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  if iszero(dterm2) then
    return(FAIL);
  end_if;
  if has(ode::normal((a0:= diff(dterm0,x))),x) or 
     has(ode::normal((a1:= diff(dterm1,x))),x) or 
     has(ode::normal((a2:= diff(dterm2,x))),x) then 
    return(FAIL);
  end_if;
  b0:= dterm0 | x=0;
  b1:= dterm1 | x=0;
  b2:= dterm2 | x=0;
  D:= a1^2 - 4*a0*a2;
  if ode::odeIszero(a2 - 1) and 
     ode::odeIszero(b2) and 
     ode::odeIszero(a1) and 
     ode::odeIszero(a0) and 
     iszero(expand(a2 - 1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(b2,optIgnoreAnalyticConstraints)) and 
     iszero(expand(a1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(a0,optIgnoreAnalyticConstraints)) then 
    if type((nu:= float(1-b1))) = DOM_FLOAT then 
      nu:= abs(1-b1);
    else
      nu:= 1-b1;
    end_if;
    if traperror((res:= {x^((1-b1)/2)*besselJ(nu,2*sqrt(b0*x)), 
            x^((1-b1)/2)*besselY(nu,2*sqrt(b0*x))})) = 0 then 
      return(res)
    end_if;
  end_if;
  if ode::odeIszero(a2 - 1) and 
     ode::odeIszero(b2) and 
     ode::odeIszero(a1) and 
     ode::odeIszero(b0) and 
     iszero(expand(a2 - 1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(b2,optIgnoreAnalyticConstraints)) and 
     iszero(expand(a1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(b0,optIgnoreAnalyticConstraints)) then 
    if type((nu:= float(1-b1))) = DOM_FLOAT then 
      nu:= abs(1/2*(1-b1));
    else
      nu:= 1/2*(1-b1);
    end_if;
    if traperror((res:= {x^((1-b1)/2)*besselJ(nu,sqrt(a0)*x), 
            x^((1-b1)/2)*besselY(nu,sqrt(a0)*x)})) = 0 then 
      return(res);
    end_if;
  end_if;
  if ode::odeIszero(a2 - 1) and 
     ode::odeIszero(b2) and 
     iszero(expand(a2 - 1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(b2,optIgnoreAnalyticConstraints)) and 
     not iszero(expand(b1,optIgnoreAnalyticConstraints)) then 
    c:= b0/b1;
    if ode::odeIszero(a0-c*(a1-c)) and 
       iszero(expand(a0-c*(a1-c),optIgnoreAnalyticConstraints)) then 
      if traperror((res:= {exp(-c*x), 
              exp(-c*x)*int(exp(-(a1*x+b1*ln(x)))/exp(-c*x)^2,x,intOptions)})) = 0 then 
        return(res)
      end_if;
    end_if;
  end_if;
  if ode::odeIszero(a2 - 1) and 
     ode::odeIszero(b2) and 
     ode::odeIszero(a0-(a1/2)^2) and 
     ode::odeIszero(b0-a1*b1/2) and 
     iszero(expand(a2 - 1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(b2,optIgnoreAnalyticConstraints)) and 
     iszero(expand(a0-(a1/2)^2,optIgnoreAnalyticConstraints)) and 
     iszero(expand(b0-a1*b1/2,optIgnoreAnalyticConstraints))  then 
    if not iszero(expand(b1-1,optIgnoreAnalyticConstraints)) then 
      if traperror((res:= {exp(-a1/2*x), exp(-a1/2*x)*x^(1-b1)})) = 0 then 
        return(res)
      end_if;
    else
      if traperror((res:= {exp(-a1/2*x), exp(-a1/2*x)*ln(x)})) = 0 then 
        return(res)
      end_if;
    end_if;
  end_if;  
  if ode::odeIszero(a2-1) and 
     ode::odeIszero(b0-a1) and 
     ode::odeIszero(a0) and 
     iszero(expand(a2-1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(b0-a1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(a0,optIgnoreAnalyticConstraints)) then 
    if traperror((res:= {exp(-int((a1*x+b1-1)/(x+b2),x,intOptions)),
            exp(-int((a1*x+b1-1)/(x+b2),x))*int(exp(-int((a1*x+b1)/(x+b2),x,intOptions))/
                exp(-int((a1*x+b1-1)/(x+b2),x))^2,x,intOptions)})) = 0 then 
      return(res)
    end_if;
  end_if;
  if ode::odeIszero(a2-1) and 
     ode::odeIszero(b2) and 
     ode::odeIszero(a1+1) and 
     ode::odeIszero(a0) and 
     iszero(expand(a2-1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(b2,optIgnoreAnalyticConstraints)) and 
     iszero(expand(a1+1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(a0,optIgnoreAnalyticConstraints)) then 
    if traperror((res:= {exp(x)^(1/2)*(1/x^b1)^(1/2)*whittakerM(-b0-b1/2,1/2-b1/2,-x), 
            exp(x)^(1/2)*(1/x^b1)^(1/2)*whittakerW(-b0-b1/2,1/2-b1/2,-x)})) = 0 then 
      return(res);
    end_if;  
  end_if; 
  if not iszero(expand(a2,optIgnoreAnalyticConstraints)) and 
     not iszero(expand(a1^2 - 4*a0*a2,optIgnoreAnalyticConstraints)) then 
    k:= (sqrt(D) - a1)/(2*a2);
    Bk:= b2*k^2 + b1*k + b0;
    if not iszero(2*a2*k + a1) then 
      lambda:= -a2/(2*a2*k + a1);
      mu:= -b2/a2;
      if not iszero(lambda) then 
        z:= (x-mu)/lambda;
        a:= Bk/(2*a2*k + a1);
        b:= (a2*b1 - a1*b2)/(a2^2);
        if traperror((w:= kummerU(a,b,z))) = 0 then 
          y1:= combine(exp(k*x)*w,optIgnoreAnalyticConstraints);
          if traperror((y2:= combine(exp(k*x)*hypergeom([a],[b],z),optIgnoreAnalyticConstraints))) = 0 then  
            return({y1,y2});
          elif not iszero(a2*x+b2) then 
            F:= int((a1*x+b1)/(a2*x+b2),x,intOptions);
            if traperror((y2:= y1*int(exp(-F)/y1^2,x,intOptions))) = 0 then 
              return({y1,y2});
            end_if;
          end_if;
        end_if;  
      end_if;
    end_if;
  end_if;
  if ode::odeIszero(a2) and 
     iszero(expand(a2,optIgnoreAnalyticConstraints)) and
     not iszero(expand(a1,optIgnoreAnalyticConstraints)) then 
    k:= -a0/a1;
    Bk:= b2*k^2 + b1*k + b0;
    lambda:= 1;
    mu:= -(2*b2*k+b1)/a1;
    z:= (x-mu)/lambda;
    a:= Bk/(2*a1);
    if not iszero(expand(b2,optIgnoreAnalyticConstraints)) then 
      beta:= -a1/(2*b2);
      if traperror((w:= kummerU(a,1/2,beta*z^2))) = 0 then 
        y1:= combine(exp(k*x)*w,optIgnoreAnalyticConstraints);
        if traperror((y2:= exp(k*x)*hypergeom([a],[1/2],beta*z^2))) = 0 then 
          return({y1,y2});
        elif not iszero(a2*x+b2) then 
          F:= int((a1*x+b1)/(a2*x+b2),x,intOptions);
          if traperror((y2:= y1*int(exp(-F)/y1^2,x,intOptions))) = 0 then 
            return({y1,y2});    
          end_if;
        end_if;
      end_if;      
    end_if;
  end_if;
  if not iszero(expand(a2,optIgnoreAnalyticConstraints)) and 
     ode::odeIszero(a1^2 - 4*a0*a2) and 
     iszero(expand(a1^2 - 4*a0*a2,optIgnoreAnalyticConstraints)) then 
    k:= -a1/(2*a2);
    Bk:= b2*k^2 + b1*k + b0;
    lambda:= a2;
    mu:= -b2/a2;
    z:= (x-mu)/lambda;
    nu:= 1-(2*b2*k+b1)/a2;
    beta:= 2*sqrt(Bk);
    if not iszero(beta) then 
      w:= z^(nu/2)*besselY(nu,beta*sqrt(z));
      y1:= combine(exp(k*x)*w,optIgnoreAnalyticConstraints);
      if not iszero(a2*x+b2) then 
        F:= int((a1*x+b1)/(a2*x+b2),x,intOptions);
        y2:= y1*int(exp(-F)/y1^2,x,intOptions);
        return({y1,y2});    
      end_if;
    end_if;
  end_if;
  if ode::odeIszero(a2) and 
     ode::odeIszero(a1) and
     iszero(expand(a2,optIgnoreAnalyticConstraints)) and 
     iszero(expand(a1,optIgnoreAnalyticConstraints)) and
     not iszero(expand(a0,optIgnoreAnalyticConstraints)) then 
    if not iszero(expand(b2,optIgnoreAnalyticConstraints)) then 
      k:= -b1/(2*b2);
      Bk:= b2*k^2 + b1*k + b0;
      lambda:= 1;
      mu:= (b1^2-4*b0*b2)/(4*a0*b2);
      z:= (x-mu)/lambda;
      beta:= 2/3*(a0/b2)^(1/2);
      if not iszero(beta) then 
        w:= z^(1/2)*besselY(1/3,beta*z^(3/2));
        y1:= combine(exp(k*x)*w,optIgnoreAnalyticConstraints);
        if not iszero(a2*x+b2) then 
          F:= int((a1*x+b1)/(a2*x+b2),x,intOptions);
          y2:= y1*int(exp(-F)/y1^2,x,intOptions);
          return({y1,y2});        
        end_if;
      end_if;
    end_if;
  end_if;

  return(FAIL);
end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.2-3. eq. 65,66,67,68, p. 220)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_3_eq_65_66_67_68_p_220:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, a, b, c, n, nn, aa, nu, xpow, tmp,
        intOptions, optIgnoreAnalyticConstraints, t;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  a:= FAIL;
  if has(dterm1,x) then 
    if has((c:= ode::normal(diff(dterm2/dterm1,x))),x) then 
      return(FAIL);
    elif not iszero(expand(c,optIgnoreAnalyticConstraints)) then 
      a:= 1/c;
      dterm0:= ode::normal(dterm0/(c*dterm1));
      dterm1:= a;
      dterm2:= x;
    end_if;
  else
    if traperror((t:= iszero(expand(dterm2 | x=0)))) = 0 then 
      if t = TRUE and 
         not has((c:= expand(diff(dterm2,x)/*,optIgnoreAnalyticConstraints*/)),x) then 
        if c <> 1 and not iszero(expand(c,optIgnoreAnalyticConstraints)) then  
          a:= dterm1/c;
          dterm0:= dterm0/c;
          dterm2:= x;
        else
          a:= dterm1;
        end_if;
      end_if;
    end_if; 
  end_if;
  if a = FAIL or 
     not iszero(expand(diff(dterm2,x)-1,optIgnoreAnalyticConstraints)) or 
     not iszero(expand(diff(dterm1,x),optIgnoreAnalyticConstraints)) then  
    return(FAIL);
  end_if;
  xpow:= {};
  misc::maprec(dterm0,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  if nops(xpow) <> 1 and nops(xpow) <> 2 then 
    return(FAIL);
  end_if;
  if nops(xpow) = 1 then 
    n:= xpow[1];
    if traperror((tmp:= subs(dterm0,x^n=0,EvalChanges))) = 0 and ode::odeIszero(tmp) and 
       iszero(expand(tmp,optIgnoreAnalyticConstraints)) and 
       not iszero(expand(n+1,optIgnoreAnalyticConstraints)) and 
       traperror((b:= dterm0 | x=1)) = 0 then
      if ode::odeIszero(a-(1-n)/2) and 
         ode::odeIszero(dterm0-b*x^(1-2*a)) and 
         iszero(expand(a-(1-n)/2,optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0-b*x^(1-2*a),optIgnoreAnalyticConstraints)) then // 65.
        return({sin(sqrt(b)/(a-1)*x^(1-a)), cos(sqrt(b)/(a-1)*x^(1-a))})
      end_if;
      if ode::odeIszero(a-1+3*(n+1)/2) and 
         iszero(expand(a-1+3*(n+1)/2,optIgnoreAnalyticConstraints)) then  // 66.
        nn:= (n+1)/2;
        aa:= (-b/nn^2)^(1/2);
        return({(aa*x^nn+1)*exp(-aa*x^nn), (-aa*x^nn+1)*exp(aa*x^nn)})
      end_if; 
      if ode::odeIszero(diff(a,x)) and 
         ode::odeIszero(dterm0-b*x^n) and 
         iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0-b*x^n)) and 
         not iszero(expand(b,optIgnoreAnalyticConstraints)) then // 67.
        nu:= (1-a)/(n+1);
        if type(float(nu)) = DOM_FLOAT then 
          nu:= abs(nu)
        end_if;
        return({x^(1/2-a/2)*besselJ(nu,(2*b^(1/2)*x^(n/2+1/2))/(n+1)), 
                x^(1/2-a/2)*besselY(nu, (2*b^(1/2)*x^(n/2+1/2))/(n+1))});
      else
        return(FAIL); 
      end_if;
    end_if;
  else
    if ode::odeIszero(xpow[1]-2*xpow[2]-1) and 
       iszero(expand(xpow[1]-2*xpow[2]-1,optIgnoreAnalyticConstraints)) then 
      n:= xpow[2];
    elif ode::odeIszero(xpow[2]-2*xpow[1]-1) and 
         iszero(expand(xpow[2]-2*xpow[1]-1,optIgnoreAnalyticConstraints)) then 
      n:= xpow[1]; 
    else
      return(FAIL);
    end_if;
    if ode::odeIszero(n+1) and 
       iszero(expand(n+1,optIgnoreAnalyticConstraints)) then 
      return(FAIL)
    end_if;
    dterm0:= combine(dterm0,optIgnoreAnalyticConstraints);
    if not iszero(expand(a+n,optIgnoreAnalyticConstraints)) then 
      b:= ode::normal(subs(dterm0, [x^(2*n+1)=0,x^(n)=1])/(a+n)); 
      if ode::odeIszero(subs(dterm0, [x^(2*n+1)=1,x^(n)=0])+b^2) and 
         iszero(expand(subs(dterm0, [x^(2*n+1)=1,x^(n)=0])+b^2,optIgnoreAnalyticConstraints)) then 
        return({exp(-b/(n+1)*x^(n+1)), 
                exp(-b/(n+1)*x^(n+1))*int(1/(x^a*exp(-(2*b)/(n+1)*x^(n+1))),x,intOptions)})
      end_if;
    end_if;        
  end_if;

  return(FAIL);
end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.2-4. eq. 118,119,132,
                                133,134,143,144,145,147, 
                                p. 226,228,229,230)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_4_eq_118_119_132_133_134_143_144_145_147_p_226_228_229_230:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, a, b, c, xpow0, lcf, x1, x2, x3, 
        n, m, intOptions, nu, aRoot, input_eq, res, xpow1, t, optIgnoreAnalyticConstraints;
  save MAXEFFORT;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  input_eq:= eq;
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  if traperror((t:= iszero(expand(dterm2 | x=0/*,optIgnoreAnalyticConstraints*/)))) <> 0 or t <> TRUE then 
    return(FAIL);
  end_if; 
    
  if has((lcf:= expand(diff(dterm2,x,x)/2/*,optIgnoreAnalyticConstraints*/)),x) or iszero(lcf) then 
    return(FAIL);
  else
    if lcf <> 1 then  
      dterm0:= dterm0/lcf;
      dterm1:= dterm1/lcf;
      dterm2:= x^2;
    end_if;
  end_if; 

  xpow0:= {};
  misc::maprec(dterm0,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow0:= xpow0 union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  xpow1:= {};
  misc::maprec(dterm1,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow1:= xpow1 union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  x1:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  x2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1]));
  x3:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,x1,x2]));

  if not has((a:= diff(dterm1,x)),x) and 
     ode::odeIszero(dterm1 | x=0) and 
     iszero(expand(dterm1 | x=0,optIgnoreAnalyticConstraints)) then
    if nops(xpow0) = 1 then 
      n:= xpow0[1];
      t:= subs(dterm0, [x^n =x1]);
      if not has(t,x) and traperror((c:= t | x1=0)) = 0 and not has(((b:= diff(t,x1)),x1)) then         
        nu:= 1/n*sqrt((1-a)^2-4*c);
        if (a=1 and n=2 and b=1) or (a=1 and n=2 and b=-1) then 
          return(ode::specfunc(input_eq,y,x,{},{}));
        end_if;
        if not iszero(expand(b,optIgnoreAnalyticConstraints)) then 
          return({x^((1-a)/2)*besselJ(nu, 2/n*sqrt(b)*x^(n/2)),
                  x^((1-a)/2)*besselY(nu, 2/n*sqrt(b)*x^(n/2))});
        end_if;
      end_if;
    elif not has((b:= diff(dterm0,x)),x) and 
         not(iszero(expand(b,optIgnoreAnalyticConstraints))) and 
         traperror((c:= dterm0 | x=0)) = 0 then 
      n:= 1;
      nu:= 1/n*sqrt((1-a)^2-4*c);
      return({x^((1-a)/2)*besselJ(nu, 2/n*sqrt(b)*x^(n/2)),
              x^((1-a)/2)*besselY(nu, 2/n*sqrt(b)*x^(n/2))});
    end_if;
  end_if;

  if ode::odeIszero(dterm1) and 
     iszero(expand(dterm1,optIgnoreAnalyticConstraints)) and nops(xpow0) = 2 then 
    n:= xpow0[1];
    m:= xpow0[2];
    if iszero(expand(n-2*m,optIgnoreAnalyticConstraints)) or iszero(expand(m-2*n,optIgnoreAnalyticConstraints)) then 
      if iszero(expand(n-2*m,optIgnoreAnalyticConstraints)) then 
        n:= m;
      end_if;
      t:= subs(dterm0,[x^(2*n)=x1,x^n=x2]);
      if not has(t,x) then 
        a:= -diff(t,x1);
        if not iszero(expand(a,optIgnoreAnalyticConstraints)) then 
          if type(a) = "_power" and op(a,2) = 2 then 
            aRoot:= op(a,1);
          else
            aRoot:= sqrt(a);
          end_if;
          b:= ode::normal(-(diff(t,x2)/aRoot-1+n)/2);
          if ode::odeIszero((t | (x1=0,x2=0)) + b*(b-1)) and 
             iszero(expand((t | (x1=0,x2=0)) + b*(b-1),optIgnoreAnalyticConstraints)) then 
            return({x^b*exp(aRoot*x^n/n),
                    x^b*exp(aRoot*x^n/n)*int(1/(x^(2*b)*exp(2*aRoot*x^n/n)),x,intOptions)});  
          end_if;
          b:= ode::normal(-(diff(t,x2)/(-aRoot)-1+n)/2);
          if ode::odeIszero((t | (x1=0,x2=0)) + b*(b-1)) and 
             iszero(expand((t | (x1=0,x2=0)) + b*(b-1),optIgnoreAnalyticConstraints)) then 
            return({x^b*exp(-aRoot*x^n/n),
                    x^b*exp(-aRoot*x^n/n)*int(1/(x^(2*b)*exp(-2*aRoot*x^n/n)),x,intOptions)});  
          end_if;
        end_if;
      end_if;
    end_if;   
  end_if;

  if not has(diff(dterm1,x),x) and 
     not has(dterm0,x) then
    res:= ode::specfunc(input_eq,y,x,solveOptions,odeOptions union {IgnoreAnalyticConstraints});
    if res <> {} then 
      return(res)
    end_if;
  end_if;

  if nops(xpow1) = 1 and nops(xpow0) > 0 then
    n:= xpow1[1];
    t:= subs(dterm1,x^n=x1);
    if not has(t,x) then 
      a:= diff(t,x1); 
      if not iszero(a) and 
         not has((b:= -ode::normal(diff(subs(dterm0,x^n=x1),x1)/a)),x) and 
         not has((c:= -ode::normal(diff(subs(dterm0,x^(n-1)=x1),x1)/a)),x) and 
         ode::odeIszero(dterm0 + a*b*x^n+a*c*x^(n-1)+b^2*x^2+2*b*c*x+c^2-c) and 
         iszero(expand(dterm0 + a*b*x^n+a*c*x^(n-1)+b^2*x^2+2*b*c*x+c^2-c,optIgnoreAnalyticConstraints)) then  
        return({x^c*exp(b*x), 
                x^c*exp(b*x)*int(1/(x^(2*c)*exp(2*b*x+a/(n-1)*x^(n-1))),x,intOptions)});   
      end_if;
    end_if;
  end_if;

  if nops(xpow1) = 1 and nops(xpow0) > 0 then
    n:= xpow1[1];
    t:= subs(dterm1,x^n=x1);
    if not has(t,x) and 
       ode::odeIszero(t | x1=0) and 
       iszero(expand(t | x1=0,optIgnoreAnalyticConstraints)) then 
      a:= diff(t,x1);
      if not iszero(a) then 
        t:= subs(dterm0, x^(n-1)=x1);
        m:= ode::normal(diff(t,x1)/a);
        t:= subs(dterm0, x^(n+2*m)=x1); 
        b:= ode::normal(diff(t,x1)/a);
        if ode::odeIszero(dterm0 - (a*b*x^(n+2*m) - b^2*x^(4*m+2) + a*m*x^(n-1) - m^2 - m)) and 
           iszero(expand(dterm0 - (a*b*x^(n+2*m) - b^2*x^(4*m+2) + a*m*x^(n-1) - m^2 - m),optIgnoreAnalyticConstraints)) then 
          return({1/x^m*exp(-b/(2*m+1)*x^(2*m+1)), 
                  1/x^m*exp(-b/(2*m+1)*x^(2*m+1))*int(exp(-a/(n-1)*x^(n-1))/((1/x^m*exp(-b/(2*m+1)*x^(2*m+1)))^2),x,
                                                      intOptions)})
        end_if;
      end_if;                                                
    end_if;
  end_if;
  
  if nops(xpow1) = 1 then 
    n:= xpow1[1]-1;
    t:= subs(dterm1, x^(n+1)=x1);
    a:= diff(t,x1);
    if not iszero(expand(a,optIgnoreAnalyticConstraints)) then 
      b:= diff((t | x1=0), x);
      if not has(a,x) and not has(b,x) and not iszero(expand(b,optIgnoreAnalyticConstraints)) then 
        if ode::odeIszero(dterm0 - (b*(a*x^n-1))) and 
           iszero(expand(dterm0 - (b*(a*x^n-1)),optIgnoreAnalyticConstraints)) then
          return({x^(-b), x^(-b)*int(x^b*exp(-a/n*x^n),x,intOptions)})     
        end_if;
      end_if;
    end_if;
  end_if;

  return(FAIL);
end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.2-5. eq. 149, p. 230)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_5_eq_149_p_230:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, n;
  save MAXEFFORT;
begin
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)));
  dterm1:= combine(expand(diff(eq,y1)));
  dterm2:= combine(expand(diff(eq,y2)));
  if has(dterm0,x) then 
    return(FAIL);
  end_if;
  if not iszero(dterm1) then 
    return(FAIL);
  end_if;
  if ode::odeIszero(dterm2 + (1-x^2)) and 
     iszero(expand(dterm2 + (1-x^2))) then 
    dterm0:= combine(expand(-dterm0));
    dterm2:= combine(expand(-dterm2));
  end_if;
  if not iszero(expand(dterm2 - (1-x^2))) then 
    return(FAIL);
  end_if;
  n:= (4*dterm0 + 1)^(1/2)/2 + 1/2;
  return({(x - x^3)*hypergeom([n/2 + 1, 3/2 - n/2], [3/2], x^2), 
          (x^2 - 1)*hypergeom([1 - n/2, n/2 + 1/2], [1/2], x^2)})
end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.2-5. eq. 153 & 154 & 156, p. 231)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_5_eq_154_p_230:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, nu;
  save MAXEFFORT;
begin
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)));
  dterm1:= combine(expand(diff(eq,y1)));
  dterm2:= combine(expand(diff(eq,y2)));
  if has(dterm0,x) then 
    return(FAIL);
  end_if;
  if ode::odeIszero(expand(dterm2 + (1-x^2))) and 
     iszero(expand(dterm2 + (1-x^2))) then 
    dterm0:= combine(expand(-dterm0));
    dterm1:= combine(expand(-dterm1));
    dterm2:= combine(expand(-dterm2));
  end_if;
  if ode::odeIszero(dterm2 - (1-x^2)) and 
     ode::odeIszero(dterm1 + 2*x) and 
     ode::odeIszero(diff(dterm0,x)) and 
     iszero(expand(dterm2 - (1-x^2))) and 
     iszero(expand(dterm1 + 2*x)) and 
     iszero(expand(diff(dterm0,x))) then 
    nu:= - (4*dterm0 + 1)^(1/2)/2 - 1/2;
    if indets(nu) <> {} then 
      return({orthpoly::legendre(nu, x),
              orthpoly::legendre(nu, x)*int(1/(orthpoly::legendre(nu, x)^2*(x^2 - 1)), x, intOptions)});
    end_if;
  end_if;
  return(FAIL);
end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.2-5. eq. 171, 172, p. 233, 234)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_5_eq_hypergeom:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, a, b, c, d, t1, t2, lcf, 
        res, t; 
  save MAXEFFORT;
begin 
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)));
  dterm1:= combine(expand(diff(eq,y1)));
  dterm2:= combine(expand(diff(eq,y2)));
  if has(diff(dterm2,x,x),x) or 
     has(diff(dterm1,x,x),x) or 
     has(diff(dterm0,x,x),x) then 
    return(FAIL)
  end_if;
 
  if iszero(diff((lcf:= ode::normal(dterm2/(x*(x-1)))),x)) and 
     iszero(diff(dterm1,x,x)) and 
     iszero(diff(dterm0,x)) then  
    if not iszero(expand(lcf)) then  
      t1:= diff(dterm1,x)/lcf;
      t2:= dterm0/lcf;
      a:= t1/2+(t1^2-2*t1-4*t2+1)^(1/2)/2-1/2;
      b:= t1/2-(t1^2-2*t1-4*t2+1)^(1/2)/2-1/2;
      c:= -((dterm1/lcf) | x=0);
      if not iszero(expand(c)) and not testtype(c, Type::Integer) then 
        return({hypergeom([a, b],[c],x), 
                x^(1-c)*hypergeom([b+1-c, a+1-c],[2-c],x)})
      elif iszero(expand(c)) or testtype(c, Type::NegInt) then 
        return({x^(1-c)*hypergeom([b+1-c, a+1-c],[2-c],x),
                x^(1-c)*hypergeom([b+1-c, a+1-c],[2-c],x)*
                int(exp(-int(((a+b+1)*x-c)/(x^2-x),x))/
                    (x^(2-2*c)*hypergeom([b+1-c, a+1-c],[2-c],x)^2),x)})
      elif testtype(c, Type::NonNegInt) then 
        return({hypergeom([a, b],[c],x), 
                hypergeom([a, b],[c],x)*int(exp(-int(((a+b+1)*x-c)/(x^2-x),x))/
                hypergeom([a, b],[c],x)^2,x)})      
      end_if;
    end_if;          
  end_if;

  if ode::odeIszero(dterm2 | x=0) and 
     ode::odeIszero(diff(dterm1,x,x)) and
     ode::odeIszero(diff(dterm0,x)) and 
     iszero(expand(dterm2 | x=0)) and 
     iszero(expand(diff(dterm1,x,x))) and
     iszero(expand(diff(dterm0,x))) then 
    a:= diff(dterm2,x) | x=0;
    if not iszero(a) then 
      b:= diff(dterm1,x);
      c:= dterm1 | x=0;
      d:= dterm0; 
      lcf:= diff(dterm2,x,x)/2;    
      if not iszero(expand(lcf)) then 
        if lcf <> 1 then 
          a:= a/lcf;
          b:= b/lcf;
          c:= c/lcf;
          d:= dterm0/lcf; 
        end_if;
        res:= ode::PZ_Sec_212_5_eq_hypergeom(
                 x*(1-x)*diff(y(x),x,x)+(c/a-b*x)*diff(y(x),x)-d*y(x),
                 y,x,solveOptions={},odeOptions={});
        if res = FAIL then 
          res:= ode::PZ_Sec_212_5_eq_169_170_p_233_234(
                 x*(1-x)*diff(y(x),x,x)+(c/a-b*x)*diff(y(x),x)-d*y(x),
                 y,x,solveOptions={},odeOptions={});
        end_if;          
        t:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
        res:= map(res, intlib::changevar, x=-t/a, t);
        res:= subs(res, x=-x/a);
        res:= subs(res, t=x); 
        return(res);        
      end_if;  
    end_if;
  end_if;
  
  return(FAIL);

end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.2-5. eq. 169, 170, 171, p. 233, 234)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_5_eq_169_170_p_233_234:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, a, b, c, d, t1, t2, lambda, 
        res, m, k, csts;
  save MAXEFFORT;
begin 
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)));
  dterm1:= combine(expand(diff(eq,y1)));
  dterm2:= combine(expand(diff(eq,y2)));
  if has(diff(dterm2,x,x),x) or 
     has(diff(dterm1,x,x),x) or 
     has(diff(dterm0,x,x),x) then 
    return(FAIL)
  end_if;
  a:= diff(dterm2,x,x)/2;
  b:= dterm2 | x=0;
  if ode::odeIszero(diff(dterm2,x) | x=0) and 
     ode::odeIszero(diff(dterm0,x) | x=0) and 
     ode::odeIszero(diff(dterm1,x) | x=0) and 
     iszero(expand(diff(dterm2,x) | x=0)) and 
     iszero(expand(diff(dterm0,x) | x=0)) and 
     iszero(expand(diff(dterm1,x) | x=0)) then 
    c:= diff(dterm1,x,x)/2;
    d:= dterm1 | x=0;
    t1:= diff(dterm0,x,x)/2;
    t2:= dterm0 | x=0;
    if not iszero(expand(t2)) and not iszero(expand(a-b*t1/t2)) then 
      lambda:= ode::normal((c-d*t1/t2)/(a-b*t1/t2));
      if not iszero(expand(lambda)) then
        return({exp(-lambda*x),
                exp(-lambda*x)*int(exp(-int((c*x^2+d)/(a*x^2+b),x))*exp(2*lambda*x),x)})
      end_if;          
    end_if;   
  end_if;
  if ode::odeIszero(diff(dterm2,x) | x=0) and 
     ode::odeIszero(diff(dterm0,x) | x=0) and 
     iszero(expand(diff(dterm2,x) | x=0)) and 
     iszero(expand(diff(dterm0,x) | x=0)) and
     not iszero(expand(b)) then 
    lambda:= ode::normal((dterm1 | x=0)/(2*b));
    c:= (diff(dterm1,x) | x=0) + a;
    if not iszero(expand(lambda)) and 
       ode::odeIszero(diff(dterm1,x,x)/2 - lambda*(c+a)) and 
       ode::odeIszero(dterm0-lambda^2*(c*x^2+b)) and 
       iszero(expand(diff(dterm1,x,x)/2 - lambda*(c+a))) and 
       iszero(expand(dterm0-lambda^2*(c*x^2+b))) then
      return({(lambda*x+1)*exp(-lambda*x), 
              (lambda*x+1)*exp(-lambda*x)*int(exp(-int((lambda*(c+a)*x^2+(c-a)*x+
                                                        2*b*lambda)/(a*x^2+b),x))/((lambda*x+1)^2*
                                                        exp(-2*lambda*x)),x)});
    end_if;
  end_if;
  
  if diff(dterm2,x,x) = 2 then
    a:= (diff(dterm2,x) | x=0)/2;
    b:= dterm2 | x=0; 
    m:= sqrt(-dterm0);
    if ode::odeIszero(dterm1-(a+x)) and 
       ode::odeIszero(diff(dterm0,x)) and 
       iszero(expand(dterm1-(a+x))) and 
       iszero(expand(diff(dterm0,x))) and 
       not iszero(expand(m)) then 
      return({(x+a+sqrt(x^2+2*a*x+b))^m, (x+a+sqrt(x^2+2*a*x+b))^(-m)})
    end_if;
  end_if;

  a:= diff(dterm2,x,x)/2;
  b:= diff(dterm2,x) | x=0;
  c:= dterm2 | x=0;

  if ode::odeIszero(diff(dterm1,x,x)) and 
     ode::odeIszero(diff(dterm0,x)) and 
     iszero(expand(diff(dterm1,x,x))) and 
     iszero(expand(diff(dterm0,x))) then 
    d:= diff(dterm1,x);
    k:= dterm1 | x=0;
    if ode::odeIszero(dterm0 - (d-2*a)) and 
       iszero(expand(dterm0 - (d-2*a))) then
      res:= solve(ode((a*x^2+b*x+c)*diff(y(x),x) + ((d-2*a)*x+k-b)*y(x) = genident("C"),y(x)));
      csts:= [op(freeIndets(res) minus (indets(eq)))];
      if nops(csts) = 2 then 
        res:= subs(res,csts[1]=1,csts[2]=0) union subs(res,csts[1]=0,csts[2]=1);
        return(res)      
      end_if;  
    end_if;
  end_if;

  return(FAIL);

end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.2-6. eq. 186, 190, 194, p. 237)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_6_eq_186_190_194_p_237:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, lcf, a, b, c, d, s, z,
        a0, b0, a1, b1, a2, res, optIgnoreAnalyticConstraints, dterm0_new, dterm1_new, 
        dterm2_new, tmp;
  save MAXEFFORT;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
 
  if iszero(dterm0) or iszero(dterm1) or iszero(dterm2) or 
     not iszero(expand(diff(dterm2,x,x,x,x),optIgnoreAnalyticConstraints)) or 
     not iszero(expand(diff(dterm1,x,x,x),optIgnoreAnalyticConstraints)) or
     not iszero(expand(diff(dterm0,x,x),optIgnoreAnalyticConstraints)) then
    return(FAIL);
  end_if;

  if iszero(diff((lcf:= ode::normal(dterm2/x^3)),x)) then 
    if dterm2 <> x^3 then
      dterm0_new:= dterm0/lcf;
      dterm1_new:= dterm1/lcf;
      dterm2_new:= dterm2/lcf;
    else 
      dterm0_new:= dterm0;
      dterm1_new:= dterm1;
      dterm2_new:= dterm2;
    end_if;
    if ode::odeIszero(dterm1_new | x=0) and 
       iszero(expand(dterm1_new | x=0,optIgnoreAnalyticConstraints)) then 
      a:= diff(dterm1_new,x,x)/2;
      b:= diff(dterm1_new,x) | x=0;
      if indets(a) <> {} and indets(b) <> {} then
        c:= diff(dterm0_new,x);
        d:= dterm0_new | x=0;
        if indets(c) <> {} and indets(d) <> {} then 
          return({x^(1/2 - (a^2 - 2*a - 4*c + 1)^(1/2)/2 - a/2)*
                  hypergeom([((b*(a - 1))/2 - d + (b*(a^2 - 2*a - 4*c + 1)^(1/2))/2)/b], 
                            [(a^2 - 2*a - 4*c + 1)^(1/2) + 1], b/x), 
                  x^(1/2 - (a^2 - 2*a - 4*c + 1)^(1/2)/2 - a/2)*
                  kummerU(((b*(a - 1))/2 - d + (b*(a^2 - 2*a - 4*c + 1)^(1/2))/2)/b, 
                          (a^2 - 2*a - 4*c + 1)^(1/2) + 1, b/x)});
        end_if;
      end_if;
    end_if;
  end_if;

  if not has((lcf:= diff(dterm2,x,x,x)/6),x) and 
     not iszero(expand(lcf,optIgnoreAnalyticConstraints)) and 
     ode::odeIszero(dterm2 | x=0) and 
     iszero(expand(dterm2 | x=0,optIgnoreAnalyticConstraints)) then 
    if traperror((tmp:= expand(dterm2 | x=0,optIgnoreAnalyticConstraints))) = 0 and 
       iszero(tmp) then // implementing eq. 192 additionally 
      a:= lcf;
      b:= (diff(dterm2,x,x)/2) | x=0;
      if ode::odeIszero(dterm2-(x^2*(a*x+b))) and 
         ode::odeIszero(dterm1+(2*x*(a*x+2*b))) and
         ode::odeIszero(dterm0-2*(a*x+3*b)) and 
         iszero(expand(dterm2-(x^2*(a*x+b)),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm1+(2*x*(a*x+2*b)),optIgnoreAnalyticConstraints)) and
         iszero(expand(dterm0-2*(a*x+3*b),optIgnoreAnalyticConstraints)) and 
         not iszero(a*x+b) then
        return({x^2/(a*x+b),x^3/(a*x+b)})   
      end_if;  
    end_if; // end of implementation of eq. 192
    if lcf <> 1 then 
      dterm0:= dterm0/lcf;
      dterm1:= dterm1/lcf;
      dterm2:= dterm2/lcf;
    end_if;  
    a:= diff(dterm2,x) | x=0;
    if not iszero(a) and 
       ode::odeIszero(dterm2 - (x^3+a*x)) and 
       iszero(expand(dterm2 - (x^3+a*x),optIgnoreAnalyticConstraints)) then
      if not has(subs(dterm1,x^2=y0),x) then 
        c:= dterm1 | x=0;
        b:= diff(dterm1,x,x)/2;
        if not has((s:= diff(dterm0,x)),x) then 
          z:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
          res:= ode::PZ_Sec_212_5_eq_hypergeom(
                  z*(1-z)*diff(y(z),z,z) + 1/2*(1+c/a-(1+b)*z)*diff(y(z),z)-1/4*s*y(z),
                  y, z, solveOptions,odeOptions);
          if res = FAIL then 
            res:= ode::PZ_Sec_212_5_eq_169_170_p_233_234(
                    z*(1-z)*diff(y(z),z,z) + 1/2*(1+c/a-(1+b)*z)*diff(y(z),z)-1/4*s*y(z),
                    y, z, solveOptions,odeOptions);
          end_if;  
          res:= map(res, intlib::changevar, z=-x^2/a, x);
          res:= subs(res, z=-x^2/a);
          return(res);        
        end_if;
      end_if;
    end_if;
    
    a2:= diff(dterm2,x,x)/2 | x=0;
    if not iszero(a2) and 
       ode::odeIszero(dterm2 - (x^3+a2*x^2)) and 
       iszero(expand(dterm2 - (x^3+a2*x^2),optIgnoreAnalyticConstraints)) then 
      if not has((b1:= diff(dterm1,x,x)/2),x) and 
         ode::odeIszero(dterm1 | x=0) and 
         iszero(expand(dterm1 | x=0,optIgnoreAnalyticConstraints)) then 
        a1:= (diff(dterm1,x)) | x=0;
        if not has((b0:= diff(dterm0,x)),x) then 
          a0:= dterm0 | x=0;
          if traperror((res:= {x^((a1^2 - 2*a1*a2 + a2^2 - 4*a0*a2)^(1/2)/(2*a2) - a1/(2*a2) + 1/2)*
                   hypergeom([((a2*b1)/2 - a1/2 + (a1^2 - 2*a1*a2 + a2^2 - 4*a0*a2)^(1/2)/2 + 
                              (a2*(b1^2 - 2*b1 - 4*b0 + 1)^(1/2))/2)/a2, -(a1/2 - (a2*b1)/2 - 
                              (a1^2 - 2*a1*a2 + a2^2 - 4*a0*a2)^(1/2)/2 + 
                              (a2*(b1^2 - 2*b1 - 4*b0 + 1)^(1/2))/2)/a2], 
                             [(a2 + (a1^2 - 2*a1*a2 + a2^2 - 4*a0*a2)^(1/2))/a2], -x/a2), 
                  x^(1/2 - (a1^2 - 2*a1*a2 + a2^2 - 4*a0*a2)^(1/2)/(2*a2) - a1/(2*a2))*
                   hypergeom([-(a1/2 - (a2*b1)/2 + (a1^2 - 2*a1*a2 + a2^2 - 4*a0*a2)^(1/2)/2 + 
                              (a2*(b1^2 - 2*b1 - 4*b0 + 1)^(1/2))/2)/a2, -(a1/2 - (a2*b1)/2 + 
                              (a1^2 - 2*a1*a2 + a2^2 - 4*a0*a2)^(1/2)/2 - (a2*(b1^2 - 2*b1 - 
                              4*b0 + 1)^(1/2))/2)/a2], 
                             [(a2 - (a1^2 - 2*a1*a2 + a2^2 - 4*a0*a2)^(1/2))/a2], -x/a2)})) = 0 then 
            if nops(select(res, hastype, "hypergeom")) = 2 then 
              return(res)
            end_if;
          end_if;
        end_if;
      end_if;
    end_if;
  end_if;

  return(FAIL);
end_proc:


ode::PZ_Sec_212_6_eq_195_196_197_199_200_201_205_206_207_208_209_210_p_238:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, a, b, c, aa, bb, cc, Int,
        lambda, k, m, d, res, optIgnoreAnalyticConstraints, xpow, t, tmp;
  save MAXEFFORT;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);

  Int:= if has(odeOptions,freeze) then freeze(int) else int end_if;

  if ode::odeIszero(diff(dterm2,x,x,x,x)) and 
     iszero(expand(diff(dterm2,x,x,x,x),optIgnoreAnalyticConstraints)) then
    if traperror((tmp:= expand(dterm2 | x=0,optIgnoreAnalyticConstraints))) = 0 and 
       iszero(tmp) then
      a:= diff(dterm2,x,x,x)/6;
      b:= (diff(dterm2,x,x) | x=0)/2;
      c:= (diff(dterm2,x) | x=0);
      if ode::odeIszero(diff(dterm1,x,x,x)) and 
         iszero(expand(diff(dterm1,x,x,x),optIgnoreAnalyticConstraints)) then
        aa:= diff(dterm1,x,x)/2;
        bb:= diff(dterm1,x) | x=0;
        if ode::odeIszero((dterm1 | x=0)-2*c) and 
           iszero(expand((dterm1 | x=0)-2*c,optIgnoreAnalyticConstraints)) then
          if ode::odeIszero(dterm0-(bb-2*b)) and 
             iszero(expand(dterm0-(bb-2*b),optIgnoreAnalyticConstraints)) then 
            y0:= 2*a-aa+(2*b-bb)/x;
            if not iszero(y0) then 
              y1:= y0*Int(exp(-Int((aa*x^2+bb*x+2*c)/(a*x^3+b*x^2+c*x),x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          end_if;
          if ode::odeIszero(dterm0+(aa*x+2*b-bb)) and 
             iszero(expand(dterm0+(aa*x+2*b-bb),optIgnoreAnalyticConstraints)) and 
             not iszero(expand(aa-a,optIgnoreAnalyticConstraints)) then 
            lambda:= (c*aa+(b-bb)*(2*b-bb))/(aa-a);
            y0:= aa*x+2*(bb-b)+lambda/x;
            if not iszero(y0) then 
              y1:= y0*Int(exp(-Int((aa*x^2+bb*x+2*c)/(a*x^3+b*x^2+c*x),x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          end_if;
        end_if;
        if ode::odeIszero(aa+2*a) and 
           ode::odeIszero(b+1+bb) and 
           iszero(expand(aa+2*a,optIgnoreAnalyticConstraints)) and 
           iszero(expand(b+1+bb,optIgnoreAnalyticConstraints)) then 
          k:= dterm1 | x=0; 
          if ode::odeIszero(dterm0-2*(a*x+1)) and 
             iszero(expand(dterm0-2*(a*x+1),optIgnoreAnalyticConstraints)) then 
            y0:= (a*k+b-1)*x^2+(c+k)*(2*x-k);
            if not iszero(y0) then 
              y1:= y0*Int(exp(-Int((-2*a*x^2-(b+1)*x+k)/(a*x^3+b*x^2+c*x),x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          end_if;
        end_if;
        /* ================================================================
           Implementation of ODE 198: the solution 'x^(1-k)' is not correct;
           There must be a typo in the book by Polyanin & Zaitsev. 

               k:= dterm1 | x=0;
               if iszero(expand(dterm0-((k-1)*((aa-a*k)*x+bb-b*k)),optIgnoreAnalyticConstraints)) then 
                 y0:= x^(1-k);
                 y1:= y0*Int(exp(-Int((aa*x^2+bb*x+k)/(a*x^3+b*x^2+c*x),x,intOptions))/y0^2,x,intOptions);
                 return({y0,y1});
               end_if;

           End of implementation of ODE 198. 
           ================================================================ */
        if ode::odeIszero(diff(dterm0,x,x)) and  
           iszero(expand(diff(dterm0,x,x),optIgnoreAnalyticConstraints)) and iszero((dterm0 | x=0) - 1) then 
          m:= -diff(dterm0,x)/2; 
          if ode::odeIszero(dterm1-((m-a)*x^2+(2*c*m-1)*x-c)) and 
             iszero(expand(dterm1-((m-a)*x^2+(2*c*m-1)*x-c),optIgnoreAnalyticConstraints)) then
            y0:= (a+m)*x^2+(2*b+4*c*m-1)*(x+c);
            if not iszero(y0) then 
              y1:= y0*Int(exp(-Int((((m-a)*x^2+(2*c*m-1)*x-c))/(a*x^3+b*x^2+c*x),x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          end_if;
        end_if;
        k:= dterm1 | x=0; 
        if ode::odeIszero(dterm0+2*(a+aa)*x-1) and 
           ode::odeIszero(2*(2*a+aa)*(c+k)+(2*b+2*bb+1)*(bb+1+2*k*(a+aa))) and 
           iszero(expand(dterm0+2*(a+aa)*x-1,optIgnoreAnalyticConstraints)) and 
           iszero(expand(2*(2*a+aa)*(c+k)+(2*b+2*bb+1)*(bb+1+2*k*(a+aa)),optIgnoreAnalyticConstraints)) then
          y0:= (2*a+aa)*x^2+(2*b+2*bb+1)*(x-k); 
          if not iszero(y0) then 
            y1:= y0*Int(exp(-Int((aa*x^2+bb*x+k)/(a*x^3+b*x^2+c*x),x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;
        end_if;
      end_if;
    end_if;
    a:= diff(dterm2,x,x,x)/6;
    if traperror((b:= dterm2 | x=0)) = 0 then 
      if ode::odeIszero(dterm2-(a*x^3+x^2+b)) and 
         ode::odeIszero(dterm1-(a^2*x*(x^2-b))) and 
         ode::odeIszero(dterm0+(a^3*b*x)) and 
         iszero(expand(dterm2-(a*x^3+x^2+b),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm1-(a^2*x*(x^2-b)),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0+(a^3*b*x),optIgnoreAnalyticConstraints)) then
        y0:= (a*x+2)*exp(-a*x);
        if not iszero(y0) then 
          y1:= y0*Int(exp(-Int((a^2*x*(x^2-b))/(a*x^3+x^2+b),x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      // now treat the case for -1*eq instead of eq (also possible)
      elif ode::odeIszero(dterm2+(-a*x^3+x^2-b)) and 
         ode::odeIszero(dterm1+(a^2*x*(x^2+b))) and 
         ode::odeIszero(dterm0-(a^3*b*x)) and 
         iszero(expand(dterm2+(-a*x^3+x^2-b),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm1+(a^2*x*(x^2+b)),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0-(a^3*b*x),optIgnoreAnalyticConstraints)) then  
        y0:= (-a*x+2)*exp(a*x);
        if not iszero(y0) then 
          y1:= y0*Int(exp(-Int((a^2*x*(x^2+b))/(-a*x^3+x^2-b),x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      end_if;
      if ode::odeIszero(diff(dterm0,x,x)) and 
         iszero(expand(diff(dterm0,x,x),optIgnoreAnalyticConstraints)) then 
        lambda:= dterm0 | x=0;
        if ode::odeIszero(dterm1 + (x^2-lambda^2)) and 
           ode::odeIszero(dterm0 - (x+lambda)) and 
           iszero(expand(dterm1 + (x^2-lambda^2),optIgnoreAnalyticConstraints)) and 
           iszero(expand(dterm0 - (x+lambda),optIgnoreAnalyticConstraints)) then 
          y0:= x-lambda;
          if not iszero(dterm2) and not iszero(y0) then 
            y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;
        end_if;
      end_if;
      if not has(dterm0,x) then 
        lambda:= dterm0;
        a:= a/2;
        if traperror((b:= diff(dterm2,x,x)/4 | x=0)) = 0 and 
           traperror((c:= diff(dterm2,x)/2 | x=0)) = 0 and 
           traperror((d:= (dterm2 | x=0)/2)) = 0 then 
          if ode::odeIszero(dterm1 - (3*a*x^2+2*b*x+c)) and 
             iszero(expand(dterm1 - (3*a*x^2+2*b*x+c),optIgnoreAnalyticConstraints)) and 
             not iszero(a*x^3+b*x^2+c*x+d) and 
             not iszero(lambda) then 
            res:= {exp((2^(1/2)*(-lambda)^(1/2)*x)/2), 
                   1/exp((2^(1/2)*(-lambda)^(1/2)*x)/2)};
            res:= subs(res, x = Int(1/sqrt(a*x^3+b*x^2+c*x+d),x,intOptions));        
            //if not has(res,int) and has(res,ln) then 
            //  return(simplify(res,IgnoreAnalyticConstraints))
            //else
              return(res);
            //end_if;          
          end_if;
        end_if;  
      end_if;
      if ode::odeIszero(diff(dterm0,x,x)) and 
         iszero(expand(diff(dterm0,x,x),optIgnoreAnalyticConstraints)) then 
        a:= a/2;
        if traperror((b:= diff(dterm2,x,x)/4 | x=0)) = 0 and 
           traperror((c:= diff(dterm2,x)/2 | x=0)) = 0 and 
           traperror((d:= (dterm2 | x=0)/2)) = 0 then 
          if ode::odeIszero(dterm1 - 3*(3*a*x^2+2*b*x+c)) and 
             iszero(expand(dterm1 - 3*(3*a*x^2+2*b*x+c),optIgnoreAnalyticConstraints)) then 
            lambda:= (dterm0 | x=0) - 2*b;
            if ode::odeIszero(dterm0 - (6*a*x+2*b+lambda)) and 
               iszero(expand(dterm0 - (6*a*x+2*b+lambda),optIgnoreAnalyticConstraints)) and 
               not iszero(a*x^3+b*x^2+c*x+d) then 
              if not iszero(lambda) then 
                return({exp(Int(((-2)*lambda)^(1/2)/(2*(a*x^3+ b*x^2+c*x+d)^(1/2)),x,intOptions))/(a*x^3+b*x^2+c*x+d)^(1/2), 
                        1/(exp(Int(((-2)*lambda)^(1/2)/(2*(a*x^3+b*x^2+c*x+d)^(1/2)),x,intOptions))*(a*x^3+b*x^2+c*x+d)^(1/2))});
              else 
                y0:= 1/(a*x^3+b*x^2+c*x+d)^(1/2);
                if not iszero(y0) then 
                  y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
                  return({y0,y1});
                end_if;
              end_if;
            end_if;
          end_if;     
        end_if;  
      end_if;
    end_if;  
    if ode::odeIszero(diff(dterm1,x,x,x)) and 
       ode::odeIszero(diff(dterm0,x,x)) and 
       iszero(expand(diff(dterm1,x,x,x),optIgnoreAnalyticConstraints)) and 
       iszero(expand(diff(dterm0,x,x),optIgnoreAnalyticConstraints)) then
      aa:= -diff(dterm0,x);
      if traperror((bb:= - (dterm0 | x=0))) = 0 then 
        if not iszero(bb) then 
          cc:= (dterm1 | x=0)/bb;
          if ode::odeIszero(dterm1 - (aa*x^2+(aa*cc+bb)*x+bb*cc)) and 
             iszero(expand(dterm1 - (aa*x^2+(aa*cc+bb)*x+bb*cc),optIgnoreAnalyticConstraints)) then
            y0:= x + cc;
            y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;
        else // bb = 0
          if ode::odeIszero(dterm1 | x=0) and 
             iszero(expand(dterm1 | x=0,optIgnoreAnalyticConstraints)) and not iszero(aa) then 
            cc:= (diff(dterm1,x) | x=0)/aa;
            if ode::odeIszero(dterm1 - (aa*x^2+(aa*cc+bb)*x+bb*cc)) and 
               iszero(expand(dterm1 - (aa*x^2+(aa*cc+bb)*x+bb*cc),optIgnoreAnalyticConstraints)) then
              y0:= x + cc;
              y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          end_if;  
        end_if;   
      end_if;  
    end_if;
    if has(dterm1,x^3) and 
       ode::odeIszero(diff(dterm1,x,x,x,x)) and 
       iszero(expand(diff(dterm1,x,x,x,x),optIgnoreAnalyticConstraints)) then 
      lambda:= combine((dterm1 | x=0)^(1/3),IgnoreAnalyticConstraints);
      if ode::odeIszero(dterm1 - (x^3+lambda^3)) and 
         ode::odeIszero(dterm0 + (x^2-lambda*x+lambda^2)) and 
         iszero(expand(dterm1 - (x^3+lambda^3),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0 + (x^2-lambda*x+lambda^2),optIgnoreAnalyticConstraints)) then 
        y0:= x + lambda;
        y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
    if traperror((tmp:= expand(dterm2 | x=0,optIgnoreAnalyticConstraints))) = 0 and 
       iszero(tmp) then 
      a:= a/2;
      if traperror((b:= diff(dterm2,x,x)/4 | x=0)) = 0 and 
         traperror((c:= diff(dterm2,x)/2 | x=0)) = 0 then 
        xpow:= {};
        misc::maprec(dterm0,
                     {"_power"} = proc(elem)
                                    begin
                                      if op(elem,1) = x then 
                                        xpow:= xpow union {op(elem,2)}
                                      end_if;
                                      elem;
                                    end_proc);  
        xpow:= [op(xpow)];
        if nops(xpow) = 1 and 
           traperror(subs(dterm0,x^(xpow[1])=0, EvalChanges)) = 0 and
           ode::odeIszero(subs(dterm0,x^(xpow[1])=0)) and 
           iszero(expand(subs(dterm0,x^(xpow[1])=0),optIgnoreAnalyticConstraints)) then 
          lambda:= subs(dterm0,x^(xpow[1])=1);
          k:= xpow[1]-1; 
          if not has(lambda,x) and 
             ode::odeIszero(dterm1-(a*(2-k)*x^2+b*(1-k)*x-c*k)) and 
             iszero(expand(dterm1-(a*(2-k)*x^2+b*(1-k)*x-c*k))) and 
             not iszero(a*x^2+b*x+c) then 
            t:= specfunc::Wurzelbehandlung(x^k/(a*x^2+b*x+c)); 
            return({exp((2^(1/2)*(-lambda)^(1/2)*int(t,x,intOptions))/2), 
                    1/exp((2^(1/2)*(-lambda)^(1/2)*int(t,x,intOptions))/2)});
          end_if;
        end_if;
      end_if;  
    end_if;
  end_if;  

  return(FAIL);
end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.2-7. eq. 215,217,218,219,228,229,230,231,233,234, p. 240)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_7_eq_215_217_218_219_228_229_230_231_233_234_p_240:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, optIgnoreAnalyticConstraints, lcf, xpow1,
        n, Int, a, b, c, d, p, q, w, res, z, k, aa, bb, xpow0, m, lambda, mu, dterm0_new, dterm2_new, eq_new,
        k1, k2, A, csts, tmp1, tmp2, tmp3;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   

  Int:= if has(odeOptions,freeze) then freeze(int) else int end_if;

  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]); 
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  
  if traperror(subs(dterm2,x^4=0,EvalChanges)) = 0 and 
     traperror(subs(dterm2,x^4=1,EvalChanges)) = 0 and
     not has(subs(dterm2,x^4=0),x) then 
    lcf:= subs(dterm2,x^4=1);
    if lcf <> 1 and not iszero(lcf) then 
      dterm0:= dterm0/lcf;
      dterm1:= dterm1/lcf;
      dterm2:= dterm2/lcf;
    end_if;
    xpow1:= {};
    misc::maprec(dterm1,
                 {"_power"} = proc(elem)
                                begin
                                  if op(elem,1) = x then 
                                    xpow1:= xpow1 union {op(elem,2)}
                                  end_if;
                                  elem;
                                end_proc);  
    xpow1:= [op(xpow1)];
    if nops(xpow1) = 1 then 
      n:= xpow1[1]; 
      if traperror(subs(dterm1,x^n=1,EvalChanges)) = 0 and 
         traperror(subs(dterm1,x^n=0,EvalChanges)) = 0 and 
         traperror(subs(dterm0, [x^(n-1)=0,x^(n-2)=0],EvalChanges)) = 0 and 
         not has((a:=subs(dterm1,x^n=1)),x) and 
         ode::odeIszero(subs(dterm1,x^n=0)) and  
         iszero(expand(subs(dterm1,x^n=0),optIgnoreAnalyticConstraints)) then 
        b:= combine((-subs(dterm0, [x^(n-1)=0,x^(n-2)=0]))^(1/2),IgnoreAnalyticConstraints);
        if not has(b,x) then 
          if ode::odeIszero(dterm0 + (a*x^(n-1)+a*b*x^(n-2)+b^2)) and 
             iszero(expand(dterm0 + (a*x^(n-1)+a*b*x^(n-2)+b^2),optIgnoreAnalyticConstraints)) then 
            y0:=x*exp(-b/x);
            y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          elif ode::odeIszero(dterm0 + (a*x^(n-1)-a*b*x^(n-2)+b^2)) and 
               iszero(expand(dterm0 + (a*x^(n-1)-a*b*x^(n-2)+b^2),optIgnoreAnalyticConstraints)) then 
            y0:=x*exp(b/x);
            y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;
        end_if;
      end_if;
    end_if;
  end_if;
  if not iszero(dterm2) and 
     ode::odeIszero(diff(dterm2,x,x,x,x,x)) and 
     iszero(expand(diff(dterm2,x,x,x,x,x),optIgnoreAnalyticConstraints)) then 
    if ode::odeIszero(dterm1) and iszero(expand(dterm1,optIgnoreAnalyticConstraints)) then 
      if iszero(dterm2-1) and testtype(dterm0,Type::PolyExpr(x)) = FALSE then 
        eq_new:= numer(eq);
        dterm0_new:= combine(expand(diff(eq_new,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
        dterm2_new:= combine(expand(diff(eq_new,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
      else 
        dterm0_new:= dterm0;
        dterm2_new:= dterm2;
      end_if;  
      if testtype(dterm0_new,Type::PolyExpr(x)) and 
         testtype(dterm2_new,Type::PolyExpr(x)) and 
         ode::odeIszero(diff(dterm0_new,x,x,x)) and 
         ode::odeIszero(dterm2_new | x=0) and 
         ode::odeIszero(dterm2_new | x=1) and 
         iszero(expand(diff(dterm0_new,x,x,x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm2_new | x=0,optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm2_new | x=1,optIgnoreAnalyticConstraints)) and
         not has((a:= subs(dterm2_new,[x^4=1,x^3=0,x^2=0])),x) then 
        if ode::odeIszero(expand(dterm2_new - a*x^2*(x-1)^2)) and 
           iszero(expand(dterm2_new - a*x^2*(x-1)^2)) then 
          b:= diff(dterm0_new,x,x)/2;
          c:= diff(dterm0_new,x) | x=0;
          d:= dterm0_new | x=0;
          p:= ((a-4*d)/a)^(1/2)/2+1/2; 
          q:= (-(4*b-a+4*c+4*d)/a)^(1/2)/2+1/2;
          w:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
          res:= ode::PZ_Sec_212_5_eq_hypergeom(
                   a*x*(x-1)*diff(w(x),x,x)+2*a*((p+q)*x-p)*diff(w(x),x)+(2*a*p*q-c-2*d)*w(x),
                   w, x, solveOptions, odeOptions);
          if res = FAIL then 
            res:= ode::PZ_Sec_212_5_eq_169_170_p_233_234(
                   a*x*(x-1)*diff(w(x),x,x)+2*a*((p+q)*x-p)*diff(w(x),x)+(2*a*p*q-c-2*d)*w(x),
                   w, x, solveOptions, odeOptions);
          end_if;  
          if res <> FAIL then 
            return({x^p*(x-1)^q*res[1],x^p*(x-1)^q*res[2]})
          end_if;
        end_if; 
      end_if;
    end_if;    
    if traperror((tmp1:= expand(dterm2 | x=0,optIgnoreAnalyticConstraints))) = 0 and 
       traperror((tmp2:= subs(dterm2, [x^4=0,x^2=0]))) = 0 and 
       traperror((tmp3:= expand(diff(dterm0,x),optIgnoreAnalyticConstraints))) = 0 and 
       iszero(tmp1) and iszero(tmp2) and iszero(tmp3) then 
      lcf:= diff(dterm2,x,x,x,x)/24;
      if lcf <> 1 and not iszero(lcf) then 
        dterm0:= dterm0/lcf;
        dterm1:= dterm1/lcf;
        dterm2:= dterm2/lcf;
      end_if; 
      a:= diff(dterm2,x,x)/2 | x=0;
      if ode::odeIszero(diff(dterm1,x,x,x,x)) and 
         iszero(expand(diff(dterm1,x,x,x,x),optIgnoreAnalyticConstraints)) and 
         iszero(subs(dterm1,[x^3=0,x=0])) then 
        c:= diff(dterm1,x);
        b:= diff(dterm1,x,x,x)/6;
        d:= dterm0;
        z:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
        res:= ode::PZ_Sec_212_6_eq_186_190_194_p_237(
                4*z^2*(z+a)*diff(y(z),z,z)+2*z*((b+1)*z+a+c)*diff(y(z),z)+d*y(z),
                y,z,solveOptions,odeOptions);
        return(res | z=x^2);
      end_if;
    end_if;
    if traperror((tmp1:= subs(dterm2,[x^4=0,x^2=0],EvalChanges))) = 0 and 
      not has(tmp1,x) then 
      a:= combine((diff(dterm2,x,x,x,x)/24)^(1/2),IgnoreAnalyticConstraints);
      if traperror((b:= combine(expand((dterm2 | x=0)^(1/2)/*,optIgnoreAnalyticConstraints*/),
                                IgnoreAnalyticConstraints))) = 0 then 
        for aa in [a,-a] do
          for bb in [b,-b] do
            if not iszero(aa) and 
               ode::odeIszero(dterm2-(aa*x^2+bb)^2) and 
               ode::odeIszero(diff(dterm0,x)) and 
               iszero(expand(dterm2-(aa*x^2+bb)^2,optIgnoreAnalyticConstraints)) and 
               iszero(expand(diff(dterm0,x),optIgnoreAnalyticConstraints)) then 
              k:= dterm0;
              if ode::odeIszero(diff(dterm1,x,x,x,x)) and 
                 iszero(expand(diff(dterm1,x,x,x,x),optIgnoreAnalyticConstraints)) and 
                 not iszero(bb) then 
                c:= (dterm1 | x=0)/bb;
                if ode::odeIszero(dterm1 - (2*aa*x+c)*(aa*x^2+bb)) and 
                   iszero(expand(dterm1 - (2*aa*x+c)*(aa*x^2+bb),optIgnoreAnalyticConstraints))  then 
                  res:= {1/exp((arctan((aa*x)/(aa*bb)^(1/2))*(c/2 + (c^2 - 4*k)^(1/2)/2))/(aa*bb)^(1/2)), 
                         1/exp((arctan((aa*x)/(aa*bb)^(1/2))*(c/2 - (c^2 - 4*k)^(1/2)/2))/(aa*bb)^(1/2))};
                  if nops(res) = 2 then
                    return(res);
                  elif nops(res) = 1 then 
                    y0:= res[1];
                    if not iszero(y0) then 
                      y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
                      return({y0,y1});
                    end_if; 
                  end_if;
                end_if;
              end_if;
            end_if;
          end_for;
        end_for;
        for aa in [a,-a] do 
          for bb in  [b,-b] do 
            if ode::odeIszero(diff(dterm1,x,x,x,x,x)) and 
               iszero(expand(diff(dterm1,x,x,x,x,x),optIgnoreAnalyticConstraints)) and 
               not iszero(aa) and not iszero(bb) then 
              c:= diff(dterm1,x,x,x,x)/(24*aa);
              if traperror((d:= (dterm1 | x=0)/bb)) = 0 then 
                if ode::odeIszero(dterm1 - (aa*x^2+bb)*(c*x^2+d)) and
                   ode::odeIszero(dterm0 - 2*(bb*c-aa*d)*x) and 
                   iszero(expand(dterm1 - (aa*x^2+bb)*(c*x^2+d))) and
                   iszero(expand(dterm0 - 2*(bb*c-aa*d)*x)) then
                  y0:= exp(-int((c*x^2+d)/(aa*x^2+bb),x));
                  if not iszero(y0) then 
                    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
                    return({y0,y1});
                  end_if; 
                end_if;
              end_if;
            end_if;
          end_for;
        end_for;
      end_if;  
      if a <> 1 and not iszero(a) then 
        dterm0:= dterm0/a^2;
        dterm1:= dterm1/a^2;
        dterm2:= dterm2/a^2;
      end_if;
      if traperror((a:= combine((dterm2 | x=0)^(1/2),IgnoreAnalyticConstraints))) = 0 then 
        if ode::odeIszero(dterm2 - (x^2+a)^2) and 
           iszero(expand(dterm2 - (x^2+a)^2)) then 
          xpow0:= {};
          misc::maprec(dterm0,
                       {"_power"} = proc(elem)
                                      begin
                                        if op(elem,1) = x then 
                                          xpow0:= xpow0 union {op(elem,2)}
                                        end_if;
                                        elem;
                                      end_proc);  
          xpow0:= [op(xpow0)];
          if nops(xpow0) = 1 then 
            n:= xpow0[1]-1; 
            if traperror(subs(dterm0,x^(n+1)=0,EvalChanges)) = 0 and 
               traperror(subs(dterm0,x^(n+1)=1,EvalChanges)) = 0 and 
               ode::odeIszero(a+subs(dterm0,x^(n+1)=0)) and 
               iszero(expand(a+subs(dterm0,x^(n+1)=0),optIgnoreAnalyticConstraints)) then 
              b:= -subs(dterm0,x^(n+1)=1)-a;
              if ode::odeIszero(expand(dterm1 - (b*x^n*(x^2+a)),optIgnoreAnalyticConstraints)) and 
                 iszero(expand(dterm1 - (b*x^n*(x^2+a)),optIgnoreAnalyticConstraints)) then 
                y0:= sqrt(x^2+a);
                if not iszero(y0) then 
                  y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
                  return({y0,y1});
                end_if; 
              end_if;
            end_if;
          elif nops(xpow0) = 2 and has(xpow0,2) then 
            n:= ({op(xpow0)} minus {2})[1]-1;
            z:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
            if not has((b:= diff(subs(dterm1,x^(n+2)=z),z)),x) then 
              if ode::odeIszero(dterm1 - (b*x^n*(x^2+a))) and 
                 iszero(expand(dterm1 - (b*x^n*(x^2+a)),optIgnoreAnalyticConstraints)) and
                 not iszero(b) then 
                m:= -diff(subs(dterm0,x^(n+1)=z),z)/b;
                if ode::odeIszero(dterm0 + m*(b*x^(n+1)+(m-1)*x^2+a)) and 
                   iszero(expand(dterm0 + m*(b*x^(n+1)+(m-1)*x^2+a),optIgnoreAnalyticConstraints)) then 
                 y0:= (x^2+a)^(m/2);
                  if not iszero(y0) then 
                    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
                    return({y0,y1});
                  end_if; 
                end_if;
              end_if;
            end_if;          
          end_if; 
        end_if;
      end_if;    
    end_if;
    if ode::odeIszero(diff(dterm0,x)) and 
       iszero(expand(diff(dterm0,x),optIgnoreAnalyticConstraints)) then 
      mu:= dterm0; 
      res:= ode::solveWithoutProperties(dterm2=0,x,IgnoreProperties,IgnoreSpecialCases,op(solveOptions));      
      if type(res) = piecewise then 
        res:= piecewise::disregardPoints(res);
      end_if;
      if type(res) = DOM_SET and nops(res) = 2 then 
        a:= res[1];
        b:= res[2];
        res:= ode::normal(dterm1/((x-a)*(x-b)));
        if ode::odeIszero(diff(res,x,x)) and 
           iszero(expand(diff(res,x,x),optIgnoreAnalyticConstraints)) then 
          lambda:= res | x=0;
          if ode::odeIszero(res - (2*x+lambda)) and 
             iszero(expand(res - (2*x+lambda),optIgnoreAnalyticConstraints)) then 
            z:= (x-a)/(x-b);
            res:= ode::solveWithoutProperties((a-b)^2*x^2+(a-b)*(a+b+lambda)*x+mu,x,IgnoreProperties,IgnoreSpecialCases,
                        op(solveOptions));
            if type(res) = piecewise then 
              res:= piecewise::disregardPoints(res);
            end_if;
            if type(res) = DOM_SET and nops(res) = 2 then 
              k1:= res[1];
              k2:= res[2];
              return({z^k1, z^k2});
            elif type(res) = DOM_SET and nops(res) = 1 then 
              k:= res[1];
              return({z^k, z^k*ln(z)})
            end_if;
          end_if;
        end_if;
      end_if; 
    end_if;
    if ode::odeIszero(diff(dterm0,x)) and 
       iszero(expand(diff(dterm0,x),optIgnoreAnalyticConstraints)) then 
      A:= dterm0;
      a:= combine((diff(dterm2,x,x,x,x)/24)^(1/2),IgnoreAnalyticConstraints);
      if not iszero(a) then 
        b:= (diff(dterm2,x,x,x) | x=0)/(12*a);
        if not iszero(b) then 
          c:= (diff(dterm2,x) | x=0)/(2*b);
          if ode::odeIszero(dterm2-(a*x^2+b*x+c)^2) and 
             ode::odeIszero(dterm1) and 
             iszero(expand(dterm2-(a*x^2+b*x+c)^2,optIgnoreAnalyticConstraints)) and 
             iszero(expand(dterm1,optIgnoreAnalyticConstraints)) then // eq. 234
            return({(a*x^2 + b*x + c)^(1/2)/exp((int(1/(a*x^2 + b*x + c), x)*(b^2 - 4*A - 4*a*c)^(1/2))/2), 
                     exp((int(1/(a*x^2 + b*x + c), x)*(b^2 - 4*A - 4*a*c)^(1/2))/2)*(a*x^2 + b*x + c)^(1/2)})
          end_if;
          if ode::odeIszero(dterm2-(a*x^2+b*x+c)^2) and 
             iszero(expand(dterm2-(a*x^2+b*x+c)^2,optIgnoreAnalyticConstraints)) then // eq. 237
            m:= dterm0;
            k:= ode::normal(dterm1/(a*x^2+b*x+c))-2*a*x;
            if ode::odeIszero(diff(k,x)) and 
               iszero(expand(diff(k,x),optIgnoreAnalyticConstraints)) then 
              z:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
              res:= solve(ode(diff(y(z),z,z)+(k-b)*diff(y(z),z)+m*y(z),y(z)));
              csts:= [op(freeIndets(res) minus (indets(eq) union {z}))];
              if nops(csts) = 2 then 
                res:= subs(res,csts[1]=1,csts[2]=0) union subs(res,csts[1]=0,csts[2]=1);
                if has(res,int) then 
                  res:= map(res, intlib::changevar, 
                            z = int(1/(a*x^2+b*x+c),x,intOptions), x);
                end_if; 
                return(subs(res,z = int(1/(a*x^2+b*x+c),x,intOptions)));  
              end_if;
            end_if;
          end_if;
        end_if;
      end_if;         
    end_if;
  end_if;
    
  return(FAIL);
end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.2-8. eq. 242,243,245,247,248,
                                                     249,250,251,252,
                                                     254 to 265 
                                                  p. 244,245,246)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_8_eq_242_243_245_247_248_249_250_251_252_254_to_265_p_244:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, optIgnoreAnalyticConstraints, xpow, lcf,  
        a, b, n, Int, z, xpow1, m, t, lambda1, lambda2, lambda3, lambda4, c, d,
        dterm0_new, dterm1_new, dterm2_new, n_new, p, q, r, s, A, B, aa, bb, cc,
        u, res, i, csts, lambda, t1, t2, t3;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   

  Int:= if has(odeOptions,freeze) then freeze(int) else int end_if;

  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);

  if ode::odeIszero(diff(ode::normal(dterm2/x),x)) and 
     iszero(expand(diff(ode::normal(dterm2/x),x),optIgnoreAnalyticConstraints)) then 
    xpow:= [1];
  else
    xpow:= {};
    misc::maprec(dterm2,
                 {"_power"} = proc(elem)
                                begin
                                  if op(elem,1) = x then 
                                    xpow:= xpow union {op(elem,2)}
                                  end_if;
                                  elem;
                                  end_proc);  
    xpow:= [op(xpow)];
  end_if;

  if nops(xpow) = 1 and 
     traperror(subs(dterm2, x^(xpow[1])=1,EvalChanges)) = 0 then 
    n:= xpow[1]; 
    lcf:= subs(dterm2, x^n=1);
    if ode::odeIszero(diff(lcf,x)) and 
       iszero(expand(diff(lcf,x),optIgnoreAnalyticConstraints)) then 
      if not iszero(lcf) and lcf <> 1 then 
        dterm0:= dterm0/lcf;
        dterm1:= dterm1/lcf;
        dterm2:= dterm2/lcf;
      end_if;
    end_if;

    if ode::odeIszero(dterm2-x^n) and 
       iszero(expand(dterm2-x^n,optIgnoreAnalyticConstraints)) then 

      if ode::odeIszero(diff(dterm1,x,x)) and traperror(ode::odeIszero(dterm1 | x=0)) = 0 and 
         ode::odeIszero(dterm1 | x=0) and iszero(expand(diff(dterm1,x,x),optIgnoreAnalyticConstraints)) and
         iszero(expand(dterm1 | x=0,optIgnoreAnalyticConstraints)) then 
        a:= diff(dterm1,x);
        z:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
        b:= combine(expand((-diff(subs(dterm0,x^n=z),z))^(1/2)/*,optIgnoreAnalyticConstraints*/),IgnoreAnalyticConstraints);
        if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and 
           ode::odeIszero(dterm0 + (b^2*x^n+2*b*x^(n-1)+a*b*x+a)) and
           iszero(expand(dterm0 + (b^2*x^n+2*b*x^(n-1)+a*b*x+a),optIgnoreAnalyticConstraints)) then 
          y0:= x*exp(b*x);
          if not iszero(y0) then 
            y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if; 
        end_if;
      end_if;

      if ode::odeIszero(diff(dterm1,x,x)) and ode::odeIszero(diff(dterm0,x)) and 
         iszero(expand(diff(dterm1,x,x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(diff(dterm0,x),optIgnoreAnalyticConstraints)) then 
        a:= diff(dterm1,x);
        b:= dterm1 | x=0;
        if not has(a,x) and not has(b,x) and ode::odeIszero(dterm0+a) and
           iszero(expand(dterm0+a,optIgnoreAnalyticConstraints)) then 
          y0:= a*x+b;
          if not iszero(y0) then 
            y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if; 
        end_if;
      end_if;

      if iszero(diff(dterm0,x)) then 
        b:= dterm0;
        z:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
        if n <> 3 then 
          a:= diff(subs(dterm1,x^2=z),z);
        else 
           a:= diff(subs(dterm1,x^2=z),z)-2;
        end_if; 
        if ode::odeIszero(diff(a,x)) and ode::odeIszero(dterm1 - (2*x^(n-1)+a*x^2+b*x)) and 
           iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
           iszero(expand(dterm1 - (2*x^(n-1)+a*x^2+b*x),optIgnoreAnalyticConstraints)) then 
          y0:= a+b/x;
          if not iszero(y0) then 
            y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if; 
        end_if;
      end_if;

      if iszero(diff(dterm0,x,x)) then 
        z:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
        a:= diff(subs(dterm1,x^n=z),z);
        if not has(a,x) and not has(a,z) and not iszero(a) then
          b:= ode::normal(dterm0/(a^2*x));  
          if ode::odeIszero(diff(b,x)) and 
             ode::odeIszero(dterm1 - (a*x^n-x^(n-1)+a*b*x+b)) and 
             iszero(expand(diff(b,x),optIgnoreAnalyticConstraints)) and 
             iszero(expand(dterm1 - (a*x^n-x^(n-1)+a*b*x+b),optIgnoreAnalyticConstraints)) then 
            y0:= (a*x+1)*exp(-a*x);
            if not iszero(y0) then 
              y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if; 
          end_if;
        end_if;  
      end_if;
  
      if traperror(subs(dterm1,x=1,EvalChanges)) = 0 and 
         not has((a:= subs(dterm1,x=1,EvalChanges)-1),x) then
        xpow1:= {};
        misc::maprec(dterm1,
                     {"_power"} = proc(elem)
                                    begin
                                      if op(elem,1) = x then 
                                        xpow1:= xpow1 union {op(elem,2)}
                                      end_if;
                                      elem;
                                      end_proc);  
        xpow1:= [op(xpow1)];
        if nops(xpow1) = 0 and 
           ode::odeIszero(diff(dterm1,x,x)) and 
           iszero(expand(diff(dterm1,x,x),optIgnoreAnalyticConstraints)) then 
          xpow1:= [1];
        end_if;
        if nops(xpow1) = 1 then 
          m:= xpow1[1]-n;
          if ode::odeIszero(dterm1-(a*x^(n+m)+1)) and ode::odeIszero(dterm0-a*x^m*(1+m*x^(n-1))) and 
             iszero(expand(dterm1-(a*x^(n+m)+1),optIgnoreAnalyticConstraints)) and 
             iszero(expand(dterm0-a*x^m*(1+m*x^(n-1)),optIgnoreAnalyticConstraints)) then 
            y0:= exp(-a/(m+1)*x^(m+1));
            if not iszero(y0) then 
              y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if; 
          end_if;      
        end_if;
      end_if;

    end_if;
  end_if;

  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  if ode::odeIszero(diff(ode::normal(dterm2),x,x)) and 
     iszero(expand(diff(ode::normal(dterm2),x,x),optIgnoreAnalyticConstraints)) then 
    xpow:= [1];
  end_if;
  z:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  u:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,z]));

  if nops(xpow) = 1 then 
    n:= xpow[1];

    if traperror(subs(dterm2,x^n=0,EvalChanges)) = 0 and 
       traperror(subs(dterm2,x^n=1,EvalChanges)) = 0 and 
       not has((b:= subs(dterm2,x^n=0)),x) and 
       not has((a:= subs(dterm2,x^n=1)-b),x) and 
       ode::odeIszero(dterm2-(a*x^n+b)) and 
       iszero(expand(dterm2-(a*x^n+b),optIgnoreAnalyticConstraints)) then 
      if not iszero(a) and 
         traperror(subs(dterm1,x^n=0,EvalChanges)) = 0 and 
         traperror(subs(dterm1,x^n=1,EvalChanges)) = 0 and 
         not has((d:= subs(dterm1,x^n=0)),x) and 
         not has((c:= subs(dterm1,x^n=1)-d),x) and 
         ode::odeIszero(dterm1-(c*x^n+d)) and 
         iszero(expand(dterm1-(c*x^n+d),optIgnoreAnalyticConstraints)) then 
        t:= diff(subs(dterm0,x^n=z),z);
        lambda1:= simplify((c+(c^2-4*a*t)^(1/2))/(2*a)-c/a,IgnoreAnalyticConstraints);
        lambda2:= simplify((c-(c^2-4*a*t)^(1/2))/(2*a),IgnoreAnalyticConstraints);
        lambda3:= simplify((c+(c^2-4*a*t)^(1/2))/(2*a),IgnoreAnalyticConstraints);
        lambda4:= simplify((c-(c^2-4*a*t)^(1/2))/(2*a)-c/a,IgnoreAnalyticConstraints);
        if not has(lambda1,x) and 
           ode::odeIszero(dterm0-lambda1*((c-a*lambda1)*x^n+d-b*lambda1)) and 
           iszero(expand(dterm0-lambda1*((c-a*lambda1)*x^n+d-b*lambda1),optIgnoreAnalyticConstraints)) then 
          y0:= exp(-lambda1*x);
          y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        elif not has(lambda2,x) and 
           ode::odeIszero(dterm0-lambda2*((c-a*lambda2)*x^n+d-b*lambda2)) and 
           iszero(expand(dterm0-lambda2*((c-a*lambda2)*x^n+d-b*lambda2),optIgnoreAnalyticConstraints)) then 
          y0:= exp(-lambda2*x);
          y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        elif not has(lambda3,x) and 
           ode::odeIszero(dterm0-lambda3*((c-a*lambda3)*x^n+d-b*lambda3)) and 
           iszero(expand(dterm0-lambda3*((c-a*lambda3)*x^n+d-b*lambda3),optIgnoreAnalyticConstraints)) then 
          y0:= exp(-lambda3*x);
          y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        elif not has(lambda4,x) and 
           ode::odeIszero(dterm0-lambda4*((c-a*lambda4)*x^n+d-b*lambda4)) and  
           iszero(expand(dterm0-lambda4*((c-a*lambda4)*x^n+d-b*lambda4),optIgnoreAnalyticConstraints)) then 
          y0:= exp(-lambda4*x);
          y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;      
      end_if;       
    end_if;

    if iszero(dterm1) and 
       not has((a:= diff(subs(dterm2,x^n=z),z)),x) and 
       not has(a,z) and traperror(subs(dterm2,[x^n=0,x=0],EvalChanges)) = 0 then
      b:= diff(subs(dterm2,x^n=z),x);
      c:= subs(dterm2,[x^n=0,x=0]);
      if ode::odeIszero(diff(b,x)) and ode::odeIszero(diff(c,x)) and ode::odeIszero(dterm0 + (a*n*(n-1)*x^(n-2))) and 
         iszero(expand(diff(b,x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(diff(c,x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0 + (a*n*(n-1)*x^(n-2)),optIgnoreAnalyticConstraints)) then 
          y0:= dterm2;
          if not iszero(y0) then 
            y1:= y0*Int(1/y0^2,x,intOptions);
            return({y0,y1});
          end_if;                         
      end_if;
    end_if;

    if ode::odeIszero(diff(ode::normal(dterm2/(x^n+x)),x)) and 
       iszero(expand(diff(ode::normal(dterm2/(x^n+x)),x),optIgnoreAnalyticConstraints)) then 
      lcf:= diff(subs(dterm2, x^n=z),z);
      if ode::odeIszero(diff(lcf,x)) and 
         iszero(expand(diff(lcf,x),optIgnoreAnalyticConstraints)) and 
         not iszero(lcf) and 
         lcf <> 1 then 
        dterm0_new:= dterm0/lcf;
        dterm1_new:= dterm1/lcf;
        dterm2_new:= dterm2/lcf;
      else 
        dterm0_new:= dterm0;
        dterm1_new:= dterm1;
        dterm2_new:= dterm2;
      end_if;
      if ode::odeIszero(dterm2_new-(x^n+x)) and 
         iszero(expand(dterm2_new-(x^n+x),optIgnoreAnalyticConstraints)) and 
         traperror(subs(dterm1_new,x^(n-1)=0,EvalChanges)) = 0 then 
        n_new:= n-1;
        a:= subs(dterm1_new,x^n_new=0)+n_new;
        b:= -diff(subs(dterm1_new,x^n_new=z),z)+a;
        if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and  
           ode::odeIszero(dterm0_new - b*(1-a)*x^(n_new-1)) and 
           iszero(expand(dterm0_new - b*(1-a)*x^(n_new-1),optIgnoreAnalyticConstraints)) then
          y0:= (x^n_new+1)^(b/n_new);
          if not iszero(y0) then 
            y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;                         
        end_if;
      end_if;
    end_if;

    if not has((lcf:= diff(subs(dterm2,x^n=z),z)),x) 
       and not iszero(lcf) then 
      dterm0_new:= dterm0/lcf;
      dterm1_new:= dterm1/lcf;
      dterm2_new:= dterm2/lcf;
      n_new:= (n-1)/2;
      a:= diff(subs(dterm2_new,x^n=z),x);
      if not iszero(n_new) and not has(a,x) and not has(a,z) and 
         ode::odeIszero(dterm2_new-x^(2*n_new+1)-a*x) and 
         ode::odeIszero(dterm1_new-(x^(2*n_new)+a-a*n_new)) and 
         iszero(expand(dterm2_new-x^(2*n_new+1)-a*x,optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm1_new-(x^(2*n_new)+a-a*n_new),optIgnoreAnalyticConstraints)) and 
         traperror(subs(dterm0_new,x^(2*n_new-1)=1,EvalChanges)) = 0 then
        b:= combine((-subs(dterm0_new,x^(2*n_new-1)=1))^(1/2),IgnoreAnalyticConstraints);
        for bb in [b,-b] do 
          if not has(b,x) and not has(b,z) and 
             ode::odeIszero(dterm0_new+bb^2*x^(2*n_new-1)) and 
             iszero(expand(dterm0_new+bb^2*x^(2*n_new-1),optIgnoreAnalyticConstraints)) and 
             not iszero(bb) and 
             ode::odeIszero(diff(bb,x)) and 
             iszero(expand(diff(bb,x),optIgnoreAnalyticConstraints)) then 
            return({(x^n_new+sqrt(x^(2*n_new)+a))^(bb/n_new),
                    (x^n_new+sqrt(x^(2*n_new)+a))^(-bb/n_new)})  
          end_if;
        end_for;  
      end_if;
    end_if;
  end_if;

  if nops(xpow) = 2 then 
    if contains(xpow,2) > 0 then 
      n:= op({op(xpow)} minus {2})-2;
      a:= diff(subs(dterm2,x^(n+2)=z),z);
      if not iszero(n) and 
         not iszero(a) and 
         not has(a,x) and not has(a,z) and 
         ode::odeIszero(dterm2-a*x^(n+2)+x^2) and 
         iszero(expand(dterm2-a*x^(n+2)+x^2,optIgnoreAnalyticConstraints)) then 
        p:= diff(subs(dterm1,x^(n+1)=z),z)/a;
        q:= diff(subs(dterm1,x^(n+1)=z),x);
        if ode::odeIszero(diff(p,x)) and 
           iszero(expand(diff(p,x),optIgnoreAnalyticConstraints)) and not has(p,z) and 
           ode::odeIszero(diff(q,x)) and 
           iszero(expand(diff(q,x),optIgnoreAnalyticConstraints)) and not has(q,z) and 
           ode::odeIszero(dterm1-(a*p*x^(n+1)+x*q)) and 
           iszero(expand(dterm1-(a*p*x^(n+1)+x*q),optIgnoreAnalyticConstraints)) and 
           traperror(subs(dterm0,x^n=0,EvalChanges)) = 0 then 
          r:= diff(subs(dterm0,x^n=z),z)/a;
          s:= subs(dterm0,x^n=0);
          if ode::odeIszero(diff(r,x)) and ode::odeIszero(diff(s,x)) and ode::odeIszero(dterm0-(a*r*x^n+s)) and 
             iszero(expand(diff(r,x),optIgnoreAnalyticConstraints)) and 
             iszero(expand(diff(s,x),optIgnoreAnalyticConstraints)) and 
             iszero(expand(dterm0-(a*r*x^n+s),optIgnoreAnalyticConstraints)) then  
            A:= ode::solveWithoutProperties(x^2-(q+1)*x-s,x,IgnoreSpecialCases,IgnoreProperties);
            B:= ode::solveWithoutProperties(x^2-(p-1)*x+r,x,IgnoreSpecialCases,IgnoreProperties);
            if nops(A) = 2 and nops(B) = 2 then 
              c:= A[1];
              aa:= (A[1]+B[1])/n;
              bb:= (A[1]+B[2])/n;
              cc:= (A[1]-A[2])/n+1;
              if not has(aa,x) and not has(aa,z) and 
                 not has(bb,x) and not has(bb,z) and 
                 not has(cc,x) and not has(cc,z) then  
                res:= ode::PZ_Sec_212_5_eq_169_170_p_233_234(
                        z*(z-1)*diff(u(z),z,z)+((aa+bb+1)*z-cc)*diff(u(z),z)+aa*bb*u(z),  
                        u, z, solveOptions, odeOptions);
                if res = FAIL then 
                  res:= ode::PZ_Sec_212_5_eq_hypergeom(
                          z*(z-1)*diff(u(z),z,z)+((aa+bb+1)*z-cc)*diff(u(z),z)+aa*bb*u(z),  
                          u, z, solveOptions, odeOptions);
                end_if;          
              end_if;          
              if res <> FAIL then 
                if has(res,int) then 
                  res:= map(res, intlib::changevar, z=a*x^n, x);
                end_if;
                res:= map(subs(res,z=a*x^n), elem -> elem*x^c);
                return(res):
              end_if;
            end_if;
          end_if;
        end_if;
      end_if;
    end_if;

    if ((t:= iszero(expand(xpow[1]-2*xpow[2]/*,optIgnoreAnalyticConstraints*/)))) or  
       iszero(expand(xpow[2]-2*xpow[1],optIgnoreAnalyticConstraints)) then 
      if t = TRUE then 
        n:= xpow[2];
      else 
        n:= xpow[1];
      end_if;

      if not has((lcf:= diff(subs(dterm2,x^(2*n)=z),z)),x) 
         and not iszero(lcf) and 
         traperror(subs(dterm2/lcf,[x^(2*n)=0,x^n=0],EvalChanges)) = 0 then 
        dterm0_new:= dterm0/lcf;
        dterm1_new:= dterm1/lcf;
        dterm2_new:= dterm2/lcf;

        aa:= combine((subs(dterm2_new,[x^(2*n)=0,x^n=0]))^(1/2),IgnoreAnalyticConstraints);
        for a in [aa,-aa] do 
          if iszero(dterm1_new) and not has(a,x) then 
            if ode::odeIszero(diff(a,x)) and ode::odeIszero(dterm2_new-(x^(2*n)+2*a*x^n+a^2)) and 
               iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
               iszero(expand(dterm2_new-(x^(2*n)+2*a*x^n+a^2),optIgnoreAnalyticConstraints)) then
              t:= diff(subs(dterm0_new,x^(2*n-2)=z),z);
              b:= 1/2+simplify(sqrt(1/4-t),IgnoreAnalyticConstraints);
              if not has(b,x) and not has(b,z) and 
                 ode::odeIszero(dterm0_new+b*x^(n-2)*((b-1)*x^n+a*(n-1))) and 
                 iszero(expand(dterm0_new+b*x^(n-2)*((b-1)*x^n+a*(n-1)),optIgnoreAnalyticConstraints)) then
                y0:= (x^n+a)^(b/n);
                if not iszero(y0) then 
                  y1:= y0*Int(1/y0^2,x,intOptions);
                  return({y0,y1});
                end_if;                         
              end_if;
              b:= 1/2-simplify(sqrt(1/4-t),IgnoreAnalyticConstraints);
              if not has(b,x) and not has(b,z) and 
                 ode::odeIszero(dterm0_new+b*x^(n-2)*((b-1)*x^n+a*(n-1))) and 
                 iszero(expand(dterm0_new+b*x^(n-2)*((b-1)*x^n+a*(n-1)),optIgnoreAnalyticConstraints)) then
                y0:= (x^n+a)^(b/n);
                if not iszero(y0) then 
                  y1:= y0*Int(1/y0^2,x,intOptions);
                  return({y0,y1});
                end_if;                         
              end_if;
            end_if;
          end_if;
        end_for;

        aa:= combine((subs(dterm2_new,[x^(2*n)=0,x^n=0]))^(1/2),IgnoreAnalyticConstraints);
        if not iszero(dterm1_new) and not has(aa,x) then 
          xpow1:= {};
          misc::maprec(dterm1_new,
                       {"_power"} = proc(elem)
                                      begin
                                        if op(elem,1) = x then 
                                          xpow1:= xpow1 union {op(elem,2)}
                                        end_if;
                                        elem;
                                        end_proc);  
          xpow1:= [op(xpow1)];
          if nops(xpow1) = 2 then 
            if ode::odeIszero(xpow1[1]-n-xpow1[2]) and 
               iszero(expand(xpow1[1]-n-xpow1[2],optIgnoreAnalyticConstraints)) then 
              m:= xpow1[2];
            elif ode::odeIszero(xpow1[2]-n-xpow1[1]) and 
                 iszero(expand(xpow1[2]-n-xpow1[1],optIgnoreAnalyticConstraints)) then 
              m:= xpow1[1];
            else 
              m:= FAIL;
            end_if;
            if m <> FAIL then 
              b:= diff(subs(dterm1_new,x^(m+n)=z),z);
              if not has(b,x) and not has(b,z) then 
                for a in [aa,-aa] do 
                  if ode::odeIszero(dterm1_new-b*x^m*(x^n+a)) and 
                     ode::odeIszero(dterm0_new+x^(n-2)*(b*x^(m+1)+a*n-a)) and 
                     iszero(expand(dterm1_new-b*x^m*(x^n+a),optIgnoreAnalyticConstraints)) and 
                     iszero(expand(dterm0_new+x^(n-2)*(b*x^(m+1)+a*n-a),optIgnoreAnalyticConstraints)) then
                    y0:= (x^n+a)^(1/n);
                    if not iszero(y0) then 
                      y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
                      return({y0,y1});
                    end_if;                         
                  end_if;
                end_for;
              end_if;  
            end_if;  
          end_if;
        end_if;
        
      end_if;

      aa:= simplify((diff(subs(dterm2,x^(2*n)=z),z))^(1/2),IgnoreAnalyticConstraints);
      bb:= simplify((subs(dterm2,[x^(2*n)=0,x^n=0]))^(1/2),IgnoreAnalyticConstraints);
      if not has(aa,x) and not has(aa,z) and not has(bb,x) and not has(bb,z) then 
        for a in [aa,-aa] do
          for b in [bb,-bb] do 
            if not iszero(a) and 
               ode::odeIszero(dterm2-(a*x^n+b)^2) and 
               iszero(expand(dterm2-(a*x^n+b)^2,optIgnoreAnalyticConstraints)) and 
               traperror(subs((t:= combine(expand(ode::normal(dterm1/(a*x^n+b))),IgnoreAnalyticConstraints)),[x^n=0])) = 0 then 
              c:= diff(subs(t,x^n=z),z);
              d:= subs(t,[x^n=0]);
              if not has(c,x) and not has(c,z) and not has(d,x) and not has(d,z) and
                 ode::odeIszero(dterm0-n*(b*c-a*d)*x^(n-1)) and 
                 iszero(expand(dterm0-n*(b*c-a*d)*x^(n-1),optIgnoreAnalyticConstraints)) then 
                y0:= exp(-Int((c*x^n+d)/(a*x^n+b),x,intOptions));
                if not iszero(y0) then 
                  y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
                  return({y0,y1});
                end_if;                         
              end_if;
            end_if;
          end_for;
        end_for;
      end_if;

      if traperror(subs(dterm2,[x^(2*n)=0,x^n=0],EvalChanges)) = 0 then 
        aa:= simplify((diff(subs(dterm2,x^(2*n)=z),z))^(1/2),IgnoreAnalyticConstraints);
        bb:= simplify((subs(dterm2,[x^(2*n)=0,x^n=0]))^(1/2),IgnoreAnalyticConstraints);
        if not has(aa,x) and not has(aa,z) and not has(bb,x) and not has(bb,z) then 
          for a in [aa,-aa] do
            for b in [bb,-bb] do 
              t:= simplify(combine(expand(ode::normal(dterm1/(a*x^n+b))),IgnoreAnalyticConstraints),IgnoreAnalyticConstraints);
              xpow1:= {};
              misc::maprec(t,
                           {"_power"} = proc(elem)
                                          begin
                                            if op(elem,1) = x then 
                                              xpow1:= xpow1 union {op(elem,2)}
                                            end_if;
                                            elem;
                                          end_proc);  
              xpow1:= [op(xpow1)];
              if nops(xpow1) = 1 then 
                m:= xpow1[1];
              elif nops(xpow1) = 0 and ode::odeIszero(diff(t,x,x)) and 
                   iszero(expand(diff(t,x,x),optIgnoreAnalyticConstraints)) then 
                xpow1:= [1];
                m:= 1;
              end_if;
              if nops(xpow1) = 1 and traperror((c:= subs(t,x^m=1,EvalChanges))) = 0 then               
                if not iszero(c) and 
                   ode::odeIszero(dterm1-c*x^m*(a*x^n+b)) and 
                   ode::odeIszero(dterm0-(c*x^m-a*n*x^(n-1)-1)) and 
                   iszero(expand(dterm1-c*x^m*(a*x^n+b),optIgnoreAnalyticConstraints)) and 
                   iszero(expand(dterm0-(c*x^m-a*n*x^(n-1)-1),optIgnoreAnalyticConstraints))
                 then
                  y0:= exp(-Int(1/(a*x^n+b),x,intOptions));
                  if not iszero(y0) then 
                    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
                    return({y0,y1});
                  end_if;                         
                end_if;
              end_if;
            end_for;
          end_for;
        end_if;
      end_if;
    end_if;
  end_if;
  
  if nops(xpow) = 3 then 
    if (i:= contains(xpow,2)) > 0 and 
       ode::odeIszero(diff(dterm0,x)) and 
       iszero(expand(diff(dterm0,x),optIgnoreAnalyticConstraints)) then 
      c:= dterm0;
      delete xpow[i]; 
      if ode::odeIszero(diff(dterm0,x)) and 
         iszero(expand(diff(dterm0,x),optIgnoreAnalyticConstraints)) then 
        t:= TRUE;
        if ode::odeIszero(xpow[1]-2*xpow[2]+2) and 
           iszero(expand(xpow[1]-2*xpow[2]+2,optIgnoreAnalyticConstraints)) then 
          n:= xpow[2]-2;
        elif ode::odeIszero(xpow[2]-2*xpow[1]+2) and 
             iszero(expand(xpow[2]-2*xpow[1]+2,optIgnoreAnalyticConstraints)) then
          n:= xpow[1]-2;
        else 
          t:= FALSE;
        end_if;
        if t = TRUE and 
           traperror(subs(dterm2,[x^(2*n+2)=0,x^(n+2)=0,x^2=1],EvalChanges)) = 0 then 
          aa:= simplify((diff(subs(dterm2,x^(2*n+2)=z),z))^(1/2),IgnoreAnalyticConstraints);
          bb:= simplify((subs(dterm2,[x^(2*n+2)=0,x^(n+2)=0,x^2=1]))^(1/2),IgnoreAnalyticConstraints);
          if not has(aa,x) and not has(aa,z) and not has(bb,x) and not has(bb,z) then 
            for a in [aa,-aa] do
              for b in [bb,-bb] do 
                if not iszero(a) and 
                   not iszero(b) and 
                   not iszero(n) and 
                   ode::odeIszero(dterm2-(x^2*(a*x^n+b)^2)) and 
                   ode::odeIszero(dterm1-((n+1)*x*(a^2*x^(2*n)-b^2))) and 
                   iszero(expand(dterm2-(x^2*(a*x^n+b)^2),optIgnoreAnalyticConstraints)) and 
                   iszero(expand(dterm1-((n+1)*x*(a^2*x^(2*n)-b^2)),optIgnoreAnalyticConstraints)) then 
                  res:= solve(ode(diff(y(z),z,z)-b*(n+2)*diff(y(z),z)+c*y(z),y(z)));
                  csts:= [op(freeIndets(res) minus (indets(eq) union {z}))];
                  if nops(csts) = 2 then 
                    res:= subs(res,csts[1]=1,csts[2]=0) union subs(res,csts[1]=0,csts[2]=1);
                    if has(res,int) then 
                      res:= map(res, intlib::changevar, 
                                z = 1/(n*b)*ln((a*x^n)/(a*x^n+b)), x);
                    end_if;
                    return(subs(res,z = 1/(n*b)*ln((a*x^n)/(a*x^n+b))))  
                  end_if;
                end_if;
              end_for;
            end_for;    
          end_if;
        end_if;
      end_if;
    end_if;
  end_if;

  if nops(xpow) = 1 then 
    n:= xpow[1];
    t:= subs(dterm2,x^n=z);
    if not iszero(expand(diff(t,x),optIgnoreAnalyticConstraints)) and 
       ode::odeIszero(diff(t,x,x)) and 
       iszero(expand(diff(t,x,x),optIgnoreAnalyticConstraints)) then 
      xpow:= xpow.[1];
    end_if;
  end_if;

  if nops(xpow) = 2 and 
     (((t:= iszero(expand(xpow[1]-xpow[2]-1/*,optIgnoreAnalyticConstraints*/)))) or 
      iszero(expand(xpow[2]-xpow[1]-1,optIgnoreAnalyticConstraints))) then
    if t = TRUE then 
      n:= xpow[2];
    else 
      n:= xpow[1];
    end_if;
    if traperror(subs(dterm2,[x^(n+1)=0,x^n=z],EvalChanges)) = 0 and 
       traperror(subs(dterm2,[x^(n+1)=0,x^n=0],EvalChanges)) = 0 then 
      a:= diff(subs(dterm2,x^(n+1)=z),z);
      b:= diff(subs(dterm2,[x^(n+1)=0,x^n=z]),z);
      c:= subs(dterm2,[x^(n+1)=0,x^n=0]);
      if not has(a,x) and not has(a,z) and 
         not has(b,x) and not has(b,z) and 
         not has(c,x) and not has(c,z) and 
         ode::odeIszero(dterm2-(a*x^(n+1)+b*x^n+c)) and 
         iszero(expand(dterm2-(a*x^(n+1)+b*x^n+c),optIgnoreAnalyticConstraints)) and 
         traperror(subs(dterm1,[x^n=0,x^(n-1)=z],EvalChanges)) = 0 and 
         traperror(subs(dterm1,[x^n=0,x^(n-1)=0],EvalChanges)) = 0 then 
        aa:= diff(subs(dterm1,x^n=z),z);
        bb:= diff(subs(dterm1,[x^n=0,x^(n-1)=z]),z);
        cc:= subs(dterm1,[x^n=0,x^(n-1)=0]);
        if not has(aa,x) and not has(aa,z) and 
           not has(bb,x) and not has(bb,z) and 
           not has(cc,x) and not has(cc,z) and 
           ode::odeIszero(dterm1-(aa*x^n+bb*x^(n-1)+cc)) and 
           ode::odeIszero(dterm0-(n*(aa-a-a*n)*x^(n-1)+(n-1)*(bb-b*n)*x^(n-2))) and 
           iszero(expand(dterm1-(aa*x^n+bb*x^(n-1)+cc),optIgnoreAnalyticConstraints)) and 
           iszero(expand(dterm0-(n*(aa-a-a*n)*x^(n-1)+(n-1)*(bb-b*n)*x^(n-2)),optIgnoreAnalyticConstraints)) then
          y0:= exp(Int(((a*n+a-aa)*x^n+(b*n-bb)*x^(n-1)-cc)/dterm2,x,intOptions));
          if not iszero(y0) then 
            y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;                         
        end_if;
      end_if;
    end_if;
  end_if;

  if nops(xpow) = 1 or nops(xpow) = 2 then
    t:= FALSE;
    if nops(xpow) = 2 then 
      if xpow[1] <> 1 then 
        n:= xpow[1];
        m:= xpow[2];
      else 
        n:= xpow[2];
        m:= xpow[1];
      end_if;
      a:= diff(subs(dterm2,x^n=z),z);
      if traperror((b:= diff(subs(dterm2,[x^n=0,x^m=z],EvalChanges),z))) = 0 and 
         traperror((c:= subs(dterm2,[x^n=0,x^m=0],EvalChanges))) = 0 and 
         not has(a,x) and not has(a,z) and 
         not has(b,x) and not has(b,z) and 
         not has(c,x) and not has(c,z) and 
         ode::odeIszero(dterm2-(a*x^n+b*x^m+c)) and 
         iszero(expand(dterm2-(a*x^n+b*x^m+c),optIgnoreAnalyticConstraints)) then 
        t:= TRUE;    
      end_if;
    else 
      n:= xpow[1];
      a:= diff(subs(dterm2,x^n=z),z);
      if traperror((c:= subs(dterm2,[x^n=0],EvalChanges))) = 0 and
         not has(a,x) and not has(a,z) and 
         not has(c,x) and not has(c,z) and 
         ode::odeIszero(dterm2-(a*x^n+c)) and 
         iszero(expand(dterm2-(a*x^n+c),optIgnoreAnalyticConstraints)) then 
        t:= TRUE;
      end_if;
    end_if;
    
    if t = TRUE then 
      if ode::odeIszero(diff(dterm1,x)+1) and 
         ode::odeIszero(dterm0-1) and 
         iszero(expand(diff(dterm1,x)+1,optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0-1)) then 
        y0:= -dterm1;
        if not iszero(y0) then 
          y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;                         
      end_if;
      if ode::odeIszero(diff(dterm1,x)+2*x) and 
         ode::odeIszero(diff(dterm0,x)-1) and 
         iszero(expand(diff(dterm1,x)+2*x,optIgnoreAnalyticConstraints)) and 
         iszero(expand(diff(dterm0,x)-1,optIgnoreAnalyticConstraints)) then 
        lambda:= dterm0 | x=0;
        if ode::odeIszero(dterm1-(lambda^2-x^2)) and 
           ode::odeIszero(dterm0-(x+lambda)) and 
           iszero(expand(dterm1-(lambda^2-x^2),optIgnoreAnalyticConstraints)) and 
           iszero(expand(dterm0-(x+lambda),optIgnoreAnalyticConstraints)) then 
          y0:= x-lambda;
          if not iszero(y0) then 
            y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;                         
        end_if;
      end_if;
      if ode::odeIszero(1/2*diff(dterm2,x)-dterm1) and 
         ode::odeIszero(diff(dterm0,x)) and 
         iszero(expand(1/2*diff(dterm2,x)-dterm1,optIgnoreAnalyticConstraints)) and 
         iszero(expand(diff(dterm0,x))) then 
        return({exp((2^(1/2)*(-dterm0)^(1/2)*int(1/(1/2*dterm2)^(1/2), x,intOptions))/2), 
                1/exp((2^(1/2)*(-dterm0)^(1/2)*int(1/(1/2*dterm2)^(1/2), x,intOptions))/2)})
      end_if;
    end_if;             
  end_if;

  if not has(dterm1,x) then 
    xpow1:= [];
  elif ode::odeIszero(diff(dterm1,x,x)) and 
       iszero(expand(diff(dterm1,x,x),optIgnoreAnalyticConstraints)) then 
    xpow1:= [1];
  else
    xpow1:= {};
    misc::maprec(dterm1,
                 {"_power"} = proc(elem)
                                begin
                                  if op(elem,1) = x then 
                                    xpow1:= xpow1 union {op(elem,2)}
                                  end_if;
                                  elem;
                                  end_proc);  
    xpow1:= [op(xpow1)];
  end_if;

  if nops(xpow1) = 1 then 
    n:= xpow1[1];
    a:= diff(subs(dterm1,x^n=z),z);
    if not has(a,x) and not has(a,z) and  
       traperror((b:= subs(dterm1,x^n=0,EvalChanges))) = 0 and
       not has(b,x) and not has(b,z) and 
       not iszero(a) and 
       not iszero(n) and 
       ode::odeIszero(dterm1-(a*x^n+b)) and 
       iszero(expand(dterm1-(a*x^n+b),optIgnoreAnalyticConstraints)) and  
       traperror((m:= -subs(dterm0,x^(n-1)=1,EvalChanges)/(a*n))) = 0 then
      if ode::odeIszero(expand(diff(m,x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(diff(m,x),optIgnoreAnalyticConstraints)) then 
        if ode::odeIszero(dterm0+a*n*m*x^(n-1)) and 
           ode::odeIszero(dterm2-(a*x^n+b)^(m+1)) and 
           iszero(expand(dterm0+a*n*m*x^(n-1),optIgnoreAnalyticConstraints)) and 
           iszero(expand(dterm2-(a*x^n+b)^(m+1),optIgnoreAnalyticConstraints)) then
          y0:= exp(-Int(1/(a*x^n+b)^m,x,intOptions));
          if not iszero(y0) then 
            y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;                         
        end_if;
      end_if;
    end_if;
  end_if;

  // Code for pattern 265: comment in the notation of the book stated in the 
  //                       introductory comment to this procedure
  if not iszero(dterm0) then 
    t1:= ode::normal(expand(dterm2/x/*,optIgnoreAnalyticConstraints*/)); // this gives P_n(x) 
    t2:= ode::normal(expand(dterm1-2*t1-x*dterm0/*,optIgnoreAnalyticConstraints*/)); // this gives a*x^2*Q_{n-2}(x)
    t3:= ode::normal(expand(t2/(x^2*dterm0)/*,optIgnoreAnalyticConstraints*/)); // this gives a/b
    if iszero(diff(t3,x)) then  
      y0:= t3 + 1/x; // since ODE is linear an d hom., this is equivalent to 
                     // the solution a + b/x, but since we al
      if not iszero(y0) then 
        y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;                         
    end_if;
  end_if;
  
  return(FAIL);
end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.2-2. eq. 47,183,202 p. 218)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_212_1_to_8_eq_47_183_202_p_218:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, optIgnoreAnalyticConstraints, xpow, Int, 
        dterm0_new, dterm1_new, dterm2_new, z, a, b, c, n, csts, res, t, lcf,
        lambda;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   

  Int:= if has(odeOptions,freeze) then freeze(int) else int end_if;

  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);

  z:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));

  if ode::odeIszero(diff(dterm2,x)) and 
     iszero(expand(diff(dterm2,x),optIgnoreAnalyticConstraints)) and 
     has(dterm1,x) then 
    dterm0_new:= combine(expand(ode::normal(dterm0/dterm2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);;
    dterm1_new:= combine(expand(ode::normal(dterm1/dterm2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);;
    dterm2_new:= 1;
    if ode::odeIszero(diff(dterm1_new,x,x)) and 
       iszero(expand(diff(dterm1_new,x,x),optIgnoreAnalyticConstraints)) then 
      xpow:= [1];
    else
      xpow:= {};
      misc::maprec(dterm1_new,
                   {"_power"} = proc(elem)
                                  begin
                                    if op(elem,1) = x then 
                                      xpow:= xpow union {op(elem,2)}
                                    end_if;
                                    elem;
                                    end_proc);  
      xpow:= [op(xpow)];
    end_if;
    if nops(xpow) = 1 then 
      n:= xpow[1];
      a:= diff(subs(dterm1_new,x^n=z),z);
      if not has(a,x) and not has(a,z) and 
         ode::odeIszero(dterm1_new-a*x^n) and 
         iszero(expand(dterm1_new-a*x^n,optIgnoreAnalyticConstraints)) then 
        b:= diff(subs(dterm0_new,x^(2*n)=z),z);
        c:= diff(subs(dterm0_new,x^(n-1)=z),z);
        if not has(b,x) and not has(b,z) and 
           not has(c,x) and not has(c,z) and 
           ode::odeIszero(dterm0_new-(b*x^(2*n)+c*x^(n-1))) and 
           iszero(expand(dterm0_new-(b*x^(2*n)+c*x^(n-1)),optIgnoreAnalyticConstraints)) and 
           not iszero(n+1) then 
          res:= solve(ode((n+1)^2*z*diff(y(z),z,z)+(n+1)*(a*z+n)*diff(y(z),z)+(b*z+c)*y(z),y(z)));
          csts:= [op(freeIndets(res) minus (indets(eq) union {z}))];
          if nops(csts) = 2 then 
            res:= subs(res,csts[1]=1,csts[2]=0) union subs(res,csts[1]=0,csts[2]=1);
            if traperror(evalAt(res,z = x^(n+1))) = 0 and 
               traperror((res:= map(res, intlib::changevar, z = x^(n+1), x))) = 0 then                 
                res:= evalAt(res, z = x^(n+1));
              return(res);
            end_if; 
          end_if;
        end_if;
      end_if;
    end_if;
  end_if;

  if ode::odeIszero(dterm1) and  
     iszero(expand(dterm1,optIgnoreAnalyticConstraints)) then 
    lcf:= ode::normal(dterm2/x^2);
    dterm0_new:= combine(expand(ode::normal(dterm0/lcf)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm2_new:= x^2;
    xpow:= {};
    misc::maprec(dterm0_new,
                 {"_power"} = proc(elem)
                                begin
                                  if op(elem,1) = x then 
                                    xpow:= xpow union {op(elem,2)}
                                  end_if;
                                  elem;
                                  end_proc);  
    xpow:= [op(xpow)];
    if nops(xpow) = 2 then 
      t:= FALSE;
      if iszero(expand(2*xpow[1]-3*xpow[2],optIgnoreAnalyticConstraints)) then 
        t:= TRUE;  
        n:= ode::normal(xpow[1]/3);
      elif iszero(expand(3*xpow[1]-2*xpow[2],optIgnoreAnalyticConstraints)) then    
        t:= TRUE;
        n:= ode::normal(xpow[2]/3);
      end_if;
      if t = TRUE then 
        a:= diff(subs(dterm0_new,x^(3*n)=z),z);
        b:= diff(subs(dterm0_new,x^(2*n)=z),z);
        if not has(a,x) and not has(a,z) and 
           not has(b,x) and not has(b,z) and 
           ode::odeIszero(dterm0_new-(a*x^(3*n)+b*x^(2*n)+1/4-1/4*n^2)) and 
           iszero(expand(dterm0_new-(a*x^(3*n)+b*x^(2*n)+1/4-1/4*n^2),optIgnoreAnalyticConstraints)) then  
          res:= solve(ode(diff(y(z),z,z)+1/(a*n)^2*z*y(z),y(z)));
          csts:= [op(freeIndets(res) minus (indets(eq) union {z}))];
          if nops(csts) = 2 then 
            res:= subs(res,csts[1]=1,csts[2]=0) union subs(res,csts[1]=0,csts[2]=1);
            if traperror(subs(res,z = a*x^n+b,EvalChanges)) = 0 then 
              res:= map(res, intlib::changevar, z = a*x^n+b, x);
              res:= subs(res, z = a*x^n+b);
              res:= map(res, elem -> elem*x^((1-n)/2));
              return(res);
            end_if; 
          end_if;
        end_if;
      end_if;
    end_if;
  end_if;
  
  if ode::odeIszero(diff((lcf:= ode::normal(dterm2/x^3)),x)) and 
     iszero(expand(diff((lcf:= ode::normal(dterm2/x^3)),x),optIgnoreAnalyticConstraints)) and 
     not iszero(lcf) then 
    dterm0_new:= combine(expand(ode::normal(dterm0/lcf)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm1_new:= combine(expand(ode::normal(dterm1/lcf)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm2_new:= x^3;
    if ode::odeIszero(diff(dterm1_new,x,x,x)) and 
       ode::odeIszero(diff(dterm0_new,x,x)) and 
       iszero(expand(diff(dterm1_new,x,x,x),optIgnoreAnalyticConstraints)) and 
       iszero(expand(diff(dterm0_new,x,x),optIgnoreAnalyticConstraints)) then 
      a:= diff(subs(dterm1_new,x^2=z),z);
      b:= subs(dterm1_new,x=0);
      c:= diff(dterm0_new,x);
      if not iszero(c) and 
         not iszero(b) and 
         not has(a,x) and not has(a,z) and 
         not has(b,x) and not has(b,z) and 
         not has(c,x) and not has(c,z) and 
         ode::odeIszero(dterm1_new-(a*x^2+b)) and 
         ode::odeIszero(dterm0_new-c*x) and 
         iszero(expand(dterm1_new-(a*x^2+b),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0_new-c*x,optIgnoreAnalyticConstraints)) then 
        if traperror((
          res:= {x^(1/2-(a^2-2*a-4*c+1)^(1/2)/2-a/2)*
                 kummerU(a/4+(a^2-2*a-4*c+1)^(1/2)/4-1/4, 
                        (a^2-2*a-4*c+1)^(1/2)/2+1,b/(2*x^2)),
                 x^(1/2-(a^2-2*a-4*c+1)^(1/2)/2-a/2)*
                 hypergeom([a/4+(a^2-2*a-4*c+1)^(1/2)/4-1/4], 
                           [(a^2-2*a-4*c+1)^(1/2)/2+1],b/(2*x^2))})) = 0 then   
          return(res);
        end_if;
      end_if;
    end_if;
  end_if;

  if ode::odeIszero(diff(dterm2,x,x,x,x)) and 
     ode::odeIszero(diff(dterm1,x,x,x)) and 
     ode::odeIszero(diff(dterm0,x,x,x)) and 
     iszero(expand(diff(dterm2,x,x,x,x),optIgnoreAnalyticConstraints)) and 
     iszero(expand(diff(dterm1,x,x,x),optIgnoreAnalyticConstraints)) and 
     iszero(expand(diff(dterm0,x,x,x),optIgnoreAnalyticConstraints)) then 
    a:= diff(dterm2,x,x,x)/12;
    b:= (diff(dterm2,x,x) | x=0)/4;
    c:= (diff(dterm2,x) | x=0)/2;
    if not has(a,x) and not has(a,z) and 
       not has(b,x) and not has(b,z) and 
       not has(c,x) and not has(c,z) and       
       ode::odeIszero(dterm2-(2*a*x^3+2*b*x^2+2*c*x)) and 
       ode::odeIszero(dterm1-(a*x^2-c)) and 
       iszero(expand(dterm2-(2*a*x^3+2*b*x^2+2*c*x),optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm1-(a*x^2-c),optIgnoreAnalyticConstraints)) then 
      lambda:= diff(dterm0,x,x)/2;
      if ode::odeIszero(dterm0-lambda*x^2) and 
         iszero(expand(dterm0-lambda*x^2,optIgnoreAnalyticConstraints)) then 
        res:= solve(ode(2*diff(y(z),z,z)+lambda*y(z),y(z)));
        csts:= [op(freeIndets(res) minus (indets(eq) union {z}))];
        if nops(csts) = 2 then 
          res:= subs(res,csts[1]=1,csts[2]=0) union subs(res,csts[1]=0,csts[2]=1);
          res:= map(res, intlib::changevar, z = int((x/(a*x^2+b*x+c))^(1/2),x), x);
          res:= subs(res, z = int((x/(a*x^2+b*x+c))^(1/2),x));
          return(res);
        end_if;
      end_if;
    end_if;
  end_if;

  return(FAIL);
end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.9.-1. eq. 1 to 52 p. 285 to 289)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/


ode::PZ_Sec_219_1_eq_1_to_52_p_285_to_289:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, optIgnoreAnalyticConstraints,
        Int, dterm0_new, dterm1_new, dterm2_new, a, t, z, xpow, n, L, b, c, f, lambda,
        m, exp_pow, exp_pow2, aa, bb, cc, dd, functionArgs, cos_pow, sin_pow, t1, t0,
        tan_pow, ff, cot_pow;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   

  Int:= if has(odeOptions,freeze) then freeze(int) else int end_if;

  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);

  dterm0:= combine(dterm0, exp);
  dterm1:= combine(dterm1, exp);
  dterm2:= combine(dterm2, exp);

  if iszero(expand(dterm2,optIgnoreAnalyticConstraints)) then 
    return(FAIL)
  end_if;

  z:= genident(); //solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 4
  // ---------------------------

  if not iszero(expand(dterm1_new,optIgnoreAnalyticConstraints)) and 
     not iszero(expand(dterm0_new,optIgnoreAnalyticConstraints)) and 
     not iszero(expand(diff(dterm1_new,x),optIgnoreAnalyticConstraints)) and 
     traperror((a:= ode::normal(diff(dterm0_new,x)/diff(dterm1_new,x)))) = 0 then 
    if ode::odeIszero(diff(a,x)) and 
       ode::odeIszero(dterm0_new-a*dterm1_new+a^2) and 
       iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm0_new-a*dterm1_new+a^2,optIgnoreAnalyticConstraints)) then 
      y0:= exp(-a*x);
      y1:= y0*int(exp(-int(dterm1_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1})    
    end_if;  
  end_if;

  // ---------------------------
  // look-up for eq. 5
  // ---------------------------

  t:= diff(subsex(dterm0_new, dterm1_new = z),z); // t should be a*x^n
  xpow:= {};
  misc::maprec(t,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  xpow:= [op(xpow)];
  if nops(xpow) = 1 then 
    n:= xpow[1];
    a:= diff(subsex(t,x^n=z),z);
    if n <> -1 and 
       ode::odeIszero(dterm0_new - (a*(x^n*dterm1_new-a*x^(2*n)+n*x^(n-1)))) and 
       iszero(expand(dterm0_new - (a*(x^n*dterm1_new-a*x^(2*n)+n*x^(n-1))))) then 
      y0:= exp(-a/(n+1)*x^(n+1));
      y1:= y0*int(exp(-int(dterm1_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1})    
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 7
  // ---------------------------

  /*
    COMMENT. (possible optimization and extensions strategy)

             The following look-ups using the heuristic of localizing 
             a power of the form 'x^n' within the given ODE can be 
             extended. Currently they check whether there is only a 
             single appearance of a power of the form 'x^n'. If there 
             are two or more appearances, the methods fail. Hence, a 
             possiblility for extending the methods is to replace the 
             if-statements 'if nops(xpow) = 1 then ...'  at the beginning 
             of each look-up block by a for-look of the form 
             'for n in xpow do ...'. This increases the success rate 
             drastically, but on the cost of efficiency (note that in 
             lots of the cases equations need to be solved to find a 
             possible matching). Note that the arbitrary function 'f(x)'
             from the patterns could also be a polynomial of arbitrary  
             degree. If this polynomial has many non-zero coefficients, 
             it is not a good idea to use a for-loop iterating over 
             all elements of 'xpow'. Maybe it's worth thinking about 
             restrictions of the form 'if nops(xpow) < 4 then for 
             n in xpow do ...'.              
  */
  
  xpow:= {};
  misc::maprec(dterm1_new,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  xpow:= [op(xpow)]; 
  if nops(xpow) = 1 then 
    n:= xpow[1];
    if n <> -1 then 
      a:= diff(subs(dterm1_new,x^n=z),z);
      if iszero(diff(a,x)) and iszero(diff(a,z)) then 
        L:= ode::solveWithoutProperties(z^2 - dterm1_new*z + dterm0_new-n*a*x^(n-1), z, 
                  IgnoreSpecialCases, optIgnoreAnalyticConstraints);
        L:= select(map(L, elem -> ode::normal(elem-a*x^n)),_not@has, x);
        for b in L do
          f:= dterm1_new-a*x^n-b;
          if ode::odeIszero(dterm0_new-((a*x^n+b)*f+a*n*x^(n-1))) and  
             iszero(expand(dterm0_new-((a*x^n+b)*f+a*n*x^(n-1)),optIgnoreAnalyticConstraints)) then 
            y0:= exp(-a/(n+1)*x^(n+1)-b*x);
            y1:= y0*int(exp(-int(dterm1_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;
        end_for;
      end_if;
    end_if;
  end_if;

  // =================================================
  // different ode::normalization using original input data
  // =================================================

  if dterm2 <> x then 
    dterm2_new:= x;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    
  delete a,b,f:

  // ---------------------------
  // look-up for eq. 8
  // ---------------------------

  f:= ode::normal(dterm1_new/x);
  L:= ode::solveWithoutProperties(z^2 + (2+dterm1_new)*z + (x*dterm0_new+dterm1_new) = 0, z, 
            IgnoreSpecialCases, optIgnoreAnalyticConstraints);
  L:= select(map(L, elem -> ode::normal(elem/x)),_not@has, x);
  for a in L do 
    if ode::odeIszero(dterm0_new+((a*x+1)*f+a*(a*x+2))) and 
       iszero(expand(ode::normal(dterm0_new+((a*x+1)*f+a*(a*x+2))),optIgnoreAnalyticConstraints)) then 
      y0:= x*exp(a*x);
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_for;  

  // ---------------------------
  // look-up for eq. 9
  // ---------------------------

  if traperror((a:= (dterm1_new | x=0))) = 0 and 
     a <> 1 then 
    f:= ode::normal(dterm0_new/(a-1));
    if ode::odeIszero(dterm1_new-(x*f+a)) and 
       ode::odeIszero(dterm0_new-(a-1)*f) and 
       iszero(expand(ode::normal(dterm1_new-(x*f+a)),optIgnoreAnalyticConstraints)) and 
       iszero(expand(ode::normal(dterm0_new-(a-1)*f),optIgnoreAnalyticConstraints)) then 
      y0:= x^(1-a);
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 10
  // ---------------------------

  L:= ode::solveWithoutProperties(x^2*z^3 + (-x*(dterm1_new+1))*z^2 + x*dterm0_new*z + dterm0_new = 0, z, 
            IgnoreSpecialCases, MaxDegree = 3, optIgnoreAnalyticConstraints);  
  L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
  for a in L do 
    if not iszero(a) then 
      f:= ode::normal(dterm0_new/(a^2*x));
      if ode::odeIszero(dterm1_new-((a*x+1)*f+a*x-1)) and 
         iszero(expand(ode::normal(dterm1_new-((a*x+1)*f+a*x-1)),optIgnoreAnalyticConstraints)) then 
        y0:= (a*x+1)*exp(-a*x);
        y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
  end_for;

  // ---------------------------
  // look-up for eq. 12
  // ---------------------------

  xpow:= {};
  misc::maprec(dterm1_new,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  xpow:= [op(xpow)];
  if nops(xpow) = 1 then 
    n:= xpow[1]-1;
    L:= ode::solveWithoutProperties(x^(n+1)*z^2 - (dterm1_new+n)*z + dterm0_new/x^n, z,
              IgnoreSpecialCases, optIgnoreAnalyticConstraints);  
    L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
    for a in L do 
      f:= dterm1_new - a*x^(n+1);
      if ode::odeIszero(dterm0_new-a*x^n*(f+n)) and 
         iszero(expand(ode::normal(dterm0_new-a*x^n*(f+n)),optIgnoreAnalyticConstraints)) then 
        y0:= exp(-a/(n+1)*x^(n+1));
        y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_for;
  end_if;

  // ---------------------------
  // look-up for eq. 13
  // ---------------------------
  xpow:= {};
  misc::maprec(dterm1_new,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  xpow:= [op(xpow)];
  if nops(xpow) = 1 then 
    n:= xpow[1];
    a:= diff(subs(dterm1_new,x^n=z),z);
    if iszero(diff(a,x)) and iszero(diff(a,z)) then 
      f:= (ode::normal(dterm1_new-a*x^n))/x;
      if ode::odeIszero(dterm0_new-(a*x^n-1)*f-a*n*x^(n-1)) and 
         iszero(expand(ode::normal(dterm0_new-(a*x^n-1)*f-a*n*x^(n-1)),optIgnoreAnalyticConstraints)) then 
        y0:= x*exp(-a*x^n/n);
        y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 14
  // ---------------------------

  xpow:= {};
  misc::maprec(dterm1_new,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  xpow:= [op(xpow)];
  if nops(xpow) = 1 then 
    n:= xpow[1];
    L:= ode::solveWithoutProperties(n^2*x^n*z^3-n*(dterm1_new+2*n-1)*z^2+x^(1-n)*dterm0_new*z+x^(1-2*n)*dterm0_new, z,
              IgnoreSpecialCases, MaxDegree = 3, optIgnoreAnalyticConstraints);
    L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
    for a in L do
      if not iszero(a) then  
        f:= ode::normal(dterm0_new/(a^2*n*x^(2*n-1)));    
        if ode::odeIszero(dterm1_new-((a*x^n+1)*f+a*n*x^n+1-2*n)) and 
           iszero(expand(ode::normal(dterm1_new-((a*x^n+1)*f+a*n*x^n+1-2*n)),
                         optIgnoreAnalyticConstraints)) then 
          y0:= (a*x^n+1)*exp(-a*x^n);
          y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      end_if;
    end_for;
  end_if;

  // =================================================
  // different ode::normalization using original input data
  // =================================================
  if dterm2 <> x^2 then 
    dterm2_new:= x^2;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    
  delete a,b,f:

  // ---------------------------
  // look-up for eq. 18
  // ---------------------------

  // Note. Heuristic only works if 'f' has constant term equal to '0'. 
  //       Otherwise no idea how to match the pattern (two non-linear 
  //       equations for three quantities 'a', 'b' and 'f(x)' do not 
  //       give unique access to the quantities.  

  if traperror((a:= ode::normal((diff(dterm1_new,x)/2) | x = 0))) = 0 then 
    f:= ode::normal(dterm1_new/x) - 2*a;
    L:= ode::solveWithoutProperties((z*x+a)*f-z^2*x^2 +a*(a-1) = dterm0_new, z, 
              IgnoreSpecialCases, optIgnoreAnalyticConstraints);
    L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
    for b in L do 
      y0:= x^(-a)*exp(-b*x);
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_for;
  elif type(dterm1_new) = "_plus" then  
    if ((t:= select({op(ode::normal(dterm1_new/(2*x)))},_not@has,x))) = {} then 
      a:= 0;
    else 
      a:= _plus(op(t));
    end_if;      
    f:= ode::normal(dterm1_new/x) - 2*a;
    L:= ode::solveWithoutProperties((z*x+a)*f-z^2*x^2 +a*(a-1) = dterm0_new, z, 
              IgnoreSpecialCases, optIgnoreAnalyticConstraints);
    L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
    for b in L do 
      y0:= x^(-a)*exp(-b*x);
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_for;
  end_if;
  
  // ---------------------------
  // look-up for eq. 19
  // ---------------------------

  // Note. Heuristic only works if 'f' has no polynomial part. 
  //       Otherwise no idea how to match the pattern. 

  f:= ode::normal(dterm1_new/x);
  xpow:= {};
  misc::maprec(dterm0_new,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  xpow:= [op(xpow)];
  if nops(xpow) = 2 then 
    if xpow[1] = 2*xpow[2] then 
      n:= ode::normal((xpow[2]-1)/2);
    elif xpow[2] = 2*xpow[1] then 
      n:= ode::normal((xpow[1]-1)/2);
    else
      n:= 0
    end_if;
    if n <> 0 then
      L:= ode::solveWithoutProperties((z*x^(2*n+1)+n)*f-z^2*x^(4*n+2)-n^2-n = dterm0_new, z,
                IgnoreSpecialCases/*, optIgnoreAnalyticConstraints*/);
      L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
      for a in L do 
        if not iszero(2*n+1) and 
           ode::odeIszero(dterm0_new-((a*x^(2*n+1)+n)*f-a^2*x^(4*n+2)-n^2-n)) and
           iszero(expand(dterm0_new-((a*x^(2*n+1)+n)*f-a^2*x^(4*n+2)-n^2-n),
                         optIgnoreAnalyticConstraints)) then 
          y0:= x^(-n)*exp(-a/(2*n+1)*x^(2*n+1));
          y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      end_for;
    end_if;   
  end_if;

  // ---------------------------
  // look-up for eq. 20
  // ---------------------------

  if not iszero(dterm0) then 
    f:= ode::normal(-dterm1/dterm0);
    if has(f,x) and 
       ode::odeIszero(diff(f,x,x)) and ode::odeIszero(dterm1*diff(f,x) + dterm0*f) and 
       iszero(expand(diff(f,x,x),optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm1*diff(f,x) + dterm0*f,optIgnoreAnalyticConstraints)) then 
      y0:= f;
      y1:= y0*int(exp(-int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;

  // =================================================
  // different ode::normalization using original input data
  // =================================================

  if dterm2 <> x^4 then 
    dterm2_new:= x^4;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 21
  // ---------------------------

  f:= ode::normal(dterm1_new/x^2);
  L:= ode::solveWithoutProperties((z-x)*f-z^2 = dterm0_new, z,
            IgnoreSpecialCases, optIgnoreAnalyticConstraints);
  L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
  for lambda in L do 
    y0:= x*exp(lambda/x);
    y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_for;  

  // ---------------------------
  // look-up for eq. 22
  // ---------------------------

  if has(dterm2,x^4) and ode::odeIszero(diff(dterm2,x,x,x,x,x)) and 
     iszero(expand(diff(dterm2,x,x,x,x,x),optIgnoreAnalyticConstraints)) then 
    a:= diff(dterm2,x,x,x,x)/24;
    b:= ((diff(dterm2,x,x) | x=0))/2; 
    if not iszero(a*x^2+b) then 
      if traperror((f:= ode::normal(dterm1/(x*(a*x^2+b))))) = 0 then 
        if ode::odeIszero(dterm0+((a*x^2-b)*f+2*b)) and 
           iszero(expand(dterm0+((a*x^2-b)*f+2*b),optIgnoreAnalyticConstraints)) then 
          y0:= a*x+b/x;
          y1:= y0*int(exp(-int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      end_if;  
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 23 & 24
  // ---------------------------


  if has(dterm2,x^4) and 
     iszero(expand(diff(dterm2,x,x,x,x,x),optIgnoreAnalyticConstraints)) and 
     ode::odeIszero(diff(dterm2,x,x,x,x,x)) then 
    a:= ((diff(dterm2,x,x) | x=0))/4; 
    if ode::odeIszero(dterm2-(x^2+a)^2) and 
       iszero(expand(dterm2-(x^2+a)^2,optIgnoreAnalyticConstraints)) then
      f:= ode::normal(dterm1/(x^2+a));
      // look-up for eq. 23 
      if ode::odeIszero(dterm0+(x*f+a)) and 
         iszero(expand(dterm0+(x*f+a),optIgnoreAnalyticConstraints)) then
        y0:= sqrt(x^2+a);
        y1:= y0*int(exp(-int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
      // look-up for eq. 23 & 24
      if not iszero(a) and traperror((m:= ode::normal((dterm0 | x=0)/(-a)))) = 0 then 
        if not has(m,x) and 
           ode::odeIszero(dterm0+m*(x*f+(m-1)*x^2+a)) and 
           iszero(expand(dterm0+m*(x*f+(m-1)*x^2+a),optIgnoreAnalyticConstraints)) then 
          y0:= (x^2+a)^(m/2);
          y1:= y0*int(exp(-int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 25 & 26
  // ---------------------------

  xpow:= {};
  misc::maprec(dterm2,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  xpow:= [op(xpow)];
  if nops(xpow) = 1 then 
    n:= xpow[1];
    a:= diff(subs(dterm2,x^n=z),z);
    b:= dterm2 - a*x^n;
    // look-up for eq. 25 
    if not has(a,x) and not has(b,x) then 
      f:= ode::normal(dterm1/dterm2);
      if ode::odeIszero(dterm2-(a*x^n+b)) and 
         ode::odeIszero(dterm1-(a*x^n+b)*f) and 
         ode::odeIszero(dterm0+a*n*x^(n-2)*(x*f+n-1)) and
         iszero(expand(dterm2-(a*x^n+b),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm1-(a*x^n+b)*f,optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0+a*n*x^(n-2)*(x*f+n-1),optIgnoreAnalyticConstraints)) then 
        y0:= a*x^n+b;
        y1:= y0*int(exp(-int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
    // look-up for eq. 26
    if not has(a,x) and ode::odeIszero(diff(b,x,x)) and 
       iszero(expand(diff(b,x,x),optIgnoreAnalyticConstraints)) then 
      b:= diff(b,x);
      f:= ode::normal(dterm1/dterm2);
      if ode::odeIszero(dterm2-(a*x^n+b*x)) and 
         ode::odeIszero(dterm1-(a*x^n+b*x)*f) and 
         ode::odeIszero(dterm0+((a*n*x^(n-1)+b)*f+a*n*(n-1)*x^(n-2))) and 
         iszero(expand(dterm2-(a*x^n+b*x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm1-(a*x^n+b*x)*f,optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0+((a*n*x^(n-1)+b)*f+a*n*(n-1)*x^(n-2)),optIgnoreAnalyticConstraints)) then 
        y0:= a*x^n+b*x;
        y1:= y0*int(exp(-int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 27 & 28
  // ---------------------------

  xpow:= {};
  misc::maprec(dterm2,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  xpow:= [op(xpow)];

  if nops(xpow) = 2 then 
    if iszero(expand(xpow[1]-2*xpow[2],optIgnoreAnalyticConstraints)) then 
      n:= xpow[2];
    elif iszero(expand(xpow[2]-2*xpow[1],optIgnoreAnalyticConstraints)) then 
      n:= xpow[1];
    else
      n:= FAIL;
    end_if;
    if n <> FAIL then 
      // look-up for eq. 27 
      t:= diff(subs(dterm2,x^(2*n)=z),z);
      if not has(t,x) and t <> 1 and t <> 0 then 
        dterm2_new:= combine(expand(dterm2_new/t/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
        dterm1_new:= combine(expand(dterm1_new/t/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
        dterm0_new:= combine(expand(dterm0_new/t/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
      else
        dterm2_new:= dterm2;
        dterm1_new:= dterm1;
        dterm0_new:= dterm0;
      end_if;    
      a:= ode::normal(diff(subs(dterm2_new,x^n=z),z)/2);
      if not has(a,x) and not has(a,z) then 
        f:= ode::normal(dterm1_new/(x^n+a));
        if ode::odeIszero(dterm2_new-(x^n+a)^2) and 
           ode::odeIszero(dterm1_new-(x^n+a)*f) and 
           ode::odeIszero(dterm0_new+x^(n-2)*(x*f+a*n-a)) and 
           iszero(expand(dterm2_new-(x^n+a)^2,optIgnoreAnalyticConstraints)) and 
           iszero(expand(dterm1_new-(x^n+a)*f,optIgnoreAnalyticConstraints)) and 
           iszero(expand(dterm0_new+x^(n-2)*(x*f+a*n-a),optIgnoreAnalyticConstraints)) then
          y0:= (x^n+a)^(1/n);
          y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      end_if; 
      // look-up for eq. 27 
      if not iszero(t) and not has(t,x) and not has(t,z) then 
        t:= combine(sqrt(t),IgnoreAnalyticConstraints);
        for a in {-t,t} do 
          b:= ode::normal(diff(subs(dterm2,x^n=z),z)/(2*a));
          if not has(b,x) and not has(b,z) then 
            f:= ode::normal(dterm1/(a*x^n+b));
            if ode::odeIszero(dterm2-(a*x^n+b)^2) and 
               ode::odeIszero(dterm1-(a*x^n+b)*f) and 
               ode::odeIszero(dterm0-(f-a*n*x^(n-1)-1)) and 
               iszero(expand(dterm2-(a*x^n+b)^2,optIgnoreAnalyticConstraints)) and 
               iszero(expand(dterm1-(a*x^n+b)*f,optIgnoreAnalyticConstraints)) and 
               iszero(expand(dterm0-(f-a*n*x^(n-1)-1),optIgnoreAnalyticConstraints)) then
              y0:= exp(-int(1/(a*x^n+b),x,intOptions));
              y1:= y0*int(exp(-int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          end_if;
        end_for;
      end_if;      
    end_if;    
  end_if;  

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 30
  // ---------------------------

  exp_pow:= {};
  misc::maprec(dterm0_new,
               {"exp"} = proc(elem)
                              begin
                                if has(elem,x) and 
                                   iszero(expand(diff(op(elem,1),x,x),
                                                 optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
  exp_pow:= [op(exp_pow)];

  lambda:= FAIL;
  if nops(exp_pow) = 2 then 
    if iszero(expand(exp_pow[1]-2*exp_pow[2],optIgnoreAnalyticConstraints)) then 
      lambda:= exp_pow[2];
    elif iszero(expand(exp_pow[2]-2*exp_pow[1],optIgnoreAnalyticConstraints)) then 
      lambda:= exp_pow[1];
    end_if;
  end_if;
  if lambda <> FAIL then
    f:= dterm1_new;
    L:= ode::solveWithoutProperties(z*exp(lambda*x)*(f-z*exp(lambda*x)+lambda) = dterm0_new, z,
              IgnoreSpecialCases, optIgnoreAnalyticConstraints);
    L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
    for a in L do 
      y0:= exp(-a/lambda*exp(lambda*x));
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_for;             
  end_if;  
  
  // ---------------------------
  // look-up for eq. 31
  // ---------------------------

  exp_pow:= {};
  misc::maprec(dterm1_new,
               {"exp"} = proc(elem)
                              begin
                                if has(elem,x) and 
                                   iszero(expand(diff(op(elem,1),x,x),
                                                 optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
  exp_pow:= [op(exp_pow)];
  if nops(exp_pow) = 1 then 
    lambda:= exp_pow[1];
    a:= diff(subs(dterm1_new,exp(lambda*x)=z),z);
    if not has(a,x) and not has(a,z) then 
      f:= dterm1_new-a*exp(lambda*x);
      if ode::odeIszero(dterm0_new-a*exp(lambda*x)*(f+lambda)) and 
         iszero(expand(dterm0_new-a*exp(lambda*x)*(f+lambda),
                optIgnoreAnalyticConstraints)) then 
        y0:= exp(-a/lambda*exp(lambda*x));
        y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
  end_if;
  
  // ---------------------------
  // look-up for eq. 32
  // ---------------------------

  exp_pow2:= {};
  misc::maprec(dterm0_new,
               {"exp"} = proc(elem)
                              begin
                                if has(elem,x) and 
                                   iszero(expand(diff(op(elem,1),x,x),
                                                 optIgnoreAnalyticConstraints)) then 
                                  exp_pow2:= exp_pow2 union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
  exp_pow2:= [op(exp_pow2)];

  if nops(exp_pow) = 1 and 
     nops(exp_pow2) = 1 and 
     iszero(expand(exp_pow[1]-exp_pow2[1],optIgnoreAnalyticConstraints)) then 
    lambda:= exp_pow[1]/2;
    aa:= ode::normal((dterm0_new+lambda^2)/lambda);
    bb:= ode::normal((dterm1_new-aa)/(2*exp(2*lambda*x))); // should be b*f(x) if pattern matches
    cc:= ode::normal((dterm1_new+aa)/2); // should be a*f(x) if pattern matches
    if not iszero(cc) then 
      dd:= ode::normal(bb/cc);
      if ode::odeIszero(diff(dd,x)) and 
         iszero(expand(diff(dd,x),optIgnoreAnalyticConstraints)) then 
        ff:= dd*exp(lambda*x) + exp(-lambda*x);
        if ode::odeIszero(diff(ff,x,x) + dterm1_new*diff(ff,x) + dterm0_new*ff) and 
           iszero(combine(expand(diff(ff,x,x) + dterm1_new*diff(ff,x) + dterm0_new*ff,
                                 optIgnoreAnalyticConstraints),
                  exp)) then 
          y0:= ff;
          y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 33
  // ---------------------------

  exp_pow:= {};
  misc::maprec(dterm2,
               {"exp"} = proc(elem)
                              begin
                                if has(elem,x) and 
                                   iszero(expand(diff(op(elem,1),x,x),
                                                 optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
  exp_pow:= [op(exp_pow)];

  lambda:= FAIL;
  if nops(exp_pow) = 2 then 
    if iszero(expand(exp_pow[1]-2*exp_pow[2],optIgnoreAnalyticConstraints)) then 
      lambda:= exp_pow[2];
    elif iszero(expand(exp_pow[2]-2*exp_pow[1],optIgnoreAnalyticConstraints)) then 
      lambda:= exp_pow[1];
    end_if;
  end_if;
  if lambda <> FAIL then
    t:= combine(sqrt(diff(subs(dterm2,exp(2*lambda*x)=z),z)),IgnoreAnalyticConstraints);
    if ode::odeIszero(diff(t,x)) and 
       iszero(expand(diff(t,x),IgnoreAnalyticConstraints)) then 
      for a in {-t,t} do
        if not iszero(a) then 
          b:= diff(subs(dterm2,exp(lambda*x)=z),z)/(2*a);
          if ode::odeIszero(diff(b,x)) and 
             ode::odeIszero(dterm2-(a*exp(lambda*x)+b)^2) and 
             iszero(expand(diff(b,x),IgnoreAnalyticConstraints)) and 
             iszero(expand(dterm2-(a*exp(lambda*x)+b)^2,IgnoreAnalyticConstraints)) then 
            f:= combine(ode::normal(expand(dterm1/(a*exp(lambda*x)+b))),exp);
            // we have to solve ((-exp(2*lambda*x))*z^2 + (f*exp(lambda*x) + b*lambda*exp(lambda*x))*z = dterm0
            // rewrite this as U^2 - (f+b*lambda)*U + dterm0 = 0 where U = exp(lambda*x)*z
            // so we could also do            
            // L:= map(solve(z^2 - (f+b*lambda)*z + dterm0 = 0, z, IgnoreSpecialCases, optIgnoreAnalyticConstraints), ode::normal@_divide, exp(lambda*x));
            
            L:= ode::solveWithoutProperties((-exp(2*lambda*x))*z^2 + (f*exp(lambda*x) + b*lambda*exp(lambda*x))*z = dterm0,z,
                      IgnoreSpecialCases, optIgnoreAnalyticConstraints);
            L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
            for c in L do 
              y0:= (a*exp(lambda*x)+b)^(-c/(a*lambda));
              y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_for;     
          end_if;
        end_if;
      end_for;
    end_if;
  end_if;  

  // We need the coefficients without applying 'expand' as done above. 
  // This is due to the fact that we do not want 'sinh' or 'cosh'
  // expressions to be expanded. We need to undo this expansion
  // later using 'simplify' and this is even more costly when 'expand' 
  // was applied before. 
  dterm0_new:= combine(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/);
  dterm1_new:= combine(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/);
  dterm2_new:= combine(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/);

  if dterm2_new <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1_new/dterm2_new/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm0_new/dterm2_new/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  end_if;    

  // ---------------------------
  // look-up for eq. 34
  // ---------------------------
 
  if has(dterm1_new,sinh) and has(dterm0_new,cosh) then  
    dterm1_new:= simplify(dterm1_new);
    dterm0_new:= simplify(dterm0_new);
  end_if;
  functionArgs:= {};
  misc::maprec(dterm1_new,
               {"sinh"} = proc(elem)
                               begin
                                 if has(elem,x) and 
                                    iszero(expand(diff(op(elem,1),x,x),
                                                  optIgnoreAnalyticConstraints)) then 
                                   functionArgs:= functionArgs union {diff(op(elem,1),x)}
                                 end_if;
                                 elem;
                               end_proc);  
  functionArgs:= [op(functionArgs)];
  if nops(functionArgs) = 1 and not has((a:= functionArgs[1]),x) and not iszero(sinh(a*x)) then 
    f:= ode::normal(dterm1_new/sinh(a*x));
    functionArgs:= {};
    misc::maprec(dterm0_new,
                 {"cosh"} = proc(elem)
                                 begin
                                   if has(elem,x) and 
                                      iszero(expand(diff(op(elem,1),x,x),
                                                    optIgnoreAnalyticConstraints)) then 
                                     functionArgs:= functionArgs union {diff(op(elem,1),x)}
                                   end_if;
                                   elem;
                                 end_proc);  
    if nops(functionArgs) = 1 and 
       ode::odeIszero(functionArgs[1]-a) and 
       ode::odeIszero(dterm0_new+a*(a+f*cosh(a*x))) and 
       iszero(expand(functionArgs[1]-a,optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm0_new+a*(a+f*cosh(a*x)),optIgnoreAnalyticConstraints)) then 
      y0:= sinh(a*x);
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;
  
  // ---------------------------
  // look-up for eq. 35
  // ---------------------------
 
  if has(dterm1_new,cosh) and has(dterm0_new,sinh) then  
    dterm1_new:= simplify(dterm1_new);
    dterm0_new:= simplify(dterm0_new);
  end_if;

  functionArgs:= {};
  misc::maprec(dterm1_new,
               {"cosh"} = proc(elem)
                               begin
                                 if has(elem,x) and 
                                    iszero(expand(diff(op(elem,1),x,x),
                                                  optIgnoreAnalyticConstraints)) then 
                                   functionArgs:= functionArgs union {diff(op(elem,1),x)}
                                 end_if;
                                 elem;
                               end_proc);  
  functionArgs:= [op(functionArgs)];
  if nops(functionArgs) = 1 and not has((a:= functionArgs[1]),x) and not iszero(cosh(a*x)) then 
    f:= ode::normal(dterm1_new/cosh(a*x));
    functionArgs:= {};
    misc::maprec(dterm0_new,
                 {"sinh"} = proc(elem)
                                 begin
                                   if has(elem,x) and 
                                      iszero(expand(diff(op(elem,1),x,x),
                                                    optIgnoreAnalyticConstraints)) then 
                                     functionArgs:= functionArgs union {diff(op(elem,1),x)}
                                   end_if;
                                   elem;
                                 end_proc);  
    if nops(functionArgs) = 1 and 
       ode::odeIszero(functionArgs[1]-a) and 
       ode::odeIszero(dterm0_new+a*(a+f*sinh(a*x))) and 
       iszero(expand(functionArgs[1]-a,optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm0_new+a*(a+f*sinh(a*x)),optIgnoreAnalyticConstraints)) then 
      y0:= cosh(a*x);
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 40
  // ---------------------------
 
  if has(dterm1_new,sin) and has(dterm0_new,cos) then  
    dterm1_new:= simplify(dterm1_new);
    dterm0_new:= simplify(dterm0_new);
  end_if;
  functionArgs:= {};
  misc::maprec(dterm1_new,
               {"sin"} = proc(elem)
                               begin
                                 if has(elem,x) and 
                                    iszero(expand(diff(op(elem,1),x,x),
                                                  optIgnoreAnalyticConstraints)) then 
                                   functionArgs:= functionArgs union {diff(op(elem,1),x)}
                                 end_if;
                                 elem;
                               end_proc);  
  functionArgs:= [op(functionArgs)];
  if nops(functionArgs) = 1 and not has((a:= functionArgs[1]),x) and not iszero(sin(a*x)) then 
    f:= ode::normal(dterm1_new/sin(a*x));
    functionArgs:= {};
    misc::maprec(dterm0_new,
                 {"cos"} = proc(elem)
                                 begin
                                   if has(elem,x) and 
                                      iszero(expand(diff(op(elem,1),x,x),
                                                    optIgnoreAnalyticConstraints)) then 
                                     functionArgs:= functionArgs union {diff(op(elem,1),x)}
                                   end_if;
                                   elem;
                                 end_proc);  
    if nops(functionArgs) = 1 and 
       ode::odeIszero(functionArgs[1]-a) and 
       ode::odeIszero(dterm0_new-a*(a-f*cos(a*x))) and 
       iszero(expand(functionArgs[1]-a,optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm0_new-a*(a-f*cos(a*x)),optIgnoreAnalyticConstraints)) then 
      y0:= sin(a*x);
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 41
  // ---------------------------
 
  if has(dterm1_new,cos) and has(dterm0_new,sin) then  
    dterm1_new:= simplify(dterm1_new);
    dterm0_new:= simplify(dterm0_new);
  end_if;
  functionArgs:= {};
  misc::maprec(dterm1_new,
               {"cos"} = proc(elem)
                               begin
                                 if has(elem,x) and 
                                    iszero(expand(diff(op(elem,1),x,x),
                                                  optIgnoreAnalyticConstraints)) then 
                                   functionArgs:= functionArgs union {diff(op(elem,1),x)}
                                 end_if;
                                 elem;
                               end_proc);  
  functionArgs:= [op(functionArgs)];
  if nops(functionArgs) = 1 and not has((a:= functionArgs[1]),x) and not iszero(cos(a*x)) then 
    f:= ode::normal(dterm1_new/cos(a*x));
    functionArgs:= {};
    misc::maprec(dterm0_new,
                 {"sin"} = proc(elem)
                                 begin
                                   if has(elem,x) and 
                                      iszero(expand(diff(op(elem,1),x,x),
                                                    optIgnoreAnalyticConstraints)) then 
                                     functionArgs:= functionArgs union {diff(op(elem,1),x)}
                                   end_if;
                                   elem;
                                 end_proc);  
    if nops(functionArgs) = 1 and 
       ode::odeIszero(functionArgs[1]-a) and 
       ode::odeIszero(dterm0_new-a*(a+f*sin(a*x))) and 
       iszero(expand(functionArgs[1]-a,optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm0_new-a*(a+f*sin(a*x)),optIgnoreAnalyticConstraints)) then 
      y0:= cos(a*x);
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;

  // ode::normalization 
  if dterm2 <> x then 
    dterm2_new:= x;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 36
  // ---------------------------
  
  if ode::odeIszero(dterm1_new+dterm0_new*x*ln(x)-1) and 
     iszero(expand(dterm1_new+dterm0_new*x*ln(x)-1,optIgnoreAnalyticConstraints)) then
    y0:= ln(x);
    y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // ---------------------------
  // look-up for eq. 37
  // ---------------------------

  if has(dterm1_new, ln) and traperror((f:= subs(dterm1_new, ln(x)=0))) = 0 then 
    a:= ode::normal((dterm1_new-f)/(x*ln(x)));
    if ode::odeIszero(diff(a,x)) and 
       ode::odeIszero(dterm0_new-a*(f*ln(x)+1)) and 
       iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm0_new-a*(f*ln(x)+1),optIgnoreAnalyticConstraints)) then 
      y0:= exp(a*x)*x^(-a*x);
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;

  // ode::normalization 
  if dterm2 <> x^2 then 
    dterm2_new:= x^2;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 38
  // ---------------------------

  if has(dterm1_new,ln) and has(dterm0_new,ln) then
    aa:= ode::normal(dterm1_new/(2*x));
    bb:= 1/4-dterm0_new;
    f:= expand((bb-aa)/2,optIgnoreAnalyticConstraints);
    if not iszero(expand(f,optIgnoreAnalyticConstraints)) then 
      a:= ode::normal(aa/f-ln(x));
      if ode::odeIszero(diff(a,x)) and 
         ode::odeIszero(dterm1_new-2*x*(ln(x)+a)*f) and 
         ode::odeIszero(dterm0_new-(1/4-(ln(x)+a+2)*f)) and 
         iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm1_new-2*x*(ln(x)+a)*f,optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0_new-(1/4-(ln(x)+a+2)*f),optIgnoreAnalyticConstraints)) then 
        y0:= sqrt(x)*(ln(x)+a);
        y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 39
  // ---------------------------

  if has(dterm1_new,ln) and has(dterm0_new,ln) and 
     traperror((f:=subs(dterm1_new,ln(x)=0))) = 0 then
    f:= ode::normal(f/x);
    a:= ode::normal(diff(subs(dterm1_new,ln(x)=z),z)/x);
    if ode::odeIszero(diff(a,x)) and 
       ode::odeIszero(dterm1_new-x*(f+a*ln(x))) and 
       ode::odeIszero(dterm0_new-a*(f*ln(x)-ln(x)+1)) and 
       iszero(expand(diff(a,x),optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm1_new-x*(f+a*ln(x)),optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm0_new-a*(f*ln(x)-ln(x)+1),optIgnoreAnalyticConstraints)) then 
      y0:= exp(-1/2*a*ln(x)^2);
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 42
  // ---------------------------

  if has(dterm1_new, sin) and has(dterm0_new, sin) and has(dterm0_new, cos) then 
    a:= diff(subs(dterm1_new, sin(x)=z),z);
    if not has(a,z) and not has(a,x) then 
      f:= dterm1_new - a*sin(x);
      if ode::odeIszero(dterm0_new-a*(f*sin(x)+cos(x))) and 
         iszero(expand(dterm0_new-a*(f*sin(x)+cos(x)),optIgnoreAnalyticConstraints)) then 
        y0:= exp(a*cos(x));
        y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 43
  // ---------------------------

  if has(dterm1_new, cos) and has(dterm0_new, cos) and has(dterm0_new, sin) then 
    a:= diff(subs(dterm1_new, cos(x)=z),z);
    if not has(a,z) and not has(a,x) then 
      f:= dterm1_new - a*cos(x);
      if ode::odeIszero(dterm0_new-a*(f*cos(x)-sin(x))) and 
         iszero(expand(dterm0_new-a*(f*cos(x)-sin(x)),optIgnoreAnalyticConstraints)) then 
        y0:= exp(-a*sin(x));
        y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 44
  // ---------------------------

  cos_pow:= {};
  misc::maprec(dterm1_new,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = cos(x) then 
                                  cos_pow:= cos_pow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  cos_pow:= [op(cos_pow)];

  if nops(cos_pow) = 1 then 
    n:= cos_pow[1];
    a:= diff(subs(dterm1_new, cos(x)^n=z),z);
    if not has(a,x) and not has(a,z) then 
      f:= dterm1_new - a*cos(x)^n;
      if ode::odeIszero(dterm0_new-a*cos(x)^(n-1)*(f*cos(x)-n*sin(x))) and 
         iszero(expand(dterm0_new-a*cos(x)^(n-1)*(f*cos(x)-n*sin(x)),
                       optIgnoreAnalyticConstraints)) then 
        y0:= exp(-a*Int(cos(x)^n,x,intOptions));
        y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 45
  // ---------------------------

  sin_pow:= {};
  misc::maprec(dterm1_new,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = sin(x) then 
                                  sin_pow:= sin_pow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  sin_pow:= [op(sin_pow)];

  if nops(sin_pow) = 1 then 
    n:= sin_pow[1];
    a:= diff(subs(dterm1_new, sin(x)^n=z),z);
    if not has(a,x) and not has(a,z) then 
      f:= dterm1_new - a*sin(x)^n;
      if ode::odeIszero(dterm0_new-a*sin(x)^(n-1)*(f*sin(x)+n*cos(x))) and 
         iszero(expand(dterm0_new-a*sin(x)^(n-1)*(f*sin(x)+n*cos(x)),
                        optIgnoreAnalyticConstraints)) then 
         y0:= exp(-a*Int(sin(x)^n,x,intOptions));
         y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
         return({y0,y1});
       end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 46
  // ---------------------------
  
  if has(dterm2,sin) or has(dterm1,sin) or has(dterm0,sin) then 
    // ode::normalization 
    if dterm2 <> sin(x)^2 then 
      dterm2_new:= sin(x)^2;
      dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
      dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    else
      dterm2_new:= dterm2;
      dterm1_new:= dterm1;
      dterm0_new:= dterm0;
    end_if;  
    t1:= ode::normal(dterm1_new/sin(x));
    if traperror((t0:= subs(dterm0_new, cos(x)=0, EvalChanges))) = 0 then     
      L:= simplify({t1/2+(t1^2-4*t0)^(1/2)/2,
                    t1/2-(t1^2-4*t0)^(1/2)/2}, 
                   IgnoreAnalyticConstraints);  
      L:= select(L, _not@has, x);
      for a in L do 
        f:= t1-a;
        if ode::odeIszero(dterm1_new-sin(x)*(f+a)) and ode::odeIszero(dterm0_new-a*(f-cos(x))) and 
           iszero(combine(expand(dterm1_new-sin(x)*(f+a),optIgnoreAnalyticConstraints),
                          IgnoreAnalyticConstraints)) and 
           iszero(combine(expand(dterm0_new-a*(f-cos(x)),optIgnoreAnalyticConstraints),
                          IgnoreAnalyticConstraints)) then 
          y0:= cot(1/2*x)^a;
          y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      end_for;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 47
  // ---------------------------

  if has(dterm2,cos) or has(dterm1,cos) or has(dterm0,cos) then 
    // ode::normalization 
    if dterm2 <> cos(x)^2 then 
      dterm2_new:= cos(x)^2;
      dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
      dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    else
      dterm2_new:= dterm2;
      dterm1_new:= dterm1;
      dterm0_new:= dterm0;
    end_if;  
    t1:= ode::normal(dterm1_new/cos(x));
    if traperror((t0:= subs(dterm0_new, sin(x)=0, EvalChanges))) = 0 then     
      L:= simplify({t1/2+(t1^2-4*t0)^(1/2)/2,
                    t1/2-(t1^2-4*t0)^(1/2)/2}, 
                   IgnoreAnalyticConstraints);  
      L:= select(L, _not@has, x);
      for a in L do 
        f:= t1-a;
        if ode::odeIszero(dterm1_new-cos(x)*(f+a)) and ode::odeIszero(dterm0_new-a*(f+sin(x))) and 
           iszero(combine(expand(dterm1_new-cos(x)*(f+a),optIgnoreAnalyticConstraints),
                          IgnoreAnalyticConstraints)) and 
           iszero(combine(expand(dterm0_new-a*(f+sin(x)),optIgnoreAnalyticConstraints),
                          IgnoreAnalyticConstraints)) then 
          y0:= cot(1/2*x+1/4*PI)^a;
          y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      end_for;
    end_if;
  end_if;
  
  // ---------------------------
  // look-up for eq. 48
  // ---------------------------

  // We need the coefficients without applying 'expand' as done above. 
  // This is due to the fact that we do not want 'tan' expressions
  // expressions to be expanded. 
  dterm0_new:= combine(diff(eq,y0),optIgnoreAnalyticConstraints);
  dterm1_new:= combine(diff(eq,y1),optIgnoreAnalyticConstraints);
  dterm2_new:= combine(diff(eq,y2),optIgnoreAnalyticConstraints);

  // ode::normalization 
  if dterm2_new <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(ode::normal(dterm1_new/dterm2_new)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(ode::normal(dterm0_new/dterm2_new)/*,optIgnoreAnalyticConstraints*/);
  end_if;
    
  if has(dterm0_new, tan) then 
    tan_pow:= {};
    lambda:= FAIL;
    misc::maprec(dterm0_new,
                 {"_power"} = proc(elem)
                                begin
                                  if type(op(elem,1)) = "tan" and 
                                     not has((lambda:= diff(op(op(elem,1)),x)),x) then 
                                    tan_pow:= tan_pow union {op(elem,2)}
                                  end_if;
                                  elem;
                                end_proc);  
    tan_pow:= [op(tan_pow)];
    if nops(tan_pow) = 1 and tan_pow[1] = 2 and lambda <> FAIL and lambda <> 0 then 
      t:= diff(subs(dterm0_new,tan(lambda*x)^2=z),z);
      if not has(t,z) then 
        L:= ode::solveWithoutProperties(z*(lambda-z)-t = 0, z, IgnoreSpecialCases, optIgnoreAnalyticConstraints);
        L:= select(simplify(L,IgnoreAnalyticConstraints), _not@has, x);
        for a in L do 
          f:= dterm1_new;
          if ode::odeIszero(dterm0_new-a*(lambda+f*tan(lambda*x)+(lambda-a)*tan(lambda*x)^2)) and 
             iszero(expand(dterm0_new-a*(lambda+f*tan(lambda*x)+(lambda-a)*tan(lambda*x)^2),
                           IgnoreAnalyticConstraints)) then 
            y0:= cos(lambda*x)^(a/lambda);
            y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});  
          end_if;
        end_for;
      end_if;
    end_if;
  end_if;  

  // ---------------------------
  // look-up for eq. 51
  // ---------------------------

  // requires not expanded form to macht as eq. 48!!!
 
  if has(dterm0_new, cot) then 
    cot_pow:= {};
    lambda:= FAIL;
    misc::maprec(dterm0_new,
                 {"_power"} = proc(elem)
                                begin
                                  if type(op(elem,1)) = "cot" and 
                                     not has((lambda:= diff(op(op(elem,1)),x)),x) then 
                                    cot_pow:= cot_pow union {op(elem,2)}
                                  end_if;
                                  elem;
                                end_proc);  
    cot_pow:= [op(cot_pow)];
    if nops(cot_pow) = 1 and cot_pow[1] = 2 and lambda <> FAIL and lambda <> 0 then 
      t:= diff(subs(dterm0_new,cot(lambda*x)^2=z),z);
      if not has(t,z) then 
        L:= ode::solveWithoutProperties(z*(lambda-z)-t = 0, z, IgnoreSpecialCases, optIgnoreAnalyticConstraints);
        L:= select(simplify(L,IgnoreAnalyticConstraints), _not@has, x);
        for a in L do 
          f:= dterm1_new;
          if ode::odeIszero(dterm0_new-a*(lambda-f*cot(lambda*x)+(lambda-a)*cot(lambda*x)^2)) and 
             iszero(expand(dterm0_new-a*(lambda-f*cot(lambda*x)+(lambda-a)*cot(lambda*x)^2),
                           IgnoreAnalyticConstraints)) then 
            y0:= sin(lambda*x)^(a/lambda);
            y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});  
          end_if;
        end_for;
      end_if;
    end_if;
  end_if;  

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 49
  // ---------------------------

  if has(dterm1_new,tan) and has(dterm0_new,tan) then 
    a:= diff(subs(dterm1_new,tan(x)=z),z);
    if not has(a,x) and not has(a,z) then 
      f:= dterm1_new-a*tan(x);
      if ode::odeIszero(dterm0_new-(a+1)*(f*tan(x)+1)) and 
         iszero(expand(dterm0_new-(a+1)*(f*tan(x)+1),optIgnoreAnalyticConstraints)) then 
        y0:= cos(x)^(a+1);
        y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});  
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 50
  // ---------------------------
  
  tan_pow:= {};
  misc::maprec(dterm0_new,
               {"_power"} = proc(elem)
                              begin
                                if type(op(elem,1)) = "tan" then 
                                  tan_pow:= tan_pow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  tan_pow:= [op(tan_pow)];
  if nops(tan_pow) = 1 and tan_pow[1] = 2 then 
    if traperror((t0:= subs(dterm0_new,tan(x)^2=0,EvalChanges))) = 0 and 
       traperror((t1:= subs(dterm1_new,tan(x)=1,EvalChanges))) = 0 then 
      a:= ode::normal((t0+t1-1)/3); 
      if not has(a,x) then  
        f:= ode::normal(dterm1_new/tan(x))-a+1;
        if ode::odeIszero(dterm1_new-tan(x)*(f+a-1)) and 
           ode::odeIszero(dterm0_new-((a*tan(x)^2-1)*f+2*a+2)) and 
           iszero(expand(dterm1_new-tan(x)*(f+a-1),optIgnoreAnalyticConstraints)) and 
           iszero(expand(dterm0_new-((a*tan(x)^2-1)*f+2*a+2),optIgnoreAnalyticConstraints)) then 
          y0:= sin(x)*cos(x)^a;
          y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});  
        end_if;
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 52
  // ---------------------------

  if has(dterm1_new,cot) and has(dterm0_new,cot) then 
    a:= diff(subs(dterm1_new,cot(x)=z),z);
    if not has(a,x) and not has(a,z) then 
      f:= dterm1_new-a*cot(x);
      if ode::odeIszero(dterm0_new-(a-1)*(f*cot(x)-1)) and 
         iszero(expand(dterm0_new-(a-1)*(f*cot(x)-1),optIgnoreAnalyticConstraints)) then 
        y0:= sin(x)^(1-a);
        y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});  
      end_if;
    end_if;
  end_if;

  // ---------------------------

  return(FAIL);
end_proc:

/*
  ----------
  REFERENCE. Implementation of (Section 2.1.9.-2. eq. 53 to 91 p. 289 to 292)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/

ode::PZ_Sec_219_2_eq_53_to_91_p_289_to_292:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, optIgnoreAnalyticConstraints,
        Int, dterm0_new, dterm1_new, dterm2_new, z, a, L, f, t, xpow, F, f0, f1, res,
        csts, b, c, n, exp_pow;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   

  Int:= if has(odeOptions,freeze) then freeze(int) else int end_if;

  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) or (not has(eq,y0) and not has(eq,y1)) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);

  if iszero(expand(dterm2,optIgnoreAnalyticConstraints)) then 
    return(FAIL)
  end_if;

  z:= genident(); //solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 54
  // ---------------------------

  f:= dterm1_new;
  L:= ode::solveWithoutProperties(f^2*z^2+(f^2+diff(f,x))*z+dterm0_new, z,
            IgnoreSpecialCases, optIgnoreAnalyticConstraints);
  L:= select(ode::normal(L), _not@has, x);
  for a in L do 
    if ode::odeIszero(dterm0_new+(a*(a+1)*f^2+a*diff(f,x))) and 
       iszero(expand(ode::normal(dterm0_new+(a*(a+1)*f^2+a*diff(f,x))),optIgnoreAnalyticConstraints)) then 
      y0:= exp(a*int(f,x));
      y1:= y0*int(exp(-int(dterm1_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});   
    end_if;
  end_for;    

  // ---------------------------
  // look-up for eq. 55
  // ---------------------------
 
  f:= dterm1_new/2;
  if ode::odeIszero(dterm0_new-(f^2+diff(f,x))) and 
     iszero(expand(ode::normal(dterm0_new-(f^2+diff(f,x))),optIgnoreAnalyticConstraints)) then 
    return({exp(-int(f,x)), x*exp(-int(f,x))})
  end_if;

  // ---------------------------
  // look-up for eq. 56
  // ---------------------------
  f:= dterm1_new;
  L:= ode::solveWithoutProperties(z*f^2/(1-z)^2+z*diff(f,x)/(1-z) = -dterm0_new,z, 
            IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  L:= select(ode::normal(L), _not@has, x);
  for a in L do 
    if not iszero(1-a) and 
       ode::odeIszero(dterm0_new+a*(f^2/(1-a)^2+diff(f,x)/(1-a))) and 
       iszero(expand(dterm0_new+a*(f^2/(1-a)^2+diff(f,x)/(1-a)),
                     optIgnoreAnalyticConstraints)) then 
      y0:= exp(a/(1-a)*int(f,x));
      y1:= y0*int(exp(-int(dterm1_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});   
    end_if;
  end_for;

  // ---------------------------
  // look-up for eq. 58
  // ---------------------------

  f:= dterm1_new/2;
  a:= combine(ode::normal(dterm0_new-f^2-diff(f,x)),IgnoreAnalyticConstraints);
  if not has(a,x) and not iszero(a) then 
    return({exp((-a)^(1/2)*x-int(f,x)), 1/exp((-a)^(1/2)*x+int(f,x))});
  end_if;

  // ---------------------------
  // look-up for eq. 59
  // ---------------------------

  f:= dterm1_new/2;
  t:= combine(expand(dterm0_new-f^2-diff(f,x)/*,optIgnoreAnalyticConstraints*/),
              optIgnoreAnalyticConstraints);  
  xpow:= {};
  misc::maprec(t,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  if xpow = {4} then 
    xpow:= {4,1}
  end_if;
  if nops(xpow) = 2 and type(t) = "_plus" and nops(t) = 2 then 
    L:= ode::PZ_Sec_212_1_eq_10_p_215(diff(y(x),x,x)+t*y(x),y,x,solveOptions,odeOptions);
    //L:= ode::lookUp2ndOrderLinear(diff(y(x),x,x)+t*y(x),y,x,solveOptions,odeOptions);
    if L <> FAIL and L <> {} then 
      y0:= L[1]*exp(-int(f,x));
      y1:= L[2]*exp(-int(f,x));
      return({y0,y1});
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 60
  // ---------------------------
  
  F:= ode::normal(dterm1_new/2); // F = f+a/2
  if iszero(diff((f0:= expand(F^2-dterm1_new*F+dterm0_new-diff(F,x),
                             IgnoreAnalyticConstraints)),x)) and 
     iszero(diff(expand((f1:= dterm1_new-2*F),IgnoreAnalyticConstraints),x)) then 
    res:= solve(ode(diff(y(x),x,x) + f1*diff(y(x),x) + f0*y(x),y(x)));
    csts:= [op(freeIndets(res) minus (indets(eq)))];
    if nops(csts) = 2 then 
      res:= subs(res,csts[1]=1,csts[2]=0) union subs(res,csts[1]=0,csts[2]=1);
      y0:= res[1]*exp(-int(F,x));
      y1:= res[2]*exp(-int(F,x));
      return({y0,y1});
    end_if;  
  end_if;         

  // =================================================
  // different ode::normalization using original input data
  // =================================================

  if dterm2 <> x then 
    dterm2_new:= x;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 62
  // ---------------------------

  f:= ode::normal(dterm1_new/x);
  if ode::odeIszero(dterm0_new-f-x*diff(f,x)) and 
     iszero(expand(dterm0_new-f-x*diff(f,x),optIgnoreAnalyticConstraints)) then 
    y0:= x*exp(-int(f,x));
    y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});   
  end_if;

  // ---------------------------
  // look-up for eq. 63
  // ---------------------------
 
  if traperror((a:= (dterm1_new | x=0))) = 0 then 
    f:= ode::normal((dterm1_new-a)/x);
    if ode::odeIszero(dterm0_new-a*f-x*diff(f,x)) and 
       iszero(expand(dterm0_new-a*f-x*diff(f,x),optIgnoreAnalyticConstraints)) then 
      y0:= exp(-int(f,x));
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});   
    end_if;
  end_if; 

  // ode::normalization 
  if dterm2 <> x^2 then 
    dterm2_new:= x^2;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    
    
  // ---------------------------
  // look-up for eq. 65
  // ---------------------------

  f:= ode::normal(dterm1_new/x);
  t:= combine(expand(dterm0_new - x*diff(f,x)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  L:= ode::solveWithoutProperties(z*f-z*(z+1)=dterm0_new-x*diff(f,x),z, 
            IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
  if nops(L) > 0 then 
    a:= L[1];
    y0:= x^(a+1)*exp(-int(f/x,x));
    y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});   
  end_if; 

  // ---------------------------
  // look-up for eq. 66
  // ---------------------------

  f:= ode::normal(dterm1_new/(2*x));
  t:= combine(expand(dterm0_new-x*diff(f,x)-f^2+f/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  if ode::odeIszero(diff(t,x,x,x)) and 
     iszero(expand(diff(t,x,x,x),optIgnoreAnalyticConstraints)) and 
     traperror((c:= t | x=0)) = 0 then 
    a:= ode::normal(diff(t,x,x)/2);
    b:= diff(t-a*x^2-c,x);
    res:= ode::specfunc(x^2*diff(y(x),x,x) + (a*x^2+b*x+c)*y(x), y, x, solveOptions, odeOptions);
    if nops(res) = 2 then 
      y0:= res[1]*exp(-int(f/x,x));
      y1:= res[2]*exp(-int(f/x,x));
      return({y0,y1});
    elif nops(res) = 1 then 
      y0:= res[1]*exp(-int(f/x,x));
      y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;  
  end_if;

  // -----------------------------------
  // look-up for eq. 68 (includes 67,70)
  // -----------------------------------

  F:= -ode::normal((dterm1_new/x)/2);
  f1:= combine(expand(dterm1_new+2*x*F/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  f0:= combine(expand(-F+dterm0_new+x*diff(F,x)+F^2+1/x*F*dterm1_new/*,optIgnoreAnalyticConstraints*/),
               optIgnoreAnalyticConstraints);
  L:= ode::PZ_Sec_212_4_eq_118_119_132_133_134_143_144_145_147_p_226_228_229_230(x^2*diff(y(x),x,x)+
          f1*diff(y(x),x)+f0*y(x),y,x,solveOptions,odeOptions);
  if L <> FAIL and L <> {} then 
    y0:= L[1]*exp(int(F/x,x));
    y1:= L[2]*exp(int(F/x,x));
    return({y0,y1});
  elif ode::odeIszero(f1) and 
       iszero(expand(f1,optIgnoreAnalyticConstraints)) then 
    // look-up for eq. 70 (after transformation introduced via 'F', the term 'f0' must
    //                     be of the form 'a1*x^(2*n)+a2*x^n+a0')
    xpow:= {};
    misc::maprec(f0,
                {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
    n:= FAIL;
    if nops(xpow) = 1 and xpow[1]=2 then 
      n:= 2;
    elif nops(xpow) = 2 and 
         iszero(expand(xpow[1]-2*xpow[2],optIgnoreAnalyticConstraints)) then 
      n:= xpow[2];
    elif nops(xpow) = 2 and 
         iszero(expand(xpow[2]-2*xpow[1],optIgnoreAnalyticConstraints)) then 
      n:= xpow[1];
    end_if;  
    if n <> FAIL then 
      a:= diff(subs(f0,x^(2*n)=z),z);
      b:= diff(subs(eval(f0-x^(2*n)),x^n=z),z);
      c:= combine(expand(f0-a*x^(2*n)-b*x^n/*,optIgnoreAnalyticConstraints*/),
                  optIgnoreAnalyticConstraints);
      if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and not has(c,x) and 
         traperror((L:= 
            {x^(1/2 - n/2)*whittakerM(-(b*I)/(2*a^(1/2)*n), ((4*c - 1)^(1/2)*I)/(2*n), 
                                       (a^(1/2)*x^n*2*I)/n), 
             x^(1/2 - n/2)*whittakerW(-(b*I)/(2*a^(1/2)*n), ((4*c - 1)^(1/2)*I)/(2*n), 
                                       (a^(1/2)*x^n*2*I)/n)}
                   )) = 0 then 
        y0:= L[1]*exp(int(F/x,x));
        y1:= L[2]*exp(int(F/x,x));
        return({y0,y1});
      end_if;
    end_if;
  end_if;

  // -----------------------------------------------------------------------------
  // look-up for equations of the form 
  //
  //               x^2*y''(x) + (a*x^(2*n)+b*x^n+c)*y(x) = 0
  // 
  // not listed in the book, this is a self-made look-up procedure (Abfallprodukt
  // des Look-ups fr Gleichungen vom Typ eq. 70). 
  // -----------------------------------------------------------------------------
  if ode::odeIszero(dterm1_new) and ode::odeIszero(dterm2_new-x^2) and 
     iszero(expand(dterm1_new,optIgnoreAnalyticConstraints)) and 
     iszero(expand(dterm2_new-x^2,optIgnoreAnalyticConstraints)) then 
    // look-up for eq. 70 (after transformation introduced via 'F', the term 'f0' must
    //                     be of the form 'a1*x^(2*n)+a2*x^n+a0')
    xpow:= {};
    misc::maprec(dterm0_new,
                {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
    n:= FAIL;
    if nops(xpow) = 1 and xpow[1]=2 then 
      n:= 2;
    elif nops(xpow) = 2 and 
         iszero(expand(xpow[1]-2*xpow[2],optIgnoreAnalyticConstraints)) then 
      n:= xpow[2];
    elif nops(xpow) = 2 and 
         iszero(expand(xpow[2]-2*xpow[1],optIgnoreAnalyticConstraints)) then 
      n:= xpow[1];
    end_if;  
    if n <> FAIL then 
      a:= diff(subs(dterm0_new,x^(2*n)=z),z);
      b:= diff(subs(eval(dterm0_new-x^(2*n)),x^n=z),z);
      c:= combine(expand(dterm0_new-a*x^(2*n)-b*x^n/*,optIgnoreAnalyticConstraints*/),
                  optIgnoreAnalyticConstraints);
      if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and not has(c,x) and 
         traperror((L:= 
            {x^(1/2 - n/2)*whittakerM(-(b*I)/(2*a^(1/2)*n), ((4*c - 1)^(1/2)*I)/(2*n), 
                                       (a^(1/2)*x^n*2*I)/n), 
             x^(1/2 - n/2)*whittakerW(-(b*I)/(2*a^(1/2)*n), ((4*c - 1)^(1/2)*I)/(2*n), 
                                       (a^(1/2)*x^n*2*I)/n)}
                   )) = 0 then 
        y0:= L[1]*exp(int(F/x,x));
        y1:= L[2]*exp(int(F/x,x));
        return({y0,y1});
      end_if;
    end_if;
  end_if;
  // -----------------------------------------------------------------------------

  // NOTE: We need to operate and the non-ode::normalized input data here!!!

  // ---------------------------
  // look-up for eq. 71
  // ---------------------------

  f:= ode::normal(dterm2/2);
  a:= dterm0;
  if not has(a,x) and ode::odeIszero(dterm1-diff(f,x)) and 
     iszero(expand(dterm1-diff(f,x),optIgnoreAnalyticConstraints)) then 
    res:= solve(ode(2*diff(y(x),x,x) + a*y(x), y(x)));
    csts:= [op(freeIndets(res) minus (indets(eq)))];
    if nops(csts) = 2 then 
      res:= subs(res,csts[1]=1,csts[2]=0) union subs(res,csts[1]=0,csts[2]=1);
      res:= res | x = int(1/sqrt(f),x);
      return(res);
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 72
  // ---------------------------

  f:= dterm1;
  if not iszero(f) then 
    a:= combine(expand(ode::normal(-dterm0/f^3)/*,optIgnoreAnalyticConstraints*/),
                optIgnoreAnalyticConstraints);
    if not has(a,x) and ode::odeIszero(dterm1+diff(f,x)) and 
       iszero(expand(dterm1+diff(f,x),optIgnoreAnalyticConstraints)) then 
      t:= sqrt(a)*int(f,x);
      return({exp(t),exp(-t)});
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 73
  // ---------------------------

  f:= ode::normal(dterm2);
  if has(f,x) then 
    a:= combine(expand(ode::normal(-dterm1/diff(f,x))/*,optIgnoreAnalyticConstraints*/),
                optIgnoreAnalyticConstraints);
    b:= combine(expand(ode::normal(-dterm0/f^(2*a+1))/*,optIgnoreAnalyticConstraints*/),
                optIgnoreAnalyticConstraints);
    if not has(a,x) and not has(b,x) then 
      t:= sqrt(b)*int(f^a,x);
      return({exp(t),exp(-t)})
    end_if;  
  end_if;

  // ---------------------------
  // look-up for eq. 74
  // ---------------------------

  f:= dterm2;
  b:= combine(expand(ode::normal(dterm0/f^3)/*,optIgnoreAnalyticConstraints*/),
              optIgnoreAnalyticConstraints);
  a:= combine(expand(ode::normal((-dterm1-diff(f,x))/f^2)/*,optIgnoreAnalyticConstraints*/),
              optIgnoreAnalyticConstraints);
  if not has(a,x) and not has(b,x) then 
    L:= ode::solveWithoutProperties(z^2-a*z+b=0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    if nops(L) = 2 then 
      return({exp(L[1]*int(f,x)),exp(L[2]*int(f,x))});
    end_if;  
  end_if;

  // ---------------------------
  // look-up for eq. 75
  // ---------------------------

  f:= dterm2;
  if has(f,x) then 
    a:= combine(expand(ode::normal(-(dterm1+diff(f,x))/f)/*,optIgnoreAnalyticConstraints*/),
                optIgnoreAnalyticConstraints);
    if not has(a,x) then 
      L:= ode::solveWithoutProperties(z*f^2*(a+z*f) = -dterm0, z,
                IgnoreSpecialCases, optIgnoreAnalyticConstraints);
      L:= select(ode::normal(L), _not@has, x);
      if nops(L) > 0 then 
        b:= L[1];   
        y0:= exp(-b*int(f,x));
        y1:= y0*int(exp(-int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1})
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 76
  // ---------------------------

  f:= dterm2;
  if has(f,x) then 
    a:= combine(expand(ode::normal(-(dterm1+diff(f,x))/(2*f))/*,optIgnoreAnalyticConstraints*/),
                optIgnoreAnalyticConstraints);
    b:= combine(sqrt(expand(ode::normal((dterm0-a*diff(f,x)-a^2*f)/(-f^3))/*,optIgnoreAnalyticConstraints*/)),
                IgnoreAnalyticConstraints); 
    if not has(b,x) then 
      y0:= exp(a*x+b*int(f,x));    
      y1:= y0*int(exp(-int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1})
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 77
  // ---------------------------

  t:= combine(sqrt(combine(dterm2,exp)),IgnoreAnalyticConstraints);
  delete f;
  for f in {-t,t} do 
    if has(f,x) then 
      a:= combine(expand(ode::normal(dterm1/f)-diff(f,x)/*,optIgnoreAnalyticConstraints*/),
                  optIgnoreAnalyticConstraints);
      b:= dterm0;
      if not has(a,x) and not has(b,x) then
        res:= solve(ode(diff(y(x),x,x) + a*diff(y(x),x) + b*y(x),y(x)));
        csts:= [op(freeIndets(res) minus (indets(eq)))];
        if nops(csts) = 2 then 
          res:= subs(res,csts[1]=1,csts[2]=0) union subs(res,csts[1]=0,csts[2]=1);
          res:= map(res, elem -> subs(elem, x = int(1/f,x)));
          return(res);
        end_if;  
      end_if;
    end_if;
  end_for;

  // ---------------------------
  // look-up for eq. 81 & 82
  // ---------------------------

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  dterm0_new:= combine(dterm0_new,exp);
  exp_pow:= {};
  misc::maprec(dterm0_new,
               {"exp"} = proc(elem)
                              begin
                                exp_pow:= exp_pow union {ode::normal(op(elem,1)/2)};
                                elem;
                              end_proc);  
  exp_pow:= [op(exp_pow)];
  if nops(exp_pow) = 1 then 
    f:= exp_pow[1];
    if ode::odeIszero(expand(dterm1_new+diff(f,x),optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm1_new+diff(f,x),optIgnoreAnalyticConstraints)) then 
      a:= combine(sqrt(ode::normal(combine(dterm0_new/exp(2*f),exp))),IgnoreAnalyticConstraints);
      if not has(a,x) then 
        return({sin(a*int(exp(f),x)),cos(a*int(exp(f),x))});
      end_if;
    end_if;   
  end_if;

  // ---------------------------
  // look-up for eq. 84
  // ---------------------------

  // Note: we need the input data in its original form (i.e. no 'expanded'
  //       since this reduces the number of cases which can be recognized
  //       due to the fact that we cannot (and do not want to) use arbitary 
  //       factorization methods here for reasons of efficiency. 
  dterm0_new:= diff(eq,y0);
  dterm1_new:= diff(eq,y1);
  dterm2_new:= diff(eq,y2);

  f:= combine(sqrt(dterm2_new/4),IgnoreAnalyticConstraints);
  if iszero(dterm1_new) then 
    a:= combine(expand(-dterm0_new-2*f*diff(f,x,x)+diff(f,x)^2/*,optIgnoreAnalyticConstraints*/),
                optIgnoreAnalyticConstraints);  
    if not has(a,x) then 
      if not iszero(a) then 
        return({sqrt(f)*exp(1/2*sqrt(a)*int(1/f,x)), 
                sqrt(f)*exp(-1/2*sqrt(a)*int(1/f,x))});
      else
        return({sqrt(f),sqrt(f)*int(1/f,x)});
      end_if;
    end_if;
  end_if;


  // ---------------------------

  return(FAIL);
end_proc:


/*
  ----------
  REFERENCE. Implementation of (Section 2.1.3.-2. eq. 1 to 79 p. 246 to 252)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/


ode::PZ_Sec_213_1_eq_1_to_79_p_246_to_252:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, optIgnoreAnalyticConstraints,
        Int, dterm0_new, dterm1_new, dterm2_new, z, exp_pow, lambda, a, b, t0, t1, c, 
        L, mu, elem, ind, k, t, csts, d, m, n, l, xpow, bb;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   

  Int:= if has(odeOptions,freeze) then freeze(int) else int end_if;

  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) or (not has(eq,y0) and not has(eq,y1)) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);

  dterm0:= combine(dterm0, exp);
  dterm1:= combine(dterm1, exp);
  dterm2:= combine(dterm2, exp);

  if ode::odeIszero(dterm2) and 
     iszero(expand(dterm2,optIgnoreAnalyticConstraints)) then 
    return(FAIL)
  end_if;

  z:= genident(); //solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm0_new:= combine(dterm0_new, exp);
    dterm1_new:= combine(dterm1_new, exp);
  
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 1
  // ---------------------------

  if ode::odeIszero(dterm1_new) and 
     iszero(expand(dterm1_new,optIgnoreAnalyticConstraints)) and 
     has(dterm0_new, exp) then 
    exp_pow:= {};
    dterm0_new:= combine(dterm0_new,exp);
    misc::maprec(combine(dterm0_new,exp),
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then 
      lambda:= exp_pow[1];
      a:= diff(subs(dterm0_new,exp(lambda*x)=z),z);
      if not has(a,x) and not has(a,z) and 
         ode::odeIszero(dterm0_new-a*exp(lambda*x)) and 
         iszero(expand(dterm0_new-a*exp(lambda*x),optIgnoreAnalyticConstraints)) then 
        L:= {besselJ(0,2/lambda*a^(1/2)*exp(1/2*lambda*x)),
             besselY(0,2/lambda*a^(1/2)*exp(1/2*lambda*x))};      
        if nops(L) = 2 then 
          return(L);
        elif nops(L) = 1 then 
          y0:= L[1];
          y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 2
  // ---------------------------

  if ode::odeIszero(dterm1_new) and 
     iszero(expand(dterm1_new,optIgnoreAnalyticConstraints)) and 
     has(dterm0_new,exp) then 
    a:= diff(subs(dterm0_new,exp(x)=z),z);
    if not has(a,x) and not has(a,z) and traperror((b:= eval(-subs(dterm0_new,exp(x)=0)))) = 0 
       and not has(b,x) and not has(b,z) then 
      if ode::odeIszero(dterm0_new-a*exp(x)+b) and 
         iszero(expand(dterm0_new-a*exp(x)+b,optIgnoreAnalyticConstraints)) then 
        L:= {besselJ(2*sqrt(b),2*sqrt(a)*exp(x/2)),
             besselY(2*sqrt(b),2*sqrt(a)*exp(x/2))};
        if nops(L) = 2 then 
          return(L);
        elif nops(L) = 1 then 
          y0:= L[1];
          y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 4
  // ---------------------------

  delete a,b:
  if ode::odeIszero(dterm1_new) and 
     iszero(expand(dterm1_new,optIgnoreAnalyticConstraints)) and 
     has(dterm0_new,exp(x)) and has(dterm0_new,exp(2*x)) then 
    t0:= combine(sqrt(-diff(subs(dterm0_new,exp(2*x)=z),z)),IgnoreAnalyticConstraints);
    if not has(t0,x) and traperror((
         t1:= combine(sqrt(-subs(dterm0_new,[exp(2*x)=0,exp(x)=0])),IgnoreAnalyticConstraints)
       )) = 0 and not has(t1,x) then  
      for a in [t0,-t0] do 
        for b in [t1,-t1] do 
          if ode::odeIszero(dterm0_new+a^2*exp(2*x)+a*(2*b+1)*exp(x)+b^2) and 
             iszero(expand(dterm0_new+a^2*exp(2*x)+a*(2*b+1)*exp(x)+b^2,IgnoreAnalyticConstraints)) then
            y0:= exp(a*exp(x)+b*x);
            y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});   
          end_if;
        end_for;
      end_for;    
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 5
  // ---------------------------

  if ode::odeIszero(dterm1_new) and
     iszero(expand(dterm1_new,optIgnoreAnalyticConstraints)) and 
     has(dterm0_new,exp) then 
    exp_pow:= {};
    misc::maprec(dterm0_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    lambda:= FAIL;
    if nops(exp_pow) = 2 then 
       if iszero(expand(exp_pow[1]-2*exp_pow[2],optIgnoreAnalyticConstraints)) then 
        lambda:= exp_pow[2];
      elif iszero(expand(exp_pow[2]-2*exp_pow[1],optIgnoreAnalyticConstraints)) then 
        lambda:= exp_pow[1];
      end_if;
    elif nops(exp_pow) = 1 then 
      lambda:= ode::normal(exp_pow[1]/2);
    end_if;
    if lambda <> FAIL then 
      a:= -diff(subs(dterm0_new,exp(2*lambda*x)=z),z);
      if not has(a,x) and not has(a,z) then 
        b:= -diff(subs(dterm0_new,exp(lambda*x)=z),z);
        if not has(b,x) and not has(b,z) and 
           traperror((c:= -subs(dterm0_new,[exp(2*lambda*x)=0,exp(lambda*x)=0]))) = 0 then 
          /* 
          if not iszero(lambda) then 
            k:= sqrt(c)/lambda;
            L:= ode::lookUp2ndOrderLinear(//ode::PZ_Sec_212_1_eq_4_and_6_p_214(
                            lambda^2*x*diff(y(x),x,x)+lambda^2*(2*k+1)*diff(y(x),x)-(a*x+b)*y(x),y,x,
                            solveOptions,odeOptions);
            if nops(L) = 2 and not has(L,int) then  
              return(map(L, elem -> ((elem*x^k) | x = exp(lambda*x))));
            end_if;            
          end_if;
           */
          if not iszero(a) and not iszero(b) and traperror((L:= 
               {whittakerM(-b/(2*a^(1/2)*lambda),c^(1/2)/lambda,
                           (2*a^(1/2)*exp(lambda*x))/lambda)/exp((lambda*x)/2), 
                whittakerW(-b/(2*a^(1/2)*lambda),c^(1/2)/lambda,
                           (2*a^(1/2)*exp(lambda*x))/lambda)/exp((lambda*x)/2)}
              )) = 0 then 
            if nops(L) = 2 then 
              return(L);
            elif nops(L) = 1 then 
              y0:= L[1];
              y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          elif iszero(a) and traperror((L:= 
               {besselJ(2/lambda*c^(1/2),2*I/lambda*b^(1/2)*exp(1/2*lambda*x)),
                besselY(2/lambda*c^(1/2),2*I/lambda*b^(1/2)*exp(1/2*lambda*x))}
              )) = 0 then 
            if nops(L) = 2 then 
              return(L);
            elif nops(L) = 1 then 
              y0:= L[1];
              y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          elif iszero(b) and traperror((L:= 
               {besselJ(1/lambda*c^(1/2),I/lambda*a^(1/2)*exp(lambda*x)),
                besselY(1/lambda*c^(1/2),I/lambda*a^(1/2)*exp(lambda*x))}
              )) = 0 then 
            if nops(L) = 2 then 
              return(L);
            elif nops(L) = 1 then 
              y0:= L[1];
              y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          end_if;
        end_if;
      end_if;  
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 6
  // ---------------------------

  if ode::odeIszero(dterm1_new) and 
     iszero(expand(dterm1_new,optIgnoreAnalyticConstraints)) and 
     has(dterm0_new,exp) and type(dterm0_new) = "_plus" then 
    t0:= combine(sqrt(-4*_plus(op(select({op(dterm0_new)},_not@has,x)))),
                 IgnoreAnalyticConstraints);
    for lambda in {t0,-t0} do
      if traperror((a:= diff(subs(dterm0_new,exp(4*lambda*x)=z),z))) = 0 and 
         traperror((b:= diff(subs(dterm0_new,exp(3*lambda*x)=z),z))) = 0 and          
         traperror((c:= diff(subs(dterm0_new,exp(2*lambda*x)=z),z))) = 0 and 
         not has(a,x) and not has(b,x) and not has(c,x) and 
         not has(a,z) and not has(b,z) and not has(c,z) and 
         not iszero(lambda) and 
         ode::odeIszero(dterm0_new-(a*exp(4*lambda*x)+b*exp(3*lambda*x)+c*exp(2*lambda*x)-1/4*lambda^2)) and 
         iszero(expand(dterm0_new-
                       (a*exp(4*lambda*x)+b*exp(3*lambda*x)+c*exp(2*lambda*x)-1/4*lambda^2),
                optIgnoreAnalyticConstraints)) then
        L:= ode::lookUp2ndOrderLinear(//ode::PZ_Sec_212_1_eq_4_and_6_p_214(
                        diff(y(x),x,x)+1/lambda^2*(a*x^2+b*x+c)*y(x),y,x,
                        solveOptions,odeOptions);
        if nops(L) = 2 and not has(L,int) then  
          return(map(L, elem -> (elem | x = exp(lambda*x))*exp(-lambda*x/2)));
        end_if;
      end_if;
    end_for;        
  end_if;
  
  // ---------------------------
  // look-up for eq. 10
  // ---------------------------

  a:= dterm1_new;
  if not has(a,x) and not has(a,z) and has(dterm0_new,exp) then 
    exp_pow:= {};
    misc::maprec(dterm0_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then 
      lambda:= exp_pow[1];
      b:= diff(subs(dterm0_new,exp(lambda*x)=z),z);
      if not has(b,x) and not has(b,z) and 
         traperror((c:= subs(dterm0_new,exp(lambda*x)=0))) = 0 then  
        if traperror((L:= {exp(-a*x/2)*besselJ(sqrt(a^2-4*c)/lambda,2/lambda*sqrt(b)*exp(lambda*x/2)),
               exp(-a*x/2)*besselY(sqrt(a^2-4*c)/lambda,2/lambda*sqrt(b)*exp(lambda*x/2))})) = 0 then 
          if nops(L) = 2 then 
            return(L);
          elif nops(L) = 1 then 
            y0:= L[1];
            y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;
        end_if; 
      end_if; 
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 11
  // ---------------------------

  if ode::odeIszero(1+dterm1_new) and 
     iszero(expand(1+dterm1_new,optIgnoreAnalyticConstraints)) and 
     has(dterm0_new,exp) then 
    exp_pow:= {};
    misc::maprec(dterm0_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    lambda:= FAIL;
    if nops(exp_pow) = 2 then 
       if iszero(expand(2*exp_pow[1]-3*exp_pow[2],optIgnoreAnalyticConstraints)) then 
        lambda:= exp_pow[2]/2;
      elif iszero(expand(2*exp_pow[2]-3*exp_pow[1],optIgnoreAnalyticConstraints)) then 
        lambda:= exp_pow[1]/2;
      end_if;
    end_if;
    if lambda <> FAIL then 
      a:= diff(subs(dterm0_new,exp(3*lambda*x)=z),z);
      b:= diff(subs(dterm0_new,exp(2*lambda*x)=z),z);
      if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and 
         ode::odeIszero(dterm0_new-(a*exp(3*lambda*x)+b*exp(2*lambda*x)+1/4-1/4*lambda^2)) and 
         iszero(expand(dterm0_new-(a*exp(3*lambda*x)+b*exp(2*lambda*x)+1/4-1/4*lambda^2),
                       optIgnoreAnalyticConstraints)) then 
        L:= ode::lookUp2ndOrderLinear(//ode::PZ_Sec_212_1_eq_4_and_6_p_214(
                        x^2*diff(y(x),x,x)+(a*x^(3*lambda)+b*x^(2*lambda)+1/4-1/4*lambda^2)*y(x),y,x,
                        solveOptions,odeOptions);
        if nops(L) = 2 and not has(L,int) then  
          return(map(L, elem -> (elem | x = exp(x))));
        end_if;          
      end_if;
    end_if;  
  end_if;

  // ---------------------------
  // look-up for eq. 15
  // ---------------------------

  if has(dterm1_new,exp) and has(dterm0_new,exp) then 
    dterm0_new:= combine(dterm0_new,exp);
    dterm1_new:= combine(dterm1_new,exp);
    exp_pow:= {};
    misc::maprec(dterm1_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then
      lambda:= exp_pow[1]; 
      a:= diff(subs(dterm1_new,exp(lambda*x)=z),z);
      if not has(a,x) and not has(a,z) and 
         ode::odeIszero(dterm1_new-a*exp(lambda*x)) and 
         iszero(expand(dterm1_new-a*exp(lambda*x),optIgnoreAnalyticConstraints)) then 
        exp_pow:= {};
        misc::maprec(dterm0_new,
                     {"exp"} = proc(elem)
                                    begin
                                      if has(elem,x) and 
                                         iszero(expand(diff(op(elem,1),x,x),
                                                       optIgnoreAnalyticConstraints)) then 
                                        exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                      end_if;
                                      elem;
                                    end_proc);  
        exp_pow:= [op(exp_pow)];
        mu:= FAIL; 
        if nops(exp_pow) = 2 then
          if ode::odeIszero(exp_pow[1]-2*exp_pow[2]) and 
             ode::odeIszero(exp_pow[2]-lambda) and  
             iszero(combine(expand(exp_pow[1]-2*exp_pow[2],optIgnoreAnalyticConstraints),exp)) and 
             iszero(combine(expand(exp_pow[2]-lambda,optIgnoreAnalyticConstraints),exp)) then
            mu:= lambda;
          elif ode::odeIszero(exp_pow[2]-2*exp_pow[1]) and ode::odeIszero(exp_pow[1]-lambda) and 
             iszero(combine(expand(exp_pow[2]-2*exp_pow[1],optIgnoreAnalyticConstraints),exp)) and 
             iszero(combine(expand(exp_pow[1]-lambda,optIgnoreAnalyticConstraints),exp)) then
            mu:= lambda;  
          end_if;
        elif nops(exp_pow) = 3 then 
          ind:= [[2,3,1],[3,2,1],[1,3,2],[3,1,2],[1,2,3],[2,1,3]]:
          for elem in ind do
            if ode::odeIszero(exp_pow[elem[1]]-2*exp_pow[elem[2]]) and 
               ode::odeIszero(exp_pow[elem[3]]-lambda-exp_pow[elem[2]]) and 
               iszero(combine(expand(exp_pow[elem[1]]-2*exp_pow[elem[2]],optIgnoreAnalyticConstraints),exp)) and 
               iszero(combine(expand(exp_pow[elem[3]]-lambda-exp_pow[elem[2]],optIgnoreAnalyticConstraints),exp)) then 
              mu:= exp_pow[elem[2]];
              break;
            end_if;
          end_for;  
        end_if;
        if mu <> FAIL and type(dterm0_new) = "_plus" then
          b:= _plus(op(select({op(combine(expand(dterm0_new/(-mu*exp(mu*x)),exp)))},_not@has, exp)));
          if not has(b,x) and not has(b,z) and 
             ode::odeIszero(dterm0_new+b*exp(mu*x)*(a*exp(lambda*x)+b*exp(mu*x)+mu)) and 
             iszero(combine(expand(dterm0_new+b*exp(mu*x)*(a*exp(lambda*x)+b*exp(mu*x)+mu),
                           optIgnoreAnalyticConstraints),exp)) then 
            y0:= exp(b/mu*exp(mu*x));
            y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});   
          end_if;
        end_if;
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 16
  // ---------------------------

  if has(dterm1_new,exp) and has(dterm0_new,exp) then 
    dterm0_new:= combine(dterm0_new,exp);
    dterm1_new:= combine(dterm1_new,exp);
    exp_pow:= {};
    misc::maprec(dterm1_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then
      mu:= exp_pow[1]; 
      k:= ode::normal(diff(subs(dterm1_new,[exp(expand(mu*x))=z,exp(mu*x)=z]),z)/2);
      if ode::odeIszero(dterm1_new - 2*k*exp(mu*x)) and 
         iszero(expand(dterm1_new - 2*k*exp(mu*x),optIgnoreAnalyticConstraints)) and
         not has(k,x) and not has(k,z) then
        t:= combine(dterm0_new - k^2*exp(2*mu*x) - k*mu*exp(mu*x),exp);
        L:= ode::PZ_Sec_213_1_eq_1_to_79_p_246_to_252(diff(y(x),x,x)+t*y(x),y,x,
                                                 solveOptions,odeOptions);
        if nops(L) = 2 and not has(L,int) then 
          L:= map(L, elem -> elem*exp(-k/mu*exp(mu*x)));
          return(L);
        end_if;
      end_if;
    end_if;
  end_if;  

  // ---------------------------
  // look-up for eq. 19
  // ---------------------------
  if has(dterm1_new,exp) and has(dterm0_new,exp) then 
    dterm0_new:= combine(dterm0_new,exp);
    dterm1_new:= combine(dterm1_new,exp);
    exp_pow:= {};
    misc::maprec(dterm1_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then
      lambda:= exp_pow[1];
      a:= diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z);
      if not has(a,x) and not has(a,z) and 
         ode::odeIszero(dterm1_new-(a*exp(lambda*x)-lambda),exp) and 
         iszero(expand(combine(dterm1_new-(a*exp(lambda*x)-lambda),exp),optIgnoreAnalyticConstraints)) then 
        b:= combine(ode::normal(dterm0_new/exp(2*lambda*x)),exp);
        if not has(b,x) and not has(b,z) and 
           ode::odeIszero(dterm0_new-b*exp(2*lambda*x)) and 
           iszero(expand(combine(dterm0_new-b*exp(2*lambda*x),exp),optIgnoreAnalyticConstraints)) then 
          L:= solve(ode(lambda^2*diff(y(x),x,x)+a*lambda*diff(y(x),x)+b*y(x),y(x)));
          csts:= [op(freeIndets(L) minus (freeIndets(eq)))];
          if nops(csts) = 2 and not has(L,int) then 
            L:= evalAt(L,csts[1]=1,csts[2]=0) union subs(L,csts[1]=0,csts[2]=1);
            L:= map(L, elem -> elem | (x=exp(lambda*x)));
            return(L)      
          end_if;  
        end_if;
      end_if; 
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 22
  // ---------------------------

  if has(dterm1_new,exp) and has(dterm0_new,exp) then 
    dterm0_new:= combine(dterm0_new,exp);
    dterm1_new:= combine(dterm1_new,exp);
    exp_pow:= {};
    misc::maprec(dterm1_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then
      lambda:= exp_pow[1];
      if traperror((b:= subs(dterm1_new+3*lambda,[exp(expand(lambda*x))=0,exp(lambda*x)=0],EvalChanges))) = 0 and 
         not has(b,x) and not has(b,z) and not iszero(b) then 
        a:= ode::normal(diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z)/b);
        if not has(a,x) and 
           ode::odeIszero(dterm1_new-(a*b*exp(lambda*x)+b-3*lambda)) and 
           ode::odeIszero(dterm0_new-a^2*lambda*(b-lambda)*exp(2*lambda*x)) and 
           iszero(expand(combine(dterm1_new-(a*b*exp(lambda*x)+b-3*lambda),exp),optIgnoreAnalyticConstraints)) and 
           iszero(expand(combine(dterm0_new-a^2*lambda*(b-lambda)*exp(2*lambda*x),exp),optIgnoreAnalyticConstraints)) then 
          y0:= (a*exp(lambda*x)+1)*exp(-a*exp(lambda*x));
          y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1})
        end_if;
      end_if;
    end_if;
  end_if;

  // -----------------------------
  // look-up for eq. 23 & 28 & 29
  // -----------------------------

   if has(dterm1_new,exp) and has(dterm0_new,exp) then 
    dterm0_new:= combine(dterm0_new,exp);
    dterm1_new:= combine(dterm1_new,exp);
    exp_pow:= {};
    misc::maprec(dterm1_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then
      lambda:= exp_pow[1];
      a:= diff(subs(ode::normal((dterm1_new+lambda)/2),[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z);
      b:= expand(combine(dterm1_new-2*a*exp(lambda*x)+lambda,exp)/*,optIgnoreAnalyticConstraints*/);
      if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and
         ode::odeIszero(dterm1_new-(2*a*exp(lambda*x)+b-lambda)) and 
         iszero(expand(dterm1_new-(2*a*exp(lambda*x)+b-lambda),optIgnoreAnalyticConstraints)) then 
        t:= combine(dterm0_new-a^2*exp(2*lambda*x)-a*b*exp(lambda*x),exp);
        exp_pow:= {};
        misc::maprec(t,
                     {"exp"} = proc(elem)
                                   begin
                                     if has(elem,x) and 
                                        iszero(expand(diff(op(elem,1),x,x),
                                                      optIgnoreAnalyticConstraints)) then 
                                       exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                     end_if;
                                     elem;
                                   end_proc);  
        exp_pow:= [op(exp_pow)];
        if nops(exp_pow) = 2 and type(t) = "_plus" then 
          mu:= FAIL;
          if iszero(expand(exp_pow[1]-2*exp_pow[2],optIgnoreAnalyticConstraints)) then 
            mu:= exp_pow[2];
          elif iszero(expand(exp_pow[2]-2*exp_pow[1],optIgnoreAnalyticConstraints)) then 
            mu:= exp_pow[1];
          end_if;
        elif nops(exp_pow) = 1 then 
          mu:= exp_pow[1];
        end_if;
        if mu <> FAIL then 
          if nops(exp_pow) = 1 then 
            c:= 0;
          else 
            c:= diff(subs(t,[exp(expand(2*mu*x))=z,exp(2*mu*x)=z]),z);      
          end_if;
          if not has(c,x) and not has(c,z) then 
            d:= diff(subs(t,[exp(expand(mu*x))=z,exp(mu*x)=z]),z);
            if not has(d,x) and not has(d,z) then 
              k:= 0;
              if type(t) = "_plus" then 
                k:= _plus(op(select({op(t)},_not@has,x)));   
              end_if;
              if ode::odeIszero(dterm0_new-(a^2*exp(2*lambda*x)+a*b*exp(lambda*x)+c*exp(2*mu*x)+d*exp(mu*x)+k)) and 
                 iszero(expand(combine(dterm0_new-(a^2*exp(2*lambda*x)+a*b*exp(lambda*x)+c*exp(2*mu*x)+d*exp(mu*x)+k),
                                       exp),optIgnoreAnalyticConstraints)) then 
                L:= ode::PZ_Sec_213_1_eq_1_to_79_p_246_to_252(
                             diff(y(x),x,x)+(c*exp(2*mu*x)+d*exp(mu*x)+k-1/4*(b-lambda)^2)*y(x),y,x,
                             solveOptions,odeOptions);
                if nops(L) = 2 and not has(L,int) then 
                  L:= map(L, elem -> elem*exp(-a/lambda*exp(lambda*x)-(b-lambda)/2*x));
                  return(L);
                end_if;
              end_if;
            end_if;
          end_if;
        end_if;
      end_if;        
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 30
  // ---------------------------

   if has(dterm1_new,exp) and has(dterm0_new,exp) then 
    dterm0_new:= combine(dterm0_new,exp);
    dterm1_new:= combine(dterm1_new,exp);
    exp_pow:= {};
    misc::maprec(dterm1_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 2 then
      for lambda in exp_pow do 
        mu:= op({op(exp_pow)} minus {lambda});
        a:= diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z);
        b:= diff(subs(dterm1_new,[exp(expand(mu*x))=z,exp(mu*x)=z]),z);
        if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and
           ode::odeIszero(dterm1_new-(a*exp(lambda*x)+b*exp(mu*x))) and
           ode::odeIszero(dterm0_new-a*exp(lambda*x)*(b*exp(mu*x)+lambda)) and 
           iszero(combine(expand(dterm1_new-(a*exp(lambda*x)+b*exp(mu*x)),
                                 optIgnoreAnalyticConstraints),exp)) and
           iszero(combine(expand(dterm0_new-a*exp(lambda*x)*(b*exp(mu*x)+lambda),
                          optIgnoreAnalyticConstraints),exp)) then
          y0:= exp(-a/lambda*exp(lambda*x));
          y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1})        
        end_if;
      end_for;
      // ---------------------------
      // look-up for eq. 31
      // ---------------------------
      for lambda in exp_pow do 
        mu:= ode::normal((op({op(exp_pow)} minus {lambda})-lambda)/2);
        a:= diff(subs(dterm1_new,[exp(expand(lambda*x+2*mu*x))=z,exp(lambda*x+2*mu*x)=z]),z);
        b:= diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z);
        if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and
           ode::odeIszero(dterm1_new-exp(lambda*x)*(a*exp(2*mu*x)+b)) and
           ode::odeIszero(dterm0_new-mu*(exp(lambda*x)*(b-a*exp(2*mu*x))-mu)) and 
           iszero(combine(expand(dterm1_new-exp(lambda*x)*(a*exp(2*mu*x)+b),
                                 optIgnoreAnalyticConstraints),exp)) and
           iszero(combine(expand(dterm0_new-mu*(exp(lambda*x)*(b-a*exp(2*mu*x))-mu),
                          optIgnoreAnalyticConstraints),exp)) then
          y0:= a*exp(mu*x)+b*exp(-mu*x);
          y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1})        
        end_if;
      end_for;
      // ---------------------------
      // look-up for eq. 33
      // ---------------------------
      for lambda in exp_pow do 
        mu:= op({op(exp_pow)} minus {lambda});
        a:= diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z);
        b:= diff(subs(dterm1_new,[exp(expand(mu*x))=z,exp(mu*x)=z]),z);
        c:= 0;
        if type(dterm1_new) = "_plus" and nops((L:= select({op(dterm1_new)},_not@has,x))) > 0 then 
          c:= _plus(op(select({op(dterm1_new)},_not@has,x)))
        end_if;
        if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and
           not has(c,x) and not has(c,z) and
           ode::odeIszero(dterm1_new-(a*exp(lambda*x)+b*exp(mu*x)+c)) and
           ode::odeIszero(dterm0_new-(a*b*exp((lambda+mu)*x)+a*c*exp(lambda*x)+b*mu*exp(mu*x))) and 
           iszero(combine(expand(dterm1_new-(a*exp(lambda*x)+b*exp(mu*x)+c),
                                 optIgnoreAnalyticConstraints),exp)) and
           iszero(combine(expand(dterm0_new-(a*b*exp((lambda+mu)*x)+a*c*exp(lambda*x)+b*mu*exp(mu*x)),
                          optIgnoreAnalyticConstraints),exp)) then
          y0:= exp(-b/mu*exp(mu*x)-c*x);
          y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1})        
        end_if;
      end_for;
      // ---------------------------
      // look-up for eq. 34
      // ---------------------------
      for lambda in exp_pow do 
        mu:= op({op(exp_pow)} minus {lambda});
        a:= diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z);
        b:= ode::normal(diff(subs(dterm1_new,[exp(expand(mu*x))=z,exp(mu*x)=z]),z)/2);
        if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and
           ode::odeIszero(dterm1_new-(a*exp(lambda*x)+2*b*exp(mu*x)-lambda)) and 
           iszero(expand(dterm1_new-(a*exp(lambda*x)+2*b*exp(mu*x)-lambda),optIgnoreAnalyticConstraints)) then 
          c:= diff(subs(dterm0_new,[exp(expand(2*lambda*x))=z,exp(2*lambda*x)=z]),z); 
          if not has(c,x) and not has(c,z) and 
             ode::odeIszero(dterm0_new-(a*b*exp((lambda+mu)*x)+c*exp(2*lambda*x)+ 
                       b^2*exp(2*mu*x)+b*(mu-lambda)*exp(mu*x))) and 
             iszero(combine(expand(dterm0_new-(a*b*exp((lambda+mu)*x)+c*exp(2*lambda*x)+
                                       b^2*exp(2*mu*x)+b*(mu-lambda)*exp(mu*x)),
                            optIgnoreAnalyticConstraints),exp)) then
 
            L:= solve(ode(diff(y(z),z,z) + a*diff(y(z),z) + c*y(z),y(z)));
            csts:= [op(freeIndets(L) minus (freeIndets(eq) union {z}))];
            if nops(csts) = 2 then 
              L:= subs(L,csts[1]=1,csts[2]=0) union subs(L,csts[1]=0,csts[2]=1);
              L:= evalAt(L,z=1/lambda*exp(lambda*x));
              L:= map(L, elem -> elem*exp(-b/mu*exp(mu*x)));
              return(L);
            end_if;                 
          end_if;
        end_if;      
      end_for;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 35
  // ---------------------------

   if has(dterm1_new,exp) and has(dterm0_new,exp) then 
    dterm0_new:= combine(dterm0_new,exp);
    dterm1_new:= combine(dterm1_new,exp);
    exp_pow:= {};
    misc::maprec(dterm1_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 3 then
      lambda:= FAIL;
      mu:= FAIL;
      for l in exp_pow do 
        for m in {op(exp_pow)} minus {l} do 
          if iszero(expand(m+l-op({op(exp_pow)} minus {l,m}),
                           optIgnoreAnalyticConstraints)) then 
            lambda:= l; 
            mu:= m;
          end_if;
        end_for;
      end_for;
      if lambda <> FAIL and mu <> FAIL and not iszero(lambda) then 
        a:= ode::normal(diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z)/lambda);
        b:= diff(subs(dterm1_new,[exp(expand(mu*x))=z,exp(mu*x)=z]),z);
        if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and 
           ode::odeIszero(dterm1_new-(a*b*exp((lambda+mu)*x)+a*lambda*exp(lambda*x)+
                                             b*exp(mu*x)-2*lambda)) and 
           ode::odeIszero(dterm0_new-(a^2*b*lambda*exp((2*lambda+mu)*x))) and 
           iszero(combine(expand(dterm1_new-(a*b*exp((lambda+mu)*x)+a*lambda*exp(lambda*x)+
                                             b*exp(mu*x)-2*lambda),
                                 optIgnoreAnalyticConstraints),exp)) and 
           iszero(combine(expand(dterm0_new-(a^2*b*lambda*exp((2*lambda+mu)*x)),
                                 optIgnoreAnalyticConstraints),exp)) then 
          y0:= (a*exp(lambda*x)+1)*exp(-a*exp(lambda*x));
          y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1})        
        end_if;
      end_if;
      t:= lambda;
      lambda:= mu;
      mu:= t;
      if lambda <> FAIL and mu <> FAIL and not iszero(lambda) then 
        a:= ode::normal(diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z)/lambda);
        b:= diff(subs(dterm1_new,[exp(expand(mu*x))=z,exp(mu*x)=z]),z);
        if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and 
           ode::odeIszero(dterm1_new-(a*b*exp((lambda+mu)*x)+a*lambda*exp(lambda*x)+
                                             b*exp(mu*x)-2*lambda)) and 
           ode::odeIszero(dterm0_new-(a^2*b*lambda*exp((2*lambda+mu)*x))) and 
           iszero(combine(expand(dterm1_new-(a*b*exp((lambda+mu)*x)+a*lambda*exp(lambda*x)+
                                             b*exp(mu*x)-2*lambda),
                                 optIgnoreAnalyticConstraints),exp)) and 
           iszero(combine(expand(dterm0_new-(a^2*b*lambda*exp((2*lambda+mu)*x)),
                                 optIgnoreAnalyticConstraints),exp)) then 
          y0:= (a*exp(lambda*x)+1)*exp(-a*exp(lambda*x));
          y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1})        
        end_if;
      end_if;
    end_if;
  end_if;

  // No more ode::normalization needed! Work with the original input data

  // ---------------------------
  // look-up for eq. 37
  // ---------------------------
  
  if has(dterm2,exp) and has(dterm0,exp) and 
     ode::odeIszero(dterm1) and 
     iszero(expand(dterm1,optIgnoreAnalyticConstraints)) then 
    dterm0_new:= combine(dterm0,exp);
    dterm2_new:= combine(dterm2,exp);
    exp_pow:= {};
    misc::maprec(dterm2_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then
      lambda:= exp_pow[1];
      a:= diff(subs(dterm2_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z);
      b:= expand(combine(dterm2_new-a*exp(lambda*x),exp)/*,optIgnoreAnalyticConstraints*/);
      if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and 
         ode::odeIszero(dterm0_new+a*lambda^2*exp(lambda*x)) and 
         iszero(combine(expand(dterm0_new+a*lambda^2*exp(lambda*x),
                               optIgnoreAnalyticConstraints),exp)) then
        y0:= a*exp(lambda*x)+b;
        y1:= y0*Int(1/y0^2,x,intOptions);
        return({y0,y1})        
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 38
  // ---------------------------
  
  if has(dterm2,exp) and has(dterm0,exp) and not iszero(dterm1) and 
     ode::odeIszero(diff(dterm1,x)) and  
     iszero(expand(diff(dterm1,x),optIgnoreAnalyticConstraints)) then 
    dterm0_new:= combine(dterm0,exp);
    dterm1_new:= dterm1;
    dterm2_new:= combine(dterm2,exp);
    exp_pow:= {};
    misc::maprec(dterm2_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then
      lambda:= ode::normal(exp_pow[1]/2);
      a:= combine(sqrt(diff(subs(dterm2_new,[exp(expand(2*lambda*x))=z,exp(2*lambda*x)=z]),z)),
                  IgnoreAnalyticConstraints);
      b:= expand(combine(dterm2_new-a^2*exp(2*lambda*x),exp)/*,optIgnoreAnalyticConstraints*/);
      if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and 
         ode::odeIszero(dterm1_new+b*lambda) and 
         iszero(expand(dterm1_new+b*lambda,optIgnoreAnalyticConstraints)) then 
        k:= combine(sqrt(ode::normal(-dterm0_new/(a^2*lambda^2*exp(2*lambda*x)))),IgnoreAnalyticConstraints);
        if not has(k,x) and not has(k,z) then 
          return({(a*exp(lambda*x)+sqrt(a^2*exp(2*lambda*x)+b))^k,
                  (a*exp(lambda*x)+sqrt(a^2*exp(2*lambda*x)+b))^(-k)})
        end_if;
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 42
  // ---------------------------

  if has(dterm2,exp) and has(dterm1,exp) and has(dterm0,exp) and 
     iszero(expand(diff((k:= (dterm2-exp(x))),x),IgnoreAnalyticConstraints)) then 
    dterm0_new:= combine(dterm0,exp);
    dterm1_new:= combine(dterm1,exp);
    exp_pow:= {};
    misc::maprec(dterm1_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 2 then 
      lambda:= exp_pow[1];
      mu:= exp_pow[2];
      a:= diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z);
      b:= diff(subs(dterm1_new,[exp(expand(mu*x))=z,exp(mu*x)=z]),z);  
      c:= expand(combine(dterm1_new-a*exp(lambda*x)-b*exp(mu*x),exp)/*,optIgnoreAnalyticConstraints*/);
      if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) and 
         ode::odeIszero(diff(c,x)) and 
         ode::odeIszero(dterm0_new-(a*lambda*exp(lambda*x)+b*mu*exp(mu*x)-exp(x))) and 
         iszero(expand(diff(c,x),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0_new-(a*lambda*exp(lambda*x)+b*mu*exp(mu*x)-exp(x)),
                       optIgnoreAnalyticConstraints)) then 
        L:= solve(ode((exp(x)+k)*diff(y(x),x)+(a*exp(lambda*x)+b*exp(mu*x)+c-exp(x))*y(x)=0,
                      y(x)));
        if not has(L,FAIL) and L <> {} then 
          csts:= [op(freeIndets(L) minus (freeIndets(eq)))];
          if nops(csts) = 1 then 
            y0:= op(subs(L,csts[1]=1));
            y1:= y0*Int(exp(-int(dterm1_new/dterm2,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1})        
          end_if;
        end_if;  
      end_if;
    end_if;      
  end_if;

  // ---------------------------
  // look-up for eq. 45
  // ---------------------------

  if has(dterm2,exp) and has(dterm1,exp) and 
     iszero(expand(diff((m:= dterm0),x),IgnoreAnalyticConstraints)) then 
    dterm1_new:= combine(dterm1,exp);
    dterm2_new:= combine(dterm2,exp);
    exp_pow:= {};
    misc::maprec(dterm2_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    lambda:= FAIL;
    if nops(exp_pow) = 2 then 
      if iszero(expand(exp_pow[1]-2*exp_pow[2],optIgnoreAnalyticConstraints)) then 
        lambda:= exp_pow[2];
      elif iszero(expand(exp_pow[2]-2*exp_pow[1],optIgnoreAnalyticConstraints)) then
        lambda:= exp_pow[1];      
      end_if;
      if lambda <> FAIL then 
        a:= combine(sqrt(diff(subs(dterm2_new,[exp(expand(2*lambda*x))=z,exp(2*lambda*x)=z]),z)),
                    IgnoreAnalyticConstraints);
        if not iszero(a) and not has(a,x) and not has(a,z) then 
          b:= ode::normal(diff(subs(dterm2_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z)/(2*a));
          if not has(b,x) and not has(b,z) and 
             ode::odeIszero(dterm2_new-(a*exp(lambda*x)+b)^2) and 
             iszero(combine(expand(dterm2_new-(a*exp(lambda*x)+b)^2,
                                   optIgnoreAnalyticConstraints),exp)) then 
            c:= expand(combine(ode::normal(-(lambda*exp(2*lambda*x)*a^2+b*lambda*exp(lambda*x)*a-
                                 dterm1_new)/(b+a*exp(lambda*x))),
                               exp)/*,optIgnoreAnalyticConstraints*/);
            if not has(c,x) then 
              L:= solve(ode(diff(y(x),x,x)+c*diff(y(x),x)+m*y(x),y(x)));
              csts:= [op(freeIndets(L) minus (freeIndets(eq)))];
              if nops(csts) = 2 then 
                L:= subs(L,csts[1]=1,csts[2]=0) union subs(L,csts[1]=0,csts[2]=1);
                L:= map(L, elem -> elem | x = int(1/(a*exp(lambda*x)+b),x));
                return(L)      
              end_if;  
            end_if;
          end_if;
        end_if;      
      end_if;
    end_if;    
  end_if;

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 48
  // ---------------------------

  if has(dterm1_new,exp) and has(dterm0_new,exp) then 
    dterm1_new:= combine(dterm1_new,exp);
    dterm0_new:= combine(dterm0_new,exp);
    exp_pow:= {};
    misc::maprec(dterm1_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then 
      lambda:= exp_pow[1];
      a:= diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z);
      if not has(a,x) and not has(a,z) and 
         ode::odeIszero(dterm1_new-a*exp(lambda*x)) and 
         iszero(combine(expand(dterm1_new-a*exp(lambda*x),
                                   optIgnoreAnalyticConstraints),exp)) then 
        xpow:= {};
        misc::maprec(dterm0_new,
                     {"_power"} = proc(elem)
                                    begin
                                      if op(elem,1) = x then 
                                        xpow:= xpow union {op(elem,2)}
                                      end_if;
                                      elem;
                                      end_proc);  
        xpow:= [op(xpow)];
        if nops(xpow) = 3 then 
          n:= FAIL;
          for ind in [[2,3,1],[3,2,1],[1,3,2],[3,1,2],[1,2,3],[2,1,3]] do
            if iszero(expand(xpow[ind[2]]-2*xpow[ind[1]],optIgnoreAnalyticConstraints)) and 
               iszero(expand(xpow[ind[3]]-xpow[ind[1]]+1,optIgnoreAnalyticConstraints)) then 
              n:= xpow[ind[1]];
              break;
            end_if;
          end_for;
          if n <> FAIL and n <> -1 then 
            bb:= combine(sqrt(-diff(subs(dterm0_new,[x^(expand(2*n))=z,x^(2*n)=z]),z)),
                         IgnoreAnalyticConstraints);
            for b in {-bb,bb} do 
              if ode::odeIszero(dterm0_new-b*(a*x^n*exp(lambda*x)-b*x^(2*n)+n*x^(n-1))) and 
                 iszero(combine(expand(dterm0_new-b*(a*x^n*exp(lambda*x)-b*x^(2*n)+n*x^(n-1)),
                                     optIgnoreAnalyticConstraints),exp)) then 
                y0:= exp(-b/(n+1)*x^(n+1));
                y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
                return({y0,y1})
              end_if;
            end_for;
          end_if; 
        end_if;        
      end_if;
    end_if;
  end_if;

  // special ode::normalization strategy
  if dterm2 <> x then 
    dterm2_new:= x;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 56
  // ---------------------------

  a:= ode::normal(-diff(dterm1_new,x,x)/4);
  if has(dterm0_new,exp) and not has(a,x) and 
     ode::odeIszero(dterm1_new+(2*a*x^2+1)) and 
     iszero(expand(dterm1_new+(2*a*x^2+1),optIgnoreAnalyticConstraints)) then
    exp_pow:= {};
    misc::maprec(dterm0_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x^2) and 
                                     iszero(expand(diff(op(elem,1),x,x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {ode::normal(diff(op(elem,1),x,x)/4)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then 
      lambda:= exp_pow[1];
      b:= ode::normal(dterm0_new/(4*x^3*exp(2*lambda*x^2)));
      if not has(b,x) and ode::odeIszero(dterm0_new-4*b*x^3*exp(2*lambda*x^2)) and 
         iszero(combine(expand(dterm0_new-4*b*x^3*exp(2*lambda*x^2),
                                     optIgnoreAnalyticConstraints),exp)) then
        y0:= exp(1/2*a*x^2)*besselJ(a/(2*lambda),sqrt(b)*exp(lambda*x^2)/lambda);
        y1:= exp(1/2*a*x^2)*besselY(a/(2*lambda),sqrt(b)*exp(lambda*x^2)/lambda);
        return({y0,y1})
      end_if;
    end_if;      
  end_if;
 
  // ---------------------------
  // look-up for eq. 66
  // ---------------------------

  if has(dterm0_new,exp) and has(dterm1_new,exp) then
    exp_pow:= {};
    misc::maprec(dterm0_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) then 
                                    exp_pow:= {op(elem,1)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then 
      xpow:= {};
      misc::maprec(exp_pow[1],
                   {"_power"} = proc(elem)
                                  begin
                                    if op(elem,1) = x then 
                                      xpow:= xpow union {op(elem,2)}
                                    end_if;
                                    elem;
                                    end_proc);  
      xpow:= [op(xpow)];
      if xpow = [] then 
        xpow:= [1];
      end_if;      
      if nops(xpow) = 1 then
        m:= xpow[1]; 
        b:= diff(subs(exp_pow[1],x^m=z),z);
        if not has(b,x) and not has(b,z) and 
           ode::odeIszero(exp_pow[1]-b*x^m) and 
           iszero(expand(exp_pow[1]-b*x^m,optIgnoreAnalyticConstraints)) then 
          t:= diff(subs(dterm1_new,[exp(b*x^m)=z,exp(expand(b*x^m))=z]),z); 
          xpow:= {};
          misc::maprec(t,
                       {"_power"} = proc(elem)
                                       begin
                                         if op(elem,1) = x then 
                                           xpow:= xpow union {op(elem,2)}
                                         end_if;
                                         elem;
                                         end_proc);  
          xpow:= [op(xpow)];
          if xpow = [] then 
            xpow:= [1];
          end_if;
          if nops(xpow) = 1 then
            n:= xpow[1];
            a:= diff(subs(t,x^n=z),z); 
            if not has(a,x) and 
               traperror((c:= evalAt(dterm1_new,[exp(b*x^m)=0,exp(expand(b*x^m))=0]))) = 0 and 
               ode::odeIszero(dterm1_new-(a*x^n*exp(b*x^m)+c)) and 
               ode::odeIszero(dterm0_new-a*(c-1)*x^(n-1)*exp(b*x^m)) and 
               iszero(combine(expand(dterm1_new-(a*x^n*exp(b*x^m)+c),optIgnoreAnalyticConstraints),exp)) and 
               iszero(combine(expand(dterm0_new-a*(c-1)*x^(n-1)*exp(b*x^m),optIgnoreAnalyticConstraints),exp)) then 
              y0:= x^(1-c);
              y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1})
            end_if;          
          end_if;
        end_if; 
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 68
  // ---------------------------

  if ode::odeIszero(dterm1) and iszero(expand(dterm1,optIgnoreAnalyticConstraints)) then 
    // special ode::normalization strategy
    if dterm2 <> 4*x^2 then 
      dterm2_new:= 4*x^2;
      dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    else
      dterm2_new:= dterm2;
      dterm0_new:= dterm0;
    end_if;    
    exp_pow:= {};
    misc::maprec(dterm0_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) then 
                                    exp_pow:= {op(elem,1)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then 
      xpow:= {};
      misc::maprec(exp_pow[1],
                   {"_power"} = proc(elem)
                                  begin
                                    if op(elem,1) = x then 
                                      xpow:= xpow union {op(elem,2)}
                                    end_if;
                                    elem;
                                    end_proc);  
      xpow:= [op(xpow)];
      if xpow = [] then 
        xpow:= [1];
      end_if;      
      if nops(xpow) = 1 then
        n:= xpow[1]; 
        b:= diff(subs(exp_pow[1],x^n=z),z);
        a:= ode::normal(diff(subs(dterm0_new,[exp(b*x^n)=z,exp(expand(b*x^n))=z,
                                         x^(2*n)=z,x^(expand(2*n))=z]),z,z)/2); 
        if not has(a,x) and not has(a,z) and not iszero(b*n) and 
           ode::odeIszero(dterm0_new-(a*x^(2*n)*exp(b*x^n)+1-n^2)) and 
           iszero(combine(expand(dterm0_new-(a*x^(2*n)*exp(b*x^n)+1-n^2),
                                 optIgnoreAnalyticConstraints),exp)) and  
           not iszero(combine(expand(eq-(4*y2+a*(b*n)^(-2)*exp(x)*y0),optIgnoreAnalyticConstraints),exp)) then 
          L:= ode::PZ_Sec_213_1_eq_1_to_79_p_246_to_252(
                   4*diff(y(x),x,x)+a*(b*n)^(-2)*exp(x)*y(x),y,x,
                                                   solveOptions,odeOptions);
          if nops(L) = 2 and not has(L,int) then 
            L:= map(L, elem -> evalAt(elem, x=b*x^n)*x^((1-n)/2));
            return(L);
          end_if;
        end_if;
      end_if;
    end_if; 
  end_if;

  // special ode::normalization strategy
  if dterm2 <> x^2 then 
    dterm2_new:= x^2;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 70
  // ---------------------------

  if has(dterm0_new,exp) and has(dterm1_new,exp) then
    exp_pow:= {};
    misc::maprec(dterm1_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then 
      lambda:= exp_pow[1];
      a:= ode::normal(diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z)/x);
      if not iszero(a) and not has(a,x) and not has(a,z) and 
         ode::odeIszero(dterm1_new-a*x*exp(lambda*x)) and 
         iszero(combine(expand(dterm1_new-a*x*exp(lambda*x),optIgnoreAnalyticConstraints),exp)) then 
        b:= ode::normal(diff(subs(dterm0_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z)/a);
        if not has(a,x) and not has(a,z) and 
           ode::odeIszero(dterm0_new-b*(a*exp(lambda*x)-b-1)) and 
           iszero(combine(expand(dterm0_new-b*(a*exp(lambda*x)-b-1),
                                 optIgnoreAnalyticConstraints),exp)) then 
          y0:= x^(-b);
          y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
        end_if;
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 71
  // ---------------------------
  
  if has(dterm0_new,exp) and has(dterm1_new,exp) then
    exp_pow:= {};
    misc::maprec(dterm1_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then 
      lambda:= exp_pow[1];
      a:= ode::normal(diff(subs(dterm1_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z)/x);
      if not iszero(a) and not has(a,x) and not has(a,z) then 
        b:= ode::normal(diff(combine(expand(dterm1_new-x*a*exp(lambda*x)
                                       /*,optIgnoreAnalyticConstraints*/),exp),x)/2);
        if not has(b,x) and
           ode::odeIszero(dterm1_new-x*(a*exp(lambda*x)+2*b)) and 
           iszero(combine(expand(dterm1_new-x*(a*exp(lambda*x)+2*b),
                                 optIgnoreAnalyticConstraints),exp)) then
          c:= ode::normal(diff(diff(subs(dterm0_new,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z),x)/a);
          if not has(c,x) and not has(c,z) and 
             ode::odeIszero(dterm0_new-(a*(c*x+b)*exp(lambda*x)-c^2*x^2+b*(b-1))) and 
             iszero(combine(expand(dterm0_new-(a*(c*x+b)*exp(lambda*x)-c^2*x^2+b*(b-1)),
                                 optIgnoreAnalyticConstraints),exp)) then           
            y0:= x^(-b)*exp(-c*x);
            y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});            
          end_if;
        end_if;
      end_if; 
    end_if;
  end_if;  

  // -------------------------------------
  // look-up for eq. 73 (includes eq. 72)
  // -------------------------------------

  if ode::odeIszero(dterm1) and iszero(expand(dterm1,optIgnoreAnalyticConstraints)) and 
     has(dterm0,exp) then 
    // special ode::normalization strategy
    if dterm2 <> x^4 then 
      dterm2_new:= x^4;
      dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
      dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    else
      dterm2_new:= dterm2;
      dterm1_new:= dterm1;
      dterm0_new:= dterm0;
    end_if;    
    exp_pow:= {};
    misc::maprec(dterm0_new,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(ode::normal(1/op(elem,1)),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {ode::normal(1/diff(ode::normal(1/op(elem,1)),x))}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    lambda:= FAIL;
    if nops(exp_pow) = 2 then 
      if iszero(expand(exp_pow[1]-2*exp_pow[2],optIgnoreAnalyticConstraints)) then
        lambda:= exp_pow[2];
      elif iszero(expand(exp_pow[2]-2*exp_pow[1],optIgnoreAnalyticConstraints)) then
        lambda:= exp_pow[1];
      end_if;
    elif nops(exp_pow) = 1 then 
      lambda:= exp_pow[1];
    end_if;
    if lambda <> FAIL then  
      a:= diff(subs(dterm0_new,[exp(expand(2*lambda/x))=z,exp(2*lambda/x)=z]),z);
      b:= diff(subs(dterm0_new,[exp(expand(lambda/x))=z,exp(lambda/x)=z]),z);
      c:= combine(expand(dterm0_new-a*exp(2*lambda/x)-b*exp(lambda/x)
                         /*,optIgnoreAnalyticConstraints*/),exp);
      if not has(a,x) and not has(a,z) and 
         not has(b,x) and not has(b,z) and 
         not has(c,x) and not has(c,z) and
         ode::odeIszero(dterm0_new-(a*exp(2*lambda/x)+b*exp(lambda/x)+c)) and 
         iszero(combine(expand(dterm0_new-(a*exp(2*lambda/x)+b*exp(lambda/x)+c),
                              optIgnoreAnalyticConstraints),exp)) then 
        L:= ode::PZ_Sec_213_1_eq_1_to_79_p_246_to_252(
                 diff(y(x),x,x)+(a*exp(2*lambda*x)+b*exp(lambda*x)+c)*y(x),y,x,
                                                 solveOptions,odeOptions);
        if nops(L) = 2 and not has(L,int) then 
          L:= map(L, elem -> evalAt(elem, x=1/x)*x);
          return(L);
        end_if;
      end_if;
    end_if;
  end_if;

  // no ode::normalization here!!! work on roginal data

  if ode::odeIszero(dterm1) and ode::odeIszero(diff(dterm0,x)) and 
     iszero(expand(dterm1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(diff(dterm0,x),optIgnoreAnalyticConstraints)) and 
     has(dterm2,exp) then 
    exp_pow:= {};
    misc::maprec(dterm2,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then 
      lambda:= exp_pow[1];
      t:= diff(subs(dterm2,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z);
      a:= diff(t,x);
      b:= t-a*x;
      if not has(a,x) and not has(a,z) and not has(b,x) and not has(b,z) then 
        c:= combine(expand(dterm2-(a*x+b)*exp(lambda*x)/*,optIgnoreAnalyticConstraints*/),exp);
        if not has(c,x) and ode::odeIszero(dterm0+c*lambda^2) and 
           iszero(expand(dterm0+c*lambda^2,optIgnoreAnalyticConstraints)) then
          y0:= c*exp(-lambda*x)+a*x+b;
          y1:= y0*Int(1/y0^2,x,intOptions);
          return({y0,y1});            
        end_if;
      end_if;
    end_if;      
  end_if;

  // ---------------------------

  return(FAIL);
end_proc:


/*
  ----------
  REFERENCE. Implementation of (Section 2.1.5.-1. eq. 1 to 40 p. 257 to 259)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/


ode::PZ_Sec_215_1_eq_1_to_40_p_257_to_259:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, optIgnoreAnalyticConstraints,
        Int, dterm0_new, dterm1_new, dterm2_new, z, t, a, b, ln_pow, ln_arg, xpow, n,
        L, csts, c, m, d, lambda;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   

  Int:= if has(odeOptions,freeze) then freeze(int) else int end_if;

  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) or (not has(eq,y0) and not has(eq,y1)) then 
    return(FAIL)
  end_if;
  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);

  if ode::odeIszero(dterm2) and 
     iszero(expand(dterm2,optIgnoreAnalyticConstraints)) then 
    return(FAIL)
  end_if;

  z:= genident(); //solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 1
  // ---------------------------

  if ode::odeIszero(dterm1_new) and 
     iszero(expand(dterm1_new,optIgnoreAnalyticConstraints)) and has(dterm0_new,ln) then 
    t:= subs(dterm0_new,ln(x)=z);
    if ode::odeIszero(diff(t,z,z,z)) and 
       iszero(expand(diff(t,z,z,z),optIgnoreAnalyticConstraints)) then 
      a:= ode::normal(-(t | z=0));
      if not has(a,x) and ode::odeIszero(dterm0_new+(a^2*x^2*ln(x)^2+a*ln(x)+a)) and 
         iszero(expand(dterm0_new+(a^2*x^2*ln(x)^2+a*ln(x)+a,
                                        optIgnoreAnalyticConstraints))) then 
        y0:= exp(-a*x^2/4)*x^(a*x^2/2);
        y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1})        
      end_if;
      // ---------------------------
      // look-up for eq. 2 & 3
      // ---------------------------
      t:= combine(ode::normal(-(t | z=0)),optIgnoreAnalyticConstraints);
      xpow:= {};
      if has(t,x) then 
        misc::maprec(t,
                     {"_power"} = proc(elem)
                                    begin
                                      if op(elem,1) = x then 
                                        xpow:= xpow union {op(elem,2)}
                                      end_if;
                                      elem;
                                      end_proc);  
        xpow:= [op(xpow)];
      end_if;
      if nops(xpow) = 0 and ode::odeIszero(diff(t,x,x)) and 
         iszero(expand(diff(t,x,x),optIgnoreAnalyticConstraints)) then 
        xpow:= [1];          
      end_if;
      if nops(xpow) = 1 then 
        n:= xpow[1]+1;
        a:= diff(subs(t,[x^(n-1)=z,x^(expand(n-1))=z]),z); 
        if ode::odeIszero(dterm0_new+(a^2*x^(2*n)*ln(x)^2+a*n*x^(n-1)*ln(x)+a*x^(n-1))) and 
           iszero(expand(dterm0_new+(a^2*x^(2*n)*ln(x)^2+a*n*x^(n-1)*ln(x)+a*x^(n-1)),
                         optIgnoreAnalyticConstraints)) and n <> -1 then 
          y0:= exp(-a*x^(n+1)/(n+1)^2)*x^(a*x^(n+1)/(n+1));
          y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1})        
        end_if;
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 4
  // ---------------------------

  // original data; special ode::normalization necessary
  dterm0_new:= diff(eq,y0);
  dterm1_new:= diff(eq,y1);
  dterm2_new:= diff(eq,y2);

  if dterm2_new <> x then 
    dterm1_new:= combine(ode::normal(x*dterm1_new/dterm2_new),optIgnoreAnalyticConstraints);
    dterm0_new:= combine(ode::normal(x*dterm0_new/dterm2_new),optIgnoreAnalyticConstraints);
    dterm2_new:= x;
  end_if;    

  dterm0_new:= combine(dterm0_new);

  if ode::odeIszero(dterm1_new) and 
     iszero(expand(dterm1_new,optIgnoreAnalyticConstraints)) and has(dterm0_new,ln) then 
    ln_arg:= {};
    misc::maprec(dterm0_new,
                 {"ln"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    ln_arg:= ln_arg union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    ln_arg:= [op(ln_arg)];
    if nops(ln_arg) = 1 then 
      b:= ln_arg[1];
      ln_pow:= {};
      misc::maprec(dterm0_new,
                   {"_power"} = proc(elem)
                                  begin
                                    if op(elem,1) = ln(b*x) then 
                                      ln_pow:= ln_pow union {op(elem,2)}
                                    end_if;
                                    elem;
                                    end_proc);  
      ln_pow:= [op(ln_pow)];
      n:= FAIL;
      if ln_pow = [2] then 
        n:= 2;
      elif nops(ln_pow) = 2 then 
        if iszero(expand(ln_pow[1]-2*(ln_pow[2]+1),optIgnoreAnalyticConstraints)) then 
          n:= ode::normal(ln_pow[1]/2);
        elif iszero(expand(ln_pow[2]-2*(ln_pow[1]+1),optIgnoreAnalyticConstraints)) then
          n:= ode::normal(ln_pow[2]/2);
        end_if;
      end_if;
      if n <> FAIL then 
        a:= ode::normal(-diff(subs(dterm0_new,[ln(b*x)^(n-1)=z,ln(expand(b*x))^(n-1)=z]),z)/n);
        if not has(a,x) and not has(a,z) then 
          if ode::odeIszero(dterm0_new+(a^2*x*ln(b*x)^(2*n)+a*n*ln(b*x)^(n-1))) and 
             iszero(expand(dterm0_new+(a^2*x*ln(b*x)^(2*n)+a*n*ln(b*x)^(n-1)),optIgnoreAnalyticConstraints)) then 
            y0:= exp(a*int(ln(b*x)^n,x));
            y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1})        
          end_if;
        end_if;
      end_if;
    end_if;   
  end_if;

  if iszero(dterm1_new) and has(dterm0_new,ln) then 
    // special ode::normalization strategy
    if dterm2 <> x^2 then 
      dterm2_new:= x^2;
      dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    else
      dterm2_new:= dterm2;
      dterm0_new:= dterm0;
    end_if;    
    // ---------------------------
    // look-up for eq. 5
    // ---------------------------
    a:= diff(subs(dterm0_new,ln(x)=z),z);
    if not iszero(a) and not has(a,x) and not has(a,z) then 
      b:= dterm0_new -a*ln(x);
      if ode::odeIszero(diff(b,x)) and 
         iszero(expand(diff(b,x),optIgnoreAnalyticConstraints))  then 
        L:= solve(ode(diff(y(x),x,x)+x/a^2*y(x),y(x)));
        csts:= [op(indets(L) minus (indets(eq)))];
        if nops(csts) = 2 and not has(L,int) then 
          L:= evalAt(L,csts[1]=1,csts[2]=0) union subs(L,csts[1]=0,csts[2]=1);
          if traperror((L:= map(L, elem -> (elem | (x=a*ln(x)+b-1/4))*x^(1/2)))) = 0 then 
            return(L)      
          end_if;
        end_if;  
      end_if;
    end_if;
    // ---------------------------
    // look-up for eq. 6
    // ---------------------------
    t:= evalAt(dterm0_new,ln(x)=z);
    if ode::odeIszero(diff(t,z,z,z)) and 
       iszero(expand(diff(t,z,z,z),optIgnoreAnalyticConstraints)) then 
      a:= ode::normal(diff(t,z,z)/2);
      b:= diff(t,z) | z=0;
      c:= t | z=0;
      if not has(a,x) and not has(b,x) and not has(c,x) and 
         ode::odeIszero(dterm0_new-(a*ln(x)^2+b*ln(x)+c)) and 
         iszero(expand(dterm0_new-(a*ln(x)^2+b*ln(x)+c),optIgnoreAnalyticConstraints)) then 
        L:= solve(ode(diff(y(x),x,x)+(a*x^2+b*x+c-1/4)*y(x),y(x)));
        csts:= [op(freeIndets(L) minus (freeIndets(eq)))];
        if nops(csts) = 2 and not has(L,int) then 
          L:= evalAt(L,csts[1]=1,csts[2]=0) union subs(L,csts[1]=0,csts[2]=1);
          if traperror((L:= map(L, elem -> (elem | x=ln(x))*x^(1/2)))) = 0 then 
            return(L)      
          end_if;
        end_if;          
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 7
  // ---------------------------

  // original data; special ode::normalization necessary
  dterm0_new:= diff(eq,y0);
  dterm1_new:= diff(eq,y1);
  dterm2_new:= diff(eq,y2);

  if dterm2_new <> x^2 then 
    dterm1_new:= combine(ode::normal(x^2*dterm1_new/dterm2_new),optIgnoreAnalyticConstraints);
    dterm0_new:= combine(ode::normal(x^2*dterm0_new/dterm2_new),optIgnoreAnalyticConstraints);
    dterm2_new:= x^2;
  end_if;    

  ln_arg:= {};
  misc::maprec(dterm0_new,
               {"_power"} = proc(elem)
                              begin
                                if has(op(elem,1),ln(x)) and 
                                   iszero(expand(diff(subs(op(elem,1),ln(x)=z),z,z),
                                                 optIgnoreAnalyticConstraints)) then 
                                  ln_arg:= ln_arg union {[op(elem,1),op(elem,2)]}
                                end_if;
                                elem;
                                end_proc);  
  ln_arg:= [op(ln_arg)];
  if nops(ln_arg) = 1 then 
    n:= ln_arg[1][2];
    b:= diff(subs(ln_arg[1][1],ln(x)=z),z);
    c:= combine(expand(ln_arg[1][1]-b*ln(x),
                       optIgnoreAnalyticConstraints),
                optIgnoreAnalyticConstraints);
    a:= ode::normal((dterm0_new-1/4)/(b*ln(x)+c)^n);
    if not has(a,x) and not iszero(b) and  
       ode::odeIszero(dterm0_new-(a*(b*ln(x)+c)^n+1/4)) and 
       iszero(expand(dterm0_new-(a*(b*ln(x)+c)^n+1/4),optIgnoreAnalyticConstraints)) then 
      L:= solve(ode(diff(y(x),x,x)+a/b^2*x^n*y(x),y(x)));
      csts:= [op(freeIndets(L) minus (freeIndets(eq)))];
      if nops(csts) = 2 and not has(L,int) then 
        L:= evalAt(L,csts[1]=1,csts[2]=0) union subs(L,csts[1]=0,csts[2]=1);
        if traperror((L:= map(L, elem -> (elem | x=b*ln(x)+c)*x^(1/2)))) = 0 then 
          return(L)      
        end_if;
      end_if;                  
    end_if;
  end_if;    

  // ---------------------------
  // look-up for eq. 8
  // ---------------------------

  // original data; special ode::normalization necessary
  dterm0_new:= diff(eq,y0);
  dterm1_new:= diff(eq,y1);
  dterm2_new:= diff(eq,y2);

  if ode::odeIszero(dterm1_new) and 
     ode::odeIszero(dterm0_new-1) and 
     iszero(expand(dterm1_new,optIgnoreAnalyticConstraints)) and 
     iszero(expand(dterm0_new-1,optIgnoreAnalyticConstraints)) and 
     has(dterm2_new,x^2) and has(dterm2_new,ln) then 
    t:= ode::normal(dterm2_new/x^2);
    if type(t) = "ln" then 
      t:= op(t,1);
      if ode::odeIszero(diff(t,x,x)) and 
         iszero(expand(diff(t,x,x),optIgnoreAnalyticConstraints)) then 
        a:= diff(t,x);
        if ode::odeIszero(dterm2_new-x^2*ln(a*x)) and 
           iszero(expand(dterm2_new-x^2*ln(a*x),optIgnoreAnalyticConstraints)) then 
          return({ln(a*x),ln(a*x)*int(1/ln(a*x)^2,x)})
        end_if; 
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 9
  // ---------------------------
    
  if ode::odeIszero(dterm1) and
     ode::odeIszero(diff(dterm0,x)) and 
     iszero(expand(dterm1,optIgnoreAnalyticConstraints)) and
     iszero(expand(diff(dterm0,x),optIgnoreAnalyticConstraints)) then
    a:= -dterm0;
    t:= expand(dterm2-a*x^2*ln(x)/*,optIgnoreAnalyticConstraints*/);
    if ode::odeIszero(diff(t,x,x,x)) and 
       iszero(expand(diff(t,x,x,x),optIgnoreAnalyticConstraints)) then
      b:= ode::normal(diff(t,x,x)/2);
      c:= expand(diff(t-b*x^2,x)/*,optIgnoreAnalyticConstraints*/);
      if ode::odeIszero(dterm2-x*(a*x*ln(x)+b*x+c)) and 
         iszero(expand(dterm2-x*(a*x*ln(x)+b*x+c),optIgnoreAnalyticConstraints)) then 
        y0:= a*x*ln(x)+b*x+c;
        y1:= y0*Int(1/y0^2,x,intOptions);
        return({y0,y1})        
      end_if;
    end_if;
    // ---------------------------
    // look-up for eq. 10
    // ---------------------------
    a:= dterm0;
    t:= expand(dterm2-a*x^2*ln(x)/*,optIgnoreAnalyticConstraints*/);
    if ode::odeIszero(diff(t,x,x,x,x)) and 
       iszero(expand(diff(t,x,x,x,x),optIgnoreAnalyticConstraints)) then
      b:= ode::normal(diff(t,x,x,x)/6);
      c:= expand(diff(t-b*x^3,x,x)/2/*,optIgnoreAnalyticConstraints*/);
      if ode::odeIszero(dterm2-x^2*(a*ln(x)+b*x+c)) and 
         iszero(expand(dterm2-x^2*(a*ln(x)+b*x+c),optIgnoreAnalyticConstraints)) then 
        y0:= a*ln(x)+b*x+c;
        y1:= y0*Int(1/y0^2,x,intOptions);
        return({y0,y1})        
      end_if;
    end_if;
  end_if;

  // special ode::normalization strategy
  if dterm2 <> x then 
    dterm2_new:= x;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 17
  // ---------------------------

  if has(dterm1_new,ln) and has(dterm0_new,ln) then 
    a:= ode::normal(diff(subs(dterm1_new,ln(x)=z),z)/(2*x));
    if not has(a,x) and 
       ode::odeIszero(dterm1_new-(2*a*x*ln(x)+1)) and 
       ode::odeIszero(dterm0_new-(a^2*x*ln(x)^2+a*ln(x)+a)) and 
       iszero(expand(dterm1_new-(2*a*x*ln(x)+1),optIgnoreAnalyticConstraints)) and 
       iszero(expand(dterm0_new-(a^2*x*ln(x)^2+a*ln(x)+a),optIgnoreAnalyticConstraints)) then 
      return({exp(a*x)*x^(-a*x),exp(a*x)*x^(-a*x)*ln(x)});
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 18
  // ---------------------------
  
  if has(dterm1_new,ln) and has(dterm0_new,ln) then 
    t:= diff(subs(dterm1_new,ln(x)=z),z);
    if not has(t,z) and 
       ode::odeIszero(diff(t,x,x)) and 
       iszero(expand(diff(t,x,x),optIgnoreAnalyticConstraints)) then 
      a:= diff(t,x);
      b:= expand(t-a*x/*,optIgnoreAnalyticConstraints*/);
      if not has(a,x) and 
         ode::odeIszero(dterm1_new-ln(x)*(a*x+b)) and ode::odeIszero(dterm0_new-a*(b*ln(x)^2+1)) and 
         iszero(expand(dterm1_new-ln(x)*(a*x+b),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0_new-a*(b*ln(x)^2+1),optIgnoreAnalyticConstraints)) then 
        y0:= exp(a*x)*x^(-a*x);
        y1:= y0*Int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1})
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 27
  // ---------------------------

  // original data; special ode::normalization necessary
  dterm0_new:= diff(eq,y0);
  dterm1_new:= diff(eq,y1);
  dterm2_new:= diff(eq,y2);

  dterm0_new:= combine(dterm0_new,ln);

  if dterm2_new <> x^2 then 
    dterm1_new:= combine(ode::normal(x^2*dterm1_new/dterm2_new),optIgnoreAnalyticConstraints);
    dterm0_new:= combine(ode::normal(x^2*dterm0_new/dterm2_new),ln);
    dterm2_new:= x^2;  
  end_if;    

  if ode::odeIszero(dterm1_new-x) and 
     iszero(expand(dterm1_new-x,optIgnoreAnalyticConstraints)) then 
    ln_arg:= {};
    misc::maprec(dterm0_new,
                 {"ln"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    ln_arg:= ln_arg union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    ln_arg:= [op(ln_arg)];
    if nops(ln_arg) = 1 then 
      b:= ln_arg[1];
      ln_pow:= {};
      misc::maprec(dterm0_new,
                   {"_power"} = proc(elem)
                                  begin
                                    if op(elem,1) = ln(b*x) then 
                                      ln_pow:= ln_pow union {op(elem,2)}
                                    end_if;
                                    elem;
                                    end_proc);  
      ln_pow:= [op(ln_pow)];
      n:= FAIL;
      if nops(ln_pow) = 1 then 
        n:= ln_pow[1];
      elif ln_pow = [] then 
        n:= 1;
      end_if;
      if n <> FAIL then 
        a:= expand(diff(subs(dterm0_new,ln(b*x)^n=z),z)/*,optIgnoreAnalyticConstraints*/):
        m:= 1/2*(n+2);
        if not has(a,x) and not has(a,z) and not iszero(m) and 
           ode::odeIszero(dterm0_new-a*ln(b*x)^n) and 
           iszero(expand(dterm0_new-a*ln(b*x)^n,optIgnoreAnalyticConstraints)) then 
          return({sqrt(ln(b*x))*besselJ(1/(2*m),sqrt(a)/m*ln(b*x)^m), 
                  sqrt(ln(b*x))*besselY(1/(2*m),sqrt(a)/m*ln(b*x)^m)})
        end_if;
      end_if;
    end_if;            
  end_if;

  // ode::normalization 
  if dterm2 <> x^2 then 
    dterm2_new:= x^2;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // ---------------------------
  // look-up for eq. 28
  // ---------------------------

  if ode::odeIszero(dterm1_new-x) and  
     iszero(expand(dterm1_new-x,optIgnoreAnalyticConstraints)) then 
    ln_pow:= {};
    misc::maprec(dterm0_new,
                 {"_power"} = proc(elem)
                                begin
                                  if op(elem,1) = ln(x) then 
                                    ln_pow:= ln_pow union {op(elem,2)}
                                  end_if;
                                  elem;
                                  end_proc);  
    ln_pow:= [op(ln_pow)];
    n:= FAIL;
    if ln_pow = [2] then 
      n:= 1;
    elif ln_pow = [4] then 
      n:= 2
    elif nops(ln_pow) = 2 then 
      if iszero(expand(ln_pow[1]-2*(ln_pow[2]+1),optIgnoreAnalyticConstraints)) then 
        n:= ode::normal(ln_pow[1]/2);
      elif iszero(expand(ln_pow[2]-2*(ln_pow[1]+1),optIgnoreAnalyticConstraints)) then
        n:= ode::normal(ln_pow[2]/2);
      end_if;
    end_if;
    if n <> FAIL then 
      a:= diff(subs(dterm0_new,[ln(x)^(2*n)=z,ln(x)^(expand(2*n))=z]),z);
      b:= FAIL;
      if n <> 2 and n <> 1 then  
        b:= diff(subs(dterm0_new,[ln(x)^(n-1)=z,ln(x)^(expand(n-1))=z]),z);
      elif n = 1 then 
        b:= expand(dterm0_new-a*ln(x)^(2*n)/*,optIgnoreAnalyticConstraints*/);
      elif traperror((evalAt(diff(subs(dterm0_new,ln(x)=z),z),z=0))) = 0 then 
        b:= evalAt(diff(subs(dterm0_new,ln(x)=z),z),z=0);          
      end_if;
      if b <> FAIL and ode::odeIszero(dterm0_new-(a*ln(x)^(2*n)+b*ln(x)^(n-1))) and 
         iszero(expand(dterm0_new-(a*ln(x)^(2*n)+b*ln(x)^(n-1)),
                       optIgnoreAnalyticConstraints)) then 
        L:= solve(ode(diff(y(x),x,x)+(a*x^(2*n)+b*x^(n-1))*y(x),y(x)));
        csts:= [op(freeIndets(L) minus (freeIndets(eq)))];
        if nops(csts) = 2 and not has(L,int) then 
          L:= evalAt(L,csts[1]=1,csts[2]=0) union subs(L,csts[1]=0,csts[2]=1);
          if traperror((L:= map(L, elem -> (elem | x=ln(x))))) = 0 then 
            return(L)      
          end_if;
        end_if;                  
      end_if;
    end_if;    
  end_if;
   
  // ---------------------------
  // look-up for eq. 30
  // ---------------------------

  if has(dterm1_new,ln) and has(dterm0_new,ln) then 
    t:= subs(dterm0_new,[ln(x)^2=z^2,ln(x)=z]);
    if ode::odeIszero(diff(t,z,z,z)) and 
       iszero(expand(diff(t,z,z,z),optIgnoreAnalyticConstraints)) then 
      a:= diff(t,z) | z=0;
      b:= t | z=0;
      if not has(a,x) and not has(b,x) and 
         ode::odeIszero(dterm1_new-x*(2*ln(x)+a+1)) and 
         ode::odeIszero(dterm0_new-(ln(x)^2+a*ln(x)+b)) and 
         iszero(expand(dterm1_new-x*(2*ln(x)+a+1),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0_new-(ln(x)^2+a*ln(x)+b),optIgnoreAnalyticConstraints)) then 
        L:= solve(ode(diff(y(x),x,x)+a*diff(y(x),x)+(b-1)*y(x),y(x)));
        csts:= [op(freeIndets(L) minus (freeIndets(eq)))];
        if nops(csts) = 2 and not has(L,int) then 
          L:= evalAt(L,csts[1]=1,csts[2]=0) union subs(L,csts[1]=0,csts[2]=1);
          if traperror((L:= map(L, elem -> (elem | x=ln(x))*exp(-1/2*ln(x)^2)))) = 0 then 
            return(L)      
          end_if;
        end_if;                  
      end_if;
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 34
  // ---------------------------

  if ode::odeIszero(diff(dterm2,x,x,x)) and 
     iszero(expand(diff(dterm2,x,x,x),optIgnoreAnalyticConstraints)) and 
     has(dterm1,ln) and ode::odeIszero(diff(dterm0,x)) and  
     iszero(expand(diff(dterm0,x),optIgnoreAnalyticConstraints)) then 
    a:= (diff(dterm2,x) | x=0);
    b:= dterm0;
    if ode::odeIszero(dterm2-x*(x+a)) and 
       iszero(expand(dterm2-x*(x+a),optIgnoreAnalyticConstraints)) and 
       traperror((c:= diff(subs(dterm1,ln(x)=0,EvalChanges),x))) = 0 and 
       ode::odeIszero(dterm1-x*(b*ln(x)+c)) and 
       iszero(expand(dterm1-x*(b*ln(x)+c),optIgnoreAnalyticConstraints)) then 
      y0:= exp(-int((b*ln(x)+c-1)/(x+a),x));
      y1:= y0*Int(exp(-int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1})
    end_if;
  end_if;

  // ---------------------------
  // look-up for eq. 36
  // ---------------------------

  if has(dterm1,ln) or has(dterm1,ln) or has(dterm0,ln) then 
    t:= subs(dterm2,ln(x)=z);
    if not has(t,x) and ode::odeIszero(diff(t,z,z)) and  
       iszero(expand(diff(t,z,z),optIgnoreAnalyticConstraints)) then 
      a:= diff(t,z);
      b:= expand(t-a*z/*,optIgnoreAnalyticConstraints*/);
      if ode::odeIszero(dterm2-(a*ln(x)+b)) and 
         iszero(expand(dterm2-(a*ln(x)+b),optIgnoreAnalyticConstraints)) then 
        t:= subs(dterm1,ln(x)=z);
        if not has(t,x) and ode::odeIszero(diff(t,z,z)) and 
           iszero(expand(diff(t,z,z),optIgnoreAnalyticConstraints)) then 
          c:= diff(t,z);
          d:= expand(t-c*z/*,optIgnoreAnalyticConstraints*/);
          if ode::odeIszero(dterm1-(c*ln(x)+d)) and 
             iszero(expand(dterm1-(c*ln(x)+d),optIgnoreAnalyticConstraints)) and 
             traperror((t:= evalAt(dterm0,ln(x)=0))) = 0 then
            L:= ode::solveWithoutProperties(z*(d-b*z) = t,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
            L:= select(ode::normal(L), _not@has, x);
            for lambda in L do 
              if ode::odeIszero(dterm0-lambda*((c-a*lambda)*ln(x)+d-b*lambda)) and 
                 iszero(expand(dterm0-lambda*((c-a*lambda)*ln(x)+d-b*lambda),
                               optIgnoreAnalyticConstraints)) then 
                y0:= exp(-lambda*x);
                y1:= y0*Int(exp(-int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
                return({y0,y1})
              end_if;
            end_for;
          end_if;
        end_if;      
      end_if;
    end_if;         
  end_if;

  // ---------------------------

  return(FAIL);
end_proc:


/*
  ----------
  REFERENCE. Implementation of (Section 2.1.6.-1. eq. 1 to 132 p. 260 to 271)

               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------
            eq -- expression of the form f(x)*y''(x) + g(x)*y'(x) + h(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' 

*/


ode::PZ_Sec_215_1_eq_1_to_132_p_260_to_271:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, optIgnoreAnalyticConstraints,
        Int, dterm0_new, dterm1_new, dterm2_new, z, L, a, b, c, ind, sin_arg, n, 
        xpow, dterm0_orig, dterm1_orig, dterm2_orig, k, kk, m, t, cc, aa, i, bb, 
        SinCos, SinCos_arg, SinCos_pow, lambda, tan_arg, tan_pow, cos_arg, cos_pow, f, 
        k1, k2, sin_pow, exp_pow;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();    
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   

  Int:= if has(odeOptions,freeze) then freeze(int) else int end_if;

  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) or (not has(eq,y0) and not has(eq,y1)) then 
    return(FAIL)
  end_if;

  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  dterm0_orig:= combine(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/);
  dterm1_orig:= combine(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/);
  dterm2_orig:= combine(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/);

  if ode::odeIszero(dterm2) and 
     iszero(expand(dterm2,optIgnoreAnalyticConstraints)) then 
    return(FAIL)
  end_if;

  z:= genident(); //solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));

  // ode::normalization 
  if dterm2_orig <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(dterm1_orig/dterm2_orig,sincos);
    dterm0_new:= combine(dterm0_orig/dterm2_orig,sincos);
  else
    dterm2_new:= combine(dterm2_orig,sincos);
    dterm1_new:= combine(dterm1_orig,sincos);
    dterm0_new:= combine(dterm0_orig,sincos);
  end_if;    

  // ---------------------------
  // look-up for eq. 4
  // ---------------------------
 
  if has(dterm1_new,sin) and has(dterm0_new,sin) then 
    sin_arg:= {};
    misc::maprec(dterm1_new,
                 {"sin"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    sin_arg:= sin_arg union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    sin_arg:= [op(sin_arg)];
    if nops(sin_arg) = 1 then 
      b:= sin_arg[1];
      a:= diff(subs(dterm1_new,sin(b*x)=z),z);
      if not has(a,x) and not has(a,z) and ode::odeIszero(a*sin(b*x)-dterm1_new) and 
         iszero(expand(a*sin(b*x)-dterm1_new,optIgnoreAnalyticConstraints)) then 
        xpow:= {};
        misc::maprec(expand(dterm0_new,optIgnoreAnalyticConstraints),
                     {"_power"} = proc(elem)
                                       begin
                                         if op(elem,1) = x then 
                                           xpow:= xpow union {op(elem,2)}
                                         end_if;
                                         elem;
                                       end_proc);              
        xpow:= [op(xpow)];
        n:= FAIL;
        if xpow = [2] then
          n:= 1;       
        elif {op(xpow)} = {-1/2,1/2} then
          n:= 1/2;       
        elif nops(xpow) = 3 then 
          for ind in [[2,3,1],[3,2,1],[1,3,2],[3,1,2],[1,2,3],[2,1,3]] do 
            if iszero(expand(xpow[ind[1]]-2*xpow[ind[2]],optIgnoreAnalyticConstraints)) and 
               iszero(expand(xpow[ind[2]]-xpow[ind[3]]-1,optIgnoreAnalyticConstraints)) then 
              n:= xpow[ind[2]];
            end_if;
          end_for;
        end_if;
        if n <> FAIL then 
          L:= ode::solveWithoutProperties(z*(a*x^n*sin(b*x)-z*x^(2*n)+n*x^(n-1)) = dterm0_new,z,
                    IgnoreSpecialCases,optIgnoreAnalyticConstraints);
          L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
          for c in L do 
            if ode::odeIszero(dterm0_new-c*(a*x^n*sin(b*x)-c*x^(2*n)+n*x^(n-1))) and 
               iszero(expand(dterm0_new-c*(a*x^n*sin(b*x)-c*x^(2*n)+n*x^(n-1)),
                             optIgnoreAnalyticConstraints))then 
              y0:= exp(-c/(n+1)*x^(n+1));
              y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          end_for;
        end_if;
        // same thing again, but without using 'expand' when trying to compute 'n';
        // this is ugly, but I tried and tried and there seems no better workaround
        xpow:= {};
        misc::maprec(dterm0_new,
                     {"_power"} = proc(elem)
                                       begin
                                         if op(elem,1) = x then 
                                           xpow:= xpow union {op(elem,2)}
                                         end_if;
                                         elem;
                                       end_proc);              
        xpow:= [op(xpow)];
        n:= FAIL;
        if xpow = [2] then
          n:= 1;       
        elif {op(xpow)} = {-1/2,1/2} then
          n:= 1/2;       
        elif nops(xpow) = 3 then 
          for ind in [[2,3,1],[3,2,1],[1,3,2],[3,1,2],[1,2,3],[2,1,3]] do 
            if iszero(expand(xpow[ind[1]]-2*xpow[ind[2]],optIgnoreAnalyticConstraints)) and 
               iszero(expand(xpow[ind[2]]-xpow[ind[3]]-1,optIgnoreAnalyticConstraints)) then 
              n:= xpow[ind[2]];
            end_if;
          end_for;
        end_if;
        if n <> FAIL then 
          L:= ode::solveWithoutProperties(z*(a*x^n*sin(b*x)-z*x^(2*n)+n*x^(n-1)) = dterm0_new,z,
                    IgnoreSpecialCases,optIgnoreAnalyticConstraints);
          L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
          for c in L do 
            if ode::odeIszero(dterm0_new-c*(a*x^n*sin(b*x)-c*x^(2*n)+n*x^(n-1))) and 
               iszero(expand(dterm0_new-c*(a*x^n*sin(b*x)-c*x^(2*n)+n*x^(n-1)),
                             optIgnoreAnalyticConstraints)) then 
              y0:= exp(-c/(n+1)*x^(n+1));
              y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          end_for;
        end_if;
      end_if;
    end_if;
  end_if;
 
  for SinCos in [sin,cos,tan,cot] do 
    // ---------------------------
    // look-up for eq. 5 & 31 & 57
    // ---------------------------
    // ode::normalization 
    if dterm2 <> 1 then 
      dterm2_new:= 1;
      dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
      dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    else
      dterm2_new:= dterm2;
      dterm1_new:= dterm1;
      dterm0_new:= dterm0;
    end_if;    
    if has(dterm1_new,SinCos) and has(dterm0_new,SinCos) then 
      SinCos_arg:= {};
      misc::maprec(dterm1_new,
                   {expr2text(SinCos)} = proc(elem)
                                  begin
                                    if has(elem,x) and 
                                       iszero(expand(diff(op(elem,1),x,x),
                                                     optIgnoreAnalyticConstraints)) then 
                                      SinCos_arg:= SinCos_arg union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
      SinCos_arg:= [op(SinCos_arg)];
      if nops(SinCos_arg) = 1 then 
        b:= SinCos_arg[1];
        if not has(b,x) and not has(b,z) then 
          SinCos_pow:= {};
          misc::maprec(dterm1_new,
                       {"_power"} = proc(elem)
                                         begin
                                           if op(elem,1) = SinCos(b*x) then 
                                             SinCos_pow:= SinCos_pow union {op(elem,2)}
                                           end_if;
                                           elem;
                                         end_proc);              
          SinCos_pow:= [op(SinCos_pow)];
          n:= FAIL;
          if nops(SinCos_pow) = 1 then 
            n:= SinCos_pow[1];
          elif nops(SinCos_pow) = 0 then 
            n:= 1;
          end_if;
          if n <> FAIL then 
            // still eq. 5 & 31 & 57
            a:= diff(subs(dterm1_new,SinCos(b*x)^n=z),z);
            if ode::odeIszero(dterm1_new-a*SinCos(b*x)^n) and 
               iszero(expand(dterm1_new-a*SinCos(b*x)^n,optIgnoreAnalyticConstraints)) then 
              L:= ode::solveWithoutProperties(z*(a*SinCos(b*x)^n-z) = dterm0_new,z,
                        IgnoreSpecialCases,optIgnoreAnalyticConstraints);
              L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
              for c in L do 
                if ode::odeIszero(dterm0_new-c*(a*SinCos(b*x)^n-c)) and 
                   iszero(expand(dterm0_new-c*(a*SinCos(b*x)^n-c),
                                 optIgnoreAnalyticConstraints)) then 
                  y0:= exp(-c*x);
                  y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
                  return({y0,y1});
                end_if;
              end_for;
            end_if;
            // ---------------------------
            // look-up for eq. 6 & 32 & 60
            // ---------------------------
            c:= expand(dterm1_new-a*SinCos(b*x)^n/*,optIgnoreAnalyticConstraints*/);
            if not has(c,x) and 
               ode::odeIszero(dterm1_new-a*SinCos(b*x)^n-c) and 
               ode::odeIszero(dterm0_new-a*c*SinCos(b*x)^n) and 
               iszero(expand(dterm1_new-a*SinCos(b*x)^n-c,optIgnoreAnalyticConstraints)) and 
               iszero(expand(dterm0_new-a*c*SinCos(b*x)^n,optIgnoreAnalyticConstraints)) then
              y0:= exp(-c*x);
              y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;    
            // --------------------------------
            // look-up for eq. 9 & 35 & 64 & 94
            // --------------------------------
            m:= n;
            t:= a; 
            xpow:= {};
            misc::maprec(t,
                         {"_power"} = proc(elem)
                                           begin
                                             if op(elem,1) = x then 
                                               xpow:= xpow union {op(elem,2)}
                                             end_if;
                                             elem;
                                           end_proc);              
            xpow:= [op(xpow)];
            n:= FAIL;
            if nops(xpow) = 1 then 
              n:= xpow[1]
            elif nops(xpow) = 0 then 
              n:= 1;
            end_if;
            a:= diff(subs(t,x^n=z),z); 
            if ode::odeIszero(dterm1_new-a*x^n*SinCos(b*x)^m) and 
               iszero(expand(dterm1_new-a*x^n*SinCos(b*x)^m,optIgnoreAnalyticConstraints)) then 
              xpow:= {};
              misc::maprec(dterm0_new,  
                           {"_power"} = proc(elem)  
                                             begin
                                               if op(elem,1) = x then 
                                                 xpow:= xpow union {op(elem,2)}
                                               end_if;
                                               elem;
                                             end_proc);              
              xpow:= [op(xpow)];
              k:= FAIL;
              if nops(xpow) = 3 then 
                for ind in [[2,3,1],[3,2,1],[1,3,2],[3,1,2],[1,2,3],[2,1,3]] do 
                  kk:= xpow[ind[2]]-n;
                  if iszero(expand(xpow[ind[1]]-2*kk,optIgnoreAnalyticConstraints)) and 
                     iszero(expand(kk-xpow[ind[3]]-1,optIgnoreAnalyticConstraints)) then 
                    k:= kk;
                  end_if;
                end_for;
              elif nops(xpow) = 2 and contains(xpow,2) > 0 then 
                k:= 1;
              elif nops(xpow) = 2 then 
                if iszero(expand(2*(xpow[1]+1)-xpow[2],optIgnoreAnalyticConstraints)) or 
                   iszero(expand(2*(xpow[2]+1)-xpow[1],optIgnoreAnalyticConstraints)) then 
                  k:= n;
                end_if;
              end_if;
              if k = FAIL and traperror((t:= evalAt(dterm0_new,SinCos(b*x)^m=0))) = 0 then
                xpow:= {};
                misc::maprec(dterm0_new,  
                             {"_power"} = proc(elem)  
                                               begin
                                                 if op(elem,1) = x then 
                                                   xpow:= xpow union {op(elem,2)}
                                                 end_if;
                                                 elem;
                                               end_proc);              
                xpow:= [op(xpow)];
                k:= FAIL;
                if nops(xpow) = 2 then 
                  if iszero(expand(2*(xpow[1]+1)-xpow[2],optIgnoreAnalyticConstraints)) then 
                    k:= xpow[1];
                  elif iszero(expand(2*(xpow[2]+1)-xpow[1],optIgnoreAnalyticConstraints)) then 
                    k:= xpow[2];
                  end_if;
                elif nops(xpow) = 1 then 
                  k:= 1;
                end_if;
              end_if;    
              if k <> FAIL and not iszero(k) then
                c:= FAIL;
                if k <> 1 then 
                  c:= ode::normal(diff(subs(dterm0_new,[x^(k-1)=z,x^(expand(k-1))=z]),z)/k);
                else 
                  cc:= select({op(dterm0_new)},_not@has,x);
                  if nops(cc) = 1 then 
                    c:= cc[1];
                  end_if;
                end_if;
                if c <> FAIL and 
                   ode::odeIszero(dterm0_new-c*(a*x^(n+k)*SinCos(b*x)^m-c*x^(2*k)+k*x^(k-1))) and 
                   iszero(expand(dterm0_new-c*(a*x^(n+k)*SinCos(b*x)^m-c*x^(2*k)+k*x^(k-1)),
                                 optIgnoreAnalyticConstraints)) then 
                  y0:= exp(-c/(k+1)*x^(k+1));
                  y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
                  return({y0,y1});
                end_if;
              end_if;
            end_if;
          end_if;
        end_if;  
      end_if;  
    end_if;

    if SinCos <> cot then 
      // ----------------------------
      // look-up for eq. 15 & 41 & 74
      // ----------------------------
    
      // ode::normalization 
      if dterm2 <> x^2 then 
        dterm2_new:= x^2;
        dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
        dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
      else
        dterm2_new:= dterm2;
        dterm1_new:= dterm1;
        dterm0_new:= dterm0;
      end_if;    
    
      if has(dterm1_new,SinCos) and has(dterm0_new,SinCos) then
        SinCos_pow:= {};
        misc::maprec(dterm1_new,
                     {"_power"} = proc(elem)
                                       begin
                                         if op(elem,1) = SinCos(x) then 
                                           SinCos_pow:= SinCos_pow union {op(elem,2)}
                                         end_if;
                                         elem;
                                       end_proc);              
        SinCos_pow:= [op(SinCos_pow)];
        n:= FAIL;
        if nops(SinCos_pow) = 1 then 
          n:= SinCos_pow[1];
        elif nops(SinCos_pow) = 0 then 
          n:= 1;
        end_if;
        if n <> FAIL then 
          a:= diff(diff(subs(dterm1_new,SinCos(x)^n=z),z),x);
          if not iszero(a) and not has(a,x) and not has(a,z) and 
             ode::odeIszero(dterm1_new-x*(a*SinCos(x)^n+1)) and 
             iszero(expand(dterm1_new-x*(a*SinCos(x)^n+1),optIgnoreAnalyticConstraints)) then 
            b:= ode::normal(diff(subs(dterm0_new,SinCos(x)^n=z),z)/a);  
            if not has(b,x) and not has(b,z) and 
               ode::odeIszero(dterm0_new-b*(a*SinCos(x)^n-b)) and 
               iszero(expand(dterm0_new-b*(a*SinCos(x)^n-b),optIgnoreAnalyticConstraints)) then
              y0:= x^(-b);
              y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          end_if;
          // ----------------------------
          // look-up for eq. 16 & 42 & 75
          // ----------------------------
          b:= ode::normal((dterm1_new-a*x*SinCos(x)^n)/x);
          if not has(b,x) and not has(b,z) and 
             ode::odeIszero(dterm1_new-x*(a*SinCos(x)^n+b)) and 
             ode::odeIszero(dterm0_new-b*(a*SinCos(x)^n-1)) and 
             iszero(expand(dterm1_new-x*(a*SinCos(x)^n+b),optIgnoreAnalyticConstraints)) and 
             iszero(expand(dterm0_new-b*(a*SinCos(x)^n-1),optIgnoreAnalyticConstraints)) then 
            y0:= x^(-b);
            y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;
        end_if;
      end_if;  
    
      // ----------------------------
      // look-up for eq. 17 & 43 & 76
      // ----------------------------
    
      if has(dterm1_new,SinCos) and has(dterm0_new,SinCos) then 
        SinCos_arg:= {};
        misc::maprec(dterm1_new,
                     {expr2text(SinCos)} = proc(elem)
                                    begin
                                      if has(elem,x) and 
                                         iszero(expand(diff(op(elem,1),x,x),
                                                       optIgnoreAnalyticConstraints)) then 
                                        SinCos_arg:= SinCos_arg union {diff(op(elem,1),x)}
                                      end_if;
                                      elem;
                                    end_proc);  
        SinCos_arg:= [op(SinCos_arg)];
        if nops(SinCos_arg) = 1 then 
          b:= SinCos_arg[1];
          if not has(b,x) and not has(b,z) then 
            SinCos_pow:= {};
            misc::maprec(dterm1_new,
                         {"_power"} = proc(elem)
                                           begin
                                             if op(elem,1) = SinCos(b*x) then 
                                               SinCos_pow:= SinCos_pow union {op(elem,2)}
                                             end_if;
                                             elem;
                                           end_proc);              
            SinCos_pow:= [op(SinCos_pow)];
            m:= FAIL;
            if nops(SinCos_pow) = 1 then 
              m:= SinCos_pow[1];
            elif nops(SinCos_pow) = 0 then 
              m:= 1;
            end_if;
            if m <> FAIL then 
              t:= diff(subs(dterm1_new,SinCos(b*x)^m=z),z);
              xpow:= {};
              misc::maprec(t,
                           {"_power"} = proc(elem)
                                             begin
                                               if op(elem,1) = x then 
                                                 xpow:= xpow union {op(elem,2)}
                                               end_if;
                                               elem;
                                             end_proc);              
              xpow:= [op(xpow)];
              n:= FAIL;
              if nops(xpow) = 1 then 
                n:= xpow[1]
              elif nops(xpow) = 0 then 
                n:= 1;
              end_if;
              a:= diff(subs(t,x^n=z),z); 
              if not has(a,x) and not has(a,z) and 
                 ode::odeIszero(dterm1_new-a*x^n*SinCos(b*x)^m) and 
                 iszero(expand(dterm1_new-a*x^n*SinCos(b*x)^m,optIgnoreAnalyticConstraints)) then 
                L:= ode::solveWithoutProperties(z*(a*x^(n-1)*SinCos(b*x)^m-z-1) = dterm0_new, z, IgnoreSpecialCases, optIgnoreAnalyticConstraints);
                L:= select(L, _not@has, x);
                for c in L do 
                  if ode::odeIszero(dterm0_new-c*(a*x^(n-1)*SinCos(b*x)^m-c-1)) and 
                     iszero(expand(dterm0_new-c*(a*x^(n-1)*SinCos(b*x)^m-c-1),
                                   optIgnoreAnalyticConstraints)) then 
                    y0:= x^(-c);
                    y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
                    return({y0,y1});
                  end_if;
                end_for;
              end_if;
            end_if;
          end_if;
        end_if;
      end_if;  
  
      if SinCos in {sin,cos} then 
        // ---------------------------
        // look-up for eq. 20
        // ---------------------------
        // ode::normalization 
        if dterm2 <> SinCos(2*x) and dterm2 <> 2*cos(x)*SinCos(x) then 
          dterm2_new:= SinCos(2*x);
          dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
          dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
        else
          dterm2_new:= dterm2;
          dterm1_new:= dterm1;
          dterm0_new:= dterm0;
        end_if;    
      
        if ode::odeIszero(dterm1_new+1) and 
           iszero(expand(dterm1_new+1,optIgnoreAnalyticConstraints)) then 
          aa:= combine(sqrt(ode::normal(dterm0_new/(2*SinCos(x)^2))),IgnoreAnalyticConstraints);
          if not has(aa,x) then 
            for a in [-aa,aa] do
              if ode::odeIszero(dterm0_new-2*a^2*SinCos(x)^2) and 
                 iszero(expand(dterm0_new-2*a^2*SinCos(x)^2)) then 
                return({SinCos(a*Int(sqrt(tan(x)),x)), 
                        cos(a*Int(sqrt(tan(x)),x))});
              end_if;
            end_for;
          end_if;
        end_if;
      
        // ---------------------------
        // look-up for eq. 22 & 45
        // ---------------------------
      
        if ode::odeIszero(dterm1) and 
           iszero(expand(dterm1,optIgnoreAnalyticConstraints)) then 
          // ode::normalization 
          if dterm2 <> SinCos(x)^2 then   
            dterm2_new:= SinCos(x)^2;
            dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
          else
            dterm2_new:= dterm2;
            dterm0_new:= dterm0;
          end_if; 
          a:= diff(subs(-dterm0_new,SinCos(x)^2=z),z);   
          if not has(a,x) and not has(a,z) and 
             traperror((t:= subs(-dterm0_new,SinCos(x)^2=0,EvalChanges))) = 0 and 
             not has(t,x) then 
            L:= ode::solveWithoutProperties(z*(z-1) = t, z, IgnoreSpecialCases, optIgnoreAnalyticConstraints);
            L:= select(L, _not@has, x);
            for n in L do 
              if ode::odeIszero(dterm0_new+(a*SinCos(x)^2+n*(n-1))) and 
                 iszero(expand(dterm0_new+(a*SinCos(x)^2+n*(n-1)),
                               optIgnoreAnalyticConstraints)) and 
                 type(n) = DOM_INT and n > 0 then 
                y0:= exp(x*sqrt(a));
                y1:= exp(-x*sqrt(a));
                for i from 1 to n do 
                  y0:= combine(1/SinCos(x)*diff(y0,x),sincos);
                  y1:= combine(1/SinCos(x)*diff(y1,x),sincos);
                end_for; 
                y0:= SinCos(x)^n*y0;
                y1:= SinCos(x)^n*y1;
                return({y0,y1});
              end_if;
            end_for;
          end_if;
          // ---------------------------
          // look-up for eq. 24
          // ---------------------------
          t:= _plus(op(select(op(expand(-dterm0_new)),_not@has,x)));
          L:= ode::solveWithoutProperties(z^2-z = t,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints); 
          L:= select(ode::normal(L), _not@has, x);
          for a in L do 
            if not has(a,x) and not iszero(a) then 
              bb:= combine(sqrt(ode::normal((diff(subsex(-dterm0_new,SinCos(x)^2=z),z)+(a+1)^2)/a^2)),
                           IgnoreAnalyticConstraints); 
              for b in [-bb,bb] do 
                if ode::odeIszero(dterm0_new+(((a^2*b^2-(a+1)^2)*SinCos(x)^2+a*(a+1)*b*SinCos(2*x)+a*(a-1)))) and 
                   iszero(expand(dterm0_new+(((a^2*b^2-(a+1)^2)*SinCos(x)^2+a*(a+1)*b*SinCos(2*x)+a*(a-1))),
                                 optIgnoreAnalyticConstraints)) then 
                  y0:= exp(a*b*x)*SinCos(x)^a*(cos(x)+b*SinCos(x));
                  y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
                  return({y0,y1});
                end_if;
              end_for;
            end_if;
          end_for;
        end_if;
      end_if;    
    end_if;
  end_for;

  // ---------------------------
  // look-up for eq. 51
  // ---------------------------
  
  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  if iszero(dterm1_new) then 
    a:= diff(subs(dterm0_new,tan(x)^2=z),z);
    if not has(a,x) and not has(a,z) then 
      b:= dterm0_new-a*tan(x)^2;
      if not has(b,x) and ode::odeIszero(dterm0_new-a*tan(x)^2-b) and 
         iszero(expand(dterm0_new-a*tan(x)^2-b,optIgnoreAnalyticConstraints)) then 
        L:= ode::solveWithoutProperties(z^2+z+a=0,z,IgnoreSpecialCases,PrincipalValue);
        if nops(L) = 1 then 
          k:= L[1];
          L:= ode::lookUp2ndOrderLinear(x*(x-1)*diff(y(x),x,x)+((1-k)*x-1/2)*diff(y(x),x)-1/4*(k+b)*y(x),
                                        y,x,solveOptions,odeOptions);
          if nops(L) = 2 and not has(L,int) then 
            if traperror((L:= map(L, elem -> ((elem | (x=sin(x)^2))/cos(x)^k)))) = 0 then 
              return(L)      
            end_if;
          end_if;  
        end_if:
      end_if;
    end_if;
  end_if;

  /* ========================== not used; incorrect transformation ==========================
  // ---------------------------
  // look-up for eq. 55
  // ---------------------------
  // transformation stated in the book seems to be wrong; found this 
  // out after I did the implementation; the code is still here since 
  // a fix might be found in the future
  tan_arg:= {};
  misc::maprec(dterm1_new,
               {"tan"} = proc(elem)
                              begin
                                if has(elem,x) and 
                                   iszero(expand(diff(op(elem,1),x,x),
                                                 optIgnoreAnalyticConstraints)) then 
                                  tan_arg:= tan_arg union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
  tan_arg:= [op(tan_arg)];
  if nops(tan_arg) = 1 then 
    lambda:= tan_arg[1];
    if iszero(expand(dterm1_new+2*lambda*tan(lambda*x),optIgnoreAnalyticConstraints)) and 
       iszero(expand(diff(dterm0_new,x,x,x),optIgnoreAnalyticConstraints)) then
      L:= ode::lookUp2ndOrderLinear(diff(y(x),x,x)+(dterm0_new+lambda^2)*y(x),
                                    y,x,solveOptions,odeOptions);
      if nops(L) = 2 then 
        if traperror((L:= map(L, elem -> elem/cos(lambda*x)))) = 0 then 
          return(L)      
        end_if;
      end_if;  
    end_if;
  end_if;
  ========================== not used; incorrect transformation ========================== */

  // ---------------------------
  // look-up for eq. 58
  // ---------------------------

  if has(dterm1_new,tan) and has(dterm0_new,tan) then 
    tan_pow:= {};
    misc::maprec(dterm1_new,
                 {"_power"} = proc(elem)
                                   begin
                                     if type(op(elem,1)) = "tan" then 
                                       tan_pow:= tan_pow union {op(elem,2)};
                                     end_if;
                                     elem;
                                   end_proc);      
    n:= FAIL;        
    if nops(tan_pow) = 1 then
      n:= tan_pow[1];
    elif tan_pow = {} then 
      n:= 1;
    end_if;
    if n <> FAIL then 
      tan_arg:= {};
      misc::maprec(dterm1_new,
                   {"tan"} = proc(elem)
                                  begin
                                    if has(elem,x) and 
                                       iszero(expand(diff(op(elem,1),x,x),
                                                     optIgnoreAnalyticConstraints)) then 
                                      tan_arg:= tan_arg union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
      if nops(tan_arg) = 1 then 
        lambda:= tan_arg[1];
        a:= diff(subs(dterm1_new,[tan(lambda*x)^n=z]),z);
        if not has(a,x) and not has(a,z) and not iszero(lambda) and 
           ode::odeIszero(dterm1_new-a*tan(lambda*x)^n) and 
           iszero(expand(dterm1_new-a*tan(lambda*x)^n,optIgnoreAnalyticConstraints)) and 
           type(dterm0_new) = "_plus" then
          b:= ode::normal(_plus(op(select({op(dterm0_new)},_not@has,x)))/lambda);
          if ode::odeIszero(dterm0_new-b*(a*tan(lambda*x)^(n+1)+(lambda-b)*tan(lambda*x)^2+lambda)) and 
             iszero(expand(dterm0_new-b*(a*tan(lambda*x)^(n+1)+(lambda-b)*tan(lambda*x)^2+lambda),
                           optIgnoreAnalyticConstraints)) then 
            y0:= cos(lambda*x)^(b/lambda);
            y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1});
          end_if;
        end_if;
      end_if;        
    end_if;
  end_if;  

  // ----------------------------------
  // look-up for eq. 61 (including 59)
  // ----------------------------------

  if has(dterm1_new,tan) and has(dterm0_new,tan) then 
    tan_pow:= {};
    misc::maprec(dterm1_new,
                 {"_power"} = proc(elem)
                                   begin
                                     if type(op(elem,1)) = "tan" then 
                                       tan_pow:= tan_pow union {op(elem,2)};
                                     end_if;
                                     elem;
                                   end_proc);      
    n:= FAIL;        
    if nops(tan_pow) = 1 then
      n:= tan_pow[1]-1;
    elif tan_pow = {} then 
      n:= 0;
    end_if;
    if n <> FAIL then 
      a:= diff(subs(dterm1_new,[tan(x)^(n+1)=z,tan(x)^(expand(n+1))=z]),z);
      t:= expand(dterm1_new-a*tan(x)^(n+1)/*,optIgnoreAnalyticConstraints*/);
      if not has(a,x) and not has(a,z) and 
         traperror((b:= expand(evalAt(diff(subs(t,tan(x)=z),z),z=0)+1
                               /*,optIgnoreAnalyticConstraints*/))) = 0 then 
        if ode::odeIszero(dterm1_new-tan(x)*(a*tan(x)^n+b-1)) and 
           ode::odeIszero(dterm0_new-(a*b*tan(x)^(n+2)-a*tan(x)^n+2*b+2)) and 
           iszero(expand(dterm1_new-tan(x)*(a*tan(x)^n+b-1),optIgnoreAnalyticConstraints)) and 
           iszero(expand(dterm0_new-(a*b*tan(x)^(n+2)-a*tan(x)^n+2*b+2),optIgnoreAnalyticConstraints)) then 
          y0:= sin(x)*cos(x)^b;
          y1:= y0*Int(exp(-Int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1});
        end_if;          
      end_if;
    end_if;
  end_if;
 
  // -------------------------------------------------------------
  // look-up for eq. 102 & 103 & 104 (but generalized to arbitrary 
  // functions):
  //  'x^2*diff(y(x),x,x)+x*(f(x)+1)*diff(y(x),x)+b*(f(x)-b)*y(x)'
  // has solution x^(-b). 
  // -------------------------------------------------------------
  
  // ode::normalization 
  if dterm2 <> x^2 then 
    dterm2_new:= x^2;
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  f:= ode::normal(dterm1_new/x)-1;
  L:= ode::solveWithoutProperties(z*(f-z) = dterm0_new,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
  if nops(L) > 0 then
    b:= L[1];
    y0:= x^(-b);
    y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // --------------------
  // look-up for eq. 107
  // --------------------

  // ode::normalization 
  if dterm2_orig <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(dterm1_orig/dterm2_orig,sincos);
    dterm0_new:= combine(dterm0_orig/dterm2_orig,sincos);
  else
    dterm2_new:= combine(dterm2_orig,sincos);
    dterm1_new:= combine(dterm1_orig,sincos);
    dterm0_new:= combine(dterm0_orig,sincos);
  end_if;    

  if ode::odeIszero(dterm1_new) and 
     iszero(expand(dterm1_new,optIgnoreAnalyticConstraints)) then 
    for SinCos in {sin,cos} do  
      SinCos_arg:= {};  
      misc::maprec(dterm0_new,
                   {expr2text(SinCos)} = proc(elem)
                                         begin
                                           if has(elem,x) and 
                                              iszero(expand(diff(op(elem,1),x,x),
                                                            optIgnoreAnalyticConstraints)) then 
                                             SinCos_arg:= SinCos_arg union {diff(op(elem,1),x)}
                                           end_if;
                                           elem;
                                         end_proc);  
      SinCos_arg:= [op(SinCos_arg)];
      for b in SinCos_arg do 
        if not has(b,x) and not has(b,z) then 
          a:= ode::normal(-diff(subs(dterm0_new,[SinCos(b*x)=z,SinCos(expand(b*x))=z]),z)/b);
          if SinCos = cos and ode::odeIszero(dterm0_new+a*(a*sin(b*x)^2+b*cos(b*x))) and 
             iszero(expand(combine(dterm0_new+a*(a*sin(b*x)^2+b*cos(b*x)),sincos),
                                            optIgnoreAnalyticConstraints)) then
            y0:= exp(-a/b*cos(b*x));
            y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1})
          elif SinCos = sin and ode::odeIszero(dterm0_new+a*(a*cos(b*x)^2+b*sin(b*x))) and 
               iszero(expand(combine(dterm0_new+a*(a*cos(b*x)^2+b*sin(b*x)),sincos),
                                            optIgnoreAnalyticConstraints)) then
            y0:= exp(-a/b*sin(b*x));
            y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1})
          end_if;
        end_if;
      end_for;
    end_for;
  end_if;

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  // --------------------
  // look-up for eq. 110
  // --------------------

  b:= diff(subs(dterm0_new,cos(x)=z),z);
  if not has(b,x) and not has(b,z) then
    t:= dterm0_new-b*cos(x);
    sin_pow:= {};
    misc::maprec(t,
                 {"_power"} = proc(elem)
                                   begin
                                     if type(op(elem,1)) = "sin" then 
                                       sin_pow:= sin_pow union {op(elem,2)};
                                     end_if;
                                     elem;
                                   end_proc);      
    n:= FAIL;
    if nops(sin_pow) = 1 then
      n:= sin_pow[1]-1;
    end_if; 
    if n <> FAIL then 
      a:= diff(subs(expand(dterm1_new-b*sin(x)/*,optIgnoreAnalyticConstraints*/),sin(x)^n=z),z);
      if not has (a,x) and not has(a,z) and 
         ode::odeIszero(dterm1_new-(a*sin(x)^n+b*sin(x))) and 
         ode::odeIszero(dterm0_new-b*(a*sin(x)^(n+1)+cos(x))) and 
         iszero(expand(dterm1_new-(a*sin(x)^n+b*sin(x)),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0_new-b*(a*sin(x)^(n+1)+cos(x)),optIgnoreAnalyticConstraints)) then 
        y0:= exp(b*cos(x));        
        y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1})
      end_if;
    end_if;
  end_if;
  // --------------------
  // look-up for eq. 112  
  // --------------------
  b:= -diff(subs(dterm0_new,sin(x)=z),z);
  if not has(b,x) and not has(b,z) then 
    t:= dterm0_new+b*sin(x);
    cos_pow:= {};  
    misc::maprec(t,
                   {"_power"} = proc(elem)
                                     begin
                                       if type(op(elem,1)) = "cos" then 
                                         cos_pow:= cos_pow union {op(elem,2)};
                                       end_if;
                                       elem;
                                     end_proc);      
    n:= FAIL;
    if nops(cos_pow) = 1 then
      n:= cos_pow[1]-1;
    end_if; 
    if n <> FAIL then 
      a:= diff(subs(expand(dterm1_new-b*cos(x)/*,optIgnoreAnalyticConstraints*/),cos(x)^n=z),z);
      if not has (a,x) and not has(a,z) and 
         ode::odeIszero(dterm1_new-(a*cos(x)^n+b*cos(x))) and 
         ode::odeIszero(dterm0_new-b*(a*cos(x)^(n+1)-sin(x))) and 
         iszero(expand(dterm1_new-(a*cos(x)^n+b*cos(x)),optIgnoreAnalyticConstraints)) and 
         iszero(expand(dterm0_new-b*(a*cos(x)^(n+1)-sin(x)),optIgnoreAnalyticConstraints)) then 
        y0:= exp(-b*sin(x));        
        y1:= y0*int(exp(-int(dterm1_new/dterm2_new,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1})
      end_if;
    end_if;
  end_if;

  // --------------------
  // look-up for eq. 113  
  // --------------------

  // ode::normalization 
  if dterm2 <> sin(x) then 
    dterm2_new:= sin(x);
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  if ode::odeIszero(dterm1_new-cos(x)) and 
     iszero(expand(dterm1_new-cos(x),optIgnoreAnalyticConstraints)) and 
     has(dterm0_new,sin) then 
    t:= ode::normal(dterm0_new/sin(x)):
    if not has(t,x) then
      L:= ode::lookUp2ndOrderLinear((1-x^2)*diff(y(x),x,x)-2*x*diff(y(x),x)+t*y(x),
                                    y,x,solveOptions,odeOptions);
      if nops(L) = 2 then 
        L:= map(L,intlib::changevar,x=cos(z),z);
        L:= subs(L,x=cos(x));
        L:= subs(L,z=x); 
        return(L);        
      end_if;  
    end_if;
  end_if;

  if has(dterm2_orig,cos) and has(dterm1_orig,sin) and has(dterm0_orig,sin) and has(dterm0_orig,cos) then 
    cos_arg:= {};  
    misc::maprec(dterm2_orig,
                 {"cos"} = proc(elem)
                             begin
                               if has(elem,x) and 
                                  iszero(expand(diff(op(elem,1),x,x),
                                                optIgnoreAnalyticConstraints)) then 
                                  cos_arg:= cos_arg union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
    cos_arg:= [op(cos_arg)];
    for a in cos_arg do 
      if not iszero(a*sin(2*a*x)) and 
         ode::odeIszero(dterm2_orig-cos(a*x)^2) and 
         iszero(expand(dterm2_orig-cos(a*x)^2,optIgnoreAnalyticConstraints)) then 
        n:= ode::normal(combine(dterm1_orig/(a*sin(2*a*x)),sincos)+1);
        if not has(n,x) and 
           ode::odeIszero(dterm1_orig-(n-1)*a*sin(2*a*x)) and 
           ode::odeIszero(dterm0_orig-n*a^2*((n-1)*sin(a*x)^2+cos(a*x)^2)) and 
           iszero(expand(combine(dterm1_orig-(n-1)*a*sin(2*a*x),sincos),optIgnoreAnalyticConstraints)) and 
           iszero(expand(combine(dterm0_orig-n*a^2*((n-1)*sin(a*x)^2+cos(a*x)^2),sincos),optIgnoreAnalyticConstraints)) then 
          y0:= cos(a*x)^n;
          y1:= y0*int(exp(-int(dterm1_orig/dterm2_orig,x,intOptions))/y0^2,x,intOptions);
          return({y0,y1})
        end_if;
      end_if;
    end_for; 
  end_if;

  // --------------------
  // look-up for eq. 121  
  // --------------------

  // ode::normalization 
  if dterm2 <> sin(x)*cos(x)^2 then 
    dterm2_new:= sin(x);
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    
 
  b:= ode::normal(combine(dterm0_new/sin(x)^3,sincos));
  if not has(b,x) then 
    a:= ode::normal(combine((dterm1_new+cos(x))/(sin(x)^2*cos(x)),sincos));
    if not has(a,x) then 
      L:= ode::solveWithoutProperties(z^2-a*z+b,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
      if nops(L) = 2 then 
        k1:= L[1];
        k2:= L[2];
        return({cos(x)^k1,cos(x)^k2})
      elif nops(L) = 1 then 
        k1:= L[1];
        y0:= cos(x)^k1;
        y1:= y0*int(exp(-int(dterm1_orig/dterm2_orig,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1})
      end_if;
    end_if;
  end_if;

  // --------------------
  // look-up for eq. 125  
  // --------------------

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= combine(expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  if has(dterm1_new,tan) /*and (has(dterm0_new,cos)*/ or  has(dterm0_new,cot) then 
    n:= ode::normal(dterm1_new/tan(x));
    if not has(n,x) then 
      a:= combine(sqrt(dterm0_new/cos(x)^(2*n)),IgnoreAnalyticConstraints);
      if not has(a,x) then 
        return({sin(a*int(cos(x)^n,x)),cos(a*int(cos(x)^n,x))})
      end_if;
    end_if;
    // --------------------
    // look-up for eq. 126  
    // --------------------
    if n = 1 then 
      sin_pow:= {};
      misc::maprec(dterm0_new,
                   {"_power"} = proc(elem)
                                     begin
                                       if op(elem,1) = sin(x) then 
                                         sin_pow:= sin_pow union {op(elem,2)};
                                       end_if;
                                       elem;
                                     end_proc);      
      n:= FAIL;
      if nops(sin_pow) = 1 then
        n:= ode::normal((sin_pow[1]+2)/2);
      end_if; 
      if n <> FAIL then 
      a:= combine(sqrt(dterm0_new/(cos(x)^2*sin(x)^(2*n-2))),IgnoreAnalyticConstraints);
        if not has(a,x) then 
          return({sqrt(sin(x))*besselJ(1/(2*n),a/n*sin(x)^n), 
                  sqrt(sin(x))*besselY(1/(2*n),a/n*sin(x)^n)})
        end_if;
      end_if;
      // --------------------
      // look-up for eq. 127  
      // --------------------
      t:= ode::normal(rewrite(dterm0_new/(-cot(x)^2),exp));
      if not has(t,x) then 
        L:= ode::solveWithoutProperties(z*(z-1)=t,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
        L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
        for a in L do 
          if a <> 1/2 then 
            return({sin(x)^a, sin(x)^(1-a)}) 
          else
            return({sqrt(sin(x)),sqrt(sin(x))*ln(sin(x))})
          end_if; 
        end_for;
      end_if;
    end_if;
  end_if;

  // --------------------
  // look-up for eq. 128  
  // --------------------
  if dterm2_orig = 1 then 
    if has(dterm1_orig,cot) and has(dterm0_orig,sin) then 
      t:= misc::maprec(dterm1_orig,
                       {"cot"} = proc(elem)
                                    begin
                                      if has(elem,x) and 
                                         iszero(expand(diff(op(elem,1),x,x),
                                                       optIgnoreAnalyticConstraints)) then 
                                        z; else elem; 
                                      end_if;
                                    end_proc);
      a:= ode::normal(diff(t,z)/(-2));
      if not has(a,x) and not has(a,z) and 
         ode::odeIszero(dterm1_orig/(2*a*cot(2*a*x))+1) and 
         iszero(expand(ode::normal(dterm1_orig/(2*a*cot(2*a*x)))+1,optIgnoreAnalyticConstraints)) then 
        b:= combine(sqrt(ode::normal(-dterm0_orig/sin(2*a*x)^2)),IgnoreAnalyticConstraints);
        if not has(b,x) and not has(b,z) then 
          return({exp(b/a*sin(a*x)^2),exp(-b/a*sin(a*x)^2)}) 
        end_if;
      end_if;
    else // try rewrite to 'exp' strategy  
      exp_pow:= {};
      misc::maprec(ode::normal(rewrite(dterm1_orig,exp),Expand=FALSE),
                   {"exp"} = proc(elem)
                               begin
                                 if has(elem,x) and 
                                    iszero(expand(diff(op(elem,1),x,x),
                                                  optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {ode::normal(diff(op(elem,1),x)/(4*I))}
                                  end_if;
                                  elem;
                                end_proc);  
      for a in exp_pow do 
        if  ode::odeIszero(dterm1_orig/(2*a*cot(2*a*x))) and 
            iszero(expand(ode::normal(rewrite(dterm1_orig/(2*a*cot(2*a*x)),exp))+1,
                         optIgnoreAnalyticConstraints)) then 
          b:= combine(sqrt(ode::normal(rewrite(-dterm0_orig/sin(2*a*x)^2,exp))),IgnoreAnalyticConstraints);
          if not has(b,x) and not has(b,z) then 
            return({exp(b/a*sin(a*x)^2),exp(-b/a*sin(a*x)^2)}) 
          end_if;
        end_if;  
      end_for;
    end_if;
  end_if;        

  // --------------------
  // look-up for eq. 129  
  // --------------------

  if has(dterm1_new,cot) and has(dterm0_new,sin) then 
    n:= ode::normal(dterm1_new/(-cot(x)));
    if not has(n,x) then 
      a:= combine(sqrt(dterm0_new/(sin(x)^(2*n))),IgnoreAnalyticConstraints);
      if not has(a,x) then 
        return({sin(a*int(sin(x)^n,x)),cos(a*int(sin(x)^n,x))})
      end_if;
    end_if;
  else // try rewrite to 'exp' strategy  
    n:= ode::normal(rewrite(dterm1_new/(-cot(x)),exp));
    if not has(n,x) then 
      a:= combine(sqrt(ode::normal(rewrite(dterm0_new/(sin(x)^(2*n)),exp))),IgnoreAnalyticConstraints);
      if not has(a,x) then 
        return({sin(a*int(sin(x)^n,x)),cos(a*int(sin(x)^n,x))})
      end_if;
    end_if;
  end_if;

  // --------------------
  // look-up for eq. 130  
  // --------------------

  if (
      iszero(expand(dterm1_new+2*cot(2*x))) or
      iszero(ode::normal(rewrite(dterm1_new + 2*cot(2*x), exp)))
     ) and not has((a:= ode::normal(dterm0_new/tan(x)^2)),x) then 
    L:= ode::lookUp2ndOrderLinear(x^2*diff(y(x),x,x)-x*diff(y(x),x)+a*y(x),
                                  y,x,solveOptions,odeOptions);
    if nops(L) = 2 /*and not has(L,int)*/ then 
      if traperror((L:= map(L,intlib::changevar,x=cos(z),z))) = 0 and 
         traperror((L:= evalAt(evalAt(L,x=cos(x)),z=x))) = 0 then 
        return(L)      
      end_if;
    end_if;
  end_if;

  // --------------------
  // look-up for eq. 132  
  // --------------------

  // ode::normalization 
  if dterm2 <> sin(2*x) then 
    dterm2_new:= sin(2*x);
    dterm1_new:= combine(expand(dterm2_new*dterm1/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= combine(expand(dterm2_new*dterm0/dterm2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  if ode::odeIszero(diff(dterm1_new,x)) and 
     iszero(expand(diff(dterm1_new,x),optIgnoreAnalyticConstraints)) then
    n:= ode::normal(dterm1_new/(-2));
    a:= combine(sqrt(dterm0_new/(2*sin(x)^2*tan(x)^(2*n-1))),IgnoreAnalyticConstraints);
    if not has(a,x) then 
      return({sin(a*int(tan(x)^n,x)),cos(a*int(tan(x)^n,x))})  
    end_if;
  end_if;

  // ---------------------------

  return(FAIL);
end_proc:

/*
  Special heuristics for determining particular class of ODEs that seem 
  to occur more often in standard literature:
    - Heuristic to find solutions of the form 'exp(-a/lambda*exp(lambda*x))'
    - Heuristic to find solutions of the form 'x^n'
    - Heuristic to find solutions of the form 'exp(lambda*x)'
    - Heuristic to find solutions of the form 'exp(lambda*x^2)'
    - Heuristic to find solutions of the form 'exp(lambda/x)'
    - Heuristic to find solutions of the form 'ln(lambda*x)'
    - Heuristic to find solutions of the form 'x*exp(lambda*x)'
    - Heuristic to find solutions of the form 'x^2*exp(lambda*x)'
    - Heuristic to find solutions of the form '1/x*exp(lambda*x)'
    - Heuristic to find solutions of the form '1/x^2*exp(lambda*x)'
    - Heuristic to find various trigonometric/hyperbolic functions 
      as solutions. 
    - Heuristics for some further arbitrary functions patterns. 
*/

ode::PZ_2nd_order_linear_look_up_heuristics_1:= proc(eq,y,x,solveOptions={},odeOptions={})
  local y0, y1, y2, dterm0, dterm1, dterm2, intOptions, optIgnoreAnalyticConstraints,
        Int, exp_pow, lambda, b, bb, z, a, L, dterm0_new, dterm1_new, dterm2_new, f,
        SinCos, SinCos_arg, dterm0_orig, dterm1_orig, dterm2_orig;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   

  Int:= if has(odeOptions,freeze) then freeze(int) else int end_if;

  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  eq:= subs(eq, [diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y2) or (not has(eq,y0) and not has(eq,y1)) then 
    return(FAIL)
  end_if;

  dterm0:= combine(expand(diff(eq,y0)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  dterm1:= combine(expand(diff(eq,y1)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  dterm2:= combine(expand(diff(eq,y2)/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/);
  dterm0_orig:= diff(eq,y0);
  dterm1_orig:= diff(eq,y1);
  dterm2_orig:= diff(eq,y2);

  if ode::odeIszero(dterm2) and 
     iszero(expand(dterm2,optIgnoreAnalyticConstraints)) then 
    return(FAIL)
  end_if;

  z:= genident(); //solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));

  // Heuristic to find solutions of the form 'exp(-a/lambda*exp(lambda*x))'
  if has(dterm0,exp) then
    exp_pow:= {};
    misc::maprec(dterm0,
                 {"exp"} = proc(elem)
                             begin
                               if has(elem,x) and 
                                  iszero(expand(diff(op(elem,1),x,x),
                                                optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
    exp_pow:= [op(exp_pow)];
    lambda:= FAIL; 
    if nops(exp_pow) = 2 then
      if ode::odeIszero(exp_pow[1]-2*exp_pow[2]) and 
         iszero(expand(exp_pow[1]-2*exp_pow[2],optIgnoreAnalyticConstraints)) then
        lambda:= exp_pow[2];
      elif ode::odeIszero(exp_pow[2]-2*exp_pow[1]) and 
           iszero(expand(exp_pow[2]-2*exp_pow[1],optIgnoreAnalyticConstraints)) then 
        lambda:= exp_pow[1];
      end_if;
      if lambda <> FAIL then 
        bb:= combine(sqrt(ode::normal(-diff(subs(dterm0,[exp(2*lambda*x)=z,exp(expand(2*lambda*x))=z]),z))),
                     IgnoreAnalyticConstraints);
        if not has(bb,x) then 
          for b in [-bb,bb] do 
            y0:= exp(-b/lambda*exp(lambda*x));
            if ode::odeIszero(dterm2*diff(y0,x,x)+dterm1*diff(y0,x)+dterm0*y0) and 
               iszero(expand(combine(dterm2*diff(y0,x,x)+dterm1*diff(y0,x)+dterm0*y0, 
                                     exp),optIgnoreAnalyticConstraints)) then 
              y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
              return({y0,y1});
            end_if;
          end_for;
        end_if;    
      end_if;  
    elif nops(exp_pow) = 1 and has(dterm1,exp) then 
      lambda:= exp_pow[1];
      a:= diff(subs(dterm1,[exp(lambda*x)=z,exp(expand(lambda*x))=z]),z);
      y0:= exp(-a/lambda*exp(lambda*x));
      if ode::odeIszero(dterm2*diff(y0,x,x)+dterm1*diff(y0,x)+dterm0*y0) and 
         iszero(expand(combine(dterm2*diff(y0,x,x)+dterm1*diff(y0,x)+dterm0*y0, 
                               exp),optIgnoreAnalyticConstraints)) then 
        y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_if;
  end_if;

  // Heuristic to find solutions of the form 'x^n'
  L:= ode::solveWithoutProperties(dterm2/x^2*z^2 + (dterm1/x-dterm2/x^2)*z + dterm0,z, 
            IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  L:= select(ode::normal(L),_not@has,x);
  if type(L) = DOM_SET and L <> {} then 
    y0:= x^(L[1]);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find solutions of the form 'exp(lambda*x)'
  L:= ode::solveWithoutProperties(dterm2*z^2 + dterm1*z + dterm0,z, 
            IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  L:= select(ode::normal(L),_not@has,x);
  if type(L) = DOM_SET and L <> {} then 
    y0:= exp(L[1]*x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find solutions of the form 'exp(lambda*x^2)'
  L:= ode::solveWithoutProperties((4*dterm2*x^2)*z^2 + (2*dterm2+2*dterm1*x)*z + dterm0,z, 
            IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  L:= select(ode::normal(L),_not@has,x);
  if type(L) = DOM_SET and L <> {} then 
    y0:= exp(L[1]*x^2);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find solutions of the form 'exp(lambda/x)'
  L:= ode::solveWithoutProperties((dterm2/x^4)*z^2 + ((2*dterm2)/x^3-dterm1/x^2)*z + dterm0,z, 
            IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  L:= select(ode::normal(L),_not@has,x);
  if type(L) = DOM_SET and L <> {} then 
    y0:= exp(L[1]/x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find solutions of the form 'ln(lambda*x)'
  if not iszero(dterm0) then 
    lambda:= expand(ode::normal(exp((dterm2 - dterm1*x)/(dterm0*x^2))/x));
    if not has(lambda,x) then 
      y0:= ln(lambda*x);
      y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;  

  // Heuristic to find solutions of the form 'x*exp(lambda*x)'
  L:= ode::solveWithoutProperties(dterm2*z^2 + (dterm1+(2*dterm2)/x)*z + dterm0+dterm1/x,z, 
            IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  L:= select(ode::normal(L),_not@has,x);
  if type(L) = DOM_SET and L <> {} then 
    y0:= x*exp(L[1]*x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;
 
  // Heuristic to find solutions of the form 'x^2*exp(lambda*x)'
  L:= ode::solveWithoutProperties(dterm2*z^2 + (dterm1+(4*dterm2)/x)*z + dterm0+(2*dterm1)/x+(2*dterm2)/x^2,z,  
            IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  L:= select(ode::normal(L),_not@has,x);
  if type(L) = DOM_SET and L <> {} then 
    y0:= x^2*exp(L[1]*x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find solutions of the form '1/x*exp(lambda*x)'
  L:= ode::solveWithoutProperties(dterm2*z^2 + (dterm1-(2*dterm2)/x)*z + dterm0-dterm1/x+(2*dterm2)/x^2,z, 
            IgnoreSpecialCases,optIgnoreAnalyticConstraints);

  L:= select(ode::normal(L),_not@has,x);
  if type(L) = DOM_SET and L <> {} then 
    y0:= 1/x*exp(L[1]*x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find solutions of the form '1/x^2*exp(lambda*x)'
  L:= ode::solveWithoutProperties(dterm2*z^2 + (dterm1-(4*dterm2)/x)*z + dterm0-(2*dterm1)/x+(6*dterm2)/x^2,z,  
            IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  L:= select(ode::normal(L),_not@has,x);
  if type(L) = DOM_SET and L <> {} then 
    y0:= 1/x^2*exp(L[1]*x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;
    
  // Heuristic to find solutions of the form '1 + lambda/x'  
  L:= ode::solveWithoutProperties(((dterm0*x^2-dterm1*x+2*dterm2)/x^3)*z + dterm0,z,  
            IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  L:= select(ode::normal(L),_not@has,x);
  if type(L) = DOM_SET and L <> {} then 
    y0:= 1+lambda/x;
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'sin(x)'
  if ode::odeIszero(dterm1*cos(x) + dterm0*sin(x) - dterm2*sin(x)) and 
     iszero(expand(ode::normal(dterm1*cos(x) + dterm0*sin(x) - dterm2*sin(x)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= sin(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;
    
  // Heuristic to find the solution 'cos(x)'
  if ode::odeIszero(dterm0*cos(x) - dterm2*cos(x) - dterm1*sin(x)) and 
     iszero(expand(ode::normal(dterm0*cos(x) - dterm2*cos(x) - dterm1*sin(x)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= cos(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'tan(x)'
  if ode::odeIszero(dterm1 + dterm1*tan(x)^2 + 2*dterm2*tan(x)^3 + dterm0*tan(x) + 2*dterm2*tan(x)) and 
     iszero(expand(ode::normal(dterm1 + dterm1*tan(x)^2 + 2*dterm2*tan(x)^3 + dterm0*tan(x) + 2*dterm2*tan(x)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= cos(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;
  
  // Heuristic to find the solution 'sinh(x)'
  if ode::odeIszero(dterm1*cosh(x) + dterm0*sinh(x) + dterm2*sinh(x)) and 
     iszero(expand(ode::normal(dterm1*cosh(x) + dterm0*sinh(x) + dterm2*sinh(x)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= sinh(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;
  
  // Heuristic to find the solution 'cosh(x)'
  if ode::odeIszero(dterm0*cosh(x) + dterm2*cosh(x) + dterm1*sinh(x)) and 
     iszero(expand(ode::normal(dterm0*cosh(x) + dterm2*cosh(x) + dterm1*sinh(x)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= cosh(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'tanh(x)'
  if ode::odeIszero(dterm1 - dterm1*tanh(x)^2 + 2*dterm2*tanh(x)^3 + dterm0*tanh(x) - 2*dterm2*tanh(x)) and 
     iszero(expand(ode::normal(dterm1 - dterm1*tanh(x)^2 + 2*dterm2*tanh(x)^3 + dterm0*tanh(x) - 2*dterm2*tanh(x)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= tanh(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'arcsin(x)'
  if ode::odeIszero((dterm1 + dterm2*x - dterm1*x^2 + dterm0*arcsin(x)*(1 - x^2)^(3/2))/(1 - x^2)^(3/2)) and 
     iszero(expand(ode::normal((dterm1 + dterm2*x - dterm1*x^2 + dterm0*arcsin(x)*(1 - x^2)^(3/2))/(1 - x^2)^(3/2)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= arcsin(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'arccos(x)'
  if ode::odeIszero(-(dterm1 + dterm2*x - dterm1*x^2 - dterm0*arccos(x)*(1 - x^2)^(3/2))/(1 - x^2)^(3/2)) and 
     iszero(expand(ode::normal(-(dterm1 + dterm2*x - dterm1*x^2 - 
                           dterm0*arccos(x)*(1 - x^2)^(3/2))/(1 - x^2)^(3/2)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= arccos(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'arctan(x)'
  if ode::odeIszero((dterm1 - 2*dterm2*x + dterm1*x^2 + dterm0*arctan(x) + 
                           2*dterm0*x^2*arctan(x) + dterm0*x^4*arctan(x))/(x^4 + 2*x^2 + 1)) and 
     iszero(expand(ode::normal((dterm1 - 2*dterm2*x + dterm1*x^2 + dterm0*arctan(x) + 
                           2*dterm0*x^2*arctan(x) + dterm0*x^4*arctan(x))/(x^4 + 2*x^2 + 1)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= arctan(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'arcsinh(x)'
  if ode::odeIszero((dterm1 - dterm2*x + dterm1*x^2 + dterm0*arcsinh(x)*(x^2 + 1)^(3/2))/(x^2 + 1)^(3/2)) and 
     iszero(expand(ode::normal((dterm1 - dterm2*x + dterm1*x^2 + dterm0*arcsinh(x)*(x^2 + 1)^(3/2))/(x^2 + 1)^(3/2)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= arcsinh(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'arccosh(x)'
  if ode::odeIszero(-(dterm1 + dterm2*x - dterm1*x^2 - dterm0*arccosh(x)*(x^2 - 1)^(3/2))/(x^2 - 1)^(3/2)) and 
     iszero(expand(ode::normal(-(dterm1 + dterm2*x - dterm1*x^2 - dterm0*arccosh(x)*(x^2 - 1)^(3/2))/(x^2 - 1)^(3/2)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= arccosh(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'arctanh(x)'
  if ode::odeIszero((dterm1 + 2*dterm2*x - dterm1*x^2 + 
                           dterm0*arctanh(x) - 2*dterm0*x^2*arctanh(x) +
                           dterm0*x^4*arctanh(x))/(x^4 - 2*x^2 + 1)) and 
     iszero(expand(ode::normal((dterm1 + 2*dterm2*x - dterm1*x^2 + 
                           dterm0*arctanh(x) - 2*dterm0*x^2*arctanh(x) +
                           dterm0*x^4*arctanh(x))/(x^4 - 2*x^2 + 1)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= arctanh(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'cot(x)'
  if ode::odeIszero(dterm0*cot(x) - dterm1 + 2*dterm2*cot(x) - dterm1*cot(x)^2 + 2*dterm2*cot(x)^3) and 
     iszero(expand(ode::normal(dterm0*cot(x) - dterm1 + 2*dterm2*cot(x) - dterm1*cot(x)^2 + 2*dterm2*cot(x)^3),
                   optIgnoreAnalyticConstraints)) then 
    y0:= cot(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'arccot(x)'
  if iszero(expand(ode::normal((2*dterm2*x - dterm1 - dterm1*x^2 + dterm0*arccot(x) + 
                           2*dterm0*x^2*arccot(x) + dterm0*x^4*arccot(x))/(x^4+2*x^2+1)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= arccot(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'coth(x)'
  if ode::odeIszero(dterm1 + dterm0*coth(x) - 2*dterm2*coth(x) - dterm1*coth(x)^2 + 
                          2*dterm2*coth(x)^3) and 
     iszero(expand(ode::normal(dterm1 + dterm0*coth(x) - 2*dterm2*coth(x) - dterm1*coth(x)^2 + 
                          2*dterm2*coth(x)^3),
                   optIgnoreAnalyticConstraints)) then 
    y0:= coth(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'arccoth(x)'
  if ode::odeIszero((dterm1 + 2*dterm2*x - dterm1*x^2 + dterm0*arccoth(x) - 
                           2*dterm0*x^2*arccoth(x) + dterm0*x^4*arccoth(x))/(x^4-2*x^2+1)) and 
     iszero(expand(ode::normal((dterm1 + 2*dterm2*x - dterm1*x^2 + dterm0*arccoth(x) - 
                           2*dterm0*x^2*arccoth(x) + dterm0*x^4*arccoth(x))/(x^4-2*x^2+1)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= arccoth(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'sec(x)'
  if ode::odeIszero((dterm0 + 3*dterm2 + dterm1*sin(2*x) + dterm0*cos(2*x) - dterm2*cos(2*x))/(2*cos(x)^3)) and 
     iszero(expand(ode::normal((dterm0 + 3*dterm2 + dterm1*sin(2*x) + dterm0*cos(2*x) - dterm2*cos(2*x))/(2*cos(x)^3)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= sec(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'arcsec(x)'
  if ode::odeIszero((dterm2 - dterm1*x + dterm1*x^3 - 2*dterm2*x^2 + 
                           dterm0*x^2*arccos(1/x)*(x^2 - 1)^(3/2))/(x^2*(x^2 - 1)^(3/2))) and 
     iszero(expand(ode::normal((dterm2 - dterm1*x + dterm1*x^3 - 2*dterm2*x^2 + 
                           dterm0*x^2*arccos(1/x)*(x^2 - 1)^(3/2))/(x^2*(x^2 - 1)^(3/2))),
                   optIgnoreAnalyticConstraints)) then 
    y0:= arcsec(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'sech(x)'
  if ode::odeIszero((dterm0 - 3*dterm2 - dterm1*sinh(2*x) + dterm0*cosh(2*x) + 
                           dterm2*cosh(2*x))/(2*cosh(x)^3)) and 
     iszero(expand(ode::normal((dterm0 - 3*dterm2 - dterm1*sinh(2*x) + dterm0*cosh(2*x) + 
                           dterm2*cosh(2*x))/(2*cosh(x)^3)),
                   optIgnoreAnalyticConstraints)) then 
    y0:= sech(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // Heuristic to find the solution 'arcsech(x)'
  if ode::odeIszero((dterm0*x^2*arccosh(1/x)*(x^2 - 1)^(3/2) - dterm2*x^2*2*I - 
                           dterm1*x*I + dterm1*x^3*I + dterm2*I)/(x^2*(x^2 - 1)^(3/2))) and 
     iszero(expand(ode::normal((dterm0*x^2*arccosh(1/x)*(x^2 - 1)^(3/2) - dterm2*x^2*2*I - 
                           dterm1*x*I + dterm1*x^3*I + dterm2*I)/(x^2*(x^2 - 1)^(3/2))),
                   optIgnoreAnalyticConstraints)) then 
    y0:= arcsech(x);
    y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
    return({y0,y1});
  end_if;

  // ode::normalization 
  if dterm2 <> 1 then 
    dterm2_new:= 1;
    dterm1_new:= expand(dterm1/dterm2/*,optIgnoreAnalyticConstraints*/);
    dterm0_new:= expand(dterm0/dterm2/*,optIgnoreAnalyticConstraints*/);
  else
    dterm2_new:= dterm2;
    dterm1_new:= dterm1;
    dterm0_new:= dterm0;
  end_if;    

  /*
     All ODEs of the form y''(x) + f(x)*y'(x) + b*(b+f(x)*tan(b*x))*y(x) = 0 have 
     the solution y0(x) = cos(b*x). The following lines mathc this pattern of ODEs
     and compute the solution. 
  */
   
  f:= dterm1_new;
  if type(dterm0_new) = "_plus" then 
    if ((L:= select({op(dterm0_new)},_not@has,x))) <> {} then 
      bb:= combine(sqrt(_plus(op(L))),IgnoreAnalyticConstraints);
    else 
      bb:= 0;
    end_if;
    for b in [-bb,bb] do 
      if ode::odeIszero(dterm0_new-b*(b+f*tan(b*x))) and 
         iszero(combine(expand(rewrite(dterm0_new-b*(b+f*tan(b*x)),sincos),optIgnoreAnalyticConstraints),
                        optIgnoreAnalyticConstraints)) then 
        y0:= cos(b*x);
        y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_for;    
  end_if;

  /*
     All ODEs of the form y''(x) + f(x)*y'(x) + b*(b-f(x)*cot(b*x))*y(x) = 0 have 
     the solution y0(x) = sin(b*x). The following lines mathc this pattern of ODEs
     and compute the solution. 
  */
   
  if type(dterm0_new) = "_plus" then 
    if ((L:= select({op(dterm0_new)},_not@has,x))) <> {} then 
      bb:= combine(sqrt(_plus(op(L))),IgnoreAnalyticConstraints);
    else 
      bb:= 0;
    end_if;
    for b in [-bb,bb] do 
      if not iszero(b) and 
         ode::odeIszero(dterm0_new-b*(b-f*cot(b*x))) and 
         iszero(combine(expand(rewrite(dterm0_new-b*(b-f*cot(b*x)),sincos),optIgnoreAnalyticConstraints),
                        optIgnoreAnalyticConstraints)) then 
        y0:= sin(b*x);
        y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_for;    
  end_if;

  /*
     All ODEs of the form y''(x) + f(x)*y'(x) - b*(b+f(x)*tanh(b*x))*y(x) = 0 have 
     the solution y0(x) = cosh(b*x). The following lines mathc this pattern of ODEs
     and compute the solution. 
  */
   
  if type(dterm0_new) = "_plus" then 
    if ((L:= select({op(dterm0_new)},_not@has,x))) <> {} then 
      bb:= combine(sqrt(-_plus(op(L))),IgnoreAnalyticConstraints);
    else 
      bb:= 0;
    end_if;
    for b in [-bb,bb] do 
      if ode::odeIszero(dterm0_new+b*(b+f*tanh(b*x))) and 
         iszero(combine(expand(rewrite(dterm0_new+b*(b+f*tanh(b*x)),sincos),optIgnoreAnalyticConstraints),
                        optIgnoreAnalyticConstraints)) then 
        y0:= cosh(b*x);
        y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_for;    
  end_if;

  /*
     All ODEs of the form y''(x) + f(x)*y'(x) - b*(b+f(x)*coth(b*x))*y(x) = 0 have 
     the solution y0(x) = sinh(b*x). The following lines mathc this pattern of ODEs
     and compute the solution. 
  */
   
  if type(dterm0_new) = "_plus" then 
    if ((L:= select({op(dterm0_new)},_not@has,x))) <> {} then 
      bb:= combine(sqrt(-_plus(op(L))),IgnoreAnalyticConstraints);
    else 
      bb:= 0;
    end_if;
    for b in [-bb,bb] do 
      if not iszero(b) and 
         ode::odeIszero(dterm0_new+b*(b+f*coth(b*x))) and 
         iszero(combine(expand(rewrite(dterm0_new+b*(b+f*coth(b*x)),sincos),optIgnoreAnalyticConstraints),
                        optIgnoreAnalyticConstraints)) then 
        y0:= sinh(b*x);
        y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
        return({y0,y1});
      end_if;
    end_for;    
  end_if;

  // eq. 53 & 54 from Section 2.1.4-5., p. 256

  if ode::odeIszero(expand(dterm1_orig,optIgnoreAnalyticConstraints)) and 
     iszero(expand(dterm1_orig,optIgnoreAnalyticConstraints)) then 
    for SinCos in {sinh,cosh} do  
      SinCos_arg:= {};  
      misc::maprec(dterm0_orig,
                   {expr2text(SinCos)} = proc(elem)
                                         begin
                                           if has(elem,x) and 
                                              iszero(expand(diff(op(elem,1),x,x),
                                                            optIgnoreAnalyticConstraints)) then 
                                             SinCos_arg:= SinCos_arg union {diff(op(elem,1),x)}
                                           end_if;
                                           elem;
                                         end_proc);  
      SinCos_arg:= [op(SinCos_arg)];
      for b in SinCos_arg do 
        if not has(b,x) and not has(b,z) then 
          a:= ode::normal(-diff(subs(dterm0_orig,[SinCos(b*x)=z,SinCos(expand(b*x))=z]),z)/b);
          if SinCos = cosh and ode::odeIszero(dterm0_orig+a*(a*sinh(b*x)^2+b*cosh(b*x))) and 
             iszero(expand(combine(dterm0_orig+a*(a*sinh(b*x)^2+b*cosh(b*x)),sincos),
                                            optIgnoreAnalyticConstraints)) then
            y0:= exp(a/b*cosh(b*x));
            y1:= y0*int(exp(-int(dterm1_orig/dterm2_orig,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1})
          elif SinCos = sinh and ode::odeIszero(dterm0_orig+a*(a*cosh(b*x)^2+b*sinh(b*x))) and 
               iszero(expand(combine(dterm0_orig+a*(a*cosh(b*x)^2+b*sinh(b*x)),sincos),
                                            optIgnoreAnalyticConstraints)) then
            y0:= exp(a/b*sinh(b*x));
            y1:= y0*int(exp(-int(dterm1_orig/dterm2_orig,x,intOptions))/y0^2,x,intOptions);
            return({y0,y1})
          end_if;
        end_if;
      end_for;
    end_for;
  end_if;

  // Heuristic to find solutions of the form 'exp(lambda*x)' 
  // in case of trigonometric/hyperbolic coefficients.
  if has(dterm2*z^2+dterm1*z+dterm0,[exp,sin,cos,tan,sinh,cosh,tanh]) then 
    L:= ode::solveWithoutProperties(rewrite(dterm2*z^2 + dterm1*z + dterm0,exp),z, 
              IgnoreSpecialCases,optIgnoreAnalyticConstraints);  
    L:= select(ode::normal(L),_not@has,x);
    if type(L) = DOM_SET and L <> {} then 
      y0:= exp(L[1]*x);
      y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;

  // Heuristic to find solutions of the form 'exp(lambda*x^2)'
  // in case of trigonometric/hyperbolic coefficients.
  if has(dterm2*z^2+dterm1*z+dterm0,[exp,sin,cos,tan,sinh,cosh,tanh]) then 
    L:= ode::solveWithoutProperties(rewrite((4*dterm2*x^2)*z^2 + (2*dterm2+2*dterm1*x)*z + dterm0, exp),z, 
              IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    L:= select(ode::normal(L),_not@has,x);
    if type(L) = DOM_SET and L <> {} then 
      y0:= exp(L[1]*x^2);
      y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;
  
  // Heuristic to find solutions of the form 'exp(lambda/x)'
  // in case of trigonometric/hyperbolic coefficients.
  if has(dterm2*z^2+dterm1*z+dterm0,[exp,sin,cos,tan,sinh,cosh,tanh]) then 
    L:= ode::solveWithoutProperties(rewrite((dterm2/x^4)*z^2 + ((2*dterm2)/x^3-dterm1/x^2)*z + dterm0,exp),z, 
              IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    L:= select(ode::normal(L),_not@has,x);
    if type(L) = DOM_SET and L <> {} then 
      y0:= exp(L[1]/x);
      y1:= y0*Int(exp(-Int(dterm1/dterm2,x,intOptions))/y0^2,x,intOptions);
      return({y0,y1});
    end_if;
  end_if;  
  

  // ---------------------------

  return(FAIL);
end_proc: