/*
      REFERENCES: 
       [1] Manuel Bronstein, Sebastian Lafaille: Solutions of linear ordinary 
           differential equations in terms of special functions, ISSAC 2004

      ARGUMENTS:
                eq -- expression encoding a 2nd order homogeneous linear ODEs
                 y -- the dependent ODE variable 
                 x -- the independent ODE variable 
      solveOptions -- set of options to be passed in calls of 'solve'
        odeOptions -- set of internal ODE options  

      RETURN VALUE:
       -- The method returns a set of solutions. In the generic situations 
          where the algorithms are successful, the set will contain two 
          independent solutions. 
       -- In degenerate situations only one particular solution may be found 
          (this is treated appropriately by the methods calling 
           'ode::specfunc'). 
       -- When the methods cannot be applied, i.e. no solution can be computed, 
          the empty set is returned. 
 
*/

ode::specfunc:=
proc(eq,y,x,solveOptions,odeOptions)
  local unimod,v,nuinfty,dv,ddv,u,dvi,vi,wi,lQ,i,j,hi,Q,dQ,n,eqz,p,a,P,r,uu,lc,sol,
        N,z,m,mu,nu,eps,valuation,vars,optIgnoreAnalyticConstraints,f1,f2,
        degr,coeffi;
save MAXEFFORT;
begin
  // Local procedure - wrapper call to 'degree' in order to speed things up 
  degr:= proc(p,x) 
           local res;
         begin 
           res:= degree(p,x);
           if res <> FAIL then 
             return(res);
           else 
             return(degree(p,[x]));
           end_if;
         end_proc:
  // Local procedure - wrapper call to 'coeff' in order to speed things up 
  coeffi:= proc(p,x,i)
             local res;
           begin 
             res:= coeff(p,x,i);
             if res <> FAIL then 
               return(res);
             else 
               return(coeff(p,[x],i));
             end_if;
           end_proc:
  /*
     Local Procedure 'valuation' 
      -- valuation function from [1]

     PARAMETERS: 
      q -- a polynomial or rational expression 
      x -- the independent variable 
      p -- a polynomial or 'infinity'  

     NOTE: see page 3, beginning of Section 4, in [1] for details 
           on the valuation
                 
     RETURN VALUE: 
      -- returns an integer or 'infinity' (see [1] for details)   
     
     The function 'valuation' called 'nu_p(q)' is such that:
      -- nu_infty(0) = nu_p(0) = 0 for p in C[x]
      -- nu_infty(q) = -degree(q) for q in C[x]\{0}
      -- nu_p(q) = max{n in N | p^n|q} for p in C[x] irreducible
      -- nu_infty(a/b) = nu_infty(a) - nu_infty(b) and
      -- nu_p(a/b) = nu_p(a) - nu_p(b) for polynomials a and b      
  */
  valuation :=
  proc(q,x,p)
    local dq,nq,ddq,dnq;
  begin
    if p <> infinity and not testtype(p, Type::PolyExpr(x)) then
      error("Illegal third argument")
    end_if;
    if iszero(q) then
      return(infinity);
    elif testtype(q, Type::RatExpr(x)) or has(odeOptions, IgnoreAnalyticConstraints) then
        q:= ode::normal(q,Rationalize=None); 
       dq:= denom(q); 
       nq:= numer(q);
      ddq:= degr(dq,x); 
      dnq:= degr(nq,x);
      if p = infinity then
        return(-dnq+ddq);
      else 
        error("Illegal third argument");
      end_if;  
      /* 
      The following code is not yet in use. The helper method 'valuation' is 
      always called with 'p = infinity'. For extending this method, see [1] and 
      activate the following lines.  

      elif irreducible(p) then
        n:= ddq+1; 
        repeat 
          n:= n-1; 
        until 
          iszero(divide(dq, p^n, [x], Rem)) 
        end_repeat:
        m:= dnq+1; 
        repeat 
          m:= m-1; 
        until 
          iszero(divide(nq, p^m, [x], Rem)) 
        end_repeat:
        return(m-n);
      else
        error("Illegal third argument : polynomial not irreducible")
      end_if
      */
    else
      error("Illegal first argument : not a rational function")
    end_if;
  end_proc:

  // =================================================================
  // MAIN of 'ode::specfunc' 
  // =================================================================

  MAXEFFORT:= MAXEFFORT/5; 

  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  
  if has(odeOptions, IgnoreAnalyticConstraints) then 
    eq:= ode::isLODE(eq, y(x), Hlode, solveOptions, odeOptions);
  else 
    eq:= ode::isLODE(eq, y(x), HlodeOverRF, solveOptions, odeOptions);
  end_if;
  
  if eq <> FALSE and eq[4] = 2 then
    // Compute unimodular form. 
    unimod:= ode::mkUnimodular(eq, Transform, solveOptions, odeOptions);
    v:= ode::normal(-unimod[3][1]);
    nuinfty:= valuation(v,x,infinity);
    dv:= denom(v); 
    ddv:= diff(dv,x);
    u:= gcd(dv,ddv); 
    vi:= divide(dv,u,[x],Quo); 
    wi:= divide(ddv,u,[x],Quo);
    lQ:=[]; 
    i:=1;
    repeat
      dvi:= diff(vi,x);
      hi:= gcd(vi,wi-dvi);
      vi:= divide(vi,hi,[x],Quo); 
      wi:= divide(wi-dvi,hi,[x],Quo);
      i:= i+1;
      lQ:= append(lQ,hi);
    until 
      vi=1 or vi=-1 
    end_repeat:
    r:= genident();
    a:= genident("a");
    uu:= genident("u");
    mu:= genident("m");
    nu:= genident("n");
    eps:= genident("e");
    
    /*----------------------------------------------------------------------------
      CASE: 'Airy'

      See [1], Section 5.1, for details.
    ------------------------------------------------------------------------------*/
    Q:= _mult(lQ[3*i+2]^i $ i=1..floor((nops(lQ)-2)/3));
    dQ:= degr(Q,x);
    n:= max(dQ, dQ + floor((2-nuinfty)/3));
    eqz:= 3*diff(y(x),x$2)^2-2*diff(y(x),x)*diff(y(x),x$3)+
          4*y(x)*diff(y(x),x)^4-4*v*diff(y(x),x)^2; // [1] eq. (15)
    p:= numer(evalAt(eqz, y(x)=r(x)/Q));
    P:= _plus(a.i*x^i $ i=0..n);
    p:= evalAt(p, r(x)=P);
    lc:= {coeffi(p,x,i) $ i=0..degr(p,x)};
    N:= select([i $ i=0..n], ()->coeffi(Q,x,args(1))<>0)[1];
    lc:= lc union {1 - _plus((coeffi(Q,x,i)*a.N - coeffi(Q,x,N)*a.i)*a.(n+i+1) $ i=0..n)};
    vars:= [a.i $ i=0..2*n+1];
    // We explicitly need 'IgnoreSpecialCases' independent of the fact whether it 
    // is contained in 'solveOptions' or not. 
    lc:=solve(lc, vars, VectorFormat, IgnoreSpecialCases, op(solveOptions));
    sol:= solvelib::getElement(lc);
    if sol <> FAIL then
      z:= ode::normal(evalAt(P/Q, [zip(vars, [op(sol)], _equal)]),Rationalize=None);
      m:= ode::normal(1/diff(z,x),Rationalize=None);
      m:= factor(simplify(unimod[4]^2*m,optIgnoreAnalyticConstraints));
      // This is just to make the code more robust. I do not have an example 
      // at hand where we get a 'piecewise' here, but I want to make sure that
      // we treat this case appropriately if it occurs. 
      if type(m) = piecewise then 
        m:= piecewise::disregardPoints(m);
        if type(m) = piecewise then 
          return({})
        end_if;
      end_if;
      // Note: 'm' must be of type 'Factored'. This can be converted to 'DOM_LIST'.
      m:= coerce(m, DOM_LIST):
      m:=_mult(m[2*i]^(m[2*i+1]/2) $ i in select({j $j=1..nops(m) div 2},
                                                 ()->has(m[2*args(1)],x)));
      if not has(m,x) then
        return({airyAi(z),airyBi(z)})
      else
        return({m*airyAi(z),m*airyBi(z)})
      end_if;
    end_if;
    
    /*----------------------------------------------------------------------------
      CASE: 'Bessel'

      See [1], Section 5.2, for details.
    ------------------------------------------------------------------------------*/
    Q:= _mult(lQ[2*i+2]^i $ i=1..floor((nops(lQ)-2)/2));
    dQ:= degr(Q,x);
    n:= max(dQ + 1, dQ + floor(1-nuinfty/2));
    eqz:= 3*diff(y(x),x$2)^2-2*diff(y(x),x)*diff(y(x),x$3)+(4*nu^2-1)*diff(y(x),x)^4/y(x)^2-
          4*eps*diff(y(x),x)^4-4*v*diff(y(x),x)^2; // [1] eq. (17)
    p:= numer(evalAt(eqz, y(x)=r(x)/Q));
    P:= _plus(a.i*x^i $ i=0..n); 
    p:= evalAt(p, r(x)=P);
    lc:= {coeffi(p,x,i) $ i=0..degr(p,x)};
    N:= select([i $ i=0..n], ()->coeffi(Q,x,args(1))<>0)[1];
    lc:= lc union {1 - _plus((coeffi(Q,x,i)*a.N - coeffi(Q,x,N)*a.i)*a.(n+i+1) $ i=0..n)};
    vars:= [eps, nu, a.i $ i=0..2*n+1];
    // We explicitly need 'IgnoreSpecialCases' independent of the fact whether it 
    // is contained in 'solveOptions' or not. 
    lc:=solve(lc union {eps^2-1}, vars, VectorFormat, IgnoreSpecialCases, op(solveOptions));
    sol:= solvelib::getElement(lc);
    if sol <> FAIL then
      z:= evalAt(ode::normal(P/Q,Rationalize=None), [zip(vars, [op(sol)], _equal)]);
      m:= ode::normal(z/diff(z,x),Rationalize=None);
      m:= factor(simplify(unimod[4]^2*m,optIgnoreAnalyticConstraints));
      // This is just to make the code more robust. I do not have an example 
      // at hand where we get a 'piecewise' here, but I want to make sure that
      // we treat this case appropriately if it occurs. 
      if type(m) = piecewise then 
        m:= piecewise::disregardPoints(m);
        if type(m) = piecewise then 
          return({})
        end_if;
      end_if;
      // Note: 'm' must be of type 'Factored'. This can be converted to 'DOM_LIST'.
      m:= coerce(m, DOM_LIST);
      m:=_mult(m[2*i]^(m[2*i+1]/2) $ i in select({j $j=1..nops(m) div 2}, ()->has(m[2*args(1)],x)));
      nu:= evalAt(nu, zip(vars, [op(sol)], _equal));
      eps:= evalAt(eps, zip(vars, [op(sol)], _equal));
      if eps = 1 then
        if not has(m,x) then
          return({besselJ(nu,z), besselY(nu,z)})
        else
          return({m*besselJ(nu,z), m*besselY(nu,z)})
        end_if;
      else
        if not has(m,x) then
          return({besselI(nu,z), besselK(nu,z)})
        else
          return({m*besselI(nu,z), m*besselK(nu,z)})
        end_if;  
      end_if;
    end_if;
    
    /*----------------------------------------------------------------------------
      CASE: 'Kummer' is missing. The reason is that Kummer and Whittaker functions
            are rationally equivalent (see introductory remarks for Section 5 of 
            [1]). 
            Since the Whittaker case is implemented below (taking care of rational 
            equivalence transformations), there is no need for additionally doing 
            the Kummer case (anyway, introduction of the Kummer case gives 
            fundamental sets of solutions in terms of 'kummerU' and 'hypergeom', 
            which is less elemantary than the solutions in terms of Whittaker 
            functions). 
    ------------------------------------------------------------------------------*/
    
    /*----------------------------------------------------------------------------
      CASE: 'Whittaker'

      See [1], Section 5.4, for details.
    ------------------------------------------------------------------------------*/
    Q:= _mult(lQ[2*i+2]^i $ i=1..floor((nops(lQ)-2)/2));
    dQ:= degr(Q,x);
    n:= max(dQ + 1, dQ + floor(1-nuinfty/2));
    eqz:= 3*diff(y(x),x$2)^2-2*diff(y(x),x)*diff(y(x),x$3)+
          (1-4*mu/y(x)-(1-4*nu^2)/y(x)^2)*diff(y(x),x)^4-4*v*diff(y(x),x)^2; // [1] eq. (20)
    p:= numer(evalAt(eqz, y(x)=r(x)/Q));
    P:= _plus(a.i*x^i $ i=0..n); 
    p:= evalAt(p, r(x)=P);
    lc:= {coeffi(p,x,i) $ i=0..degr(p,x)};
    N:= select([i $ i=0..n], ()->coeffi(Q,x,args(1))<>0)[1];
    lc:= lc union {1 - _plus((coeffi(Q,x,i)*a.N - coeffi(Q,x,N)*a.i)*a.(n+i+1) $ i=0..n)};
    vars:= [mu, nu, a.i $ i=0..2*n+1];
    lc:= solve(lc, vars, VectorFormat, IgnoreSpecialCases, op(solveOptions));
    sol:= solvelib::getElement(lc);
    if sol <> FAIL then
      z:= subs(ode::normal(P/Q,Rationalize=None), zip(vars, [op(sol)], _equal));
      m:= ode::normal(1/diff(z,x),Rationalize=None);
      m:= factor(simplify(unimod[4]^2*m,optIgnoreAnalyticConstraints));
      // This is just to make the code more robust. I do not have an example 
      // at hand where we get a 'piecewise' here, but I want to make sure that
      // we treat this case appropriately if it occurs. 
      if type(m) = piecewise then 
        m:= piecewise::disregardPoints(m);
        if type(m) = piecewise then 
          return({})
        end_if;
      end_if;
      // Note: 'm' must be of type 'Factored'. This can be converted to 'DOM_LIST'.
      m:= coerce(m, DOM_LIST):
      m:= _mult(m[2*i]^(m[2*i+1]/2) $ i in select({j $j=1..nops(m) div 2}, ()->has(m[2*args(1)],x)));
      mu:= sol[1]; 
      nu:= sol[2];
      if not has(m,x) then
        if traperror((f1:= whittakerM(mu,nu,z))) <> 0 then 
          f1:= undefined;
        end_if;
        if traperror((f2:= whittakerW(mu,nu,z))) <> 0 then 
          f2:= undefined;
        end_if;  
        return({f1,f2} minus {undefined});
      else
        if traperror((f1:= m*whittakerM(mu,nu,z))) <> 0 then 
          f1:= undefined;
        end_if;
        if traperror((f2:= m*whittakerW(mu,nu,z))) <> 0 then 
          f2:= undefined;
        end_if;  
        return({f1,f2} minus {undefined});
      end_if;
    end_if;

    /*----------------------------------------------------------------------------
      CASE: 'Bessel with algebraic transformation'

      See [1], Section 6.2, for details.
    ------------------------------------------------------------------------------*/
    Q:= _mult(lQ[i]^(i-2) $ i=3..nops(lQ));
    dQ:= degr(Q,x);
    n:= max(0, min(2*(dQ+2)-1,2*dQ+2-nuinfty)); // should be max, but the algebraic solver is too slow
    eqz:= 3*diff(y(x),x$2)^2-2*diff(y(x),x)*diff(y(x),x$3)+
          (4*nu^2-1)*diff(y(x),x)^4/y(x)^2-4*eps*diff(y(x),x)^4-4*v*diff(y(x),x)^2; // [1] eq. (17)
    p:= evalAt(eqz, y(x)=r(x)/Q^(1/2));
    P:= _plus(a.i*x^(i/2) $ i=0..n); 
    p:= subs(p, r(x)=P, EvalChanges);
    p:= misc::maprec(p,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x and 
                                   denom(op(elem,2)) = 2 then 
                                  _power(uu,numer(op(elem,2))) 
                                else
                                  elem;
                                end_if;                                
                              end_proc);  
    p:= subs(p,x=uu^2);                          
    // p:= numer(normal(p/*,Rationalize=None*/));
    p:= expand(ode::normal(p,Expand=FALSE,List)[1],ArithmeticOnly);
    lc:= {coeffi(p,uu,i) $ i=0..degr(p,uu)};
    N:= select([i $ i=0..n], () -> coeffi(Q,x,args(1)) <> 0)[1];
    lc:= lc union {1 - _plus((coeffi(Q,x,i)*a.N - coeffi(Q,x,N)*a.i)*a.(n+i+1) $ i=0..n)};
    vars:= [eps, nu, a.i $ i=0..2*n+1];
    lc:= solve(lc union {eps^2-1}, vars, VectorFormat, IgnoreSpecialCases, op(solveOptions));
    sol:= solvelib::getElement(lc);
    if sol <> FAIL then
      z:= evalAt(ode::normal(P/Q^(1/2),Rationalize=None,Expand=FALSE), zip(vars, [op(sol)], _equal));
      m:= ode::normal(z/diff(z,x),Rationalize=None,Expand=FALSE);
      m:= factor(simplify(unimod[4]^2*m,optIgnoreAnalyticConstraints));
      // This is just to make the code more robust. I do not have an example 
      // at hand where we get a 'piecewise' here, but I want to make sure that
      // we treat this case appropriately if it occurs. 
      if type(m) = piecewise then 
        m:= piecewise::disregardPoints(m);
        if type(m) = piecewise then 
          return({})
        end_if;
      end_if;
      // Note: 'm' must be of type 'Factored'. This can be converted to 'DOM_LIST'.
      m:= coerce(m, DOM_LIST);
      m:= _mult(m[2*i]^(m[2*i+1]/2) $ i in select({j $j=1..nops(m) div 2}, ()->has(m[2*args(1)],x)));
      nu:= evalAt(nu, zip(vars, [op(sol)], _equal));
      eps:= evalAt(eps, zip(vars, [op(sol)], _equal));
      if eps = 1 then
        if not has(m,x) then
          return({besselJ(nu,z), besselY(nu,z)})
        else
          return({m*besselJ(nu,z), m*besselY(nu,z)})
        end_if;
      else
        if not has(m,x) then
          return({besselI(nu,z), besselK(nu,z)})
        else
          return({m*besselI(nu,z), m*besselK(nu,z)})
        end_if;  
      end_if;
    end_if;

    /*----------------------------------------------------------------------------
      CASE: 'Whittaker with algebraic transformation'

      See [1], Section 6.2, for details.
    ------------------------------------------------------------------------------*/
    /*
    
    NOTE: In the definition of 'n' below we need to use the maximum in order to get 
          satisfactory results. But this means solving larger systems of polynomial 
          equations. 
    
    
    Q:= _mult(lQ[i]^(i-2) $ i=3..nops(lQ));
    dQ:= degr(Q,x);
    n:= max(0, min(2*(dQ+2)-1, 2*dQ+2-nuinfty)); // should be max, but the algebraic solver is too slow
    eqz:= 3*diff(y(x),x$2)^2-2*diff(y(x),x)*diff(y(x),x$3)+
          (1-4*mu/y(x)-(1-4*nu^2)/y(x)^2)*diff(y(x),x)^4-4*v*diff(y(x),x)^2; // [1] eq. (20)
    p:= evalAt(eqz, y(x)=r(x)/Q^(1/2));
    P:= _plus(a.i*x^(i/2) $ i=0..n); 
    p:= evalAt(p, r(x)=P);
    p:= misc::maprec(p,
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x and 
                                   denom(op(elem,2)) = 2 then 
                                  _power(uu,numer(op(elem,2))) 
                                else
                                  elem;
                                end_if;                                
                              end_proc);  
    p:= subs(p,x=uu^2);                          
    // p:= numer(normal(p/*,Rationalize=None*/));
    p:= expand(ode::normal(p,Expand=FALSE,List)[1],ArithmeticOnly);
    lc:= {coeffi(p,uu,i) $ i=0..degr(p,uu)};
    N:= select([i $ i=0..n], () -> coeffi(Q,x,args(1)) <> 0)[1];
    lc:=lc union {1-_plus((coeffi(Q,x,i)*a.N - coeffi(Q,x,N)*a.i)*a.(n+i+1) $ i=0..n)};
    vars:= [mu, nu, a.i $ i=0..2*n+1];
    lc:= solve(lc, vars, VectorFormat, IgnoreSpecialCases, op(solveOptions));
    sol:= solvelib::getElement(lc);
    if sol <> FAIL then
      z:= evalAt(ode::normal(P/Q^(1/2),Rationalize=None,Expand=FALSE), zip(vars, [op(sol)], _equal));
      m:= ode::normal(1/diff(z,x),Rationalize=None,Expand=FALSE);
      m:= factor(simplify(unimod[4]^2*m,optIgnoreAnalyticConstraints));
      // This is just to make the code more robust. I do not have an example 
      // at hand where we get a 'piecewise' here, but I want to make sure that
      // we treat this case appropriately if it occurs. 
      if type(m) = piecewise then 
        m:= piecewise::disregardPoints(m);
        if type(m) = piecewise then 
          return({})
        end_if;
      end_if;
      // Note: 'm' must be of type 'Factored'. This can be converted to 'DOM_LIST'.
      m:= coerce(m, DOM_LIST):
      m:=_mult(m[2*i]^(m[2*i+1]/2) $ i in select({j $j=1..nops(m) div 2},
                                                 () -> has(m[2*args(1)],x)));
      mu:= evalAt(mu, zip(vars, [op(sol)], _equal));
      nu:= evalAt(nu, zip(vars, [op(sol)], _equal));
      if not has(m,x) then
        if traperror((f1:= whittakerM(mu,nu,z))) <> 0 then 
          f1:= undefined;
        end_if;
        if traperror((f2:= whittakerW(mu,nu,z))) <> 0 then 
          f2:= undefined;
        end_if;  
        return({f1,f2} minus {undefined});
      else
        if traperror((f1:= m*whittakerM(mu,nu,z))) <> 0 then 
          f1:= undefined;
        end_if;
        if traperror((f2:= m*whittakerW(mu,nu,z))) <> 0 then 
          f2:= undefined;
        end_if;  
        return({f1,f2} minus {undefined});
      end_if;
    end_if;
    */
  end_if;
  // ------------------------------------------------------------------------------
  
  
  return({});
end_proc:
