/* 

  DESCRIPTION:
   This file contains functions for finding power series solutions of ordinary
   differential equations with initial conditions by the method of repeated
   differentiating.

  FUNCTIONS: 
    - used Parameter:
    - ode::repdiff

  EXAMPLES: 
    >> ode::repdiff(diff(y(x),x)+sin(x)/x,y,x,[0,0],6)
    >> ode::repdiff(diff(y(x),x)^2+sin(x)/x,y,x,[0,0],6)
    >> ode::repdiff(sqrt(diff(y(x),x))+sin(x)/x,y,x,[0,0],6)
    >> ode::repdiff(diff(y(x),x,x)+sin(x)/x,y,x,[0,1,2],6)
    >> ode::repdiff(diff(y(x),x)-y(x)^2-x^2,y,x,[0,1],6)
    >> ode::repdiff(diff(y(x),x)^2-y(x)^2-x^2,y,x,[0,1],6)
    >> ode::repdiff(diff(y(x),x,x)^(1/2)+diff(y(x),x)-sin(x)/x,y,x,[0,1,2],6)
*/


ode::repdiff:= proc(eq,y,x,ivs=[],order=ORDER,dir=Real,solveOptions={},odeOptions={})
  local n,f,s,e,res,r,r0,i,sl0,dfgen,sol,intOptions;
begin
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  
  if type(ivs[1])=stdlib::Infinity then // expansion at infinity not yet implemented
    return(FAIL)
  end_if:
  
  // determine order of eq
  n:= ode::getOrder(eq,y(x),solveOptions,odeOptions);
  // generate missing initial values
  if ivs=[] then
    ivs:= [0,(D@@i)(y)(0) $i=0..n-1];
  elif nops(ivs)<n+1 then
    ivs:= ivs.[(D@@i)(y)(ivs[1]) $ i=nops(ivs)-1..n-1];
  end_if;
  // handle trivial cases (first stage)
  if n>=order then 
    return({taylor(_plus(ivs[i+2]/fact(i)*(x-ivs[1])^i $i=0..order-1),
                   x=ivs[1],order,dir)});
  elif n=-infinity then
    return({taylor(0,x=ivs[1],order,dir)});
  end_if;
  // resolve equation
  dfgen:= genident("__repDiffy");
  s:= solvelib::discreteSolve(subs(eq,diff(y(x),x$n)=dfgen),dfgen,op(solveOptions));
  if domtype(s)<>DOM_SET or nops(s)=0 then
    userinfo(2, "Cannot resolve equation for ".expr2text(diff(y(x),x$n)));
    return(FAIL);
  end_if;
  s:= {diff(y(x),x$n)=sol $ sol in s};
  // handle trivial cases (second stage)
  if n=0 then
    return(map(s,e->taylor(op(op(e),2),x=ivs[1],order, dir)));
  end_if;
  // step through any resolution
  res:= {};
  r0:= _plus(ivs[i+2]/fact(i)*(x-ivs[1])^i $i=0..n-1);
  sl0:= [x=ivs[1],diff(y(x),x$i)=ivs[i+2] $i=0..n-1];
  for e in s do
    f:= op(op(e),2); //  y^(n)=f(x,y,y'',..y^(n-1))
    r:= r0;
    for i from n to order-1 do
      if traperror((r:=r+subs(f,sl0,EvalChanges)/fact(i)*(x-ivs[1])^i))<>0 then
        // expansion around singular point
        if ode::getOrder(f,y(x),solveOptions,odeOptions)<0 then
          // treat special cases
          case n
            of 1 do
              return({series(int(f,x,intOptions),x=ivs[1],order,dir)+ivs[2]});
              break;
            otherwise
              for i from 1 to n do
                f:= expr(series(int(f,x,intOptions),x=ivs[1],order,dir)+ivs[n-i+2]);
              end_for;
              return({taylor(f,x=ivs[1],order,dir)});
          end_case;
        else
          userinfo("Only expansions around regular points are treated.");
          return(FAIL);
        end_if;
      end_if;
      // repeated differentiation
      f:= subs(diff(f,x),e,EvalChanges);
    end_for;
    res:= res union {taylor(r,x=ivs[1],order,dir)};
  end_for;
  
  return(res);
end_proc:


