/* 

  DESCRIPTION:
    This file contains functions for computing the companion system of a
    linear ordinary (homogeneous) differential equations.

  FUNCTIONS: 
    - used Parameter:
       ++ Ly, y(x)   :DOM_EXPR
       ++ y, x       :DOM_IDENT
       ++ n          :DOM_INT
       ++ R          :DOM_DOMAIN
       ++ A          :Dom::Matrix
       ++ v          :DOM_LIST or Dom::Matrix
       ++ Ly    = ordinary linear homogeneous differential equation 
       ++ y(x)  = the operator (function) of Ly  
       ++ y     = name of the operator of Ly
       ++ x     = independent variable (of the operator y)
       ++ n     = order of Ly
       ++ R     = field of functions or numbers of characteristic zero 
       ++ A     = square matrix representing a linear differential system 
       ++ v     = (row) vector of rational functions
       
    - ode::companionSystem(Ly, y(x) <, R> )
    - ode::compSys(Ly, y, x, n <, R> )
       ++ returns the companion system matrix associated with the differential 
       ++ equation Ly.  
       
    - ode::cyclicVector(A,x <, v>)
       ++ reduces the linear system Y'=A*Y to a single scalar differential
       ++ equation using cyclic vectors. For definition of cyclic vectors see [2].
       ++ For this the companion matrix C of that equation is computed such that
       ++ Z=P*Y and Z'=C*Z. To make it possible to determine solutions of A,
       ++ a list containing the last row of C as first entry and P as second
       ++ entry is returned. 
       ++ When optionally a vector v is given, only this vector is tested, if it
       ++ is cyclic; otherwise vectors are generated by random. After 2*size of A
       ++ generated (bad) vectors this process is stopped and a warning is 
       ++ printed out.
       
    - ode::isCyclicVector(A,x,v <, Coeffs>)
       ++ tests if v is a cyclic vector and returns TRUE if v is cyclic and 
       ++ FALSE otherwise.
       ++ When the option 'Coeffs' is given, then if v is cyclic a list containing
       ++ the last row of C as first entry and P as second entry is returned
       ++ (see ode::cyclicVector) otherwise FALSE.
       ++ Note: There is no argument check.
       
    - ode::scalarEquation(A,x,y <, Transform>)  
       ++ returns a scalar differential equation in y(x) which is equivalent to
       ++ the linear differential system Y'=A*Y. Note this equation is not 
       ++ uniquely determined; it depends on the choice of the cyclic vector.
       ++ When option 'Transform' is given a list consisting of the scalar
       ++ equation and the matrix P is returned (see ode::cyclicVector).


   REFERENCES:
    [1] Walter, W.: Gewhnliche Differentialgleichungen. 4th Edition.
        Berlin, Heidelberg, New York: Springer, 1990
    [2] Barkatou, M.A.: An algorithm to compute the exponential part  
        of a formal fundamental matrix solution of a linear differential system.
        J. of Appl. Alg. in Eng. Comm. and Comp. 8, pp. 1-23, 1997


   EXAMPLES: 
     > eq:= 4*x^2*diff(y(x), x$2)+4*x*diff(y(x),x)-y(x);
     > A:= ode::companionSystem(eq,y(x));
     > ode::scalarEquation(A,x,y);

*/


ode::companionSystem:= proc(eq,z,R=Dom::ExpressionField(normal),solveOptions={},odeOptions={})
  local eqs;
begin
  eqs:= ode::isLODE(eq,z,Hlode,solveOptions,odeOptions);
  if eqs=FAIL then
    error("Not an ordinary linear homogeneous differential equation");
  else
    return(ode::compSys(eqs,R,solveOptions,odeOptions));
  end_if;
end_proc:



ode::compSys:= proc(eq,y,x,n,R=Dom::ExpressionField(normal),solveOptions,odeOptions)
  local M,v,i,j;
begin
  M:= Dom::Matrix(R);
  v:= ode::vectorize(eq,y,x,n,solveOptions,odeOptions);
  assert(not iszero(v[n+1]));
  v:= map(map(v,_mult,-1/v[n+1]),R::new);
  delete v[n+1];
  return(M([[(if i+1=j then R::one else R::zero end_if) $ j=1..n] $ i=1..n-1,v]));
end_proc:



ode::scalarEquation:= proc(A:Dom::Matrix,x:DOM_IDENT,y:DOM_IDENT,o=" ",
                           solveOptions={},odeOptions={}) // Option: Transform
  local n,v;
begin
  if (n:= linalg::nrows(A)) <> linalg::ncols(A) then
    error("Matrix should be square")
  end_if;
  v:= ode::cyclicVector(A,x,[],solveOptions,odeOptions);
  if v=[] then FAIL
  elif o=Transform then 
    return([-ode::mkODE(v[1].[-1],y,x,solveOptions,odeOptions),v[2]]);
  else 
    return(-ode::mkODE(v[1].[-1],y,x,solveOptions,odeOptions));
  end_if;
end_proc:



ode::cyclicVector:= proc(A:Dom::Matrix,x:DOM_IDENT,v=[],solveOptions={},odeOptions={})
  local n,sol,triedVectors,isCyclic,i,maxTries,p,d,j;
begin
  n:= linalg::nrows(A);
  if n=1 then 
    return([expr(op(A))]);
  elif v<>[] then
    if nops(v)<>n then 
      error("Vector has wrong size");
    else
      sol:= ode::isCyclicVector(A,x,v,Coeffs,solveOptions,odeOptions);
      maxTries:= 0;
    end_if;
  else
    maxTries:= 2*n; // this is simply heuristic
  end_if;
  // test random generated vectors
  triedVectors:= {};
  isCyclic:= FALSE;
  i:= 1;
  while not isCyclic and i<=maxTries do
    repeat
      d:= (i div 2) - 1 + modp(i,2);
      if i=1 then 
        v:= [1, 0 $ (n-1)]; // should be cyclic already in many cases
      elif iszero(d) then // necessary, since Degree=0 in randpoly is not allowed
        p:= polylib::randpoly([x],IntMod(2),Degree=n-1);
        v:= [1,coeff(p,x,j) $ j=1..n-1];
      else
        v:= [1,expr(polylib::randpoly([x],IntMod(2),Degree=d)) $j=1..n-1];
      end_if;
    until not contains(triedVectors,v) 
    end_repeat;
    triedVectors:= triedVectors union {v};
    userinfo(10,"try random generated vector",v);
    sol:= ode::isCyclicVector(A,x,v,Coeffs,solveOptions,odeOptions);
    if sol<>FALSE then 
      isCyclic:= TRUE; 
    end_if;
    i:= i+1;
  end_while;
  if i>maxTries and maxTries<>0 and ode::printWarningsFlag then
    ode::odeWarning("No cyclic vector found"); 
  end_if;
  if sol=FALSE then 
    return([]); 
  else 
    return(sol); 
  end_if;

end_proc:



ode::isCyclicVector:= proc(A:Dom::Matrix,x:DOM_IDENT,v,o=" ",solveOptions={},odeOptions={}) // Options: Coeffs
  local n,M,P,i,Bn,Cn,cis,sys,solset;
begin
  // some initial definitions
  M:= Dom::Matrix(Dom::ExpressionField(normal));
  A:= M(A);
  n:= linalg::nrows(A);
  v:= M(v);
  if linalg::nrows(v)<>1 then 
    v:= M::transpose(v);
  end_if;
  cis:= [genident("c") $ i=1..n];
  // compute matrix P, consisting of inductively defined row vectors 
  P:= linalg::stackMatrix(v, (v:= diff(v,x)+v*A;) $ i=1..n-1);
  // nth derivative of v, which is the nth row of matrix B (=P'+P*A)
  Bn:= diff(v,x)+v*A;
  // generic last row of companion matrix C
  Cn:= M::transpose(M(cis));
  // test if condition C*P=B, and thus especially Cn*P=Bn is satisfied
  sys:= {op(zip(map([op(Cn*P)],expr),map([op(Bn)],expr),_equal))};
  solset:= linsolve(sys,cis);
  if solset=FAIL then 
    return(FALSE);
  elif has((Cn:=[op(solset,[i,2]) $ i=1..n]),cis) or has(Cn, FAIL) then 
    return(FALSE); 
    // rank<n
    // now for Y'=A*Y it holds: Z'=C*Z and Z=P*Y 
  elif o=Coeffs then 
    return([Cn,P]);
  else 
    return(TRUE);
  end_if;
  
end_proc:



