/*
   ode::distributePiecewise(pw, indeps, csts)

   PARAMETER: 

     pw     -- any set (function has no effect unless pw is piecewise) 
     indeps -- DOM_SET, consisting of independent variables of the equation
     csts   -- DOM_SET, consisting of integration constants


   DETAILS: 

     If the solution of an ODE is a 'piecewise' of sets, we want to turn it
     into a set of 'piecewise' defined functions - the set may be 'piecewise',
     but with the conditions not depending on the independent variable. 


  EXAMPLES: 

    (1) > ode::distributePiecewise(piecewise([ln(- (4*exp(5*C3 + 5*x) + 
                                              1)/(exp(5*C3 + 5*x) - 1) - 1) = 5*C3 + 5*x + 
                                              ln(4 - (4*exp(5*C3 + 5*x) + 1)/(exp(5*C3 + 5*x) - 1)), 
                                              {-(4*exp(ln(-(5*exp(5*C3 + 5*x))/(exp(5*C3 + 5*x) - 1)) - 
                                               ln(-5/(exp(5*C3 + 5*x) - 1))) + 1)/(exp(ln(-(5*exp(5*C3 + 
                                               5*x))/(exp(5*C3 + 5*x) - 1)) - ln(-5/(exp(5*C3 + 5*x) - 1))) - 
                                               1)}], 
                                              [ln(- (4*exp(5*C3 + 5*x) + 1)/(exp(5*C3 + 5*x) - 1) - 1) <> 
                                               5*C3 + 5*x + ln(4 - (4*exp(5*C3 + 5*x) + 1)/(exp(5*C3 + 5*x) - 1)), 
                                               {}]), 
                                   {x}, {C3}, {}, {#inits = {y(x0) = 1}})

        
    (2) > ode::distributePiecewise(piecewise([a <> 0, {-(2*3^(1/2)*(-a*x)^(1/2))/3, (2*3^(1/2)*(-a*x)^(1/2))/3}], 
                                             [a = 0 and x <> 0, {0, C54^2/x}], [C54 = 0 and a = 0 and x = 0, C_], 
                                             [C54 <> 0 and a = 0 and x = 0, {0}]), 
                                   {x}, {C54}, {}, {#inits = {y(0) = 0}})

    (3) In case no 'piecewise' is given as 1st argument, the argument remains 
        unchanged: 

        > ode::distributePiecewise({0, 1/((2*cos(x))/5 + (4*sin(x))/5 + C58*exp(2*x))^(1/2), 
                                     -1/((2*cos(x))/5 + (4*sin(x))/5 + C58*exp(2*x))^(1/2)}, 
                                   {x}, {C58}, {}, {#inits = {y(0) = 1}})
    
*/

ode::distributePiecewise:= proc(pw,indeps:DOM_SET,csts:DOM_SET,solveOptions:DOM_SET,odeOptions:DOM_SET)
  local result,F,branches,j,i,S:DOM_SET,distributeBranch:DOM_PROC;
begin  
  csts:= csts minus Type::ConstantIdents;
 
  // Local procedure. 
  // return the set of solutions of the branch 'br'
  distributeBranch:= proc(br)
    local cond,dep,indep,dummy,res;
  begin
    cond:= op(br,1);
    case type(cond)
      of "_and" do
        [dep,indep,dummy]:= split(cond,has,S);
        if indep <> TRUE then
          return(piecewise([indep,distributeBranch([dep,op(br,2)])],
                           [not indep,{}])
           )
        end_if;
        // fall through
      otherwise
        // default: if 'cond' does not depend on 'z', we want a 'piecewise'
        // of solutions; if it does depend on 'z', we want one solution that is
        // a 'piecewise'
        if not has(cond, S) then
          return(piecewise(stdlib::branch(br), [not cond, {}] ))
        else
          F:= fp::unapply
          (piecewise::disregardPoints(piecewise([cond, `#x`])),
           `#x`);
          if F = undefined then
            res:= undefined
          else
            res:= ode::mapsol(op(br,2),F,solveOptions,odeOptions)
          end_if;
          if res = undefined then
            return({})
          end_if;
          return(misc::maprec(res, {DOM_SET} = (x -> x minus {undefined})));
        end_if;
    end_case;
  end_proc;

  S:= indeps union csts;
  if type(pw) <> piecewise then
    result:= pw
  else
    branches:= [op(pw)];
    if ((j:= contains(map(branches,op,1),Otherwise))) > 0 then
      // replace the condition 'Otherwise' by the negation of the 
      // disjunction of the remaining other conditions 
      branches[j][1]:= not _or(branches[i][1] $i=1..j-1,
                               branches[i][1] $i=j+1..nops(branches))
    end_if;
    result:= _union(op(map(branches,distributeBranch)))
  end_if;

  result:= misc::maprec(result, {piecewise} =
               proc(pw)
                 local vars;
               begin
                 vars:= indets(piecewise::expressions(pw)); 
                 // we can ignore conditions on integration constants that
                 // do not actually appear in the solution
                 piecewise::assumeIdentsTRUE(pw, csts minus vars)
               end_proc
               );

  // collect piecewises with one branch and equal sets in their only branch
  if type(result) = DOM_SET then
    result:= [op(result)];
    for i from 1 to nops(result) do
      for j from i+1 to nops(result) do
        if type(result[i]) = piecewise and
           type(result[j]) = piecewise and
           nops(result[i]) = 1 and 
           nops(result[j]) = 1 and
           piecewise::expression(result[i],1) = piecewise::expression(result[j],1) then
          // combine the conditions
          result[i]:= piecewise::mapConditions(result[i],_or,piecewise::condition(result[j],1));
          // and throw away the second copy
          result[j]:= FAIL;
        end_if;
      end_for;
    end_for;
    return({op(select(result,_unequal,FAIL))});
  else // result is not a finite set
    return(result);
  end_if;
  
end_proc:

