/*
  ode::removeConstants

   PARAMETER:            'S' -- set of solutions produced by ODE 
                         'y' -- the dependent variable 
                         'x' -- the independent variable 
                     'inits' -- set of initial conditions (may be {})
              'solveOptions' -- the usual options for 'solve'
                'odeOptions' -- the usual options for ODE  

   DETAILS: 

    Procedure 'ode::removeConstants' serves to delete constant elements
    contained in sets of solutions, which are in fact no solutions of the 
    ODE under consideration. 

    The function 'ode::autonomousEq' may introduce constants, which are no 
    solution at all. This is not a bug in 'ode::autonomousEq', but rather a 
    consequence of the algorithmic way of proceeding, which seems to be 
    standard in the literature. So we do not need to change 
    'ode::autonomousEq', but we have to check the results of this function 
    if the ODE under consideration does not admit constant solutions. 
*/

ode::removeConstants:= proc(S,y,x,inits,solveOptions,odeOptions)
  local removeConstantsRecursively;
begin
  // Local method to move recursively through the things returned by the 
  // solver. This method can return 'null()' (e.g. for 'S = C_'). We use 
  // the outer procedure just to make sure that 'FAIL' can be returned in 
  // such cases. 
  removeConstantsRecursively:= proc(S,y,x,inits,solveOptions,odeOptions)
                               begin 
                                 case type(S)
                                   of "_union" do
                                     S:= map([op(S)], removeConstantsRecursively, 
                                             y,x,inits,solveOptions,odeOptions);
                                     return(_union(op(S)));
                                   of DOM_SET do
                                     return(map(S, s -> removeConstantsRecursively(s,y,x,inits,
                                                                                   solveOptions,
                                                                                   odeOptions)));
                                   of piecewise do 
                                     return(piecewise::extmap(S,removeConstantsRecursively,
                                                              y,x,inits,solveOptions,
                                                              odeOptions));
                                   of "solve" do
                                     if has(S,x) or has(S,y) then 
                                       return(S);                             
                                     else
                                       return(null());
                                     end_if;
                                   of solvelib::BasicSet do 
                                     return(null());
                                 end_case;
                                 if not has(S,x) then 
                                   return(null());
                                 else
                                   return(S);
                                 end_if;
                               end_proc:    
  S:= removeConstantsRecursively(S,y,x,inits,solveOptions,odeOptions);                 
  if S = null() then 
    return(FAIL);
  else 
    return(S);
  end_if;     

end_proc:

