/*
   ode::selectSolutions(S,cond,y,z,solveOptions,odeOptions)

   PARAMETERS: 

         S - set consisting of functions z -> f(z) or FAIL
      cond - condition depending on y(z) 
         y - dependent variable
         z - independent variable
    
      solveOptions, odeOptions - additional options as usual 

   RETURN VALUE: 

     Returns the set of all functions f(z) such that cond | y(z) = f(z) is TRUE
     or FAIL. 
*/

ode::selectSolutions:= proc(S: Type::Union(Type::Set, DOM_FAIL), cond, y, z, solveOptions, odeOptions)
  local solutions, res,
        check: DOM_PROC;
begin
  if S=FAIL then
    return(FAIL);
  end_if;
  if not has(cond, y(z)) then
    return(piecewise([cond,S],[not cond,{}]));
  end_if;

  // ------------------------- local procedure ---------------------------
  check:= proc(sol: Type::Set, eq, x)
    local res,tmp;
  begin
    if type(eq) = "_equal" then
      eq:= op(eq,2)-op(eq,1);
    end_if;
    case type(sol)
      of DOM_SET do
        // apply only weak zero-test due to efficiency
        res:= select(sol, s -> if traperror((tmp:= eq | x=s))=0 and iszero(tmp) then 
                                 TRUE; 
                               else 
                                 FALSE;
                               end_if);
        return(res);                       
      of "_union" do
      of "_minus" do  
        return(map(sol,check,eq,x));
    end_case;
    return(solvelib::solve_intersect(hold(solve)(eq, x),sol));
  end_proc;
  // ---------------------------------------------------------------------

  case type(S)
  of DOM_SET do
    // check each element s of S by evaluating the condition cond at y(z) = s
    return(_union(op(
           map(S,
               proc(s)
                 local c;
               begin
                 if traperror((c:= evalAt(cond,y(z)=s)))<> 0 then 
                   return({}); 
                 else   
                   return(piecewise([c,{s}],[not c,{}]));
                 end_if;  
               end_proc
               )
           )
          )
       )
  of "_union" do 
    // we can apply selectSolutions to each operand independently   
    res:= map({op(S)}, ode::selectSolutions, cond, y, z, solveOptions, odeOptions);
    res:= select(res,_not@has,FAIL);
    if res = {} then 
      return(FAIL);
    else
      return(_union(op(res)));
    end_if;
  end_case;
  // default: understand cond as an algebraic or differential equation in
  // y(z), solve it and intersect with S
  if not has(ode::odeIndets(cond, {y}, solveOptions, odeOptions), 
             {hold(D), hold(diff)}) then
    // does not contain derivatives of y(z) 
    // call algebraic solver 
    solutions:= solve(cond, y(z), op(solveOptions));          
  elif  type(cond) = "_equal" then
    // as a condition, 'cond' cannot be an expression only, but 
    // must be an equation or inequality or whatever; here 
    // we treat the case of an equation (inequalities, whatever, .., cannot 
    // be treated, so we return FAIL below in these cases)
    solutions:= ode::solve_eq(op(cond, 1) - op(cond, 2), y, z, {}, 
                              solveOptions, odeOptions);
  else
    // give up: no equation, something complicated
    return(FAIL);
  end_if;
  if type(S) = "solve" then
    // check whether the solutions obtained by solving the condition for 'y(z)' 
    // are elements of the set 'S' of solutions of the underlying differential
    // equation; i.e. 'S' is of the form 'solve(F(x) = 0, x)', now check the 
    // elements 's' of 'solutions' whether they satisfy 'F(s) = 0'. 
    return(check(solutions,op(S,1),op(S,2)));
  end_if;
  // return the intersection of S and solutions; since S containes solutions
  // of the underlying differential equation, the intersection of S and 
  // solutions is always a solution of this ODE consisting of those functions,
  // which satisfy cond. 
  return(solvelib::solve_intersect(S, solutions));

end_proc:

