/* 

   DESCRIPTION: 

     This file contains functions for computing the Wronskian matrix and 
     determinant of given functions or of a given linear differential equation.
     In particular, this functions can be used to create linear homogeneous
     differential equations.


   FUNCTIONS: 

     - used Parameter:
        ++ Ly, y(x)   :DOM_EXPR
        ++ v_i        :DOM_EXPR
        ++ x,y        :DOM_IDENT
        ++ R          :DOM_DOMAIN
        ++ Ly    = ordinary linear differential equation 
        ++ y(x)  = the operator (function) of Ly  
        ++ y     = name of the operator of Ly
        ++ x     = independant variable (of the operator y)
        ++ n     = order of Ly
        ++ v_i   = function in x 
        ++ R     = differential ring
    
     - ode::wronskianMatrix([v_i], x, < R >)
        ++ returns the Wronskian matrix of the functions v_i  w.r.t. x.
        ++ When the option R is given, then the specified differential ring 
        ++ will be chosen for reprensenting the matrix entries, otherwise 
        ++ Dom::ExpressionField(id, iszero@normal) will be used instead. 

     - ode::wronskian([v_i], x, < R >)
       ode::wronskian(Ly, y(x))
        ++ returns the Wronskian (determinant) of the functions v_i  w.r.t. x 
        ++ or the Wronskian of the equation Ly.

     - ode::wronskianOf(Ly,y,x,n)
        ++ returns the wronskian of Ly, which is exp(-int(a_(n-1)/a_n, x)) if
        ++ a_n and a_(n-1) are the coefficients of diff(y(x),x$n) and
        ++ diff(y(x),x$(n-1)).
        ++ NOTE: no check of arguments!                                       


   EXAMPLES:

     >> ode::wronskian([sqrt(x),1/sqrt(x),y(x)],x);      
     
                        diff(y(x), x, x)   y(x)   diff(y(x), x)
                      - ---------------- + ---- - -------------
                               x              3         2
                                           4 x         x
     
     >> ode::wronskianMatrix([sqrt(x),1/sqrt(x),y(x)],x);                    
     
                     +-                                      -+
                     |     1/2       1                        |
                     |    x   ,     ----,         y(x)        |
                     |               1/2                      |
                     |              x                         |
                     |                                        |
                     |     1          1                       |
                     |   ------,  - ------,   diff(y(x), x)   |
                     |      1/2        3/2                    |
                     |   2 x        2 x                       |
                     |                                        |
                     |      1        3                        |
                     |  - ------,  ------,  diff(y(x), x, x)  |
                     |       3/2      5/2                     |
                     |    4 x      4 x                        |
                     +-                                      -+
*/

ode::wronskianMatrix:= proc(vis,x,R=Dom::ExpressionField(id,iszero@normal), 
                            solveOptions={},odeOptions={})
  local i, mat;
begin
  if R::hasProp(Dom::BaseDomain) = FAIL 
     or not(R::hasProp(Cat::CommutativeRing) and
            R::hasProp(Cat::DifferentialRing) and
            iszero( R::characteristic() )) then 
    error("illegal coefficient ring") 
  end_if;
  mat:= [map(vis, R::new)];
  for i from 1 to nops(vis)-1 do
    mat:= append(mat, map(mat[i], diff, x)) 
  end_for;
  
  //return((Dom::Matrix(R))::create(mat));
  return((Dom::Matrix(R))(mat));
end_proc:


ode::wronskian:= proc()
  local eq,solveOptions,odeOptions;
begin
  if domtype(args(1))=DOM_LIST then
    if args(0) <= 3 then 
      expr(linalg::det(ode::wronskianMatrix( args() )));
    else 
      if args(0) = 4 then 
        solveOptions:= args(args(0)-1); 
        odeOptions:= args(args(0));
        expr(linalg::det(ode::wronskianMatrix(args(1..2), 
                                              Dom::ExpressionField(id, iszero@normal),
                                              solveOptions,odeOptions)));
      elif args(0)= 5 then 
        expr(linalg::det(ode::wronskianMatrix( args() )));                                           
      end_if;
    end_if;
  elif args(0)=2 or args(0)=3 then
    eq:=ode::isLODE(args(1..2),Lode);
    if eq<>FALSE then
      ode::wronskianOf(eq);
    else
      error("Not a linear ordinary differential equation");
    end_if;
  elif args(0) > 3 then 
    solveOptions:= args(args(0)-1); 
    odeOptions:= args(args(0));
    eq:= ode::isLODE(args(1..2),Lode,solveOptions,odeOptions);
    if eq<>FALSE then
      ode::wronskianOf(eq,solveOptions,odeOptions);
    else
      error("Not a linear ordinary differential equation");
    end_if;
  else
    error("wrong no of args");
  end_if;
  
end_proc:


ode::wronskianOf:= proc(eq,y,x,n,solveOptions={},odeOptions={})
  local intOptions, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
              IgnoreAnalyticConstraints;
            else
              null();
            end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
//  return(normal(expand(
//           exp(-int(coeff(eq,[diff(y(x),x$(n-1))],1)/coeff(eq,[diff(y(x),x$n)],1),x,intOptions)), 
//           optIgnoreAnalyticConstraints)));
  return(ode::normal(exp(-int(coeff(eq,[diff(y(x),x$(n-1))],1)/coeff(eq,[diff(y(x),x$n)],1),x,intOptions)),
                Expand=FALSE));
           
end_proc:

