
/* 
   =============================
   METHODS FOR SOLVING ABEL ODES
   =============================
 
   REFERENCES: [1] E. Kamke: "Differentialgleichungen", pp. 24-28 
               [2] G.M. Murphy: "Ordinary Differential Equations and Their Solution",
                   pp. 23-26
               [3] E.S. Cheb-Terrab & A.D. Roche: "Abel ODEs: Equivalence and Integrable
                   Classes", Comp. Phys. Comm., 130 (2000) 197, p. 204-241.

   CALL: 'ode::abel(eq,y,x)' tries to solve the first-order ODE 'eq = 0' wrt 'y(x)'
         when 'eq = 0' is of Abel type. Reference: Murphy pp. 23-26 and Kamke pp. 24-28.

     Applicable to equations of the form :
       - y' = f3(x) y^3 + f2(x) y^2 + f1(x) y + f0(x) 
         (Abel equations of the first kind)
 
       - (g0(x) + g1(x) y) y' = f2(x) y^2 + f1(x) y + f0(x) 
         (Abel equations of the second kind)
         It can be always be transformed to an Abel ODE of first kind.

   Algorithm used is not complete for solving Abel ODEs of first kind (for a complete 
   one, one may have a look at [3]. The only "easy" cases are implemented ...

   Consider:
      s3:= f0f3^2+1/3(2f2^3/9-f1f2f3+f3f2'-f2f3') and
      s5:= f3s3'-3s3(f3'+f1f3-f2^2/3)
   
   What is implemented for the moment are the cases s3=0 and s5^3/s3^5 is constant. They 
   both reduce to separable ODEs.
   
   Examples:

   1) Equation of second kind :

     >> ode::solve(x*(y(x)+4)*diff(y(x),x)-y(x)^2-2*y(x)-2*x, y(x));

     >> ode::solve(x*(2*y(x)+x-1)*diff(y(x),x)-y(x)*(y(x)+2*x+1), y(x));


   2) Equation of first kind :

     >> ode::solve(diff(y(x),x)-x^a*y(x)^3+3*y(x)^2-x^(-a)*y(x)-x^(-2*a)+a*x^(-a-1),y(x));

*/

ode::abel :=
proc(eq, y, x, solveOptions, odeOptions)
  local yp,P,g,d,g0,g1,f0,f1,f2,f3,F1,F2,F3,handleRes: DOM_PROC, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;

  // ----------------------------------------------------------------
  // local method handleRes(S)
  // computes {1/u-g0; u in S, u <>0}
  // ----------------------------------------------------------------
  handleRes:= proc(S)
    local l: DOM_LIST, pw;
    save MAXEFFORT;
  begin
    case type(S)
      of DOM_FAIL do
        return(FAIL)
      of piecewise do
        pw:= piecewise::extmap(S, handleRes);
        if has(pw, FAIL) then
          return(FAIL)
        else
          return(pw)
        end_if
      of "_union" do
        l:= map([op(S)], handleRes);
        if contains(l, FAIL) > 0 then
          return(FAIL)
        else
          return(_union(op(l)))
        end_if
      of "solve" do
      of RootOf do
        return(solve(subs(op(S, 1), op(S,2)=1/(op(S,2)+g0)), op(S, 2), 
                     op(solveOptions)))
    end_case;
    // default (applies, e.g., to DOM_SET) 
    ode::mapsol(S, u -> if u<>0 then 
                          1/u-g0 
                        else 
                          undefined 
                        end_if, 
                solveOptions, odeOptions)
  end_proc;
  // ----------------------------------------------------------------
  
  yp:= genident();
  eq:= subs(eq,diff(y(x),x)=yp,y(x)=y,EvalChanges);
  userinfo(2,"trying to recognize an Abel equation");
  if testtype(eq,Type::PolyExpr(yp)) then
    if degree((eq:=poly(eq,[yp])))=1 then
      if testtype((P:=coeff(eq,0)),Type::PolyExpr(y)) then
        P:= poly(-P,[y]);
        if testtype((g:=coeff(eq,1)),Type::PolyExpr(y)) then
          d:= degree((g:=poly(g,[y])));
          if d=1 and degree(P)<=2 then
            userinfo(1,"Abel equation of second kind");
            g1:= coeff(g,1);
            assert(not iszero(g1));
            g0:= coeff(g,0)/g1;
            f2:= coeff(P,2)/g1;
            f1:= coeff(P,1)/g1;
            f0:= coeff(P,0)/g1;
            F3:= -combine(ode::normal(f2*g0^2-f1*g0+f0,Expand=FALSE),optIgnoreAnalyticConstraints);
            F2:= -combine(ode::normal(f1-2*f2*g0+diff(g0,x),Expand=FALSE),optIgnoreAnalyticConstraints);
            F1:= -f2;
            userinfo(2,"transformed Abel equation of first kind ".
                     "is:", diff(y(x),x) = F3*y(x)^3+F2*y(x)^2+F1*y(x));
            // The solution of the initial ODE is 1/u-g0 where u is a solution
            // of the transformed equation
            return(handleRes(ode::Abel1st(F3,F2,F1,0,x,solveOptions,odeOptions)))
          elif d=0 and degree(P)<=3 then
            userinfo(1,"Abel equation of first kind");
            g0:=coeff(g,0);
            assert(not iszero(g0));
            f3:=combine(ode::normal(coeff(P,3)/g0),optIgnoreAnalyticConstraints);
            f2:=combine(ode::normal(coeff(P,2)/g0),optIgnoreAnalyticConstraints);
            f1:=combine(ode::normal(coeff(P,1)/g0),optIgnoreAnalyticConstraints);
            f0:=combine(ode::normal(coeff(P,0)/g0),optIgnoreAnalyticConstraints);
            return(ode::Abel1st(f3,f2,f1,f0,x,solveOptions,odeOptions))
          end_if
        end_if
      end_if
    end_if
  end_if;
  if contains(odeOptions, Type = Abel) then 
    ode::odeWarning("cannot detect Abel equation");
  end_if;  
  FAIL
end_proc:



//  Main procedure for solving Abel equations of first kind.
//  Only the simple cases are implemented, i.e. when s3=0 and s5^3/s3^5=Ct.

ode::Abel1st:= proc(f3,f2,f1,f0,x,solveOptions,odeOptions)
  local y,z,C,s3,F,v,s5,inv,X,crs3,i,c0,c1,cubic,zero,integral,u,aux,sol,tmp,
        intOptions,optIgnoreAnalyticConstraints;
  save MAXEFFORT;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y:= genident();
  C:= genident("C");
  u:= genident();
  if not iszero(f3) then
    s3:= combine(ode::normal(f0*f3^2+1/3*(2*f2^3/9-f1*f2*f3+f3*diff(f2,x)-f2*diff(f3,x)),
                        Expand=FALSE),optIgnoreAnalyticConstraints);
    if iszero(s3) then
      assert(not iszero(f3));
      F:= ode::normal(-f2/3/f3);
      // Note that the following use of 'expand' should not be removed. We get 
      // better and simpler results if we do this 'expand' even if this does 
      // not seem obvious at first glance. Do not remove this without running 
      // performance comparison tests!
      v:= expand(exp(int(f1+f2*F,x,intOptions)),optIgnoreAnalyticConstraints);
      z:= int(f3*v^2,x,intOptions);
      // The solution is y(x)=u(z)*v(x)+F(x) where u'(z)=u(z)^3 ,
      //      i.e. u(z)=0 or (+/-)1/sqrt(C-2*z)
      // v and z may be piecewise
      return(piecewise::zip(v, z,
                            proc(V, Z)
                              begin
                                {F, v/sqrt(-2*z+C)+F, -v/sqrt(-2*z+C)+F}
                            end_proc
                            )
             )
     else 
      // The case s3 <> 0. 
      s5:= ode::normal(combine(f3*diff(s3,x)-3*s3*(diff(f3,x)+f1*f3-f2^2/3), 
                          optIgnoreAnalyticConstraints),
                  Expand=FALSE);
      if not has((inv:=ode::normal(s5^3/s3^5,Expand=FALSE)),x) then
        /* 
           Here are some explanations of the following....
           The theory says (see Murphy p.24) that in this case the general solution
           is y(x)=u(x)v(x)+F(x), where F=-f2/3/f3, v=1/f3*s3^(1/3) and u is solution
           of the ODE :
                  u'(x)=X(x)*(1-A*u(x)+u(x)^3)
           with (3A)^3=inv and X=1/f3*s3^(2/3).
           Therefore, u is solution of the following algebraic equation :
                  solve(-int(X,x)+int(1/(1-A*u+u^3), u)+C=0, u)
               or u=RootOf(_Z^3-A*_Z+1, _Z).
           And it's easy to see that u(x)*v(x) is solution of :
                  solve(-int(X,x)+v^2*int(1/(c0-c1*y+y^3), y)+C=0, y)
               or a solution of _Z^3-c1*_Z+c0=0
           with c0=s3/f3^3 and c1=1/3/f3^2/s3*s5.
           Here I computed directly the integral :
                  integral:=int(1/(c0-c1*y+y^3), y)
                           =sum(ln(u-R)/(3*R^2-c1)) for R in solve(c0-c1*y+y^3, y)
           and therefore, y=uv+F is solution of
                  solve(-int(X,x)+v^2*subs(integral, u=u-F)+C=0, u)
               or a solution of (_Z-F)^3-c1*(_Z-F)+c0=0.
           Since X=v^2/f3, we can ignore the content in the factorization of s3
           and simplify the cubic root of s3 (crs3 in the next) : only divide
           the exponents by 3.
        */
        s3:= factor(s3);
        tmp:= coerce(s3, DOM_LIST);
        crs3:= _mult(tmp[2*i]^(tmp[2*i+1]/3) $ i=1..nops(tmp) div 2);
        assert(not iszero(f3));
        v:= ode::normal(1/f3*crs3,Expand=FALSE);
        F:= ode::normal(-f2/3/f3,Expand=FALSE);
        X:= ode::normal(1/f3*crs3^2,Expand=FALSE);
        // The solution is y(x)=u(x)*v(x)+F(x) where u'(x)=X(x)(1-A*u(x)+u(x)^3)
        c1:= ode::normal(1/3*s5/f3^2/s3,Expand=FALSE);
        c0:= ode::normal(s3/f3^3,Expand=FALSE);
        cubic:= c0-c1*y+y^3;
        if type((zero:= solve(cubic,y,IgnoreSpecialCases,op(solveOptions)))) = DOM_SET then
          if nops(zero) = 3 then
            integral:= _plus(ln(u-F-i)/(3*i^2-c1) $ i in zero)
          else
            integral:= int(1/subs(cubic,y=u),u,intOptions)
          end_if:
        else //zero is {RootOf(cubic,y)}
          integral:= _plus(ln(u-F-i)/(3*i^2-c1) $ i in op(zero))
        end_if;
        aux:= v^2*integral-int(X,x,intOptions);
        aux:= piecewise::extmap(aux,
                                proc(g)
                                  local result;
                                begin
                                  result:= factor(g);
                                  if not has(op(result, 1),{u,x}) then
                                    expr(result)/op(result,1)
                                  else
                                    expr(result)
                                  end_if
                                end_proc
                                );
        sol:= ode::mapsol(solve(aux-ln(C),u,op(solveOptions)),expand,solveOptions,odeOptions);
        return(solve((y-F)^3-c1*(y-F)+c0,y,op(solveOptions)) union sol)
      else
        userinfo(1,"Abel equation failed");
      end_if
    end_if
  end_if;
  return(FAIL);
end_proc:
  
