/*  
   
   ===========================
   METHODS FOR AUTONOMOUS ODES
   ===========================
 
   REFERENCES: [1] D. Zwillinger: "Handbook of Differential Equations", 
                   Section 48, pp. 214

   CALL: 'ode::abel(eq,y,x)' tries to solve the first-order ODE 
         'eq = 0' wrt 'y(x)'

         ode::autonomousEq(eq,y,x,n) solves the autonomous equation 
         eq=0 of order n in y(x), using the algorithm described in [1]. 

   RETURN VALUE: FAIL or a list of solutions for y(x).

   EXAMPLE:  

     >> ode::autonomousEq(diff(y(x),x,x)-diff(y(x),x)-2*y(x)*diff(y(x),x),y,x,2);
 
          {        /           1/2          \                     }
          {        | (4 C3 - 1)    (C5 + x) |           1/2       }
          {     tan| ---------------------- | (4 C3 - 1)          }
          {        \            2           /                     }
          { C6, ------------------------------------------- - 1/2 }
          {                          2                            }
 
*/

/* Currently not used. 
ode::autonomous:=
proc()
  local components,solveOptions,odeOptions;
begin
  solveOptions:= args(args(0)-1);
  odeOptions:= args(args(0));
  components:=ode::getComponents(args());
  if nops(components[2])>1 then
    error("autonomous method works only for scalar equations.")
  end_if;
  components[4]:=ode::order(components[1],{components[2]},solveOptions,odeOptions);
  ode::autonomousEq(components,solveOptions,odeOptions);
end_proc:
*/
    
ode::autonomousEq:= proc(eq,y,x,n,solveOptions,odeOptions)
  local eq0,i,u,solu, optIgnoreAnalyticConstraints, 
        autoTransform: DOM_PROC,
        createSols: DOM_PROC;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                                   IgnoreAnalyticConstraints;
                                 else
                                   null();
                                 end_if;  
              
  // -------------------------------------------------------------------
  // local methods
  // -------------------------------------------------------------------
  // gives the value of diff(y(x),x$n) in terms of u(y)=diff(y(x),x)
  // uses y and u from outer proc
  autoTransform:= proc(n)
    local e;
  begin
    e:=u(y);
    while n>1 do
      e:= expand(u(y)*diff(e,y),optIgnoreAnalyticConstraints); // see Note 3
                                                               // in [1] for 
                                                               // explanations
                                                               // on this transform     
                                                          
      n:= n-1;
    end_while;
    return(e);
  end_proc:
  // -------------------------------------------------------------------

  /* m a i n  */

  // Recognize autonomous ODE. 
  userinfo(2,"trying to recognize an autonomous equation");
  u:= genident("u");
  eq0:= eq;
  eq:= subs(eq,diff(y(x),x$i)=autoTransform(i) $ i=1..n,y(x)=y,EvalChanges);
  if has(eq,x) then
    return(FAIL)
  end_if;

  // Now we know that we have an autonomous ODE.    
  userinfo(1,"autonomous equation");
  sysassign(ode::depth,ode::depth+1);
  solu:= ode::solve_eq(eq,u,y,{},solveOptions,odeOptions);
  sysassign(ode::depth,ode::depth-1);
  if has(solu,hold(solve)) or has(solu,FAIL) then
    userinfo(1,"autonomous method failed");
    return(FAIL)
  end_if;
  
  userinfo(2,"solutions for ".expr2text(u)."(y) are",solu);
  
  // Now each solution is of the form u = u(y(x)) and we have to solve 
  // the auxiliary ODE y'(x) = u(y(x)) for each of these solutions. 
  // We cannot use any automatic method for mapping, hence, we have to 
  // do it by hand.   
  createSols:= proc(S,solveOptions,odeOptions)
    local res, sols, intvars, tmp;
  begin
    case type(S)
      of "_union" do
        S:= select(map([op(S)],createSols,solveOptions,odeOptions),_not@has,FAIL); 
        if S = [] then
          return(FAIL)
        else
          return(_union(op(S)))
        end_if
      /* This branch is not helpful.  
        of piecewise do
        // the conditions of S depend on y; for each branch [cond(y), S],
        // only those y that both solve y'(x) = u(y(x)) for some u in S and
        // satisfy cond(y) are solutions of the original eq.
        branches:= [op(S)];
        res:= {};
        for branch in branches do
          sols:= createSols(branch[2],solveOptions,odeOptions);
          if sols = FAIL then
            return(FAIL)
          end_if;
          // This strategy is questionable. There does not seem to be much
          // hope at all in the case of 'piecewise'-objects here. 
          res:= res union (solve(branch[1],y,op(solveOptions)) intersect sols);
        end_for;
        return(res);
      */
      of DOM_SET do
        intvars:= {};
        // We need to find out if there are unresolved integrals where the 
        // integration variable is y. If this is the case, we have to give 
        // up, since the right-hand-side of the auxiliary ODE y'(x) = u(y(x)), 
        // which we need to solve, would contain these integrals (meaning 
        // that we would in fact not have an ODE). 
        misc::maprec(S,
                     {"int"} = proc(elem)
                                begin
                                  if op(elem,2) = y then 
                                    intvars:= intvars union {y};
                                  end_if;
                                  return(elem);
                                end_proc):
        if contains(intvars,y) then 
          // give up
          return(FAIL);
        end_if;
        // Solve the auxiliary ODE y'(x) = u(y(x)) for each element in S. 
        res:= map(S, s -> ode::solve_eq(diff(y(x),x)-subs(s,y=y(x)),
                                        y,x,{},solveOptions,odeOptions));                                        
        if not has(odeOptions,#inits) then                                 
          res:= select(res,_not@has,FAIL);
          res:= select(res,_not@has,hold(solve));
          if res = {} or not has(res,x) then  
            return(FAIL);
          end_if;
        elif has(res, {hold(solve), FAIL}) then  
          return(FAIL);  
        end_if;
        /*
          It is a well-known problem that the method for autonomous ODEs 
          can introduce constant solutions which are in fact no solutions to 
          the input ODE. The following 'mapping procedure' serves to filter 
          out these incorrect constant solution. 
        */
        res:= split(res,testtype,DOM_SET);
        res[1]:= map(res[1], 
                     s -> map(s, elem -> if has(elem,x) then 
                                           elem 
                                         elif traperror((tmp:= expand(eq0 | y(x)=elem,
                                                                      optIgnoreAnalyticConstraints))) = 0 
                                              and iszero(tmp) then
                                           elem
                                         end_if));                                        
        res:= _union(op(res));                                   
        if res <> {} then
          return(_union(op(res)));
        else 
          return(FAIL);
        end_if;
    end_case;
    return(FAIL);
  end_proc;

  return(createSols(solu, solveOptions, odeOptions));
end_proc:


