/*  
   ==========================
   METHODS FOR BERNOULLI ODES
   ==========================
 
   REFERENCE: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 49, pp. 214

   eq is a first-order ODE in y(x). The method tries to recognize and solve a 
   Bernoulli equation, i.e. and equation of the form
 
	   y' + P(x)*y = Q(x)*y^n

   and returns FAIL otherwise.

   EXAMPLES:

    >> ode::solve(diff(y(x),x)+y(x)-y(x)^3*sin(x),y(x)); 

    >> ode::bernoulli(diff(x(y),y)-y^2*x(y)-y^5/x(y),x,y);

*/

ode::bernoulli:= proc(eq,y,x,solveOptions={},odeOptions={})
  local  s, yp, P, n, Q, optIgnoreAnalyticConstraints, ypow, z1, z2, tmp;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  // delete yp, P, n;
  yp:= genident();
  z1:= genident();
  z2:= genident();
  eq:= ode::normal(subs(eq,diff(y(x),x)=yp),Expand=FALSE);
  if testtype(eq,Type::PolyExpr(yp)) and degree(eq,[yp])=1 then // linear in y'(x) 
    eq:= coeff(eq,[yp],0)/coeff(eq,[yp],1); // should match P(x)*y-Q(x)*y^n
    eq:= combine(expand(subs(eq,y(x)=y),optIgnoreAnalyticConstraints),
                 optIgnoreAnalyticConstraints); // combine added
    ypow:= {}; 
    misc::maprec(eq,
                 {"_power"} = proc(elem)
                                begin
                                  if op(elem,1) = y then 
                                    ypow:= ypow union {op(elem,2)}
                                  end_if;
                                  elem;
                                end_proc);  
    if nops(ypow) = 1 then 
      n:= ypow[1];
      tmp:= subs(eq,[y^n=z1,y=z2]);
      Q:= -coeff(tmp,[z1],1);
      P:= coeff(tmp,[z2],1);
      if ode::odeIszero(eq-P*y+Q*y^n) and iszero(expand(eq-P*y+Q*y^n)) then 
        userinfo(1,"Bernoulli equation");
        userinfo(2,"P=",P,"Q=",Q,"n=",n);
        s:= ode::bernoulliEq(P,Q,n,x,solveOptions,odeOptions);
        // for positive n, zero is another (singular) solution
        return(
           piecewise([n > 0, {0} union s],
                     [not n > 0, s])
               );
      end_if;  
    end_if;  
  end_if;
  if contains(odeOptions, Type = Bernoulli) then 
    ode::odeWarning("cannot detect Bernoulli equation");
  end_if;  
  FAIL
end_proc:


/* 
   STRATEGY:  

      We solve y'(x) + P(x) y(x) = Q(x) y(x)^n by changing to u(x) = y(x)^(1-n), 
      we obtain the equation 1/(1-n)*u'(x) + P(x) u(x) = Q(x), which can be solved 
      exactly using integrating factors, and then y(x)=u(x)^(1/(1-n)) 
*/

ode::bernoulliEq :=
proc(P,Q,n,x,solveOptions,odeOptions)
  local intP,y, res, intOptions, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions,IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions,IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;
  intP:= int(P,x,intOptions);
  y:= genident();
  res:= combine(exp((n-1)*intP)*(genident("C")+(1-n)*int(exp((1-n)*intP)*Q,x,intOptions)),
                exp,optIgnoreAnalyticConstraints);
  if not has(res, int) then
    res:= ode::normal(res,Expand=FALSE)^(1/(1-n));
  else
    res:= res^(1/(1-n));
  end_if;
  if not (type(n) in {DOM_INT, DOM_RAT}) then
    return({res});
  else
    return(map(solve(y^(1-n)=1,y,op(solveOptions)),_mult,res));
  end_if;
end_proc:


