/* 
   ===========================================================
   METHODS FOR SOLVING ODE SYSTEMS VIA INTEGRABLE CONBINATIONS
   ===========================================================
 
   REFERENCE: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 75, pp. 303

   DETAILS: 

     ode::combination(sys,unk,t) tries to solve the system of
     ODEs sys by the method of integrable combinations.
     The implementation below works only for first-order systems with
     polynomial coefficients.

     It returns either FAIL or a sequence of solutions.


   EXAMPLES:

     >> ode::solve({diff(x(t),t)=-3*y(t)*z(t),diff(y(t),t)=3*x(t)*z(t),
                    diff(z(t),t)=-x(t)*y(t)},{x(t),y(t),z(t)});

*/

ode::combination :=
proc(sys,unk,t,solveOptions, odeOptions)
  local l,i,n,v,eq,intOptions;
begin
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
   userinfo(1,"trying integrable combinations method");
   n:= nops(unk);
   l:= [diff(op(unk,i),t)=genident() $ i=1..n, 
        op(unk,i)=op(unk,[i,0]) $ i=1..n];
   sys:= subs(sys,l);
   sys:= map(sys,()->(if type(args(1))="_equal" then 
                        op(args(1),1)-op(args(1),2); 
                      else 
                        args(1); 
                      end_if));
   v:= map(l,op,2);
   for eq in sys do
     if not testtype(eq,Type::PolyExpr(v)) then
       return(FAIL)
     end_if
   end_for;
   userinfo(1,"trying to find integrable combinations");
   sys:= [op(map(sys,poly,v))];
   /* traperror is needed because gbasis can fail on non rational polynomials
       (Error: Illegal argument [icontent]) */
   if traperror((sys:= groebner::gbasis(sys))) <> 0 then 
     return(FAIL) 
   end_if;
   sys:= map(sys,expr);
   sys:= subs(sys,op(l,[2*n-i,2])=op(l,[2*n-i,1])$ i=0..2*n-1);
   sys:= map(sys,int,t,intOptions);
   if has(sys, hold(int)) then
     return(FAIL)
   end_if;
   sys:= {sys[i]+genident("C") $ i=1..nops(sys)};
   sys:= solve(sys,unk,op(solveOptions));
   userinfo(1,"integrable combinations method worked");
   return(sys);
end_proc:
