/* 
   ===============================
   DIFFERENTIATION METHOD FOR ODES
   ===============================
 
   REFERENCE: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 56 pp. 239

   DETAILS:

     Input: eq is an ODE in y(x).
     Output: a solution or FAIL.
  
     Description of algorithm:
     we are given an (implicit) differential equation

              eq:= F(y^{(n)}(x), ..., y(x), x)

     and want to solve eq = 0.

     We differentiate both sides with respect to x and obtain
     d/dx eq = 0.

     Solving this (factor by factor, if possible), we get the solutions of
     eq = C.

     We have to solve C = 0 w.r.t. the integration constants contained
     in the solutions; for simplicity, we solve this for one variable
     (to improve: how to select this variable?)
   

   EXAMPLES: 

     >> ode::solve(2*y(x)*diff(y(x),x,x)-diff(y(x),x)^2-1/3*(diff(y(x),x)
                   -x*diff(y(x),x,x))^2,y(x));

*/

ode::differentiation:= proc(eq,_0_y,_0_x,solveOptions,odeOptions)
  local neweq,e,sol,s,l,c,_f,_fc,i,INT;
begin 
  userinfo(2,"trying differentiation method");
  neweq:= factor(diff(eq,_0_x));
  if type(neweq) = Factored then 
    neweq:= Factored::convert_to(neweq,DOM_EXPR);
  else 
    return(FAIL);
  end_if;    
  if type(neweq)="_mult" then
    neweq:= select(neweq, has,{_0_x,_0_y});
  end_if;
  if type(neweq)="_mult" then
    neweq:= [op(neweq)];
  else
    neweq:= [neweq];
  end_if;
  if max(op(map(neweq,length))) >= length(eq) then // to avoid loops
    userinfo(10,"Differentiation method failed because ".
             "derivative is not simpler  than original equation");
    return(FAIL);
  end_if;
  sol:={};
  for e in neweq do
    sysassign(ode::depth,ode::depth+1);
    s:= ode::solve_eq(e,_0_y,_0_x,{},solveOptions,odeOptions);
    sysassign(ode::depth,ode::depth-1);
    if type(s) <> DOM_SET or s = {FAIL} then
      return(FAIL);
    end_if;
    if s = {} then
      next;
    end_if;
    s:= op(s, 1);
    if traperror((e:=expr(factor(evalAt(eq,_0_y(_0_x)=s))))) = 0 then
      if iszero(e) then
        sol:= sol union {s}
      else // check in the initial equation
        if type(e)="_mult" then
          e:= [op(e)];
        else
          e:= [e];
        end_if;
        for _f in e do
          l:= indets(_f) minus indets(eq) minus Type::ConstantIdents;
          // l contains the free integration constants
          if l={} then
            // no solution, go on to next factor of e
            next;
          end_if;
          // we prefer to solve a linear equation
          c:= FAIL;
          for i from 1 to nops(l) do
            if Type::Linear(_f, [op(l, i)]) <> FALSE then
              c:= op(l, i);
              break;
            end_if;
          end_for;
          if c = FAIL then
            c:= op(l,1);
          end_if;
          INT:= genident();
          /* 
             Note by Kai: We need to speed up the case where lots of 
             nested integrals are involved and 'solve' evaluates and 
             and expands them in its intermediate steps which is 
             completely useless in the process of finding solutions. 
          */
          _fc:= subs(solve(subs(_f, hold(int) = INT, Unsimplified),
                           c,op(solveOptions)),
                     INT = hold(int),Unsimplified);
          if _fc <> FAIL then
            _fc:= solvelib::substituteBySet(s, c, _fc);
          end_if;
          if _fc = FAIL then
            return(FAIL);
          end_if;
          sol:= sol union _fc;
        end_for
      end_if;
    end_if
  end_for; // for e in neweq
  
  return(sol);
end_proc:

