/*
  ==========================
  METHODS FOR FACTORING ODES
  ==========================

  REFERENCES: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 66 pp. 266, 
 
    ode::factor(eq,y,z,n) tries to solve the ODE eq in y(z)
    using the factoring method.

    NOTE: The implementation below works only for 2nd order linear 
          ODEs. 

    EXAMPLES: 
      >> ode::factor((x^2-x^3)*diff(u(x),x,x)+(2*x^2-4*x)*diff(u(x),x)+
                     (6-2*x)*u(x),u,x,2);

    Called with an element of domtype "ode", it returns a list of
    factors for the equation.
*/

ode::factor:= proc(eq, y, z, n, solveOptions={}, odeOptions={})
  local l, sol, mapSolve, handleSolution;
begin
  //-----------------------------------------------------------------
  // local methods
  //-----------------------------------------------------------------
  // handleSolution(solution)
  // solves the ode l[1] = solution
  handleSolution:=
  proc(solution)
  begin
    ode::solve_eq(l[1]=solution,y,z,{},solveOptions,odeOptions)
  end_proc;
  //-----------------------------------------------------------------
  // mapSolve
  // computes the union of all solutions of l[1] = f,
  // where f runs through all elements of S
  // currently handles only finite sets and piecewises
  //-----------------------------------------------------------------
  mapSolve:=
  proc(S)
  begin
    case type(S)
      of DOM_SET do
        _union(op(map(S,handleSolution)));
        break
      of piecewise do
        S:= piecewise::extmap(S,mapSolve);
        if has(S,FAIL) then
          FAIL
        else
          S
        end_if;
        break
      otherwise
        FAIL
    end_case
  end_proc;
  //-----------------------------------------------------------------

  // M A I N    of   ode::factor
  userinfo(1,"trying to factor differential operator");
  if type(eq)=ode then
    y:= op(eq,2); 
    z:= op(eq,3); 
    eq:= op(eq,1);
    n:= ode::order(eq,{y},solveOptions,odeOptions);
    l:= ode::factor_aux(eq,y,z,n,solveOptions,odeOptions);
    return(map(l,ode,y(z)));
  else
    l:= ode::factor_aux(eq,y,z,n,solveOptions,odeOptions);
    if nops(l)=1 then
      userinfo(1,"factoring method failed");
      return(FAIL)
    end_if;
    userinfo(1,"differential operator factors");
    sol:= {0};
    while l<>[] do
      userinfo(3,"solving",l[1]); 
      sol:= mapSolve(sol);
      delete l[1];
      // Note by Kai: In certain cases allow returning integrals. This 
      // branch is only taken in the very particular situation that 
      // 'ode::factor' is called from the heuristics collected within
      // 'ode::lookUp2ndOrderLinear'. 
      if not has(sol,FAIL) and has(sol,int) and contains(solveOptions,IgnoreAnalyticConstraints) then 
        return(sol);
      end_if;
      if has(sol,FAIL) or (has(sol,int) and l<>[]) then
	      // do not allow nested Liouvillian solutions 
        userinfo(1,"factoring method failed");
        return(FAIL);
      end_if;
    end_while;
    userinfo(1,"factoring method worked");
    return(sol);
  end_if
end_proc:

/* 
  returns a list L of differential equations
  so that the original equation is L[1] @ L[2] @ ... @ L[nops(L)] 
  
  NOTE: Method works only for 2nd order case!     
*/
ode::factor_aux:= proc(eq,y,z,n,solveOptions,odeOptions)
  local l,a,b,c,d,f,g,e,i,g0,g1;
begin
  if n=2 then
    if (l:= Type::Linear(eq,[diff(y(z),z$i)$i=0..n]))<>FALSE then
      a:=l[3]; 
      b:=l[2]; 
      c:=l[1];
      if map({a,b,c}, testtype, Type::RatExpr(z))={TRUE} then
        /* 
           if a*y''+b*y'+c*y factors into (d*y'+e) @ (f*y'+g),
           then a=d*f, b=d*f'+d*g+e*f and c=d*g'+e*g 
        */
        g0:= `#g0`;
        g1:= `#g1`;
        g:= g0+g1*z;
        if poly(a)=FAIL then
          return([eq]);
        end_if;
        for d in polylib::divisors(a) do
          f:= divide(a,d,Exact);
          e:= (b-d*diff(f,z)-d*g)/f;
          eq:= d*diff(g,z)+e*g-c;
          eq:= numer(eq);
          eq:= coeff(eq,[z]);
          if eq<>FAIL and
             type((eq:=solvelib::discreteSolve({eq},{g0,g1},
                                               op(solveOptions))))=DOM_SET
             and eq<>{} then
            eq:= map(op(eq,1),ode::normal,Expand=FALSE);
            eq:= subs(map(eq,map,expr),
                          RootOf=proc(e,x)
                                 begin
                                   e:=solvelib::discreteSolve(e,x,
                                                              op(solveOptions));
                                   if type(e)<>DOM_SET or e={} then
                                     FAIL
                                   else
                                     op(e,1)
                                   end_if
                                 end_proc, 
                     EvalChanges);
            /* 
               do 2 substitutions due to bug in gbasis
               (univariate polynomial first instead of last) 
            */
            e:= ode::normal(subs(e,op(eq),op(eq)),Expand=FALSE);
            g:= subs(g,op(eq),op(eq));
            return([d*diff(y(z),z)+e*y(z)+l[4],f*diff(y(z),z)+g*y(z)]);
          end_if
        end_for
      end_if
    end_if;
  end_if;
  return([eq]);
end_proc:

