/* 
   REFERENCES: [1] G.M. Murphy: "Ordinary Differential Equations and Their 
                                 Solutions", Section 8-2., pp. 36

   DETAILS: 

     ode::homogeneousG(eq,y,x) tries to solve the ODE eq with respect to y(x)
     using the method of homogeneous G (gleichgradig in German) equations. It
     returns either a set of solutions or FAIL.
  
     Applicable to: first order ODEs of the form y' = x^(n-1)*f(y/x^n), also 
     called "Isobaric Equations", where f is an analytic function and n a 
     rational number.


   EXAMPLES: 

    >> ode::homogeneousG(diff(y(x),x)-a*y(x)^3-b/x^(3/2), y,x);

    >> ode::solve((4*x*y(x)+3*y(x)^4)+(2*x^2+5*x*y(x)^3)*diff(y(x),x), y(x));
*/


ode::homogeneousG:= proc(eq,y,x,solveOptions,odeOptions) 
  local yp, zz, fx, fy, alpha, sol, u, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  yp:= genident();
  u:= genident();
  userinfo(2, "trying to recognize an homogeneous equation of type G");
  eq:= subs(eq,{diff(y(x), x)=yp,y(x)=y});
  if testtype(eq, Type::PolyExpr(yp)) then
    if degree(eq, [yp], yp) = 1 then
      eq:= -coeff(eq,[yp],yp,0)/coeff(eq,[yp],yp,1);
      zz:= ode::normal(x*eq/y);
      if has(zz,y) then
        fx:= ode::normal(x*diff(zz,x),Expand=FALSE);
        fy:= ode::normal(y*diff(zz,y),Expand=FALSE);
        if fy <> 0 then 
          //alpha:= normal(expand(normal(fx/fy), optIgnoreAnalyticConstraints)) // alpha=-n in Murphy p36
          alpha:= ode::normal(fx/fy); // alpha=-n in Murphy p36
        else
          return(FAIL);
        end_if
      elif not has(zz, x) then
        alpha:= 0
      else return(FAIL);
      end_if;
      if not has(alpha, [x, y]) then
        userinfo(1,"type G homogeneous equation");
        // We used to apply 'combine@expand' to 'u(x)*(evalAt(zz,y=u(x)/x^alpha)+alpha)' 
        // before CR. This does not seem necessary here anymore. 
        sol:= ode::separate(x*diff(u(x),x)-u(x)*(evalAt(zz,y=u(x)/x^alpha)+alpha),u,x,1,
                            solveOptions,odeOptions);
        if sol <> FAIL then
          userinfo(1,"homogeneous G worked");
          //ind:= indets(sol) minus {u,x};
          //sol:= piecewise::assumeIdentsTrue(sol,ind);
          return(ode::mapsol(sol, _mult, x^(-alpha), solveOptions, odeOptions));
        else
          userinfo(1,"homogeneous G failed");
        end_if;
      end_if;
    end_if;
  end_if;
  
  return(FAIL);
end_proc:

