/* 
   REFERENCES: [1] D. Zwillinger: "Handbook of Differential Equations", 
                   Section 80, pp. 327  

   DETAILS: 

    ode::interchange(eq,y,x,n) tries to solve the n-th order ODE eq=0 wrt y(x)
    by interchanging y and x. 

   NOTE. The following comment refers to an older edition of [1]: 

           "There is a typo in (68.5): the solution is 
            x(y) = sqrt(A*exp(2*y^3/3)-y^3-3/2)."

         In the current version of [1] everything is o.k. 
         

   EXAMPLES:

     >> ode::interchange(diff(y(x),x)-x/(x^2*y(x)^2+y(x)^5),y,x,1);

     >> ode::interchange(diff(y(x),x)^3-4*x*y(x)*diff(y(x),x)+8*y(x)^2,y,x,1);

     >> setuserinfo(Any,1):
     >> ode::interchange(diff(y(x),x,x)+x*y(x)*diff(y(x),x)^3,y,x,2);
*/

ode::interchange:= proc(eq,y,x,n,solveOptions,odeOptions) 
  local i,l,sols, X, intterm, constantsols, eqnew, optIgnoreAnalyticConstraints,
        mapinverse: DOM_PROC;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  
  // --------------------------------------------------------------------------------
  // local method mapsolve
  //   mapinverse(S)
  //      S - set of functions in y
  //   returns {f@@(-1)(y); f in S} where each f@@(-1)
  //   is expressed as function of x
  //   in other words, return the union of all solve(f=x, y) where f ranges over S
  // --------------------------------------------------------------------------------
  mapinverse:=
  proc(S)
    local res, i;
  begin
    case type(S)
      of "_union" do
        res:= map({op(S)}, mapinverse);
        if contains(res, FAIL) then
          return(FAIL)
        else
          return(_union(op(res)))
        end_if
      of DOM_SET do  
        /*
          Note by Kai: Maybe we should use 'MaxDegree = 3' in the following call
                       to find explicit representation of more solutions. Currently, 
                       this strategy is not efficient. The user must give 
                       'MaxDegree = 3' via 'solveOptions'. 
        */
        res:= [op(map(S, f -> solve(f=x, y, op(solveOptions), IgnoreProperties)))];
        for i from 1 to nops(res) do
          if (hastype(res[i], "solve") and not hastype(res[i], "int")) or 
              hastype(res[i], Dom::ImageSet) then  
            res[i]:= FAIL;
          else 
            if hastype(res[i], piecewise) then
              res[i]:= piecewise::disregardPoints(res[i]);
            end_if;
          end_if;
        end_for;
        res:= _union(op(select(res,_unequal,FAIL)));
        //res:= {op(map(res, elem -> if elem <> FAIL then if type(elem) = DOM_SET then op(elem) else elem end_if; end_if))};
        if res = {} then
          return(FAIL)
        else
          return(res)
        end_if;
      of "solve" do
      of RootOf do  
        // S is an implicit solution S = \{ x(y) ; g(x, y) = 0 \}
        // then the inverses simply are  \{ y(x); g(x, y) = 0 \}
        res:= solve(subs(op(S, 1), op(S, 2) = x), y, op(solveOptions));
        if type(res) = "solve" then
          return(FAIL)
        else
          return(res)
        end_if
      of piecewise do 
        S:= piecewise::disregardPoints(S);
        if type(S) = piecewise then 
          return(piecewise::disregardPoints(mapinverse(_union(S[i] $ i = 1..nops(S)))));
        end_if;    
    end_case;
    FAIL
  end_proc;
  
  // ----------------------------- M A I N ----------------------------------
 
  X:= genident();
  /* 
      the following lines are needed because one cannot deal with other
      functions of x, for example p(x)*diff(y(x),x)+q(x) 
  */
  l:= select(map(select(indets(eq,PolyExpr),testtype,"diff"),op,1),has,x);
  if l minus {y(x)} <> {} then
    return(FAIL)
  end_if; 
  // Do not interchange if there are expressions like int(f(x),x):
  intterm:= misc::subExpressions(eq, "int");
  if has(map(intterm,op,2),x) then
    return(FAIL);
  end_if;
  userinfo(10,"trying to interchange dependent and independent variable");
  l:= [1/diff(X(y),y)];
  for i from 2 to n do
    l:= append(l, expand(diff(l[i-1],y)*l[1], optIgnoreAnalyticConstraints))
  end_for;
  eqnew:= numer(subs(eq,diff(y(x),x$i)=l[i] $ i=1..n,
                     y(x)=y,x=x(y),X=x));
   /* the following line avoids that we try to interchange back,
     but the call of "numer" above is crucial to have a normal form */
  //sysassign(ode::interchange(eqnew,x,y,n,solveOptions,odeOptions),FAIL);
  // check the depth here in order not to enter an infinite loop 
  sysassign(ode::depth,ode::depth+1);
  unassume(x);
  assume(y in R_);
  sols:= ode::solve_eq(eqnew,x,y,{},solveOptions,odeOptions);
  sysassign(ode::depth,ode::depth-1);
  if has(sols,FAIL) then
    userinfo(10,"interchanging failed");
    return(FAIL);
  end_if;
  sols:= mapinverse(sols);
  if sols = FAIL then
    userinfo(10,"computing the inverse failed");
    return(FAIL);
  end_if;
  // the functional inverse of y does not exist if y is constant
  // Hence, we have to compute constant solutions separately
  if traperror((l:= subs(eq, y(x) = y, EvalChanges))) <> 0 then 
    constantsols:= {};
  else   
    if has(l, x) then
      constantsols:= {}
    else
      constantsols:= solve(l, y, op(solveOptions))
    end_if;
  end_if;  
  
  return(sols union constantsols);
end_proc:

