/* 

   DESCRIPTION: 
     This file contains functions for solving scalar differential equations and
     systems of differential equations using the Laplace transform.  
   
   FUNCTIONS: 
   
    - used Parameter:
       ++ eq,unk   :DOM_EXPR
       ++ sys,unks :DOM_SET
       ++ inits    :DOM_SET
       ++ x        :DOM_IDENT
       ++ eq    = ordinary differential equation in unk
       ++ unk   = unknown function
       ++ sys   = set of ordinary differential equations together with initial
       ++         conditions              
       ++ unks  = set of unknown functions in one independent variable
       ++ x     = independent variable 
       ++ inits = set of initial conditions
   
    - ode::laplace(sys,unks)
      ode::laplace(ode(sys,unks))
      ode::laplace(eq,unk)
      ode::laplace(ode(eq,unk))
       ++ returns the solutions of the system of ordinary differential equations
       ++ with respect to the unknown functions unks or
       ++ the expression transform::laplace(ode(..)) if called from outside 
       ++ ode::solve and FAIL otherwise using the laplace method.
       ++ No special cases for equations with variable coefficients are 
       ++ implemented. Thus it is best to use for equations with constant
       ++ coefficients.
    - ode::laplaceScalar(eq,unk,x,inits)
       ++ performs the laplace method for a scalar equation.
    - ode::laplaceSystem(sys,unks,x,inits)
       ++ performs the laplace method for a system of equations.


   EXAMPLE: 

     >> eq:= diff(y(x),x$2)+4*y(x)=sin(w*x):
     >> ode::laplace(eq,y(x))
     
     >> eq0:= diff(y(t),t$2)+5*diff(y(t),t)+6*y(t)=0:
     >> ode::laplaceScalar(eq0,y,t,{y(0)=0,D(y)(0)=1})
     
     >> eq1:= m*diff(x(t),t$2)-e*H*diff(y(t),t)=0
     >> eq2:= m*diff(y(t),t$2)+e*H*diff(x(t),t)=e*Ef
     
     >> eqs:= {eq1,eq2}
     >> inits:= {x(0)=0,D(x)(0)=0,y(0)=0,D(y)(0)=0}
     >> ode::laplaceSystem(eqs,{x,y},t,inits)
     
     ode::laplace(ode(eqs union inits,{x(t),y(t)}))
     transform::laplace(ode(eqs union inits,{x(t),y(t)}))
*/

ode::laplace:= proc()
  local components,result,solveOptions,odeOptions;
begin
  // This is currently only a first primitive version.
  // Since the chances to run into an error while using transfrom::laplace
  // transform::invlaplace and solve are really good, I think it is a
  // good idea to use traperror to prevent that errors.
  if args(0) > 2 then
    components:=ode::getComponents(args(1..args(0)-2));
    // The last two arguments are 'solveOptions' and 'odeOptions'.
    solveOptions:= args(args(0)-1);
    odeOptions:= args(args(0));
  else 
    components:=ode::getComponents(args());
    solveOptions:= {};
    odeOptions:= {};
  end_if;
  result:=FAIL;
  if domtype(components[1])<>DOM_SET then 
    traperror( (result:=ode::laplaceScalar(components,solveOptions,odeOptions)) )
  else
    traperror( (result:=ode::laplaceSystem(components,solveOptions,odeOptions)) )
  end_if;
  if calledFromWithinDSolve<>TRUE and result=FAIL then
    return(hold(transform::laplace)(ode((if domtype(components[1])<>DOM_SET then
                                    {components[1]}
                                  else components[1]
                                  end_if) union components[4],
                                 (if domtype(components[1])<>DOM_SET then
                                    components[2](components[3])
                                  else map(components[2],e->e(components[3]))
                                  end_if))));
  else
    return(result);
  end_if
end_proc:

// this procedure is already contained in ode::laplaceSystem, but yields better
// simplifications and moreover the result is a set of expressions and not
// a set of equations.
ode::laplaceScalar:= proc(eq,y,x,inits={},solveOptions,odeOptions)
  local s,X,EQs,imSet;
begin
  s:= genident("s");
  X:= genident("X");
  EQs:= subs(transform::laplace(eq,x,s),transform::laplace(y(x),x,s)=X,EvalChanges);
  EQs:= subs(EQs,inits,EvalChanges);
  imSet:= solvelib::discreteSolve(EQs,X,op(solveOptions));
  return(map(imSet,transform::invlaplace,s,x));
end_proc:

ode::laplaceSystem:= proc(eqs,ys,x,inits={},solveOptions,odeOptions)
  local s,X,EQs,imSet,e,i,sol;
begin
  s:= genident("s");
  if domtype(ys)<>DOM_SET then ys:={ys} end_if;
  ys:= [op(ys)];
  X:= [genident("X") $ e in ys];
  EQs:= map(eqs, transform::laplace,x,s);
  EQs:= subs(EQs,transform::laplace(ys[i](x),x,s)=X[i] $ i=1..nops(ys),EvalChanges);
  EQs:= subs(EQs,inits,EvalChanges);
  imSet:= solvelib::discreteSolve(EQs,X,op(solveOptions));
  sol:= subs(map(imSet,
                 e->map(e,ee->op(ee,1)=transform::invlaplace(op(ee,2),s,x))),
             X[i]=ys[i](x) $ i=1..nops(ys));
  //if nops(sol)=1 and domtype(sol)=DOM_SET then sol:={op(op(sol))} end_if;
  return(sol);
end_proc:


