/* 
  =============================
  METHODS FOR LINEAR ODE SYTEMS
  =============================
 
  REFERENCES: [1] E. Kamke: "Differentialgleichungen"


  DETAILS:
    Solve a set of linear odes in z.  
      ++ l is a list [u,v,...] of indeterminates for the functions u(z), 
         v(z), ...  
      ++ A is a table of (ord+1) lists of n lists of n coefficients
      ++ b is a list
      ++ n = nops(l) is the number of unknown functions
      ++ ord = nops(A)-1 is the maximal order of the equations
      ++ nn is the number of wanted solutions (nn=n when ord=1)
 
    The coefficient of diff(l[j](z),z$k) in equation i is stored in A[k][i][j]
    and the inhomogeneous coefficient of equation i is stored in _b[i]. 

  EXAMPLES: 
    >> ode::linsys(table(0 = [[-1, -8], [-9, -2]], 1 = [[0, 1], [1, 0]]),
                   [-1, 0], {y1, y2}, t, 2, 1); 

       solves the first-order system

           y2'(t)-y1(t)-8*y2(t)=1, y1'(t)-9*y1(t)-2*y2(t)=0 
*/


ode::linsys:= proc(AA,_b,l,z,n,ord,nn,solveOptions,odeOptions)
  local k,lc,M,Id,invlc,C,eigenvals,lambda,eigenvects,sol,v,i,m,A,j,hsol,
        T,maxord,vec,vl,s,gm,f1i,f1,f2,f1z,delta,ls,leq,a,b,lab,zz,inte,
        eqx,solx,xx,yy,eqy,intOptions,optIgnoreAnalyticConstraints,S,Sol,
        indConj,cvec;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  if ord>=2 then
    if AA[0]=[[0$n]$n] then
      sol:= ode::linsys(table(i=AA[i+1]$i=0..ord-1),_b,l,z,n,ord-1,nn,
                        solveOptions,odeOptions);
      if sol = FAIL then 
        return(FAIL);
      end_if;                 
      sol:=map(op(sol),
               proc(s)
               begin
                 op(s,1)= int(op(s,2),z,intOptions) + genident("C")
               end_proc );
      return({sol});
    end_if;
    // reduce to a system of smaller order
    /* l[1..n] represent the original functions,
       l[n+1..2*n] their derivatives,
       l[2*n+1..3*n] their second derivatives, ... */
    A:= table();
    T:= table();
    maxord:= table();
    for j from 1 to n do
      for k from ord downto 1 do
        for i from 1 to n do
          if AA[k][i][j] <> 0 then
            break;
          end_if
        end_for;
        if i<=n then
          break
        end_if
      end_for;
      maxord[j]:= k;
      T[j,0]:= j;
      while k>1 do
        k:= k-1;
        l:= append(l,genident());
        T[j,k]:= nops(l)
      end_while;
    end_for;
    v:= [0 $ nops(l)];
    A[0]:= [v $ nops(l)];
    A[1]:= A[0];
    for i from 1 to n do
      // equations (1) to (n)
      for j from 1 to n do
        A[0][i][j]:= AA[0][i][j];
        A[1][i][j]:= AA[1][i][j];
        for k from 2 to maxord[j] do
          A[1][i][T[j,k-1]]:= AA[k][i][j]
        end_for
      end_for;
    end_for;
    i:= n;
    for j from 1 to n do
      for k from 1 to maxord[j]-1 do
        // diff(y[j],z $ k)' = diff(y[j],z $ k+1)
        i:= i+1;
        A[0][i][T[j,k]]:= 1;
        A[1][i][T[j,k-1]]:= -1;
      end_for;
    end_for;
    return(ode::linsys(A,[op(_b),0$(nops(l)-n)],l,z,nops(l),1,n,
                       solveOptions,odeOptions));
   end_if;
   // now ord=1 
   // first transform to matrices 
   M:= Dom::Matrix(Dom::ExpressionField(normal));
   A[0]:= M(AA[0]);
   A[1]:= M(AA[1]);
   // normalize the leading coefficient
   lc:= A[1];
   Id:= M(n,n,1,Diagonal);
   if lc <> Id then
     if (invlc:= lc^(-1))=FAIL then
       return(FAIL);
     end_if;
     A[0]:= invlc * A[0];
   else
     invlc:= Id;
   end_if;
   if not has(expr(A[0]),z) then 
     // solve Y' + A[0] * Y = -_b
     lc:= []; // will contain the integration constants
     // find the eigenvalues
     eigenvects:= linalg::eigenvectors(A[0], "CalledFromODELinsys");
     if type(eigenvects) <> DOM_LIST then
       return(FAIL)
     end_if;
     eigenvals:= map(eigenvects,op,1);
     userinfo(2,"eigenvalues are",eigenvals);
     sol:= M(n,1); 
     k:= table();
     Sol:= {};
     for lambda in eigenvects do
       // expr is needed if one uses ExpressionField(normal)
       m:= lambda[2]; 
       v:= lambda[3]; 
       lambda:= expr(lambda[1]);
       userinfo(2,"for eigenvalue",lambda);
       userinfo(2,"dimension of eigenspace is",m);
       userinfo(2,"eigenvectors for value",lambda,"are",v);
       if v=[] then 
         return(FAIL);
       end_if;
       k:= exp(-lambda*z);
       if nops(v)=m then // There are enough eigenvectors
         for i from 1 to m do
           Sol:= Sol union {k*v[i]};
         end_for;
       else  
         // There are not enough eigenvectors, we will have
         // to find some generalized eigenvectors.
         gm:= 0;
         for vec in v do
           vl := [];
           while nops(vec) > 0 and vec <> NIL do
             vl:= append(vl,vec);
             // Note that we want the generalized eigenvalues for
             // -A[0]
             vec:= linalg::matlinsolve(-A[0]+lambda*Id,vec,Special);
           end_while;
           gm:= gm+nops(vl);
           for i from 1 to nops(vl) do
             s:= _plus(z^(j-1)/fact(j-1)*vl[i-j+1]$j=1..i);
             Sol:= Sol union {k*s};
           end_for;
         end_for;
         if gm<>m then
           return(FAIL);
         end_if;
       end_if;
     end_for;
     if has(Sol,I) and not has(AA,I) then 
       // ---------------------------
       // We want fundamental systems not involving the complex unit 'I'. 
       // If 'vec' is a complex solution vector of the fundamental system, 
       // then 'conjugate(vec)' is also an element of the fundamental system 
       // and due to the linearity of the system we can replace 'vec' and 
       // 'conjugate(vec)' by 'Re(vec)' and 'Im(vec)'. 
       S:= {};
       Sol:= [op(Sol)]; // list of solutions; we need an ordered data structure here 
       while Sol <> [] do
         vec:= Sol[1]; 
         if has(vec,I) then
           // Given 'vec' we now search for its conjugate in 'Sol'. 
           cvec := conjugate(vec) assuming z in R_;
           indConj:= contains(Sol,cvec); 
           if indConj = 0 then 
             for i from 1 to nops(Sol) do 
               // Experiments showed that using 'testeq' with small number of 'Steps' and 
               // 'NumberOfRandomTests' is the only practical and efficient(!) way to 
               // further detect conjugates. If this fails due to missing simplification
               // steps, we need to accept it. Remember that we are trying to make the 
               // results more convenient and such post-processing should not make up 
               // a significant part of the time consumed for the whole computation.
               // Furthermore, if this heuristic fails, this does not mean that we obtain 
               // incorrect results. They are simply less nice. 
               if (testeq(Sol[i],cvec,
                          Steps=5,NumberOfRandomTests=5) 
                          assuming z in R_) = TRUE then 
                 indConj:= i;     
                 break;
               end_if;  
             end_for;  
           end_if;  
           if indConj > 0 then 
             S:= S union {map(vec, x -> Re(rectform(expr(x))) assuming z in R_),
                          map(vec, x -> Im(rectform(expr(x))) assuming z in R_)};
             delete Sol[indConj];
             delete Sol[1];
           else 
             S:= S union {vec};
             delete Sol[1];
           end_if;  
         else 
           S:= S union {vec};
           delete Sol[1];
         end_if;
       end_while;
     else 
       S:= {op(Sol)};
     end_if;
     // Now create the solutions involving the constants of integration 
     // to return the components of the general solution vector as 
     // scalar functions. 
     sol:= M(n,1); // container for the solutions 
     lc:= []; // collects the constants of integration 
     for i from 1 to nops(S) do 
       C:= genident("C");
       lc:= append(lc,C);
       sol:= sol + C*S[i];  
     end_for;
     //---------------------------
     hsol:= {l[i](z)=expr(sol[i]) $ i=1..nn};
     if {op(_b)} = {0} then
       // homogeneous system
       return({hsol}); 
     else
      lc:= linsolve({expr(_plus(A[1][i,j]*sol[j]$j=1..n))+_b[i] $ i=1..n},{op(lc)});
      if nops(lc)=0 or lc=FAIL then
        userinfo(1,"unable to solve for integration constants");
        return(FAIL);
      else
        lc:= map(lc,(eq -> (op(eq,1) = op(eq,1)+int(op(eq,2),z,intOptions))));
        lc:= subs(hsol,lc,EvalChanges);
        return({map(map(lc,expand),proc()
                            begin
                              op(args(1),1)=combine(op(args(1),2),exp,optIgnoreAnalyticConstraints)
                            end_proc )});
      end_if;
    end_if;
  elif n = 2 and 
       (iszero(A[1][1,1]) or iszero(A[1][1,2])) and 
       (iszero(A[1][2,1]) or iszero(A[1][2,2])) then
    // Check if the system is of the same type of Example 8.20 p.611 [1]
    userinfo(1,"linear differential system with nonconstant coefficients");
    userinfo(1,"try a particular method");
    f1i:= expr((A[0]*M(l))[1]); 
    f2:= expr((A[0]*M(l))[2]);
    f1:= factor(f1i);
    f1:= coerce(f1, DOM_LIST);
    f1z:= select([i $ i=1..nops(f1) div 2], ()->has(f1[2*args(1)], z));
    f1:= expand(_mult(f1[2*i]^f1[2*i+1] $ i in f1z),optIgnoreAnalyticConstraints);
    assert(not iszero(f1));
    A[0]:= expr(map(A[0],() -> ode::normal(1/f1*args())));
    if not has(A[0],z) then
      _b:= expr(invlc*M(_b));
      if (delta:= expand(A[0][1,1]*A[0][2,2]-A[0][1,2]*A[0][2,1],optIgnoreAnalyticConstraints)) <> 0 then
        s:= genident("s");
        ls:= solvelib::discreteSolve(s^2-s*(A[0][1,1]+A[0][2,2])+delta,s,op(solveOptions));
        leq:= {};        
        if nops(ls)=2 then
          for s in ls do
            a:= genident("a"); 
            b:= genident("b");
            lab:= linsolve({a*A[0][1,1]+b*A[0][2,1]-s*a,
                            a*A[0][1,2]+b*A[0][2,2]-s*b}, [a,b]);
            if nops(lab)=1 then
              if not has(lab,a) then
                lab:= ([subs(op(lab),a=1,EvalChanges), a=1]);
              else
                lab:= ([subs(op(lab),b=1,EvalChanges), b=1]);
              end_if;
            end_if;
            zz:= ode::solve_linear(subs([s*f1,1,a*_b[1,1]+b*_b[2,1]],lab,EvalChanges),
                                   z,1,solveOptions,odeOptions);
            leq:= leq union {subs(a*l[1]+b*l[2]-op(zz),lab,EvalChanges)};
          end_for;
          return({subs(linsolve(leq, {l[1], l[2]}), {l[i]=l[i](z) $ i=1..2},EvalChanges)});
        elif nops(ls)=1 then
          s:= op(ls);
          a:= genident("a"); 
          b:= genident("b");
          lab:= linsolve({a*A[0][1,1]+b*A[0][2,1]-s*a,
                          a*A[0][1,2]+b*A[0][2,2]-s*b}, [a,b]);
          if lab = [] then 
            return(FAIL); 
          end_if;               
          if nops(lab)=1 then
            if not has(lab,a) then
              lab:= ([subs(op(lab),a=1,EvalChanges), a=1]);
            else
              lab:= ([subs(op(lab),b=1,EvalChanges), b=1]);
            end_if;
          end_if;
          zz:= ode::solve_linear(subs([s*f1,1,a*_b[1,1]+b*_b[2,1]],lab,EvalChanges),z,1,
                                 solveOptions,odeOptions);
          zz:= op(zz);
          if not iszero(A[0][1,2]) then
            eqx:= subs([s*f1,1,_b[1,1]+1/b*A[0][1,2]*zz*f1],lab,EvalChanges);
            xx:= ode::solve_linear(eqx,z,1,solveOptions,odeOptions);
            return({[l[1](z)=op(xx),l[2](z)=subs(1/b*(zz-a*op(xx)),lab,EvalChanges)]})
          else
            eqy:=subs([A[0][2,2]*f1,1,_b[2,1]+A[0][2,1]/a*zz*f1],lab,EvalChanges);
            yy:=ode::solve_linear(eqy,z,1,solveOptions,odeOptions);
            return({[subs(l[1](z)=1/a*zz,lab,EvalChanges), l[2](z)=op(yy)]})
          end_if;
        end_if;
      elif not iszero(A[0][1,1]) and not iszero(A[0][1,2]) then
        assert(not iszero(f1i));
        // It may be possible to remove the simplification step in the future. 
        // This requires an extending the test base of solving ODE systems. 
        lambda:= simplify(f2/f1i,optIgnoreAnalyticConstraints);
        inte:= int(lambda*_b[1,1]-_b[2,1], z,intOptions);
        C:= genident("C");
        eqx:= [ A[0][1,1]*f1+lambda*A[0][1,2]*f1,1,
               _b[1,1]+A[0][1,2]*f1*(inte+C) ];
        solx:= ode::solve_linear(eqx,z,1,solveOptions,odeOptions);
        return({[l[1](z)=op(solx), l[2](z)=lambda*op(solx)+inte+C]});
      else
        return(FAIL);
      end_if:
     else
       return(FAIL);
    end_if:
  else
    return(FAIL);
  end_if
end_proc:
