/*
  DESCRIPTION:

    This file contains functions for creating and handling objects of type ode.
    A valid differential equation must at least contain one given unknown 
    differential function.

  FUNCTIONS: 
                              
    - used Parameter:
       ++ a        : this domain 
       ++ eq,unk   : DOM_EXPR
       ++ sys,unks : DOM_SET
       ++ indepVar : DOM_IDENT
       ++ depVars  : DOM_IDENT or DOM_SET
       ++ a        = an element of domain ode
       ++ eq       = ordinary differential equation in unk
       ++ unk      = unknown function
       ++ sys      = set of ordinary differential equations together with initial
       ++            conditions              
       ++ unks     = set of unknown functions in one independent variable
       ++ indepVar = independent variable
       ++ depVars  = dependent variable or set of dependent variables
   
    - ode::new(eq,unk)
      ode::new(sys,unks)
       ++ creates from an ordinary differential equation eq in unk or a system
       ++ of ordinary differential equations in the unknowns unks an element of
       ++ the domain ode.
       ++ Currently the internal representation is a sequence of
                - set of differential expressions (representing the equations)
                  or a differential expression
                - set of dependent variables or dependent variable
                - independent variable
                - set of initial conditions

    - ode::getComponents(a,..)
      ode::getComponents(sys,unks,..)
       ++ returns a sequence of the internal representation together with
       ++  all remaining args
       ++ for the domain element a. It is assumed that a is a valid object of
       ++ type ode. 

    - ode::makeEquation(eq,depVars,indepVar)
       ++ returns FAIL or a valid differential expression representing a 
       ++ differtial equation in at least one depVars[i](indepVar).

    - ode::makeInitialCondition(eq,depVars,indepVar)
       ++ returns FAIL or a (not proper checked!) initial condition.
       ++ Such intial conditions are at least of the form 
       ++  D(..D(depVars[i])..)=e or depVars[i](e1)=e2 
       ++ where the ei's are not allowed to contain depVars.

    - ode::print(ode)
       ++ prints an element of ode out.

    - ode::evaluate(ode)
       ++ evaluates an element of ode.

    - ode::splitSys(sys,depVars,indepVar)
       ++ returns a sequence of two sets, where the first set contains the 
       ++ differential expressions and the second the initial conditions.

  EXAMPLES:

    >> o:= ode({y'(x)+1,z'(x)+y(x),D(y)(0)=2,y(0)=1},{y(x),z(x)}): 
    >> ode::getComponents(o)                 
*/

ode::new:= proc(eq, unk=FAIL)
  local indepVar,components,tmp;
begin
  if args(0)=0 then 
    error("Wrong number of arguments"); 
  end_if;
  if type(eq)=ode then 
    return(eq); 
  end_if;
  if args(0)<>2 then 
    error("Wrong number of arguments"); 
  end_if;
  // argument testing: the unknowns
  case type(unk)
    of "function" do
      if nops(unk)<>1 then
        error("Function must be univariate")
      else
        indepVar:=op(unk,1);
      end_if;
      break;
    of DOM_LIST do 
      unk:={op(unk)};
    of DOM_SET do
      if map(unk, type)<>{"function"} then
        error("Unknowns must be functions")
      end_if;
      tmp:= map(unk,op);
      if nops(tmp) <> 1 then
        error("Unknowns must be univariate functions ".
              "in the same variable")
      else
        indepVar:=op(tmp,1)
      end_if;
      break;
    of DOM_FAIL do 
      error("Missing unknowns");
    otherwise
      error("Illegal unknowns")
  end_case;
  // argument testing: the equation(s)
  case domtype(eq)
    of DOM_EXPR do
      break;
    of DOM_LIST do 
      eq:={op(eq)};
    of DOM_SET do
      if map(eq, domtype)<>{DOM_EXPR} then
        error("Illegal equations");
      end_if;
      break;
    otherwise
      error("First argument must be equation or set of equations");
  end_case;
  // end of argument testing 
    
  components:=ode::makeComponents(eq,unk);
  new(ode,components);
end_proc:


ode::makeEquation:= proc(eq,depVars,indepVar)
begin
  if has(eq,indepVar) then
    eq:= rewrite(eq,diff);
    if has(eq,D) then 
      return(FAIL); 
    end_if;
    if type(eq)="_equal" then 
      eq:= op(eq,1)-op(eq,2);
    end_if;
    if has(eq,depVars(indepVar)) then 
      return(eq);
    else 
      return(FAIL);
    end_if;
  else
    return(FAIL);
  end_if;
end_proc:


ode::makeInitialCondition:= proc(eq,depVars,indepVar)
begin
  if has(eq,indepVar) or type(eq) <> "_equal" then
    return(FAIL);
  else
    return(eq);
  end_if
end_proc:


ode::splitSys:= proc(sys,depVars,indepVar)
  local eqs,inits,eq,h;
begin
  eqs:= null(); 
  inits:= null();
  if domtype(sys)<>DOM_SET then 
    sys:={sys};
  end_if;
  for eq in sys do
    if (h:= ode::makeEquation(eq,depVars,indepVar)) = FAIL then
      if (h:=ode::makeInitialCondition(eq,depVars,indepVar)) = FAIL then
        error("Invalid equation or initial condition contained");
      else
        inits:= inits,h;
      end_if;
    else
      eqs:= eqs,h;
    end_if;
  end_for;
  return({eqs},{inits});
end_proc:


ode::makeComponents:= proc(sys,unks)
  local inits,depVars,indepVar,l,z,f,eqs,components,ODEsys;
begin
  case type(unks)
    of DOM_SET do
      l:= {}; 
      z:= {};
      for f in unks do
        if type(f) = "function" then
          if type((depVars:=op(f,0))) = DOM_IDENT and nops(f) = 1 then
            l:= l union {depVars};
            z:= z union {op(f)};
            next;
          end_if;
        end_if;
        error("Invalid unknown function");
      end_for;
      if nops(z) <> 1 then
        error("Wrong number of independent variables") 
      end_if;
      z:= op(z);
      // split equations and initial conditions
      eqs:= ode::splitSys(sys,l,z);
      ODEsys:= eqs[1]; 
      inits:= eqs[2]; 
      if nops(unks) = 1 and nops(ODEsys) = 1 then 
        unks:=op(unks)
      else // system of differential equations
        components:= ODEsys,l,z,inits;
        break;
      end_if;
    of "function" do // only one unknown function
      depVars:= op(unks,0);
      if not has(sys,depVars) then
        error("Differential function does not appear");
      end_if;
      if nops(unks)<>1 then
        l:= ode::diffvars(sys,depVars);
        if nops(l)=1 then  //only one variable is differential 
          z:= op(l);
          l:= [op(unks)];
          sys:= ode::subsvars(sys,depVars,contains(l,z),l);
          unks:= depVars(z);
        else
          error("Resolution of partial differential equations ".
                "not yet implemented");
        end_if
      end_if;
      indepVar:= op(unks);
      if {type(depVars),type(indepVar)} <> {DOM_IDENT} then
        error("Invalid arguments");
      end_if;
      if depVars = indepVar then
        error("Dependent and independent variable should differ");
      end_if;
      eqs:= ode::splitSys(sys,depVars,indepVar);
      inits:= eqs[2];
      eqs:= eqs[1];
      if nops(eqs)=1 then 
        components:= op(eqs),depVars,indepVar,inits;
      else 
        error("Number of equations differs from number of functions");
      end_if; 
      break;
    otherwise
      error("Invalid arguments")
  end_case;
   
  return(components);
end_proc:


ode::getComponents:= proc(a)
  local sys,unks,components,restargs;
begin
  restargs:= null();
  if args(0)=0 then 
    error("Wrong number of arguments");
  end_if;
  if domtype(a)=ode then
    components:= op(a);
    if args(0)>1 then
      restargs:= args(2..args(0))
    end_if;
  elif args(0)<2 then
    error("Wrong number of arguments")
  else
    sys:= args(1);
    unks:= args(2);
    components:= ode::makeComponents(sys,unks);
    if args(0)>2 then 
      restargs:= args(3..args(0)); 
    end_if;
  end_if;
  
  return(components,restargs);
end_proc:


ode::evaluate:= proc(a: ode)
  local components, evComponents;
begin
  components:= [ode::getComponents(a)];
  evComponents:= map(components, eval);
  if components[2] <> evComponents[2] or
     components[3] <> evComponents[3] then
    error("Some dependent or independent variable has a value");
  end_if;
  
  return(new(ode,op(evComponents)));
end_proc:

ode::print:= proc(a)
  local components;
begin
  components:=ode::getComponents(a);
  components[1] := rewrite(components[1], D);
  if domtype(components[1])=DOM_SET then
    components[1] := map(components[1], generate::sortSums);
  else
    components[1] := generate::sortSums(components[1]);
  end_if;
  return(
    hold(ode)(if domtype(components[1])=DOM_SET then
                components[1] union components[4]
              elif nops(components[4])=0 then
                components[1]
              else
                {components[1]} union components[4]
              end_if,
              map(components[2],e->e(components[3])))
  );            
end_proc:

ode::TeX:= proc(o, prio)
  local p;
begin
  p:= dom::print(o);
  return(generate::tex(p, prio));
end_proc:

ode::subs:= proc(a)
  local b;
begin
  b:= subs([extop(a)],args(2..args(0)));
  if testtype(b[1], DOM_SET) then
    // a was a system
    return(ode::new(b[1] union b[4], b[2](b[3])));
  else
    return(ode::new({b[1]} union b[4], b[2](b[3])));
  end;
  
end_proc:


ode::notSupported:= () -> error("this operation is not supported for objects of type 'ode'"):

ode::_mult:= ode::notSupported:
ode::_plus:= ode::notSupported:
ode::_invert:= ode::notSupported:
ode::_power:= ode::notSupported:



