/* 
   =================================
   METHOD OF SEPARATION OF VARIABLES
   =================================

   DETAILS: 

     ode::separate(eq,y,z,n) tries to solve the n-th order equation eq=0 for y(z)
     either recognizing a direct n-th order equation (D@@n)(y)(x) = f(x)or 
     separate equations

   EXAMPLES: 

     >> ode::separate(diff(y(x),x,x)-exp(x),y,x,2);
 
     >> ode::separate(diff(y(x),x)+y(x)^2,y,x,1);

     >> ode::separate(x+y(x)+1+(2*x+2*y(x)-1)*diff(y(x),x),y,x,1);
*/

ode::separate:= proc(eq,y,z,n,solveOptions,odeOptions)
  local yp, s, res, i, delElems, elem, rootFound, foundExpOfLn,
  handleSolution: DOM_PROC;
  // save MAXEFFORT;
begin
  
  // ----------------------------------------------------------------------
  // LOCAL METHOD for processing sets of solutions
  // ----------------------------------------------------------------------
  handleSolution:=
  proc(s)
    save MAXEFFORT;
  begin 
    case type(s)
      of DOM_SET do
        // Try to avoid 'MAXEFFORT' outside of the 'solver' and the 
        // 'simplifier'. The test base seems stable against removing this, 
        // to to keep better control stop using 'MAXEFFORT' in this central 
        // routines.  
        // 
        //if nops(s) > 1 then   
        //   MAXEFFORT:= MAXEFFORT/nops(s)
        //end_if;
        s:= map(s,ode::separateExplicit,y,z,n,solveOptions,odeOptions);
        s:= select(s,_not@has,FAIL);
        if s={} then
          return(FAIL);
        else
          return(_union(op(s)));
        end_if;
      of "_union" do
        //MAXEFFORT:= MAXEFFORT/nops(s);
        s:= map([op(s)], handleSolution);
        s:= map(s, elem -> if not has(elem,FAIL) then elem end_if);
        if contains(s,FAIL) > 0 then
          return(FAIL);
        else
          return(_union(op(s)));
        end_if;
      of solvelib::BasicSet do
        if s <> C_ then
          break;
        end_if;
        return(C_);
      of Dom::ImageSet do
        if nops(op(s,2))>=2 then
          // Probably too complicated, do not attempt this. 
          return(FAIL);
        end_if;
        res:= ode::separateExplicit(op(s, 1), y, z, n, solveOptions, odeOptions);
        if type(res) = Dom::ImageSet then  
          res:= {op(res,1)}    
        end_if;
        if res=FAIL then
          break;
        else
          res:= solvelib::Union(res, op(s, [2, 1]), op(s, [3, 1]));
          res:= solvelib::avoidAliasProblem(res,{y, z});
          return(res);
        end_if;
      of piecewise do
        // The simple case: if the conditions do not contain y(z), then
        // we may simply map:
        if not has(piecewise::conditions(s), y) then
          s:= piecewise::extmap(s, handleSolution);
          if has(s, FAIL) then
            return(FAIL);
          else
            return(s);
          end_if;
        end_if;
        // If the conditions contain y(z), we have to compute the intersection
        // of those y(z) that satisfy the explicit ode and those that satisfy
        // the condition: 
        s:= piecewise::replaceOtherwise(s);
        assert(piecewise::condition(s, nops(s)) <> Otherwise);
        //MAXEFFORT:= MAXEFFORT/nops(s);
        res:= [ode::selectSolutions
               (handleSolution(piecewise::expression(s, i)),
                piecewise::condition(s, i), y, z, solveOptions, odeOptions)
                    $ i =1..nops(s)];
//        if contains(res,FAIL) = 0 then
//          return(_union(op(res)));
//        end_if;
        res:= select(res,_not@has,FAIL);
        if res <> [] then 
          return(_union(op(res)));
        end_if;  
    end_case;
    return(FAIL);
  end_proc;
  // ----------------------------------------------------------------------
  
  // ---------------------- MAIN OF 'ode::separate' ----------------------- 
  
  // find explicit form of ode, and solve
  yp:= genident(); 
  s:= subs(eq,diff(y(z),z$n)=yp,EvalChanges);
  s:= solve(s, yp, op(solveOptions));
  /* 
    Note by Kai:  
    Here one tried to use 's:= piecewise::disregardPoints(s, z)' instead of 
    only using 's:= piecewise::disregardPoints(s)' as stated below, i.e.
    one wanted to have points disregarded only with respect to 'x'. Currently,
    the implementation of 'piecewise::disregardPoints' with additional
    argument does not work as desired in the context of ODE. It removes 
    not only branches of piecewises, but also mutilates conditions such 
    that the corresponding branch together with the corresponding 
    condition becomes wrong. Hence, we use 'piecewise::disregardPoints'
    without giving 'x' as additional argument. This may remove more branches
    than necessary and lead to incomplete case distinctions, but it does 
    not provide incorrect results. 
  */
  s:= piecewise::disregardPoints(s);         
  res:= handleSolution(s);
  if type(res) = piecewise and contains(solveOptions,IgnoreSpecialCases) then 
    res:= piecewise::disregardPoints(res);    
  end_if;
  
  if type(res) = DOM_SET then
    res:= [op(res)];
    for i from 1 to nops(res) do 
      if testtype(res[i],Type::Arithmetical) then 
        if rationalize(res[i],DescendInto=TRUE,ReplaceHardToEval)[2] = {} then  
          foundExpOfLn:= FALSE;
          misc::maprec(res[i],
                       {"exp"} = proc(elem)
                                 begin
                                   if has(elem,ln) then 
                                     foundExpOfLn:= misc::breakmap();
                                   end_if;
                                   return(elem);
                                 end_proc):
          if foundExpOfLn then 
            res[i]:= simplify::exp(res[i]);
          end_if;
        end_if;  
      end_if;  
    end_for;  
    res:= {op(res)};
  end_if;  
  
  /* =======================================================================
     Note by Kai: We have to be careful in using 'IgnoreSpecialCases'
     especially in the context of equations containing roots of 
     the indeterminate. Hence, we should first check the expression 'inty'
     to see if roots are contained. If this is the case, we need to check
     whether the solutions obtained are correct. Since in computing with 
     expressions involving roots we do not have good symplification 
     routines, we use 'testeq' as a heuristic to filter out incorrect
     solutions. 

      CONCRETE EXAMPLE: Solving 'diff(v(t), t) = (1-v(t)^2)^(3/2)'
                        without the following heuristic provides
                        a wrong result ('-sqrt(..)' is not a solution).
      ======================================================================= 
  */ 
  if type(res) = DOM_SET and length(eq) < 200 then 
    rootFound := FALSE:
    misc::maprec(eq,
                 {"_power"} = proc(x)
                                begin
                                  if type(op(x,2)) = DOM_RAT then
                                    rootFound := misc::breakmap();
                                  end_if;
                                  args()
                                end_proc):
    if rootFound then 
      delElems:= {};
      for elem in res do 
        if not contains({DOM_SET, "_union", solvelib::BasicSet, 
                        Dom::ImageSet, piecewise}, type(elem)) then                         
          if length(elem) < 100 and traperror(eq | y(z) = elem) = 0 and 
             testeq((simplify(eq | y(z) = elem,IgnoreAnalyticConstraints)) = 0, Steps = 2, 
                    NumberOfRandomTests = 50) = FALSE then 
            delElems:= delElems union {elem};
          end_if;
        end_if;
      end_for;  
      res:= res minus delElems;
    end_if;
  end_if;  
  // =======================================================================

  return(res);
end_proc:
 

ode::separateExplicit:= proc(s,y,z,n,solveOptions,odeOptions)
  local eq, parameq, i, l, inty, u, soll, Rhs, found, 
        newC, S, origEq, intOptions,optIgnoreAnalyticConstraints, sol;
begin
  // We want to remove subexpressions 'exp(anything + ln(anything with x))' 
  // and substitute them by 'exp(anything) * (anything with x)'. Because 
  // a global application of 'simplify' would probably be too expensive 
  // in the generic situation, we use 'misc::maprec' and 'expand' only 
  // on those subexpressions of 's', which are to be modified. This is not 
  // an action in the sense of the 'IgnoreAnalyticConstraints'-concept. 
  // No analytic constraints are ignored to proceed this way. 
  s:= misc::maprec(s,
                   {"exp"} = proc(elem)
                                  begin
                                    if has(elem,ln) then 
                                      expand(elem)
                                    else 
                                      elem;
                                    end_if;
                                  end_proc);  
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();  
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  eq:= hold(diff)(y(z), z$n) - s;
  origEq:= s;
  parameq:= indets(eq) minus {y,z,PI,CATALAN,EULER};
  newC:= genident("C");
  if not has(s,y) then 
    // try to integrate directly
    // cf Zwillinger page 147
    userinfo(1,"trivial n-th order equation");
    // s:= simplify(s,optIgnoreAnalyticConstraints);
    for i from 1 to n do
      userinfo(2,"integrand is",s);
      s:= int(s,z,intOptions)+genident("C");
    end_for;
    return({s});
  elif n=1 then // try to separate
    s:= Factored::convert_to(factor(origEq),DOM_EXPR);
    if type(s)<>"_mult" then 
      s:= hold(_mult)(s);
    end_if;
    ([l,s,inty]):= split(s,has,y);
    l:= subs(l,y(z)=y,EvalChanges);
    // Don't give up so soon! Try some more. Use 'expand' before factoring 
    // because this will help in the situation of trigonometric expressions 
    // like 
    //
    //   factor(expand(cos(x+y(x)) + sin(x-y(x)))) = 
    //   (cos(x) + sin(x))*(cos(y(x)) - sin(y(x)))
    //
    // But restrict this strictly to the situation where there are trigonometric 
    // expressions of this form. 
    found:= FALSE; 
    if has(l,z) and (has(l,sin) or has(l,cos)) then 
      if not found then 
        misc::maprec(l,
                     {"sin"} = proc(elem)
                                    begin
                                      if has(elem,y) and has(elem,z) then 
                                        found:= misc::breakmap();
                                      end_if;
                                      elem;
                                    end_proc);  
      end_if;
      if not found then 
        misc::maprec(l,
                     {"cos"} = proc(elem)
                                    begin
                                      if has(elem,y) and has(elem,z) then 
                                        found:= misc::breakmap();
                                      end_if;
                                      elem;
                                    end_proc);  
      end_if;
    end_if;
    if found then 
      s:= Factored::convert_to(factor(expand(origEq,optIgnoreAnalyticConstraints)),DOM_EXPR);
      if type(s)<>"_mult" then 
        s:= hold(_mult)(s); 
      end_if;
      ([l,s,inty]):= split(s,has,y);
      l:= subs(l,y(z)=y,EvalChanges);
    end_if;
    
    if not has(l,z) then
      // dy/dz = l(y)*s(z)
      // this is true either if y is constant and l(y) = 0 (singular sol.)
      // or if dy/l = s*dz
      // integrating the latter gives
      // int(dy/l)=int(s,z)
      userinfo(2,"separate equation:",diff(y(z),z)/l=s);
      // important special case:
      if (u:= Type::Linear(l, [y])) <> FALSE then
        // l= u[1]*y + u[2]
        // solving int(1/l, y) = int(s, z) gives
        // ln(l)/u[1] = int(s, z)
        // u[1]*y + u[2] = exp(u[1] * int(s, z))
        // y = (exp(u[1] * int(s, z)) - u[2])/u[1]
        if not iszero(u[1]) then 
          sol:= ((genident("C")* exp(u[1] *int(s, z, intOptions))) - u[2])/u[1];
          return(piecewise([u[1] = 0, {int(u[2]*s,z,intOptions)+genident("C")}],
                           [u[1] <> 0, {sol}]))
        else 
          return({int(u[2]*s,z,intOptions)+genident("C")})
        end_if;            
      end_if;
      // We have to be careful when we divide by l, we may lose some
      //   singular solutions. Let soll be the set of singular solutions
      if contains(indets(l), y) then
        soll:= solve(l, y, op(solveOptions));
        soll:= subs(soll, C_ = {genident("C")});
      else
        soll:= {}
      end_if;
      inty:=int(1/l,y,intOptions);
      s:= int(s, z, intOptions);
      // since int returns no integration constant, we have to find all y with
      // inty(y) = s(z) + C  
      Rhs:= genident("C");
      // rewrite abs and sign as piecewise because solve understands them
      // as complex functions while int and ode do not
      inty:= subs(inty,
                       [hold(abs) = (x-> piecewise([x>=0, x], [x<=0, -x])),
                        hold(sign)= (x-> piecewise([x>0, 1], [x<0, -1]))
                        ], EvalChanges);
      // we first find the functional inverse of inty, then
      // have to substitute s + Rhs into it
      S:= solve(inty = Rhs, y, IgnoreSpecialCases, op(solveOptions));
      S:= solvelib::avoidAliasProblem(S, {y, z});
      s:= subs(S, Rhs = s + Rhs, EvalChanges);
      s:= subs(s, C_ = {genident("C")});
      if type(s) = piecewise then
        s:= ode::distributePiecewise(s, {z},
                                     indets(s) minus ({y, z} union parameq), 
                                     solveOptions,odeOptions)
      end;
      userinfo(1,"separate method worked");
      return(soll union s)
    else
      // l depends on z. We have to solve y' = l(y, z) * s(z)
      // try with a change of function
      l:= Factored::convert_to(factor(l),DOM_EXPR);
      if type(l)="_mult" then
        u:= numer(select(l,has,y));
      else
        u:= l;
      end_if;
      l:= s*l;
      // now have to solve dy/dz = l(y, z)
      S:= /*piecewise::disregardPoints*/(solve(u,y,IgnoreSpecialCases,op(solveOptions)));
      if type(S) = DOM_SET and nops(S) <> 0 then
        u:= op(S,1);
        l:= ode::normal(subs(l,y=y+u,EvalChanges));
        // letting ynew = y - u(z) 
        // then diff(ynew, z) = diff(y, z) - diff(u, z)
        // thus have to solve dynew/dz = l - du/dz
        if not has(l,z) and not has((i:=diff(u,z)),z) then
          // dynew/dz = l-i
          // we may separate
          s:= ode::separateExplicit(l-i, y, z, 1, solveOptions, odeOptions);
          // for each ynew found, solve ynew = y - u(z) for y:
          // that is, add u(z)
          s:= ode::mapsol(s, _plus, u, solveOptions, odeOptions);
          return(s);
        end_if // not has(l, z) ...
      /*
        Note by Kai: This branch was introduced as a first
        step to deal with image sets in this situation. It is
        just a first expermint and a heuristic. 
      elif type(S) = Dom::ImageSet then 
        u:= op(S,1);
        l:= ode::normal(subs(l,y=y+u,EvalChanges));
        // letting ynew = y - u(z) 
        // then diff(ynew, z) = diff(y, z) - diff(u, z)
        // thus have to solve dynew/dz = l - du/dz
        if not has(l,z) and not has((i:=diff(u,z)),z) then
          // dynew/dz = l-i
          // we may separate
          s:= ode::separateExplicit(l-i, y, z, 1, solveOptions, odeOptions);
          // for each ynew found, solve ynew = y - u(z) for y:
          // that is, add u(z)
          s:= ode::mapsol(s, _plus, u, solveOptions, odeOptions);
          return(s);
        end_if; // not has(l, z) ...
      */
      end_if; // u nonempty finite set
    end_if; // not has(l, z)
  end_if; // n <= 1
  
  return(FAIL);
end_proc:


