/* 
   ===============================
   METHODS FOR ODES SOLVABLE FOR Y
   ===============================
 
   REFERENCE: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 92, pp. 371.

   DETAILS: 

   ode::solvable_for_y(eq,y,x) tries to solve the first-order ODE eq=0 wrt y(x)
   using the method of "solvable in y" equations.
   There is a typo in the example given by Zwillinger: the equation right after
   (92.5) should be p' = p/x/(1-p^2) instead of p' = px/(1-p^2), but the solution 
   is correct.

   Output: either FAIL or a set of solutions.

   EXAMPLE: 

     >> ode::solvable_for_y(y(x)-x-x*diff(y(x),x),y,x);
     
     >> ode::solvable_for_y(x-y(x)*diff(y(x),x)+x*diff(y(x),x)^2,y,x);
*/

ode::solvable_for_y:= proc(eq,y,x,solveOptions,odeOptions) 
  local p,yy,sol,sol1;
begin 
   userinfo(2,"trying to recognize an equation solvable for y");
   p:= genident("p");
   eq:= subs(eq,diff(y(x),x)=p,y(x)=y);
   // Note, I use MaxDegree only, since otherwise solve needs too much time.
   sol:= solvelib::discreteSolve(eq,y,MaxDegree=1,op(solveOptions)); 
   if not ode::isResultOfSolveOK(sol,"NoRootOf",solveOptions,odeOptions) or sol={0} or nops(sol)<>1
     then return(FAIL);
   end_if;
   userinfo(1,"equation solvable for y");
   yy:= op(sol,1); // yy=f(x,p) 
   sol:= diff(subs(yy,p=p(x)),x)-p(x);
   sysassign(ode::depth,ode::depth+1);
   sol:= ode::solve_eq(sol,p,x,{},solveOptions,odeOptions);
   sysassign(ode::depth,ode::depth-1);
   if type(sol)<>DOM_SET or has(sol,FAIL) or (type(sol)=DOM_SET and hastype(sol,DOM_LIST)) then
     if not (type(sol) = piecewise and map({op(sol)},elem -> type(op(elem,2)))={DOM_SET}) then 
       userinfo(1,"solvable for y method failed");
       return(FAIL);
     end_if;  
   end_if;
   sol:= map(sol, proc()
                  begin 
                    subs(eq,p=args(1)) 
                  end_proc);
   sol1:= map(sol,solvelib::discreteSolve,y,op(solveOptions));
   if sol1={FAIL} then
       sol:= map(sol,RootOf,y);
   else
       sol:= sol1;
   end_if;
   userinfo(1,"solvable for y method worked");
   
   return(map(sol,() -> (if type(args(1))="solve" or domtype(args(1))=RootOf then
                           args(1) 
                         else 
                           op(args(1)) 
                         end_if)));
end_proc:


             