//    

/* main file of the ODE library solver. This solver works as follows:
  the user calls solve(ode(...)) which calls in turn ode::solve(...)
  by the overloading mechanism of solve, and which calls ode::dsolve(...)
  (see below for details).

  The main reference for the methods used in this library is the book
  by Daniel Zwillinger, Handbook of Differential Equations, Academic Press.
*/


// the following (protected) identifier is used to detect if methods are
// called from within the solver. This becomes necessary due to some lacks
// in the algebraic solver. Thus one can avoid implicit results in many cases.
// E.g. ode::separate will often work and is one of the first tried methods,
// but produce mostly implicit results (due to the algebraic solver) which
// in turn yields horrible results of the ode solver.
// sysassign(calledFromWithinDSolve,FALSE): //yet predefined in LIBFILES/stdlib.mu


/* ode::solve(sys,yofz) solves the system (or equation) sys with
  respect to yofz.
  Input:  sys is either a single equation (or expression) or a set of equations
          yofz is either of the form y(z) or a set of such unknown functions
  Output: a (possibly empty) list of solutions. In the case of one unknown
          function, each solution is an expression. In the case of several 
          unknown functions, each solution is a list of assignments for these 
          functions.

  Examples: ode::solve(diff(y(x),x,x)=0,y(x));
            ode::solve({diff(y(x),x,x)=0,y(0)=a,D(y)(0)=b},y(x));
  ode::solve({diff(y1(t),t)=9*y1(t)+2*y2(t),diff(y2(t),t)=y1(t)+8*y2(t)},
              {y1(t),y2(t)});
*/

ode::solve:=
  proc()
    local t: DOM_IDENT, sol, components, solveOptions, odeOptions,
          restargs;
    begin
    /*
      First reset the static variable in 'ode::odeWarning' which stores
      the warnings given by ODE. See file 'ODE/TOOLS/warning.mu' for 
      explanations and details. 
    */      
    ode::odeWarning(Clear);
    components:=ode::getComponents(args());
    /*
    Now we have 'components = eq, y, x, inits, restargs', i.e. we
    have to check the arguments starting with the 5th argument
    and filter out the valid options for 'solve' and those for 
    'ode' itself. 
    */
    restargs:= {};
    solveOptions:= {};
    odeOptions:= {};
    if nops(components) > 4 then
      
      /* OLD version of the code where no additional arguments 
         where allowed. 

         warning("Additional argument(s) ".
              expr2text(op(components,5..nops(components))).
              " are ignored") 
      */
                    
      restargs:= {components[5..nops(components)]};
      restargs:= map(restargs, elem -> if elem <> {} then elem end_if);
              
      // Insert valid options to be used in 'solve' here. 
      /* solveOptions:= map(restargs,indets) intersect 
                     {Real,IgnoreSpecialCases,MaxDegree};*/
         
      // IMPORTANT NOTE by Kai: There is a duplicate of this strategy 
      // in the function 'lodo::solve'. Note that every change 
      // here has to be synchronized with the strategy implemented 
      // in 'Dom::linearDifferentialOperator'. 
      restargs:= map(restargs, 
                     elem -> if type(elem) = "_equal" then 
                               if rhs(elem) = FALSE then 
                                 null()
                               elif rhs(elem) = TRUE then  
                                 lhs(elem)
                               else 
                                 elem
                               end_if
                             else 
                               elem
                             end_if);        
     
      solveOptions:= map(restargs, 
                         elem -> if indets(elem) intersect 
                                      {Real, 
                                       MaxDegree,
                                       IgnoreAnalyticConstraints, 
                                       IgnoreSpecialCases, 
                                       IgnoreProperties} 
                                    <> {}  
                                 then elem end_if);                     
                     
      // Insert valid options to be used in 'ode' here. 
      odeOptions:= restargs intersect { Type = ExactFirstOrder, 
                                        Type = ExactSecondOrder,
                                        Type = Homogeneous,
                                        Type = Riccati,
                                        Type = Lagrange,
                                        Type = Bernoulli,
                                        Type = Abel,
                                        Type = Chini,
                                        Type = Clairaut,
                                        Type = IntegratingFactor};
                                          
      restargs:= restargs minus (solveOptions union odeOptions); 
      if restargs <> {} and restargs <> {"ConsiderAllHopelessCases"} then 
        ode::odeWarning("Additional argument(s) ".
              expr2text(op(restargs)).
              " are ignored")
      end_if;
      components:= components[1..4];
    end_if;
   
    if restargs = {} and 
       ode::hopelessCases(components[1],components[2],components[3],solveOptions,odeOptions) then 
      return(hold(solve)(args()));
    end_if;
    
    // handle properties of independent variable
    t:=components[3];
    assert(domtype(t) = DOM_IDENT);
    save t; 
   
    assume(t in R_, _and);
     
    sol:=ode::solve_intern(components, solveOptions, odeOptions);

    if has(sol,FAIL) then
      if odeOptions = {} then 
        hold(solve)(args());
      else 
        return(FAIL);
      end_if;
    else
      sol
    end_if
  end_proc:


ode::solve_intern :=
  proc(sys)
    save calledFromWithinDSolve;
    local depVars,indepVar,inits,components,solveOptions,odeOptions,
          L, ODEsys, R, eq, f, i, ind, l, n, x, z, Sol, ListFunction, 
          ListPower, Res, deletedODEs, systemComplexity, tmp;
  begin
    /* 
     Note by Kai: I made a grep and 'ode::solve' seems to 
     be the only function in the whole ODE library which
     calls 'ode::solve_intern'. Since 'ode::solve' cares 
     for an appropriate treatment of additional options 
     for 'solve' and 'ode', we can assume here that 
     the 
         'args(args(0)-1) = solveOptions'  
     and 
           'args(args(0)) = odeOptions'.
     No further additional invalid arguments can be 
     contained in 'args()'. 
     */
    if domtype(sys)=ode then
      components:=ode::getComponents(args());
      sys:=components[1];
      depVars:=components[2];
      indepVar:=components[3];
      inits:=components[4];
    else
      depVars:=args(2);
      indepVar:=args(3);
      inits:=args(4);
    end_if;
    
    solveOptions:= args(args(0)-1);
    odeOptions:= args(args(0));
    
    // set flag for calling methods within dsolve
    // sysassign(calledFromWithinDSolve,TRUE);
  
    // depth in routines of ODE, to avoid infinite loops 
    sysassign(ode::depth,0);

    if type(sys) = DOM_SET then

        ODEsys:= sys;
        l:= depVars;
        z:= indepVar;
        tmp:= l;     
        for i from 1 to nops(l) do 
          ind:= l[i];
          if not has(ODEsys,ind) then 
            warning("unknown functions not appearing in the system are ignored");
            tmp:= tmp minus {ind};
          end_if;  
        end_for;  
        l:= tmp;
        
        components:= FAIL;
        
        if nops(ODEsys)<>nops(l) then
          // Try to treat some easy cases of non-square ODE systems 
          // consisting of more ODEs than dependent variables. 
          deletedODEs:= {};
          if nops(ODEsys) > nops(l) then 
            warning("more equations than unknowns: trying heuristics to reduce to square system");
            // collected deleted ODEs for consistency check below 
            n:= ode::order(ODEsys,l,{},{});
            x:= genident();
            for i from 1 to nops(l) do 
              L:= subs(ODEsys, diff(l[i](z),z$n) = x);
              for i from 1 to nops(L) do 
                // We try to reduce the number of equations. Therefore we use
                // the following strategy: we select an ODE from 'ODEsys' which 
                // contains an n-th order derivative of an unknown function.
                // If this derivative appears in "easy" form, then we solve 
                // the equation for this derivative. Then we insert it into 
                // the remaining equations and thus have reduced the number
                // of equations by 1. We try to apply this strategy successively 
                // unitl the number of ODEs equals the number of independent 
                // variables. 
                eq:= L[i]; 
                if has(eq,x) /*and not has(diff(eq,x),x)*/ then 
                  if has(L minus {eq},x) then 
                    // There must be at least 1 more equation containing 'x' 
                    // such that we can insert the result obtained from 
                    // isolating 'x' from 'eq' into the system. Otherwise 
                    // we would simply leave out an equation which will 
                    // lead to inconsistent solutions. 
                    L:= L minus {eq};
                    deletedODEs:= deletedODEs union {subs(eq, x = diff(l[i](z),z$n))};
                    break;
                  end_if;  
                end_if;
                // No equation for heuristic elimination strategy could be 
                // found. Assign 'FAIL' to the variable 'eq'. 
                eq:= FAIL;
              end_for;
              if eq <> FAIL then 
                // Solve equation 'eq' for identifier 'x' and substitute it 'x'  
                // by the result in 'L'.
                R:= solve(eq,x,IgnoreSpecialCases);
                // Treat only return values that are sufficiently easy to be 
                // processed with reasonable effort. 
                if type(R) = DOM_SET then 
                  L:= subs(L, x = R[1]);
                  if nops(L) = nops(l) then 
                    if nops(L) = 1 then 
                      // This means we have reduced the system to a single 
                      // ODE. Here the first two elements in the sequence
                      // components need to be an expression (the ODE) and 
                      // an identifier (the dependent ODE variable).
                      components:= op(L), op(l), z, inits;
                    else   
                      // If 'nops(L) > 1' we still have a system of ODEs. 
                      // Set of the components in the way 'ode::solve_sys' 
                      // needs it. 
                      components:= L, l, z, inits;
                    end_if;  
                    break;
                  end_if; 
                end_if;  
              end_if;  
            end_for; 
            // Heuristics were not successful. Raise an error. 
            if components = FAIL then 
              error("Cannot reduce to square system: number of equations differs from number of unknowns");
            end_if;  
          elif nops(ODEsys) < nops(l) then 
            // In case that there are more independent variables than ODEs, we remove 
            // a subset from set S of independent variables until then number of 
            // independent variables equals the number of ODEs. Then we give the solution 
            // of the ODE system in parametrized form treating the elements of S as 
            // generic functions (and not a solution variables). 
            warning("more unknowns than equations: trying to parametrize solutions in terms unknowns");
            ListPower:= {}:
            ListFunction:= {}:
            /*-----------------------------------------------------------------
               Local procedure to identify which of the dependent variables 
               appear linear, which appear in terms of powers and which as 
               arguments of functions. The lists 'ListPower' and 'ListFunction' 
               serve as global containers. 
                
                   ARGUMENTS:   system -- set of expressions encoding the ODE 
                                          system 
                              indepVar -- an independent (from the ODE system)   

               RETURN VALUES: Modifies the sets 'ListPower' and 'ListFunction'
                              which serve as global containers to identify 
                              whether an independent variable (or any of its
                              derivatives) has type '_power' or 'function'.  
            -----------------------------------------------------------------*/
            systemComplexity:= proc(system,indepVar)
              local eq, sysIndepVar;
            begin
              sysIndepVar:= select(system,has,indepVar);
              for eq in sysIndepVar do 
                case type(eq) 
                  of "_plus" do 
                  of "_mult" do 
                    return(systemComplexity({op(eq)},indepVar))
                  of "_power" do 
                    ListPower:= ListPower union {indepVar};
                    break;
                  of "function" do
                    if op(eq,0) <> indepVar then 
                      ListFunction:= ListFunction union {indepVar};
                      break;
                    end_if;
                end_case;
              end_for;
              return(null())
            end_proc:
            //-----------------------------------------------------------------
            for i from 1 to nops(l) do 
              systemComplexity(ODEsys,l[i]);
            end_for;  
            while nops(ODEsys) < nops(l) and ListFunction <> {} do 
              f:= ListFunction[1];
              l:= l minus {f};
              ListFunction:= ListFunction minus {f};
            end_while;
            while nops(ODEsys) < nops(l) and ListPower <> {} do 
              f:= ListPower[1];
              l:= l minus {f};
              ListPower:= ListPower minus {f};
            end_while;
            while nops(ODEsys) < nops(l) do 
              f:= l[1];
              l:= l minus {f};
            end_while;
            if nops(ODEsys) = 1 then 
              components:= op(ODEsys),op(l),z,inits;
            else 
              components:= ODEsys,l,z,inits;
            end_if;  
          end_if;  
        end_if;
      if components = FAIL then
        return(ode::solve_sys(sys,depVars,indepVar,inits,solveOptions,odeOptions));
      else 
        if type(components[1]) = DOM_SET then 
          Sol:= ode::solve_sys(components,solveOptions,odeOptions);
          Res:= Sol;
          if Res <> FAIL and Res <> {} then 
            // Check if the solution of the ODE also solves the equations 
            // in 'deletedODEs'. Remove inconsistent solutions. 
            for eq in deletedODEs do 
              for i from 1 to nops(Sol) do 
                if traperror((tmp:= eq | Sol[i])) <> 0 then 
                  Res:= Res minus {Sol[i]};
                elif testeq(tmp,0,NumberOfRandomTests = 5,Steps = 10) <> TRUE then                  
                  Res:= Res minus {Sol[i]};
                end_if;  
              end_for;  
            end_for;  
            return(Res);
          end_if;  
        else 
          // System has been reduced to a single ODE.
          Sol:= ode::solve_eq(components,solveOptions,odeOptions);
          Res:= Sol;
          // Check if the solution of the ODE also solves the equations 
          // in 'deletedODEs'. Remove inconsistent solutions. 
          for eq in deletedODEs do 
            for i from 1 to nops(Sol) do 
              if traperror((tmp:= eq | components[2](components[3])=Sol[i])) <> 0 then 
                Res:= Res minus {Sol[i]};
              elif testeq(tmp,0,NumberOfRandomTests = 5,Steps = 10) <> TRUE then                  
                Res:= Res minus {Sol[i]};
              end_if;  
            end_for;  
          end_for;  
        end_if;  
        return(Res);
      end_if;  
    else
      return(ode::solve_eq(sys,depVars,indepVar,inits,solveOptions,odeOptions));
    end_if;
  end_proc:
  
  
ode::isResultOfSolveOK:=
  proc(r,o=" ")  // option: "AllowImplicit", "NoRootOf"
    local solveOptions, odeOptions;
  begin
    if args(0) = 3 or args(0) = 4 then 
      solveOptions:= args(args(0)-1);
      odeOptions:= args(args(0));
    end_if;
    if type(r)=DOM_SET then
      if o="NoRootOf" and hastype(r,RootOf) then return(FALSE)
      else return(TRUE) end_if;
    end_if;
//    if type(r)="solve" and o="AllowImplicit" then return(TRUE) end_if;
    if (type(r)="solve" or domtype(r)=RootOf) and o="AllowImplicit" then
      return(TRUE)
    end_if;
    if type(r) = "_union" then
      if args(0) = 3 or args(0) = 4 then
        return(_and(map(op(r), ode::isResultOfSolveOK, o, 
                        solveOptions, odeOptions)))
      else 
        return(_and(map(op(r), ode::isResultOfSolveOK, o)))
      end_if;
    end_if;
    FALSE
  end_proc:
    

// end of file 
