/*
  ode::solve_sys(sys, l, z, inits)

    sys - a set of expressions
    l   - a set of identifiers {x1, x2, ...}
    z   - variable
    inits - set of initial conditions

  Returns a set of lists {[x1(z) = ..., x2(z) = ...]}.
*/

ode::solve_sys:= proc(sys:DOM_SET,_l:DOM_SET,z,inits,solveOptions,odeOptions) // unks is a set
  local is_linear,eq,maxord,s,i,j,A,ll,_b,m,k,sol,n,indet,unks,optIgnoreAnalyticConstraints,
        initialConditions:DOM_PROC, appendSol:DOM_PROC;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;              
  // -------------------------- LOCAL PROCEDURES -------------------------             
  // handle initial conditions
  // because of the inconsistent return value formats of the
  // particular solvers called, sol may be a set of lists
  // {[x(t) =...., y(t)=...,. ...],...}
  // or a set of sets
  // {{x(t) = ..., y(t) = ..., ...}, ...}
  initialConditions:=
  proc(sol: Type::Set, inits: DOM_SET, indet, _l, z, solveOptions, odeOptions)
  begin
    // convert to lists
    if type(sol) = piecewise then
      return(piecewise::extmap(sol,initialConditions,inits,indet,_l,z,solveOptions,odeOptions));
    end_if;
    assert(type(sol)=DOM_SET);
    sol:= map(sol, x -> if type(x)=DOM_LIST then x else [op(x)] end_if);
    _l:= {op(_l)};
    if inits = {} then
      return(sol);
    end_if;
    sol:= map(sol,ode::solvesysinit,indet,inits,_l,z,solveOptions,odeOptions);
    return(_union(op(sol)));
  end_proc;
  // --------------------------------------------------------------------             
  // appendSol
  // for each element s in sol, solve (sys| x=s) for unk and append the
  // coordinate x=s to each solution vector 
  // --------------------------------------------------------------------             
  appendSol:=
  proc(sys, unk, x, sol, solveOptions, odeOptions)
    local l, s, i, appends: DOM_PROC;
  begin
    // append x=s to each element of S
    appends:=
    proc(S,s)
      local T;
    begin
      case type(S)
        of DOM_SET do
          return(map(S, append,x=s))
        of piecewise do  
          T:= piecewise::extmap(S,appends,s);
          if has(T,FAIL) then
            return(FAIL);
          else
            return(T);
          end_if
      end_case;
      return(FAIL);
    end_proc;
    
    case type(sol)
      of DOM_SET do
        unk:= map(unk, op, 0);
        l:= [appends(ode::solve_sys(subs(sys, x=op(sol, i), EvalChanges), unk, z, {},
                                    solveOptions, odeOptions),
                                    op(sol, i))
             $i=1..nops(sol)];
        if contains(l, FAIL) > 0 then
          return(FAIL);
        else
          return(_union(op(l)));
        end_if
      of "_union" do
        l:= map([op(sol)], s-> appendSol(sys, unk, x, s, solveOptions, odeOptions));
        if contains(l, FAIL) > 0 then
          return(FAIL);
        else
          return(_union(op(l)));
        end_if
      of piecewise do
        l:= piecewise::extmap(sol, s-> appendSol(sys, unk, x, s, solveOptions, odeOptions));
        if has(l, FAIL) then
          return(FAIL);
        else
          return(l);
        end_if
    end_case;
    return(FAIL);
  end_proc;
  // ----------------------- END OF LOCAL PROCEDURES ----------------------             

  // ------------------------------- M A I N ------------------------------
  
  // Get a list of the indeterminates from the original system
  // this will be needed to handle initial conditions
  indet:= {};
  for eq in sys union inits do
    indet:= indet union indets(eq);
  end_for;

  n:= nops(_l); // number of unknown functions
  unks:= map(_l,e->e(z)); // set of unknown functions

  // check the equations and compute the maximal order
  maxord:= ode::order(sys,_l,solveOptions,odeOptions);
  userinfo(1,"differential system of order ".maxord." with ".n." functions");
  // Treat the case of 'maxord = 0' first, i.e. 'sys' is a non-differential
  // system that can be tackled by the generic solver. 
  if maxord = 0 then 
    s:= solve(sys,unks,IgnoreSpecialCases,op(solveOptions));
    if type(s) <> DOM_SET then 
      if type(s) = piecewise then 
        s:= piecewise::disregardPoints(s);
        if type(s) <> DOM_SET then 
          s:= {};
        end_if;
      end_if; 
    end_if;
    return(s);
  end_if;
  // ------------------------------------------------------------------------
  // recognize linear systems
  // ------------------------------------------------------------------------
  _l:= [op(_l)]; // to set the order
  s:= [(diff(_l[i](z),z$j) $ i=1..n) $ j=0..maxord];
  m:= n*(maxord+1);
  is_linear:= TRUE;
  A:= table();
  _b:= [];
  for k from 0 to maxord do
    A[k]:= [[0$n]$n]
  end_for;
  for i from 1 to n do
    if (ll:= Type::Linear(expand(op(sys,i), optIgnoreAnalyticConstraints),s)) <> FALSE then
      _b:= append(_b,ll[m+1]);
      for k from 0 to maxord do
        for j from 1 to n do
          A[k][i][j]:= ll[k*n+j];
        end_for;
      end_for;
    else
      is_linear:=FALSE;
      break;
    end_if
  end_for;
  sol:= FAIL;
  if is_linear and nops(A) > 0 then
    sol:= ode::linsys(A,_b,_l,z,n,maxord,n,solveOptions,odeOptions);
  else
    userinfo(1,"nonlinear system");
    if maxord = 1 then
      sol:= ode::combination(sys,unks,z,solveOptions,odeOptions);
      if sol = FAIL then
        sol:= ode::matrixRiccati(sys,unks,z,solveOptions,odeOptions);
      end_if;
    end_if;
  end_if;
  if sol <> FAIL then
    return(initialConditions(sol,inits,indet,_l,z,solveOptions,odeOptions));
  end_if;
  // ------------------------------------------------------------------------
  // try to recognize a triangular system
  // ------------------------------------------------------------------------
  for eq in sys do
    s:= misc::subExpressions(eq,"function") intersect unks;
    if nops(s) = 1 then
      userinfo(1,"found one independent equation");
      s:= op(s); // x(t)
      sol:= ode::solve_eq(eq,op(s,0),op(s),{},solveOptions,odeOptions);
      if has(sol, FAIL) or not contains({DOM_SET,piecewise},domtype(sol)) then
        // cannot handle more general results of equation solver yet
        next;
      end_if;
      sys:= sys minus {eq};
      unks:= unks minus {s};
      if unks = {} then
        if sys = {} and type(sol) = DOM_SET then
          return({[s=op(sol,i)] $ i=1..nops(sol)});
        else
          return(FAIL);
        end_if;
      end_if;
      sol:= appendSol(sys,unks,s,sol,solveOptions,odeOptions);
      if sol = FAIL then
        return(FAIL);
      else
        return(initialConditions(sol,inits,indet,_l,z,solveOptions,odeOptions));
      end_if;
      /*
      map(sol,
        proc(solution, sys, unks, s)
          begin
            solution:= {s = solution};
            if nops(unks) = 1 then
              s:=op(unks);
              sys:=ode::solve_eq(op(sys),op(s,0),op(s));
              if has(sys, FAIL) then
                return(FAIL)
              end_if;
              
          sys:= ode::mapsol(sys,
                            proc(sol)
                            begin
                              {s = sol}
                            end_proc
                            );
          sys:= map(sys,
                    proc()
                    begin
                      if type(args(1))=DOM_SET then
                        args(1)
                      else
                        {args(1)}
                      end_if
                    end_proc
                    );
              
          else
          sys:=ode::solve_sys(ode::getComponents(sys,unks));
          assert(testtype(sys, Type::SetOf(Type::ListOf("_equal"))))
          end_if; // nops(unks) = 1
          if type(sys)=DOM_SET then
            return(initialConditions(map(sys,
                                         proc(e)
                                         begin
                                           if domtype(e)=DOM_LIST then
                                             e.[op(solution)]
                                           else
                                             [op(e), op(solution)]
                                           end_if
                                         end_proc),
                                     inits,indet,_l,z))
          else
          return(FAIL)
          end_if
          end_proc,
          sys, unks, s);
      if contains(%, FAIL) then
        return(FAIL)
      else
        return(_union(op(%)))
      end_if
      */
    end_if;
  end_for;

  /*
  if has(sol, FAIL) then
    FAIL
  else
    initialConditions({sol},inits,indet,_l,z)
  end_if;
  */
  
  return(FAIL);
end_proc:


/*******************************************
ode::solvesysinit - initial condition solver

arguments:
sol - solution of a system of diff. eq. (list Of equations)
indet - set of free parameters
inits - set of initial conditions
fns   - set of identifiers representing the functions to solve for
z     - the variable on which the functions in fns depend

solveOptions - options for calling 'solve'
odeOptions   - options for calling 'ode'

********************************************/
ode::solvesysinit:=
proc(sol:Type::ListOf("_equal"), indet:DOM_SET,
     inits:Type::SetOf("_equal"), fns:DOM_SET, z:DOM_IDENT,
     solveOptions, odeOptions)
  local csts: DOM_SET, // set of constants
        eq,s,Lhs,a,i,csteq,aux,e,t,spsol,soln,optIgnoreAnalyticConstraints,f,newIndets;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  // replace indefinite integrals by definite ones
  // (this is our only chance to plug in values)
  sol:= misc::maprec(sol, {"int"} =
                     proc(J: "int")
                       local X;
                     begin
                       if type(op(J, 2)) <> "_equal" then
                         // write int(f(x), x) as
                         // int(f(X), X=0..x)
                         X:= solvelib::getIdent(R_, indets(J));
                         subsop(J, 
                                1 = subs(op(J,1), op(J, 2) = X),
                                // use one of the initial values as 
                                // starting point for integration 
                                2 = (X = op(lhs(inits[1]),1)..op(J, 2))
                                )
                       else
                         J
                       end_if
                     end_proc
                     );

  // Find the constants
  csts:= indets(sol) minus (indet union fns union {z});
  userinfo(3,"general solution is",sol);
  // here solve_eq.mu has a test to see if the solution has type
  // RootOf
  userinfo(3,"initial conditions are",inits);
  // Now put the initial conditions into a more useful form.
  // The result of the following steps will be [f,i,a,b], which then 
  // means (D@@i)(f)(a)=b.
  s:= {}; 
  for eq in inits do
    if has(op(eq,2),fns) then
      //error("dependent variable not allowed in rhs of init conds");
      s:= {}; 
      break;
    end_if;
    Lhs:= op(eq,1);
    i:= 0;
    a:= op(Lhs);
    Lhs:= op(Lhs,0); // This should be f or D(...(D(f))...)
    case type(Lhs)
      of DOM_IDENT do
        break;
      of "D" do
        repeat
          Lhs:= op(Lhs);
          i:= i+1;
        until type(Lhs) <> "D"
        end_repeat;
        break;
      otherwise
        error("invalid initial conditions");
    end_case;
    if not(contains(fns,Lhs)) then
      error("invalid initial conditions");
    end_if;
    s:= s union {[Lhs,i,a,op(eq,2)]};
  	// [f,i,a,b] means (D@@i)(f)(a)=b
	  // solve_eq has a test here to make sure that the denominator
	  // of eq does not vanish
  end_for;
  // solve_eq.mu has the next part in "traperror".
  // It will take the initial conditions and create a system of 
  // equations which can be solved for the constants
  csteq:= {};
 
  for t in s do
    if has(sol,t[1]) then
      // get the right hand side of t[1](z) = ....
      eq:= op(select(sol,has,t[1]),[1,2]); 
    end_if;
    // The substitution "int=fun(0)" is an OC one. To be observed!
    csteq := csteq union {subs(diff(eq,z$t[2]),
                                    // hold(int)=0, !! K.F.G.: causes problems with intial values !!
                               z=t[3],EvalChanges) = t[4]};
  end_for;
 
  if nops(csts) = 0 then
    return(piecewise([_and(op(csteq)),{sol}],[Otherwise,{}]));
  end_if;

  if csteq <> {} then 
    // Now solve the equations for the constants
    if map(csteq,
           proc()
             begin
               nops(_intersect(indets(args()) minus Type::ConstantIdents, csts))
           end_proc
           )
      = {1}
      // solve the equations independently because solvelib::discreteSolve may introduce
      // some output of type "_in" for instance.
      then
      aux:= [];
      for e in csteq do
        aux:= append(aux,op(op(solvelib::discreteSolve(e,csts intersect indets(e), 
                                                       op(solveOptions)))))
      end_for:
      csteq:= {aux};
    else
      csteq:= solvelib::discreteSolve(csteq,csts,op(solveOptions));
    end_if:
  end_if;

  if csteq = {} then 
    aux:= evalAt(inits, (f = fp::unapply(rhs(select(sol,has,f)[1]),z)) $ f in fns);
    // The next line can be removed when the solver is be called with option
    // 'IgnoreAnalyticConstraints'.
    aux:= map(aux,elem->numer(op(elem,1))*denom(op(elem,2)) = 
                        numer(op(elem,2))*denom(op(elem,1)));
    csteq:= solve(aux,csts,op(solveOptions));
  end_if;

  // Note that before the introduction of the three following lines no 
  // solution was returned when 'csteq' was a piecewise. Now at least
  // in the case when 'disregardPoints' applied to 'csteq' gives a 
  // a set of solution, a result will be obtained. 

  if type(csteq) = piecewise then 
    csteq:= piecewise::disregardPoints(csteq);
  end_if;

  csteq := select(csteq,proc()
                        begin
                          contains({DOM_LIST,DOM_SET},
                                   type(args(1)))
                        end_proc );
  
  // What to do if the system of equations for the constants cannot
  // be solved?  Return the general solution with the initial
  // conditions, I suppose
  
  if type(csteq) <> DOM_SET then
    //if ode::printWarningsFlag then
    //  ode::odeWarning("Initial conditions cannot be solved")
    //end_if;
    //return({sol.[op(inits)]})
    // Note by Kai: I do not think it is a good idea to return 'sol.[op(inits)]' when 
    //              the initial problem could not be solved. It is more consistent 
    //              with the philosophy of the ODE solver to return '{}' here. 
    return({})
  elif nops(csteq) = 0 then
    // If 'nops(csteq) = 0', then 'csteq = {}' and this means that the initial 
    // conditions CANNOT be fulfilled. 
    return({});
  else
    newIndets:= freeIndets(csteq) minus (fns union freeIndets(sol) union freeIndets(inits));
    sol:= {subs(sol,op(csteq,i),EvalChanges) $ i = 1..nops(csteq)};
    sol:= subs(sol,newIndets[i]=genident("C") $ i = 1..nops(newIndets));
    spsol := {};
    for soln in sol do
     // soln := eval(subs(soln,op(csteq,i)) $ i = 1..nops(csteq));
      if has(soln,{sin,cos}) then
        // Expanding expressions that contain unresolved integrals 
        // leads to re-evaluation of the integrals that will most 
        // probably not produce a more explicit form the results, 
        // but consume a lot of time instead. Hence, replace integrals
        // by identifiers, do the expansion and afterwards do a back 
        // substitution. 
        if hastype(soln,"int") then 
          soln:= maprat(soln,x -> combine(expand(x,optIgnoreAnalyticConstraints),
                                          sincos,optIgnoreAnalyticConstraints),
                        DescendInto=TRUE,ReplaceHardToEval);
        else  
          soln:= combine(expand(soln, optIgnoreAnalyticConstraints),sincos, optIgnoreAnalyticConstraints);
        end_if;  
      end_if;
      spsol := spsol union {soln}
    end_for;

    return(spsol)
  end_if;

end_proc: /* ode::solvesysinit */

