/* 
   ===============================================
   METHOD FOR ODES WITH MISSING DEPENDENT VARIABLE
   ===============================================
 
   REFERENCE: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 55, pp. 237.

   DETAILS: 
 
     Input: a n-th order ODE in y(x).
     Output: a solution or FAIL.
     Uses the dependent variable missing method. 

   EXAMPLES: 
     >> ode::ymissing(diff(y(x),x,x)+diff(y(x),x)-x,y,x,2);


     From Murphy, G.M.: "Ordinary Differential Equations and their solutions". 
                        Example 306 (non-linear second order), p. 409.

     >> eq:=diff(y(x),x,x)^3-12*diff(y(x),x)*(x*diff(y(x),x,x)-2*diff(y(x),x)):
     >> ode::ymissing(eq,y,x,2);
     
     {                4 }               /      /
     {            19 x  }              |       |
     { C10, C10 + ----- } union C10 +  |  solve|
     {              54  }             /        \
     
          /
         |        3          3  9       3  2       3          6
         |  1 / (x  RootOf(u4  x  - 38 x  y  + 27 y  - 12 u4 x  y +
        /
     
                                                          \
               3  2       2  6                            |
        27 u4 x  y  + 9 u4  x  y, u4)) dy - ln(x) - C9, y | dx
                                                          /
     
     REMARK: the above general solution could be simplified to C1*(x-C1)^3+C2
*/

ode::ymissing:= proc(eq,y,x,n,solveOptions,odeOptions) 
  local i,sol,C,lord,intOptions;
begin 
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  eq:= subs(eq,diff(y(x),x$i)=`#yp`[i] $ i=1..n);
  if has(eq,y(x)) then
    return(FAIL);
  else
    userinfo(1,"using dependent variable missing method");
    lord:= 1; 
    while not has(eq,`#yp`[lord]) do 
      lord:= lord+1;
    end_while;
    // lord is the lowest derivative order in y(x)
    eq:= subs(eq,`#yp`[i]=diff(y(x),x$(i-lord)) $ i=lord..n,EvalChanges);
    userinfo(2,"solving new equation",eq);
    sysassign(ode::depth,ode::depth+1);
    sol:= ode::solve_eq(eq,y,x,{},solveOptions,odeOptions);
    sysassign(ode::depth,ode::depth-1);
    if has(sol,FAIL) or has(sol,hold(solve)) then
      return(FAIL);
    end_if;
    for i from 1 to lord do
      C:= genident("C");
      sol:= ode::mapsol(sol,t->int(t,x,intOptions)+C,solveOptions,odeOptions);
      //sol:= eval(sol);
    end_for;
    userinfo(1,"dependent variable missing method worked");
    return(sol);
  end_if;
  
end_proc:

