//      
/*++ chebyshev1(n, x) - generates the nth Chebyshev polynomial (first kind)

T(0,x) = 1
T(1,x) = x
T(n,x) = 2 * x * T(n-1,x) - T(n-2,x)

Explicit expressions are

 T(n, x) = cos(n*arccos(x))

and

 T(n, x) = (  1/((x - 1)^(1/2) - (x + 1)^(1/2))^(2*n) 
            + 1/((x - 1)^(1/2) + (x + 1)^(1/2))^(2*n)
           ) * 2^(n-1)

The implementation uses chebyshev1_rec(n, x) (recursively n -> n/2)

--*/

orthpoly::chebyshev1:= proc(n, x)
local k, tmp, result;
begin
  if args(0) <> 2 then error("wrong number of arguments"); end_if;
  if not (domtype(n) = DOM_INT) then
     return(procname(args()));
  end_if;
  if n < 0 then error("1st argument must not be negative"); end_if;
  if domtype(x) = DOM_FLOAT and specfunc::abs(x)<=1
     then if n=0 then return(float(1)); end_if;
          if iszero(x) 
          then if n mod 2 = 0
                 then return(float((-1)^(n/2)))
                 else return(float(0))
               end_if;
          else return(cos(n*arccos(x)));
          end_if;
  end_if;

  if not testtype(x, Type::Arithmetical) then
     error("illegal 2nd argument");
  end_if;
  if domtype(x) = DOM_IDENT or type(x)="_index"
  then /* Compute the Chebyshev polynomial by an iteration:
          Wolfram Koepf, "Efficient Computation of Orthogonal Polynomials 
          in Computer Algebra", Preprint SC 95-42, Dec. 1995, ZIB, page 10.
          This is adequate, if the output is to be of expanded form
          (as done by poly(..))
          Use explicit formula T[n](x) = sum(c[n,k]*x^k,k=0..n) with
          a simple recursion c[n,k-1]=-known_factor(n,k)*c[n,k].
       */
       case n
       of 0 do return(poly(1,[x]));
       of 1 do return(poly(x,[x]));
       otherwise result:= [ 0 $ 1 + (n div 2)];
                 tmp:= 2^(n-1);  // coefficient of leading term
                 result[1]:= [tmp, n]; // store coefficient in result list
                 for k from 1 to n div 2 do
                    // next lower coefficient can be computed from last coefficient
                    tmp:= -tmp*((1/4/k)*((n - 2*k) + 2)*((n - 2*k) + 1))/(n - k);
                    // store in result list
                    result[k+1]:= [tmp,n-2*k];
                 end_for;
                 // build poly from coefficients stored in result
                 return( poly(result,[x]) );
       end_case;
  else 
       if has({DOM_INT, DOM_RAT, DOM_FLOAT, DOM_COMPLEX}, domtype(x)) then
            orthpoly::chebyshev1_rec(n,x);
       else orthpoly::chebyshev1_rec_expand(n,x);
       end_if:
  end_if;
end_proc:

/*--
Computing the Chebyshev polynomial, represented in a not distributed way,
by the recursion 2*T[n](x)*T[m](x) = T[n+m](x)+T[n-m](x) (n>m), i.e.,
T[n+m] = 2*T[n](x)*T[m](x) - T[n-m](x).

With m = n:   T[2*n]   = 2*T[n](x)^2 - T[0](x).
With m = n-1: T[2*n-1] = 2*T[n](x)*T[n-1](x) - T[1](x).

See Wolfram Koepf, "Efficient Computation of Orthogonal Polynomials in 
Computer Algebra", Preprint SC 95-42, Dec. 1995, ZIB, page 11
This is appropriate, if a non distributed representation 
(x-a1)((x-a2)^2 + ..) is ok, otherwise final expansion is
too expensive.
--*/

orthpoly::chebyshev1_rec:= proc(n, x)
option remember;
begin
   if n=0 then return(1) end_if;  
   if n=1 then return(x) end_if;  
   if n mod 2 = 0 
   then 2*orthpoly::chebyshev1_rec(n/2, x)^2 - 1;
   else 2*orthpoly::chebyshev1_rec((n-1)/2,x)*
          orthpoly::chebyshev1_rec((n+1)/2,x) - x;
   end_if
end_proc:

orthpoly::chebyshev1_rec_expand:= proc(n, x)
option remember;
begin
   if n=0 then return(1) end_if;  
   if n=1 then return(expand(x)) end_if;  
   if n mod 2 = 0 
   then expand(2*orthpoly::chebyshev1_rec_expand(n/2, x)^2 - 1);
   else expand(2*orthpoly::chebyshev1_rec_expand((n-1)/2,x)*
               orthpoly::chebyshev1_rec_expand((n+1)/2,x) - x);
   end_if
end_proc:

/* ---- end of file ---- */
