//      

/*++
chebyshev2(n, x) - generates the n-th Chebyshev polynomial of the second kind

U(0,x) = 1
U(1,x) = 2*x
U(n,x) = 2*x*U(n-1,x) - U(n-2,x)
--*/

orthpoly::chebyshev2:= proc(n, x)
local y;
begin
  if args(0) <> 2 then error("wrong number of arguments"); end_if;
  if not (domtype(n) = DOM_INT) then
     return(procname(args()));
  end_if;
  if n < 0 then error("1st argument must not be negative"); end_if;
  if domtype(x) = DOM_FLOAT and specfunc::abs(x)<=1
     then if n=0 then return(float(1)); end_if;
          if iszero(x) 
          then if n mod 2 = 0
                 then return((-1)^(n/2)*1.0)
                 else return(float(0))
               end_if;
          end_if;
          if iszero(x-1) then 
              return(float(n+1))
          end_if;
          if iszero(x+1) then 
              return(float((-1)^n*(n+1)))
          end_if;
          // now x<> -1, 0, 1
          y:= arccos(x);
          return(sin((n+1)*y)/sin(y));
  end_if;
  if not testtype(x, Type::Arithmetical) then
     error("illegal 2nd argument");
  end_if;
  if domtype(x) = DOM_IDENT or type(x)="_index"
  then case n
       of 0 do return(poly( 1 ,[x]));
       of 1 do return(poly(2*x,[x]));
       otherwise // You may use standard recursion:
                 // r0 := poly( 1 ,[x]);
                 // r1 := poly(2*x,[x]);
                 // for i from 1 to n-1 do
                 //   [r1, r0]:=[poly(2*x,[x])*r1 - r0, r1]
                 // end_for;
                 // return(r1);

                 // However, with (1-x^2)*U[n](x) = x*T[n](x) - T[n+2](x)
                 // we can make use of the very fast T[n](x)
                 return(
                 divide(poly(x, [x])*orthpoly::chebyshev1(n+1,x) -
                        orthpoly::chebyshev1(n+2,x),
                        poly(1-x^2, [x]), Quo)
                       );
       end_case;
  else if has({DOM_INT, DOM_RAT, DOM_FLOAT, DOM_COMPLEX}, domtype(x)) then
            orthpoly::chebyshev2_rec(n,x);
       else orthpoly::chebyshev2_rec_expand(n,x);
       end_if:
  end_if;
end_proc:

// Recursive call for U=chebychev2, T=chebychev1:
// use U[2*n+1](x) = 2*U[n](x)*T[n+1](x)
//     U[2*n](x)   = 2*U[n](x)*T[n](x) -1
// This is appropriate for a not expanded form such as
// (x-a1)*((x-a2)*..) + ..

orthpoly::chebyshev2_rec:= proc(n, x)
begin
   if n=0 then return(1) end_if;
   if n=1 then return(2*x) end_if;
   if n mod 2 = 0
   then 2*orthpoly::chebyshev2_rec(n/2, x)*
          orthpoly::chebyshev1(n/2, x) - 1;
   else 2*orthpoly::chebyshev2_rec((n-1)/2,x)*
          orthpoly::chebyshev1((n+1)/2,x);
   end_if
end_proc:

orthpoly::chebyshev2_rec_expand:= proc(n, x)
begin
   if n=0 then return(1) end_if;
   if n=1 then return(expand(2*x)) end_if;
   if n mod 2 = 0
   then expand(2*orthpoly::chebyshev2_rec_expand(n/2, x)*
               orthpoly::chebyshev1(n/2, x) - 1);
   else expand(2*orthpoly::chebyshev2_rec_expand((n-1)/2,x)*
               orthpoly::chebyshev1((n+1)/2,x));
   end_if
end_proc:

/* --- end of file --- */
