//      

/*++
gegenbauer(n, a, x) - generates the nth Gegenbauer polynomial

G(0,a, x) = 1
G(1,a, x) = 2*a*x
G(n,a, x) = 2*(n+a-1)/n*x*G(n-1,a ,x) - (n+2*a-2)/n*G(n-2, a,x)
--*/

orthpoly::gegenbauer := proc(n, a, x)
local i, expand_coeffs, r0, r1;
begin
  if args(0) <> 3 then error("wrong number of arguments"); end_if;
  if not (domtype(n) = DOM_INT) then
     return(procname(args()));
  end_if;
  if n < 0 then error("1st argument must not be negative"); end_if;
  if not testtype(a, Type::Arithmetical) then
     error("illegal 2nd argument");
  end_if;
  if has({DOM_FLOAT, DOM_COMPLEX, DOM_INT,DOM_RAT}, domtype(a)) and
     has({DOM_FLOAT, DOM_COMPLEX, DOM_INT,DOM_RAT}, domtype(x))
     then expand_coeffs:= FALSE
     else expand_coeffs:= TRUE
  end_if;
  if not testtype(x, Type::Arithmetical) then
     error("illegal 3rd argument");
  end_if;

  //-------------------------------------------------------
  // Special case G(n,1,x)= chebyshev2(n,x)
  //-------------------------------------------------------

  if iszero(a-1) then
     userinfo(1, "calling orthpoly::chebyshev2");
     return(orthpoly::chebyshev2(n, x));
  end_if;

  //-------------------------------------------------------
  // Special case G(n,1/2,x)= legendre(n,x)
  //-------------------------------------------------------

  if iszero(a-1/2) then
     userinfo(1, "calling orthpoly::legendre2");
     return(orthpoly::legendre(n, x));
  end_if;

  //-------------------------------------------------------
  // General case G(n,a,x):
  //-------------------------------------------------------

  if domtype(x) = DOM_IDENT or type(x)="_index"
  then case n
       of 0 do return(poly(1, [x])); break;
       of 1 do return(poly(2*a*x, [x])); break;
       otherwise r0 := poly(1, [x]);
                 r1 := poly(2*a*x, [x]);
                 if expand_coeffs
                 then for i from 1 to n-1 do
                        [r1, r0]:= [mapcoeffs(
                                         poly(2*(a+i)*x,[x])*r1
                                       - poly(2*a+i-1,[x])*r0,
                                                 expand@_divide,i+1),
                                        r1];
                      end_for;
                 else for i from 1 to n-1 do
                        [r1, r0]:= [mapcoeffs(
                                         poly(2*(a+i)*x,[x])*r1
                                       - poly(2*a+i-1,[x])*r0,
                                         _divide,i+1),
                                        r1];
                   end_for;
                 end_if;
                 return(r1);
       end_case;
  else case n
       of 0 do return(1); break;
       of 1 do return(expand(2*a*x)); break;
       otherwise r0 := 1;
                 r1 := 2*a*x;
                 if expand_coeffs
                 then for i from 1 to n-1 do
                        [r1, r0]:= [expand(2*(a+i)/(i+1)*x*r1 
                                             - (2*a+i-1)/(i+1)*r0 ),
                                         r1];
                      end_for;
                 else for i from 1 to n-1 do
                        [r1, r0]:= [ ( 2*(a+i)*x*r1 
                                         - (2*a+i-1)*r0 
                                         )/(i+1),
                                         r1];
         
                      end_for;
                 end_if;
                 return(r1);
      end_case;
  end_if;
end_proc:

/* ---- end of file ---- */
