//      


// New implementation by Walter, 5.11.99
/*++
jacobi(n,a,b,x) - generates the nth Jacobi polynomial

n   -- nonnegative integer
a,b -- arbitrary arithmetical expressions
x   -- identifier or indexed identifier  --> output is poly(..,[x])
       any arithmetical expression    --> output is expression

P(0,a,b,x) = 1
P(1,a,b,x) = (a/2-b/2) + (1+a/2+b/2)*x
P(n,a,b,x) = ( (2*n+a+b-1)* ( (2*n-2+a+b)*(2*n+a+b)*x +a^2 - b^2
                             )*P(n-1,a,b,x)
                            - 2*(n-1+a)*(n-1+b)*(2*n+a+b)*P(n-2,a,b,x)
             ) / ( 2*n*(n+a+b)*(2*n-2+a+b) )

Special cases:  P(n,a,-a,x) special implementation
                P(n,-1/2,-1/2,x) = (2*n)!  /(n!)^2  /2^(2*n)*chebyshev1(n,x)
                P(n, 1/2, 1/2,x) = (2*n+1)!/n!/(n+1)/2^(2*n)*chebyshev2(n,x)
                P(n,  0,   0 ,x) = legendre(n,x)
--*/

orthpoly::jacobi := proc(n,a,b,x)
local i, expand_coeffs, amb, apb, r0, r1;
begin
  if args(0) <> 4 then error("wrong number of arguments"); end_if;
  if not (domtype(n) = DOM_INT) then
     return(procname(args()));
  end_if;
  if n < 0 then error("1st argument must not be negative"); end_if;
  if not testtype(a, Type::Arithmetical) then
     error("illegal 2nd argument");
  end_if;
  if not testtype(b, Type::Arithmetical) then
     error("illegal 3rd argument");
  end_if;
  if not testtype(x, Type::Arithmetical) then
     error("illegal 4th argument");
  end_if;
  expand_coeffs:= FALSE; // default
  if not has({DOM_FLOAT, DOM_COMPLEX, DOM_INT, DOM_RAT}, domtype(a))
     then expand_coeffs:= TRUE
  end_if;
  if not has({DOM_FLOAT, DOM_COMPLEX, DOM_INT, DOM_RAT}, domtype(b))
     then expand_coeffs:= TRUE
  end_if;
  if not has({DOM_FLOAT, DOM_COMPLEX, DOM_INT, DOM_RAT}, domtype(x))
     then expand_coeffs:= TRUE
  end_if;
  
  //-------------------------------------------------------
  // Special case P(n,0,0,x) = legendre poly P(n,x).
  // This implementation is faster than the most general
  // code below, so call it:
  //-------------------------------------------------------

  if iszero(a) and iszero(b) then
     userinfo(1, "calling orthpoly::legendre");
     return(orthpoly::legendre(n, x))
  end_if;

  //-------------------------------------------------------
  // Special case P(n,-1/2,-1/2,x) = 2^(2n)*(n!)^2/(2n)!* chebyshev1(n,x)
  // The implementation of chebyshev1 is extremely fast, so use ist:
  //-------------------------------------------------------

  if iszero(a+1/2) and iszero(b+1/2) then
     userinfo(1, "calling orthpoly::chebyshev1");
     i:= orthpoly::chebyshev1(n, x);
     if domtype(i) = DOM_POLY 
     then return(mapcoeffs(i,_mult,(2*n)!/(n!)^2/2^(2*n)));
     else if expand_coeffs then
               return(expand(i * (2*n)!/(n!)^2/2^(2*n) ));
          else return(       i * (2*n)!/(n!)^2/2^(2*n) );
          end_if;
     end_if;
  end_if;

  //-------------------------------------------------------
  // Special case P(n,1/2,1/2,x) = 2^(2n)*n!*(n+1)!/(2n+1)!*chebyshev2(n,x)
  // The implementation of chebyshev2 is fast, so use ist:
  //-------------------------------------------------------

  if iszero(a-1/2) and iszero(b-1/2) then
     userinfo(1, "calling orthpoly::chebyshev2");
     i:= orthpoly::chebyshev2(n, x);
     if domtype(i) = DOM_POLY 
     then return(mapcoeffs(i,_mult,(2*n+1)!/n!/(n+1)!/2^(2*n)));
     else if expand_coeffs then
               return(expand( i *  (2*n+1)!/n!/(n+1)!/2^(2*n) ));
          else return(        i *  (2*n+1)!/n!/(n+1)!/2^(2*n) );
          end_if;
     end_if;
  end_if;

  //-------------------------------------------------------
  // Special case P(n,a,-a,b):
  /* For b=-a the general recursion
   P(n,a,b,x) = ( (2*n+a+b-1)* ( (2*n-2+a+b)*(2*n+a+b)*x +a^2 - b^2
                               )*P(n-1,a,b,x)
                  - 2*(n-1+a)*(n-1+b)*(2*n+a+b)*P(n-2,a,b,x)
                 ) / ( 2*n*(n+a+b)*(2*n-2+a+b) )
   simplifies to
   P(n,a,-a,x) = (2*n-1)/n*x*P(n-1,a,-a,x)
                 - (n-1+a)*(n-1-a)/n/(n-1)*P(n-2,a,-a,x)
   No division by terms involving a occurs, so we can use it.
   Its faster than the most general case below.
  */
  //-------------------------------------------------------

  if iszero(a+b) then 
  userinfo(1, "using special recursion formula");
  if domtype(x) = DOM_IDENT or type(x)="_index"
  then case n
       of 0 do return(poly(1, [x]));
       of 1 do return(poly(a + x, [x]));
       otherwise r0 := poly(1, [x]):
                 r1 := poly(a+x, [x]):
                 if expand_coeffs
                 then for i from 2 to n do
                       [r1, r0] := [mapcoeffs(
                                    poly((2*i-1)*x, [x]) * r1
                                  - poly(i-1-a^2/(i-1), [x]) *r0,
                                     expand@_divide, i),
                                    r1];
                      end_for:
                 else for i from 2 to n do
                       [r1, r0] := [mapcoeffs(
                                    poly((2*i-1)*x, [x]) * r1
                                  - poly(i-1-a^2/(i-1), [x]) *r0,
                                    _divide, i),
                                    r1];
                      end_for:
                 end_if;
                 return(r1);
       end_case;
  else case n
       of 0 do return(1);
       of 1 do return(expand(a + x));
       otherwise r0 := 1; 
                 r1 := expand(a+x);
                 if expand_coeffs then
                    for i from 2 to n do
                      [r1, r0] := [((2*i-1)*x*r1 - (i-1-a^2/(i-1))*r0)/i, r1];
                      r1:= expand(r1);
                    end_for:
                    return(r1);
                 else
                    for i from 2 to n do
                      [r1, r0] := [((2*i-1)*x*r1 - (i-1-a^2/(i-1))*r0)/i, r1];
                    end_for:
                    return(r1);
                 end_if;
       end_case;
  end_if;
  end_if;

  // ------------ general case: any a, b: ---------------------
  /* Use the following recursion formula:

   2*i*P(i,a,b,x) = ( (a-b-2*i+2) + (a+b+2*i)*x )*P(n-1,a+1,b,x)
                   + (b+i-1)*(x-1)*P(i-2,a+2,b,x);

  Advantage: no division by terms involving a, b.
             So arbitrary a,b are allowed!
             The coefficients are automatically polynomials in
             a and b, nor normalization for symbolic a, b needed!
  Disadvantage: Involves computing P(n-i, a-i, b, x), i= 0.. n; 
                This is more costly than P(n-i,a,b,x), because
                the second argument has higher complexity!
  ---------------------------------------------------------------*/
  userinfo(1, "using general recursion formula");
  if domtype(x) = DOM_IDENT or type(x)="_index"
  then case n
       of 0 do return(poly(1, [x]));
       of 1 do return(poly((a/2-b/2) + (1+a/2+b/2)*x, [x]));
       otherwise a:= a+n-1;
                 amb:= a-b;
                 apb:= a+b;
                 r0 := poly(1, [x]):    //=P(0,a+n,b,x)
                 r1 := poly(amb/2 + (1+apb/2)*x, [x]); //=P(1, a+n-1,b,x)
                 if expand_coeffs
                 then for i from 2 to n do
                       [r1, r0] := [ mapcoeffs(
                                      poly( (amb-3*(i-1)) + (apb+1+i)*x, [x])*r1  
                                    + poly( (b+i-1)*(x-1), [x])*r0
                                              , expand@_divide,2*i),
                                     r1];
                      end_for:
                 else for i from 2 to n do
                       [r1, r0] := [ mapcoeffs(
                                     poly( (amb-3*(i-1)) + (apb+1+i)*x, [x])*r1  
                                   + poly( (b+i-1)*(x-1), [x])*r0
                                  , _divide,2*i),
                                     r1];
                      end_for:
                 end_if;
                 return(r1);
       end_case;
  else case n
       of 0 do return(1);
       of 1 do return(expand((a-b)/2 + (1+(a+b)/2)*x));
       otherwise a:= a+n-1;
                 amb:= a-b;
                 apb:= a+b;
                 r0 := 1;                   //=P(0,a+n,b,x);
                 r1 := amb/2 + (1+apb/2)*x; //=P(1, a+n-1,b,x)
                 r1 := expand(r1);
                 if expand_coeffs then
                    for i from 2 to n do
                      [r1, r0] := [ ( ((amb-3*(i-1)) + (apb+1+i)*x)*r1  
                                                  + ((b+i-1)*(x-1))*r0
                                     )/(2*i), r1];
                      r1:= expand(r1);
                    end_for:
                    return(r1);
                 else
                    for i from 2 to n do
                      [r1, r0] := [ ( ((amb-3*(i-1)) + (apb+1+i)*x)*r1  
                                                  + ((b+i-1)*(x-1))*r0
                                     )/(2*i), r1];
                    end_for:
                    return(r1);
                 end_if;
       end_case;
  end_if;
end_proc:

/* ---- end of file ----- */
