//      

/*++
laguerre(n, a, x) - generates the nth Laguerre polynomial

L(0,a,x) = 1
L(1,a,x) = -x + 1 + a
L(n,a,x) = (2*n+a-1-x)/n*L(n-1,a,x) - (n+a-1)/n*L(n-2,a,x)

Special cases: L(n,-n,x)=(-1)^n*x^n/n!
--*/
orthpoly::laguerre := proc(n,a,x)
local i, expand_coeffs, r0, r1;
begin
  if args(0) <> 3 then error("wrong number of arguments"); end_if;
  if not (domtype(n) = DOM_INT) then
     return(procname(args()));
  end_if;
  if n < 0 then error("1st argument must not be negative"); end_if;
  if not testtype(a, Type::Arithmetical) then
     error("illegal 2nd argument");
  end_if;
  if has({DOM_FLOAT, DOM_COMPLEX, DOM_INT,DOM_RAT}, domtype(a)) and
     has({DOM_FLOAT, DOM_COMPLEX, DOM_INT,DOM_RAT}, domtype(x))
     then expand_coeffs:= FALSE
     else expand_coeffs:= TRUE
  end_if;
  if not testtype(x, Type::Arithmetical) then
     error("illegal 3rd argument");
  end_if;

  //-------------------------------------------------------
  // Special case L(n,-n,x):
  //-------------------------------------------------------
  if iszero(n+a)
  then userinfo(1, "using special formula");
       if domtype(x) = DOM_IDENT or type(x)="_index"
          then return(poly((-1)^n/n!*x^n, [x] ))
          else return((-1)^n/n!*x^n)
       end_if;
  end_if;

  //-------------------------------------------------------
  // General case L(n,a,x):
  //-------------------------------------------------------

  if domtype(x) = DOM_IDENT or type(x)="_index"
  then case n
       of 0 do return(poly(1, [x])); break;
       of 1 do return(poly(1+a-x, [x])); break;
       otherwise r0 := poly(1, [x]);
                 r1 := poly(1 + a - x, [x]);
                 if expand_coeffs
                 then for i from 1 to n-1 do
                        [r1, r0] := [ mapcoeffs(
                                 poly((2*i+1 + a -x)/(i+1),[x])*r1
                                 - poly((i+a)/(i+1),[x])*r0,
                                           expand),
                                      r1]:
                      end_for;
                 else for i from 1 to n-1 do
                        [r1, r0] := [
                                 poly((2*i+1 + a -x)/(i+1),[x])*r1
                                 - poly((i+a)/(i+1),[x])*r0,
                                      r1]:
                   end_for;
                 end_if;
                 return(r1);
       end_case;
  else case n
       of 0 do return(1); break;
       of 1 do return(1+a-x); break;
       otherwise r0 := 1; 
                 r1 := expand(1+a-x);
                 if expand_coeffs
                 then for i from 1 to n-1 do
                        [r1, r0] := [ expand(((2*i+1+a-x)*r1 
                                                  - (i+a)*r0
                                             )/(i+1)),
                                      r1]:
                      end_for;
                 else for i from 1 to n-1 do
                        [r1, r0] := [ ((2*i+1+a-x)*r1 
                                           - (i+a)*r0)/(i+1),
                                      r1]:
                      end_for;
                 end_if;
                 return(r1);
      end_case;
  end_if;
end_proc:
