//    
// tonner, 23/06/98 

/*++
output::Content -- Domain of Content markup
++*/

output:

/** Creating Content trees
 *
 *   This domain implements the infrastructure to create Content
 *   trees
 */
domain Content
  category output::CatContent;
  inherits Dom::BaseDomain;
  
  Name := "Content";
    
  interface := {};
    
  create :=
  proc(typString : DOM_STRING, typInfo : DOM_STRING, attributes : DOM_STRING)
    local typ, fixed, nArgs;
  begin
    if args(0) < 3 then
      error("internal output error")
    end_if;
    if map({args(4..args(0))}, domtype) minus
       {DOM_STRING, DOM_INT, DOM_IDENT, DOM_FLOAT, Content} <> {} then
      error("children may only be idents, numbers, strings or elements of Content");
    end_if;
    if contains(ContentLayout::arities, typString) then;
      nArgs := ContentLayout::arities[typString];
      fixed := TRUE;
      if nArgs[-1] = "*" then
        nArgs := nArgs[1..-2];
        fixed := FALSE;
      end_if;
      nArgs := text2expr(nArgs);
      if args(0)-3 < nArgs then
        error("too few arguments for template '".typString."', at least ".nArgs." expected")
      end_if;
      if fixed and args(0)-3 > nArgs then
        error("got ".expr2text(args(0)-3)." arguments for fixed sized template '".typString."', ".nArgs." argument(s) expected");
      end_if;
    else
      error("template name '".typString."' unexpected");
    end_if;

    typ := dom::ContentTable[typString];
    if type(typ) = "_index" then
      error("No template \"".typString."\" defined");
    end_if;
    
    new(dom, args())
  end_proc;

  // access element entries
  typString := proc(c)
  begin
    if contains({DOM_INT,DOM_STRING,DOM_FLOAT,DOM_IDENT}, c::dom) then
        expr2text(c::dom)
    else
        extop(c, 1)
    end
  end;

  attributString := proc(c)
  begin
    if contains({DOM_INT,DOM_STRING,DOM_FLOAT,DOM_IDENT}, c::dom) then 
        ""
    else
      extop(c, 2)
    end
  end;

  infos := proc(c)
  begin
    if contains({DOM_INT,DOM_STRING,DOM_FLOAT,DOM_IDENT}, c::dom) then 
      ""
    else
      extop(c, 3)
    end
  end;
    
  op := proc(c,i=-1)
               begin
                 if i = -1 then
                   extop(c, 4..extnops(c))
                 elif testtype(i, "_range") then
                   extop(c, 3+op(i,1)..3+op(i,2))
                 else
                   extop(c, 3+i)
                 end_if
               end_proc;

  nops := c -> extnops(c) - 3;

  Content2tree :=
  proc(x)
    local res;
  begin
    if x::dom <> dom then
      if x::dom = DOM_STRING then
        [expr2text(x)]
      else
        [x]
      end_if
    else
      res := [dom::typString(x).
       ", info: ".expr2text(dom::infos(x)).", attrib : ".
       expr2text(dom::attributString(x))];
      if nops(x)=0 then res
      else append(res, op(map([op(x)], dom::Content2tree)));
      end_if;
    end_if
  end_proc;

  print := 
  proc(x)
  begin
    output::tree(dom::Content2tree(x), {hold(Small)});
  end_proc;
    
  subs := proc(c, eq)
            local idx, signString;
          begin
            if testtype(op(eq, 1), dom) and
               dom::typString(c) = "plus" and
               dom::typString(op(eq, 1)) = "plus" and
                   op(text2expr(dom::attributString(lhs(eq))), 2) = "-" then
              signString := op(text2expr(dom::attributString(c)), 2);
              idx := contains([op(c)], op(op(eq, 1)));
              if idx > 0 then
                signString[idx]:=
                (if signString[idx] = "+" then
                   "-"
                 else
                   "+"
                 end);
                return(new(dom, extop(c,1),
                           expr2text(hold(Signs)=signString), extop(c, 3),
                           subs(op(c, 1..idx-1), eq, Unsimplified), op(rhs(eq)),
                           subs(op(c, idx+1..nops(c)), eq, Unsimplified)))
              end_if;
            end_if;
            new(dom, subs(extop(c), eq, Unsimplified));
          end_proc;
     
    /**
     *  findAbbreviations(cont, varname)
     *
     *  Finds common subexpressions in the Content-tree (or one of the leaf-types DOM_IDENT,
     *  DOM_FLOAT, DOM_INT, DOM_STRING) cont and replaces them by abbreviations. The result
     *  is either cont itself or cont rewritten with abbreviations.
     */
  findAbbreviations := 
  proc(cont, varname = Pref::abbreviateOutput(hold(Placeholder)) : DOM_IDENT)
    local occurrences, significance, res, getSubExpressions, subExprTab,
          i, j, maxLength, candidates, nextPlaceholder, placeholders, newPlaceholders,
          bestSignificance, indexIndets, saveSubExprTab, abbrevLength, skipFurtherCandidates,
          removedPlaceholders;
  begin
    assert(contains({dom, DOM_IDENT, DOM_FLOAT, DOM_INT, DOM_STRING}, domtype(cont)));
    if  domtype(cont) <> dom then
      return(cont);
    end_if;
    
    res := [cont];
    occurrences := Pref::abbreviateOutput(hold(Occurrences));
    // compute the significance of a possible abbreviation
    // take relative length and number of occurrences c into account
    significance := 
      (c, l) -> ceil(((1/maxLength*(1-l/maxLength)^4)*20*l^2))*
               ceil((if c - occurrences + 1 > 0 then c/2 else 0 end_if)^2);
  
    // returns a generator for placeholders
    placeholders := [];
    nextPlaceholder :=
    proc(placeHolder, Ex)
      local count, idents, PHbase;
      option escape;
    begin
      PHbase := "".placeHolder;
      idents := map(map(indets(Ex, All), expr2text),
                    stringlib::subs, "`"="");
      count := 0;
        
      proc()
      begin
        // make sure that our desired placeholder is not yet used in Ex
        repeat
          count := count + 1;
        until idents intersect {PHbase."{".count."}", PHbase.count} = {} and
              not has(Ex, [hold(_index)(hold(``).(PHbase[1..-2]), count),
                           hold(_index)(hold(``).PHbase, count)])
        end_repeat;
        if PHbase[-1] = "_" then
          Content(hold(_index)(hold(``).(PHbase[1..-2]), count));
        else
          Content(hold(``).PHbase.count)
        end_if;
      end_proc;
    end_proc(varname, res);
    
    // an equivalent of numeric::indets, working on Content trees
    indexIndets :=
    proc(cont)
    begin
      if testtype(cont, DOM_IDENT) then
        return({cont});
      end_if;
      if not testtype(cont, Content) then
        return({});
      end_if;
      if Content::typString(cont) = "index" and
         domtype(op(cont, 1)) = DOM_IDENT then
        return({cont});
      end_if;
      _union(op(map({op(cont)}, indexIndets)));
    end_proc;
      
    // the return value is the length of ex
    getSubExpressions :=
    proc(ex)
      local entry, attributes, prio;
    begin
      if contains(subExprTab, ex) then
        entry := subExprTab[ex];
        entry[1] := entry[1]+1;
        subExprTab[ex] := entry;
        return(entry[2]);
      end_if;
    
      if ex = op(ex) or 
         Content::typString(ex) = "index" and domtype(op(ex,1)) = DOM_IDENT then
        return(1);
      end_if;
    
      entry := [1, _plus(1, getSubExpressions(op(ex, i)) $ i=1..nops(ex))];
      prio := infinity;
      attributes := table(ContentLayout::attributes(Content::LayoutTable[Content::typString(ex)]));
      if contains(attributes, "prio") then
         prio := text2expr(attributes["prio"]);
      end_if;
      if entry[2] >= Pref::abbreviateOutput(hold(MinLength)) and 
         prio > output::Priority::Relation  then
        subExprTab[ex] := entry;
      end_if;
      entry[2];
    end_proc;
    
    
    removedPlaceholders := {};
    repeat
      // subExprTab is a table of subexpressions in ex with entries
      // subExpr = [count, length]
      subExprTab := table();
      
      maxLength := max(op(map(op(res), getSubExpressions)));

      // the expression is short enough, done
      if maxLength < Pref::abbreviateOutput(hold(Threshold)) then
        break;
      end_if;
    
      // entries are: [significance, count, length, expr]
      candidates := [[significance(op(i[2])), op(i[2]), i[1]] $ i in subExprTab];
    
      // avoid {#1}, #1="huge expression" by removing all candidates,
      // which use up nearly the complete maxLength
      candidates := select(candidates, x -> x[3] < 0.9*maxLength and x[1] > 0
                                              and contains(res, x[4]) = 0);
    
      // out of candidates, done
      if candidates = [] then break; end_if;

      // sort candidates by significance
      // the resulting list is ordered in ascending significance
      candidates := sort(candidates);
    
      bestSignificance := candidates[-1][1];
      for i from nops(candidates) downto 1 do
        // Use all candidates with similar good significance as abbreviations,
        // not only the best one. This reduces the number of repeat loop
        // cycles which are more expensive.
        skipFurtherCandidates := FALSE;
        if candidates[i][1] > 0.8*bestSignificance then
          placeholders := placeholders.[genident("#mumpitz")];
          res := subs(res, candidates[i][4] = placeholders[-1],
                      eval(candidates[i][4]) = placeholders[-1], Unsimplified).
               [candidates[i][4]];
          
          // abbreviations in previously found abbreviations may lead to
          // abbreviations smaller than MinLength. 
          saveSubExprTab := subExprTab;
          subExprTab := table();
          abbrevLength := map(res[2..-1], getSubExpressions);
          subExprTab := saveSubExprTab;
          assert(nops(abbrevLength) = nops(placeholders));
          for j from nops(abbrevLength) downto 1 do
            if abbrevLength[j] < Pref::abbreviateOutput(hold(MinLength)) then
              res := subs(res, placeholders[j]=
                          subs(res[j+1],  placeholders[j]=res[-1], Unsimplified),
                          Unsimplified);
              skipFurtherCandidates := TRUE;
              removedPlaceholders := removedPlaceholders union {placeholders[j]};
              delete placeholders[j], res[j+1];
              assert(indets(res, All) intersect removedPlaceholders = {});
            end_if;
          end_for;
          if skipFurtherCandidates = TRUE then
            // leave loop over candidates
            break;
          end_if;
          // just in case that further candidats contain the just replaced
          // substitution
          candidates := subs(candidates[1..i-1], 
                             candidates[i][4] = placeholders[-1],
                             eval(candidates[i][4]) = placeholders[-1],
                             Unsimplified);
          assert(indets(candidates, All) intersect removedPlaceholders = {});
        else
          break;
        end_if;
      end_for;
    until FALSE end;
  
    // sort replacements such that whenever the abbreviation a uses the abbreviation b,
    // a is listed *above* b in the final output
    if nops(res) > 2 then
      traperror((
        proc()
          local sortOrder, i, j, plh_set, G;
        begin
          plh_set := {op(placeholders)};
          G := [([placeholders[i-1], j] $ j in indexIndets(res[i]) intersect plh_set) $ i = 2..nops(res)];
          if nops(G) > 0 then
            G := Graph(placeholders, G, Directed);
            sortOrder := Graph::topSort(G)[1];
            res[2..-1] := [res[1+contains(placeholders, sortOrder[i])] $ i = 1..nops(sortOrder)];
            sortOrder := [op(sortOrder)];
            res := subs(res,
                        map(sortOrder, eq -> op(eq, 2)=placeholders[op(eq,1)]), Unsimplified);
          end_if;
        end_proc()
      ));
    end_if;
    
    // now replace the intermediate placeholders with the real ones
    newPlaceholders := [nextPlaceholder() $ i = 1..nops(placeholders)];
    res := subs(res, zip(placeholders, newPlaceholders, _equal), Unsimplified);
    
    assert(indets(res, All) intersect ({op(placeholders)} union removedPlaceholders) = {});
    
    res := [res[1], op(zip(newPlaceholders, res[2..-1], (x,y)->Content::create("eq", "", "", x, y)))];
    if nops(res) > 1 then
      Content::Cabbreviation(res[1],
                             Content(hold(where)),
                             Content::Crow(Content(hold(`&Space;&Space;`)),
                                           res[i]) $ i =2..nops(res));
    else
      res[1]
    end_if;
  end_proc;
    
    // XML output used in package doc and for xmsg protocol
    printXML_data :=
    proc(xmlfd, type, data)
    begin
      xmlprint::beginElement(xmlfd, type);
      xmlprint::data(xmlfd, data);
      xmlprint::endElement(xmlfd, type);
    end_proc;

    // take apart an ident (already converted to a string),
    // putting its accents into <iacc> elements.
    XML_accents := {"OverBar", "UnderBar", "ast", "prime", "circ",
                    "tilde", "rarr", "dot", "die", "tdot"};
    XML_chars := select(map(Symbol::symbols, expr2text),
                        x -> testtype(slot(Symbol, x), DOM_IDENT));
    
    // when changing the output of identifiers, keep GENERATE/TeX.mu up-to-date!
    printXML_ident :=
    proc(xmlfd, s)
      local recurse, res;
    begin
      recurse :=
      proc(s : DOM_STRING)
        local st, i, j, res, push, pop, popIdent, item, prevstring,
              nextToken, token, token2, symb;
      begin
        st := adt::Stack();
        push := st::push;
        pop := proc()
                 local p;
               begin
                 p := st::pop();
                 if p = FAIL then
                   ""
                 else
                   p
                 end_if;
               end_proc;
        popIdent := proc()
                      local p;
                    begin
                      p := pop();
                      if p = "" or p[1] <> "<" or
                         length(p) > 8 and p[1..8] = "<chr nm="  then
                        "<ident>".p."</ident>"
                      else
                        p
                      end_if;
                    end_proc;
        nextToken :=
        proc(s : DOM_STRING) : DOM_STRING
          local i, open;
        begin
          if s = "" then
            error("no more tokens");
            return("")
          end_if;
          if s[1] = "{" then
            open  := 0;
            for i from 1 to length(s) do
              if s[i] = "{" then
                open := open + 1;
              elif s[i] = "}" then
                open := open - 1;
                if open = 0 then
                  return(s[1..i])
                end_if;
              end_if;
            end_for;
            error("unmatched parentheses");
          else
            s[1]
          end_if;
        end_proc;
        
        // first we split the string
        i := 1;
        while i <= length(s) do
          if s[i] = "{" then
            token := nextToken(s[i..-1]);
            push(recurse(token[2..-2]));
            i := i + length(token);
            next;
          end_if;
          if s[i] = "<" then
            push("<chr nm=\"lt\"/>");
            i := i+1;
            next;
          end_if;
          if s[i] = "&" then
            j := strmatch(s[i..-1], ";", Index)[1] + i - 1;
            if contains(dom::XML_accents, s[i+1..j-1]) then
              push("<iacc nm=\"".s[i+1..j-1]."\">".popIdent()."</iacc>");
            elif s[i..j] = "&UnderLine;" then
              push("_")
            elif contains(dom::XML_chars, s[i+1..j-1]) then
              push("<chr nm=\"".s[i+1..j-1]."\"/>");
            else
              push(stringlib::subs(s[i..j], "&"="&amp;"))
            end_if;
            i := j+1;
            next;
          end_if;
          if contains({"_", "^", "$"}, s[i]) then
            symb := s[i];
            token := nextToken(s[i+1..-1]);
            i := i + 1 + length(token);
            if symb = "_" then
              push("<isub>".popIdent().recurse(token)."</isub>")
            elif symb = "^" then
              push("<isup>".popIdent().recurse(token)."</isup>")
            elif symb = "$" then
              token2 := nextToken(s[i..-1]);
              i := i + length(token2);
              push("<isubsup>".popIdent().recurse(token).recurse(token2)."</isubsup>")
            end_if;
            
            next;
          end_if;
          
          // unprocessed character, put it on the stack
          st::push(s[i]);
          i := i+1;
        end_while;
        
        res := null();
        prevstring := "";
        item := pop();
        while item <> "" do
          if "<" <> item[1] or
             (length(item) > 8 and item[1..8] = "<chr nm=") then
            prevstring := item.prevstring;
            item := pop();
            next;
          end_if;
          if prevstring <> "" then
            res := "<ident>".prevstring."</ident>", res;
            prevstring := "";
          end_if;
          res := item, res;
          item := pop();
        end_while;
        if prevstring <> "" then
          res := "<ident>".prevstring."</ident>", res;
        end_if;
        if nops(res) > 1 then
          _concat("<irow>", res, "</irow>");
        elif nops(res) = 0 then
          "<ident/>"
        else
          res
        end_if;
      end_proc;
      
      // mask _ for all protected variables, but not for #-variables
      if (s = "" or s[1] = "#" or
          subs(hold(protected(#)), #=hold(``).s, EvalChanges) <> Error) and
         traperror((res := recurse(s))) = 0 then
        xmlprint::raw(xmlfd, res)
      else
        // some syntax error during processing, print ident in literal form
        xmlprint::beginElement(xmlfd, "ident"),
        xmlprint::data(xmlfd, s);
        xmlprint::endElement(xmlfd, "ident");
      end_if;
    end_proc;
    
    printXML :=
    proc(data, xmlfd)
      local t, s, floatFormat, typString;
    begin
      typString := dom::typString(data);
      case typString
        of "DOM_STRING" do
          dom::printXML_data(xmlfd, "str", data);
          return();
        of "DOM_IDENT" do
          s := expr2text(data);
          if s[1]="`" then
            s := s[2..-2];
          end_if;
          dom::printXML_ident(xmlfd, s);
          return();
        of "DOM_INT" do
          if data < 0 then
            xmlprint::beginElement(xmlfd, "plus",
                                   "sgn" = "-");
            dom::printXML_data(xmlfd, "num", expr2text(-data));
            xmlprint::endElement(xmlfd, "plus");
          else
            dom::printXML_data(xmlfd, "num", expr2text(data));
          end_if;
          return();
        of "DOM_FLOAT" do
          floatFormat := Pref::floatFormat();
          if floatFormat = "h" then
            Pref::floatFormat("g");
          end_if;
          if abs(data) > 1e-1000 and abs(data)< 1e10000 then
            Pref::floatFormat("f");
          else
            Pref::floatFormat("g");
          end_if;
          if traperror((s := expr2text(data))) <> 0 then
            s := "<<<Error>>>"
          end_if;
          Pref::floatFormat(floatFormat);
          if strmatch(s, "e") then
            t := strmatch(s, "e", Index);
            [t, s] := [s[1..t[1]-1], s[t[1]+1..-1]];
            xmlprint::beginElement(xmlfd, "expr", "tmp"="times");
            if t[1] = "-" then
              xmlprint::beginElement(xmlfd, "plus",
                                     "sgn" = "-");
              dom::printXML_data(xmlfd, "num", t[2..-1]);
              xmlprint::endElement(xmlfd, "plus");
            else
              dom::printXML_data(xmlfd, "num", t);
	    end_if;
	    xmlprint::beginElement(xmlfd, "expr", "tmp"="power");
	    dom::printXML_data(xmlfd, "num", "10");
            if s[1] = "-" then 
              xmlprint::beginElement(xmlfd, "plus",
                                     "sgn" = "-");
              dom::printXML_data(xmlfd, "num", s[2..-1]);
              xmlprint::endElement(xmlfd, "plus");
            else
              dom::printXML_data(xmlfd, "num", s);
	    end_if;
	    xmlprint::endElement(xmlfd, "expr");
	    xmlprint::endElement(xmlfd, "expr");
          else
            dom::printXML_data(xmlfd, "num", s);
          end_if;
          return();
          
        of "plus" do
          xmlprint::beginElement(xmlfd, "plus",
                                 subs(text2expr(dom::attributString(data)),
                                      hold(Signs)=hold(sgn)));
          map([op(data)], Content::printXML, xmlfd);
          xmlprint::endElement(xmlfd, "plus");
          return();
          
      end_case;
      if ContentLayout::hasOptable(dom::LayoutTable[typString])
        then
        xmlprint::beginElement(xmlfd, "tab",
                               "tmp"=typString,
                               subs(text2expr(dom::attributString(data)),
                                    [hold(Rows)=hold(rows),
                                     hold(Cols)=hold(cols),
                                     hold(rowlines)=hold(rowl),
                                     hold(columnlines)=hold(coll)]));
      else
        xmlprint::beginElement(xmlfd, "expr",
                               "tmp"=typString,
                               text2expr(dom::attributString(data)));
      end_if;
      if op(data) <> FAIL then
        map([op(data)], Content::printXML, xmlfd);
      end_if;
      if ContentLayout::hasOptable(dom::LayoutTable[typString])
         then xmlprint::endElement(xmlfd, "tab");
      else xmlprint::endElement(xmlfd, "expr");
      end_if;
    end_proc;
    
  expr :=
  proc(c)
    local i, cType, attributes, typString;
  begin
    cType := c::dom;
    typString := dom::typString(c);

    // here we need a special conversion from backquoted identifiers
    // representing MML-constants to MuPAD constants
    if cType = DOM_IDENT and contains(dom::MML2ident, c) then
       return(dom::MML2ident[c]);
    end_if;

    if not contains({DOM_STRING, DOM_INT, DOM_IDENT, DOM_FLOAT}, cType) and cType <> dom then
      error("wrong type")
    end_if;

    if nops(dom::ContentTable[typString]) > 1 then
      attributes := table(text2expr(dom::attributString(c)));
      return(dom::ContentTable[typString][2](c, attributes, dom::infos(c)));
    // ---  otherwise we would enter into an infinite loop, e.g., if c 
    // ---  is of domain type DOM_INT, and the constructor is returns
    // ---  an object of a library domain.

    // inserted  'and cType <> dom' because Franks patch broke
    // expr(Content(x!))         tonner 
    elif nops(c) = 1 and cType <> dom then
      subs(hold(XxX)(c),  hold(XxX)=op(dom::ContentTable[typString], 1), Unsimplified)
    // ----
    else
      subsop(hold(XxX)(FAIL$nops(c)),
             0=op(dom::ContentTable[typString], 1),
             i = dom::expr(op(c,i)) $ i = 1 .. nops(c),
             Unsimplified)
    end_if;
  end_proc;


/**  Table of all known Content types
 *
 *   Format of the table entries:
 *   
 *   index = [mupadType, backTranslate (optional)]
 *
 *   index              : string which adresses the template in MuPAD
 *
 *   mupadType          : type of the MuPAD object which is represented
 *                        by this Content
 *
 *   backTranslate (opt): MuPAD function which translates a Content 
 *                        element of the type index into a valid MuPAD
 *                        object
 */
  ContentTable := table();

/** Table of all known Content Layouts
 *
 */
  LayoutTable := table();

    // everthing which was written with the last call of writeLayoutSetup
  LastLayoutTable := {};
 
    // with the only option Complete a complete Setup file is written.
    // otherwise only a patch 
  writeLayoutSetup :=
  proc()
    local oldFid, fid, fName, CL, templates, update, cl,
          openFile, closeFile, DOM, Args, complete, fileName;
  begin
    complete := FALSE;
    Args := {args()};
    if contains(Args, hold(Complete)) then
      Args := Args minus {hold(Complete)};
      complete := TRUE;
    end_if;
    if nops(Args) > 1 then
      error("wrong number of arguments")
    end_if;
    if nops(Args) = 1 then
      Args := op(Args, 1);
      if Args::dom <> DOM_STRING then
        // last argument is no filename
        error("unecpected argument: ".expr2text(Args))
      else
        fileName := Args;
      end_if;
    else
      // nops(Args) = 0
      fileName := TempFile;
    end_if;
      
    DOM := dom;
    CL := ContentLayout;  // shortcut
    fName := "";
 
    // opens a new temporary setup file for writing
    openFile :=
    proc()
    begin
      oldFid := CL::OutputFile;
      if fileName = TempFile then
        fid := fopen(Text, TempFile, Write);
        fName := fname(fid);
        // get corresponding filename and insert it into a toBeDeleted
        // set for Pref::callOnExit
        sysassign(stdlib::TempFiles, stdlib::TempFiles union {fName});
      else
        fid := fopen(Text, fileName, Write);
        fName := fileName;
      end_if;
      sysassign(CL::OutputFile, fid);
    end;

    // closes the output file, iff one was opened
    // and informs the clients about the new setup File
    closeFile :=
    proc()
    begin
      if fName <> "" then
        // openFile was called, close the file and inform clients in cl
        fclose(fid);
        sysassign(CL::OutputFile, oldFid);
        map(cl, x -> traperror(clientCall(x, "templateFile", fName)));
      end_if;
    end_proc;
    
    templates := {map(op(dom::LayoutTable), op, 2)};
    if complete then
      // write complete setup file and inform the current client
      openFile();
      CL::xml(CL::mathsetup(op(DOM_SET::sort(templates))));
      cl := [clientObject()];
    else
      update := templates minus dom::LastLayoutTable;
      if update <> {} then
       // write update setup file and inform all clients
       openFile();
        CL::xml(CL::mathsetup(op(DOM_SET::sort(update))));
        cl := clientObject(All);
      else
        // update setup is empty, nothing to do
        cl := [];
      end_if;
    end_if;
    dom::LastLayoutTable := templates;
    closeFile();
    delete cl;
    null()
  end_proc;
    
/* Define new entries in Content::ContentTable */
  set :=
  proc(NAME : DOM_STRING,
       layout,
       mupadType,
       backTranslate = null() : DOM_PROC)
    option hold;
    local ltype;
  begin
    if testargs() then
      ltype := domtype(context(layout));
      if not contains({DOM_FAIL, ContentLayout}, ltype) then
        print(args());
        error("wrong type of 2. argument")
      end_if;
    end_if;
    dom::ContentTable[NAME] := [mupadType, backTranslate];
    if layout <> FAIL then
      dom::LayoutTable[NAME] := context(layout);
    end_if;
  end_proc;
  
    /*
     * token elements
     */
  Ccn:=       // Content Number
  proc(arg1) 
    local num; 
  begin 
    num := args(args(0));
    if testtype(arg1, DOM_LIST) and arg1 = ["type"="rational"] then 
      dom::create("divide", "", "", args(2), args(4))
    else
      num
    end_if
  end_proc;

  Cci:= () -> if contains(dom::ident2MML, args()) then
                 dom::Ccn(["type"="constant"], dom::ident2MML[args()]);
              else
                 args();      // Content Identifier
              end_if;

  Ccsymbol:= () -> error("not yet implemented");
    
    /*
     * basic content elements
     */

  /**  Capply - the MathML apply construct
   *  The implementation of the MathML apply construct for MuPAD Content
   *  output is quite complicated, but makes it easy to define the Content
   *  slot for MuPADs expression types.  E.g. 
   */
  Capply:=
  proc()
  begin
    if testargs() and args(1) = FAIL then
      error("trying to apply undefined entity");
    end_if;
    args(1)(args(2..args(0)))
  end_proc;   

  Creln:= () -> error("not yet implemented");
  Cfn := id;      // user-defined function
  Cinterval := // interval constructor
     proc(x, left, right)
     begin
       if x = [] then
         // kernel interval
         return(dom::create("interval", "", "", left, right));
       end_if;

       // form here we handle library intervals
       if op(x,[1,2]) = "open" then
         dom::create("intervaloo", "", "", left, right);
       elif op(x,[1,2]) = "open-closed" then
         dom::create("intervaloc", "", "", left, right);
       elif op(x,[1,2]) = "closed-open" then
         dom::create("intervalco", "", "", left, right);
       else
         dom::create("intervalcc", "", "", left, right);
       end_if;
     end_proc;

  Cinverse := () -> dom::create("inverse", "", "", args()); // functional inverse
  Csep :=  () -> dom::create("sep", "", "", args());       // separator
  // properties of an identifier
  Ccondition := () -> dom::create("condition", "", "", args());   
         
  Cdeclare := () -> error("not yet implemented");   // declaration
  Clambda := () -> dom::create("lambda", "", "", args());  // procedure
  Ccompose := () -> dom::create("compose", "", "", args());
  Cident  :=    () -> dom::create("ident", "", "", args());// identity function
  Cdomain:= () -> error("not yet implemented");
  Ccodomain:= () -> error("not yet implemented");
  Cimage:= () -> error("not yet implemented");
  Cdomainofapplication:= () -> error("not yet implemented");
  Cpiecewise :=  () -> if args(args(0)) <> hold(`&Space;otherwise`) then
                         dom::create("piecewise",
                                   expr2text(hold(Rows)=
                                             expr2text(args(0) div 2),
                                             hold(Cols)="2"), "", op(map([args()], x -> if x::dom = Content and x::dom::typString(x) = "piece2" then op(x) else x end)));
                       else
                         dom::create("piecewise3",
                                   expr2text(hold(Rows)=
                                             expr2text(args(0) div 2),
                                             hold(Cols)="2"), "",
                                     args());
                       end_if;
  Cpiece :=  () ->  (args(1), dom::create("piece2", "", "", args(2)));
  Cotherwise:= () ->  (args(), hold(`&Space;otherwise`));


    /*
     * arithmetic, algebra and logic
     */
  Cquotient :=  () -> dom::create("quotient", "", "", args());
  Cfactorial := () -> dom::create("factorial", "", "", args());
  Cdivide :=    () -> dom::create("divide", "", "", args());
  Cmax := () -> dom::create("apply", "", "", hold(max), args());
  Cmin := () -> dom::create("apply", "", "", hold(min), args());

  // negation  
  Cminus :=  () -> dom::create("plus", "Signs=\"-\"", "", args());
  Cplus :=
  proc()
    local DOM, list, Signs, s, i;
  begin
    DOM := dom;
    /* here we have to translate all operands of type
       "minus" into out special sign encoding for _plus and
       additionally we flat out nested "plus" templates */
    list := map([args()],
                contents ->
                if contents::dom = DOM then
                  if DOM::typString(contents) = "plus" and
                    DOM::attributString(contents) = "Signs=\"-\"" or
                    DOM::typString(contents) = "negate" then
                    [op(contents), "-"]
                  elif DOM::typString(contents) = "plus" then
                    s := DOM::attributString(contents);
                    [op(contents, i), substring(s, 7+i)] $
                        i = 1..nops(contents)
                  else
                    [contents, "+"]
                  end_if;
                elif contains({DOM_INT, DOM_RAT, DOM_FLOAT},
                            contents::dom) and contents < 0 then
                  [-contents, "-"]
                else
                  [contents, "+"]
                end_if);
    Signs := _concat(op(map(list, op, 2)));
    list := map(list, op, 1);
    if Signs = "+" or nops(list) = 0 then
      dom::create("apply", "", "", hold(_plus), op(list))
    else
      dom::create("plus", "Signs=".expr2text(Signs), "",
                  op(list));
    end_if;
  end_proc;
  Cpower :=     (base, expo) -> if testtype(expo, dom) then
                                  if dom::typString(expo) = "divideLinear" and
                                     length(expr2text(expr(expo))) < 6 then
                                    dom::create("power", "", "", base, expo);
                                  else
                                    dom::create("powerBigExpo", "", "", base, expo);
                                  end_if;
                                else
                                  if length(expr2text(expo)) < 6 then
                                    dom::create("power", "", "", base, expo);
                                  else
                                    dom::create("powerBigExpo", "", "", base, expo);
                                  end_if;
                                end_if;
  Crem   :=     () -> dom::create("rem", "", "", args());
  Ctimes :=     () -> dom::create("times", "", "", args());
  Croot :=      () -> if args(0) = 1 then
                        dom::create("sqrt", "", "", args());
                      else
                        dom::create("root", "", "", args(2), args(1));
                      end_if;

  Cgcd := () -> error("not yet implemented");
  Cand :=       () -> dom::create("and", "", "", args());
  Cor :=        () -> dom::create("or", "", "", args());
  Cxor :=       () -> dom::create("xor", "", "", args());
    
  Cnot :=  () -> (if testtype(args(), dom) then
                   if dom::typString(args()) = "in" then
                     return(dom::create("notin", "", "", op(args())));
                   elif dom::typString(args()) = "subset" then
                     return(dom::create("notsubset", "", "", op(args())));
                   end_if;
                  end_if;
                  dom::create("not", "", "", args());
                  );
  Cimplies   := () -> dom::create("implies", "", "", args());
  Cforall    := () -> error("not yet implemented");
  Cexists    := () -> error("not yet implemented");
  Cabs       := () -> dom::create("abs", "", "", args());
  Cconjugate := () -> dom::create("conjugate", "", "", args());
  Carg       := () -> dom::create("apply", "", "", dom(hold(arg)), args());
  Creal      := () -> dom::create("apply", "", "", dom(hold(Re)), args());
  Cimaginary := () -> dom::create("apply", "", "", dom(hold(Im)), args());
  Clcm       := () -> error("not yet implemented");
  Cfloor     := () -> dom::create("floor", "", "", args());
  Cceiling   := () -> dom::create("ceil", "", "", args());

    
    /*
     * relations
     */
  Ceq  := () -> dom::create("eq", "", "", args());
  Cneq := () -> dom::create("neq", "", "", args());
  Cgt  := () -> dom::create("gt", "", "", args());
  Clt  := () -> dom::create("lt", "", "", args());
  Cgeq := () -> dom::create("geq", "", "", args());
  Cleq := () -> dom::create("leq", "", "", args());
  // Compare Cequiv which has a different rendering, but same semantics.  
  Cequivalent := () -> error("not yet implemented");
  Capprox := () -> dom::create("approx", "", "", args());
  Cfactorof := () -> error("not yet implemented");

    /*
     * calculus and vector analysis
     */
  Cint := proc(x)
            local attrs, Args, nargs;
          begin
            if contains({DOM_LIST, DOM_SET}, x::dom) then
              attrs := expr2text(op(x));
              Args  := [args(2..args(0))];
            else
              attrs := "";
              Args  := [args()];
            end_if;
            nargs := nops(Args);
            if nargs = 4 then
                // definite integral
              dom::create("definiteint", "", attrs, Args[4], Args[1], Args[2], Args[3])
            elif nargs = 3 then
                // integral over set
              dom::create("intdomain", "", attrs, Args[3], op(Args[2]), Args[1])
            else
                // indefinite integral
              dom::create("int", "", attrs, Args[2], Args[1])
            end_if
          end_proc;
  Cdiff := () -> if args(0) = 1 then
                   if testtype(args(1), dom) then
                     if dom::typString(args(1)) = "D" then
                       return(dom::create("D2", "", "", op(args(), 1)));
                     elif dom::typString(args(1)) = "D2" then
                       return(dom::create("D3", "", "", op(args(), 1)));
                     elif dom::typString(args(1)) = "D3" then
                       return(dom::create("DN", "", "", op(args(), 1), 4));
                     elif dom::typString(args(1)) = "DN" then
                       return(dom::create("DN", "", "", op(args(), 1),
                                          dom( dom::expr(op(args(), 2))+1)));
                     end_if;
                    end_if;
                   dom::create("D", "", "", args())
                 else
                   dom::create("Dpara", "", "", args())
                 end_if;
  Cpartialdiff := (x, y, z) ->
      if args(0) = 2 then
        if x::dom = dom and
          dom::typString(x) = "list" then
          dom::create("Dpara", "", "", op(x), y);
        else
          dom::create("partialdiff", "", "", y, x);
        end_if
      else
        dom::create("partialdiff1", "", "", z, x, y);
      end_if;
  Clowlimit := () -> args();
  Cuplimit  := () -> args();
  Cbvar := () -> args();
  Cdegree := () -> args();
  Cdivergence := () -> dom::create("divergence", "", "", args());
  Cgrad :=  () -> dom::create("grad", "", "", args());
  Ccurl :=  () -> dom::create("curl", "", "", args());
  Claplacian :=  () -> dom::create("laplacian", "", "", args());


    
    /*
     * theory of sets
     */
  Cset := () ->
          if args(0) = 0 then
            dom::Cemptyset()
          elif args(0) = 3 and testtype(args(2), dom) and
            dom::typString(args(2)) = "condition" then
            // this is an imageset
            dom::create("imageset", "", "", args(3), op(args(2)));
          else
            dom::create("set", "", "", args())
          end_if;
  Clist := () -> dom::create("list", "", "", args());
  Cunion := () -> dom::create("union", "", "", args());
  Cintersect := () -> dom::create("intersect", "", "", args());
  Cin := () -> dom::create("in", "", "", args()); 
  Cnotin := () -> dom::create("notin", "", "", args());
  Csubset := () -> dom::create("subset", "", "", args());
  Cprsubset := () -> dom::create("prsubset", "", "", args());
  Cnotsubset := () -> dom::create("notsubset", "", "", args());
  Cnotprsubset := () -> dom::create("notprsubset", "", "", args());
  Csetdiff := () -> dom::create("setdiff", "", "", args());
  Ccard := () -> dom::create("card", "", "", args());
  Ccartesianproduct := () -> error("not yet implemented");

    /*
     * sequences and series
     */
  Csum := proc()
          begin
            if args(0) = 4 then
              dom::create("definitesum", "", "", args(4), args(1..3))
            elif args(0) = 3 then
              dom::create("sumrootof", "", "", args(3), args(1),
                          op(args(2), [1, 2])) // get RootOf out of condition
            else
              dom::create("sum", "", "", args(2), args(1))
            end_if
          end_proc;
                  
  Cproduct := proc()
          begin
            if args(0) = 4 then
              dom::create("definiteproduct", "", "", args(4), args(1..3))
            elif args(0) = 3 then
              dom::create("productrootof", "", "", args(3), args(1),
                          op(args(2), [1, 2])) // get RootOf out of condition
            else
              dom::create("product", "", "", args(2), args(1))
            end_if
          end_proc;

  Climit := proc()
              local arg2;
            begin
              arg2 := args(2);
              if testtype(arg2, dom) and
                dom::typString(arg2) = "condition" then
                // directed limit
                if op(arg2, 1) = "Direction = Right" then
                  dom::create("limitRight", "", "",
                              args(3), args(1), op(arg2, 2) );
                elif op(arg2, 1) = "Direction = Left" then
                  dom::create("limitLeft", "", "",
                              args(3), args(1), op(arg2, 2) );
                else  // two-sided
                  dom::create("limit", "", "",
                              args(3), args(1), op(arg2, 1) );
                end_if;
              else
                dom::create("limit", "", "", args(3), args(1..2));
              end_if;
            end_proc;

  Ctendsto := proc(x: DOM_LIST)
                option escape;
                local direction;
              begin
                if contains({"above", "below", "two-sided"}, op(op(x),2)) then
                  if op(op(x),2) = "above" then
                    direction := "Direction = Right"
                  elif op(op(x),2) = "below" then
                    direction := "Direction = Left"
                  else
                    direction := null()
                  end_if;
                  proc(bvar, tendsto)
                  begin
                    direction, tendsto
                  end_proc
                else
                  id
                end_if;
              end_proc;

    /*
     * elementary classical functions
     */
  Cexp :=       () -> dom::create("exp", "", "", args());   
  Cln := () -> dom::create("apply", "", "", hold(ln), args());
  Clog := () -> dom::create("log", "", "", args(2), args(1));
  Csin    := () -> dom::create("apply", "", "", hold(sin), args());
  Ccos    := () -> dom::create("apply", "", "", hold(cos), args());
  Ctan    := () -> dom::create("apply", "", "", hold(tan), args());
  Csec    := () -> dom::create("apply", "", "", hold(sec), args());
  Ccsc    := () -> dom::create("apply", "", "", hold(csc), args());
  Ccot    := () -> dom::create("apply", "", "", hold(cot), args());
  Csinh    := () -> dom::create("apply", "", "", hold(sinh), args());
  Ccosh    := () -> dom::create("apply", "", "", hold(cosh), args());
  Ctanh    := () -> dom::create("apply", "", "", hold(tanh), args());
  Csech    := () -> dom::create("apply", "", "", hold(sech), args());
  Ccsch    := () -> dom::create("apply", "", "", hold(csch), args());
  Ccoth    := () -> dom::create("apply", "", "", hold(coth), args());
  Carcsin    := () -> dom::create("apply", "", "", hold(arcsin), args());
  Carccos    := () -> dom::create("apply", "", "", hold(arccos), args());
  Carctan    := () -> dom::create("apply", "", "", hold(arctan), args());
  Carcsec    := () -> dom::create("apply", "", "", hold(arcsec), args());
  Carccsc    := () -> dom::create("apply", "", "", hold(arccsc), args());
  Carccot    := () -> dom::create("apply", "", "", hold(arccot), args());
  Carcsinh   := () -> dom::create("apply", "", "", hold(arcsinh), args());
  Carccosh   := () -> dom::create("apply", "", "", hold(arccosh), args());
  Carctanh   := () -> dom::create("apply", "", "", hold(arctanh), args());
  Carcsech   := () -> dom::create("apply", "", "", hold(arcsech), args());
  Carccsch   := () -> dom::create("apply", "", "", hold(arccsch), args());
  Carccoth   := () -> dom::create("apply", "", "", hold(arccoth), args());

  // Statistics
  Cmean := () -> dom::create("mean", "", "", args());
  Csdev := () -> dom::create("sdev", "", "", args());
  Cvariance := () -> dom::create("variance", "", "", args());
  Cmedian := () -> dom::create("median", "", "", args());
  Cmode := () -> dom::create("mode", "", "", args());
  Cmoment := () -> dom::create("moment", "", "", args(3), args(1), args(2));
  Cmomentabout := () -> args();

  // Linear Algebra
  Cvector := () -> error("not yet implemented");
  Cmatrix :=
  proc()
    local cols, rows, Args, constr, tmp;
  begin
    Args := [args()];
    constr := "";
    if testtype(Args[1], DOM_SET) then
      tmp := table(Args[1]);
      if contains(tmp, "constructor") then
        constr := tmp["constructor"]
      end_if;
      delete Args[1];
    end_if;
    cols := nops(Args[1]);
    rows := nops(Args);
    if cols = 0 or rows = 0 then
      dom::create("matrixEmpty", "", constr);
    else
      dom::create("matrix",
                  expr2text(hold(Rows)=expr2text(rows), 
                            hold(Cols)=expr2text(cols)),
                  constr, op(_concat(op(Args))));
    end_if;
  end_proc;
  Cmatrixrow := () -> [args()];
  Cdeterminant := () -> dom::create("determinant", "", "", args());  
  Ctranspose := () -> dom::create("transpose", "", "", args());
  Cselector := () -> error("not yet implemented");
  Cscalarproduct := () -> dom::create("scalarproduct", "", "", args()); 
  Cvectorproduct := () -> dom::create("vectorproduct", "", "", args()); 
  Couterproduct := () -> error("not yet implemented");

    /*
     * semantic mapping elements
     */
  Cannotation := () -> error("not yet implemented");
  Csemantics := () -> error("not yet implemented");
    // should be Cannotation-xml;  but this is not valid MuPAD syntax
  CannotationXml := () -> error("not yet implemented");


    /*
     *  constants and symbol elements (all new in MathML 2.0)
     */
  Cintegers := () -> dom::create("integers", "", "", args());
  Creals    := () -> dom::create("reals", "", "", args());
  Crationals := () -> dom::create("rationals", "", "", args());
  Cnaturalnumbers := () -> dom::create("naturalnumbers", "", "", args());
  Ccomplexes := () -> dom::create("complexes", "", "", args());
  Cprimes := () -> dom::create("primes", "", "", args());
  Cexponentiale := () -> dom::create("exponentiale", "", "", args());
  Cimaginaryi := () -> dom::create("imaginaryi", "", "", args());
  Cnotanumber := () -> dom::create("notanumber", "", "", args());
  Ctrue := () -> dom::create("true", "", "", args());
  Cfalse := () -> dom::create("false", "", "", args());
  Cemptyset := () -> dom::create("emptyset", "", "", args());
  Cpi := () -> dom::create("pi", "", "", args());
  Ceulergamma := () -> dom::create("eulergamma", "", "", args());
  Cinfinity := () -> dom::create("infinity", "", "", args());


    /*
     *  special MuPAD constructs
     */
  Cabbreviation :=
  proc()
    local Args, i;
  begin
    Args := [args()];
    // The insertion of empty lines her must be consistent with the
    // back-translation method below and with output::subexpr!
    Args := [Args[1], (hold(` `), Args[1+i]) $ i = 1..nops(Args)-1];
    dom::create("abbreviation",
                expr2text(hold(Rows)="".nops(Args)), "", op(Args));
  end_proc;
  Cassign   := () -> dom::create("assign", "", "", args());
  CbesselI  := () -> dom::create("besselI", "", "", args(2), args(1));
  CbesselJ  := () -> dom::create("besselJ", "", "", args(2), args(1));
  CbesselK  := () -> dom::create("besselK", "", "", args(2), args(1));
  CbesselY  := () -> dom::create("besselY", "", "", args(2), args(1));
  Cbeta     := () -> dom::create("beta", "", "", args());
  Cbinomial := () -> dom::create("binomial", "", "", args()); 
// do not remove !!  Used in doc package
  Ccomprow  := () -> dom::create("comprow", "", "", args());
  Cconcat   := () -> dom::create("concat", "", "", args());
  Cdilog    := () -> dom::create("dilog", "", "", args());
  Cdirac    := () -> if args(0) = 1 then
                       dom::create("dirac", "", "", args());
                     elif args(0) = 2 then
                       if args(2) = 1 then
                         dom::create("dirac1", "", "", args(1));
                       elif args(2) = 2 then
                         dom::create("dirac2", "", "", args(1));
                       elif args(2) = 3 then
                         dom::create("dirac3", "", "", args(1));
                       else
                         dom::create("diracD", "", "", args());
                       end_if;
                     else
                       error("not implemented")
                     end_if;
  CellipticCE  := () -> dom::create("ellipticCE", "", "", args());
  CellipticCK  := () -> dom::create("ellipticCK", "", "", args());
  CellipticCPi := () -> dom::create("ellipticCPi", "", "", args());
  CellipticE   := () -> if args(0) = 1 then
                          dom::create("ellipticE", "", "", args());
                        else
                          dom::create("ellipticE2", "", "", args());
                        end_if;
  CellipticF   := () -> dom::create("ellipticF", "", "", args());
  CellipticK   := () -> dom::create("ellipticK", "", "", args());
  CellipticNome:= () -> dom::create("ellipticNome", "", "", args());
  CellipticPi  := () -> if args(0) = 2 then
                          dom::create("ellipticPi", "", "", args());
                        else
                          dom::create("ellipticPi2", "", "", args());
                        end_if;
  Cequiv       := () -> dom::create("equiv", "", "", args());
  CequivMod    := () -> dom::create("equivMod", "", "", args());  // doc stuff
  CevalAt      := () -> dom::create("evalAt", "", "", args());
// do not remove !!  Used in calc package
  // equation system;  will be translated into a set of equations
  CEQS := () -> dom::create("EQS",
                            expr2text(hold(Rows)=expr2text(args(0) div 2),
                                      hold(Cols)="2"),
                            "", args());
// do not remove !!  Used in doc package
  // equation system;  used to output a set of equations/ineqs
  CEQS3 := (rows) -> dom::create("EQS3",
                                 expr2text(hold(Rows)=expr2text(rows),
                                           hold(Cols)="3"),
                                 "", args(2..args(0)));
  Cexprseq        := () -> dom::create("seq", "", "", args());
  Cfactorial2     := () -> dom::create("factorial2", "", "", args());
  Cfenced         := () -> dom::create("fenced", "", "", args());
  Cfnest          := () -> dom::create("fnest", "", "", args());
  CfresnelC       := () -> dom::create("fresnelC", "", "", args());
  CfresnelS       := () -> dom::create("fresnelS", "", "", args());
  // do not remove; needed for doc
  CfuncDecl       := () -> dom::create("funcDecl", "", "", args());
  Cgamma          := () -> dom::create("gamma", "", "", args());
  Chtranspose     := () -> dom::create("htranspose", "", "", args());
  Chypergeom      := () -> dom::create("hypergeom", "", "", args());
  Cigamma         := () -> dom::create("igamma", "", "", args());
  Cindex          := () -> dom::create("index", "", "", args());
  CinverseMatrix  := () -> dom::create("inverseMatrix", "", "", args());
  CintPV          := proc(x)
                       local attrs, Args, nargs;
                     begin
                       if contains({DOM_LIST, DOM_SET}, x::dom) then
                         attrs := expr2text(op(x));
                         Args  := [args(2..args(0))];
                       else
                         attrs := "";
                         Args  := [args()];
                       end_if;
                       nargs := nops(Args);
                       dom::create("definiteintpv", "", attrs, Args[4], Args[1], Args[2], Args[3]);
                     end_proc;

  CjacobiAM       := () -> dom::create("jacobiAM", "", "", args());
  CjacobiCD       := () -> dom::create("jacobiCD", "", "", args());
  CjacobiCN       := () -> dom::create("jacobiCN", "", "", args());
  CjacobiCS       := () -> dom::create("jacobiCS", "", "", args());
  CjacobiDC       := () -> dom::create("jacobiDC", "", "", args());
  CjacobiDN       := () -> dom::create("jacobiDN", "", "", args());
  CjacobiDS       := () -> dom::create("jacobiDS", "", "", args());
  CjacobiNC       := () -> dom::create("jacobiNC", "", "", args());
  CjacobiND       := () -> dom::create("jacobiND", "", "", args());
  CjacobiNS       := () -> dom::create("jacobiNS", "", "", args());
  CjacobiSC       := () -> dom::create("jacobiSC", "", "", args());
  CjacobiSD       := () -> dom::create("jacobiSD", "", "", args());
  CjacobiSN       := () -> dom::create("jacobiSN", "", "", args());
  CjacobiZeta     := () -> dom::create("jacobiZeta", "", "", args());
  CkroneckerDelta := () -> dom::create("kroneckerDelta", "", "", args());
  ClambertW       := () -> dom::create("lambertW", "", "", args(2), args(1));
  Clngamma        := () -> dom::create("lngamma", "", "", args());
  CmeijerG        := () -> dom::create("meijerG", "", "", args());
  Cnorm := () -> if args(0) = 1 then
                   dom::create("infinitynorm", "", "", args());
                 elif args(2) = hold(Infinity) then
                   dom::create("infinitynorm", "", "", args(1));
                 elif args(2) = 1 then
                   dom::create("onenorm", "", "", args(1));
                 elif args(2) = hold(Frobenius) then
                   dom::create("norm", "", "", args(1));
                 else
                   dom::create("othernorm", "", "", args());
                 end_if;
  Cpochhammer     := () -> dom::create("pochhammer", "", "", args(1), args(2));
  Cpolylog        := () -> dom::create("polylog", "", "", args());
  Cpsi     := () -> if args(0) = 1 then
                       dom::create("psi", "", "", args());
                     elif args(0) = 2 then
                       if args(2) = 1 then
                         dom::create("psi1", "", "", args(1));
                       elif args(2) = 2 then
                         dom::create("psi2", "", "", args(1));
                       elif args(2) = 3 then
                         dom::create("psi3", "", "", args(1));
                       else
                         dom::create("psiD", "", "", args());
                       end_if;
                     else
                       error("not implemented")
                     end_if;
  CquadraticMean  := () -> dom::create("quadraticMean", "", "", args());
  Crange          := () -> dom::create("range", "", "", args()); 
  Crow      := () -> dom::create("row", "", "", args());
  Cseqgen   := () -> case args(0)
                       of 1 do
                         dom::create("seqgen", "", "", args());
                         break;
                       of 2 do
                         dom::create("seqgen2", "", "", args());
                         break;
                       of 3 do
                         dom::create("seqgen3", "", "", args());
                         break;
                     end_case;
  Cseqstep   := () -> case args(0)
                       of 2 do
                         dom::create("seqstep", "", "", args());
                         break;
                       of 3 do
                         dom::create("seqstep2", "", "", args());
                         break;
                       of 4 do
                         dom::create("seqstep3", "", "", args());
                         break;
                     end_case;
  Cseqin          := () -> dom::create("seqin", "", "", args());
  Cslot           :=  () -> dom::create("slot", "", "", args());
  Ctable := () -> dom::create("table",
    expr2text(hold(Rows)=expr2text((args(0) div 2)+1),
    hold(Cols)="2"), "", dom(hold(``)), dom(hold(``)), args());
  Ctableau :=
  proc(x)
    local options; 
  begin
    if args(0)>0 and x::dom = DOM_LIST then
      options := table(op(x));
    else
      error("Attributes missing");
    end;
    
    dom::create("tableau", expr2text(op(options)), "", args(2..args(0)));
  end_proc;
  Cunknown        := () -> dom::create("unknown", "", "", args());
  CUnion          := () -> dom::create("Union", "", "", args());
  Cunit           := () -> dom::create("unit", "", "", args());
  CwhittakerM     := () -> dom::create("whittakerM", "", "", args());
  CwhittakerW     := () -> dom::create("whittakerW", "", "", args());
  CwrightOmega    := () -> dom::create("wrightOmega", "", "", args());
  Czeta     := () -> if args(0) = 1 then
                       dom::create("zeta", "", "", args());
                     elif args(0) = 2 then
                       if args(2) = 1 then
                         dom::create("zeta1", "", "", args(1));
                       elif args(2) = 2 then
                         dom::create("zeta2", "", "", args(1));
                       elif args(2) = 3 then
                         dom::create("zeta3", "", "", args(1));
                       else
                         dom::create("zetaD", "", "", args());
                       end_if;
                     else
                       error("not implemented")
                     end_if;

  end_domain:


alias(CL = ContentLayout):
alias(OP = output::Priority):

Content::set("plus",
// layout for _plus is hard-coded
             FAIL,
             _plus, 
             proc(plusExpr, attributes : DOM_TABLE)
               name convert_plus;
               local i, signs;
               save XxX;
             begin
               signs := attributes[hold(Signs)];
               if signs = "-" then
                 generate::negate(Content::expr(op(plusExpr, 1)))
               else
                 subsop(XxX((if signs[i] = "-" then
                               generate::negate(Content::expr(op(plusExpr, i)))
                             else
                               Content::expr(op(plusExpr, i))
                             end_if) $ i = 1 .. nops(plusExpr)),
                        0=hold(_plus), Unsimplified);
               end_if;
             end_proc):

// template "negate" is unused
Content::set("negate", FAIL, _power,
             obj -> _negate(Content::expr(op(obj, 1)))):
Content::set("times",
             CL::content("id"="times", "keywd"="Multiplication",
                         "accel"="*", "arity"="2*", "prio"=OP::Mult,
                        CL::layout(
                                   CL::oprow(CL::oper("prio"=OP::Mult),
                                             CL::mo("form"="infix",
                                                 "&blank;")))),
             _mult):  /* ! keine Simplifizierung */
Content::set("divide",
             CL::content("id"="divide", "keywd"="Division", "accel"="/",
                        "arity"="2", "prio"=OP::Mult+1,
                        CL::layout(CL::mrow(CL::oper("prio"=OP::Mult, "idx"="0"),
                                            CL::mo("form"="infix", "lbadness"="20000", "rbadness"="20000", "/"),
                                            CL::oper("prio"=OP::Mult, "idx"="1"))),
                        CL::layout(CL::mfrac(CL::oper("idx"="0"),
                                             CL::oper("idx"="1")))),
             _divide,
             proc(x)
               local num, den;
             begin
               num := Content::expr(op(x, 1));
               den := Content::expr(op(x, 2));
               if map({num, den}, domtype) = {DOM_INT} then
                 num/den;
               else
                 hold(_divide)(num, den);
               end_if;
             end_proc):
Content::set("divideLinear",
             CL::content("id"="divideLinear", "keywd"="Division",
                        "arity"="2", "prio"=OP::Mult+1,
                        CL::layout(
                                   CL::oprow(CL::oper("prio"=OP::Mult),
                                             CL::mo("/")))),
             _divide,
             proc(x)
               local num, den;
             begin
               num := Content::expr(op(x, 1));
               den := Content::expr(op(x, 2));
               if map({num, den}, domtype) = {DOM_INT} then
                 num/den;
               else
                 hold(_divide)(num, den);
               end_if;
             end_proc):
Content::set("powerBigExpo",
             CL::content("id"="powerBigExpo", "keywd"="Exponentiation",
                        "accel"="^", "arity"="2", "prio"=OP::Power,
                         CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Power),
                                             CL::mo("form"="infix", "^"),
                                             CL::oper("idx"="1", "prio"=OP::Power))),
                         CL::layout(CL::msup(CL::oper("idx"="0", "prio"=OP::Power),
                                             CL::oper("idx"="1")))),
             _power, proc(x)
                       local expo, base;
                     begin
                       base := Content::expr(op(x, 1));
                       expo := Content::expr(op(x, 2));
                       if base = exp(1) then
                         exp(expo)
                       else
                         hold(_power)(base, expo)
                       end;
                     end):
Content::set("power",
             CL::content("id"="power", "keywd"="Exponentiation",
                         "arity"="2", "prio"=OP::Power,
                         CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Power),
                                             CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000"),
                                             CL::msup(
                                          CL::mo("form"="infix"),
                                          CL::oper("idx"="1")))),
                         CL::layout(CL::msup(CL::oper("idx"="0", "prio"=OP::Power),
                                             CL::oper("idx"="1")))),
             _power, proc(x)
                       local expo, base;
                     begin
                       base := Content::expr(op(x, 1));
                       expo := Content::expr(op(x, 2));
                       if base = exp(1) then
                         exp(expo)
                       else
                         hold(_power)(base, expo)
                       end;
                     end):
Content::set("DOM_INT", FAIL, DOM_INT, x -> x):
Content::set("DOM_IDENT", FAIL, DOM_IDENT, x -> x):
Content::set("DOM_FLOAT", FAIL, DOM_FLOAT, x -> x):
Content::set("DOM_STRING", FAIL, DOM_STRING, x -> x):
Content::set("list",
             CL::content("id"="list", "keywd"="List", "accel"="[",
                        "arity"="0*", "dummies"="1", "flatten-seq"="0,-1",
                    CL::layout(
                          CL::mrow(
                              CL::mo("form"="prefix", "rbadness"="50000", "["),
                              CL::oprow(CL::oper("parens"="true"),
                                        CL::mo("form"="infix", "lbadness"="50000", "rbadness"="1", ",")),
                              CL::mo("form"="postfix", "lbadness"="50000", "]")))),
             DOM_LIST,
             proc(x)
               local i;
             begin
               [Content::expr(op(x, i)) $ i = 1 ..nops(x)]
             end_proc):
Content::set("set",
             CL::content("id"="set", "keywd"="Set", "accel"="{",
                        "arity"="0*", "dummies"="1", "flatten-seq"="0,-1",
                    CL::layout(
                          CL::mrow(
                              CL::mo("form"="prefix", "rbadness"="50000", "{"),
                              CL::oprow(CL::oper("parens"="true"),
                                        CL::mo("form"="infix", "lbadness"="50000", "rbadness"="1", ",")),
                              CL::mo("form"="postfix", "lbadness"="50000", "}")))),
             DOM_SET,
             proc(x)
               local i;
             begin
               {Content::expr(op(x,i)) $ i = 1 .. nops(x)}
             end_proc):
// do not remove !!  Used in calc package
Content::set("EQS",
             CL::content("id"="EQS", "keywd"="Equation System", "arity"="2*",
                          "assoc"="false",
                  CL::layout(
                         CL::optable("cols"="2",
                                     "columnalign"="rl",
                                     CL::oper(),
                                     CL::mrow(CL::mo("="),
                                              CL::oper())))),
             DOM_LIST,
             proc(x)
               local i;
             begin
               [Content::expr(op(x,2*i-1)) =
                    Content::expr(op(x,2*i))$ i = 1 .. nops(x)/2]
             end_proc):
// do not remove !!  Used in doc package
Content::set("EQS3",
             CL::content("id"="EQS3", "keywd"="General Equation System",
                         "arity"="3*",
                         "assoc"="false",
                    CL::layout(
                          CL::mrow(
                                   CL::optable("cols"="3",
                                               "columnalign"="rcl",
                                               CL::oper())))),
             eqsys,
             () -> error("\"EQS3\" -> MuPAD: not yet implemented")):
Content::set("diff",
             CL::content("id"="diff", "keywd"="Derivative",
                        "arity"="2", "prio"=OP::Mult,
                  CL::layout(
                      CL::mrow(
                          CL::mfrac(CL::mo("form"="prefix",
                                           "&DifferentialD;"),
                                    CL::mrow(CL::mo("form"="prefix",
                                                    "&DifferentialD;"),
                                             CL::oper("idx"="1"))),
                          CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&NonBreakingSpace;"),
                          CL::oper("idx"="0",
                                   "prio"=OP::Mult)))),
             diff):
Content::set("partialdiff",
             CL::content("id"="partialdiff", "keywd"="Partial Derivative",
                        "arity"="2", "prio"=OP::Mult + 10,
                  CL::layout(
                      CL::mrow(
                          CL::mfrac(CL::mo("form"="prefix",
                                           "&PartialD;"),
                                    CL::mrow(CL::mo("form"="prefix",
                                                    "&PartialD;"),
                                             CL::oper("idx"="1"))),
                          CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&NonBreakingSpace;"),
                          CL::oper("idx"="0",
                                   "prio"=OP::Mult)))),
             diff):
Content::set("partialdiff1",
             CL::content("id"="partialdiff1", "keywd"="Partial Derivative",
                        "arity"="3", "prio"=OP::Mult + 10,
                  CL::layout(
                      CL::mrow(
                          CL::mfrac(
                              CL::msup(CL::mo("form"="prefix", "&PartialD;"),
                                       CL::oper("idx"="2")),
                              CL::mrow(CL::mo("form"="prefix", "&PartialD;"),
                                       CL::msup(CL::oper("idx"="1"),
                                                CL::oper("idx"="2")))),
                          CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&NonBreakingSpace;"),
                          CL::oper("idx"="0", "prio"=OP::Mult)))),
             diff, x -> hold(diff)(Content::expr(op(x,1)), Content::expr(op(x,2)) $ Content::expr(op(x, 3)))):
// template is obsolete, but kept for backward compatibility
Content::set("partialdiff2",
             CL::content("id"="partialdiff2", "keywd"="Partial Derivative",
                        "arity"="2", "prio"=OP::Mult + 10,
               CL::layout(
                 CL::mrow(
                   CL::msub(CL::mo("form"="prefix", "&PartialD;"),
                            CL::oper("idx"="1")),
                   CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&NonBreakingSpace;"),
                   CL::mo("form"="prefix", "lbadness"="50000", "("),
                   CL::oper("idx"="0", "prio"=OP::Mult),
                   CL::mo("form"="postfix", "lbadness"="50000", ")")))),
             diff):
Content::set("eq",
             CL::content("id"="eq", "keywd"="Equal", "accel"="=",
                         "arity"="2*", "prio"=OP::Relation,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Relation),
                                              CL::mo("form"="infix", "&Equal;")))),
             _equal):
Content::set("approx",
             CL::content("id"="approx", "keywd"="Approximation",
                         "arity"="2*", "accel"="~", "prio"=OP::Relation,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Relation),
                                              CL::mo("form"="infix", "&approx;")))),
             _approx):
Content::set("lt",
             CL::content("id"="lt", "keywd"="Less", "accel"="&lt;",
                         "arity"="2*", "prio"=OP::Relation,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Relation),
                                              CL::mo("form"="infix", "&lt;")))),
             _less):
Content::set("leq",
             CL::content("id"="leq", "keywd"="LessEqual", 
                         "arity"="2*", "prio"=OP::Relation,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Relation),
                                              CL::mo("form"="infix", "&leq;")))),
             _leequal):
Content::set("neq", CL::content("id"="neq", "keywd"="NotEqual", 
                        "arity"="2*", "prio"=OP::Relation,
                        CL::layout(CL::oprow(CL::oper("prio"=OP::Relation),
                                              CL::mo("form"="infix",
                                                     "&NotEqual;")))),
             _unequal):
Content::set("gt", CL::content("id"="gt", "keywd"="Greater", "accel"="&gt;",
                        "arity"="2*", "prio"=OP::Relation,
                        CL::layout(CL::oprow(CL::oper("prio"=OP::Relation),
                                             CL::mo("form"="infix", "&gt;")))),
             _greater):
Content::set("geq", CL::content("id"="geq", "keywd"="GreaterEqual", 
                        "arity"="2*", "prio"=OP::Relation,
                        CL::layout(CL::oprow(CL::oper("prio"=OP::Relation),
                                             CL::mo("form"="infix", "&geq;")))),
             _geequal):
Content::set("and",
             CL::content("id"="and", "keywd"="And",
                        "arity"="2*", "prio"=OP::And,
                        CL::layout(CL::oprow(CL::oper("prio"=OP::And),
                                             CL::mo("&wedge;")))),
             _and):
Content::set("or",
             CL::content("id"="or", "keywd"="Or",
                        "arity"="2*", "prio"=OP::Or,
                        CL::layout(CL::oprow(CL::oper("prio"=OP::And),
                                             CL::mo("&vee;")))),
             _or):
Content::set("xor",
             CL::content("id"="xor", "keywd"="Xor",
                        "arity"="2*", "prio"=OP::Xor,
                        CL::layout(CL::oprow(CL::oper("prio"=OP::And),
                                             CL::mo("xor")))),
             _xor):
Content::set("not",
             CL::content("id"="not", "keywd"="Not",
                         "arity"="1", "prio"=OP::Not,
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "rbadness"="50000",
                                                    "&not;"),
                                          CL::oper("idx"="0", "prio"=OP::Not)))),
             _not):
Content::set("implies",
             CL::content("id"="implies", "keywd"="Implies",
                        "arity"="2", "prio"=OP::Implies,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Implies),
                                              CL::mo("form"="infix",
                                                     "&DoubleRightArrow;")))),
             _implies):
Content::set("equiv",
             CL::content("id"="equiv", "keywd"="Equivalence",
                        "arity"="2", "prio"=OP::Equiv,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Equiv),
                                              CL::mo("form"="infix",
                                                     "&DoubleLeftRightArrow;")))),
             _equiv):
// for documentation, not checked
Content::set("equivMod",
             CL::content("id"="equivMod", "keywd"="Equivalence Relation modulo",
                        "arity"="3", "prio"=OP::Exprseq,
                        CL::layout(CL::mrow(CL::oper("idx"="0"),
                                            CL::mo("lbadness"="50000", "rbadness"="50000",
                                                   "&blank;&Congruent;&blank;"),
                                            CL::oper("idx"="1"),
                                            CL::mo("lbadness"="50000", "rbadness"="50000",
                                                   "&blank;&blank;(mod&blank;"),
                                            CL::oper("idx"="2"),
                                            CL::mo("lbadness"="50000", ")")))),
             ()->error("no backtranslation for output stuff")):
Content::set("union",
              CL::content("id"="union", "keywd"="Union",
                          "arity"="2*", "prio"=OP::Union,
                          CL::layout(CL::oprow(CL::oper("prio"=OP::Intersect),
                                               CL::mo("&cup;")))),
             _union):
Content::set("Union",
              CL::content("id"="Union", "keywd"="UnionSet",
                          "arity"="3", "prio"=OP::Union,
                    CL::layout(
                          CL::mrow(
                              CL::mo("form"="prefix", "&Union;"),
                              CL::mrow(
                                 CL::mo("form"="prefix", "lbadness"="50000", "rbadness"="10000", "("),
                                 CL::oper("idx"="0", "prio"=OP::Intersect),
                                 CL::mo("form"="infix", "lbadness"="50000", "|"),
                                 CL::mrow(CL::oper("idx"="1", "prio"=OP::Relation),
                                       CL::mo("form"="infix", "&Element;"),
                                       CL::oper("idx"="2", "prio"=OP::Relation)),
                                 CL::mo("form"="postfix", "lbadness"="10000", ")")))),
                  CL::layout(
                      CL::mrow(
                          CL::munder(
                              CL::mo("form"="prefix", "&Union;"),
                              CL::mrow(CL::oper("idx"="1", "prio"=OP::Relation),
                                       CL::mo("form"="infix", "&Element;"),
                                       CL::oper("idx"="2", "prio"=OP::Relation))),
                          CL::mo("lbadness"="50000", "rbadness"="50000", "&NonBreakingSpace;"),
                          CL::oper("idx"="0", "prio"=OP::Intersect)))),
             solvelib::Union):
Content::set("intersect",
              CL::content("id"="intersect", "keywd"="Intersection",
                          "arity"="2*", "prio"=OP::Intersect,
                          CL::layout(CL::oprow(CL::oper("prio"=OP::Intersect),
                                               CL::mo("&cap;")))),
             _intersect):
Content::set("setdiff",
              CL::content("id"="setdiff", "keywd"="SetMinus",
                          "arity"="2", "prio"=OP::Minus,
                          CL::layout(CL::oprow(CL::oper("prio"=OP::Intersect),
                                               CL::mo("&setminus;")))),
             _minus):
Content::set("card",
             CL::content("id"="card", "keywd"="Cardinality", "arity"="1", 
                         CL::layout(CL::mrow(CL::mo("form"="prefix",
                                                    "&verbar;"),
                                             CL::oper("idx"="0"),
                                             CL::mo("form"="postfix",
                                                    "&verbar;")))),
             card):
Content::set("subset",
             CL::content("id"="subset", "keywd"="Subset",
                        "arity"="2", "prio"=OP::Relation,
                        CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Relation),
                                            CL::mo("form"="infix",
                                                   "&Subset;"),
                                            CL::oper("idx"="1", "prio"=OP::Relation)
                                            ))),
             _subset):
Content::set("prsubset",
             CL::content("id"="prsubset", "keywd"="Proper Subset",
                        "arity"="2", "prio"=OP::Relation,
                        CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Relation),
                                            CL::mo("form"="infix",
                                                   "&Subset;"),
                                            CL::oper("idx"="1", "prio"=OP::Relation)
                                            ))),
             _prsubset):
Content::set("notsubset",
             CL::content("id"="notsubset", "keywd"="NotSubset",
                        "arity"="2", "prio"=OP::Relation,
                        CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Relation),
                                            CL::mo("form"="infix",
                                                   "&NotSubset;"),
                                            CL::oper("idx"="1", "prio"=OP::Relation)
                                            ))),
             _notsubset):
Content::set("notprsubset",
             CL::content("id"="notprsubset", "keywd"="Not Proper Subset",
                        "arity"="2", "prio"=OP::Relation,
                        CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Relation),
                                            CL::mo("form"="infix",
                                                   "&NotSubset;"),
                                            CL::oper("idx"="1", "prio"=OP::Relation)
                                            ))),
             _notprsubset):
Content::set("exp",
             CL::content("id"="exp", "keywd"="Exponential",
                         "arity"="1", "prio"=OP::Power,
                         CL::layout(CL::mrow(CL::mo("&ee;"),
                                             CL::mo("form"="infix", "^"),
                                             CL::oper("idx"="0", "prio"=OP::Power))),
                         CL::layout(CL::msup(CL::mo("&ee;"),
                                             CL::oper("idx"="0")))),
             exp):
Content::set("lg",
             CL::content("id"="lg", "keywd"="Logarithm10", "arity"="1", 
                        CL::layout(CL::mrow(CL::msub(CL::mo("log"),
                                                   CL::mo("10")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             lg):  // defined in package calc
Content::set("log",
             CL::content("id"="log", "keywd"="Logarithm", "arity"="2", 
                        CL::layout(CL::mrow(CL::msub(CL::mo("log"),
                                                   CL::oper("idx"="1")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             log,
             X->hold(log)(Content::expr(op(X,2)), Content::expr(op(X,1)))):
Content::set("integers",
             CL::content("id"="integers", "keywd"="Integers", "arity"="0", 
                         CL::layout(CL::mo("&Zopf;"))),
             Z_, ()->Z_):
Content::set("reals",
             CL::content("id"="reals", "keywd"="Reals", "arity"="0", 
                         CL::layout(CL::mo("&Ropf;"))),
             R_, ()->R_):
Content::set("rationals",
             CL::content("id"="rationals", "keywd"="Rationals", "arity"="0", 
                         CL::layout(CL::mo("&Qopf;"))),
             Q_, ()->Q_):
Content::set("naturalnumbers",
             CL::content("id"="naturalnumbers", "keywd"="Natural Numbers",
                         "arity"="0", 
                         CL::layout(CL::mo("&Nopf;"))),
             hold(N_), ()->hold(N_)):  // dummy, no BasicSet
Content::set("complexes",
             CL::content("id"="complexes", "keywd"="Complexes", "arity"="0", 
                         CL::layout(CL::mo("&Copf;"))),
             C_, ()->C_):
Content::set("primes",
             CL::content("id"="primes", "keywd"="Primes", "arity"="0", 
                         CL::layout(CL::mo("&Popf;"))),
             hold(P_), ()->hold(P_)):        // dummy, no BasicSet
Content::set("exponentiale",
             CL::content("id"="exponentiale", "keywd"="Exponential e",
                         "arity"="0", 
                         CL::layout(CL::mo("&ee;"))),
             exp(1), ()->exp(1)):
Content::set("imaginaryi",
             CL::content("id"="imaginaryi", "keywd"="Imaginary I",
                         "arity"="0", 
                         CL::layout(CL::mo("&ii;"))),
             I, ()->I):
Content::set("notanumber",
             CL::content("id"="notanumber", "keywd"="Not a Number",
                         "arity"="0", 
                         CL::layout(CL::mo("&NotANumber;"))),
             RD_NAN, ()->RD_NAN):
Content::set("true",
             CL::content("id"="true", "keywd"="True", "arity"="0", 
                         CL::layout(CL::mo("TRUE"))),
             TRUE, ()->TRUE):
Content::set("false",
             CL::content("id"="false", "keywd"="False", "arity"="0", 
                         CL::layout(CL::mo("FALSE"))),
             FALSE, ()->FALSE):
Content::set("unknown",
             CL::content("id"="unknown", "keywd"="Unknown", "arity"="0", 
                         CL::layout(CL::mo("UNKNOWN"))),
             UNKNOWN, ()->UNKNOWN):
Content::set("emptyset",
             CL::content("id"="emptyset", "keywd"="Empty Set", "arity"="0", 
                         CL::layout(CL::mo("&emptyv;"))),
             {}, ()->{}):
Content::set("pi",
             CL::content("id"="pi", "keywd"="Pi", "arity"="0", 
                         CL::layout(CL::mo("&pi;"))),
             PI, ()->PI):
Content::set("eulergamma",
             CL::content("id"="eulergamma", "keywd"="Euler Gamma",
                         "arity"="0", 
                         CL::layout(CL::mo("EULER"))),
             EULER, ()->EULER):
Content::set("infinity",
             CL::content("id"="infinity", "keywd"="Infinity", "arity"="0", 
                         CL::layout(CL::mo("&infin;"))),
             infinity, ()->infinity):
Content::set("conjugate",
             CL::content("id"="conjugate", "keywd"="Conjugate", "arity"="1", 
                      CL::layout(CL::mover(CL::oper("idx"="0"),
                                           CL::mo("form"="postfix",
                                                  "&OverBar;")))),
             conjugate):
Content::set("definiteint",
             CL::content("id"="definiteint", "keywd"="Definite Integration",
                         "accel"=":", "arity"="4",
               CL::layout(
                   CL::mrow(
                       CL::munderover(CL::mo("form"="prefix", "&Integral;"),
                                      CL::oper("idx"="2"),
                                      CL::oper("idx"="3")),
                       CL::mo("lbadness"="50000", "rbadness"="50000"),
                       CL::oper("idx"="0", "prio"=OP::Plus),
                       CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&DifferentialD;"),
                       CL::oper("idx"="1")))),
             int,
             x -> hold(int)(Content::expr(op(x,1)),
                            Content::expr(op(x,2))=Content::expr(op(x,3)) ..
                                                   Content::expr(op(x,4)),
                            text2expr(Content::infos(x)))):
Content::set("definiteintpv",
             CL::content("id"="definiteintpv", "keywd"="Definite Integration (Principal Value)",
                         "arity"="4",
               CL::layout(
                   CL::mrow(
                       CL::munderover(CL::mo("form"="prefix", "&pvint;"),
                                      CL::oper("idx"="2"),
                                      CL::oper("idx"="3")),
                       CL::mo("lbadness"="50000", "rbadness"="50000"),
                       CL::oper("idx"="0", "prio"=OP::Plus),
                       CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&DifferentialD;"),
                       CL::oper("idx"="1")))),
             int,
             proc(x)
               local options;
             begin
               options := text2expr(Content::infos(x));
               // just in case this template is used in the editor
               if not contains({options}, PrincipalValue) then
                 options := options, PrincipalValue
               end_if;
               hold(int)(Content::expr(op(x,1)),
                         Content::expr(op(x,2))=Content::expr(op(x,3)) ..
                                                Content::expr(op(x,4)),
                         options)
             end):
Content::set("intdomain",
             CL::content("id"="intdomain", "keywd"="Integration over a Domain",
                         "arity"="3",
               CL::layout(
                   CL::mrow(
                       CL::munder(CL::mo("form"="prefix", "&Integral;"),
                                  CL::oper("idx"="1")),
                       CL::mo("lbadness"="50000", "rbadness"="50000"),
                       CL::oper("idx"="0", "prio"=OP::Plus),
                       CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&DifferentialD;"),
                       CL::oper("idx"="2")))),
             int,
             x -> hold(intlib::intOverSet)(Content::expr(op(x,1)),
                                           Content::expr(op(x,3)),
                                           op(Content::expr(op(x,2)), 2),
                                           text2expr(Content::infos(x)))):
Content::set("int",
             CL::content("id"="int", "keywd"="Indefinite Integration",
                         "arity"="2",
               CL::layout(
                   CL::mrow(CL::mo("form"="prefix", "&Integral;"),
                            CL::mo("lbadness"="50000", "rbadness"="50000"),
                            CL::oper("idx"="0", "prio"=OP::Plus),
                            CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&DifferentialD;"),
                            CL::oper("idx"="1")))),
             int,
             x -> hold(int)(Content::expr(op(x,1)),
                            Content::expr(op(x,2)),
                            text2expr(Content::infos(x)))):
Content::set("definitesum",
             CL::content("id"="definitesum", "keywd"="Definite Sum",
                        "accel"=";", "arity"="4", "prio"=OP::Plus,
                  CL::layout(
                      CL::mrow(
                          CL::munderover(
                              CL::mo("form"="prefix", "&Sum;"),
                              CL::mrow(CL::oper("idx"="1", "prio"=OP::Relation),
                                       CL::mo("form"="infix", "="),
                                       CL::oper("idx"="2", "prio"=OP::Relation)),
                              CL::oper("idx"="3")),
                          CL::mo("lbadness"="50000", "rbadness"="50000"),
                          CL::oper("idx"="0", "prio"=OP::Plus)))),
             sum,
             x -> hold(sum)(Content::expr(op(x,1)),
                            Content::expr(op(x,2))=Content::expr(op(x,3)) ..
                                                    Content::expr(op(x,4)))):
Content::set("sum",
             CL::content("id"="sum", "keywd"="Indefinite Sum",
                         "arity"="2", "prio"=OP::Plus,
                  CL::layout(
                      CL::mrow(
                          CL::munder(
                              CL::mo("form"="prefix", "&Sum;"),
                              CL::oper("idx"="1")),
                          CL::mo("lbadness"="50000", "rbadness"="50000"),
                          CL::oper("idx"="0", "prio"=OP::Plus)))),
             sum,
             x -> hold(sum)(Content::expr(op(x,1)),
                            Content::expr(op(x,2)))):
Content::set("sumrootof",
             CL::content("id"="sumrootof", "keywd"="Definite Sum over RootOf",
                         "arity"="3", "prio"=OP::Plus,
                  CL::layout(
                      CL::mrow(
                          CL::munder(
                              CL::mo("form"="prefix", "&Sum;"),
                              CL::mrow(CL::oper("idx"="1", "prio"=OP::Relation),
                                       CL::mo("form"="infix", "="),
                                       CL::oper("idx"="2", "prio"=OP::Relation))),
                          CL::mo("lbadness"="50000", "rbadness"="50000"),
                          CL::oper("idx"="0", "prio"=OP::Plus)))),
             sum,
             x -> hold(sum)(Content::expr(op(x,1)),
                            Content::expr(op(x,2))=Content::expr(op(x,3)))):
Content::set("definiteproduct",
             CL::content("id"="definiteproduct", "keywd"="Definite Product",
                         "arity"="4", "prio"=OP::Mult,
                  CL::layout(
                      CL::mrow(
                          CL::munderover(
                              CL::mo("form"="prefix", "&prod;"),
                              CL::mrow(CL::oper("idx"="1", "prio"=OP::Relation),
                                       CL::mo("form"="infix", "="),
                                       CL::oper("idx"="2", "prio"=OP::Relation)),
                              CL::oper("idx"="3")),
                          CL::mo("lbadness"="50000", "rbadness"="50000"),
                          CL::oper("idx"="0", "prio"=OP::Mult)))),
             product,
             x -> hold(product)(Content::expr(op(x,1)),
                                Content::expr(op(x,2))=
                                  Content::expr(op(x,3)) ..
                                    Content::expr(op(x,4)))):
Content::set("product",
             CL::content("id"="product", "keywd"="Indefinite Product",
                         "arity"="2", "prio"=OP::Mult,
                  CL::layout(
                      CL::mrow(
                          CL::munder(
                              CL::mo("form"="prefix", "&prod;"),
                              CL::oper("idx"="1")),
                              CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&NonBreakingSpace;"),
                          CL::mo("lbadness"="50000", "rbadness"="50000"),
                          CL::oper("idx"="0", "prio"=OP::Mult)))),
             product,
             x -> hold(product)(Content::expr(op(x,1)),
                                Content::expr(op(x,2)))):
Content::set("productrootof",
             CL::content("id"="productrootof", "keywd"="Definite Product over RootOf",
                         "arity"="3", "prio"=OP::Mult,
                  CL::layout(
                      CL::mrow(
                          CL::munder(
                              CL::mo("form"="prefix", "&prod;"),
                              CL::mrow(CL::oper("idx"="1", "prio"=OP::Relation),
                                       CL::mo("form"="infix", "="),
                                       CL::oper("idx"="2", "prio"=OP::Relation))),
                          CL::mo("lbadness"="50000", "rbadness"="50000"),
                          CL::oper("idx"="0", "prio"=OP::Mult)))),
             sum,
             x -> hold(product)(Content::expr(op(x,1)),
                            Content::expr(op(x,2))=Content::expr(op(x,3)))):
Content::set("root",
             CL::content("id"="root", "keywd"="Root", "arity"="2",
                         CL::layout(CL::mrow(CL::mo("surd"),
                              CL::mrow(
                                 CL::mo("form"="prefix", "lbadness"="50000", "("),
                                 CL::oprow(CL::oper(),
                                           CL::mo("form"="infix", "lbadness"="50000", ",")),
                                 CL::mo("form"="postfix", ")")))),
                         CL::layout(CL::mroot(CL::oper("idx"="0"),
                                              CL::oper("idx"="1")))),
             surd, 
             x -> hold(surd)(Content::expr(op(x,1)),
                             Content::expr(op(x,2)))):
Content::set("sqrt",
             CL::content("id"="sqrt", "keywd"="Square Root", "arity"="1",
                         CL::layout(CL::mrow(CL::mo("sqrt"),
                              CL::mrow(
                                 CL::mo("form"="prefix", "lbadness"="50000", "("),
                                 CL::oper("idx"="0"),
                                 CL::mo("form"="postfix", ")")))),
                         CL::layout(CL::msqrt(CL::oper("idx"="0")))),
             _power, 
             x -> hold(_power)(Content::expr(op(x)), 1/2)):
Content::set("limit",
             CL::content("id"="limit", "keywd"="Limit",
                         "arity"="3", "prio"=OP::Plus - 50,
                  CL::layout(
                      CL::mrow(
                          CL::munder(
                              CL::mo("form"="prefix", "lim"),
                              CL::mrow(CL::oper("idx"="1", "prio"=OP::Stmtseq),
                                       CL::mo("form"="infix", "&RightArrow;"),
                                       CL::oper("idx"="2", "prio"=OP::Stmtseq))),
                      CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&NonBreakingSpace;"),
                      CL::oper("idx"="0", "prio"=OP::Plus - 50)))),
             limit, 
             (x, attributes) ->
                hold(limit)(Content::expr(op(x,1)),
                            Content::expr(op(x,2))=Content::expr(op(x,3)))):
Content::set("limitRight",
             CL::content("id"="limitRight", "keywd"="Right Limit",
                         "arity"="3", "prio"=OP::Plus - 50,
                  CL::layout(
                      CL::mrow(
                          CL::munder(
                              CL::mo("form"="prefix", "lim"),
                              CL::mrow(CL::oper("idx"="1", "prio"=OP::Stmtseq),
                                       CL::mo("form"="infix", "&RightArrow;"),
                                       CL::msup(CL::oper("idx"="2",
                                                         "prio"=OP::Stmtseq),
                                                CL::msub(CL::mo(), CL::mo("+"))))),
                          CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&NonBreakingSpace;"),
                          CL::oper("idx"="0", "prio"=OP::Plus - 50)))),
             limit, 
             (x, attributes) ->
               hold(limit)(Content::expr(op(x,1)),
                           Content::expr(op(x,2))=Content::expr(op(x,3)),
                           hold(Right))):
Content::set("limitLeft",
             CL::content("id"="limitLeft", "keywd"="Left Limit",
                         "arity"="3", "prio"=OP::Plus - 50,
                  CL::layout(
                      CL::mrow(
                          CL::munder(
                              CL::mo("form"="prefix", "lim"),
                              CL::mrow(CL::oper("idx"="1", "prio"=OP::Stmtseq),
                                       CL::mo("form"="infix", "&RightArrow;"),
                                       CL::msup(CL::oper("idx"="2",
                                                         "prio"=OP::Stmtseq),
                                                CL::msub(CL::mo(), CL::mo("-"))))),
                          CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000", "&NonBreakingSpace;"),
                          CL::oper("idx"="0", "prio"=OP::Plus - 50)))),
             limit, 
             (x, attributes) ->
                hold(limit)(Content::expr(op(x,1)),
                            Content::expr(op(x,2))=Content::expr(op(x,3)),
                            hold(Left))):
Content::set("fenced",
             CL::content("id"="fenced", "keywd"="Fence",
                         "accel"=")", "arity"="1",
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "("),
                                             CL::oper("idx"="0"),
                                             CL::mo("form"="postfix", ")")))),
             id, x -> Content::expr(op(x,1))):
Content::set("factorial",
             CL::content("id"="factorial", "keywd"="Factorial",
                         "accel"="!", "arity"="1", "prio"=OP::Fact,
                      CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Fact),
                                          CL::mo("form"="postfix", "!")))),
             fact):
Content::set("factorial2",
             CL::content("id"="factorial2", "keywd"="Double Factorial Function",
                         "arity"="1", "prio"=OP::Fact,
                      CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Fact),
                                          CL::mo("form"="postfix", "!!")))),
             fact2):
Content::set("in",
             CL::content("id"="in", "keywd"="In",
                        "arity"="2", "prio"=OP::Relation,
                        CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Relation),
                                            CL::mo("form"="infix",
                                                   "&Element;"),
                                            CL::oper("idx"="1", "prio"=OP::Relation)
                                            ))),
             _in):
Content::set("notin",
             CL::content("id"="notin", "keywd"="Not In",
                        "arity"="2", "prio"=OP::Relation,
                        CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Relation),
                                            CL::mo("form"="infix",
                                                   "&NotElement;"),
                                            CL::oper("idx"="1", "prio"=OP::Relation)
                                            ))),
             _notin,
             proc(x)
               local ex1, ex2;
             begin
               ex1 := Content::expr(op(x, 1));
               ex2 := Content::expr(op(x, 2));
               hold(_not)(hold(_in)(ex1, ex2))
             end_proc):
Content::set("abs",
             CL::content("id"="abs", "keywd"="Absolute Value",
                         "accel"="|", "arity"="1", 
                         CL::layout(CL::mrow(CL::mo("form"="prefix",
                                                   "&verbar;"),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="postfix",
                                                   "&verbar;")))),
             abs):
Content::set("norm",
             CL::content("id"="norm", "keywd"="Frobenius Norm", "arity"="1",
                         CL::layout(CL::mrow(CL::mo("form"="prefix",
                                                   "&Verbar;"),
                                             CL::oper("idx"="0"),
                                             CL::mo("form"="postfix",
                                                   "&Verbar;")))),
             norm,
             proc(x)
               local ex, evex;
             begin
               ex := Content::expr(op(x));
               evex := eval(ex);
               if evex::dom::hasProp(Cat::Matrix) = TRUE then
                  hold(norm)(ex, Frobenius);
               else
                  hold(norm)(ex, 2);
               end_if;
             end_proc):
Content::set("onenorm",
             CL::content("id"="onenorm", "keywd"="1-Norm", "arity"="1",
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "&Verbar;"),
                                             CL::oper("idx"="0"),
                                             CL::mo("lbadness"="50000", "rbadness"="50000"),
                                             CL::msub(CL::mo("form"="postfix",
                                                             "&Verbar;"),
                                                      CL::mo("1")))),
                         CL::layout(CL::msub(CL::mrow(CL::mo("form"="prefix",
                                                             "&Verbar;"),
                                                      CL::oper("idx"="0"),
                                                      CL::mo("form"="postfix",
                                                             "&Verbar;")),
                                             CL::mo("1")))),
             norm,
             x -> hold(norm)(Content::expr(op(x)), 1)):
Content::set("infinitynorm",
             CL::content("id"="infinitynorm", "keywd"="Infinity Norm",
                         "arity"="1",
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "&Verbar;"),
                                             CL::oper("idx"="0"),
                                             CL::mo("lbadness"="50000", "rbadness"="50000"),
                                             CL::msub(CL::mo("form"="postfix",
                                                             "&Verbar;"),
                                                      CL::mo("&infin;")))),
                         CL::layout(CL::msub(CL::mrow(CL::mo("form"="prefix",
                                                             "&Verbar;"),
                                                      CL::oper("idx"="0"),
                                                      CL::mo("form"="postfix",
                                                             "&Verbar;")),
                                             CL::mo("&infin;")))),
             norm):
Content::set("othernorm",
             CL::content("id"="othernorm", "keywd"="Other Norm",
                         "arity"="2",
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "&Verbar;"),
                                             CL::oper("idx"="0"),
                                             CL::mo("lbadness"="50000", "rbadness"="50000"),
                                             CL::msub(CL::mo("form"="postfix",
                                                             "&Verbar;"),
                                                      CL::oper("idx"="1")))),
                         CL::layout(CL::msub(CL::mrow(CL::mo("form"="prefix",
                                                             "&Verbar;"),
                                                      CL::oper("idx"="0"),
                                                      CL::mo("form"="postfix",
                                                             "&Verbar;")),
                                             CL::oper("idx"="1")))),
                         norm):
Content::set("determinant",
             CL::content("id"="determinant", "keywd"="Determinant",
                         "arity"="1", 
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "rbadness"="50000",
                                                    "det"),
                                             CL::oper("prio"=OP::Slot,
                                                      "idx"="0")))),
             det):
Content::set("ceil",
             CL::content("id"="ceil", "keywd"="Ceil", "arity"="1",
                         CL::layout(CL::mrow(CL::mo("form"="prefix",
                                                    "&LeftCeiling;"),
                                             CL::oper("idx"="0"),
                                             CL::mo("form"="postfix",
                                                    "&RightCeiling;")))),
             ceil):
Content::set("floor",
             CL::content("id"="floor", "keywd"="Floor", "arity"="1",
                         CL::layout(CL::mrow(CL::mo("form"="prefix",
                                                    "&LeftFloor;"),
                                             CL::oper("idx"="0"),
                                             CL::mo("form"="postfix",
                                                    "&RightFloor;")))),
             floor):
Content::set("rem",
             CL::content("id"="rem", "keywd"="Remainder",
                         "arity"="2", "prio"=OP::Mod,
                         CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Mod),
                                             CL::mo("mod"),
                                             CL::oper("idx"="1", "prio"=OP::Mod)
                                             ))),
             _mod):
Content::set("quotient",
             CL::content("id"="quotient", "keywd"="Integer Quotient",
                         "arity"="2", "prio"=OP::Mod,
                         CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Mod),
                                             CL::mo("div"),
                                             CL::oper("idx"="1", "prio"=OP::Mod)
                                             ))),
             _div):
Content::set("fnest",
             CL::content("id"="fnest", "keywd"="Nested Functions",
                         "arity"="2", "prio"=OP::Fnest,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Fnest),
                                              CL::mo("form"="infix",
                                                     "&commat;&commat;")))),
             _fnest):
Content::set("lambda",
             CL::content("id"="lambda", "keywd"="Function Definition",
                        "arity"="2", "prio"=OP::Exprseq,
                        CL::layout(
                                  CL::oprow(CL::oper("prio"=OP::Exprseq),
                                           CL::mo("form"="infix",
                                                 "&rightarrow;")))),
             procdef,
             proc(x)
               local tmp;
             begin
               tmp := "(".expr2text(Content::expr(op(x,1))).") -> (".
                      expr2text(Content::expr(op(x,2))).")";
               subsop(text2expr(tmp), 12=NIL)
             end_proc):
Content::set("funcDecl",
             CL::content("id"="funcDecl", "keywd"="Function Declaration",
                        "arity"="3", "prio"=OP::Exprseq,
                        CL::layout(CL::mrow(CL::oper("idx"="0"),
                                            CL::mo("lbadness"="50000", "rbadness"="5000", ":&blank;"),
                                            CL::oper("idx"="1"),
                                            CL::mo("lbadness"="5000", "rbadness"="6000", "&blank;&rightarrow;&blank;"),
                                            CL::oper("idx"="2")))),
             ()->error("no backtranslation for output stuff")):
Content::set("slot",
             CL::content("id"="slot", "keywd"="Slot", "arity"="2", 
                         CL::layout(CL::mrow(CL::oper("idx"="0"),
                                             CL::mo("lbadness"="50000", "rbadness"="50000", "&Colon;"),
                                             CL::oper("idx"="1")))),
             slot,
             x -> hold(slot)(Content::expr(op(x,1)), expr2text(op(x, 2)))):
Content::set("dilog",
              CL::content("id"="dilog", "keywd"="Dilog",
                        "arity"="1*", 
                        CL::layout(CL::mrow(CL::msub(CL::mo("Li"),
                                                    CL::mo("2")),
                                           CL::mo("form"="prefix","lbadness"="50000", "("),
                                           CL::oprow("idx"="0,-1",
                                                    CL::oper("prio"=OP::Stmtseq),
                                                    CL::mo("lbadness"="50000", ",")),
                                           CL::mo("form"="postfix",
                                                  ")")))),
             dilog):
Content::set("polylog",
             CL::content("id"="polylog", "keywd"="Polylog", "arity"="2",
                         CL::layout(CL::mrow(CL::mo("polylog"),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("lbadness"="50000", ","),
                                          CL::oper("idx"="1"),
                                          CL::mo("form"="postfix", ")")))),
             polylog):
Content::set("D",
              CL::content("id"="D", "keywd"="Differentiation Operator",
                        "accel"="'", "arity"="1", "prio"=OP::Prime,
                          CL::layout(CL::msup(CL::oper("prio"=OP::Prime-10, "idx"="0"),
                                              CL::mo("form"="postfix", "lbadness"="50000",
                                                     "&prime;")))),
             D):
Content::set("D2",
              CL::content("id"="D2", "keywd"="Differentiation Operator",
                        "arity"="1", "prio"=OP::Prime,
                          CL::layout(CL::msup(CL::oper("prio"=OP::Prime-10, "idx"="0"),
                                       CL::mo("form"="postfix", "lbadness"="50000", "&prime;&prime;")
                                      ))),
             D,
             (x) -> hold(D@@2)(Content::expr(op(x)))
):
Content::set("D3",
              CL::content("id"="D3", "keywd"="Differentiation Operator",
                        "arity"="1", "prio"=OP::Prime,
                          CL::layout(CL::msup(CL::oper("prio"=OP::Prime-10, "idx"="0"),
                                       CL::mo("form"="postfix", "lbadness"="50000", "&prime;&prime;&prime;")
                                      ))),
             D,
             (x) -> hold(D@@3)(Content::expr(op(x)))
):
Content::set("DN",
              CL::content("id"="DN", "keywd"="Differentiation Operator",
                        "arity"="2", "prio"=OP::Prime,
                        CL::layout(CL::msup(CL::oper("prio"=OP::Prime-10, "idx"="0"),
                                        CL::mrow(
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")
                                            )))),
             D,
             (x) -> hold(_fnest)(D, (Content::expr(op(x, 2))))(Content::expr(op(x, 1)))
):
Content::set("Dpara",
              CL::content("id"="Dpara", "keywd"="Differentiation Operator",
                        "arity"="2*",  "assoc"="false",
                   CL::layout(CL::mrow(CL::msub(CL::mo("form"="prefix", "D"),
                                               CL::oprow("idx"="0,-2",
                                                        CL::oper("prio"=OP::Stmtseq),
                                                        CL::mo("lbadness"="50000", ","))),
                                      CL::mo("form"="prefix","lbadness"="50000", "("),
                                      CL::oper("idx"="-1"),
                                      CL::mo("form"="postfix",
                                             ")")))),
             D,
             proc(x)
               local i;
             begin
               hold(D)([Content::expr(op(x, i)) $ i = 1..nops(x)-1],
                       Content::expr(op(x, nops(x))))
             end):
Content::set("besselI",
             CL::content("id"="besselI", "keywd"="BesselI", "arity"="2", 
                        CL::layout(CL::mrow(CL::msub(CL::mo("I"),
                                                     CL::oper("idx"="1")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             besselI,
             x -> hold(besselI)(Content::expr(op(x, 2)),
                                Content::expr(op(x, 1)))):
Content::set("besselJ",
             CL::content("id"="besselJ", "keywd"="BesselJ", "arity"="2", 
                        CL::layout(CL::mrow(CL::msub(CL::mo("J"),
                                                     CL::oper("idx"="1")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             besselJ,
             x -> hold(besselJ)(Content::expr(op(x, 2)),
                                Content::expr(op(x, 1)))):
Content::set("besselK",
             CL::content("id"="besselK", "keywd"="BesselK", "arity"="2", 
                        CL::layout(CL::mrow(CL::msub(CL::mo("K"),
                                                     CL::oper("idx"="1")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             besselK,
             x -> hold(besselK)(Content::expr(op(x, 2)),
                                Content::expr(op(x, 1)))):
Content::set("besselY",
             CL::content("id"="besselY", "keywd"="BesselY", "arity"="2", 
                        CL::layout(CL::mrow(CL::msub(CL::mo("Y"),
                                                     CL::oper("idx"="1")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             besselY,
             x -> hold(besselY)(Content::expr(op(x, 2)),
                                Content::expr(op(x, 1)))):
Content::set("lambertW",
             CL::content("id"="lambertW", "keywd"="LambertW", "arity"="2", 
                        CL::layout(CL::mrow(CL::msub(CL::mo("W"),
                                                     CL::oper("idx"="1")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             lambertW,
             x -> hold(lambertW)(Content::expr(op(x, 2)),
                                Content::expr(op(x, 1)))):
Content::set("wrightOmega",
             CL::content("id"="wrightOmega", "keywd"="WrightOmega", "arity"="1", 
                        CL::layout(CL::mrow(CL::mo("&omega;"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="postfix", ")")))),
             wrightOmega):
Content::set("ellipticCE",
             CL::content("id"="ellipticCE", "keywd"="EllipticCE", "arity"="1", 
                        CL::layout(CL::mrow(CL::mo("E&prime;"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="postfix", ")")))),
            ellipticCE):
Content::set("ellipticCK",
             CL::content("id"="ellipticCK", "keywd"="EllipticCK", "arity"="1", 
                        CL::layout(CL::mrow(CL::mo("K&prime;"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="postfix", ")")))),
            ellipticCK):
Content::set("ellipticCPi",
             CL::content("id"="ellipticCPi", "keywd"="EllipticCPi", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("&Pi;&prime;"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            ellipticCPi):
Content::set("ellipticE",
             CL::content("id"="ellipticE", "keywd"="EllipticE", "arity"="1", 
                        CL::layout(CL::mrow(CL::mo("E"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="postfix", ")")))),
            ellipticE):
Content::set("ellipticE2",
             CL::content("id"="ellipticE2", "keywd"="EllipticE", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("E"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            ellipticE):
Content::set("ellipticF",
             CL::content("id"="ellipticF", "keywd"="EllipticF", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("F"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            ellipticF):
Content::set("ellipticK",
             CL::content("id"="ellipticK", "keywd"="EllipticK", "arity"="1", 
                        CL::layout(CL::mrow(CL::mo("K"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="postfix", ")")))),
            ellipticK):
Content::set("ellipticNome",
             CL::content("id"="ellipticNome", "keywd"="EllipticNome", "arity"="1", 
                        CL::layout(CL::mrow(CL::mo("q"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="postfix", ")")))),
            ellipticNome):
Content::set("ellipticPi",
             CL::content("id"="ellipticPi", "keywd"="EllipticPi", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("&Pi;"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            ellipticPi):
Content::set("ellipticPi2",
             CL::content("id"="ellipticPi2", "keywd"="EllipticPi", "arity"="3", 
                        CL::layout(CL::mrow(CL::mo("&Pi;"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", ";"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="2"),
                                            CL::mo("form"="postfix", ")")))),
            ellipticPi):
Content::set("fresnelC",
             CL::content("id"="fresnelC", "keywd"="FresnelC", "arity"="1", 
                        CL::layout(CL::mrow(CL::mo("C"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="postfix", ")")))),
            fresnelC):
Content::set("fresnelS",
             CL::content("id"="fresnelS", "keywd"="FresnelS", "arity"="1", 
                        CL::layout(CL::mrow(CL::mo("S"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="postfix", ")")))),
            fresnelS):
Content::set("kroneckerDelta",
             CL::content("id"="kroneckerDelta", "keywd"="KroneckerDelta",
                         "arity"="2", 
                        CL::layout(CL::msub(CL::mo("&delta;"),
                                     CL::mrow(CL::oper("idx"="0"),
                                              CL::mo("form"="infix", "lbadness"="50000", ","),
                                              CL::oper("idx"="1"))
                                   ))),
            kroneckerDelta):
Content::set("jacobiAM",
             CL::content("id"="jacobiAM", "keywd"="JacobiAM", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("am"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiAM):
Content::set("jacobiCD",
             CL::content("id"="jacobiCD", "keywd"="JacobiCD", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("cd"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiCD):
Content::set("jacobiCN",
             CL::content("id"="jacobiCN", "keywd"="JacobiCN", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("cn"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiCN):
Content::set("jacobiCS",
             CL::content("id"="jacobiCS", "keywd"="JacobiCS", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("cs"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiCS):
Content::set("jacobiDC",
             CL::content("id"="jacobiDC", "keywd"="JacobiDC", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("dc"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiDC):
Content::set("jacobiDN",
             CL::content("id"="jacobiDN", "keywd"="JacobiDN", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("dn"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiDN):
Content::set("jacobiDS",
             CL::content("id"="jacobiDS", "keywd"="JacobiDS", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("ds"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiDS):
Content::set("jacobiNC",
             CL::content("id"="jacobiNC", "keywd"="JacobiNC", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("nc"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiNC):
Content::set("jacobiND",
             CL::content("id"="jacobiND", "keywd"="JacobiND", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("nd"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiND):
Content::set("jacobiNS",
             CL::content("id"="jacobiNS", "keywd"="JacobiNS", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("ns"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiNS):
Content::set("jacobiSC",
             CL::content("id"="jacobiSC", "keywd"="JacobiSC", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("sc"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiSC):
Content::set("jacobiSD",
             CL::content("id"="jacobiSD", "keywd"="JacobiSD", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("sd"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiSD):
Content::set("jacobiSN",
             CL::content("id"="jacobiSN", "keywd"="JacobiSN", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("sn"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiSN):
Content::set("jacobiZeta",
             CL::content("id"="jacobiZeta", "keywd"="JacobiZeta", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("Z"),
                                            CL::mo("form"="prefix", "lbadness"="50000", "("),
                                            CL::oper("idx"="0"),
                                            CL::mo("form"="infix", "|"),
                                            CL::oper("idx"="1"),
                                            CL::mo("form"="postfix", ")")))),
            jacobiZeta):
Content::set("hypergeom",
             CL::content("id"="hypergeom",
                         "keywd"="Confluent Hypergeometric Function",
                         "arity"="5",
                         CL::layout(CL::mrow(CL::msub(CL::mo("&ApplyFunction;"),
                                                      CL::oper("idx"="3")),
                                             CL::mo("form"="infix", "lbadness"="50000", "rbadness"="50000"),
                         
                                             CL::msub(CL::mo("lbadness"="50000", "rbadness"="50000", "F"),
                                                      CL::oper("idx"="4")),
                                             CL::mo("form"="prefix", "lbadness"="50000", "("),
                                             CL::oper("idx"="0"),
                                             CL::mo("form"="infix", "lbadness"="50000", ";&blank;"),
                                             CL::oper("idx"="1"),
                                             CL::mo("form"="infix", "lbadness"="50000", ";&blank;"),
                                             CL::oper("idx"="2"),
                                             CL::mo("form"="postfix", ")")))),
             hypergeom,
             proc(x)
               local l1, l2;
             begin
               l1 := [Content::expr(op(x, 1))];
               if l1 = [hold(``)] then l1 := []; end;
               l2 := [Content::expr(op(x, 2))];
               if l2 = [hold(``)] then l2 := []; end;
               hold(hypergeom)(l1, l2, Content::expr(op(x, 3)))
             end):
Content::set("meijerG",
             CL::content("id"="meijerG",
                         "keywd"="Meijer's G Function",
                         "arity"="7",
                         CL::layout(CL::mrow(CL::mo("rbadness"="50000", "G"),
                            CL::munderover(CL::mo("&blank;"),
                               CL::mrow(CL::oper("idx"="5"),
                                        CL::mo("form"="infix", ",&blank;"),
                                        CL::oper("idx"="6")),
                               CL::mrow(CL::oper("idx"="3"),
                                        CL::mo("form"="infix", ",&blank;"),
                                        CL::oper("idx"="4"))),
                            CL::mo("form"="prefix", "lbadness"="50000", "rbadness"="10000", "("),
                            CL::munderover(CL::mrow(CL::mo("&blank;")),
                                           CL::mrow(CL::oper("idx"="1"),
                                                    CL::mo("&InvisibleComma;")),
                                           CL::mrow(CL::oper("idx"="0"),
                                                    CL::mo("&InvisibleComma;"))),
                            CL::mo("form"="infix", "|"),
                            CL::oper("idx"="2"),
                            CL::mo("form"="postfix", ")")))),
             meijerG,
             proc(x)
               local l1, l2;
             begin
               l1 := [Content::expr(op(x, 1))];
               if l1 = [hold(``)] then l1 := []; end;
               l2 := [Content::expr(op(x, 2))];
               if l2 = [hold(``)] then l2 := []; end;
               hold(meijerG)(Content::expr(op(x, 4)),
                                Content::expr(op(x, 5)),
                                l1, l2,
                                Content::expr(op(x, 3)))
             end):
Content::set("binomial",
             CL::content("id"="binomial", "keywd"="Binomial", "arity"="2", 
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "("),
                                             CL::mfrac("linethickness"="0",
                                                       CL::oper("idx"="0"),
                                                       CL::oper("idx"="1")),
                                             CL::mo("form"="postfix", "lbadness"="50000", ")")))),
             binomial):
Content::set("range",
             CL::content("id"="range", "keywd"="Range",
                         "arity"="2", "prio"=OP::Range,
                         CL::layout(CL::mrow(CL::oper("idx"="0"),
                                    CL::mo("form"="infix", ".."),
                                    CL::oper("idx"="1")))),
             _range):
Content::set("compose",
             CL::content("id"="compose", "keywd"="Compose Functions",
                         "accel"="@", "arity"="2*", "prio"=OP::Fconcat,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Fconcat),
                                              CL::mo("form"="infix",
                                                     "&SmallCircle;")))),
             _fconcat):
Content::set("inverse",
             CL::content("id"="inverse", "keywd"="Functional Inverse",
                         "arity"="1", "prio"=OP::Power,
                         CL::layout(CL::msup(CL::oper("idx"="0","prio"=OP::Power),
                                             CL::mo("-1")))),
             _inverse):
// notice the different back translation
Content::set("inverseMatrix",
             CL::content("id"="inverseMatrix", "keywd"="Functional Inverse",
                         "arity"="1", "prio"=OP::Power,
                         CL::layout(CL::msup(CL::oper("idx"="0","prio"=OP::Power),
                                             CL::mo("-1")))),
             inverse):
// used only temporary; not output template needed
// arity "0" is hard-coded in ContentLayout
Content::set("sep", FAIL, FAIL):
// used only temporary; not output template needed
// arity "1*" is hard-coded in ContentLayout
Content::set("condition", FAIL, FAIL):
Content::set("imageset",
             CL::content("id"="imageset", "keywd"="ImageSet", "arity"="2", 
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "{"),
                                             CL::oper("idx"="0"),
                                             CL::mo("lbadness"="5000", "rbadness"="5000", "&verbar;"),
                                             CL::oper("idx"="1"),
                                             CL::mo("form"="postfix", "}")))),
              Dom::ImageSet,
	          proc(x)
                local t;
              begin
                Dom::ImageSet::create(Content::expr(op(x,1)),
                    if Content::typString(op(x,2)) = "and" then
                      [Content::expr(op(t, 1))$t in op(x,2)],
                      [Content::expr(op(t, 2))$t in op(x,2)]
                    else
                      [Content::expr(op(x, [2,1]))],
                      [Content::expr(op(x, [2,2]))]
                    end_if)
              end_proc):
Content::set("piecewise",
             CL::content("id"="piecewise", "keywd"="Piecewise", "arity"="2*",
                          "assoc"="false",
                 CL::layout(
                     CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "{"),
                         CL::optable("cols"="2",
                                     "columnalign"="cl",
                                     "wrapchildren"="true",
                                     CL::oper(),
                                     CL::mrow(CL::mtext("&Space;if&Space;&Space;"),
                                              CL::oper()))))),
             piecewise,
             proc(x)
               local i;
             begin
               if nops(x) = 2 then
                 piecewise::create([Content::expr(op(x, 2)), Content::expr(op(x, 1))])
               else
                 piecewise::create([Content::expr(op(x, 2*i)),
                            Content::expr(op(x, 2*i-1))] $
                                i=1..nops(x)/2)
               end_if
             end_proc):
Content::set("piecewise2",
             CL::content("id"="piecewise2", "keywd"="Piecewise", "arity"="2*",
                          "assoc"="false",
               CL::layout(
                 CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "{"),
                   CL::mfrac("linethickness"="0",
                     CL::optable("cols"="2",
                                 "columnalign"="cl",
                                 "wrapchildren"="true",
                                 "idx"="0,-2",
                                 CL::oper(),
                                 CL::mrow(CL::mtext("&Space;if&Space;&Space;"),
                                          CL::oper())),
                     CL::mrow(CL::oper("idx"="-1"),
                              CL::mtext("&Space;&Space;otherwise")))))),
             piecewise,
             proc(x)
               local i;
             begin
               if nops(x) = 3 then
                 piecewise::create([Content::expr(op(x, 2)), Content::expr(op(x, 1))],
                           [Otherwise, Content::expr(op(x, nops(x)))])
               else
               piecewise::create([Content::expr(op(x, 2*i)),
                          Content::expr(op(x, 2*i-1))] $ i=1..nops(x) div 2,
                          [Otherwise, Content::expr(op(x, nops(x)))])
               end_if
             end_proc):
Content::set("piecewise3",
             CL::content("id"="piecewise3", "keywd"="Piecewise", "arity"="2*",
                          "assoc"="false",
               CL::layout(
                 CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "{"),
                     CL::optable("cols"="2",
                                 "columnalign"="cl",
                                 "wrapchildren"="true",
                                 CL::oper())))),
             piecewise,
             proc(x)
               local i;
             begin
               piecewise::create([Content::expr(op(x, 2*i)),
                          Content::expr(op(x, 2*i-1))] $ i=1..(nops(x) div 2) - 1,
                          [Otherwise, Content::expr(op(x, nops(x)-1))])
             end_proc):
Content::set("piece2",
             CL::content("id"="piece2", "keywd"="Piece", "arity"="1",
                          "assoc"="false",
               CL::layout(
                 CL::mrow(CL::mtext("&Space;if&Space;&Space;"),
                                          CL::oper("idx"="0")))),
             piecewise,
             proc(x)
             begin
               Content::expr(op(x));
             end_proc):
Content::set("table",
             CL::content("id"="table", "keywd"="Table", "arity"="0*", 
                    CL::layout(
                      CL::mrow(
                        CL::optable("cols"="2", "rowlines"="n",
                          "columnlines"="sn",
                          "framelines"="snnn",
                          "columnalign"="rl",
                          "wrapchildren"="true",
                          CL::oper())))
                         ),
             table,
             proc(x)
	       local firstrow, i;
	     begin
	       firstrow := [Content::expr(op(x, 1)),
  	       		    Content::expr(op(x, 2))];
 	       table(Content::expr(op(x, 2*i-1)) =
                     Content::expr(op(x, 2*i)) $ 
		     i=if firstrow=[hold(``),hold(``)] then 2 else 1
		     end_if..nops(x)/2);
             end_proc):
Content::set("pochhammer",
             CL::content("id"="pochhammer", "keywd"="pochhammer", "arity"="2", 
                        CL::layout(CL::mrow(CL::mo("pochhammer"),
                              CL::mrow(
                                 CL::mo("form"="prefix", "lbadness"="50000", "("),
                                 CL::oprow(CL::oper("idx"="0"),
                                           CL::mo("form"="infix", "lbadness"="50000", ",")),
                                 CL::mo("form"="postfix", ")")))),
                        CL::layout(CL::msub(CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "("),
                                            				 CL::oper("idx"="0"),
                                                     CL::mo("form"="postfix", "lbadness"="50000", ")")),
                                            CL::oper("idx"="1")))),
             pochhammer,
             x -> hold(pochhammer)(Content::expr(op(x, 1)),
                                Content::expr(op(x,2)))):
Content::set("intervaloo",
             CL::content("id"="intervaloo", "keywd"="IntervalOpenOpen",
                         "arity"="2", 
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "("),
                                             CL::oper("idx"="0"),
                                             CL::mo("lbadness"="50000", ","),
                                             CL::oper("idx"="1"),
                                             CL::mo("form"="postfix", "lbadness"="50000", ")")))),
             Dom::Interval,
             x -> Dom::Interval(Content::expr(op(x, 1)),
                                Content::expr(op(x, 2)))):
Content::set("intervaloc",
             CL::content("id"="intervaloc", "keywd"="IntervalOpenClosed",
                         "arity"="2", 
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "("),
                                             CL::oper("idx"="0"),
                                             CL::mo("lbadness"="50000", ","),
                                             CL::oper("idx"="1"),
                                             CL::mo("form"="postfix", "lbadness"="50000", "]")))),
             Dom::Interval,
             x -> Dom::Interval(Content::expr(op(x, 1)),
                                [Content::expr(op(x, 2))])):
Content::set("intervalco",
             CL::content("id"="intervalco", "keywd"="IntervalClosedOpen",
                         "arity"="2", 
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "["),
                                             CL::oper("idx"="0"),
                                             CL::mo("lbadness"="50000", ","),
                                             CL::oper("idx"="1"),
                                             CL::mo("form"="postfix", "lbadness"="50000", ")")))),
             Dom::Interval,
             x -> Dom::Interval([Content::expr(op(x, 1))],
                                Content::expr(op(x, 2)))):
Content::set("intervalcc",
             CL::content("id"="intervalcc", "keywd"="IntervalClosedClosed",
                         "arity"="2", 
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "["),
                                             CL::oper("idx"="0"),
                                             CL::mo("lbadness"="50000", ","),
                                             CL::oper("idx"="1"),
                                             CL::mo("form"="postfix", "lbadness"="50000", "]")))),
             Dom::Interval,
             x -> Dom::Interval([Content::expr(op(x, 1)),
                                Content::expr(op(x, 2))])):
Content::set("interval",
             CL::content("id"="interval", "keywd"="Interval",
                         "arity"="2", "prio"=OP::Interval,
                         CL::layout(CL::mrow(CL::oper("idx"="0"),
                                             CL::mo("form"="infix", "..."),
                                             CL::oper("idx"="1")))),
             DOM_INTERVAL,
             proc(x)
               local left, right;
             begin
               left := Content::expr(op(x,1));
               if testtype(left, DOM_IDENT) and
                 testtype(text2expr("".left), DOM_FLOAT) then
                 left := rdtext2float("".left, -1);
               end;
               right := Content::expr(op(x,2));
               if testtype(right, DOM_IDENT) and
                 testtype(text2expr("".right), DOM_FLOAT) then
                 right := rdtext2float("".right, 1);
               end;
               
               hold(hull)(left, right)
             end):

Content::set("apply",
             CL::content("id"="apply", "keywd"="Apply Function", "accel"="(",
                        "arity"="2*", "assoc"="false", "flatten-seq"="1,-1",
                         "prio"=OP::Index,
                    CL::layout(
                          CL::mrow(
                              CL::oper("idx"="0", "prio"=OP::Index-10),
                              CL::mrow(
                                 CL::mo("form"="prefix", "lbadness"="50000", "rbadness"="10000", "("),
                                 CL::oprow("idx"="1,-1",
                                           CL::oper(),
                                           CL::mo("form"="infix", "lbadness"="50000", ",")),
                              CL::mo("form"="postfix", "lbadness"="10000", ")"))))),
             fp::apply, 
             proc(x)
               local i;
               save XxX;
             begin
               delete XxX;
               
               subs(XxX(Content::expr(op(x, i)) $ i = 2 ..nops(x)),
                    XxX=Content::expr(op(x,1)), Unsimplified)
             end_proc):
Content::set("applyvoid",
             CL::content("id"="applyvoid", "keywd"="Apply Void Function",
                        "arity"="1", "prio"=OP::Index,
                    CL::layout(
                          CL::mrow(
                              CL::oper("idx"="0", "prio"=OP::Index-10),
                              CL::mrow(
                                 CL::mo("form"="prefix", "lbadness"="50000", "rbadness"="50000", "("),
                                 CL::mo("form"="postfix", ")"))))),
             fp::apply, 
             proc(x)
               save XxX;
             begin
               delete XxX;
               
               subs(XxX(), XxX=Content::expr(op(x,1)), Unsimplified)
             end_proc):
Content::set("scalarproduct",
             CL::content("id"="scalarproduct", "keywd"="Scalar Product",
                         "arity"="2", "prio"=OP::Mult,
                         CL::layout(CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "&lang;"),
                                             CL::oper("idx"="0", "prio"=OP::Stmtseq),
                                             CL::mo("form"="infix", "lbadness"="50000", ","),
                                             CL::oper("idx"="1", "prio"=OP::Stmtseq),
                                             CL::mo("form"="postfix", "lbadness"="50000", "&rang;")))),
             linalg::scalarProduct):
Content::set("vectorproduct",
             CL::content("id"="vectorproduct", "keywd"="Vector Product",
                         "arity"="2*", "prio"=OP::Mult,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Mult),
                                              CL::mo("form"="infix", "&times;")))),
             linalg::crossProduct):
Content::set("transpose",
             CL::content("id"="transpose", "keywd"="Transpose",
                         "arity"="1", "prio"=OP::Power,
                         CL::layout(CL::mrow(CL::mo("transpose"),
                              CL::mrow(
                                 CL::mo("form"="prefix", "lbadness"="50000", "("),
                                 CL::oper("idx"="0"),
                                 CL::mo("form"="postfix", ")")))),
                         CL::layout(CL::msup(CL::oper("idx"="0",
                                                      "prio"=OP::Power),
                                             CL::mo("t")))),
             transpose):
Content::set("htranspose",
             CL::content("id"="htranspose", "keywd"="Hermitian Transpose",
                         "arity"="1", "prio"=OP::Power,
                         CL::layout(CL::mrow(CL::mo("htranspose"),
                              CL::mrow(
                                 CL::mo("form"="prefix", "lbadness"="50000", "("),
                                 CL::oper("idx"="0"),
                                 CL::mo("form"="postfix", ")")))),
                         CL::layout(CL::msup(CL::mover(CL::oper("idx"="0"),
                                                       CL::mo("form"="postfix",
                                                              "&OverBar;")),
                                             CL::mo("t")))),
             htranspose):
Content::set("grad",
              CL::content("id"="grad", "keywd"="Gradient",
                        "arity"="2", "prio"=OP::Mult,
                   CL::layout(CL::mrow(CL::msub(CL::mo("&Del;"),
                                                CL::oper("idx"="1",
                                                         "prio"=OP::Noop)),
                                       CL::mo("lbadness"="50000", "rbadness"="50000", "&NonBreakingSpace;"),
                                                 CL::oper("idx"="0",
                                                         "prio"=OP::Mult)
                                                 ))),
             linalg::grad,
             x -> hold(linalg::grad)(Content::expr(op(x, 1)),
                                     [Content::expr(op(x, 2))])):
Content::set("divergence",
              CL::content("id"="divergence", "keywd"="Divergence",
                        "arity"="2", "prio"=OP::Mult,
                   CL::layout(CL::mrow(CL::msub(CL::mo("&Del;"),
                                                CL::oper("idx"="1",
                                                         "prio"=OP::Noop)),
                                       CL::mo("lbadness"="50000", "rbadness"="50000", "&cdot;"),
                                       CL::oper("idx"="0",
                                                "prio"=OP::Mult)
                                                 ))),
             linalg::divergence,
             proc(x)
               local op1, op2;
             begin
               op1 := Content::expr(op(x, 1));
               if op1::dom <> DOM_LIST and
                 op1::dom::hasProp( Cat::Matrix ) <> TRUE then
                 op1 := [op1]
               end_if;
               op2 := Content::expr(op(x, 2));
               if op2::dom <> DOM_LIST then
                 op2 := [op2]
               end_if;
               hold(linalg::divergence)(op1, op2)
             end_proc):
Content::set("curl",
              CL::content("id"="curl", "keywd"="Rotation",
                        "arity"="6", "prio"=OP::Mult,
                   CL::layout(CL::mrow(CL::msub(CL::mo("&Del;"),
                                                CL::mrow(CL::oper("idx"="3"),
                                                         CL::mo("lbadness"="50000", ","),
                                                         CL::oper("idx"="4"),
                                                         CL::mo("lbadness"="50000", ","),
                                                         CL::oper("idx"="5"))),
                                       CL::mo("lbadness"="50000", "&times;"),
                                       CL::mo("lbadness"="50000", "("),
                                       CL::oper("idx"="0"),
                                       CL::mo("lbadness"="50000", ","),
                                       CL::oper("idx"="1"),
                                       CL::mo("lbadness"="50000", ","),
                                       CL::oper("idx"="2"),
                                       CL::mo("rbadness"="50000", ")")))),
             linalg::curl,
             x -> hold(linalg::curl)([map(op(x, 1..3), Content::expr)],
                                     [map(op(x, 4..6), Content::expr)])):
Content::set("laplacian",
              CL::content("id"="laplacian", "keywd"="Laplacian",
                        "arity"="2", "prio"=OP::Mult,
                   CL::layout(CL::mrow(CL::msub(CL::mo("form"="prefix",
                                                       "&Delta;"),
                                                CL::oper("idx"="1",
                                                         "prio"=OP::Noop)),
                                       CL::mo("lbadness"="50000", "rbadness"="50000", "&NonBreakingSpace;"),
                                                 CL::oper("idx"="0",
                                                         "prio"=OP::Mult)
                                                 ))),
             linalg::laplacian,
             x -> hold(linalg::laplacian)(Content::expr(op(x, 1)),
                                          [Content::expr(op(x, 2))])):
Content::set("mean",
             CL::content("id"="mean", "keywd"="Mean Value", "arity"="1", 
                      CL::layout(CL::mover(CL::oper("idx"="0"),
                                           CL::mo("form"="postfix",
                                                  "&OverBar;")))),
             stats::mean):
Content::set("quadraticMean",
             CL::content("id"="quadraticMean", "keywd"="Quadratic Mean Value", "arity"="1", 
                      CL::layout(CL::mover(
                                    CL::msup(CL::oper("idx"="0"),
                                             CL::mo("2")),
                                           CL::mo("form"="postfix",
                                                  "&OverBar;")))),
             stats::quadraticMean):
Content::set("sdev",
             CL::content("id"="sdev", "keywd"="Standard Derivation",
                         "arity"="1*", "prio"=OP::Index,
                      CL::layout(CL::mrow(CL::mo("&sigma;"),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oprow(CL::oper(),
                                                    CL::mo("lbadness"="50000", ",")),
                                          CL::mo("form"="postfix", ")")))),
             stats::stdev):
Content::set("variance",
             CL::content("id"="variance", "keywd"="Variance",
                         "arity"="1*", "prio"=OP::Index,
                      CL::layout(CL::msup(CL::mrow(CL::mo("&sigma;"),
                                                   CL::mo("form"="prefix", "lbadness"="50000", "("),
                                                   CL::oprow(CL::oper(),
                                                             CL::mo("lbadness"="50000", ",")),
                                                   CL::mo("form"="postfix",
                                                          ")")),
                                          CL::mo("2")))),
             stats::variance):
Content::set("median",
             CL::content("id"="median", "keywd"="Median", "arity"="1*", 
                      CL::layout(CL::mrow(CL::mo("median"),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oprow(CL::oper(),
                                                    CL::mo("lbadness"="50000", ",")),
                                          CL::mo("form"="postfix", ")")))),
             stats::median):
Content::set("mode",
             CL::content("id"="mode", "keywd"="Modus",
                         "arity"="1*", "prio"=OP::Index,
                      CL::layout(CL::mrow(CL::mo("mode"),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oprow(CL::oper(),
                                                    CL::mo("lbadness"="50000", ",")),
                                          CL::mo("form"="postfix", ")")))),
             stats::modal,
             x -> hold(stats::modal)(map(op(x), Content::expr))):

Content::set("moment",
             CL::content("id"="moment", "keywd"="Moment", "arity"="3",
               CL::layout(CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "&lang;"),
                                   CL::msup(CL::oper("idx"="0", "prio"=OP::Exprseq),
                                            CL::oper("idx"="1")),
                                   CL::mo("lbadness"="50000", ","),
                                   CL::oper("idx"="2"),
                                   CL::mo("form"="postfix", "lbadness"="50000", "&rang;")))),
             stats::moment,
             x -> hold(stats::moment)(Content::expr(op(x, 2)),
                                      Content::expr(op(x, 3)),
                                      Content::expr(op(x, 1)))):

  
Content::set("matrix",
             CL::content("id"="matrix", "keywd"="Matrix",
                         "accel"="#", "arity"="1*",
                    CL::layout(
                        CL::mrow(
                            CL::mo("form"="prefix", "rbadness"="50000", "["),
                            CL::oplists(
                                CL::mrow(
                                    CL::mo("form"="prefix", "rbadness"="50000", "["),
                                    CL::oprow(
                                        CL::oper(),
                                        CL::mo("lbadness"="50000", ",")),
                                    CL::mo("form"="postfix", "lbadness"="50000", "]")),
                                CL::mo("lbadness"="50000", "rbadness"="-50000", ",")),
                            CL::mo("form"="postfix", "lbadness"="50000", "]"))),
                    CL::layout(
                          CL::mrow(
                              CL::mo("form"="prefix", "rbadness"="50000", "("),
                              CL::optable(CL::oper()),
                              CL::mo("form"="postfix", "lbadness"="50000", ")")))),
             DOM_ARRAY, 
             proc(m, attributes, infoString)
               local rows, cols, i, j, tmp, dot2zero;
             begin
               dot2zero := proc(x = null()) begin if x = hold(`&cdot; `) then 0 else x end end;
               rows := text2expr(attributes[hold(Rows)]);
               cols := text2expr(attributes[hold(Cols)]);
               // We do not want to evaluate here, but to simplify
               // (kernel simplification).  If we do not
               // evaluate.  Therefore 
               // Dom::Matrix(Dom::Fraction(Dom::Integer))([[-2, 1], [3/2, -1/2]])
               // is broken.  If we evaluate, the term order of the matrix
               // coefficients will be broken!
               tmp := [[dot2zero(Content::expr(op(m,(i-1)*cols+j)))
                        $ j=1..cols] $ i=1..rows];
               if infoString <> "" then
                 eval(text2expr(infoString))(tmp)
               else
                 array(1..rows, 1..cols, tmp);
               end_if;
             end_proc):
Content::set("matrixEmpty",
             CL::content("id"="matrixEmpty", "keywd"="Empty Matrix",
                         "arity"="0",
                    CL::layout(
                          CL::mrow(
                              CL::mo("form"="prefix", "rbadness"="50000", "("),
                              CL::mo("form"="postfix", "lbadness"="50000", ")")))),
             DOM_ARRAY, 
             proc(m, attributes, infoString)
             begin
                 eval(text2expr(infoString))()
             end_proc):
// template "invmatrix" is unused
Content::set("invmatrix",
             CL::content("id"="invmatrix", "keywd"="Inverse", "arity"="1",
                         CL::layout(CL::msup(CL::oper("idx"="0"),
                                             CL::mo("-1")))),
             _power,
             m -> hold(_power)(Content::expr(op(m, 1)), -1)):
Content::set("tableau",
             CL::content("id"="tableau", "keywd"="Tableau", "arity"="1*",
                    CL::layout(
                          CL::mrow(
                              CL::optable(CL::oper())))),
             Tableau,
             () -> error("\"tableau\" -> MuPAD: not yet implemented")):
Content::set("assign",
             CL::content("id"="assign", "keywd"="Assign",
                         "arity"="2", "prio"=OP::Noop,
                         CL::layout(CL::mrow(CL::oper("idx"="0"),
                                             CL::mo("lbadness"="10000", ":="),
                                             CL::oper("idx"="1")))),
             _assign):
Content::set("seq",
             CL::content("id"="seq", "keywd"="Sequence", "accel"=",",
                         "arity"="2*", "flatten-seq"="0,-1",  "prio"=OP::Exprseq,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Exprseq),
                                              CL::mo("lbadness"="50000", "rbadness"="1", ",")))),
             _exprseq):
Content::set("index",
             CL::content("id"="index", "keywd"="Index", "arity"="2", "prio"=OP::Index,
                         CL::layout(
                                    CL::msub(CL::oper("idx"="0", "prio"=OP::Index),
                                             CL::oper("idx"="1")))),
             _index,
             proc(x)
               local idx, i;
             begin
               idx := [Content::expr(op(x, 2))];
               subsop(hold(_index)(FAIL$nops(idx)+1),
                      1=Content::expr(op(x,1)),
                      i+1=idx[i] $ i=1..nops(idx), Unsimplified)
             end_proc):
Content::set("row",
             CL::content("id"="row", "keywd"="Layout Row",
                         "arity"="1*", "prio"=OP::Index,
                         CL::layout(CL::oprow(CL::oper(),
                                              CL::mo(" ")))),
             error,
             () -> error("\"row\" -> MuPAD: not yet implemented")):
// comprow:  documentation only?
Content::set("comprow",
             CL::content("id"="comprow", "keywd"="Layout Row of Comparisons",
                         "arity"="1*", "prio"=OP::Relation,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Relation),
                                              CL::mo("&Space;")))),
             error,
             () -> error("\"comprow\" -> MuPAD: not yet implemented")):
Content::set("concat",
             CL::content("id"="concat", "keywd"="Concatenation",
                         "arity"="2*", "prio"=OP::Concat,
                         CL::layout(CL::oprow(CL::oper("prio"=OP::Concat),
                                              CL::mo(".")))),
             _concat):
Content::set("gamma",
             CL::content("id"="gamma", "keywd"="Gamma Function",
                         "arity"="1", "prio"=OP::Index,
                      CL::layout(CL::mrow(CL::mo("&Gamma;"),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             gamma):
Content::set("igamma",
             CL::content("id"="igamma", "keywd"="Incomplete Gamma Function",
                         "arity"="2", "prio"=OP::Index,
                      CL::layout(CL::mrow(CL::mo("&Gamma;"),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("lbadness"="50000", ","),
                                          CL::oper("idx"="1"),
                                          CL::mo("form"="postfix", ")")))),
             igamma):
Content::set("lngamma",
             CL::content("id"="lngamma", "keywd"="Logarithmic Gamma Function",
                         "arity"="1", "prio"=OP::Index,
                      CL::layout(CL::mrow(CL::mo("ln&Gamma;"),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             lngamma):
Content::set("beta",
             CL::content("id"="beta", "keywd"="Beta Function",
                         "arity"="2", "prio"=OP::Index,
                      CL::layout(CL::mrow(CL::mo("&beta;"),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("lbadness"="50000", ","),
                                          CL::oper("idx"="1"),
                                          CL::mo("form"="postfix", ")")))),
             beta):
Content::set("zeta",
             CL::content("id"="zeta", "keywd"="Zeta Function",
                         "arity"="1", "prio"=OP::Index,
                      CL::layout(CL::mrow(CL::mo("&zeta;"),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             zeta):
Content::set("zeta1",
             CL::content("id"="zeta1", "keywd"="Derivative of Zeta Function",
                         "arity"="1", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                              CL::msup(CL::mo("&zeta;"),
                                       CL::mo("form"="postfix", "lbadness"="50000", "&prime;")),
                                             CL::mo("form"="prefix", "lbadness"="50000", "("),
                                             CL::oper("idx"="0"),
                                             CL::mo("form"="postfix", ")")))),
             zeta,
             x -> hold(zeta)(Content::expr(op(x)), 1)):
Content::set("zeta2",
             CL::content("id"="zeta2", "keywd"="2nd Derivative of Zeta Function",
                         "arity"="1", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                              CL::msup(CL::mo("&zeta;"),
                                CL::mo("form"="postfix", "lbadness"="50000", "&prime;&prime;")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             zeta,
             x -> hold(zeta)(Content::expr(op(x)), 2)):
Content::set("zeta3",
             CL::content("id"="zeta3", "keywd"="3rd Derivative of Zeta Function",
                         "arity"="1", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                              CL::msup(CL::mo("&zeta;"),
                                CL::mo("form"="postfix", "lbadness"="50000", "&prime;&prime;&prime;")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             zeta,
             x -> hold(zeta)(Content::expr(op(x)), 3)):
Content::set("zetaD",
             CL::content("id"="zetaD", "keywd"="n-th Derivative of Zeta Function",
                         "arity"="2", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                                 CL::msup(CL::mo("&zeta;"),
                                     CL::mrow(
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="1"),
                                          CL::mo("form"="postfix", ")"))),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             zeta):
Content::set("dirac",
             CL::content("id"="dirac", "keywd"="Dirac Distribution",
                         "arity"="1", "prio"=OP::Index,
                      CL::layout(CL::mrow(CL::mo("&delta;"),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             dirac):
Content::set("dirac1",
             CL::content("id"="dirac1", "keywd"="Derivative of Dirac Distribution",
                         "arity"="1", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                              CL::msup(CL::mo("&delta;"),
                                       CL::mo("form"="postfix", "lbadness"="50000", "&prime;")),
                                             CL::mo("form"="prefix", "lbadness"="50000", "("),
                                             CL::oper("idx"="0"),
                                             CL::mo("form"="postfix", ")")))),
             dirac,
             x -> hold(dirac)(Content::expr(op(x)), 1)):
Content::set("dirac2",
             CL::content("id"="dirac2", "keywd"="Derivative of Dirac Distribution",
                         "arity"="1", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                              CL::msup(CL::mo("&delta;"),
                                CL::mo("form"="postfix", "lbadness"="50000", "&prime;&prime;")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             dirac,
             x -> hold(dirac)(Content::expr(op(x)), 2)):
Content::set("dirac3",
             CL::content("id"="dirac3", "keywd"="Derivative of Dirac Distribution",
                         "arity"="1", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                              CL::msup(CL::mo("&delta;"),
                                CL::mo("form"="postfix", "lbadness"="50000", "&prime;&prime;&prime;")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             dirac,
             x -> hold(dirac)(Content::expr(op(x)), 3)):
Content::set("diracD",
             CL::content("id"="diracD", "keywd"="Derivative of Dirac Distribution",
                         "arity"="2", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                                 CL::msup(CL::mo("&delta;"),
                                     CL::mrow(
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="1"),
                                          CL::mo("form"="postfix", ")"))),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             dirac):
Content::set("psi",
             CL::content("id"="psi", "keywd"="Digamma Function",
                         "arity"="1", "prio"=OP::Index,
                      CL::layout(CL::mrow(CL::mo("&psi;"),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             psi):
Content::set("psi1",
             CL::content("id"="psi1", "keywd"="Derivative of Digamma Function",
                         "arity"="1", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                              CL::msup(CL::mo("&psi;"),
                                       CL::mo("form"="postfix", "lbadness"="50000", "&prime;")),
                                             CL::mo("form"="prefix", "lbadness"="50000", "("),
                                             CL::oper("idx"="0"),
                                             CL::mo("form"="postfix", ")")))),
             psi,
             x -> hold(psi)(Content::expr(op(x)), 1)):
Content::set("psi2",
             CL::content("id"="psi2", "keywd"="2nd Derivative of Digamma Function",
                         "arity"="1", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                              CL::msup(CL::mo("&psi;"),
                                CL::mrow(CL::mo("form"="postfix", "lbadness"="50000", "&prime;&prime;"))),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             psi,
             x -> hold(psi)(Content::expr(op(x)), 2)):
Content::set("psi3",
             CL::content("id"="psi3", "keywd"="3rd Derivative of Digamma Function",
                         "arity"="1", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                              CL::msup(CL::mo("&psi;"),
                                CL::mo("form"="postfix", "lbadness"="50000", "&prime;&prime;&prime;")),
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             psi,
             x -> hold(psi)(Content::expr(op(x)), 3)):
Content::set("psiD",
             CL::content("id"="psiD", "keywd"="n-th Derivative of Digamma Function",
                         "arity"="2", "prio"=OP::Index,
                         CL::layout(CL::mrow(
                                 CL::msup(CL::mo("&psi;"),
                                     CL::mrow(
                                          CL::mo("form"="prefix", "lbadness"="50000", "("),
                                          CL::oper("idx"="1"),
                                          CL::mo("form"="postfix", ")"))),
                                          CL::mo("form"="prefix", "("),
                                          CL::oper("idx"="0"),
                                          CL::mo("form"="postfix", ")")))),
             psi):
Content::set("whittakerM",
             CL::content("id"="whittakerM", "keywd"="whittakerM", "arity"="3", 
			 CL::layout(CL::mrow(CL::msub(CL::mo("M"),
						      CL::mrow(CL::oper("idx"="0"),
							       CL::mo("lbadness"="50000", ","),
							       CL::oper("idx"="1"))),
					     CL::mo("form"="prefix", "lbadness"="50000", "("),
					     CL::oper("idx"="2"),
					     CL::mo("form"="postfix", ")")))),
	     whittakerM):
Content::set("whittakerW",
             CL::content("id"="whittakerW", "keywd"="whittakerW", "arity"="3", 
			 CL::layout(CL::mrow(CL::msub(CL::mo("W"),
						      CL::mrow(CL::oper("idx"="0"),
							       CL::mo("lbadness"="50000", ","),
							       CL::oper("idx"="1"))),
					     CL::mo("form"="prefix", "lbadness"="50000", "("),
					     CL::oper("idx"="2"),
					     CL::mo("form"="postfix", ")")))),
             whittakerW):
Content::set("evalAt",
             CL::content("id"="evalAt", "keywd"="EvalAt", "arity"="2", "prio"=OP::EvalAt,
                         CL::layout(
                                    CL::msub(CL::mrow(CL::oper("idx"="0", "prio"=OP::Index),
                                                      CL::mo("form"="postfix", "lbadness"="50000", "rbadness"="50000", "|")),
                                             CL::oper("idx"="1")))),
             evalAt):

Content::set("seqgen",
             CL::content("id"="seqgen", "keywd"="Seqgen",
                        "arity"="1", "prio"=OP::Seqgen,
                        CL::layout(CL::mrow(CL::mo("form"="prefix", "rbadness"="50000", "$&blank;"),
                                            CL::oper("idx"="0", "prio"=OP::Seqgen)
                                            ))),
             _seqgen):
Content::set("seqgen2",
             CL::content("id"="seqgen2", "keywd"="Seqgen2",
                        "arity"="2", "prio"=OP::Seqgen,
                        CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Seqgen),
                                            CL::mo("form"="infix", "&blank;$&blank;"),
                                            CL::oper("idx"="1", "prio"=OP::Seqgen)
                                            ))),
             _seqgen):
Content::set("seqgen3",
             CL::content("id"="seqgen3", "keywd"="Seqgen3",
                        "arity"="3", "prio"=OP::Seqgen,
                        CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Seqgen),
                                            CL::mo("form"="infix", "&blank;$&blank;"),
                                            CL::oper("idx"="1", "prio"=OP::Seqgen),
                                            CL::mo("form"="infix", "lbadness"="50000", "&blank;=&blank;"),
                                            CL::oper("idx"="2", "prio"=OP::Seqgen)
                                            ))),
             _seqgen):
Content::set("seqstep",
             CL::content("id"="seqstep", "keywd"="Seqstep",
                        "arity"="2", "prio"=OP::Seqgen,
                        CL::layout(CL::mrow(CL::mo("form"="prefix", "$&blank;"),
                                            CL::oper("idx"="0", "prio"=OP::Seqgen),
                                            CL::mo("form"="infix", "lbadness"="50000", "&blank;step&blank;"),
                                            CL::oper("idx"="1", "prio"=OP::Seqgen)
                                            ))),
             _seqstep):
Content::set("seqstep2",
             CL::content("id"="seqstep2", "keywd"="Seqstep2",
                        "arity"="3", "prio"=OP::Seqgen,
                        CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Seqgen),
                                            CL::mo("form"="infix", "&blank;$&blank;"),
                                            CL::oper("idx"="1", "prio"=OP::Seqgen),
                                            CL::mo("form"="infix", "lbadness"="50000", "&blank;step&blank;"),
                                            CL::oper("idx"="2", "prio"=OP::Seqgen)
                                            ))),
             _seqstep):
Content::set("seqstep3",
             CL::content("id"="seqstep3", "keywd"="Seqstep3",
                        "arity"="4", "prio"=OP::Seqgen,
                        CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Seqgen),
                                            CL::mo("form"="infix", "&blank;$&blank;"),
                                            CL::oper("idx"="1", "prio"=OP::Seqgen),
                                            CL::mo("form"="infix", "lbadness"="50000", "&blank;=&blank;"),
                                            CL::oper("idx"="2", "prio"=OP::Seqgen),
                                            CL::mo("form"="infix", "lbadness"="50000", "&blank;step&blank;"),
                                            CL::oper("idx"="3", "prio"=OP::Seqgen)
                                            ))),
             _seqstep):
Content::set("seqin",
             CL::content("id"="seqin", "keywd"="Seqin",
                        "arity"="3", "prio"=OP::Seqgen,
                        CL::layout(CL::mrow(CL::oper("idx"="0", "prio"=OP::Seqgen),
                                            CL::mo("form"="infix", "&blank;$&blank;"),
                                            CL::oper("idx"="1", "prio"=OP::Seqgen),
                                            CL::mo("form"="infix", "lbadness"="50000", " &in; "),
                                            CL::oper("idx"="2", "prio"=OP::Seqgen)
                                            ))),
             _seqin):
Content::set("unit",
             CL::content("id"="unit", "keywd"="Unit",
                         "arity"="1",
                         CL::layout(CL::oper("idx"="0"))),
             unit,
             proc(x)
               local res, prefix;
             begin
               prefix := "";
               res := "".Content::expr(op(x));
               if length(res) > 4 and res[1..4] = "&mu;" then
                 prefix := "micro";
                 res := res[5..-1];
               end_if;
               if res = "&Omega;" then
                 res := "Ohm";
               end_if;
               if res = "&deg;C" then
                 res := "Celsius";
               end_if;
               if res = "&deg;F" then
                 res := "Fahrenheit";
               end_if;
               new(unit, prefix.res);
             end_proc):
Content::set("abbreviation",
             CL::content("id"="abbreviation", "keywd"="abbreviation", "arity"="1*",
                         CL::layout(
                                  CL::optable("cols"="1",
                                              "columnalign"="l",
                                              "wrapchildren"="true",
                                             CL::oper()))),
             FAIL,
             // keep in sync with example in OUTPUT/TEST/Content.dev.tst
             proc(x)
               local i;
             begin
               subs(Content::expr(op(x, 1)),
                    (Content::expr(op(x, [1+(i+1)*2, 2])) $ i=1..(nops(x) div 2)-1),
                    Unsimplified);
             end_proc):
Content::set("missingTemplate",
             CL::content("id"="missingTemplate", "keywd"="Missing Template",
                         "arity"="1*", 
                         CL::layout(CL::mrow(
                                            CL::mo("form"="prefix", "rbadness"="50000", "&lt;?"),
                                            CL::oprow(CL::oper("prio"=OP::Exprseq),
                                                      CL::mo("lbadness"="50000", ",")),
                                            CL::mo("form"="prefix", "lbadness"="50000", "?>")
                                            ))),
             FAIL):

unalias(CL):


  
// initialize identifier translation tables
map([  hold(`NIL`) = NIL, hold(`FAIL`) = FAIL, 
       /* hold(``) = null(), */ hold(`&NULL;`) = null(),
       `&pi;` = PI, Content::Cpi() = PI,
       Content::Cinfinity() = hold(infinity),
       //Content::Ceulergamma() = EULER,  
       hold(`&image;`) = hold(Im), hold(`&real;`) = hold(Re),
       Content::Cintegers() = hold(Z_), Content::Crationals() = hold(Q_),  
       Content::Creals() = hold(R_), Content::Ccomplexes() = hold(C_), 
       Content::Cnaturalnumbers() = hold(N_),  Content::Cprimes() = hold(P_)], 
     Content::translateIdents):

// end of file 
