/** output::MMLPresentation -- translates MathML Content to Presentation
 *
 *  
 */

alias(MMLP = output::MMLPresentation):
alias(XML =  adt::XML):
alias(OP =   output::Priority):

MMLP:=newDomain(hold(MMLP)):

MMLP::new :=
proc(cont : XML, outerPrio = OP::Noop : DOM_INT)
  local result, method, attrib, prio;
begin
  if testargs() then
    if XML::markupType(cont) <> "element" then
      error("only XML Elements supported")
    end_if;
  end_if;

  
  method := dom::c2p[XML::elementName(cont)];
  if type(method) <> "_index" then
    attrib := table(op(XML::attributes(cont)));
    result := method([op(attrib)], [op(cont)]);
    assert(type(result) = DOM_LIST);   assert(nops(result) = 2);
    prio := op(result, 2);   result := op(result, 1);

    // if cont has an id then add a xref to the result
    if type(attrib["id"]) <> "_index" and
       select(XML::attributes(result), x->lhs(x)="xref")=[] then
      result :=
      XML::Element(XML::elementName(result),
                   ["xref"=attrib["id"]].XML::attributes(result),
                   op(result));
    end_if;
    // depending on the values of outerPrio and prio
    // we have to set parentheses
    if prio <> OP::Noop and outerPrio >= prio then
      result := dom::mfenced(result);
    end_if;
  elif slot(dom, XML::elementName(cont)) <> FAIL then
    // cont was presentation embedded in content
    result := cont;
  else
    error("Element type '".XML::elementName(cont)."' not supported")
  end_if;
  result;
end_proc:

/** Functions to create Presentation Elements as defined by MathML:
 */

/*  Token Elements */

MMLP::mi :=            () -> XML::Element("mi",
  [if length(args())>1 and args(1)[1]<>"&"
   then "mathvariant"="normal" else null() end_if],
  args()):
MMLP::mn :=            () -> XML::Element("mn", args()):
MMLP::mo :=            () -> XML::Element("mo", args()):
MMLP::mtext :=         () -> XML::Element("mtext", args()):
// MMLP::mspace :=        () -> XML::Element("mspace", args()):
MMLP::ms :=            () -> XML::Element("ms", args()):
// MMLP::mglyph :=        () -> XML::Element("mglyph", args()):


/*  General Layout Schemata */

MMLP::mrow :=          () -> XML::Element("mrow", args()):
MMLP::mfrac :=         () -> XML::Element("mfrac", args()):
MMLP::msqrt :=         () -> XML::Element("msqrt", args()):
MMLP::mroot :=         () -> XML::Element("mroot", args()):
// MMLP::mstyle :=        () -> XML::Element("mstyle", args()):
// MMLP::merror :=        () -> XML::Element("merror", args()):
// MMLP::mpadded :=       () -> XML::Element("mpadded", args()):
// MMLP::mphantom :=      () -> XML::Element("mphantom", args()):
MMLP::mfenced :=
() -> XML::Element("mfenced",
                   if output::MMLContent::fixesForConTeXt() = TRUE then
                     ["separators"=","];
                   else
                     null();
                   end_if, args()):
// MMLP::menclose :=      () -> XML::Element("menclose", args()):


/* Script and Limit Schemata */

MMLP::msub :=          () -> XML::Element("msub", args()):
MMLP::msup :=          () -> XML::Element("msup", args()):
MMLP::msubsup :=       () -> XML::Element("msubsup", args()):
MMLP::munder :=        () -> XML::Element("munder", args()):
MMLP::mover :=         () -> XML::Element("mover", args()):
// MMLP::munderover :=    () -> XML::Element("munderover", args()):
// MMLP::mmultiscripts := () -> XML::Element("mmultiscripts", args()):

/* Tables and Matrices */


MMLP::mtable :=        () -> XML::Element("mtable", args()):
// MMLP::mlabeledtr :=    () -> XML::Element("mlabeledtr", args()):
MMLP::mtr :=           () -> XML::Element("mtr", args()):
MMLP::mtd :=           () -> XML::Element("mtd", args()):
// MMLP::maligngroup :=   () -> XML::Element("maligngroup", args()):
// MMLP::malignmark :=    () -> XML::Element("malignmark", args()):

/* Enlivening Expressions */ 

// MMLP::maction :=       () -> XML::Element("maction", args()):


// used for ln, log, ..., sin, cos, ...
MMLP::specialFunction :=
  proc(str : DOM_STRING, attr : DOM_LIST, ops : DOM_LIST)
    local trivial;
  begin
    trivial := FALSE;
    if contains({"ci", "cn"}, XML::elementName(op(ops, 1))) then
      trivial := TRUE;
    end_if;
    // leave out function parenthesis when argument is trivial
    if trivial then
      [MMLP::mrow(MMLP::mi(str),
// I think it should be there, but rendering is ugly
//                  MMLP::mo(MMLP::mmlEntity("&ApplyFunction;")),
                  // instead explicit whitespace
                  MMLP::mo(MMLP::mmlEntity("&nbsp;")),
                  MMLP::mrow(op(map(ops, MMLP, OP::Noop)))),
       OP::Mult];
    else
      [MMLP::mrow(MMLP::mi(str),
                  MMLP::mo(MMLP::mmlEntity("&ApplyFunction;")),
                  MMLP::mfenced(op(map(ops, MMLP, OP::Noop)))),
       OP::Noop];
    end_if;
  end_proc:

// used for standard functional expressions
MMLP::standardFunction :=
  proc(str : DOM_STRING, attr : DOM_LIST, ops : DOM_LIST)
    local oper;
  begin
    oper := MMLP::mmlEntity(str);
      [MMLP::mrow(MMLP::mi(oper),
                  MMLP::mo(MMLP::mmlEntity("&ApplyFunction;")),
                  MMLP::mfenced(op(map(ops, MMLP, OP::Noop)))),
       OP::Noop];
  end_proc:

// a binary operator
MMLP::binaryOperator :=
  proc(str  : DOM_STRING,  // MathML operator Entity
       prio : DOM_INT,     // operator priority -> output::Priority
       attr : DOM_LIST,    // list with attributes (equations)
       ops  : DOM_LIST)    // operands of the operator
    local oper;
  begin
    oper := MMLP::mmlEntity(str);
    [MMLP::mrow(MMLP(op(ops, 1), prio),
                MMLP::mo(oper),
                MMLP(op(ops, 2), prio)),
     prio]:
  end_proc:
// like binary, but arbitrary number of arguments allowed
MMLP::inFixOperator := 
  proc(str  : DOM_STRING,  // MathML operator Entity
       prio : DOM_INT,     // operator priority -> output::Priority
       attr : DOM_LIST,    // list with attributes (equations)
       ops  : DOM_LIST)    // operands of the operator
    local i, oper;
  begin
    oper := MMLP::mmlEntity(str);
    [MMLP::mrow(MMLP(op(ops, 1), prio),
                (MMLP::mo(oper), MMLP(op(ops, i), prio)) $ i=2..nops(ops)),
     prio]:
  end_proc:
// e.g.: n!
MMLP::postFixOperator := 
  proc(str  : DOM_STRING,  // MathML operator Entity
       prio : DOM_INT,     // operator priority -> output::Priority
       attr : DOM_LIST,    // list with attributes (equations)
       ops  : DOM_LIST)    // operands of the operator
    local oper;
  begin
    oper := MMLP::mmlEntity(str);
    [MMLP::mrow(MMLP(op(ops), prio),
                MMLP::mo(oper)),
     prio]:
  end_proc:
// e.g.: not ...
MMLP::preFixOperator := 
  proc(str  : DOM_STRING,  // MathML operator Entity
       prio : DOM_INT,     // operator priority -> output::Priority
       attr : DOM_LIST,    // list with attributes (equations)
       ops  : DOM_LIST)    // operands of the operator
    local oper;
  begin
    oper := MMLP::mmlEntity(str);
    [MMLP::mrow(MMLP::mo(oper),
                MMLP(op(ops), prio)),
     prio]:
  end_proc:

// some renderers have problems with entity names;
// these are the equivalent UNICODE values for output:
MMLP::entity2UNICODE :=
  table("&InvisibleTimes;" = "&#x02062;",
        "&ApplyFunction;" = "&#x02061;",
        "&NotEqual;" = "&#x02260;",
        "&Zopf;" = "&#x02124;",
        "&Ropf;" = "&#x0211D;",
        "&Qopf;" = "&#x0211A;",
        "&Nopf;" = "&#x02115;",
        "&Copf;" = "&#x02102;",
        "&Popf;" = "&#x02119;",
//        "&ExponentialE;" = "e",
//        "&ImaginaryI;" = "i",
        "&NotANumber;" = "NaN",
        "&true;" = "TRUE",
        "&false;" = "FALSE",
        "&unknown;" = "UNKNOWN",
        "&emptyset;" = "&#8709;",
//        "&pi;" = "&#x003C0;",
        "&gamma;" = "&#x003B3;",
        "&infin;" = "&#x0221E;",
        "&Element;" = "&#x02208;",
        "&vee;" = "&#x02228;",
        "&wedge;" = "&#x02227;",
        "&not;" = "&#x000AC;",
        "&DoubleRightArrow;" = "&#x021D2;",
        "&real;" = "&#x0211C;",
        "&image;" = "&#x02111;",
        "&cup;" = "&#x0222A;",
        "&Cup;" = "&#x022D3;",
        "&cap;" = "&#x02229;",
        "&setminus;" = "&#x02216;",
        "&rightarrow;" = "&#x02192;",
        "&Integral;" = "&#x0222B;",
        "&DifferentialD;" = "&#x02146;",
        "&nbsp;" = "&#x000A0;",
        "&subset;" = "&#x02282;",
        "&SubsetEqual;" = "&#x02286;",
        "&NotSubset;" = "&#x02284;",
        "&NotSubsetEqual;" = "&#x02288;",
        "&Sum;" = "&#x02211;",
        "&Product;" = "&#x0220F;",
        "&searrow;" = "&#x02198;",
        "&nearrow;" = "&#x02197;",
        "&Gamma;" = "&#x00393;",
        "&;" = "&#x;",
        "&;" = ""
        ):
MMLP::mmlEntity := x -> if contains(MMLP::entity2UNICODE, x) then
                          MMLP::entity2UNICODE[x]
                        else
                          x
                        end_if:

/** c2p:  tabel of conversion procedures Content -> Presentation
 *
 *  c2p is a table containing conversion procedures for
 *  MathML Content types to presentation.  The indices
 *  are the element names of the MathML Content elements.
 *
 *  These procedures get the following arguments:
 *  1.  a list of equation containing the attributes of the
 *      MathML Content element.
 *  2.  a list containing the operands of the MathML Content element.
 *
 *  Return value:  a list of
 *                 1.  the MathML Presentation tree (with domtype adt::XML)
 *                 2.  the output priority of this tree (a DOM_INT
 *                     corresponding to the values defined in outpu::Priority)
 */
MMLP::c2p := table():


// MathML Content 2.0 Token Elements

MMLP::c2p["cn"] :=
  proc(attributes, ops)
    local nType, baseAttr, res;
  begin
    nType := table(attributes)["type"];
    if testtype(nType, DOM_STRING) then
      case nType
        of "rational" do
          assert(XML::elementName(op(ops, 2)) = "sep");
          res := MMLP::mfrac(MMLP::mn(op(ops, 1)),
                             MMLP::mn(op(ops, 3))),
                 OP::Noop;
          break;
/*
        of "real" do
          assert(nops(ops) = 1);
          res := MMLP::mn(op(ops)), OP::Noop;
          break;
*/          
        of "integer" do
          // look for base
          assert(nops(ops) = 1);
          baseAttr := attributes["base"];
          if  testtype(baseAttr, DOM_STRING) then
            res := MMLP::msub(MMLP::mn(op(ops)),
                              MMLP::mn(text2expr(baseAttr))),
                   OP::Noop;          
          else
            res := MMLP::mn(op(ops)), OP::Noop;
          end_if;
          break;

/*
        of "complex-cartesian" do
          assert(XML::elementName(op(ops, 2)) = "sep");
          res := MMLP::mrow(MMLP::mn(op(ops, 1)),
                            MMLP::mo("+"),
                            MMLP::mn(op(ops, 3)),
                            MMLP::mo(MMLP::mmlEntity("&InvisibleTimes;")),
                            MMLP::mn(MMLP::mmlEntity("&ImaginaryI;"))),
                 OP::Plus;          
          break;
          
        of "complex-polar" do
          assert(XML::elementName(op(ops, 2)) = "sep");
          res := MMLP::mrow(MMLP::mi(hold(Polar)),
                            MMLP::mfenced(MMLP::mn(op(ops, 1)),
                                          MMLP::mn(op(ops, 3)))),
                 OP::Index;
          break;
*/          
          
        of "constant" do
          assert(nops(ops) = 1);
          if domtype(op(ops)) = XML then
            res :=MMLP(op(ops), OP::Noop), OP::Noop
          else
            res := MMLP::mn(op(ops)), OP::Noop:
          end_if;
          break;

        otherwise
          warning("unknown identifier type: '".nType."', ignored");
          res := MMLP::mn(op(ops)), OP::Noop:
      end_case;
    else
      assert(nops(ops) = 1);
      if domtype(op(ops)) = XML then
        res :=MMLP(op(ops), OP::Noop), OP::Noop
      else
        res := MMLP::mn(op(ops)), OP::Noop:
      end_if
    end_if;
    [res];
  end_proc:
MMLP::c2p["ci"] := (attr, ops) -> if domtype(op(ops)) = XML then
                                    [MMLP(op(ops), OP::Noop), OP::Noop]
                                  else
                                    [MMLP::mi(op(ops)), OP::Noop]
                                  end_if:
MMLP::c2p["csymbol"] := (attr, ops) -> if domtype(op(ops)) = XML then
                                    [MMLP(op(ops), OP::Noop), OP::Noop]
                                  else
                                    [MMLP::ms(op(ops)), OP::Noop]
                                  end_if:


// MathML Content 2.0 Basis Content Elements

MMLP::c2p["apply"] :=
  proc(attr, ops)
    local op0, oper, Ops, func, res;
  begin
    op0 := op(ops, 1);
    Ops := [op(ops, 2..nops(ops))];
    oper := XML::elementName(op0);
    func := MMLP::c2p[oper];
    if type(func) <> "_index" and oper <> "apply" and
//      XML::attributes(op0) = [] and
      [op(op0)] = [] then
      res := func(XML::attributes(op0), Ops)
    else
      res := [MMLP::mrow(MMLP(op0, OP::Index-1),
                        MMLP::mo(MMLP::mmlEntity("&ApplyFunction;")),
                        MMLP::mfenced(op(map(Ops, MMLP)))), OP::Index]
    end_if;
    res
  end_proc:
// MMLP::c2p["reln"]:= x -> error("reln is obsolete;  not supported"):
// MMLP::c2p["fn"]  := x -> error("fn is obsolete;  not supported"):
MMLP::c2p["interval"]:=
  proc(attr, ops)
  begin
    case  table(attr)["closure"]
      of "open" do
        [MMLP::mfenced(["open"="(", "close"=")"],
                       MMLP(op(ops, 1), OP::Noop),
                       MMLP(op(ops, 2), OP::Noop)),
         OP::Noop];
        break;
        
      of "open-closed" do
        [MMLP::mfenced(["open"="(", "close"="]"],
                       MMLP(op(ops, 1), OP::Noop),
                       MMLP(op(ops, 2), OP::Noop)),
         OP::Noop];
        break;
        
      of "closed-open" do
        [MMLP::mfenced(["open"="[", "close"=")"],
                       MMLP(op(ops, 1), OP::Noop),
                       MMLP(op(ops, 2), OP::Noop)),
         OP::Noop];
        break;
        
      of "closed" do
        // fall through
      otherwise
        [MMLP::mfenced(["open"="[", "close"="]"],
                       MMLP(op(ops, 1), OP::Noop),
                       MMLP(op(ops, 2), OP::Noop)),
         OP::Noop]        
    end_case;
  end_proc:
MMLP::c2p["inverse"]             := x -> error("not yet supported"):
// MMLP::c2p["sep"]:= x -> error("should not be called"):
MMLP::c2p["condition"]:= (attr, ops)->[MMLP(op(ops), OP::Noop), OP::Noop]:
// MMLP::c2p["declare"]             := x -> error("not yet supported"):
MMLP::c2p["lambda"]:=
(attr, ops)->MMLP::binaryOperator("&rightarrow;",
                                  OP::Exprseq, attr,
                                  [op(op(ops, 1)), op(ops, 2)]):
MMLP::c2p["compose"]             := x -> error("not yet supported"):
// MMLP::c2p["ident"]               := x -> error("not yet supported"):
// MMLP::c2p["domain"]              := x -> error("not yet supported"):
// MMLP::c2p["codomain"]            := x -> error("not yet supported"):
// MMLP::c2p["image"]               := x -> error("not yet supported"):
// MMLP::c2p["domainofapplication"] := x -> error("not yet supported"):
MMLP::c2p["piecewise"]:=
  proc(attr, ops)
    local elemName, pieces, i;
  begin
    pieces := null();
    for i in ops do
      elemName := XML::elementName(i);
      if elemName = "piece" then
        pieces := pieces, MMLP::mtr(MMLP::mtd(MMLP(op(i, 1), OP::Noop)),
                                    MMLP::mtd(MMLP::mtext("if")),
                                    MMLP::mtd(MMLP(op(i, 2), OP::Noop)))
      elif elemName = "otherwise" then
        pieces := pieces, MMLP::mtr(MMLP::mtd(MMLP(op(i), OP::Noop)),
                                    MMLP::mtd(MMLP::mtext("otherwise")))
      else
        error("unexpected piecewise entry");
      end_if;
    end_for;
    [MMLP::mrow(MMLP::mo("{"),
                MMLP::mtable(["columnalign"="center center left",
                              "columnspacing"="5pt"],
                             pieces)), OP::Noop];
  end_proc:
// MMLP::c2p["piece"]:= x -> error("makes only sense in piecewise"):
// MMLP::c2p["otherwise"]:= x -> error("makes only sense in piecewise"):

// MathML Content 2.0 Arithmetic, Algebra and Logic

MMLP::c2p["quotient"]            := x -> error("not yet supported"):
MMLP::c2p["factorial"]:= ()->MMLP::postFixOperator("!", OP::Fact, args()):
MMLP::c2p["divide"] :=
  (attr, ops) -> [MMLP::mfrac(MMLP(ops[1], OP::Noop),
                             MMLP(ops[2], OP::Noop)), OP::Mult+1]:
MMLP::c2p["min"]:=
  proc(attr, ops)
    local res;
  begin
    if nops(ops)=3 and
      XML::elementName(op(ops, 1)) = "bvar" and
      XML::elementName(op(ops, 2)) = "condition" then
      // notation with bound variable and condition
      if op(op(ops, 1)) = op(ops, 3) then
        // bound variable equals value; use simple form: set over condition
        res := MMLP::mrow(MMLP::mi("min"),
                          MMLP::mfenced(["open"="{", "close"="}"],
                                        MMLP(op(op(ops, 2))))
                          );
      else
        res := MMLP::mrow(MMLP::mub(MMLP::mi("min"), MMLP(op(op(ops, 1)))),
                          MMLP::mfenced(["open"="{", "close"="}"],
                                        MMLP(op(op(ops, 3))),
                                        MMLP::mo("|"),
                                        MMLP(op(op(ops, 2))))
                          )
      end_if;
    else
      // simple min over a set
      res := MMLP::mrow(MMLP::mi("min"),
                        MMLP::mfenced(["open"="{", "close"="}"],
                                      op(map(ops, MMLP)))
                        );
    end_if:
    [res, OP::Noop]
  end_proc:
MMLP::c2p["max"]:=
  proc(attr, ops)
    local res;
  begin
    if nops(ops)=3 and
      XML::elementName(op(ops, 1)) = "bvar" and
      XML::elementName(op(ops, 2)) = "condition" then
      // notation with bound variable and condition
      if op(op(ops, 1)) = op(ops, 3) then
        // bound variable equals value; use simple form: set over condition
        res := MMLP::mrow(MMLP::mi("max"),
                          MMLP::mfenced(["open"="{", "close"="}"],
                                        MMLP(op(op(ops, 2))))
                          );
      else
        res := MMLP::mrow(MMLP::mub(MMLP::mi("max"), MMLP(op(op(ops, 1)))),
                          MMLP::mfenced(["open"="{", "close"="}"],
                                        MMLP(op(op(ops, 3))),
                                        MMLP::mo("|"),
                                        MMLP(op(op(ops, 2))))
                          )
      end_if;
    else
      // simple max over a set
      res := MMLP::mrow(MMLP::mi("max"),
                        MMLP::mfenced(["open"="{", "close"="}"],
                                      op(map(ops, MMLP)))
                        );
    end_if:
    [res, OP::Noop]
  end_proc:
MMLP::c2p["minus"]:=
(attr, ops) -> if nops(ops) = 1 then
                 // unary minus
                 MMLP::preFixOperator("-", OP::Plus, args())
               else // binary minus
                 MMLP::binaryOperator("-", OP::Plus, args())
               end_if:
MMLP::c2p["plus"] :=
  proc(attr, ops)
    local i, Nops, res, isUnaryMinus;
  begin
    isUnaryMinus :=
    x -> bool(_lazy_and(XML::elementName(x) = "apply",
              XML::elementName(op(x, 1)) = "minus",
              nops(x) = 2));
    Nops := nops(ops);
    if isUnaryMinus(ops[1]) then
      res := MMLP::mo("-"), MMLP(op(ops[1], 2), OP::Plus);
    else
      res := MMLP(ops[1], OP::Plus);
    end_if;
    for i from 2 to Nops do
      if isUnaryMinus(ops[i]) then
        res := res, MMLP::mo("-"), MMLP(op(ops[i], 2), OP::Plus);
      else
        res := res, MMLP::mo("+"), MMLP(ops[i], OP::Plus);
      end_if;
    end_for;
    [MMLP::mrow(res), OP::Plus];
  end_proc:
MMLP::c2p["power"] :=
  (attr, ops) -> [MMLP::msup(MMLP(ops[1], OP::Power),
                            MMLP(ops[2])), OP::Power]: 
MMLP::c2p["rem"]                 := x -> error("not yet supported"):
MMLP::c2p["times"] :=
  proc(attr, ops)
    local i, Nops;
  begin
    Nops := nops(ops);
    [MMLP::mrow((MMLP(ops[i], OP::Mult),
                 MMLP::mo(MMLP::mmlEntity("&InvisibleTimes;"))) $ i = 1..Nops-1,
                MMLP(ops[Nops], OP::Mult)), OP::Mult]
  end_proc:
MMLP::c2p["root"]:=
  proc(attr, ops)
  begin
    if nops(ops) = 2 and TRUE then
      assert(XML::elementName(op(ops, 1)) = "degree");
      [MMLP::mroot(MMLP(op(ops, 2)), MMLP(op(op(ops, 1)))), OP::Noop]
    else
      [MMLP::msqrt(MMLP(op(ops))), OP::Noop]
    end_if:
  end_proc:
// MMLP::c2p["gcd"]                 := x -> error("not yet supported"):
MMLP::c2p["and"]:=     ()->MMLP::inFixOperator("&wedge;", OP::And, args()):
MMLP::c2p["or"]:=      ()->MMLP::inFixOperator("&vee;", OP::Or, args()):
MMLP::c2p["xor"]:=     ()->MMLP::inFixOperator("xor", OP::Xor, args()):
MMLP::c2p["not"]:=     ()->MMLP::preFixOperator("&not;", OP::Not, args()):
MMLP::c2p["implies"]:= ()->MMLP::binaryOperator("&DoubleRightArrow;",
                                                OP::Implies, args()):
// MMLP::c2p["forall"]              := x -> error("not yet supported"):
// MMLP::c2p["exists"]              := x -> error("not yet supported"):
MMLP::c2p["abs"]                 := x -> error("not yet supported"):
MMLP::c2p["conjugate"]           := x -> error("not yet supported"):
MMLP::c2p["arg"]                 := x -> error("not yet supported"):
MMLP::c2p["real"]:=      ()->MMLP::standardFunction("&real;", args()):
MMLP::c2p["imaginary"]:= ()->MMLP::standardFunction("&image;", args()):
MMLP::c2p["lcm"]                 := x -> error("not yet supported"):
MMLP::c2p["floor"]               := x -> error("not yet supported"):
MMLP::c2p["ceiling"]             := x -> error("not yet supported"):


// MathML Content 2.0 Relations

MMLP::c2p["eq"]:= ()->MMLP::binaryOperator("=", OP::Relation, args()):
MMLP::c2p["neq"]:=()->MMLP::binaryOperator("&NotEqual;", OP::Relation, args()):
MMLP::c2p["gt"]:= ()->MMLP::binaryOperator("&gt;", OP::Relation, args()):
MMLP::c2p["lt"]:= ()->MMLP::binaryOperator("&lt;", OP::Relation, args()):
MMLP::c2p["geq"]:=()->MMLP::binaryOperator("&gt;=", OP::Relation, args()):
MMLP::c2p["leq"]:=()->MMLP::binaryOperator("&lt;=", OP::Relation, args()):
MMLP::c2p["equivalent"]:= ()->MMLP::binaryOperator("&;", OP::Relation, args()):
MMLP::c2p["approx"]:= ()->MMLP::binaryOperator("&;", OP::Relation, args()):
// MMLP::c2p["factorof"]:= x -> error("not yet supported"):


// MathML Content 2.0 Calculus and Vector Calculus

MMLP::c2p["int"]:=
  proc(attr, ops)
    local bvar, res, ex;
  begin
    bvar := MMLP(op(ops, 1), OP::Noop);
    ex := MMLP(op(ops, nops(ops)), OP::Noop);
    if nops(ops) = 2 then
      // indefinite integral
      res := MMLP::mrow(MMLP::mo(MMLP::mmlEntity("&Integral;")), ex,
                        MMLP::mo(MMLP::mmlEntity("&DifferentialD;")), bvar);
    elif nops(ops) = 3 then
      if XML::elementName(ops[2]) = "condition" then
      // ops = bvar, condition, Expr
        res := MMLP::mrow(
                  MMLP::munder(MMLP::mo(MMLP::mmlEntity("&Integral;")),
                                MMLP(ops[2], OP::Noop)),
                          ex,
                          MMLP::mo(MMLP::mmlEntity("&DifferentialD;")),
                          bvar);
      elif XML::elementName(ops[2]) = "interval" then
      // definite integral:  ops = bvar, interval, Expr
        res := MMLP::mrow(
                  MMLP::msubsup(MMLP::mo(MMLP::mmlEntity("&Integral;")),
                                MMLP(ops[2][1], OP::Noop),
                                MMLP(ops[2][2], OP::Noop)),
                          ex,
                          MMLP::mo(MMLP::mmlEntity("&DifferentialD;")),
                          bvar);
      else
        error("wrong usage of \"int\" element");
      end_if;
    elif nops(ops) = 4 then
      // definite integral:  ops = bvar, uplimit, lowlimit, Expr
      if  XML::elementName(ops[2]) = "lowlimit" and
        XML::elementName(ops[3]) = "uplimit" then
        res := MMLP::mrow(
                  MMLP::msubsup(MMLP::mo(MMLP::mmlEntity("&Integral;")),
                                MMLP(ops[2], OP::Noop),
                                MMLP(ops[3], OP::Noop)),
                          ex,
                          MMLP::mo(MMLP::mmlEntity("&DifferentialD;")),
                          bvar);
      elif XML::elementName(ops[3]) = "lowlimit" and
        XML::elementName(ops[2]) = "uplimit" then
        res := MMLP::mrow(
                  MMLP::msubsup(MMLP::mo(MMLP::mmlEntity("&Integral;")),
                                MMLP(ops[3], OP::Noop),
                                MMLP(ops[2], OP::Noop)),
                          ex,
                          MMLP::mo(MMLP::mmlEntity("&DifferentialD;")),
                          bvar);
      else
        error("wrong usage of \"int\" element");
      end_if
    else
      error("wrong usage of \"int\" element");
    end_if;
   [res, OP::Plus]
  end_proc:
MMLP::c2p["diff"]                := x -> error("not yet supported"):
MMLP::c2p["partialdiff"]         := x -> error("not yet supported"):
MMLP::c2p["lowlimit"]:= (attr, ops)->[MMLP(op(ops), OP::Noop), OP::Noop]:
MMLP::c2p["uplimit"]:=  (attr, ops)->[MMLP(op(ops), OP::Noop), OP::Noop]:
MMLP::c2p["bvar"]:=     (attr, ops)->[MMLP(op(ops), OP::Noop), OP::Noop]:
MMLP::c2p["degree"]:=   (attr, ops)->[MMLP(op(ops), OP::Noop), OP::Noop]:
MMLP::c2p["divergence"]          := x -> error("not yet supported"):
MMLP::c2p["grad"]                := x -> error("not yet supported"):
MMLP::c2p["curl"]                := x -> error("not yet supported"):
MMLP::c2p["laplacian"]           := x -> error("not yet supported"):


// MathML Content 2.0 Theory of Sets

MMLP::c2p["set"]:=
  proc(attr, ops)
    local res;
  begin
    if nops(ops)=3 and
      XML::elementName(op(ops, 1)) = "bvar" and
      XML::elementName(op(ops, 2)) = "condition" then
      // notation with bound variable and condition
      res := MMLP::mfenced(["open"="{", "close"="}"],
                           MMLP::mrow(MMLP(op(ops, 3)),
                                      MMLP::mo("|"),
                                      MMLP(op(op(ops, 2)))))
    else
      // simple set
      res := MMLP::mfenced(["open"="{", "close"="}"],
                           op(map(ops, MMLP, OP::Noop)));
    end_if:
    [res, OP::Noop]
  end_proc:
MMLP::c2p["list"]:= // MathML order attribute ignored!
  proc(attr, ops)
    local res;
  begin
    if nops(ops)=3 and
      XML::elementName(op(ops, 1)) = "bvar" and
      XML::elementName(op(ops, 2)) = "condition" then
      // notation with bound variable and condition
      res := MMLP::mfenced(["open"="[", "close"="]"],
                           MMLP::mrow(MMLP(op(ops, 3)),
                                      MMLP::mo("|"),
                                      MMLP(op(op(ops, 2)))))
    else
      // simple set
      res := MMLP::mfenced(["open"="[", "close"="]"],
                           op(map(ops, MMLP, OP::Noop)));
    end_if:
    [res, OP::Noop]
  end_proc:
MMLP::c2p["union"]:= ()->MMLP::inFixOperator("&cup;", OP::Union, args()):
MMLP::c2p["intersect"]:= ()->MMLP::inFixOperator("&cap;",
                                                 OP::Intersect, args()):
MMLP::c2p["in"]:=  ()->MMLP::binaryOperator(MMLP::mmlEntity("&Element;"),
                                            OP::Relation, args()):
MMLP::c2p["notin"]:= ()->MMLP::binaryOperator(MMLP::mmlEntity("&notin;"),
                                              OP::Relation, args()):
MMLP::c2p["subset"]:= ()->MMLP::inFixOperator("&SubsetEqual;",
                                               OP::Relation, args()):
MMLP::c2p["prsubset"]:= ()->MMLP::inFixOperator("&subset;",
                                               OP::Relation, args()):
MMLP::c2p["notsubset"]:= ()->MMLP::inFixOperator("&NotSubsetEqual;",
                                               OP::Relation, args()):
MMLP::c2p["notprsubset"]:= ()->MMLP::inFixOperator("&NotSubset;",
                                               OP::Relation, args()):
MMLP::c2p["setdiff"]:= ()->MMLP::inFixOperator("&setminus;",
                                               OP::Minus, args()):
MMLP::c2p["card"]                := x -> error("not yet supported"):
MMLP::c2p["cartesianproduct"]    := x -> error("not yet supported"):


// MathML Content 2.0 Sequences and Series

MMLP::c2p["sum"]:=
  proc(attr, ops)
    local bvar, res, ex;
  begin
    bvar := MMLP(op(ops, 1), OP::Noop);
    ex := MMLP(op(ops, nops(ops)), OP::Noop);
    if nops(ops) = 2 then
      // indefinite sum
      res := MMLP::mrow(MMLP::mo(MMLP::mmlEntity("&Sum;")), ex);
    elif nops(ops) = 3 then
      if XML::elementName(ops[2]) = "condition" then
      // ops = bvar, condition, Expr
        res := MMLP::mrow(
                  MMLP::munder(MMLP::mo(MMLP::mmlEntity("&Sum;")),
                               MMLP(ops[2], OP::Relation)),
                          ex);
      else
        error("wrong usage of \"sum\" element");
      end_if;
    elif nops(ops) = 4 then
      // definite sum:  ops = bvar, uplimit, lowlimit, Expr
      if  XML::elementName(ops[2]) = "lowlimit" and
        XML::elementName(ops[3]) = "uplimit" then
        res := MMLP::mrow(
                  MMLP::msubsup(MMLP::mo(MMLP::mmlEntity("&Sum;")),
                                MMLP::mrow(bvar,
                                           MMLP::mo("="),
                                           MMLP(ops[2], OP::Relation)),
                                MMLP(ops[3], OP::Noop)),
                          ex);
      elif XML::elementName(ops[3]) = "lowlimit" and
        XML::elementName(ops[2]) = "uplimit" then
        res := MMLP::mrow(
                  MMLP::msubsup(MMLP::mo(MMLP::mmlEntity("&Sum;")),
                                MMLP::mrow(bvar,
                                           MMLP::mo("="),
                                           MMLP(ops[3], OP::Relation)),
                                MMLP(ops[2], OP::Noop)),
                          ex);
      else
        error("wrong usage of \"sum\" element");
      end_if
    else
      error("wrong usage of \"sum\" element");
    end_if;
   [res, OP::Plus]
  end_proc:
MMLP::c2p["product"]:= 
  proc(attr, ops)
    local bvar, res, ex;
  begin
    bvar := MMLP(op(ops, 1), OP::Noop);
    ex := MMLP(op(ops, nops(ops)), OP::Noop);
    if nops(ops) = 2 then
      // indefinite product
      res := MMLP::mrow(MMLP::mo(MMLP::mmlEntity("&Product;")), ex);
    elif nops(ops) = 3 then
      if XML::elementName(ops[2]) = "condition" then
      // ops = bvar, condition, Expr
        res := MMLP::mrow(
                  MMLP::munder(MMLP::mo(MMLP::mmlEntity("&Product;")),
                               MMLP(ops[2], OP::Relation)),
                          ex);
      else
        error("wrong usage of \"product\" element");
      end_if;
    elif nops(ops) = 4 then
      // definite product:  ops = bvar, uplimit, lowlimit, Expr
      if  XML::elementName(ops[2]) = "lowlimit" and
        XML::elementName(ops[3]) = "uplimit" then
        res := MMLP::mrow(
                  MMLP::msubsup(MMLP::mo(MMLP::mmlEntity("&Product;")),
                                MMLP::mrow(bvar,
                                           MMLP::mo("="),
                                           MMLP(ops[2], OP::Relation)),
                                MMLP(ops[3], OP::Noop)),
                          ex);
      elif XML::elementName(ops[3]) = "lowlimit" and
        XML::elementName(ops[2]) = "uplimit" then
        res := MMLP::mrow(
                  MMLP::msubsup(MMLP::mo(MMLP::mmlEntity("&Product;")),
                                MMLP::mrow(bvar,
                                           MMLP::mo("="),
                                           MMLP(ops[3], OP::Relation)),
                                MMLP(ops[2], OP::Noop)),
                          ex);
      else
        error("wrong usage of \"product\" element");
      end_if
    else
      error("wrong usage of \"product\" element");
    end_if;
   [res, OP::Plus]
  end_proc:
MMLP::c2p["limit"]:=
  proc(attr, ops)
    local bvar, res, ex;
  begin
    bvar := MMLP(ops[1], OP::Noop);
    ex :=   MMLP(ops[3], OP::Noop);
    if nops(ops) = 3 then
      if XML::elementName(ops[2]) = "condition" then
        // ops = bvar, condition, Expr
        res := MMLP::mrow(
                          MMLP::munder(MMLP::mtext("lim"),
                                       MMLP(ops[2], OP::Relation)),
                          ex);
      elif  XML::elementName(ops[2]) = "lowlimit" then
        res := MMLP::mrow(
                   MMLP::munder(MMLP::mtext("lim"),
                       MMLP::mrow(bvar,
                                  MMLP::mo(MMLP::mmlEntity("&rightarrow;")),
                                  MMLP(ops[2], OP::Relation))),
                          ex);
      else
        error("wrong usage of \"limit\" element");
      end_if;
    else
        error("wrong usage of \"limit\" element");
    end_if;
    [res, OP::Plus]
  end_proc:
MMLP::c2p["tendsto"]:=
(attr, ops)-> case table(op(attr))["type"]
                of "above" do
                  MMLP::binaryOperator("&searrow;",
                                       OP::Exprseq, attr, ops):
                  break;
                of "below" do
                  MMLP::binaryOperator("&nearrow;",
                                       OP::Exprseq, attr, ops):
                  break;
                of "two-sided" do
                  // fall through
                otherwise
                  MMLP::binaryOperator("&rightarrow;",
                                       OP::Exprseq, attr, ops):
              end_case:

// MathML Content 2.0 Elementary Classical Functions

MMLP::c2p["exp"]:=
  (attr,ops)->[MMLP::msup(MMLP::mn(MMLP::mmlEntity("&ExponentialE;")),
                                  MMLP(ops[1])),
               OP::Power]:
MMLP::c2p["ln"]:=      ()->MMLP::specialFunction("ln", args()):
MMLP::c2p["log"]:=     ()->MMLP::specialFunction("log", args()):
MMLP::c2p["sin"]:=     ()->MMLP::specialFunction("sin", args()):
MMLP::c2p["cos"]:=     ()->MMLP::specialFunction("cos", args()):
MMLP::c2p["tan"]:=     ()->MMLP::specialFunction("tan", args()):
MMLP::c2p["sec"]:=     ()->MMLP::specialFunction("sec", args()):
MMLP::c2p["csc"]:=     ()->MMLP::specialFunction("csc", args()):
MMLP::c2p["cot"]:=     ()->MMLP::specialFunction("cot", args()):
MMLP::c2p["sinh"]:=    ()->MMLP::specialFunction("sinh", args()):
MMLP::c2p["cosh"]:=    ()->MMLP::specialFunction("cosh", args()):
MMLP::c2p["tanh"]:=    ()->MMLP::specialFunction("tanh", args()):
MMLP::c2p["sech"]:=    ()->MMLP::specialFunction("sech", args()):
MMLP::c2p["csch"]:=    ()->MMLP::specialFunction("csch", args()):
MMLP::c2p["coth"]:=    ()->MMLP::specialFunction("coth", args()):
MMLP::c2p["arcsin"]:=  ()->MMLP::specialFunction("arcsin", args()):
MMLP::c2p["arccos"]:=  ()->MMLP::specialFunction("arccos", args()):
MMLP::c2p["arctan"]:=  ()->MMLP::specialFunction("arctan", args()):
MMLP::c2p["arccosh"]:= ()->MMLP::specialFunction("arccosh", args()):
MMLP::c2p["arccot"]:=  ()->MMLP::specialFunction("arccot", args()):
MMLP::c2p["arccoth"]:= ()->MMLP::specialFunction("arccoth", args()):
MMLP::c2p["arccsc"]:=  ()->MMLP::specialFunction("arccsc", args()):
MMLP::c2p["arccsch"]:= ()->MMLP::specialFunction("arccsch", args()):
MMLP::c2p["arcsec"]:=  ()->MMLP::specialFunction("arcsec", args()):
MMLP::c2p["arcsech"]:= ()->MMLP::specialFunction("arcsech", args()):
MMLP::c2p["arcsinh"]:= ()->MMLP::specialFunction("arcsinh", args()):
MMLP::c2p["arctanh"]:= ()->MMLP::specialFunction("arctanh", args()):


// MathML Content 2.0 Statistics

MMLP::c2p["mean"]                := x -> error("not yet supported"):
MMLP::c2p["sdev"]                := x -> error("not yet supported"):
MMLP::c2p["variance"]            := x -> error("not yet supported"):
MMLP::c2p["median"]              := x -> error("not yet supported"):
MMLP::c2p["mode"]                := x -> error("not yet supported"):
MMLP::c2p["moment"]              := x -> error("not yet supported"):
// MMLP::c2p["momentabout"]         := x -> error("not yet supported"):


// MathML Content 2.0 Linear Algebra

// MMLP::c2p["vector"]              := x -> error("not yet supported"):
MMLP::c2p["matrix"]:=
(attr, ops)->[MMLP::mfenced(MMLP::mtable(op(map(ops, MMLP, OP::Noop)))),
              OP::Noop]:
MMLP::c2p["matrixrow"]:=
  proc(attr, ops)
    local i;
  begin
    [MMLP::mtr(MMLP::mtd(MMLP(i, OP::Exprseq)) $ i in ops), OP::Noop]
  end_proc:
MMLP::c2p["determinant"]         := x -> error("not yet supported"):
MMLP::c2p["transpose"]           := x -> error("not yet supported"):
// MMLP::c2p["selector"]            := x -> error("not yet supported"):
MMLP::c2p["vectorproduct"]       := x -> error("not yet supported"):
MMLP::c2p["scalarproduct"]       := x -> error("not yet supported"):
// MMLP::c2p["outerproduct"]        := x -> error("not yet supported"):


// MathML Content 2.0 Semantic Mapping Elements

// MMLP::c2p["annotation"]          := x -> error("not yet supported"):
// MMLP::c2p["semantics"]           := x -> error("not yet supported"):
// MMLP::c2p["annotation-xml"]      := x -> error("not yet supported"):


// MathML Content 2.0 Contant and Symbol Elements

MMLP::c2p["integers"]:=      ()->[MMLP::mi(MMLP::mmlEntity("&Zopf;")),
                                  OP::Noop]:
MMLP::c2p["reals"]:=         ()->[MMLP::mi(MMLP::mmlEntity("&Ropf;")),
                                  OP::Noop]:
MMLP::c2p["rationals"]:=     ()->[MMLP::mi(MMLP::mmlEntity("&Qopf;")),
                                  OP::Noop]:
MMLP::c2p["naturalnumbers"]:=()->[MMLP::mi(MMLP::mmlEntity("&Nopf;")),
                                  OP::Noop]:
MMLP::c2p["complexes"]:=     ()->[MMLP::mi(MMLP::mmlEntity("&Copf;")),
                                  OP::Noop]:
MMLP::c2p["primes"]:=        ()->[MMLP::mi(MMLP::mmlEntity("&Popf;")),
                                  OP::Noop]:
MMLP::c2p["exponentiale"]:=  ()->[MMLP::mi(MMLP::mmlEntity("&ExponentialE;")),
                                  OP::Noop]:
MMLP::c2p["imaginaryi"]:=    ()->[MMLP::mi(MMLP::mmlEntity("&ImaginaryI;")),
                                  OP::Noop]:
MMLP::c2p["notanumber"]:=    ()->[MMLP::mi(MMLP::mmlEntity("&NotANumber;")),
                                  OP::Noop]:
MMLP::c2p["true"]:=          ()->[MMLP::mi(MMLP::mmlEntity("&true;")),
                                  OP::Noop]:
MMLP::c2p["false"]:=         ()->[MMLP::mi(MMLP::mmlEntity("&false;")),
                                  OP::Noop]:
MMLP::c2p["emptyset"]:=      ()->[MMLP::mi(MMLP::mmlEntity("&emptyset;")),
                                  OP::Noop]:
MMLP::c2p["pi"]:=            ()->[MMLP::mi(MMLP::mmlEntity("&pi;")),
                                  OP::Noop]:
MMLP::c2p["eulergamma"]:=    ()->[MMLP::mi(MMLP::mmlEntity("&gamma;")),
                                  OP::Noop]:
MMLP::c2p["infinity"]:=      ()->[MMLP::mi(MMLP::mmlEntity("&infin;")),
                                  OP::Noop]:

MMLP::c2p["index"] :=
(attr, ops) -> [MMLP::msub(MMLP(ops[1], OP::Index),
                            MMLP(ops[2])), OP::Noop]: 

// NOT in MathML Content 2.0:  MuPAD Extensions

MMLP::c2p["unknown"] :=      ()->[MMLP::mi(MMLP::mmlEntity("&unknown;")),
                                  OP::Noop]: 
MMLP::c2p["seq"] :=
(attr, data) -> ([MMLP::mfenced(["open"="", "close"="", "separators"=","],
				op(map(data, MMLP, OP::Exprseq))),
		  OP::Exprseq]):
MMLP::c2p["fnest"] := x -> error("not yet supported"):
MMLP::c2p["besselI"] := x -> error("not yet supported"):
MMLP::c2p["besselJ"] := x -> error("not yet supported"):
MMLP::c2p["besselK"] := x -> error("not yet supported"):
MMLP::c2p["besselY"] := x -> error("not yet supported"):
MMLP::c2p["lambertW"] := x -> error("not yet supported"):
MMLP::c2p["wrightOmega"] := x -> error("not yet supported"):
MMLP::c2p["slot"] :=
(attr, data) -> ([MMLP::mrow(MMLP(data[1], OP::Slot),
			    MMLP::mo("::"),
			    MMLP::mi(data[2])),
		 OP::Noop]):
MMLP::c2p["dilog"] := x -> error("not yet supported"):
MMLP::c2p["polylog"] := x -> error("not yet supported"):
MMLP::c2p["infinitynorm"] := x -> error("not yet supported"):
MMLP::c2p["onenorm"] := x -> error("not yet supported"):
MMLP::c2p["norm"] := x -> error("not yet supported"):
MMLP::c2p["othernorm"] := x -> error("not yet supported"):
MMLP::c2p["binomial"] := x -> error("not yet supported"):
MMLP::c2p["range"] := x -> error("not yet supported"):
MMLP::c2p["table"] := x -> error("not yet supported"):
MMLP::c2p["EQS"] := x -> error("not yet supported"):
MMLP::c2p["tableau"] := x -> error("not yet supported"):
MMLP::c2p["assign"] := x -> error("not yet supported"):
MMLP::c2p["row"] := x -> error("not yet supported"):
MMLP::c2p["concat"] := x -> error("not yet supported"):
MMLP::c2p["fenced"] := (attr, data) -> [MMLP::mfenced(op(map(data, MMLP, OP::Noop))), OP::Exprseq]:
MMLP::c2p["gamma"] := ()->MMLP::standardFunction("&Gamma;", args()):
MMLP::c2p["igamma"] := MMLP::c2p["gamma"]:

unalias(OP):
unalias(XML):
unalias(MMLP):

// end of file
